interface DebounceOptions {
    leading?: boolean;
    trailing?: boolean;
    maxWait?: number;
}
interface ThrottleOptions {
    leading?: boolean;
    trailing?: boolean;
}
/**
 * Creates a debounced function that delays invoking the provided callback until
 * after `waitMs` milliseconds have elapsed since the last time the function was called.
 *
 * @template T - The type of the callback function.
 * @param callback - The function to debounce.
 * @param waitMs - The number of milliseconds to delay (default is 0).
 * @param options - Optional configuration:
 *  - `leading` (default: false): If true, the callback will be invoked on the leading edge of the timeout.
 *  - `trailing` (default: true): If true, the callback will be invoked on the trailing edge of the timeout.
 *  - `maxWait` (default: Infinity): The maximum time the callback can be delayed before it's invoked.
 * @returns A debounced version of the callback with a `cancel` method to clear the timer.
 * @throws If `maxWait` is less than `waitMs`.
 */
export declare function debounce<T extends (...args: Parameters<T>) => void>(callback: T, waitMs?: number, options?: DebounceOptions): ((...args: Parameters<T>) => void) & {
    cancel(): void;
};
/**
 * Creates a throttled function that only invokes the provided callback at most once
 * every `waitMs` milliseconds.
 *
 * @template T - The type of the callback function.
 * @param callback - The function to throttle.
 * @param waitMs - The number of milliseconds to throttle invocations.
 * @param options - Optional configuration:
 *  - `leading` (default: true): If true, the callback will be invoked on the leading edge of the timeout.
 *  - `trailing` (default: true): If true, the callback will be invoked on the trailing edge of the timeout.
 * @returns A throttled version of the callback with a `cancel` method to clear the timer.
 */
export declare function throttle<T extends (...args: Parameters<T>) => void>(callback: T, waitMs: number, options?: ThrottleOptions): ((...args: Parameters<T>) => void) & {
    cancel(): void;
};
export {};
