'use strict';

var middlewareHostHeader = require('@aws-sdk/middleware-host-header');
var middlewareLogger = require('@aws-sdk/middleware-logger');
var middlewareRecursionDetection = require('@aws-sdk/middleware-recursion-detection');
var middlewareUserAgent = require('@aws-sdk/middleware-user-agent');
var configResolver = require('@smithy/config-resolver');
var core = require('@smithy/core');
var schema = require('@smithy/core/schema');
var middlewareContentLength = require('@smithy/middleware-content-length');
var middlewareEndpoint = require('@smithy/middleware-endpoint');
var middlewareRetry = require('@smithy/middleware-retry');
var smithyClient = require('@smithy/smithy-client');
var httpAuthSchemeProvider = require('./auth/httpAuthSchemeProvider');
var runtimeConfig = require('./runtimeConfig');
var regionConfigResolver = require('@aws-sdk/region-config-resolver');
var protocolHttp = require('@smithy/protocol-http');

const resolveClientEndpointParameters = (options) => {
    return Object.assign(options, {
        useDualstackEndpoint: options.useDualstackEndpoint ?? false,
        useFipsEndpoint: options.useFipsEndpoint ?? false,
        defaultSigningName: "ses",
    });
};
const commonParams = {
    UseFIPS: { type: "builtInParams", name: "useFipsEndpoint" },
    Endpoint: { type: "builtInParams", name: "endpoint" },
    Region: { type: "builtInParams", name: "region" },
    UseDualStack: { type: "builtInParams", name: "useDualstackEndpoint" },
};

const getHttpAuthExtensionConfiguration = (runtimeConfig) => {
    const _httpAuthSchemes = runtimeConfig.httpAuthSchemes;
    let _httpAuthSchemeProvider = runtimeConfig.httpAuthSchemeProvider;
    let _credentials = runtimeConfig.credentials;
    return {
        setHttpAuthScheme(httpAuthScheme) {
            const index = _httpAuthSchemes.findIndex((scheme) => scheme.schemeId === httpAuthScheme.schemeId);
            if (index === -1) {
                _httpAuthSchemes.push(httpAuthScheme);
            }
            else {
                _httpAuthSchemes.splice(index, 1, httpAuthScheme);
            }
        },
        httpAuthSchemes() {
            return _httpAuthSchemes;
        },
        setHttpAuthSchemeProvider(httpAuthSchemeProvider) {
            _httpAuthSchemeProvider = httpAuthSchemeProvider;
        },
        httpAuthSchemeProvider() {
            return _httpAuthSchemeProvider;
        },
        setCredentials(credentials) {
            _credentials = credentials;
        },
        credentials() {
            return _credentials;
        },
    };
};
const resolveHttpAuthRuntimeConfig = (config) => {
    return {
        httpAuthSchemes: config.httpAuthSchemes(),
        httpAuthSchemeProvider: config.httpAuthSchemeProvider(),
        credentials: config.credentials(),
    };
};

const resolveRuntimeExtensions = (runtimeConfig, extensions) => {
    const extensionConfiguration = Object.assign(regionConfigResolver.getAwsRegionExtensionConfiguration(runtimeConfig), smithyClient.getDefaultExtensionConfiguration(runtimeConfig), protocolHttp.getHttpHandlerExtensionConfiguration(runtimeConfig), getHttpAuthExtensionConfiguration(runtimeConfig));
    extensions.forEach((extension) => extension.configure(extensionConfiguration));
    return Object.assign(runtimeConfig, regionConfigResolver.resolveAwsRegionExtensionConfiguration(extensionConfiguration), smithyClient.resolveDefaultRuntimeConfig(extensionConfiguration), protocolHttp.resolveHttpHandlerRuntimeConfig(extensionConfiguration), resolveHttpAuthRuntimeConfig(extensionConfiguration));
};

class SESv2Client extends smithyClient.Client {
    config;
    constructor(...[configuration]) {
        const _config_0 = runtimeConfig.getRuntimeConfig(configuration || {});
        super(_config_0);
        this.initConfig = _config_0;
        const _config_1 = resolveClientEndpointParameters(_config_0);
        const _config_2 = middlewareUserAgent.resolveUserAgentConfig(_config_1);
        const _config_3 = middlewareRetry.resolveRetryConfig(_config_2);
        const _config_4 = configResolver.resolveRegionConfig(_config_3);
        const _config_5 = middlewareHostHeader.resolveHostHeaderConfig(_config_4);
        const _config_6 = middlewareEndpoint.resolveEndpointConfig(_config_5);
        const _config_7 = httpAuthSchemeProvider.resolveHttpAuthSchemeConfig(_config_6);
        const _config_8 = resolveRuntimeExtensions(_config_7, configuration?.extensions || []);
        this.config = _config_8;
        this.middlewareStack.use(schema.getSchemaSerdePlugin(this.config));
        this.middlewareStack.use(middlewareUserAgent.getUserAgentPlugin(this.config));
        this.middlewareStack.use(middlewareRetry.getRetryPlugin(this.config));
        this.middlewareStack.use(middlewareContentLength.getContentLengthPlugin(this.config));
        this.middlewareStack.use(middlewareHostHeader.getHostHeaderPlugin(this.config));
        this.middlewareStack.use(middlewareLogger.getLoggerPlugin(this.config));
        this.middlewareStack.use(middlewareRecursionDetection.getRecursionDetectionPlugin(this.config));
        this.middlewareStack.use(core.getHttpAuthSchemeEndpointRuleSetPlugin(this.config, {
            httpAuthSchemeParametersProvider: httpAuthSchemeProvider.defaultSESv2HttpAuthSchemeParametersProvider,
            identityProviderConfigProvider: async (config) => new core.DefaultIdentityProviderConfig({
                "aws.auth#sigv4": config.credentials,
                "aws.auth#sigv4a": config.credentials,
            }),
        }));
        this.middlewareStack.use(core.getHttpSigningPlugin(this.config));
    }
    destroy() {
        super.destroy();
    }
}

class SESv2ServiceException extends smithyClient.ServiceException {
    constructor(options) {
        super(options);
        Object.setPrototypeOf(this, SESv2ServiceException.prototype);
    }
}

class AccountSuspendedException extends SESv2ServiceException {
    name = "AccountSuspendedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "AccountSuspendedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, AccountSuspendedException.prototype);
    }
}
class AlreadyExistsException extends SESv2ServiceException {
    name = "AlreadyExistsException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "AlreadyExistsException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, AlreadyExistsException.prototype);
    }
}
class BadRequestException extends SESv2ServiceException {
    name = "BadRequestException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "BadRequestException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, BadRequestException.prototype);
    }
}
class InternalServiceErrorException extends SESv2ServiceException {
    name = "InternalServiceErrorException";
    $fault = "server";
    constructor(opts) {
        super({
            name: "InternalServiceErrorException",
            $fault: "server",
            ...opts,
        });
        Object.setPrototypeOf(this, InternalServiceErrorException.prototype);
    }
}
class NotFoundException extends SESv2ServiceException {
    name = "NotFoundException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "NotFoundException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, NotFoundException.prototype);
    }
}
class TooManyRequestsException extends SESv2ServiceException {
    name = "TooManyRequestsException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "TooManyRequestsException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, TooManyRequestsException.prototype);
    }
}
class ConcurrentModificationException extends SESv2ServiceException {
    name = "ConcurrentModificationException";
    $fault = "server";
    constructor(opts) {
        super({
            name: "ConcurrentModificationException",
            $fault: "server",
            ...opts,
        });
        Object.setPrototypeOf(this, ConcurrentModificationException.prototype);
    }
}
class ConflictException extends SESv2ServiceException {
    name = "ConflictException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "ConflictException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, ConflictException.prototype);
    }
}
class LimitExceededException extends SESv2ServiceException {
    name = "LimitExceededException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "LimitExceededException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, LimitExceededException.prototype);
    }
}
class MailFromDomainNotVerifiedException extends SESv2ServiceException {
    name = "MailFromDomainNotVerifiedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "MailFromDomainNotVerifiedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, MailFromDomainNotVerifiedException.prototype);
    }
}
class MessageRejected extends SESv2ServiceException {
    name = "MessageRejected";
    $fault = "client";
    constructor(opts) {
        super({
            name: "MessageRejected",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, MessageRejected.prototype);
    }
}
class SendingPausedException extends SESv2ServiceException {
    name = "SendingPausedException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "SendingPausedException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, SendingPausedException.prototype);
    }
}
class InvalidNextTokenException extends SESv2ServiceException {
    name = "InvalidNextTokenException";
    $fault = "client";
    constructor(opts) {
        super({
            name: "InvalidNextTokenException",
            $fault: "client",
            ...opts,
        });
        Object.setPrototypeOf(this, InvalidNextTokenException.prototype);
    }
}

const _A = "Attachment";
const _AA = "ArchiveArn";
const _AAp = "ApplicationArn";
const _ACEA = "AdditionalContactEmailAddress";
const _ACEAd = "AdditionalContactEmailAddresses";
const _AD = "AccountDetails";
const _ADt = "AttributesData";
const _AE = "AdminEmail";
const _AEE = "AlreadyExistsException";
const _AL = "AttachmentList";
const _AO = "ArchivingOptions";
const _AS = "AccountStatus";
const _ASD = "ActiveSubscribedDomains";
const _ASE = "AccountSuspendedException";
const _ASMS = "AwsSesManagedStatus";
const _AT = "AssociatedTimestamp";
const _AWE = "AutoWarmupEnabled";
const _Ag = "Aggregation";
const _At = "Attachments";
const _Att = "Attributes";
const _B = "Body";
const _BA = "BccAddresses";
const _BE = "BlacklistEntry";
const _BEC = "BulkEmailContent";
const _BEE = "BulkEmailEntry";
const _BEEL = "BulkEmailEntryList";
const _BEER = "BulkEmailEntryResult";
const _BEERL = "BulkEmailEntryResultList";
const _BEERu = "BulkEmailEntryResults";
const _BEEu = "BulkEmailEntries";
const _BEl = "BlacklistEntries";
const _BGMD = "BatchGetMetricData";
const _BGMDQ = "BatchGetMetricDataQuery";
const _BGMDQa = "BatchGetMetricDataQueries";
const _BGMDR = "BatchGetMetricDataRequest";
const _BGMDRa = "BatchGetMetricDataResponse";
const _BIN = "BlacklistItemNames";
const _BOMF = "BehaviorOnMxFailure";
const _BR = "BlacklistReport";
const _BRE = "BadRequestException";
const _BST = "BounceSubType";
const _BT = "BounceType";
const _Bo = "Bounce";
const _C = "Complaint";
const _CA = "CcAddresses";
const _CC = "CreateContact";
const _CCL = "CreateContactList";
const _CCLR = "CreateContactListRequest";
const _CCLRr = "CreateContactListResponse";
const _CCR = "CreateContactRequest";
const _CCRr = "CreateContactResponse";
const _CCS = "CreateConfigurationSet";
const _CCSED = "CreateConfigurationSetEventDestination";
const _CCSEDR = "CreateConfigurationSetEventDestinationRequest";
const _CCSEDRr = "CreateConfigurationSetEventDestinationResponse";
const _CCSR = "CreateConfigurationSetRequest";
const _CCSRr = "CreateConfigurationSetResponse";
const _CCVET = "CreateCustomVerificationEmailTemplate";
const _CCVETR = "CreateCustomVerificationEmailTemplateRequest";
const _CCVETRr = "CreateCustomVerificationEmailTemplateResponse";
const _CD = "ContentDisposition";
const _CDIP = "CreateDedicatedIpPool";
const _CDIPR = "CreateDedicatedIpPoolRequest";
const _CDIPRr = "CreateDedicatedIpPoolResponse";
const _CDTR = "CreateDeliverabilityTestReport";
const _CDTRR = "CreateDeliverabilityTestReportRequest";
const _CDTRRr = "CreateDeliverabilityTestReportResponse";
const _CDo = "ContentDescription";
const _CDr = "CreateDate";
const _CE = "ConflictException";
const _CEI = "CreateEmailIdentity";
const _CEIP = "CreateEmailIdentityPolicy";
const _CEIPR = "CreateEmailIdentityPolicyRequest";
const _CEIPRr = "CreateEmailIdentityPolicyResponse";
const _CEIR = "CreateEmailIdentityRequest";
const _CEIRr = "CreateEmailIdentityResponse";
const _CEJ = "CancelExportJob";
const _CEJR = "CancelExportJobRequest";
const _CEJRa = "CancelExportJobResponse";
const _CEJRr = "CreateExportJobRequest";
const _CEJRre = "CreateExportJobResponse";
const _CEJr = "CreateExportJob";
const _CET = "CreateEmailTemplate";
const _CETR = "CreateEmailTemplateRequest";
const _CETRr = "CreateEmailTemplateResponse";
const _CFT = "ComplaintFeedbackType";
const _CI = "ContentId";
const _CIJ = "CreateImportJob";
const _CIJR = "CreateImportJobRequest";
const _CIJRr = "CreateImportJobResponse";
const _CIa = "CampaignId";
const _CIas = "CaseId";
const _CL = "ContactLanguage";
const _CLD = "ContactListDestination";
const _CLIA = "ContactListImportAction";
const _CLN = "ContactListName";
const _CLo = "ContactList";
const _CLon = "ContactLists";
const _CME = "ConcurrentModificationException";
const _CMRE = "CreateMultiRegionEndpoint";
const _CMRER = "CreateMultiRegionEndpointRequest";
const _CMRERr = "CreateMultiRegionEndpointResponse";
const _CMS = "CustomerManagedStatus";
const _CRD = "CustomRedirectDomain";
const _CS = "ConfigurationSets";
const _CSKL = "CurrentSigningKeyLength";
const _CSN = "ConfigurationSetName";
const _CST = "ComplaintSubType";
const _CT = "ContentType";
const _CTE = "ContentTransferEncoding";
const _CTEo = "ConditionThresholdEnabled";
const _CTR = "CreateTenantRequest";
const _CTRA = "CreateTenantResourceAssociation";
const _CTRAR = "CreateTenantResourceAssociationRequest";
const _CTRARr = "CreateTenantResourceAssociationResponse";
const _CTRr = "CreateTenantResponse";
const _CTo = "CompletedTimestamp";
const _CTon = "ConditionThreshold";
const _CTr = "CreatedTimestamp";
const _CTre = "CreateTenant";
const _CV = "ConfidenceVerdict";
const _CVET = "CustomVerificationEmailTemplates";
const _CVETL = "CustomVerificationEmailTemplatesList";
const _CVETM = "CustomVerificationEmailTemplateMetadata";
const _CVT = "ConfidenceVerdictThreshold";
const _CWD = "CloudWatchDestination";
const _CWDC = "CloudWatchDimensionConfiguration";
const _CWDCl = "CloudWatchDimensionConfigurations";
const _Ca = "Cause";
const _Ch = "Charset";
const _Co = "Contact";
const _Cod = "Code";
const _Con = "Content";
const _Cont = "Contacts";
const _D = "Dimensions";
const _DA = "DkimAttributes";
const _DAa = "DashboardAttributes";
const _DC = "DiagnosticCode";
const _DCL = "DeleteContactList";
const _DCLR = "DeleteContactListRequest";
const _DCLRe = "DeleteContactListResponse";
const _DCR = "DeleteContactRequest";
const _DCRe = "DeleteContactResponse";
const _DCS = "DeleteConfigurationSet";
const _DCSED = "DeleteConfigurationSetEventDestination";
const _DCSEDR = "DeleteConfigurationSetEventDestinationRequest";
const _DCSEDRe = "DeleteConfigurationSetEventDestinationResponse";
const _DCSR = "DeleteConfigurationSetRequest";
const _DCSRe = "DeleteConfigurationSetResponse";
const _DCVET = "DeleteCustomVerificationEmailTemplate";
const _DCVETR = "DeleteCustomVerificationEmailTemplateRequest";
const _DCVETRe = "DeleteCustomVerificationEmailTemplateResponse";
const _DCe = "DefaultContent";
const _DCel = "DeleteContact";
const _DCi = "DimensionConfigurations";
const _DDC = "DomainDeliverabilityCampaign";
const _DDCL = "DomainDeliverabilityCampaignList";
const _DDCo = "DomainDeliverabilityCampaigns";
const _DDIP = "DeleteDedicatedIpPool";
const _DDIPR = "DeleteDedicatedIpPoolRequest";
const _DDIPRe = "DeleteDedicatedIpPoolResponse";
const _DDTO = "DomainDeliverabilityTrackingOption";
const _DDTOo = "DomainDeliverabilityTrackingOptions";
const _DDV = "DefaultDimensionValue";
const _DE = "DashboardEnabled";
const _DEI = "DeleteEmailIdentity";
const _DEIP = "DeleteEmailIdentityPolicy";
const _DEIPR = "DeleteEmailIdentityPolicyRequest";
const _DEIPRe = "DeleteEmailIdentityPolicyResponse";
const _DEIR = "DeleteEmailIdentityRequest";
const _DEIRe = "DeleteEmailIdentityResponse";
const _DET = "DefaultEmailTags";
const _DETR = "DeleteEmailTemplateRequest";
const _DETRe = "DeleteEmailTemplateResponse";
const _DETe = "DeleteEmailTemplate";
const _DF = "DataFormat";
const _DI = "DedicatedIp";
const _DIAWE = "DedicatedIpAutoWarmupEnabled";
const _DIL = "DedicatedIpList";
const _DIP = "DomainIspPlacements";
const _DIPe = "DedicatedIpPool";
const _DIPed = "DedicatedIpPools";
const _DIPo = "DomainIspPlacement";
const _DIe = "DedicatedIps";
const _DMRE = "DeleteMultiRegionEndpoint";
const _DMRER = "DeleteMultiRegionEndpointRequest";
const _DMRERe = "DeleteMultiRegionEndpointResponse";
const _DN = "DimensionName";
const _DNi = "DisplayName";
const _DO = "DeliveryOptions";
const _DOa = "DashboardOptions";
const _DP = "DkimPercentage";
const _DPN = "DestinationPoolName";
const _DR = "DeleteRate";
const _DS = "DkimStatus";
const _DSA = "DkimSigningAttributes";
const _DSAO = "DomainSigningAttributesOrigin";
const _DSAe = "DeliveryStreamArn";
const _DSD = "DeleteSuppressedDestination";
const _DSDR = "DeleteSuppressedDestinationRequest";
const _DSDRe = "DeleteSuppressedDestinationResponse";
const _DSPK = "DomainSigningPrivateKey";
const _DSS = "DomainSigningSelector";
const _DSSe = "DefaultSubscriptionStatus";
const _DT = "DkimTokens";
const _DTR = "DeleteTenantRequest";
const _DTRA = "DeleteTenantResourceAssociation";
const _DTRAR = "DeleteTenantResourceAssociationRequest";
const _DTRARe = "DeleteTenantResourceAssociationResponse";
const _DTRe = "DeleteTenantResponse";
const _DTRel = "DeliverabilityTestReport";
const _DTReli = "DeliverabilityTestReports";
const _DTS = "DeliverabilityTestStatus";
const _DTe = "DeleteTenant";
const _DV = "DailyVolume";
const _DVS = "DimensionValueSource";
const _DVa = "DailyVolumes";
const _Da = "Data";
const _De = "Description";
const _Des = "Destination";
const _Det = "Details";
const _Do = "Domain";
const _E = "Errors";
const _EA = "EmailAddress";
const _EAFL = "EmailAddressFilterList";
const _EAIME = "EmailAddressInsightsMailboxEvaluations";
const _EAIV = "EmailAddressInsightsVerdict";
const _EBA = "EventBusArn";
const _EBD = "EventBridgeDestination";
const _EC = "EmailContent";
const _ED = "EndDate";
const _EDD = "EventDestinationDefinition";
const _EDN = "EventDestinationName";
const _EDS = "ExportDataSource";
const _EDv = "EventDestination";
const _EDve = "EventDetails";
const _EDven = "EventDestinations";
const _EDx = "ExportDestination";
const _EDxp = "ExportDimensions";
const _EFE = "EmailForwardingEnabled";
const _EI = "EmailIdentity";
const _EIL = "EmailInsightsList";
const _EIm = "EmailInsights";
const _EIma = "EmailIdentities";
const _EIn = "EndpointId";
const _EJ = "ExportJobs";
const _EJS = "ExportJobSummary";
const _EJSL = "ExportJobSummaryList";
const _EM = "EngagementMetrics";
const _EMr = "ErrorMessage";
const _EMx = "ExportMetric";
const _EMxp = "ExportMetrics";
const _EN = "EndpointName";
const _ERC = "ExportedRecordsCount";
const _ES = "EmailSubject";
const _ESFL = "EmailSubjectFilterList";
const _EST = "ExportSourceType";
const _ESn = "EnforcementStatus";
const _ESx = "ExportStatistics";
const _ET = "EmailTags";
const _ETC = "EmailTemplateContent";
const _ETM = "EmailTemplateMetadata";
const _ETML = "EmailTemplateMetadataList";
const _ETr = "ErrorType";
const _En = "Enabled";
const _Er = "Error";
const _Es = "Esps";
const _Ev = "Events";
const _Eva = "Evaluations";
const _Ex = "Exclude";
const _F = "Filter";
const _FA = "FromAddress";
const _FEA = "FromEmailAddress";
const _FEAIA = "FromEmailAddressIdentityArn";
const _FFEA = "FeedbackForwardingEmailAddress";
const _FFEAIA = "FeedbackForwardingEmailAddressIdentityArn";
const _FFS = "FeedbackForwardingStatus";
const _FI = "FailureInfo";
const _FIe = "FeedbackId";
const _FN = "FileName";
const _FRC = "FailedRecordsCount";
const _FRSU = "FailedRecordsS3Url";
const _FRURL = "FailureRedirectionURL";
const _FS = "FilteredStatus";
const _FSDT = "FirstSeenDateTime";
const _G = "Global";
const _GA = "GuardianAttributes";
const _GAR = "GetAccountRequest";
const _GARe = "GetAccountResponse";
const _GAe = "GetAccount";
const _GBR = "GetBlacklistReports";
const _GBRR = "GetBlacklistReportsRequest";
const _GBRRe = "GetBlacklistReportsResponse";
const _GC = "GetContact";
const _GCL = "GetContactList";
const _GCLR = "GetContactListRequest";
const _GCLRe = "GetContactListResponse";
const _GCR = "GetContactRequest";
const _GCRe = "GetContactResponse";
const _GCS = "GetConfigurationSet";
const _GCSED = "GetConfigurationSetEventDestinations";
const _GCSEDR = "GetConfigurationSetEventDestinationsRequest";
const _GCSEDRe = "GetConfigurationSetEventDestinationsResponse";
const _GCSR = "GetConfigurationSetRequest";
const _GCSRe = "GetConfigurationSetResponse";
const _GCVET = "GetCustomVerificationEmailTemplate";
const _GCVETR = "GetCustomVerificationEmailTemplateRequest";
const _GCVETRe = "GetCustomVerificationEmailTemplateResponse";
const _GDDC = "GetDomainDeliverabilityCampaign";
const _GDDCR = "GetDomainDeliverabilityCampaignRequest";
const _GDDCRe = "GetDomainDeliverabilityCampaignResponse";
const _GDDO = "GetDeliverabilityDashboardOptions";
const _GDDOR = "GetDeliverabilityDashboardOptionsRequest";
const _GDDORe = "GetDeliverabilityDashboardOptionsResponse";
const _GDI = "GetDedicatedIp";
const _GDIP = "GetDedicatedIpPool";
const _GDIPR = "GetDedicatedIpPoolRequest";
const _GDIPRe = "GetDedicatedIpPoolResponse";
const _GDIR = "GetDedicatedIpRequest";
const _GDIRe = "GetDedicatedIpResponse";
const _GDIRet = "GetDedicatedIpsRequest";
const _GDIRete = "GetDedicatedIpsResponse";
const _GDIe = "GetDedicatedIps";
const _GDSR = "GetDomainStatisticsReport";
const _GDSRR = "GetDomainStatisticsReportRequest";
const _GDSRRe = "GetDomainStatisticsReportResponse";
const _GDTR = "GetDeliverabilityTestReport";
const _GDTRR = "GetDeliverabilityTestReportRequest";
const _GDTRRe = "GetDeliverabilityTestReportResponse";
const _GEAI = "GetEmailAddressInsights";
const _GEAIR = "GetEmailAddressInsightsRequest";
const _GEAIRe = "GetEmailAddressInsightsResponse";
const _GEI = "GetEmailIdentity";
const _GEIP = "GetEmailIdentityPolicies";
const _GEIPR = "GetEmailIdentityPoliciesRequest";
const _GEIPRe = "GetEmailIdentityPoliciesResponse";
const _GEIR = "GetEmailIdentityRequest";
const _GEIRe = "GetEmailIdentityResponse";
const _GEJ = "GetExportJob";
const _GEJR = "GetExportJobRequest";
const _GEJRe = "GetExportJobResponse";
const _GET = "GetEmailTemplate";
const _GETR = "GetEmailTemplateRequest";
const _GETRe = "GetEmailTemplateResponse";
const _GIJ = "GetImportJob";
const _GIJR = "GetImportJobRequest";
const _GIJRe = "GetImportJobResponse";
const _GMI = "GetMessageInsights";
const _GMIR = "GetMessageInsightsRequest";
const _GMIRe = "GetMessageInsightsResponse";
const _GMRE = "GetMultiRegionEndpoint";
const _GMRER = "GetMultiRegionEndpointRequest";
const _GMRERe = "GetMultiRegionEndpointResponse";
const _GO = "GuardianOptions";
const _GRE = "GetReputationEntity";
const _GRER = "GetReputationEntityRequest";
const _GRERe = "GetReputationEntityResponse";
const _GSD = "GetSuppressedDestination";
const _GSDR = "GetSuppressedDestinationRequest";
const _GSDRe = "GetSuppressedDestinationResponse";
const _GT = "GetTenant";
const _GTR = "GetTenantRequest";
const _GTRe = "GetTenantResponse";
const _H = "Html";
const _HP = "HttpsPolicy";
const _HVDR = "HasValidDnsRecords";
const _HVS = "HasValidSyntax";
const _He = "Headers";
const _I = "Id";
const _IC = "InboxCount";
const _ID = "ImportDestination";
const _IDS = "ImportDataSource";
const _IDT = "ImportDestinationType";
const _IDs = "IsDisposable";
const _IE = "InsightsEvent";
const _IEA = "InsightsEmailAddress";
const _IEn = "InsightsEvents";
const _II = "IdentityInfo";
const _IIL = "IdentityInfoList";
const _IJ = "ImportJobs";
const _IJS = "ImportJobSummary";
const _IJSL = "ImportJobSummaryList";
const _IN = "IspName";
const _INTE = "InvalidNextTokenException";
const _INd = "IdentityName";
const _IP = "InboxPercentage";
const _IPTO = "InboxPlacementTrackingOption";
const _IPs = "IspPlacements";
const _IPsp = "IspPlacement";
const _IRA = "IsRoleAddress";
const _IRAa = "IamRoleArn";
const _IRC = "InboxRawCount";
const _IRI = "IsRandomInput";
const _ISEE = "InternalServiceErrorException";
const _IT = "IdentityType";
const _IU = "ImageUrl";
const _IV = "IsValid";
const _Im = "Impact";
const _In = "Insights";
const _Inc = "Include";
const _Ip = "Ip";
const _Is = "Isp";
const _JI = "JobId";
const _JS = "JobStatus";
const _K = "Key";
const _KFD = "KinesisFirehoseDestination";
const _LC = "ListContacts";
const _LCF = "ListContactsFilter";
const _LCL = "ListContactLists";
const _LCLR = "ListContactListsRequest";
const _LCLRi = "ListContactListsResponse";
const _LCR = "ListContactsRequest";
const _LCRi = "ListContactsResponse";
const _LCS = "ListConfigurationSets";
const _LCSR = "ListConfigurationSetsRequest";
const _LCSRi = "ListConfigurationSetsResponse";
const _LCT = "LastCheckedTimestamp";
const _LCVET = "ListCustomVerificationEmailTemplates";
const _LCVETR = "ListCustomVerificationEmailTemplatesRequest";
const _LCVETRi = "ListCustomVerificationEmailTemplatesResponse";
const _LDDC = "ListDomainDeliverabilityCampaigns";
const _LDDCR = "ListDomainDeliverabilityCampaignsRequest";
const _LDDCRi = "ListDomainDeliverabilityCampaignsResponse";
const _LDE = "LastDeliveryEvent";
const _LDIP = "ListDedicatedIpPools";
const _LDIPR = "ListDedicatedIpPoolsRequest";
const _LDIPRi = "ListDedicatedIpPoolsResponse";
const _LDTR = "ListDeliverabilityTestReports";
const _LDTRR = "ListDeliverabilityTestReportsRequest";
const _LDTRRi = "ListDeliverabilityTestReportsResponse";
const _LEE = "LimitExceededException";
const _LEEa = "LastEngagementEvent";
const _LEI = "ListEmailIdentities";
const _LEIR = "ListEmailIdentitiesRequest";
const _LEIRi = "ListEmailIdentitiesResponse";
const _LEJ = "ListExportJobs";
const _LEJR = "ListExportJobsRequest";
const _LEJRi = "ListExportJobsResponse";
const _LET = "ListEmailTemplates";
const _LETR = "ListEmailTemplatesRequest";
const _LETRi = "ListEmailTemplatesResponse";
const _LFS = "LastFreshStart";
const _LIJ = "ListImportJobs";
const _LIJR = "ListImportJobsRequest";
const _LIJRi = "ListImportJobsResponse";
const _LKGT = "LastKeyGenerationTimestamp";
const _LMO = "ListManagementOptions";
const _LMRE = "ListMultiRegionEndpoints";
const _LMRER = "ListMultiRegionEndpointsRequest";
const _LMRERi = "ListMultiRegionEndpointsResponse";
const _LOC = "ListOfContacts";
const _LOCL = "ListOfContactLists";
const _LR = "ListRecommendations";
const _LRE = "ListReputationEntities";
const _LRER = "ListReputationEntitiesRequest";
const _LRERi = "ListReputationEntitiesResponse";
const _LRR = "ListRecommendationsRequest";
const _LRRi = "ListRecommendationsResponse";
const _LRT = "ListResourceTenants";
const _LRTR = "ListResourceTenantsRequest";
const _LRTRi = "ListResourceTenantsResponse";
const _LSD = "ListSuppressedDestinations";
const _LSDR = "ListSuppressedDestinationsRequest";
const _LSDRi = "ListSuppressedDestinationsResponse";
const _LSDT = "LastSeenDateTime";
const _LST = "LastSuccessTimestamp";
const _LT = "ListingTime";
const _LTFR = "ListTagsForResource";
const _LTFRR = "ListTagsForResourceRequest";
const _LTFRRi = "ListTagsForResourceResponse";
const _LTR = "ListTenantsRequest";
const _LTRR = "ListTenantResourcesRequest";
const _LTRRi = "ListTenantResourcesResponse";
const _LTRi = "ListTenantsResponse";
const _LTRis = "ListTenantResources";
const _LTi = "ListTenants";
const _LUT = "LastUpdatedTimestamp";
const _LUTa = "LastUpdateTime";
const _M = "Metric";
const _MDE = "MetricDataError";
const _MDEL = "MetricDataErrorList";
const _MDR = "MetricDataResult";
const _MDRL = "MetricDataResultList";
const _MDS = "MaxDeliverySeconds";
const _MDSe = "MetricsDataSource";
const _ME = "MailboxExists";
const _MET = "MatchingEventTypes";
const _MFA = "MailFromAttributes";
const _MFD = "MailFromDomain";
const _MFDNVE = "MailFromDomainNotVerifiedException";
const _MFDS = "MailFromDomainStatus";
const _MH = "MessageHeader";
const _MHL = "MessageHeaderList";
const _MHS = "Max24HourSend";
const _MI = "MessageId";
const _MIDS = "MessageInsightsDataSource";
const _MIF = "MessageInsightsFilters";
const _MP = "MissingPercentage";
const _MR = "MaxResults";
const _MRE = "MultiRegionEndpoints";
const _MREu = "MultiRegionEndpoint";
const _MRe = "MessageRejected";
const _MSR = "MaxSendRate";
const _MT = "MailType";
const _MTL = "MessageTagList";
const _MTe = "MessageTag";
const _MV = "MailboxValidation";
const _Me = "Message";
const _Met = "Metrics";
const _N = "Namespace";
const _NFE = "NotFoundException";
const _NSKL = "NextSigningKeyLength";
const _NT = "NextToken";
const _Na = "Name";
const _OCT = "OverallConfidenceThreshold";
const _OP = "OverallPlacement";
const _OSD = "OptimizedSharedDelivery";
const _OV = "OverallVolume";
const _P = "Policy";
const _PAD = "PutAccountDetails";
const _PADIWA = "PutAccountDedicatedIpWarmupAttributes";
const _PADIWAR = "PutAccountDedicatedIpWarmupAttributesRequest";
const _PADIWARu = "PutAccountDedicatedIpWarmupAttributesResponse";
const _PADR = "PutAccountDetailsRequest";
const _PADRu = "PutAccountDetailsResponse";
const _PAE = "ProductionAccessEnabled";
const _PASA = "PutAccountSendingAttributes";
const _PASAR = "PutAccountSendingAttributesRequest";
const _PASARu = "PutAccountSendingAttributesResponse";
const _PASARut = "PutAccountSuppressionAttributesRequest";
const _PASARutc = "PutAccountSuppressionAttributesResponse";
const _PASAu = "PutAccountSuppressionAttributes";
const _PAVA = "PutAccountVdmAttributes";
const _PAVAR = "PutAccountVdmAttributesRequest";
const _PAVARu = "PutAccountVdmAttributesResponse";
const _PCSAO = "PutConfigurationSetArchivingOptions";
const _PCSAOR = "PutConfigurationSetArchivingOptionsRequest";
const _PCSAORu = "PutConfigurationSetArchivingOptionsResponse";
const _PCSDO = "PutConfigurationSetDeliveryOptions";
const _PCSDOR = "PutConfigurationSetDeliveryOptionsRequest";
const _PCSDORu = "PutConfigurationSetDeliveryOptionsResponse";
const _PCSRO = "PutConfigurationSetReputationOptions";
const _PCSROR = "PutConfigurationSetReputationOptionsRequest";
const _PCSRORu = "PutConfigurationSetReputationOptionsResponse";
const _PCSSO = "PutConfigurationSetSendingOptions";
const _PCSSOR = "PutConfigurationSetSendingOptionsRequest";
const _PCSSORu = "PutConfigurationSetSendingOptionsResponse";
const _PCSSORut = "PutConfigurationSetSuppressionOptionsRequest";
const _PCSSORuto = "PutConfigurationSetSuppressionOptionsResponse";
const _PCSSOu = "PutConfigurationSetSuppressionOptions";
const _PCSTO = "PutConfigurationSetTrackingOptions";
const _PCSTOR = "PutConfigurationSetTrackingOptionsRequest";
const _PCSTORu = "PutConfigurationSetTrackingOptionsResponse";
const _PCSVO = "PutConfigurationSetVdmOptions";
const _PCSVOR = "PutConfigurationSetVdmOptionsRequest";
const _PCSVORu = "PutConfigurationSetVdmOptionsResponse";
const _PD = "PinpointDestination";
const _PDDO = "PutDeliverabilityDashboardOption";
const _PDDOR = "PutDeliverabilityDashboardOptionRequest";
const _PDDORu = "PutDeliverabilityDashboardOptionResponse";
const _PDIIP = "PutDedicatedIpInPool";
const _PDIIPR = "PutDedicatedIpInPoolRequest";
const _PDIIPRu = "PutDedicatedIpInPoolResponse";
const _PDIPSA = "PutDedicatedIpPoolScalingAttributes";
const _PDIPSAR = "PutDedicatedIpPoolScalingAttributesRequest";
const _PDIPSARu = "PutDedicatedIpPoolScalingAttributesResponse";
const _PDIWA = "PutDedicatedIpWarmupAttributes";
const _PDIWAR = "PutDedicatedIpWarmupAttributesRequest";
const _PDIWARu = "PutDedicatedIpWarmupAttributesResponse";
const _PEICSA = "PutEmailIdentityConfigurationSetAttributes";
const _PEICSAR = "PutEmailIdentityConfigurationSetAttributesRequest";
const _PEICSARu = "PutEmailIdentityConfigurationSetAttributesResponse";
const _PEIDA = "PutEmailIdentityDkimAttributes";
const _PEIDAR = "PutEmailIdentityDkimAttributesRequest";
const _PEIDARu = "PutEmailIdentityDkimAttributesResponse";
const _PEIDSA = "PutEmailIdentityDkimSigningAttributes";
const _PEIDSAR = "PutEmailIdentityDkimSigningAttributesRequest";
const _PEIDSARu = "PutEmailIdentityDkimSigningAttributesResponse";
const _PEIFA = "PutEmailIdentityFeedbackAttributes";
const _PEIFAR = "PutEmailIdentityFeedbackAttributesRequest";
const _PEIFARu = "PutEmailIdentityFeedbackAttributesResponse";
const _PEIMFA = "PutEmailIdentityMailFromAttributes";
const _PEIMFAR = "PutEmailIdentityMailFromAttributesRequest";
const _PEIMFARu = "PutEmailIdentityMailFromAttributesResponse";
const _PESD = "PendingExpirationSubscribedDomains";
const _PI = "ProjectedInbox";
const _PK = "PrivateKey";
const _PN = "PoolName";
const _PNS = "PrimaryNameServer";
const _PNo = "PolicyName";
const _PRC = "ProcessedRecordsCount";
const _PS = "PageSize";
const _PSD = "PutSuppressedDestination";
const _PSDR = "PutSuppressedDestinationRequest";
const _PSDRu = "PutSuppressedDestinationResponse";
const _PSl = "PlacementStatistics";
const _PSr = "ProjectedSpam";
const _PV = "ProjectedVolume";
const _Po = "Policies";
const _Q = "Queries";
const _R = "Results";
const _RA = "ResourceArn";
const _RC = "RawContent";
const _RD = "ReviewDetails";
const _RDR = "ReadDeleteRate";
const _RDo = "RoutesDetails";
const _RDou = "RouteDetails";
const _RE = "ReputationEntity";
const _REC = "ReplacementEmailContent";
const _REL = "ReputationEntitiesList";
const _REP = "ReputationEntityPolicy";
const _RER = "ReputationEntityReference";
const _RET = "ReputationEntityType";
const _REe = "ReputationEntities";
const _RH = "ReplacementHeaders";
const _RI = "ReportId";
const _RIe = "ReputationImpact";
const _RL = "RecommendationsList";
const _RM = "RawMessage";
const _RME = "ReputationMetricsEnabled";
const _RMP = "ReputationManagementPolicy";
const _RN = "RblName";
const _RNe = "ReportName";
const _RO = "ReputationOptions";
const _RR = "ReadRate";
const _RRP = "ReadRatePercent";
const _RT = "ReplacementTags";
const _RTA = "ReplyToAddresses";
const _RTD = "ReplacementTemplateData";
const _RTM = "ResourceTenantMetadata";
const _RTML = "ResourceTenantMetadataList";
const _RTe = "ResourceTenants";
const _RTen = "RenderedTemplate";
const _RTep = "ReplacementTemplate";
const _RTes = "ResourceType";
const _Ra = "Raw";
const _Re = "Recommendations";
const _Rea = "Reasons";
const _Reas = "Reason";
const _Rec = "Recommendation";
const _Reg = "Regions";
const _Regi = "Region";
const _Ro = "Routes";
const _Rou = "Route";
const _S = "Status";
const _SA = "SuppressionAttributes";
const _SAO = "SigningAttributesOrigin";
const _SAi = "SigningAttributes";
const _SBE = "SendBulkEmail";
const _SBER = "SendBulkEmailRequest";
const _SBERe = "SendBulkEmailResponse";
const _SC = "SpamCount";
const _SCT = "SuppressionConditionThreshold";
const _SCTu = "SuppressionConfidenceThreshold";
const _SCVE = "SendCustomVerificationEmail";
const _SCVER = "SendCustomVerificationEmailRequest";
const _SCVERe = "SendCustomVerificationEmailResponse";
const _SD = "StartDate";
const _SDA = "SuppressedDestinationAttributes";
const _SDS = "SuppressedDestinationSummaries";
const _SDSu = "SuppressedDestinationSummary";
const _SDn = "SnsDestination";
const _SDu = "SuppressedDestination";
const _SDub = "SubscribedDomain";
const _SDubs = "SubscribedDomains";
const _SE = "SigningEnabled";
const _SED = "SubscriptionExpiryDate";
const _SER = "SendEmailRequest";
const _SERe = "SendEmailResponse";
const _SEe = "SendingEnabled";
const _SEen = "SendEmail";
const _SHZ = "SigningHostedZone";
const _SI = "SendingIps";
const _SLD = "SuppressionListDestination";
const _SLH = "SentLast24Hours";
const _SLIA = "SuppressionListImportAction";
const _SM = "ScalingMode";
const _SN = "SerialNumber";
const _SO = "SendingOptions";
const _SOAR = "SOARecord";
const _SOu = "SuppressionOptions";
const _SP = "SpamPercentage";
const _SPE = "SendingPausedException";
const _SPN = "SendingPoolName";
const _SPp = "SpfPercentage";
const _SQ = "SendQuota";
const _SR = "SuppressedReasons";
const _SRC = "SpamRawCount";
const _SRURL = "SuccessRedirectionURL";
const _SRt = "StatusRecord";
const _SS = "SendingStatus";
const _SSA = "SendingStatusAggregate";
const _SSD = "SubscriptionStartDate";
const _SSu = "SubscriptionStatus";
const _SU = "S3Url";
const _SVA = "SuppressionValidationAttributes";
const _SVO = "SuppressionValidationOptions";
const _Si = "Simple";
const _St = "Statistics";
const _Su = "Subject";
const _T = "Text";
const _TA = "TenantArn";
const _TAe = "TemplateArn";
const _TAo = "ToAddresses";
const _TAop = "TopicArn";
const _TC = "TemplateContent";
const _TD = "TemplateData";
const _TDP = "TopicDefaultPreferences";
const _TF = "TopicFilter";
const _TI = "TenantId";
const _TIL = "TenantInfoList";
const _TIe = "TenantInfo";
const _TIr = "TrackedIsps";
const _TK = "TagKeys";
const _TL = "TagList";
const _TM = "TemplatesMetadata";
const _TMRE = "TooManyRequestsException";
const _TN = "TemplateName";
const _TNe = "TenantName";
const _TNo = "TopicName";
const _TO = "TrackingOptions";
const _TP = "TopicPreferences";
const _TPL = "TopicPreferenceList";
const _TPl = "TlsPolicy";
const _TPo = "TopicPreference";
const _TR = "TenantResources";
const _TRET = "TestRenderEmailTemplate";
const _TRETR = "TestRenderEmailTemplateRequest";
const _TRETRe = "TestRenderEmailTemplateResponse";
const _TRL = "TenantResourceList";
const _TRR = "TagResourceRequest";
const _TRRa = "TagResourceResponse";
const _TRa = "TagResource";
const _TRe = "TenantResource";
const _TS = "TemplateSubject";
const _Ta = "Tags";
const _Tag = "Tag";
const _Te = "Template";
const _Ten = "Tenant";
const _Tena = "Tenants";
const _Ti = "Timestamp";
const _Tim = "Timestamps";
const _To = "Topics";
const _Tok = "Tokens";
const _Top = "Topic";
const _Ty = "Type";
const _UA = "UnsubscribeAll";
const _UC = "UpdateContact";
const _UCD = "UseCaseDescription";
const _UCL = "UpdateContactList";
const _UCLR = "UpdateContactListRequest";
const _UCLRp = "UpdateContactListResponse";
const _UCR = "UpdateContactRequest";
const _UCRp = "UpdateContactResponse";
const _UCSED = "UpdateConfigurationSetEventDestination";
const _UCSEDR = "UpdateConfigurationSetEventDestinationRequest";
const _UCSEDRp = "UpdateConfigurationSetEventDestinationResponse";
const _UCVET = "UpdateCustomVerificationEmailTemplate";
const _UCVETR = "UpdateCustomVerificationEmailTemplateRequest";
const _UCVETRp = "UpdateCustomVerificationEmailTemplateResponse";
const _UDIPU = "UseDefaultIfPreferenceUnavailable";
const _UEIP = "UpdateEmailIdentityPolicy";
const _UEIPR = "UpdateEmailIdentityPolicyRequest";
const _UEIPRp = "UpdateEmailIdentityPolicyResponse";
const _UET = "UpdateEmailTemplate";
const _UETR = "UpdateEmailTemplateRequest";
const _UETRp = "UpdateEmailTemplateResponse";
const _UR = "UntagResource";
const _URECMS = "UpdateReputationEntityCustomerManagedStatus";
const _URECMSR = "UpdateReputationEntityCustomerManagedStatusRequest";
const _URECMSRp = "UpdateReputationEntityCustomerManagedStatusResponse";
const _UREP = "UpdateReputationEntityPolicy";
const _UREPR = "UpdateReputationEntityPolicyRequest";
const _UREPRp = "UpdateReputationEntityPolicyResponse";
const _URR = "UntagResourceRequest";
const _URRn = "UntagResourceResponse";
const _V = "Value";
const _VA = "VdmAttributes";
const _VAa = "ValidationAttributes";
const _VE = "VdmEnabled";
const _VFSS = "VerifiedForSendingStatus";
const _VI = "VerificationInfo";
const _VO = "VdmOptions";
const _VOa = "ValidationOptions";
const _VS = "VolumeStatistics";
const _VSe = "VerificationStatus";
const _Va = "Values";
const _WP = "WarmupPercentage";
const _WS = "WarmupStatus";
const _WURL = "WebsiteURL";
const _c = "client";
const _e = "error";
const _h = "http";
const _hE = "httpError";
const _hQ = "httpQuery";
const _m = "message";
const _s = "server";
const _sm = "smithy.ts.sdk.synthetic.com.amazonaws.sesv2";
const n0 = "com.amazonaws.sesv2";
var AdditionalContactEmailAddress = [0, n0, _ACEA, 8, 0];
var EmailSubject = [0, n0, _ES, 8, 0];
var InsightsEmailAddress = [0, n0, _IEA, 8, 0];
var PrivateKey = [0, n0, _PK, 8, 0];
var UseCaseDescription = [0, n0, _UCD, 8, 0];
var WebsiteURL = [0, n0, _WURL, 8, 0];
var AccountDetails$ = [
    3,
    n0,
    _AD,
    0,
    [_MT, _WURL, _CL, _UCD, _ACEAd, _RD],
    [
        0,
        [() => WebsiteURL, 0],
        0,
        [() => UseCaseDescription, 0],
        [() => AdditionalContactEmailAddresses, 0],
        () => ReviewDetails$,
    ],
];
var AccountSuspendedException$ = [-3, n0, _ASE, { [_e]: _c, [_hE]: 400 }, [_m], [0]];
schema.TypeRegistry.for(n0).registerError(AccountSuspendedException$, AccountSuspendedException);
var AlreadyExistsException$ = [-3, n0, _AEE, { [_e]: _c, [_hE]: 400 }, [_m], [0]];
schema.TypeRegistry.for(n0).registerError(AlreadyExistsException$, AlreadyExistsException);
var ArchivingOptions$ = [3, n0, _AO, 0, [_AA], [0]];
var Attachment$ = [
    3,
    n0,
    _A,
    0,
    [_RC, _CD, _FN, _CDo, _CI, _CTE, _CT],
    [21, 0, 0, 0, 0, 0, 0],
];
var BadRequestException$ = [-3, n0, _BRE, { [_e]: _c, [_hE]: 400 }, [_m], [0]];
schema.TypeRegistry.for(n0).registerError(BadRequestException$, BadRequestException);
var BatchGetMetricDataQuery$ = [
    3,
    n0,
    _BGMDQ,
    0,
    [_I, _N, _M, _D, _SD, _ED],
    [0, 0, 0, 128 | 0, 4, 4],
];
var BatchGetMetricDataRequest$ = [
    3,
    n0,
    _BGMDR,
    0,
    [_Q],
    [() => BatchGetMetricDataQueries],
];
var BatchGetMetricDataResponse$ = [
    3,
    n0,
    _BGMDRa,
    0,
    [_R, _E],
    [() => MetricDataResultList, () => MetricDataErrorList],
];
var BlacklistEntry$ = [3, n0, _BE, 0, [_RN, _LT, _De], [0, 4, 0]];
var Body$ = [3, n0, _B, 0, [_T, _H], [() => Content$, () => Content$]];
var Bounce$ = [3, n0, _Bo, 0, [_BT, _BST, _DC], [0, 0, 0]];
var BulkEmailContent$ = [3, n0, _BEC, 0, [_Te], [() => Template$]];
var BulkEmailEntry$ = [
    3,
    n0,
    _BEE,
    0,
    [_Des, _RT, _REC, _RH],
    [() => Destination$, () => MessageTagList, () => ReplacementEmailContent$, () => MessageHeaderList],
];
var BulkEmailEntryResult$ = [3, n0, _BEER, 0, [_S, _Er, _MI], [0, 0, 0]];
var CancelExportJobRequest$ = [3, n0, _CEJR, 0, [_JI], [[0, 1]]];
var CancelExportJobResponse$ = [3, n0, _CEJRa, 0, [], []];
var CloudWatchDestination$ = [
    3,
    n0,
    _CWD,
    0,
    [_DCi],
    [() => CloudWatchDimensionConfigurations],
];
var CloudWatchDimensionConfiguration$ = [3, n0, _CWDC, 0, [_DN, _DVS, _DDV], [0, 0, 0]];
var Complaint$ = [3, n0, _C, 0, [_CST, _CFT], [0, 0]];
var ConcurrentModificationException$ = [-3, n0, _CME, { [_e]: _s, [_hE]: 500 }, [_m], [0]];
schema.TypeRegistry.for(n0).registerError(ConcurrentModificationException$, ConcurrentModificationException);
var ConflictException$ = [-3, n0, _CE, { [_e]: _c, [_hE]: 409 }, [_m], [0]];
schema.TypeRegistry.for(n0).registerError(ConflictException$, ConflictException);
var Contact$ = [
    3,
    n0,
    _Co,
    0,
    [_EA, _TP, _TDP, _UA, _LUT],
    [0, () => TopicPreferenceList, () => TopicPreferenceList, 2, 4],
];
var ContactList$ = [3, n0, _CLo, 0, [_CLN, _LUT], [0, 4]];
var ContactListDestination$ = [3, n0, _CLD, 0, [_CLN, _CLIA], [0, 0]];
var Content$ = [3, n0, _Con, 0, [_Da, _Ch], [0, 0]];
var CreateConfigurationSetEventDestinationRequest$ = [
    3,
    n0,
    _CCSEDR,
    0,
    [_CSN, _EDN, _EDv],
    [[0, 1], 0, () => EventDestinationDefinition$],
];
var CreateConfigurationSetEventDestinationResponse$ = [3, n0, _CCSEDRr, 0, [], []];
var CreateConfigurationSetRequest$ = [
    3,
    n0,
    _CCSR,
    0,
    [_CSN, _TO, _DO, _RO, _SO, _Ta, _SOu, _VO, _AO],
    [
        0,
        () => TrackingOptions$,
        () => DeliveryOptions$,
        () => ReputationOptions$,
        () => SendingOptions$,
        () => TagList,
        () => SuppressionOptions$,
        () => VdmOptions$,
        () => ArchivingOptions$,
    ],
];
var CreateConfigurationSetResponse$ = [3, n0, _CCSRr, 0, [], []];
var CreateContactListRequest$ = [
    3,
    n0,
    _CCLR,
    0,
    [_CLN, _To, _De, _Ta],
    [0, () => Topics, 0, () => TagList],
];
var CreateContactListResponse$ = [3, n0, _CCLRr, 0, [], []];
var CreateContactRequest$ = [
    3,
    n0,
    _CCR,
    0,
    [_CLN, _EA, _TP, _UA, _ADt],
    [[0, 1], 0, () => TopicPreferenceList, 2, 0],
];
var CreateContactResponse$ = [3, n0, _CCRr, 0, [], []];
var CreateCustomVerificationEmailTemplateRequest$ = [
    3,
    n0,
    _CCVETR,
    0,
    [_TN, _FEA, _TS, _TC, _Ta, _SRURL, _FRURL],
    [0, 0, 0, 0, () => TagList, 0, 0],
];
var CreateCustomVerificationEmailTemplateResponse$ = [3, n0, _CCVETRr, 0, [], []];
var CreateDedicatedIpPoolRequest$ = [
    3,
    n0,
    _CDIPR,
    0,
    [_PN, _Ta, _SM],
    [0, () => TagList, 0],
];
var CreateDedicatedIpPoolResponse$ = [3, n0, _CDIPRr, 0, [], []];
var CreateDeliverabilityTestReportRequest$ = [
    3,
    n0,
    _CDTRR,
    0,
    [_RNe, _FEA, _Con, _Ta],
    [0, 0, () => EmailContent$, () => TagList],
];
var CreateDeliverabilityTestReportResponse$ = [3, n0, _CDTRRr, 0, [_RI, _DTS], [0, 0]];
var CreateEmailIdentityPolicyRequest$ = [
    3,
    n0,
    _CEIPR,
    0,
    [_EI, _PNo, _P],
    [[0, 1], [0, 1], 0],
];
var CreateEmailIdentityPolicyResponse$ = [3, n0, _CEIPRr, 0, [], []];
var CreateEmailIdentityRequest$ = [
    3,
    n0,
    _CEIR,
    0,
    [_EI, _Ta, _DSA, _CSN],
    [0, () => TagList, [() => DkimSigningAttributes$, 0], 0],
];
var CreateEmailIdentityResponse$ = [
    3,
    n0,
    _CEIRr,
    0,
    [_IT, _VFSS, _DA],
    [0, 2, () => DkimAttributes$],
];
var CreateEmailTemplateRequest$ = [
    3,
    n0,
    _CETR,
    0,
    [_TN, _TC, _Ta],
    [0, () => EmailTemplateContent$, () => TagList],
];
var CreateEmailTemplateResponse$ = [3, n0, _CETRr, 0, [], []];
var CreateExportJobRequest$ = [
    3,
    n0,
    _CEJRr,
    0,
    [_EDS, _EDx],
    [[() => ExportDataSource$, 0], () => ExportDestination$],
];
var CreateExportJobResponse$ = [3, n0, _CEJRre, 0, [_JI], [0]];
var CreateImportJobRequest$ = [
    3,
    n0,
    _CIJR,
    0,
    [_ID, _IDS],
    [() => ImportDestination$, () => ImportDataSource$],
];
var CreateImportJobResponse$ = [3, n0, _CIJRr, 0, [_JI], [0]];
var CreateMultiRegionEndpointRequest$ = [
    3,
    n0,
    _CMRER,
    0,
    [_EN, _Det, _Ta],
    [0, () => Details$, () => TagList],
];
var CreateMultiRegionEndpointResponse$ = [3, n0, _CMRERr, 0, [_S, _EIn], [0, 0]];
var CreateTenantRequest$ = [3, n0, _CTR, 0, [_TNe, _Ta], [0, () => TagList]];
var CreateTenantResourceAssociationRequest$ = [3, n0, _CTRAR, 0, [_TNe, _RA], [0, 0]];
var CreateTenantResourceAssociationResponse$ = [3, n0, _CTRARr, 0, [], []];
var CreateTenantResponse$ = [
    3,
    n0,
    _CTRr,
    0,
    [_TNe, _TI, _TA, _CTr, _Ta, _SS],
    [0, 0, 0, 4, () => TagList, 0],
];
var CustomVerificationEmailTemplateMetadata$ = [
    3,
    n0,
    _CVETM,
    0,
    [_TN, _FEA, _TS, _SRURL, _FRURL],
    [0, 0, 0, 0, 0],
];
var DailyVolume$ = [
    3,
    n0,
    _DV,
    0,
    [_SD, _VS, _DIP],
    [4, () => VolumeStatistics$, () => DomainIspPlacements],
];
var DashboardAttributes$ = [3, n0, _DAa, 0, [_EM], [0]];
var DashboardOptions$ = [3, n0, _DOa, 0, [_EM], [0]];
var DedicatedIp$ = [3, n0, _DI, 0, [_Ip, _WS, _WP, _PN], [0, 0, 1, 0]];
var DedicatedIpPool$ = [3, n0, _DIPe, 0, [_PN, _SM], [0, 0]];
var DeleteConfigurationSetEventDestinationRequest$ = [
    3,
    n0,
    _DCSEDR,
    0,
    [_CSN, _EDN],
    [
        [0, 1],
        [0, 1],
    ],
];
var DeleteConfigurationSetEventDestinationResponse$ = [3, n0, _DCSEDRe, 0, [], []];
var DeleteConfigurationSetRequest$ = [3, n0, _DCSR, 0, [_CSN], [[0, 1]]];
var DeleteConfigurationSetResponse$ = [3, n0, _DCSRe, 0, [], []];
var DeleteContactListRequest$ = [3, n0, _DCLR, 0, [_CLN], [[0, 1]]];
var DeleteContactListResponse$ = [3, n0, _DCLRe, 0, [], []];
var DeleteContactRequest$ = [
    3,
    n0,
    _DCR,
    0,
    [_CLN, _EA],
    [
        [0, 1],
        [0, 1],
    ],
];
var DeleteContactResponse$ = [3, n0, _DCRe, 0, [], []];
var DeleteCustomVerificationEmailTemplateRequest$ = [3, n0, _DCVETR, 0, [_TN], [[0, 1]]];
var DeleteCustomVerificationEmailTemplateResponse$ = [3, n0, _DCVETRe, 0, [], []];
var DeleteDedicatedIpPoolRequest$ = [3, n0, _DDIPR, 0, [_PN], [[0, 1]]];
var DeleteDedicatedIpPoolResponse$ = [3, n0, _DDIPRe, 0, [], []];
var DeleteEmailIdentityPolicyRequest$ = [
    3,
    n0,
    _DEIPR,
    0,
    [_EI, _PNo],
    [
        [0, 1],
        [0, 1],
    ],
];
var DeleteEmailIdentityPolicyResponse$ = [3, n0, _DEIPRe, 0, [], []];
var DeleteEmailIdentityRequest$ = [3, n0, _DEIR, 0, [_EI], [[0, 1]]];
var DeleteEmailIdentityResponse$ = [3, n0, _DEIRe, 0, [], []];
var DeleteEmailTemplateRequest$ = [3, n0, _DETR, 0, [_TN], [[0, 1]]];
var DeleteEmailTemplateResponse$ = [3, n0, _DETRe, 0, [], []];
var DeleteMultiRegionEndpointRequest$ = [3, n0, _DMRER, 0, [_EN], [[0, 1]]];
var DeleteMultiRegionEndpointResponse$ = [3, n0, _DMRERe, 0, [_S], [0]];
var DeleteSuppressedDestinationRequest$ = [3, n0, _DSDR, 0, [_EA], [[0, 1]]];
var DeleteSuppressedDestinationResponse$ = [3, n0, _DSDRe, 0, [], []];
var DeleteTenantRequest$ = [3, n0, _DTR, 0, [_TNe], [0]];
var DeleteTenantResourceAssociationRequest$ = [3, n0, _DTRAR, 0, [_TNe, _RA], [0, 0]];
var DeleteTenantResourceAssociationResponse$ = [3, n0, _DTRARe, 0, [], []];
var DeleteTenantResponse$ = [3, n0, _DTRe, 0, [], []];
var DeliverabilityTestReport$ = [
    3,
    n0,
    _DTRel,
    0,
    [_RI, _RNe, _Su, _FEA, _CDr, _DTS],
    [0, 0, 0, 0, 4, 0],
];
var DeliveryOptions$ = [3, n0, _DO, 0, [_TPl, _SPN, _MDS], [0, 0, 1]];
var Destination$ = [3, n0, _Des, 0, [_TAo, _CA, _BA], [64 | 0, 64 | 0, 64 | 0]];
var Details$ = [3, n0, _Det, 0, [_RDo], [() => RoutesDetails]];
var DkimAttributes$ = [
    3,
    n0,
    _DA,
    0,
    [_SE, _S, _Tok, _SHZ, _SAO, _NSKL, _CSKL, _LKGT],
    [2, 0, 64 | 0, 0, 0, 0, 0, 4],
];
var DkimSigningAttributes$ = [
    3,
    n0,
    _DSA,
    0,
    [_DSS, _DSPK, _NSKL, _DSAO],
    [0, [() => PrivateKey, 0], 0, 0],
];
var DomainDeliverabilityCampaign$ = [
    3,
    n0,
    _DDC,
    0,
    [_CIa, _IU, _Su, _FA, _SI, _FSDT, _LSDT, _IC, _SC, _RR, _DR, _RDR, _PV, _Es],
    [0, 0, 0, 0, 64 | 0, 4, 4, 1, 1, 1, 1, 1, 1, 64 | 0],
];
var DomainDeliverabilityTrackingOption$ = [
    3,
    n0,
    _DDTO,
    0,
    [_Do, _SSD, _IPTO],
    [0, 4, () => InboxPlacementTrackingOption$],
];
var DomainIspPlacement$ = [3, n0, _DIPo, 0, [_IN, _IRC, _SRC, _IP, _SP], [0, 1, 1, 1, 1]];
var EmailAddressInsightsMailboxEvaluations$ = [
    3,
    n0,
    _EAIME,
    0,
    [_HVS, _HVDR, _ME, _IRA, _IDs, _IRI],
    [
        () => EmailAddressInsightsVerdict$,
        () => EmailAddressInsightsVerdict$,
        () => EmailAddressInsightsVerdict$,
        () => EmailAddressInsightsVerdict$,
        () => EmailAddressInsightsVerdict$,
        () => EmailAddressInsightsVerdict$,
    ],
];
var EmailAddressInsightsVerdict$ = [3, n0, _EAIV, 0, [_CV], [0]];
var EmailContent$ = [
    3,
    n0,
    _EC,
    0,
    [_Si, _Ra, _Te],
    [() => Message$, () => RawMessage$, () => Template$],
];
var EmailInsights$ = [
    3,
    n0,
    _EIm,
    0,
    [_Des, _Is, _Ev],
    [[() => InsightsEmailAddress, 0], 0, () => InsightsEvents],
];
var EmailTemplateContent$ = [3, n0, _ETC, 0, [_Su, _T, _H], [0, 0, 0]];
var EmailTemplateMetadata$ = [3, n0, _ETM, 0, [_TN, _CTr], [0, 4]];
var EventBridgeDestination$ = [3, n0, _EBD, 0, [_EBA], [0]];
var EventDestination$ = [
    3,
    n0,
    _EDv,
    0,
    [_Na, _En, _MET, _KFD, _CWD, _SDn, _EBD, _PD],
    [
        0,
        2,
        64 | 0,
        () => KinesisFirehoseDestination$,
        () => CloudWatchDestination$,
        () => SnsDestination$,
        () => EventBridgeDestination$,
        () => PinpointDestination$,
    ],
];
var EventDestinationDefinition$ = [
    3,
    n0,
    _EDD,
    0,
    [_En, _MET, _KFD, _CWD, _SDn, _EBD, _PD],
    [
        2,
        64 | 0,
        () => KinesisFirehoseDestination$,
        () => CloudWatchDestination$,
        () => SnsDestination$,
        () => EventBridgeDestination$,
        () => PinpointDestination$,
    ],
];
var EventDetails$ = [3, n0, _EDve, 0, [_Bo, _C], [() => Bounce$, () => Complaint$]];
var ExportDataSource$ = [
    3,
    n0,
    _EDS,
    0,
    [_MDSe, _MIDS],
    [() => MetricsDataSource$, [() => MessageInsightsDataSource$, 0]],
];
var ExportDestination$ = [3, n0, _EDx, 0, [_DF, _SU], [0, 0]];
var ExportJobSummary$ = [3, n0, _EJS, 0, [_JI, _EST, _JS, _CTr, _CTo], [0, 0, 0, 4, 4]];
var ExportMetric$ = [3, n0, _EMx, 0, [_Na, _Ag], [0, 0]];
var ExportStatistics$ = [3, n0, _ESx, 0, [_PRC, _ERC], [1, 1]];
var FailureInfo$ = [3, n0, _FI, 0, [_FRSU, _EMr], [0, 0]];
var GetAccountRequest$ = [3, n0, _GAR, 0, [], []];
var GetAccountResponse$ = [
    3,
    n0,
    _GARe,
    0,
    [_DIAWE, _ESn, _PAE, _SQ, _SEe, _SA, _Det, _VA],
    [2, 0, 2, () => SendQuota$, 2, () => SuppressionAttributes$, [() => AccountDetails$, 0], () => VdmAttributes$],
];
var GetBlacklistReportsRequest$ = [3, n0, _GBRR, 0, [_BIN], [[64 | 0, { [_hQ]: _BIN }]]];
var GetBlacklistReportsResponse$ = [3, n0, _GBRRe, 0, [_BR], [() => BlacklistReport]];
var GetConfigurationSetEventDestinationsRequest$ = [3, n0, _GCSEDR, 0, [_CSN], [[0, 1]]];
var GetConfigurationSetEventDestinationsResponse$ = [
    3,
    n0,
    _GCSEDRe,
    0,
    [_EDven],
    [() => EventDestinations],
];
var GetConfigurationSetRequest$ = [3, n0, _GCSR, 0, [_CSN], [[0, 1]]];
var GetConfigurationSetResponse$ = [
    3,
    n0,
    _GCSRe,
    0,
    [_CSN, _TO, _DO, _RO, _SO, _Ta, _SOu, _VO, _AO],
    [
        0,
        () => TrackingOptions$,
        () => DeliveryOptions$,
        () => ReputationOptions$,
        () => SendingOptions$,
        () => TagList,
        () => SuppressionOptions$,
        () => VdmOptions$,
        () => ArchivingOptions$,
    ],
];
var GetContactListRequest$ = [3, n0, _GCLR, 0, [_CLN], [[0, 1]]];
var GetContactListResponse$ = [
    3,
    n0,
    _GCLRe,
    0,
    [_CLN, _To, _De, _CTr, _LUT, _Ta],
    [0, () => Topics, 0, 4, 4, () => TagList],
];
var GetContactRequest$ = [
    3,
    n0,
    _GCR,
    0,
    [_CLN, _EA],
    [
        [0, 1],
        [0, 1],
    ],
];
var GetContactResponse$ = [
    3,
    n0,
    _GCRe,
    0,
    [_CLN, _EA, _TP, _TDP, _UA, _ADt, _CTr, _LUT],
    [0, 0, () => TopicPreferenceList, () => TopicPreferenceList, 2, 0, 4, 4],
];
var GetCustomVerificationEmailTemplateRequest$ = [3, n0, _GCVETR, 0, [_TN], [[0, 1]]];
var GetCustomVerificationEmailTemplateResponse$ = [
    3,
    n0,
    _GCVETRe,
    0,
    [_TN, _FEA, _TS, _TC, _Ta, _SRURL, _FRURL],
    [0, 0, 0, 0, () => TagList, 0, 0],
];
var GetDedicatedIpPoolRequest$ = [3, n0, _GDIPR, 0, [_PN], [[0, 1]]];
var GetDedicatedIpPoolResponse$ = [3, n0, _GDIPRe, 0, [_DIPe], [() => DedicatedIpPool$]];
var GetDedicatedIpRequest$ = [3, n0, _GDIR, 0, [_Ip], [[0, 1]]];
var GetDedicatedIpResponse$ = [3, n0, _GDIRe, 0, [_DI], [() => DedicatedIp$]];
var GetDedicatedIpsRequest$ = [
    3,
    n0,
    _GDIRet,
    0,
    [_PN, _NT, _PS],
    [
        [0, { [_hQ]: _PN }],
        [0, { [_hQ]: _NT }],
        [1, { [_hQ]: _PS }],
    ],
];
var GetDedicatedIpsResponse$ = [
    3,
    n0,
    _GDIRete,
    0,
    [_DIe, _NT],
    [() => DedicatedIpList, 0],
];
var GetDeliverabilityDashboardOptionsRequest$ = [3, n0, _GDDOR, 0, [], []];
var GetDeliverabilityDashboardOptionsResponse$ = [
    3,
    n0,
    _GDDORe,
    0,
    [_DE, _SED, _AS, _ASD, _PESD],
    [2, 4, 0, () => DomainDeliverabilityTrackingOptions, () => DomainDeliverabilityTrackingOptions],
];
var GetDeliverabilityTestReportRequest$ = [3, n0, _GDTRR, 0, [_RI], [[0, 1]]];
var GetDeliverabilityTestReportResponse$ = [
    3,
    n0,
    _GDTRRe,
    0,
    [_DTRel, _OP, _IPs, _Me, _Ta],
    [() => DeliverabilityTestReport$, () => PlacementStatistics$, () => IspPlacements, 0, () => TagList],
];
var GetDomainDeliverabilityCampaignRequest$ = [3, n0, _GDDCR, 0, [_CIa], [[0, 1]]];
var GetDomainDeliverabilityCampaignResponse$ = [
    3,
    n0,
    _GDDCRe,
    0,
    [_DDC],
    [() => DomainDeliverabilityCampaign$],
];
var GetDomainStatisticsReportRequest$ = [
    3,
    n0,
    _GDSRR,
    0,
    [_Do, _SD, _ED],
    [
        [0, 1],
        [4, { [_hQ]: _SD }],
        [4, { [_hQ]: _ED }],
    ],
];
var GetDomainStatisticsReportResponse$ = [
    3,
    n0,
    _GDSRRe,
    0,
    [_OV, _DVa],
    [() => OverallVolume$, () => DailyVolumes],
];
var GetEmailAddressInsightsRequest$ = [3, n0, _GEAIR, 0, [_EA], [0]];
var GetEmailAddressInsightsResponse$ = [
    3,
    n0,
    _GEAIRe,
    0,
    [_MV],
    [() => MailboxValidation$],
];
var GetEmailIdentityPoliciesRequest$ = [3, n0, _GEIPR, 0, [_EI], [[0, 1]]];
var GetEmailIdentityPoliciesResponse$ = [3, n0, _GEIPRe, 0, [_Po], [128 | 0]];
var GetEmailIdentityRequest$ = [3, n0, _GEIR, 0, [_EI], [[0, 1]]];
var GetEmailIdentityResponse$ = [
    3,
    n0,
    _GEIRe,
    0,
    [_IT, _FFS, _VFSS, _DA, _MFA, _Po, _Ta, _CSN, _VSe, _VI],
    [0, 2, 2, () => DkimAttributes$, () => MailFromAttributes$, 128 | 0, () => TagList, 0, 0, () => VerificationInfo$],
];
var GetEmailTemplateRequest$ = [3, n0, _GETR, 0, [_TN], [[0, 1]]];
var GetEmailTemplateResponse$ = [
    3,
    n0,
    _GETRe,
    0,
    [_TN, _TC, _Ta],
    [0, () => EmailTemplateContent$, () => TagList],
];
var GetExportJobRequest$ = [3, n0, _GEJR, 0, [_JI], [[0, 1]]];
var GetExportJobResponse$ = [
    3,
    n0,
    _GEJRe,
    0,
    [_JI, _EST, _JS, _EDx, _EDS, _CTr, _CTo, _FI, _St],
    [0, 0, 0, () => ExportDestination$, [() => ExportDataSource$, 0], 4, 4, () => FailureInfo$, () => ExportStatistics$],
];
var GetImportJobRequest$ = [3, n0, _GIJR, 0, [_JI], [[0, 1]]];
var GetImportJobResponse$ = [
    3,
    n0,
    _GIJRe,
    0,
    [_JI, _ID, _IDS, _FI, _JS, _CTr, _CTo, _PRC, _FRC],
    [0, () => ImportDestination$, () => ImportDataSource$, () => FailureInfo$, 0, 4, 4, 1, 1],
];
var GetMessageInsightsRequest$ = [3, n0, _GMIR, 0, [_MI], [[0, 1]]];
var GetMessageInsightsResponse$ = [
    3,
    n0,
    _GMIRe,
    0,
    [_MI, _FEA, _Su, _ET, _In],
    [0, [() => InsightsEmailAddress, 0], [() => EmailSubject, 0], () => MessageTagList, [() => EmailInsightsList, 0]],
];
var GetMultiRegionEndpointRequest$ = [3, n0, _GMRER, 0, [_EN], [[0, 1]]];
var GetMultiRegionEndpointResponse$ = [
    3,
    n0,
    _GMRERe,
    0,
    [_EN, _EIn, _Ro, _S, _CTr, _LUT],
    [0, 0, () => Routes, 0, 4, 4],
];
var GetReputationEntityRequest$ = [
    3,
    n0,
    _GRER,
    0,
    [_RER, _RET],
    [
        [0, 1],
        [0, 1],
    ],
];
var GetReputationEntityResponse$ = [3, n0, _GRERe, 0, [_RE], [() => ReputationEntity$]];
var GetSuppressedDestinationRequest$ = [3, n0, _GSDR, 0, [_EA], [[0, 1]]];
var GetSuppressedDestinationResponse$ = [
    3,
    n0,
    _GSDRe,
    0,
    [_SDu],
    [() => SuppressedDestination$],
];
var GetTenantRequest$ = [3, n0, _GTR, 0, [_TNe], [0]];
var GetTenantResponse$ = [3, n0, _GTRe, 0, [_Ten], [() => Tenant$]];
var GuardianAttributes$ = [3, n0, _GA, 0, [_OSD], [0]];
var GuardianOptions$ = [3, n0, _GO, 0, [_OSD], [0]];
var IdentityInfo$ = [3, n0, _II, 0, [_IT, _INd, _SEe, _VSe], [0, 0, 2, 0]];
var ImportDataSource$ = [3, n0, _IDS, 0, [_SU, _DF], [0, 0]];
var ImportDestination$ = [
    3,
    n0,
    _ID,
    0,
    [_SLD, _CLD],
    [() => SuppressionListDestination$, () => ContactListDestination$],
];
var ImportJobSummary$ = [
    3,
    n0,
    _IJS,
    0,
    [_JI, _ID, _JS, _CTr, _PRC, _FRC],
    [0, () => ImportDestination$, 0, 4, 1, 1],
];
var InboxPlacementTrackingOption$ = [3, n0, _IPTO, 0, [_G, _TIr], [2, 64 | 0]];
var InsightsEvent$ = [3, n0, _IE, 0, [_Ti, _Ty, _Det], [4, 0, () => EventDetails$]];
var InternalServiceErrorException$ = [-3, n0, _ISEE, { [_e]: _s, [_hE]: 500 }, [_m], [0]];
schema.TypeRegistry.for(n0).registerError(InternalServiceErrorException$, InternalServiceErrorException);
var InvalidNextTokenException$ = [-3, n0, _INTE, { [_e]: _c, [_hE]: 400 }, [_m], [0]];
schema.TypeRegistry.for(n0).registerError(InvalidNextTokenException$, InvalidNextTokenException);
var IspPlacement$ = [3, n0, _IPsp, 0, [_IN, _PSl], [0, () => PlacementStatistics$]];
var KinesisFirehoseDestination$ = [3, n0, _KFD, 0, [_IRAa, _DSAe], [0, 0]];
var LimitExceededException$ = [-3, n0, _LEE, { [_e]: _c, [_hE]: 400 }, [_m], [0]];
schema.TypeRegistry.for(n0).registerError(LimitExceededException$, LimitExceededException);
var ListConfigurationSetsRequest$ = [
    3,
    n0,
    _LCSR,
    0,
    [_NT, _PS],
    [
        [0, { [_hQ]: _NT }],
        [1, { [_hQ]: _PS }],
    ],
];
var ListConfigurationSetsResponse$ = [3, n0, _LCSRi, 0, [_CS, _NT], [64 | 0, 0]];
var ListContactListsRequest$ = [
    3,
    n0,
    _LCLR,
    0,
    [_PS, _NT],
    [
        [1, { [_hQ]: _PS }],
        [0, { [_hQ]: _NT }],
    ],
];
var ListContactListsResponse$ = [
    3,
    n0,
    _LCLRi,
    0,
    [_CLon, _NT],
    [() => ListOfContactLists, 0],
];
var ListContactsFilter$ = [3, n0, _LCF, 0, [_FS, _TF], [0, () => TopicFilter$]];
var ListContactsRequest$ = [
    3,
    n0,
    _LCR,
    0,
    [_CLN, _F, _PS, _NT],
    [[0, 1], () => ListContactsFilter$, 1, 0],
];
var ListContactsResponse$ = [3, n0, _LCRi, 0, [_Cont, _NT], [() => ListOfContacts, 0]];
var ListCustomVerificationEmailTemplatesRequest$ = [
    3,
    n0,
    _LCVETR,
    0,
    [_NT, _PS],
    [
        [0, { [_hQ]: _NT }],
        [1, { [_hQ]: _PS }],
    ],
];
var ListCustomVerificationEmailTemplatesResponse$ = [
    3,
    n0,
    _LCVETRi,
    0,
    [_CVET, _NT],
    [() => CustomVerificationEmailTemplatesList, 0],
];
var ListDedicatedIpPoolsRequest$ = [
    3,
    n0,
    _LDIPR,
    0,
    [_NT, _PS],
    [
        [0, { [_hQ]: _NT }],
        [1, { [_hQ]: _PS }],
    ],
];
var ListDedicatedIpPoolsResponse$ = [3, n0, _LDIPRi, 0, [_DIPed, _NT], [64 | 0, 0]];
var ListDeliverabilityTestReportsRequest$ = [
    3,
    n0,
    _LDTRR,
    0,
    [_NT, _PS],
    [
        [0, { [_hQ]: _NT }],
        [1, { [_hQ]: _PS }],
    ],
];
var ListDeliverabilityTestReportsResponse$ = [
    3,
    n0,
    _LDTRRi,
    0,
    [_DTReli, _NT],
    [() => DeliverabilityTestReports, 0],
];
var ListDomainDeliverabilityCampaignsRequest$ = [
    3,
    n0,
    _LDDCR,
    0,
    [_SD, _ED, _SDub, _NT, _PS],
    [
        [4, { [_hQ]: _SD }],
        [4, { [_hQ]: _ED }],
        [0, 1],
        [0, { [_hQ]: _NT }],
        [1, { [_hQ]: _PS }],
    ],
];
var ListDomainDeliverabilityCampaignsResponse$ = [
    3,
    n0,
    _LDDCRi,
    0,
    [_DDCo, _NT],
    [() => DomainDeliverabilityCampaignList, 0],
];
var ListEmailIdentitiesRequest$ = [
    3,
    n0,
    _LEIR,
    0,
    [_NT, _PS],
    [
        [0, { [_hQ]: _NT }],
        [1, { [_hQ]: _PS }],
    ],
];
var ListEmailIdentitiesResponse$ = [
    3,
    n0,
    _LEIRi,
    0,
    [_EIma, _NT],
    [() => IdentityInfoList, 0],
];
var ListEmailTemplatesRequest$ = [
    3,
    n0,
    _LETR,
    0,
    [_NT, _PS],
    [
        [0, { [_hQ]: _NT }],
        [1, { [_hQ]: _PS }],
    ],
];
var ListEmailTemplatesResponse$ = [
    3,
    n0,
    _LETRi,
    0,
    [_TM, _NT],
    [() => EmailTemplateMetadataList, 0],
];
var ListExportJobsRequest$ = [3, n0, _LEJR, 0, [_NT, _PS, _EST, _JS], [0, 1, 0, 0]];
var ListExportJobsResponse$ = [
    3,
    n0,
    _LEJRi,
    0,
    [_EJ, _NT],
    [() => ExportJobSummaryList, 0],
];
var ListImportJobsRequest$ = [3, n0, _LIJR, 0, [_IDT, _NT, _PS], [0, 0, 1]];
var ListImportJobsResponse$ = [
    3,
    n0,
    _LIJRi,
    0,
    [_IJ, _NT],
    [() => ImportJobSummaryList, 0],
];
var ListManagementOptions$ = [3, n0, _LMO, 0, [_CLN, _TNo], [0, 0]];
var ListMultiRegionEndpointsRequest$ = [
    3,
    n0,
    _LMRER,
    0,
    [_NT, _PS],
    [
        [0, { [_hQ]: _NT }],
        [1, { [_hQ]: _PS }],
    ],
];
var ListMultiRegionEndpointsResponse$ = [
    3,
    n0,
    _LMRERi,
    0,
    [_MRE, _NT],
    [() => MultiRegionEndpoints, 0],
];
var ListRecommendationsRequest$ = [3, n0, _LRR, 0, [_F, _NT, _PS], [128 | 0, 0, 1]];
var ListRecommendationsResponse$ = [
    3,
    n0,
    _LRRi,
    0,
    [_Re, _NT],
    [() => RecommendationsList, 0],
];
var ListReputationEntitiesRequest$ = [3, n0, _LRER, 0, [_F, _NT, _PS], [128 | 0, 0, 1]];
var ListReputationEntitiesResponse$ = [
    3,
    n0,
    _LRERi,
    0,
    [_REe, _NT],
    [() => ReputationEntitiesList, 0],
];
var ListResourceTenantsRequest$ = [3, n0, _LRTR, 0, [_RA, _PS, _NT], [0, 1, 0]];
var ListResourceTenantsResponse$ = [
    3,
    n0,
    _LRTRi,
    0,
    [_RTe, _NT],
    [() => ResourceTenantMetadataList, 0],
];
var ListSuppressedDestinationsRequest$ = [
    3,
    n0,
    _LSDR,
    0,
    [_Rea, _SD, _ED, _NT, _PS],
    [
        [64 | 0, { [_hQ]: _Reas }],
        [4, { [_hQ]: _SD }],
        [4, { [_hQ]: _ED }],
        [0, { [_hQ]: _NT }],
        [1, { [_hQ]: _PS }],
    ],
];
var ListSuppressedDestinationsResponse$ = [
    3,
    n0,
    _LSDRi,
    0,
    [_SDS, _NT],
    [() => SuppressedDestinationSummaries, 0],
];
var ListTagsForResourceRequest$ = [3, n0, _LTFRR, 0, [_RA], [[0, { [_hQ]: _RA }]]];
var ListTagsForResourceResponse$ = [3, n0, _LTFRRi, 0, [_Ta], [() => TagList]];
var ListTenantResourcesRequest$ = [
    3,
    n0,
    _LTRR,
    0,
    [_TNe, _F, _PS, _NT],
    [0, 128 | 0, 1, 0],
];
var ListTenantResourcesResponse$ = [
    3,
    n0,
    _LTRRi,
    0,
    [_TR, _NT],
    [() => TenantResourceList, 0],
];
var ListTenantsRequest$ = [3, n0, _LTR, 0, [_NT, _PS], [0, 1]];
var ListTenantsResponse$ = [3, n0, _LTRi, 0, [_Tena, _NT], [() => TenantInfoList, 0]];
var MailboxValidation$ = [
    3,
    n0,
    _MV,
    0,
    [_IV, _Eva],
    [() => EmailAddressInsightsVerdict$, () => EmailAddressInsightsMailboxEvaluations$],
];
var MailFromAttributes$ = [3, n0, _MFA, 0, [_MFD, _MFDS, _BOMF], [0, 0, 0]];
var MailFromDomainNotVerifiedException$ = [
    -3,
    n0,
    _MFDNVE,
    { [_e]: _c, [_hE]: 400 },
    [_m],
    [0],
];
schema.TypeRegistry.for(n0).registerError(MailFromDomainNotVerifiedException$, MailFromDomainNotVerifiedException);
var Message$ = [
    3,
    n0,
    _Me,
    0,
    [_Su, _B, _He, _At],
    [() => Content$, () => Body$, () => MessageHeaderList, () => AttachmentList],
];
var MessageHeader$ = [3, n0, _MH, 0, [_Na, _V], [0, 0]];
var MessageInsightsDataSource$ = [
    3,
    n0,
    _MIDS,
    0,
    [_SD, _ED, _Inc, _Ex, _MR],
    [4, 4, [() => MessageInsightsFilters$, 0], [() => MessageInsightsFilters$, 0], 1],
];
var MessageInsightsFilters$ = [
    3,
    n0,
    _MIF,
    0,
    [_FEA, _Des, _Su, _Is, _LDE, _LEEa],
    [
        [() => EmailAddressFilterList, 0],
        [() => EmailAddressFilterList, 0],
        [() => EmailSubjectFilterList, 0],
        64 | 0,
        64 | 0,
        64 | 0,
    ],
];
var MessageRejected$ = [-3, n0, _MRe, { [_e]: _c, [_hE]: 400 }, [_m], [0]];
schema.TypeRegistry.for(n0).registerError(MessageRejected$, MessageRejected);
var MessageTag$ = [3, n0, _MTe, 0, [_Na, _V], [0, 0]];
var MetricDataError$ = [3, n0, _MDE, 0, [_I, _Cod, _Me], [0, 0, 0]];
var MetricDataResult$ = [3, n0, _MDR, 0, [_I, _Tim, _Va], [0, 64 | 4, 64 | 1]];
var MetricsDataSource$ = [
    3,
    n0,
    _MDSe,
    0,
    [_D, _N, _Met, _SD, _ED],
    [[2, n0, _EDxp, 0, 0, 64 | 0], 0, () => ExportMetrics, 4, 4],
];
var MultiRegionEndpoint$ = [
    3,
    n0,
    _MREu,
    0,
    [_EN, _S, _EIn, _Reg, _CTr, _LUT],
    [0, 0, 0, 64 | 0, 4, 4],
];
var NotFoundException$ = [-3, n0, _NFE, { [_e]: _c, [_hE]: 404 }, [_m], [0]];
schema.TypeRegistry.for(n0).registerError(NotFoundException$, NotFoundException);
var OverallVolume$ = [
    3,
    n0,
    _OV,
    0,
    [_VS, _RRP, _DIP],
    [() => VolumeStatistics$, 1, () => DomainIspPlacements],
];
var PinpointDestination$ = [3, n0, _PD, 0, [_AAp], [0]];
var PlacementStatistics$ = [3, n0, _PSl, 0, [_IP, _SP, _MP, _SPp, _DP], [1, 1, 1, 1, 1]];
var PutAccountDedicatedIpWarmupAttributesRequest$ = [3, n0, _PADIWAR, 0, [_AWE], [2]];
var PutAccountDedicatedIpWarmupAttributesResponse$ = [3, n0, _PADIWARu, 0, [], []];
var PutAccountDetailsRequest$ = [
    3,
    n0,
    _PADR,
    0,
    [_MT, _WURL, _CL, _UCD, _ACEAd, _PAE],
    [0, [() => WebsiteURL, 0], 0, [() => UseCaseDescription, 0], [() => AdditionalContactEmailAddresses, 0], 2],
];
var PutAccountDetailsResponse$ = [3, n0, _PADRu, 0, [], []];
var PutAccountSendingAttributesRequest$ = [3, n0, _PASAR, 0, [_SEe], [2]];
var PutAccountSendingAttributesResponse$ = [3, n0, _PASARu, 0, [], []];
var PutAccountSuppressionAttributesRequest$ = [
    3,
    n0,
    _PASARut,
    0,
    [_SR, _VAa],
    [64 | 0, () => SuppressionValidationAttributes$],
];
var PutAccountSuppressionAttributesResponse$ = [3, n0, _PASARutc, 0, [], []];
var PutAccountVdmAttributesRequest$ = [3, n0, _PAVAR, 0, [_VA], [() => VdmAttributes$]];
var PutAccountVdmAttributesResponse$ = [3, n0, _PAVARu, 0, [], []];
var PutConfigurationSetArchivingOptionsRequest$ = [
    3,
    n0,
    _PCSAOR,
    0,
    [_CSN, _AA],
    [[0, 1], 0],
];
var PutConfigurationSetArchivingOptionsResponse$ = [3, n0, _PCSAORu, 0, [], []];
var PutConfigurationSetDeliveryOptionsRequest$ = [
    3,
    n0,
    _PCSDOR,
    0,
    [_CSN, _TPl, _SPN, _MDS],
    [[0, 1], 0, 0, 1],
];
var PutConfigurationSetDeliveryOptionsResponse$ = [3, n0, _PCSDORu, 0, [], []];
var PutConfigurationSetReputationOptionsRequest$ = [
    3,
    n0,
    _PCSROR,
    0,
    [_CSN, _RME],
    [[0, 1], 2],
];
var PutConfigurationSetReputationOptionsResponse$ = [3, n0, _PCSRORu, 0, [], []];
var PutConfigurationSetSendingOptionsRequest$ = [
    3,
    n0,
    _PCSSOR,
    0,
    [_CSN, _SEe],
    [[0, 1], 2],
];
var PutConfigurationSetSendingOptionsResponse$ = [3, n0, _PCSSORu, 0, [], []];
var PutConfigurationSetSuppressionOptionsRequest$ = [
    3,
    n0,
    _PCSSORut,
    0,
    [_CSN, _SR, _VOa],
    [[0, 1], 64 | 0, () => SuppressionValidationOptions$],
];
var PutConfigurationSetSuppressionOptionsResponse$ = [3, n0, _PCSSORuto, 0, [], []];
var PutConfigurationSetTrackingOptionsRequest$ = [
    3,
    n0,
    _PCSTOR,
    0,
    [_CSN, _CRD, _HP],
    [[0, 1], 0, 0],
];
var PutConfigurationSetTrackingOptionsResponse$ = [3, n0, _PCSTORu, 0, [], []];
var PutConfigurationSetVdmOptionsRequest$ = [
    3,
    n0,
    _PCSVOR,
    0,
    [_CSN, _VO],
    [[0, 1], () => VdmOptions$],
];
var PutConfigurationSetVdmOptionsResponse$ = [3, n0, _PCSVORu, 0, [], []];
var PutDedicatedIpInPoolRequest$ = [3, n0, _PDIIPR, 0, [_Ip, _DPN], [[0, 1], 0]];
var PutDedicatedIpInPoolResponse$ = [3, n0, _PDIIPRu, 0, [], []];
var PutDedicatedIpPoolScalingAttributesRequest$ = [
    3,
    n0,
    _PDIPSAR,
    0,
    [_PN, _SM],
    [[0, 1], 0],
];
var PutDedicatedIpPoolScalingAttributesResponse$ = [3, n0, _PDIPSARu, 0, [], []];
var PutDedicatedIpWarmupAttributesRequest$ = [3, n0, _PDIWAR, 0, [_Ip, _WP], [[0, 1], 1]];
var PutDedicatedIpWarmupAttributesResponse$ = [3, n0, _PDIWARu, 0, [], []];
var PutDeliverabilityDashboardOptionRequest$ = [
    3,
    n0,
    _PDDOR,
    0,
    [_DE, _SDubs],
    [2, () => DomainDeliverabilityTrackingOptions],
];
var PutDeliverabilityDashboardOptionResponse$ = [3, n0, _PDDORu, 0, [], []];
var PutEmailIdentityConfigurationSetAttributesRequest$ = [
    3,
    n0,
    _PEICSAR,
    0,
    [_EI, _CSN],
    [[0, 1], 0],
];
var PutEmailIdentityConfigurationSetAttributesResponse$ = [3, n0, _PEICSARu, 0, [], []];
var PutEmailIdentityDkimAttributesRequest$ = [3, n0, _PEIDAR, 0, [_EI, _SE], [[0, 1], 2]];
var PutEmailIdentityDkimAttributesResponse$ = [3, n0, _PEIDARu, 0, [], []];
var PutEmailIdentityDkimSigningAttributesRequest$ = [
    3,
    n0,
    _PEIDSAR,
    0,
    [_EI, _SAO, _SAi],
    [[0, 1], 0, [() => DkimSigningAttributes$, 0]],
];
var PutEmailIdentityDkimSigningAttributesResponse$ = [
    3,
    n0,
    _PEIDSARu,
    0,
    [_DS, _DT, _SHZ],
    [0, 64 | 0, 0],
];
var PutEmailIdentityFeedbackAttributesRequest$ = [
    3,
    n0,
    _PEIFAR,
    0,
    [_EI, _EFE],
    [[0, 1], 2],
];
var PutEmailIdentityFeedbackAttributesResponse$ = [3, n0, _PEIFARu, 0, [], []];
var PutEmailIdentityMailFromAttributesRequest$ = [
    3,
    n0,
    _PEIMFAR,
    0,
    [_EI, _MFD, _BOMF],
    [[0, 1], 0, 0],
];
var PutEmailIdentityMailFromAttributesResponse$ = [3, n0, _PEIMFARu, 0, [], []];
var PutSuppressedDestinationRequest$ = [3, n0, _PSDR, 0, [_EA, _Reas], [0, 0]];
var PutSuppressedDestinationResponse$ = [3, n0, _PSDRu, 0, [], []];
var RawMessage$ = [3, n0, _RM, 0, [_Da], [21]];
var Recommendation$ = [
    3,
    n0,
    _Rec,
    0,
    [_RA, _Ty, _De, _S, _CTr, _LUT, _Im],
    [0, 0, 0, 0, 4, 4, 0],
];
var ReplacementEmailContent$ = [3, n0, _REC, 0, [_RTep], [() => ReplacementTemplate$]];
var ReplacementTemplate$ = [3, n0, _RTep, 0, [_RTD], [0]];
var ReputationEntity$ = [
    3,
    n0,
    _RE,
    0,
    [_RER, _RET, _RMP, _CMS, _ASMS, _SSA, _RIe],
    [0, 0, 0, () => StatusRecord$, () => StatusRecord$, 0, 0],
];
var ReputationOptions$ = [3, n0, _RO, 0, [_RME, _LFS], [2, 4]];
var ResourceTenantMetadata$ = [3, n0, _RTM, 0, [_TNe, _TI, _RA, _AT], [0, 0, 0, 4]];
var ReviewDetails$ = [3, n0, _RD, 0, [_S, _CIas], [0, 0]];
var Route$ = [3, n0, _Rou, 0, [_Regi], [0]];
var RouteDetails$ = [3, n0, _RDou, 0, [_Regi], [0]];
var SendBulkEmailRequest$ = [
    3,
    n0,
    _SBER,
    0,
    [_FEA, _FEAIA, _RTA, _FFEA, _FFEAIA, _DET, _DCe, _BEEu, _CSN, _EIn, _TNe],
    [0, 0, 64 | 0, 0, 0, () => MessageTagList, () => BulkEmailContent$, () => BulkEmailEntryList, 0, 0, 0],
];
var SendBulkEmailResponse$ = [
    3,
    n0,
    _SBERe,
    0,
    [_BEERu],
    [() => BulkEmailEntryResultList],
];
var SendCustomVerificationEmailRequest$ = [3, n0, _SCVER, 0, [_EA, _TN, _CSN], [0, 0, 0]];
var SendCustomVerificationEmailResponse$ = [3, n0, _SCVERe, 0, [_MI], [0]];
var SendEmailRequest$ = [
    3,
    n0,
    _SER,
    0,
    [_FEA, _FEAIA, _Des, _RTA, _FFEA, _FFEAIA, _Con, _ET, _CSN, _EIn, _TNe, _LMO],
    [
        0,
        0,
        () => Destination$,
        64 | 0,
        0,
        0,
        () => EmailContent$,
        () => MessageTagList,
        0,
        0,
        0,
        () => ListManagementOptions$,
    ],
];
var SendEmailResponse$ = [3, n0, _SERe, 0, [_MI], [0]];
var SendingOptions$ = [3, n0, _SO, 0, [_SEe], [2]];
var SendingPausedException$ = [-3, n0, _SPE, { [_e]: _c, [_hE]: 400 }, [_m], [0]];
schema.TypeRegistry.for(n0).registerError(SendingPausedException$, SendingPausedException);
var SendQuota$ = [3, n0, _SQ, 0, [_MHS, _MSR, _SLH], [1, 1, 1]];
var SnsDestination$ = [3, n0, _SDn, 0, [_TAop], [0]];
var SOARecord$ = [3, n0, _SOAR, 0, [_PNS, _AE, _SN], [0, 0, 1]];
var StatusRecord$ = [3, n0, _SRt, 0, [_S, _Ca, _LUT], [0, 0, 4]];
var SuppressedDestination$ = [
    3,
    n0,
    _SDu,
    0,
    [_EA, _Reas, _LUTa, _Att],
    [0, 0, 4, () => SuppressedDestinationAttributes$],
];
var SuppressedDestinationAttributes$ = [3, n0, _SDA, 0, [_MI, _FIe], [0, 0]];
var SuppressedDestinationSummary$ = [3, n0, _SDSu, 0, [_EA, _Reas, _LUTa], [0, 0, 4]];
var SuppressionAttributes$ = [
    3,
    n0,
    _SA,
    0,
    [_SR, _VAa],
    [64 | 0, () => SuppressionValidationAttributes$],
];
var SuppressionConditionThreshold$ = [
    3,
    n0,
    _SCT,
    0,
    [_CTEo, _OCT],
    [0, () => SuppressionConfidenceThreshold$],
];
var SuppressionConfidenceThreshold$ = [3, n0, _SCTu, 0, [_CVT], [0]];
var SuppressionListDestination$ = [3, n0, _SLD, 0, [_SLIA], [0]];
var SuppressionOptions$ = [
    3,
    n0,
    _SOu,
    0,
    [_SR, _VOa],
    [64 | 0, () => SuppressionValidationOptions$],
];
var SuppressionValidationAttributes$ = [
    3,
    n0,
    _SVA,
    0,
    [_CTon],
    [() => SuppressionConditionThreshold$],
];
var SuppressionValidationOptions$ = [
    3,
    n0,
    _SVO,
    0,
    [_CTon],
    [() => SuppressionConditionThreshold$],
];
var Tag$ = [3, n0, _Tag, 0, [_K, _V], [0, 0]];
var TagResourceRequest$ = [3, n0, _TRR, 0, [_RA, _Ta], [0, () => TagList]];
var TagResourceResponse$ = [3, n0, _TRRa, 0, [], []];
var Template$ = [
    3,
    n0,
    _Te,
    0,
    [_TN, _TAe, _TC, _TD, _He, _At],
    [0, 0, () => EmailTemplateContent$, 0, () => MessageHeaderList, () => AttachmentList],
];
var Tenant$ = [
    3,
    n0,
    _Ten,
    0,
    [_TNe, _TI, _TA, _CTr, _Ta, _SS],
    [0, 0, 0, 4, () => TagList, 0],
];
var TenantInfo$ = [3, n0, _TIe, 0, [_TNe, _TI, _TA, _CTr], [0, 0, 0, 4]];
var TenantResource$ = [3, n0, _TRe, 0, [_RTes, _RA], [0, 0]];
var TestRenderEmailTemplateRequest$ = [3, n0, _TRETR, 0, [_TN, _TD], [[0, 1], 0]];
var TestRenderEmailTemplateResponse$ = [3, n0, _TRETRe, 0, [_RTen], [0]];
var TooManyRequestsException$ = [-3, n0, _TMRE, { [_e]: _c, [_hE]: 429 }, [_m], [0]];
schema.TypeRegistry.for(n0).registerError(TooManyRequestsException$, TooManyRequestsException);
var Topic$ = [3, n0, _Top, 0, [_TNo, _DNi, _De, _DSSe], [0, 0, 0, 0]];
var TopicFilter$ = [3, n0, _TF, 0, [_TNo, _UDIPU], [0, 2]];
var TopicPreference$ = [3, n0, _TPo, 0, [_TNo, _SSu], [0, 0]];
var TrackingOptions$ = [3, n0, _TO, 0, [_CRD, _HP], [0, 0]];
var UntagResourceRequest$ = [
    3,
    n0,
    _URR,
    0,
    [_RA, _TK],
    [
        [0, { [_hQ]: _RA }],
        [64 | 0, { [_hQ]: _TK }],
    ],
];
var UntagResourceResponse$ = [3, n0, _URRn, 0, [], []];
var UpdateConfigurationSetEventDestinationRequest$ = [
    3,
    n0,
    _UCSEDR,
    0,
    [_CSN, _EDN, _EDv],
    [[0, 1], [0, 1], () => EventDestinationDefinition$],
];
var UpdateConfigurationSetEventDestinationResponse$ = [3, n0, _UCSEDRp, 0, [], []];
var UpdateContactListRequest$ = [
    3,
    n0,
    _UCLR,
    0,
    [_CLN, _To, _De],
    [[0, 1], () => Topics, 0],
];
var UpdateContactListResponse$ = [3, n0, _UCLRp, 0, [], []];
var UpdateContactRequest$ = [
    3,
    n0,
    _UCR,
    0,
    [_CLN, _EA, _TP, _UA, _ADt],
    [[0, 1], [0, 1], () => TopicPreferenceList, 2, 0],
];
var UpdateContactResponse$ = [3, n0, _UCRp, 0, [], []];
var UpdateCustomVerificationEmailTemplateRequest$ = [
    3,
    n0,
    _UCVETR,
    0,
    [_TN, _FEA, _TS, _TC, _SRURL, _FRURL],
    [[0, 1], 0, 0, 0, 0, 0],
];
var UpdateCustomVerificationEmailTemplateResponse$ = [3, n0, _UCVETRp, 0, [], []];
var UpdateEmailIdentityPolicyRequest$ = [
    3,
    n0,
    _UEIPR,
    0,
    [_EI, _PNo, _P],
    [[0, 1], [0, 1], 0],
];
var UpdateEmailIdentityPolicyResponse$ = [3, n0, _UEIPRp, 0, [], []];
var UpdateEmailTemplateRequest$ = [
    3,
    n0,
    _UETR,
    0,
    [_TN, _TC],
    [[0, 1], () => EmailTemplateContent$],
];
var UpdateEmailTemplateResponse$ = [3, n0, _UETRp, 0, [], []];
var UpdateReputationEntityCustomerManagedStatusRequest$ = [
    3,
    n0,
    _URECMSR,
    0,
    [_RET, _RER, _SS],
    [[0, 1], [0, 1], 0],
];
var UpdateReputationEntityCustomerManagedStatusResponse$ = [3, n0, _URECMSRp, 0, [], []];
var UpdateReputationEntityPolicyRequest$ = [
    3,
    n0,
    _UREPR,
    0,
    [_RET, _RER, _REP],
    [[0, 1], [0, 1], 0],
];
var UpdateReputationEntityPolicyResponse$ = [3, n0, _UREPRp, 0, [], []];
var VdmAttributes$ = [
    3,
    n0,
    _VA,
    0,
    [_VE, _DAa, _GA],
    [0, () => DashboardAttributes$, () => GuardianAttributes$],
];
var VdmOptions$ = [
    3,
    n0,
    _VO,
    0,
    [_DOa, _GO],
    [() => DashboardOptions$, () => GuardianOptions$],
];
var VerificationInfo$ = [
    3,
    n0,
    _VI,
    0,
    [_LCT, _LST, _ETr, _SOAR],
    [4, 4, 0, () => SOARecord$],
];
var VolumeStatistics$ = [3, n0, _VS, 0, [_IRC, _SRC, _PI, _PSr], [1, 1, 1, 1]];
var SESv2ServiceException$ = [-3, _sm, "SESv2ServiceException", 0, [], []];
schema.TypeRegistry.for(_sm).registerError(SESv2ServiceException$, SESv2ServiceException);
var AdditionalContactEmailAddresses = [1, n0, _ACEAd, 8, [() => AdditionalContactEmailAddress, 0]];
var AttachmentList = [1, n0, _AL, 0, () => Attachment$];
var BatchGetMetricDataQueries = [1, n0, _BGMDQa, 0, () => BatchGetMetricDataQuery$];
var BlacklistEntries = [1, n0, _BEl, 0, () => BlacklistEntry$];
var BulkEmailEntryList = [1, n0, _BEEL, 0, () => BulkEmailEntry$];
var BulkEmailEntryResultList = [1, n0, _BEERL, 0, () => BulkEmailEntryResult$];
var CloudWatchDimensionConfigurations = [1, n0, _CWDCl, 0, () => CloudWatchDimensionConfiguration$];
var CustomVerificationEmailTemplatesList = [
    1,
    n0,
    _CVETL,
    0,
    () => CustomVerificationEmailTemplateMetadata$,
];
var DailyVolumes = [1, n0, _DVa, 0, () => DailyVolume$];
var DedicatedIpList = [1, n0, _DIL, 0, () => DedicatedIp$];
var DeliverabilityTestReports = [1, n0, _DTReli, 0, () => DeliverabilityTestReport$];
var DomainDeliverabilityCampaignList = [1, n0, _DDCL, 0, () => DomainDeliverabilityCampaign$];
var DomainDeliverabilityTrackingOptions = [
    1,
    n0,
    _DDTOo,
    0,
    () => DomainDeliverabilityTrackingOption$,
];
var DomainIspPlacements = [1, n0, _DIP, 0, () => DomainIspPlacement$];
var EmailAddressFilterList = [1, n0, _EAFL, 0, [() => InsightsEmailAddress, 0]];
var EmailInsightsList = [1, n0, _EIL, 0, [() => EmailInsights$, 0]];
var EmailSubjectFilterList = [1, n0, _ESFL, 0, [() => EmailSubject, 0]];
var EmailTemplateMetadataList = [1, n0, _ETML, 0, () => EmailTemplateMetadata$];
var EventDestinations = [1, n0, _EDven, 0, () => EventDestination$];
var ExportJobSummaryList = [1, n0, _EJSL, 0, () => ExportJobSummary$];
var ExportMetrics = [1, n0, _EMxp, 0, () => ExportMetric$];
var IdentityInfoList = [1, n0, _IIL, 0, () => IdentityInfo$];
var ImportJobSummaryList = [1, n0, _IJSL, 0, () => ImportJobSummary$];
var InsightsEvents = [1, n0, _IEn, 0, () => InsightsEvent$];
var IspPlacements = [1, n0, _IPs, 0, () => IspPlacement$];
var ListOfContactLists = [1, n0, _LOCL, 0, () => ContactList$];
var ListOfContacts = [1, n0, _LOC, 0, () => Contact$];
var MessageHeaderList = [1, n0, _MHL, 0, () => MessageHeader$];
var MessageTagList = [1, n0, _MTL, 0, () => MessageTag$];
var MetricDataErrorList = [1, n0, _MDEL, 0, () => MetricDataError$];
var MetricDataResultList = [1, n0, _MDRL, 0, () => MetricDataResult$];
var MultiRegionEndpoints = [1, n0, _MRE, 0, () => MultiRegionEndpoint$];
var RecommendationsList = [1, n0, _RL, 0, () => Recommendation$];
var ReputationEntitiesList = [1, n0, _REL, 0, () => ReputationEntity$];
var ResourceTenantMetadataList = [1, n0, _RTML, 0, () => ResourceTenantMetadata$];
var Routes = [1, n0, _Ro, 0, () => Route$];
var RoutesDetails = [1, n0, _RDo, 0, () => RouteDetails$];
var SuppressedDestinationSummaries = [1, n0, _SDS, 0, () => SuppressedDestinationSummary$];
var TagList = [1, n0, _TL, 0, () => Tag$];
var TenantInfoList = [1, n0, _TIL, 0, () => TenantInfo$];
var TenantResourceList = [1, n0, _TRL, 0, () => TenantResource$];
var TopicPreferenceList = [1, n0, _TPL, 0, () => TopicPreference$];
var Topics = [1, n0, _To, 0, () => Topic$];
var BlacklistReport = [2, n0, _BR, 0, 0, () => BlacklistEntries];
var BatchGetMetricData$ = [
    9,
    n0,
    _BGMD,
    { [_h]: ["POST", "/v2/email/metrics/batch", 200] },
    () => BatchGetMetricDataRequest$,
    () => BatchGetMetricDataResponse$,
];
var CancelExportJob$ = [
    9,
    n0,
    _CEJ,
    { [_h]: ["PUT", "/v2/email/export-jobs/{JobId}/cancel", 200] },
    () => CancelExportJobRequest$,
    () => CancelExportJobResponse$,
];
var CreateConfigurationSet$ = [
    9,
    n0,
    _CCS,
    { [_h]: ["POST", "/v2/email/configuration-sets", 200] },
    () => CreateConfigurationSetRequest$,
    () => CreateConfigurationSetResponse$,
];
var CreateConfigurationSetEventDestination$ = [
    9,
    n0,
    _CCSED,
    { [_h]: ["POST", "/v2/email/configuration-sets/{ConfigurationSetName}/event-destinations", 200] },
    () => CreateConfigurationSetEventDestinationRequest$,
    () => CreateConfigurationSetEventDestinationResponse$,
];
var CreateContact$ = [
    9,
    n0,
    _CC,
    { [_h]: ["POST", "/v2/email/contact-lists/{ContactListName}/contacts", 200] },
    () => CreateContactRequest$,
    () => CreateContactResponse$,
];
var CreateContactList$ = [
    9,
    n0,
    _CCL,
    { [_h]: ["POST", "/v2/email/contact-lists", 200] },
    () => CreateContactListRequest$,
    () => CreateContactListResponse$,
];
var CreateCustomVerificationEmailTemplate$ = [
    9,
    n0,
    _CCVET,
    { [_h]: ["POST", "/v2/email/custom-verification-email-templates", 200] },
    () => CreateCustomVerificationEmailTemplateRequest$,
    () => CreateCustomVerificationEmailTemplateResponse$,
];
var CreateDedicatedIpPool$ = [
    9,
    n0,
    _CDIP,
    { [_h]: ["POST", "/v2/email/dedicated-ip-pools", 200] },
    () => CreateDedicatedIpPoolRequest$,
    () => CreateDedicatedIpPoolResponse$,
];
var CreateDeliverabilityTestReport$ = [
    9,
    n0,
    _CDTR,
    { [_h]: ["POST", "/v2/email/deliverability-dashboard/test", 200] },
    () => CreateDeliverabilityTestReportRequest$,
    () => CreateDeliverabilityTestReportResponse$,
];
var CreateEmailIdentity$ = [
    9,
    n0,
    _CEI,
    { [_h]: ["POST", "/v2/email/identities", 200] },
    () => CreateEmailIdentityRequest$,
    () => CreateEmailIdentityResponse$,
];
var CreateEmailIdentityPolicy$ = [
    9,
    n0,
    _CEIP,
    { [_h]: ["POST", "/v2/email/identities/{EmailIdentity}/policies/{PolicyName}", 200] },
    () => CreateEmailIdentityPolicyRequest$,
    () => CreateEmailIdentityPolicyResponse$,
];
var CreateEmailTemplate$ = [
    9,
    n0,
    _CET,
    { [_h]: ["POST", "/v2/email/templates", 200] },
    () => CreateEmailTemplateRequest$,
    () => CreateEmailTemplateResponse$,
];
var CreateExportJob$ = [
    9,
    n0,
    _CEJr,
    { [_h]: ["POST", "/v2/email/export-jobs", 200] },
    () => CreateExportJobRequest$,
    () => CreateExportJobResponse$,
];
var CreateImportJob$ = [
    9,
    n0,
    _CIJ,
    { [_h]: ["POST", "/v2/email/import-jobs", 200] },
    () => CreateImportJobRequest$,
    () => CreateImportJobResponse$,
];
var CreateMultiRegionEndpoint$ = [
    9,
    n0,
    _CMRE,
    { [_h]: ["POST", "/v2/email/multi-region-endpoints", 200] },
    () => CreateMultiRegionEndpointRequest$,
    () => CreateMultiRegionEndpointResponse$,
];
var CreateTenant$ = [
    9,
    n0,
    _CTre,
    { [_h]: ["POST", "/v2/email/tenants", 200] },
    () => CreateTenantRequest$,
    () => CreateTenantResponse$,
];
var CreateTenantResourceAssociation$ = [
    9,
    n0,
    _CTRA,
    { [_h]: ["POST", "/v2/email/tenants/resources", 200] },
    () => CreateTenantResourceAssociationRequest$,
    () => CreateTenantResourceAssociationResponse$,
];
var DeleteConfigurationSet$ = [
    9,
    n0,
    _DCS,
    { [_h]: ["DELETE", "/v2/email/configuration-sets/{ConfigurationSetName}", 200] },
    () => DeleteConfigurationSetRequest$,
    () => DeleteConfigurationSetResponse$,
];
var DeleteConfigurationSetEventDestination$ = [
    9,
    n0,
    _DCSED,
    {
        [_h]: [
            "DELETE",
            "/v2/email/configuration-sets/{ConfigurationSetName}/event-destinations/{EventDestinationName}",
            200,
        ],
    },
    () => DeleteConfigurationSetEventDestinationRequest$,
    () => DeleteConfigurationSetEventDestinationResponse$,
];
var DeleteContact$ = [
    9,
    n0,
    _DCel,
    { [_h]: ["DELETE", "/v2/email/contact-lists/{ContactListName}/contacts/{EmailAddress}", 200] },
    () => DeleteContactRequest$,
    () => DeleteContactResponse$,
];
var DeleteContactList$ = [
    9,
    n0,
    _DCL,
    { [_h]: ["DELETE", "/v2/email/contact-lists/{ContactListName}", 200] },
    () => DeleteContactListRequest$,
    () => DeleteContactListResponse$,
];
var DeleteCustomVerificationEmailTemplate$ = [
    9,
    n0,
    _DCVET,
    { [_h]: ["DELETE", "/v2/email/custom-verification-email-templates/{TemplateName}", 200] },
    () => DeleteCustomVerificationEmailTemplateRequest$,
    () => DeleteCustomVerificationEmailTemplateResponse$,
];
var DeleteDedicatedIpPool$ = [
    9,
    n0,
    _DDIP,
    { [_h]: ["DELETE", "/v2/email/dedicated-ip-pools/{PoolName}", 200] },
    () => DeleteDedicatedIpPoolRequest$,
    () => DeleteDedicatedIpPoolResponse$,
];
var DeleteEmailIdentity$ = [
    9,
    n0,
    _DEI,
    { [_h]: ["DELETE", "/v2/email/identities/{EmailIdentity}", 200] },
    () => DeleteEmailIdentityRequest$,
    () => DeleteEmailIdentityResponse$,
];
var DeleteEmailIdentityPolicy$ = [
    9,
    n0,
    _DEIP,
    { [_h]: ["DELETE", "/v2/email/identities/{EmailIdentity}/policies/{PolicyName}", 200] },
    () => DeleteEmailIdentityPolicyRequest$,
    () => DeleteEmailIdentityPolicyResponse$,
];
var DeleteEmailTemplate$ = [
    9,
    n0,
    _DETe,
    { [_h]: ["DELETE", "/v2/email/templates/{TemplateName}", 200] },
    () => DeleteEmailTemplateRequest$,
    () => DeleteEmailTemplateResponse$,
];
var DeleteMultiRegionEndpoint$ = [
    9,
    n0,
    _DMRE,
    { [_h]: ["DELETE", "/v2/email/multi-region-endpoints/{EndpointName}", 200] },
    () => DeleteMultiRegionEndpointRequest$,
    () => DeleteMultiRegionEndpointResponse$,
];
var DeleteSuppressedDestination$ = [
    9,
    n0,
    _DSD,
    { [_h]: ["DELETE", "/v2/email/suppression/addresses/{EmailAddress}", 200] },
    () => DeleteSuppressedDestinationRequest$,
    () => DeleteSuppressedDestinationResponse$,
];
var DeleteTenant$ = [
    9,
    n0,
    _DTe,
    { [_h]: ["POST", "/v2/email/tenants/delete", 200] },
    () => DeleteTenantRequest$,
    () => DeleteTenantResponse$,
];
var DeleteTenantResourceAssociation$ = [
    9,
    n0,
    _DTRA,
    { [_h]: ["POST", "/v2/email/tenants/resources/delete", 200] },
    () => DeleteTenantResourceAssociationRequest$,
    () => DeleteTenantResourceAssociationResponse$,
];
var GetAccount$ = [
    9,
    n0,
    _GAe,
    { [_h]: ["GET", "/v2/email/account", 200] },
    () => GetAccountRequest$,
    () => GetAccountResponse$,
];
var GetBlacklistReports$ = [
    9,
    n0,
    _GBR,
    { [_h]: ["GET", "/v2/email/deliverability-dashboard/blacklist-report", 200] },
    () => GetBlacklistReportsRequest$,
    () => GetBlacklistReportsResponse$,
];
var GetConfigurationSet$ = [
    9,
    n0,
    _GCS,
    { [_h]: ["GET", "/v2/email/configuration-sets/{ConfigurationSetName}", 200] },
    () => GetConfigurationSetRequest$,
    () => GetConfigurationSetResponse$,
];
var GetConfigurationSetEventDestinations$ = [
    9,
    n0,
    _GCSED,
    { [_h]: ["GET", "/v2/email/configuration-sets/{ConfigurationSetName}/event-destinations", 200] },
    () => GetConfigurationSetEventDestinationsRequest$,
    () => GetConfigurationSetEventDestinationsResponse$,
];
var GetContact$ = [
    9,
    n0,
    _GC,
    { [_h]: ["GET", "/v2/email/contact-lists/{ContactListName}/contacts/{EmailAddress}", 200] },
    () => GetContactRequest$,
    () => GetContactResponse$,
];
var GetContactList$ = [
    9,
    n0,
    _GCL,
    { [_h]: ["GET", "/v2/email/contact-lists/{ContactListName}", 200] },
    () => GetContactListRequest$,
    () => GetContactListResponse$,
];
var GetCustomVerificationEmailTemplate$ = [
    9,
    n0,
    _GCVET,
    { [_h]: ["GET", "/v2/email/custom-verification-email-templates/{TemplateName}", 200] },
    () => GetCustomVerificationEmailTemplateRequest$,
    () => GetCustomVerificationEmailTemplateResponse$,
];
var GetDedicatedIp$ = [
    9,
    n0,
    _GDI,
    { [_h]: ["GET", "/v2/email/dedicated-ips/{Ip}", 200] },
    () => GetDedicatedIpRequest$,
    () => GetDedicatedIpResponse$,
];
var GetDedicatedIpPool$ = [
    9,
    n0,
    _GDIP,
    { [_h]: ["GET", "/v2/email/dedicated-ip-pools/{PoolName}", 200] },
    () => GetDedicatedIpPoolRequest$,
    () => GetDedicatedIpPoolResponse$,
];
var GetDedicatedIps$ = [
    9,
    n0,
    _GDIe,
    { [_h]: ["GET", "/v2/email/dedicated-ips", 200] },
    () => GetDedicatedIpsRequest$,
    () => GetDedicatedIpsResponse$,
];
var GetDeliverabilityDashboardOptions$ = [
    9,
    n0,
    _GDDO,
    { [_h]: ["GET", "/v2/email/deliverability-dashboard", 200] },
    () => GetDeliverabilityDashboardOptionsRequest$,
    () => GetDeliverabilityDashboardOptionsResponse$,
];
var GetDeliverabilityTestReport$ = [
    9,
    n0,
    _GDTR,
    { [_h]: ["GET", "/v2/email/deliverability-dashboard/test-reports/{ReportId}", 200] },
    () => GetDeliverabilityTestReportRequest$,
    () => GetDeliverabilityTestReportResponse$,
];
var GetDomainDeliverabilityCampaign$ = [
    9,
    n0,
    _GDDC,
    { [_h]: ["GET", "/v2/email/deliverability-dashboard/campaigns/{CampaignId}", 200] },
    () => GetDomainDeliverabilityCampaignRequest$,
    () => GetDomainDeliverabilityCampaignResponse$,
];
var GetDomainStatisticsReport$ = [
    9,
    n0,
    _GDSR,
    { [_h]: ["GET", "/v2/email/deliverability-dashboard/statistics-report/{Domain}", 200] },
    () => GetDomainStatisticsReportRequest$,
    () => GetDomainStatisticsReportResponse$,
];
var GetEmailAddressInsights$ = [
    9,
    n0,
    _GEAI,
    { [_h]: ["POST", "/v2/email/email-address-insights", 200] },
    () => GetEmailAddressInsightsRequest$,
    () => GetEmailAddressInsightsResponse$,
];
var GetEmailIdentity$ = [
    9,
    n0,
    _GEI,
    { [_h]: ["GET", "/v2/email/identities/{EmailIdentity}", 200] },
    () => GetEmailIdentityRequest$,
    () => GetEmailIdentityResponse$,
];
var GetEmailIdentityPolicies$ = [
    9,
    n0,
    _GEIP,
    { [_h]: ["GET", "/v2/email/identities/{EmailIdentity}/policies", 200] },
    () => GetEmailIdentityPoliciesRequest$,
    () => GetEmailIdentityPoliciesResponse$,
];
var GetEmailTemplate$ = [
    9,
    n0,
    _GET,
    { [_h]: ["GET", "/v2/email/templates/{TemplateName}", 200] },
    () => GetEmailTemplateRequest$,
    () => GetEmailTemplateResponse$,
];
var GetExportJob$ = [
    9,
    n0,
    _GEJ,
    { [_h]: ["GET", "/v2/email/export-jobs/{JobId}", 200] },
    () => GetExportJobRequest$,
    () => GetExportJobResponse$,
];
var GetImportJob$ = [
    9,
    n0,
    _GIJ,
    { [_h]: ["GET", "/v2/email/import-jobs/{JobId}", 200] },
    () => GetImportJobRequest$,
    () => GetImportJobResponse$,
];
var GetMessageInsights$ = [
    9,
    n0,
    _GMI,
    { [_h]: ["GET", "/v2/email/insights/{MessageId}", 200] },
    () => GetMessageInsightsRequest$,
    () => GetMessageInsightsResponse$,
];
var GetMultiRegionEndpoint$ = [
    9,
    n0,
    _GMRE,
    { [_h]: ["GET", "/v2/email/multi-region-endpoints/{EndpointName}", 200] },
    () => GetMultiRegionEndpointRequest$,
    () => GetMultiRegionEndpointResponse$,
];
var GetReputationEntity$ = [
    9,
    n0,
    _GRE,
    { [_h]: ["GET", "/v2/email/reputation/entities/{ReputationEntityType}/{ReputationEntityReference}", 200] },
    () => GetReputationEntityRequest$,
    () => GetReputationEntityResponse$,
];
var GetSuppressedDestination$ = [
    9,
    n0,
    _GSD,
    { [_h]: ["GET", "/v2/email/suppression/addresses/{EmailAddress}", 200] },
    () => GetSuppressedDestinationRequest$,
    () => GetSuppressedDestinationResponse$,
];
var GetTenant$ = [
    9,
    n0,
    _GT,
    { [_h]: ["POST", "/v2/email/tenants/get", 200] },
    () => GetTenantRequest$,
    () => GetTenantResponse$,
];
var ListConfigurationSets$ = [
    9,
    n0,
    _LCS,
    { [_h]: ["GET", "/v2/email/configuration-sets", 200] },
    () => ListConfigurationSetsRequest$,
    () => ListConfigurationSetsResponse$,
];
var ListContactLists$ = [
    9,
    n0,
    _LCL,
    { [_h]: ["GET", "/v2/email/contact-lists", 200] },
    () => ListContactListsRequest$,
    () => ListContactListsResponse$,
];
var ListContacts$ = [
    9,
    n0,
    _LC,
    { [_h]: ["POST", "/v2/email/contact-lists/{ContactListName}/contacts/list", 200] },
    () => ListContactsRequest$,
    () => ListContactsResponse$,
];
var ListCustomVerificationEmailTemplates$ = [
    9,
    n0,
    _LCVET,
    { [_h]: ["GET", "/v2/email/custom-verification-email-templates", 200] },
    () => ListCustomVerificationEmailTemplatesRequest$,
    () => ListCustomVerificationEmailTemplatesResponse$,
];
var ListDedicatedIpPools$ = [
    9,
    n0,
    _LDIP,
    { [_h]: ["GET", "/v2/email/dedicated-ip-pools", 200] },
    () => ListDedicatedIpPoolsRequest$,
    () => ListDedicatedIpPoolsResponse$,
];
var ListDeliverabilityTestReports$ = [
    9,
    n0,
    _LDTR,
    { [_h]: ["GET", "/v2/email/deliverability-dashboard/test-reports", 200] },
    () => ListDeliverabilityTestReportsRequest$,
    () => ListDeliverabilityTestReportsResponse$,
];
var ListDomainDeliverabilityCampaigns$ = [
    9,
    n0,
    _LDDC,
    { [_h]: ["GET", "/v2/email/deliverability-dashboard/domains/{SubscribedDomain}/campaigns", 200] },
    () => ListDomainDeliverabilityCampaignsRequest$,
    () => ListDomainDeliverabilityCampaignsResponse$,
];
var ListEmailIdentities$ = [
    9,
    n0,
    _LEI,
    { [_h]: ["GET", "/v2/email/identities", 200] },
    () => ListEmailIdentitiesRequest$,
    () => ListEmailIdentitiesResponse$,
];
var ListEmailTemplates$ = [
    9,
    n0,
    _LET,
    { [_h]: ["GET", "/v2/email/templates", 200] },
    () => ListEmailTemplatesRequest$,
    () => ListEmailTemplatesResponse$,
];
var ListExportJobs$ = [
    9,
    n0,
    _LEJ,
    { [_h]: ["POST", "/v2/email/list-export-jobs", 200] },
    () => ListExportJobsRequest$,
    () => ListExportJobsResponse$,
];
var ListImportJobs$ = [
    9,
    n0,
    _LIJ,
    { [_h]: ["POST", "/v2/email/import-jobs/list", 200] },
    () => ListImportJobsRequest$,
    () => ListImportJobsResponse$,
];
var ListMultiRegionEndpoints$ = [
    9,
    n0,
    _LMRE,
    { [_h]: ["GET", "/v2/email/multi-region-endpoints", 200] },
    () => ListMultiRegionEndpointsRequest$,
    () => ListMultiRegionEndpointsResponse$,
];
var ListRecommendations$ = [
    9,
    n0,
    _LR,
    { [_h]: ["POST", "/v2/email/vdm/recommendations", 200] },
    () => ListRecommendationsRequest$,
    () => ListRecommendationsResponse$,
];
var ListReputationEntities$ = [
    9,
    n0,
    _LRE,
    { [_h]: ["POST", "/v2/email/reputation/entities", 200] },
    () => ListReputationEntitiesRequest$,
    () => ListReputationEntitiesResponse$,
];
var ListResourceTenants$ = [
    9,
    n0,
    _LRT,
    { [_h]: ["POST", "/v2/email/resources/tenants/list", 200] },
    () => ListResourceTenantsRequest$,
    () => ListResourceTenantsResponse$,
];
var ListSuppressedDestinations$ = [
    9,
    n0,
    _LSD,
    { [_h]: ["GET", "/v2/email/suppression/addresses", 200] },
    () => ListSuppressedDestinationsRequest$,
    () => ListSuppressedDestinationsResponse$,
];
var ListTagsForResource$ = [
    9,
    n0,
    _LTFR,
    { [_h]: ["GET", "/v2/email/tags", 200] },
    () => ListTagsForResourceRequest$,
    () => ListTagsForResourceResponse$,
];
var ListTenantResources$ = [
    9,
    n0,
    _LTRis,
    { [_h]: ["POST", "/v2/email/tenants/resources/list", 200] },
    () => ListTenantResourcesRequest$,
    () => ListTenantResourcesResponse$,
];
var ListTenants$ = [
    9,
    n0,
    _LTi,
    { [_h]: ["POST", "/v2/email/tenants/list", 200] },
    () => ListTenantsRequest$,
    () => ListTenantsResponse$,
];
var PutAccountDedicatedIpWarmupAttributes$ = [
    9,
    n0,
    _PADIWA,
    { [_h]: ["PUT", "/v2/email/account/dedicated-ips/warmup", 200] },
    () => PutAccountDedicatedIpWarmupAttributesRequest$,
    () => PutAccountDedicatedIpWarmupAttributesResponse$,
];
var PutAccountDetails$ = [
    9,
    n0,
    _PAD,
    { [_h]: ["POST", "/v2/email/account/details", 200] },
    () => PutAccountDetailsRequest$,
    () => PutAccountDetailsResponse$,
];
var PutAccountSendingAttributes$ = [
    9,
    n0,
    _PASA,
    { [_h]: ["PUT", "/v2/email/account/sending", 200] },
    () => PutAccountSendingAttributesRequest$,
    () => PutAccountSendingAttributesResponse$,
];
var PutAccountSuppressionAttributes$ = [
    9,
    n0,
    _PASAu,
    { [_h]: ["PUT", "/v2/email/account/suppression", 200] },
    () => PutAccountSuppressionAttributesRequest$,
    () => PutAccountSuppressionAttributesResponse$,
];
var PutAccountVdmAttributes$ = [
    9,
    n0,
    _PAVA,
    { [_h]: ["PUT", "/v2/email/account/vdm", 200] },
    () => PutAccountVdmAttributesRequest$,
    () => PutAccountVdmAttributesResponse$,
];
var PutConfigurationSetArchivingOptions$ = [
    9,
    n0,
    _PCSAO,
    { [_h]: ["PUT", "/v2/email/configuration-sets/{ConfigurationSetName}/archiving-options", 200] },
    () => PutConfigurationSetArchivingOptionsRequest$,
    () => PutConfigurationSetArchivingOptionsResponse$,
];
var PutConfigurationSetDeliveryOptions$ = [
    9,
    n0,
    _PCSDO,
    { [_h]: ["PUT", "/v2/email/configuration-sets/{ConfigurationSetName}/delivery-options", 200] },
    () => PutConfigurationSetDeliveryOptionsRequest$,
    () => PutConfigurationSetDeliveryOptionsResponse$,
];
var PutConfigurationSetReputationOptions$ = [
    9,
    n0,
    _PCSRO,
    { [_h]: ["PUT", "/v2/email/configuration-sets/{ConfigurationSetName}/reputation-options", 200] },
    () => PutConfigurationSetReputationOptionsRequest$,
    () => PutConfigurationSetReputationOptionsResponse$,
];
var PutConfigurationSetSendingOptions$ = [
    9,
    n0,
    _PCSSO,
    { [_h]: ["PUT", "/v2/email/configuration-sets/{ConfigurationSetName}/sending", 200] },
    () => PutConfigurationSetSendingOptionsRequest$,
    () => PutConfigurationSetSendingOptionsResponse$,
];
var PutConfigurationSetSuppressionOptions$ = [
    9,
    n0,
    _PCSSOu,
    { [_h]: ["PUT", "/v2/email/configuration-sets/{ConfigurationSetName}/suppression-options", 200] },
    () => PutConfigurationSetSuppressionOptionsRequest$,
    () => PutConfigurationSetSuppressionOptionsResponse$,
];
var PutConfigurationSetTrackingOptions$ = [
    9,
    n0,
    _PCSTO,
    { [_h]: ["PUT", "/v2/email/configuration-sets/{ConfigurationSetName}/tracking-options", 200] },
    () => PutConfigurationSetTrackingOptionsRequest$,
    () => PutConfigurationSetTrackingOptionsResponse$,
];
var PutConfigurationSetVdmOptions$ = [
    9,
    n0,
    _PCSVO,
    { [_h]: ["PUT", "/v2/email/configuration-sets/{ConfigurationSetName}/vdm-options", 200] },
    () => PutConfigurationSetVdmOptionsRequest$,
    () => PutConfigurationSetVdmOptionsResponse$,
];
var PutDedicatedIpInPool$ = [
    9,
    n0,
    _PDIIP,
    { [_h]: ["PUT", "/v2/email/dedicated-ips/{Ip}/pool", 200] },
    () => PutDedicatedIpInPoolRequest$,
    () => PutDedicatedIpInPoolResponse$,
];
var PutDedicatedIpPoolScalingAttributes$ = [
    9,
    n0,
    _PDIPSA,
    { [_h]: ["PUT", "/v2/email/dedicated-ip-pools/{PoolName}/scaling", 200] },
    () => PutDedicatedIpPoolScalingAttributesRequest$,
    () => PutDedicatedIpPoolScalingAttributesResponse$,
];
var PutDedicatedIpWarmupAttributes$ = [
    9,
    n0,
    _PDIWA,
    { [_h]: ["PUT", "/v2/email/dedicated-ips/{Ip}/warmup", 200] },
    () => PutDedicatedIpWarmupAttributesRequest$,
    () => PutDedicatedIpWarmupAttributesResponse$,
];
var PutDeliverabilityDashboardOption$ = [
    9,
    n0,
    _PDDO,
    { [_h]: ["PUT", "/v2/email/deliverability-dashboard", 200] },
    () => PutDeliverabilityDashboardOptionRequest$,
    () => PutDeliverabilityDashboardOptionResponse$,
];
var PutEmailIdentityConfigurationSetAttributes$ = [
    9,
    n0,
    _PEICSA,
    { [_h]: ["PUT", "/v2/email/identities/{EmailIdentity}/configuration-set", 200] },
    () => PutEmailIdentityConfigurationSetAttributesRequest$,
    () => PutEmailIdentityConfigurationSetAttributesResponse$,
];
var PutEmailIdentityDkimAttributes$ = [
    9,
    n0,
    _PEIDA,
    { [_h]: ["PUT", "/v2/email/identities/{EmailIdentity}/dkim", 200] },
    () => PutEmailIdentityDkimAttributesRequest$,
    () => PutEmailIdentityDkimAttributesResponse$,
];
var PutEmailIdentityDkimSigningAttributes$ = [
    9,
    n0,
    _PEIDSA,
    { [_h]: ["PUT", "/v2/email/identities/{EmailIdentity}/dkim/signing", 200] },
    () => PutEmailIdentityDkimSigningAttributesRequest$,
    () => PutEmailIdentityDkimSigningAttributesResponse$,
];
var PutEmailIdentityFeedbackAttributes$ = [
    9,
    n0,
    _PEIFA,
    { [_h]: ["PUT", "/v2/email/identities/{EmailIdentity}/feedback", 200] },
    () => PutEmailIdentityFeedbackAttributesRequest$,
    () => PutEmailIdentityFeedbackAttributesResponse$,
];
var PutEmailIdentityMailFromAttributes$ = [
    9,
    n0,
    _PEIMFA,
    { [_h]: ["PUT", "/v2/email/identities/{EmailIdentity}/mail-from", 200] },
    () => PutEmailIdentityMailFromAttributesRequest$,
    () => PutEmailIdentityMailFromAttributesResponse$,
];
var PutSuppressedDestination$ = [
    9,
    n0,
    _PSD,
    { [_h]: ["PUT", "/v2/email/suppression/addresses", 200] },
    () => PutSuppressedDestinationRequest$,
    () => PutSuppressedDestinationResponse$,
];
var SendBulkEmail$ = [
    9,
    n0,
    _SBE,
    { [_h]: ["POST", "/v2/email/outbound-bulk-emails", 200] },
    () => SendBulkEmailRequest$,
    () => SendBulkEmailResponse$,
];
var SendCustomVerificationEmail$ = [
    9,
    n0,
    _SCVE,
    { [_h]: ["POST", "/v2/email/outbound-custom-verification-emails", 200] },
    () => SendCustomVerificationEmailRequest$,
    () => SendCustomVerificationEmailResponse$,
];
var SendEmail$ = [
    9,
    n0,
    _SEen,
    { [_h]: ["POST", "/v2/email/outbound-emails", 200] },
    () => SendEmailRequest$,
    () => SendEmailResponse$,
];
var TagResource$ = [
    9,
    n0,
    _TRa,
    { [_h]: ["POST", "/v2/email/tags", 200] },
    () => TagResourceRequest$,
    () => TagResourceResponse$,
];
var TestRenderEmailTemplate$ = [
    9,
    n0,
    _TRET,
    { [_h]: ["POST", "/v2/email/templates/{TemplateName}/render", 200] },
    () => TestRenderEmailTemplateRequest$,
    () => TestRenderEmailTemplateResponse$,
];
var UntagResource$ = [
    9,
    n0,
    _UR,
    { [_h]: ["DELETE", "/v2/email/tags", 200] },
    () => UntagResourceRequest$,
    () => UntagResourceResponse$,
];
var UpdateConfigurationSetEventDestination$ = [
    9,
    n0,
    _UCSED,
    {
        [_h]: ["PUT", "/v2/email/configuration-sets/{ConfigurationSetName}/event-destinations/{EventDestinationName}", 200],
    },
    () => UpdateConfigurationSetEventDestinationRequest$,
    () => UpdateConfigurationSetEventDestinationResponse$,
];
var UpdateContact$ = [
    9,
    n0,
    _UC,
    { [_h]: ["PUT", "/v2/email/contact-lists/{ContactListName}/contacts/{EmailAddress}", 200] },
    () => UpdateContactRequest$,
    () => UpdateContactResponse$,
];
var UpdateContactList$ = [
    9,
    n0,
    _UCL,
    { [_h]: ["PUT", "/v2/email/contact-lists/{ContactListName}", 200] },
    () => UpdateContactListRequest$,
    () => UpdateContactListResponse$,
];
var UpdateCustomVerificationEmailTemplate$ = [
    9,
    n0,
    _UCVET,
    { [_h]: ["PUT", "/v2/email/custom-verification-email-templates/{TemplateName}", 200] },
    () => UpdateCustomVerificationEmailTemplateRequest$,
    () => UpdateCustomVerificationEmailTemplateResponse$,
];
var UpdateEmailIdentityPolicy$ = [
    9,
    n0,
    _UEIP,
    { [_h]: ["PUT", "/v2/email/identities/{EmailIdentity}/policies/{PolicyName}", 200] },
    () => UpdateEmailIdentityPolicyRequest$,
    () => UpdateEmailIdentityPolicyResponse$,
];
var UpdateEmailTemplate$ = [
    9,
    n0,
    _UET,
    { [_h]: ["PUT", "/v2/email/templates/{TemplateName}", 200] },
    () => UpdateEmailTemplateRequest$,
    () => UpdateEmailTemplateResponse$,
];
var UpdateReputationEntityCustomerManagedStatus$ = [
    9,
    n0,
    _URECMS,
    {
        [_h]: [
            "PUT",
            "/v2/email/reputation/entities/{ReputationEntityType}/{ReputationEntityReference}/customer-managed-status",
            200,
        ],
    },
    () => UpdateReputationEntityCustomerManagedStatusRequest$,
    () => UpdateReputationEntityCustomerManagedStatusResponse$,
];
var UpdateReputationEntityPolicy$ = [
    9,
    n0,
    _UREP,
    { [_h]: ["PUT", "/v2/email/reputation/entities/{ReputationEntityType}/{ReputationEntityReference}/policy", 200] },
    () => UpdateReputationEntityPolicyRequest$,
    () => UpdateReputationEntityPolicyResponse$,
];

class BatchGetMetricDataCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "BatchGetMetricData", {})
    .n("SESv2Client", "BatchGetMetricDataCommand")
    .sc(BatchGetMetricData$)
    .build() {
}

class CancelExportJobCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CancelExportJob", {})
    .n("SESv2Client", "CancelExportJobCommand")
    .sc(CancelExportJob$)
    .build() {
}

class CreateConfigurationSetCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateConfigurationSet", {})
    .n("SESv2Client", "CreateConfigurationSetCommand")
    .sc(CreateConfigurationSet$)
    .build() {
}

class CreateConfigurationSetEventDestinationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateConfigurationSetEventDestination", {})
    .n("SESv2Client", "CreateConfigurationSetEventDestinationCommand")
    .sc(CreateConfigurationSetEventDestination$)
    .build() {
}

class CreateContactCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateContact", {})
    .n("SESv2Client", "CreateContactCommand")
    .sc(CreateContact$)
    .build() {
}

class CreateContactListCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateContactList", {})
    .n("SESv2Client", "CreateContactListCommand")
    .sc(CreateContactList$)
    .build() {
}

class CreateCustomVerificationEmailTemplateCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateCustomVerificationEmailTemplate", {})
    .n("SESv2Client", "CreateCustomVerificationEmailTemplateCommand")
    .sc(CreateCustomVerificationEmailTemplate$)
    .build() {
}

class CreateDedicatedIpPoolCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateDedicatedIpPool", {})
    .n("SESv2Client", "CreateDedicatedIpPoolCommand")
    .sc(CreateDedicatedIpPool$)
    .build() {
}

class CreateDeliverabilityTestReportCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateDeliverabilityTestReport", {})
    .n("SESv2Client", "CreateDeliverabilityTestReportCommand")
    .sc(CreateDeliverabilityTestReport$)
    .build() {
}

class CreateEmailIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateEmailIdentity", {})
    .n("SESv2Client", "CreateEmailIdentityCommand")
    .sc(CreateEmailIdentity$)
    .build() {
}

class CreateEmailIdentityPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateEmailIdentityPolicy", {})
    .n("SESv2Client", "CreateEmailIdentityPolicyCommand")
    .sc(CreateEmailIdentityPolicy$)
    .build() {
}

class CreateEmailTemplateCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateEmailTemplate", {})
    .n("SESv2Client", "CreateEmailTemplateCommand")
    .sc(CreateEmailTemplate$)
    .build() {
}

class CreateExportJobCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateExportJob", {})
    .n("SESv2Client", "CreateExportJobCommand")
    .sc(CreateExportJob$)
    .build() {
}

class CreateImportJobCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateImportJob", {})
    .n("SESv2Client", "CreateImportJobCommand")
    .sc(CreateImportJob$)
    .build() {
}

class CreateMultiRegionEndpointCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateMultiRegionEndpoint", {})
    .n("SESv2Client", "CreateMultiRegionEndpointCommand")
    .sc(CreateMultiRegionEndpoint$)
    .build() {
}

class CreateTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateTenant", {})
    .n("SESv2Client", "CreateTenantCommand")
    .sc(CreateTenant$)
    .build() {
}

class CreateTenantResourceAssociationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "CreateTenantResourceAssociation", {})
    .n("SESv2Client", "CreateTenantResourceAssociationCommand")
    .sc(CreateTenantResourceAssociation$)
    .build() {
}

class DeleteConfigurationSetCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteConfigurationSet", {})
    .n("SESv2Client", "DeleteConfigurationSetCommand")
    .sc(DeleteConfigurationSet$)
    .build() {
}

class DeleteConfigurationSetEventDestinationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteConfigurationSetEventDestination", {})
    .n("SESv2Client", "DeleteConfigurationSetEventDestinationCommand")
    .sc(DeleteConfigurationSetEventDestination$)
    .build() {
}

class DeleteContactCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteContact", {})
    .n("SESv2Client", "DeleteContactCommand")
    .sc(DeleteContact$)
    .build() {
}

class DeleteContactListCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteContactList", {})
    .n("SESv2Client", "DeleteContactListCommand")
    .sc(DeleteContactList$)
    .build() {
}

class DeleteCustomVerificationEmailTemplateCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteCustomVerificationEmailTemplate", {})
    .n("SESv2Client", "DeleteCustomVerificationEmailTemplateCommand")
    .sc(DeleteCustomVerificationEmailTemplate$)
    .build() {
}

class DeleteDedicatedIpPoolCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteDedicatedIpPool", {})
    .n("SESv2Client", "DeleteDedicatedIpPoolCommand")
    .sc(DeleteDedicatedIpPool$)
    .build() {
}

class DeleteEmailIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteEmailIdentity", {})
    .n("SESv2Client", "DeleteEmailIdentityCommand")
    .sc(DeleteEmailIdentity$)
    .build() {
}

class DeleteEmailIdentityPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteEmailIdentityPolicy", {})
    .n("SESv2Client", "DeleteEmailIdentityPolicyCommand")
    .sc(DeleteEmailIdentityPolicy$)
    .build() {
}

class DeleteEmailTemplateCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteEmailTemplate", {})
    .n("SESv2Client", "DeleteEmailTemplateCommand")
    .sc(DeleteEmailTemplate$)
    .build() {
}

class DeleteMultiRegionEndpointCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteMultiRegionEndpoint", {})
    .n("SESv2Client", "DeleteMultiRegionEndpointCommand")
    .sc(DeleteMultiRegionEndpoint$)
    .build() {
}

class DeleteSuppressedDestinationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteSuppressedDestination", {})
    .n("SESv2Client", "DeleteSuppressedDestinationCommand")
    .sc(DeleteSuppressedDestination$)
    .build() {
}

class DeleteTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteTenant", {})
    .n("SESv2Client", "DeleteTenantCommand")
    .sc(DeleteTenant$)
    .build() {
}

class DeleteTenantResourceAssociationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "DeleteTenantResourceAssociation", {})
    .n("SESv2Client", "DeleteTenantResourceAssociationCommand")
    .sc(DeleteTenantResourceAssociation$)
    .build() {
}

class GetAccountCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetAccount", {})
    .n("SESv2Client", "GetAccountCommand")
    .sc(GetAccount$)
    .build() {
}

class GetBlacklistReportsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetBlacklistReports", {})
    .n("SESv2Client", "GetBlacklistReportsCommand")
    .sc(GetBlacklistReports$)
    .build() {
}

class GetConfigurationSetCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetConfigurationSet", {})
    .n("SESv2Client", "GetConfigurationSetCommand")
    .sc(GetConfigurationSet$)
    .build() {
}

class GetConfigurationSetEventDestinationsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetConfigurationSetEventDestinations", {})
    .n("SESv2Client", "GetConfigurationSetEventDestinationsCommand")
    .sc(GetConfigurationSetEventDestinations$)
    .build() {
}

class GetContactCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetContact", {})
    .n("SESv2Client", "GetContactCommand")
    .sc(GetContact$)
    .build() {
}

class GetContactListCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetContactList", {})
    .n("SESv2Client", "GetContactListCommand")
    .sc(GetContactList$)
    .build() {
}

class GetCustomVerificationEmailTemplateCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetCustomVerificationEmailTemplate", {})
    .n("SESv2Client", "GetCustomVerificationEmailTemplateCommand")
    .sc(GetCustomVerificationEmailTemplate$)
    .build() {
}

class GetDedicatedIpCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetDedicatedIp", {})
    .n("SESv2Client", "GetDedicatedIpCommand")
    .sc(GetDedicatedIp$)
    .build() {
}

class GetDedicatedIpPoolCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetDedicatedIpPool", {})
    .n("SESv2Client", "GetDedicatedIpPoolCommand")
    .sc(GetDedicatedIpPool$)
    .build() {
}

class GetDedicatedIpsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetDedicatedIps", {})
    .n("SESv2Client", "GetDedicatedIpsCommand")
    .sc(GetDedicatedIps$)
    .build() {
}

class GetDeliverabilityDashboardOptionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetDeliverabilityDashboardOptions", {})
    .n("SESv2Client", "GetDeliverabilityDashboardOptionsCommand")
    .sc(GetDeliverabilityDashboardOptions$)
    .build() {
}

class GetDeliverabilityTestReportCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetDeliverabilityTestReport", {})
    .n("SESv2Client", "GetDeliverabilityTestReportCommand")
    .sc(GetDeliverabilityTestReport$)
    .build() {
}

class GetDomainDeliverabilityCampaignCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetDomainDeliverabilityCampaign", {})
    .n("SESv2Client", "GetDomainDeliverabilityCampaignCommand")
    .sc(GetDomainDeliverabilityCampaign$)
    .build() {
}

class GetDomainStatisticsReportCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetDomainStatisticsReport", {})
    .n("SESv2Client", "GetDomainStatisticsReportCommand")
    .sc(GetDomainStatisticsReport$)
    .build() {
}

class GetEmailAddressInsightsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetEmailAddressInsights", {})
    .n("SESv2Client", "GetEmailAddressInsightsCommand")
    .sc(GetEmailAddressInsights$)
    .build() {
}

class GetEmailIdentityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetEmailIdentity", {})
    .n("SESv2Client", "GetEmailIdentityCommand")
    .sc(GetEmailIdentity$)
    .build() {
}

class GetEmailIdentityPoliciesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetEmailIdentityPolicies", {})
    .n("SESv2Client", "GetEmailIdentityPoliciesCommand")
    .sc(GetEmailIdentityPolicies$)
    .build() {
}

class GetEmailTemplateCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetEmailTemplate", {})
    .n("SESv2Client", "GetEmailTemplateCommand")
    .sc(GetEmailTemplate$)
    .build() {
}

class GetExportJobCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetExportJob", {})
    .n("SESv2Client", "GetExportJobCommand")
    .sc(GetExportJob$)
    .build() {
}

class GetImportJobCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetImportJob", {})
    .n("SESv2Client", "GetImportJobCommand")
    .sc(GetImportJob$)
    .build() {
}

class GetMessageInsightsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetMessageInsights", {})
    .n("SESv2Client", "GetMessageInsightsCommand")
    .sc(GetMessageInsights$)
    .build() {
}

class GetMultiRegionEndpointCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetMultiRegionEndpoint", {})
    .n("SESv2Client", "GetMultiRegionEndpointCommand")
    .sc(GetMultiRegionEndpoint$)
    .build() {
}

class GetReputationEntityCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetReputationEntity", {})
    .n("SESv2Client", "GetReputationEntityCommand")
    .sc(GetReputationEntity$)
    .build() {
}

class GetSuppressedDestinationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetSuppressedDestination", {})
    .n("SESv2Client", "GetSuppressedDestinationCommand")
    .sc(GetSuppressedDestination$)
    .build() {
}

class GetTenantCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "GetTenant", {})
    .n("SESv2Client", "GetTenantCommand")
    .sc(GetTenant$)
    .build() {
}

class ListConfigurationSetsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListConfigurationSets", {})
    .n("SESv2Client", "ListConfigurationSetsCommand")
    .sc(ListConfigurationSets$)
    .build() {
}

class ListContactListsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListContactLists", {})
    .n("SESv2Client", "ListContactListsCommand")
    .sc(ListContactLists$)
    .build() {
}

class ListContactsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListContacts", {})
    .n("SESv2Client", "ListContactsCommand")
    .sc(ListContacts$)
    .build() {
}

class ListCustomVerificationEmailTemplatesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListCustomVerificationEmailTemplates", {})
    .n("SESv2Client", "ListCustomVerificationEmailTemplatesCommand")
    .sc(ListCustomVerificationEmailTemplates$)
    .build() {
}

class ListDedicatedIpPoolsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListDedicatedIpPools", {})
    .n("SESv2Client", "ListDedicatedIpPoolsCommand")
    .sc(ListDedicatedIpPools$)
    .build() {
}

class ListDeliverabilityTestReportsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListDeliverabilityTestReports", {})
    .n("SESv2Client", "ListDeliverabilityTestReportsCommand")
    .sc(ListDeliverabilityTestReports$)
    .build() {
}

class ListDomainDeliverabilityCampaignsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListDomainDeliverabilityCampaigns", {})
    .n("SESv2Client", "ListDomainDeliverabilityCampaignsCommand")
    .sc(ListDomainDeliverabilityCampaigns$)
    .build() {
}

class ListEmailIdentitiesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListEmailIdentities", {})
    .n("SESv2Client", "ListEmailIdentitiesCommand")
    .sc(ListEmailIdentities$)
    .build() {
}

class ListEmailTemplatesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListEmailTemplates", {})
    .n("SESv2Client", "ListEmailTemplatesCommand")
    .sc(ListEmailTemplates$)
    .build() {
}

class ListExportJobsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListExportJobs", {})
    .n("SESv2Client", "ListExportJobsCommand")
    .sc(ListExportJobs$)
    .build() {
}

class ListImportJobsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListImportJobs", {})
    .n("SESv2Client", "ListImportJobsCommand")
    .sc(ListImportJobs$)
    .build() {
}

class ListMultiRegionEndpointsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListMultiRegionEndpoints", {})
    .n("SESv2Client", "ListMultiRegionEndpointsCommand")
    .sc(ListMultiRegionEndpoints$)
    .build() {
}

class ListRecommendationsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListRecommendations", {})
    .n("SESv2Client", "ListRecommendationsCommand")
    .sc(ListRecommendations$)
    .build() {
}

class ListReputationEntitiesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListReputationEntities", {})
    .n("SESv2Client", "ListReputationEntitiesCommand")
    .sc(ListReputationEntities$)
    .build() {
}

class ListResourceTenantsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListResourceTenants", {})
    .n("SESv2Client", "ListResourceTenantsCommand")
    .sc(ListResourceTenants$)
    .build() {
}

class ListSuppressedDestinationsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListSuppressedDestinations", {})
    .n("SESv2Client", "ListSuppressedDestinationsCommand")
    .sc(ListSuppressedDestinations$)
    .build() {
}

class ListTagsForResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListTagsForResource", {})
    .n("SESv2Client", "ListTagsForResourceCommand")
    .sc(ListTagsForResource$)
    .build() {
}

class ListTenantResourcesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListTenantResources", {})
    .n("SESv2Client", "ListTenantResourcesCommand")
    .sc(ListTenantResources$)
    .build() {
}

class ListTenantsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "ListTenants", {})
    .n("SESv2Client", "ListTenantsCommand")
    .sc(ListTenants$)
    .build() {
}

class PutAccountDedicatedIpWarmupAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutAccountDedicatedIpWarmupAttributes", {})
    .n("SESv2Client", "PutAccountDedicatedIpWarmupAttributesCommand")
    .sc(PutAccountDedicatedIpWarmupAttributes$)
    .build() {
}

class PutAccountDetailsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutAccountDetails", {})
    .n("SESv2Client", "PutAccountDetailsCommand")
    .sc(PutAccountDetails$)
    .build() {
}

class PutAccountSendingAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutAccountSendingAttributes", {})
    .n("SESv2Client", "PutAccountSendingAttributesCommand")
    .sc(PutAccountSendingAttributes$)
    .build() {
}

class PutAccountSuppressionAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutAccountSuppressionAttributes", {})
    .n("SESv2Client", "PutAccountSuppressionAttributesCommand")
    .sc(PutAccountSuppressionAttributes$)
    .build() {
}

class PutAccountVdmAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutAccountVdmAttributes", {})
    .n("SESv2Client", "PutAccountVdmAttributesCommand")
    .sc(PutAccountVdmAttributes$)
    .build() {
}

class PutConfigurationSetArchivingOptionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutConfigurationSetArchivingOptions", {})
    .n("SESv2Client", "PutConfigurationSetArchivingOptionsCommand")
    .sc(PutConfigurationSetArchivingOptions$)
    .build() {
}

class PutConfigurationSetDeliveryOptionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutConfigurationSetDeliveryOptions", {})
    .n("SESv2Client", "PutConfigurationSetDeliveryOptionsCommand")
    .sc(PutConfigurationSetDeliveryOptions$)
    .build() {
}

class PutConfigurationSetReputationOptionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutConfigurationSetReputationOptions", {})
    .n("SESv2Client", "PutConfigurationSetReputationOptionsCommand")
    .sc(PutConfigurationSetReputationOptions$)
    .build() {
}

class PutConfigurationSetSendingOptionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutConfigurationSetSendingOptions", {})
    .n("SESv2Client", "PutConfigurationSetSendingOptionsCommand")
    .sc(PutConfigurationSetSendingOptions$)
    .build() {
}

class PutConfigurationSetSuppressionOptionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutConfigurationSetSuppressionOptions", {})
    .n("SESv2Client", "PutConfigurationSetSuppressionOptionsCommand")
    .sc(PutConfigurationSetSuppressionOptions$)
    .build() {
}

class PutConfigurationSetTrackingOptionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutConfigurationSetTrackingOptions", {})
    .n("SESv2Client", "PutConfigurationSetTrackingOptionsCommand")
    .sc(PutConfigurationSetTrackingOptions$)
    .build() {
}

class PutConfigurationSetVdmOptionsCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutConfigurationSetVdmOptions", {})
    .n("SESv2Client", "PutConfigurationSetVdmOptionsCommand")
    .sc(PutConfigurationSetVdmOptions$)
    .build() {
}

class PutDedicatedIpInPoolCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutDedicatedIpInPool", {})
    .n("SESv2Client", "PutDedicatedIpInPoolCommand")
    .sc(PutDedicatedIpInPool$)
    .build() {
}

class PutDedicatedIpPoolScalingAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutDedicatedIpPoolScalingAttributes", {})
    .n("SESv2Client", "PutDedicatedIpPoolScalingAttributesCommand")
    .sc(PutDedicatedIpPoolScalingAttributes$)
    .build() {
}

class PutDedicatedIpWarmupAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutDedicatedIpWarmupAttributes", {})
    .n("SESv2Client", "PutDedicatedIpWarmupAttributesCommand")
    .sc(PutDedicatedIpWarmupAttributes$)
    .build() {
}

class PutDeliverabilityDashboardOptionCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutDeliverabilityDashboardOption", {})
    .n("SESv2Client", "PutDeliverabilityDashboardOptionCommand")
    .sc(PutDeliverabilityDashboardOption$)
    .build() {
}

class PutEmailIdentityConfigurationSetAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutEmailIdentityConfigurationSetAttributes", {})
    .n("SESv2Client", "PutEmailIdentityConfigurationSetAttributesCommand")
    .sc(PutEmailIdentityConfigurationSetAttributes$)
    .build() {
}

class PutEmailIdentityDkimAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutEmailIdentityDkimAttributes", {})
    .n("SESv2Client", "PutEmailIdentityDkimAttributesCommand")
    .sc(PutEmailIdentityDkimAttributes$)
    .build() {
}

class PutEmailIdentityDkimSigningAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutEmailIdentityDkimSigningAttributes", {})
    .n("SESv2Client", "PutEmailIdentityDkimSigningAttributesCommand")
    .sc(PutEmailIdentityDkimSigningAttributes$)
    .build() {
}

class PutEmailIdentityFeedbackAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutEmailIdentityFeedbackAttributes", {})
    .n("SESv2Client", "PutEmailIdentityFeedbackAttributesCommand")
    .sc(PutEmailIdentityFeedbackAttributes$)
    .build() {
}

class PutEmailIdentityMailFromAttributesCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutEmailIdentityMailFromAttributes", {})
    .n("SESv2Client", "PutEmailIdentityMailFromAttributesCommand")
    .sc(PutEmailIdentityMailFromAttributes$)
    .build() {
}

class PutSuppressedDestinationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "PutSuppressedDestination", {})
    .n("SESv2Client", "PutSuppressedDestinationCommand")
    .sc(PutSuppressedDestination$)
    .build() {
}

class SendBulkEmailCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    EndpointId: { type: "contextParams", name: "EndpointId" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "SendBulkEmail", {})
    .n("SESv2Client", "SendBulkEmailCommand")
    .sc(SendBulkEmail$)
    .build() {
}

class SendCustomVerificationEmailCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "SendCustomVerificationEmail", {})
    .n("SESv2Client", "SendCustomVerificationEmailCommand")
    .sc(SendCustomVerificationEmail$)
    .build() {
}

class SendEmailCommand extends smithyClient.Command
    .classBuilder()
    .ep({
    ...commonParams,
    EndpointId: { type: "contextParams", name: "EndpointId" },
})
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "SendEmail", {})
    .n("SESv2Client", "SendEmailCommand")
    .sc(SendEmail$)
    .build() {
}

class TagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "TagResource", {})
    .n("SESv2Client", "TagResourceCommand")
    .sc(TagResource$)
    .build() {
}

class TestRenderEmailTemplateCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "TestRenderEmailTemplate", {})
    .n("SESv2Client", "TestRenderEmailTemplateCommand")
    .sc(TestRenderEmailTemplate$)
    .build() {
}

class UntagResourceCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "UntagResource", {})
    .n("SESv2Client", "UntagResourceCommand")
    .sc(UntagResource$)
    .build() {
}

class UpdateConfigurationSetEventDestinationCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "UpdateConfigurationSetEventDestination", {})
    .n("SESv2Client", "UpdateConfigurationSetEventDestinationCommand")
    .sc(UpdateConfigurationSetEventDestination$)
    .build() {
}

class UpdateContactCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "UpdateContact", {})
    .n("SESv2Client", "UpdateContactCommand")
    .sc(UpdateContact$)
    .build() {
}

class UpdateContactListCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "UpdateContactList", {})
    .n("SESv2Client", "UpdateContactListCommand")
    .sc(UpdateContactList$)
    .build() {
}

class UpdateCustomVerificationEmailTemplateCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "UpdateCustomVerificationEmailTemplate", {})
    .n("SESv2Client", "UpdateCustomVerificationEmailTemplateCommand")
    .sc(UpdateCustomVerificationEmailTemplate$)
    .build() {
}

class UpdateEmailIdentityPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "UpdateEmailIdentityPolicy", {})
    .n("SESv2Client", "UpdateEmailIdentityPolicyCommand")
    .sc(UpdateEmailIdentityPolicy$)
    .build() {
}

class UpdateEmailTemplateCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "UpdateEmailTemplate", {})
    .n("SESv2Client", "UpdateEmailTemplateCommand")
    .sc(UpdateEmailTemplate$)
    .build() {
}

class UpdateReputationEntityCustomerManagedStatusCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "UpdateReputationEntityCustomerManagedStatus", {})
    .n("SESv2Client", "UpdateReputationEntityCustomerManagedStatusCommand")
    .sc(UpdateReputationEntityCustomerManagedStatus$)
    .build() {
}

class UpdateReputationEntityPolicyCommand extends smithyClient.Command
    .classBuilder()
    .ep(commonParams)
    .m(function (Command, cs, config, o) {
    return [middlewareEndpoint.getEndpointPlugin(config, Command.getEndpointParameterInstructions())];
})
    .s("SimpleEmailService_v2", "UpdateReputationEntityPolicy", {})
    .n("SESv2Client", "UpdateReputationEntityPolicyCommand")
    .sc(UpdateReputationEntityPolicy$)
    .build() {
}

const commands = {
    BatchGetMetricDataCommand,
    CancelExportJobCommand,
    CreateConfigurationSetCommand,
    CreateConfigurationSetEventDestinationCommand,
    CreateContactCommand,
    CreateContactListCommand,
    CreateCustomVerificationEmailTemplateCommand,
    CreateDedicatedIpPoolCommand,
    CreateDeliverabilityTestReportCommand,
    CreateEmailIdentityCommand,
    CreateEmailIdentityPolicyCommand,
    CreateEmailTemplateCommand,
    CreateExportJobCommand,
    CreateImportJobCommand,
    CreateMultiRegionEndpointCommand,
    CreateTenantCommand,
    CreateTenantResourceAssociationCommand,
    DeleteConfigurationSetCommand,
    DeleteConfigurationSetEventDestinationCommand,
    DeleteContactCommand,
    DeleteContactListCommand,
    DeleteCustomVerificationEmailTemplateCommand,
    DeleteDedicatedIpPoolCommand,
    DeleteEmailIdentityCommand,
    DeleteEmailIdentityPolicyCommand,
    DeleteEmailTemplateCommand,
    DeleteMultiRegionEndpointCommand,
    DeleteSuppressedDestinationCommand,
    DeleteTenantCommand,
    DeleteTenantResourceAssociationCommand,
    GetAccountCommand,
    GetBlacklistReportsCommand,
    GetConfigurationSetCommand,
    GetConfigurationSetEventDestinationsCommand,
    GetContactCommand,
    GetContactListCommand,
    GetCustomVerificationEmailTemplateCommand,
    GetDedicatedIpCommand,
    GetDedicatedIpPoolCommand,
    GetDedicatedIpsCommand,
    GetDeliverabilityDashboardOptionsCommand,
    GetDeliverabilityTestReportCommand,
    GetDomainDeliverabilityCampaignCommand,
    GetDomainStatisticsReportCommand,
    GetEmailAddressInsightsCommand,
    GetEmailIdentityCommand,
    GetEmailIdentityPoliciesCommand,
    GetEmailTemplateCommand,
    GetExportJobCommand,
    GetImportJobCommand,
    GetMessageInsightsCommand,
    GetMultiRegionEndpointCommand,
    GetReputationEntityCommand,
    GetSuppressedDestinationCommand,
    GetTenantCommand,
    ListConfigurationSetsCommand,
    ListContactListsCommand,
    ListContactsCommand,
    ListCustomVerificationEmailTemplatesCommand,
    ListDedicatedIpPoolsCommand,
    ListDeliverabilityTestReportsCommand,
    ListDomainDeliverabilityCampaignsCommand,
    ListEmailIdentitiesCommand,
    ListEmailTemplatesCommand,
    ListExportJobsCommand,
    ListImportJobsCommand,
    ListMultiRegionEndpointsCommand,
    ListRecommendationsCommand,
    ListReputationEntitiesCommand,
    ListResourceTenantsCommand,
    ListSuppressedDestinationsCommand,
    ListTagsForResourceCommand,
    ListTenantResourcesCommand,
    ListTenantsCommand,
    PutAccountDedicatedIpWarmupAttributesCommand,
    PutAccountDetailsCommand,
    PutAccountSendingAttributesCommand,
    PutAccountSuppressionAttributesCommand,
    PutAccountVdmAttributesCommand,
    PutConfigurationSetArchivingOptionsCommand,
    PutConfigurationSetDeliveryOptionsCommand,
    PutConfigurationSetReputationOptionsCommand,
    PutConfigurationSetSendingOptionsCommand,
    PutConfigurationSetSuppressionOptionsCommand,
    PutConfigurationSetTrackingOptionsCommand,
    PutConfigurationSetVdmOptionsCommand,
    PutDedicatedIpInPoolCommand,
    PutDedicatedIpPoolScalingAttributesCommand,
    PutDedicatedIpWarmupAttributesCommand,
    PutDeliverabilityDashboardOptionCommand,
    PutEmailIdentityConfigurationSetAttributesCommand,
    PutEmailIdentityDkimAttributesCommand,
    PutEmailIdentityDkimSigningAttributesCommand,
    PutEmailIdentityFeedbackAttributesCommand,
    PutEmailIdentityMailFromAttributesCommand,
    PutSuppressedDestinationCommand,
    SendBulkEmailCommand,
    SendCustomVerificationEmailCommand,
    SendEmailCommand,
    TagResourceCommand,
    TestRenderEmailTemplateCommand,
    UntagResourceCommand,
    UpdateConfigurationSetEventDestinationCommand,
    UpdateContactCommand,
    UpdateContactListCommand,
    UpdateCustomVerificationEmailTemplateCommand,
    UpdateEmailIdentityPolicyCommand,
    UpdateEmailTemplateCommand,
    UpdateReputationEntityCustomerManagedStatusCommand,
    UpdateReputationEntityPolicyCommand,
};
class SESv2 extends SESv2Client {
}
smithyClient.createAggregatedClient(commands, SESv2);

const paginateGetDedicatedIps = core.createPaginator(SESv2Client, GetDedicatedIpsCommand, "NextToken", "NextToken", "PageSize");

const paginateListConfigurationSets = core.createPaginator(SESv2Client, ListConfigurationSetsCommand, "NextToken", "NextToken", "PageSize");

const paginateListContactLists = core.createPaginator(SESv2Client, ListContactListsCommand, "NextToken", "NextToken", "PageSize");

const paginateListContacts = core.createPaginator(SESv2Client, ListContactsCommand, "NextToken", "NextToken", "PageSize");

const paginateListCustomVerificationEmailTemplates = core.createPaginator(SESv2Client, ListCustomVerificationEmailTemplatesCommand, "NextToken", "NextToken", "PageSize");

const paginateListDedicatedIpPools = core.createPaginator(SESv2Client, ListDedicatedIpPoolsCommand, "NextToken", "NextToken", "PageSize");

const paginateListDeliverabilityTestReports = core.createPaginator(SESv2Client, ListDeliverabilityTestReportsCommand, "NextToken", "NextToken", "PageSize");

const paginateListDomainDeliverabilityCampaigns = core.createPaginator(SESv2Client, ListDomainDeliverabilityCampaignsCommand, "NextToken", "NextToken", "PageSize");

const paginateListEmailIdentities = core.createPaginator(SESv2Client, ListEmailIdentitiesCommand, "NextToken", "NextToken", "PageSize");

const paginateListEmailTemplates = core.createPaginator(SESv2Client, ListEmailTemplatesCommand, "NextToken", "NextToken", "PageSize");

const paginateListExportJobs = core.createPaginator(SESv2Client, ListExportJobsCommand, "NextToken", "NextToken", "PageSize");

const paginateListImportJobs = core.createPaginator(SESv2Client, ListImportJobsCommand, "NextToken", "NextToken", "PageSize");

const paginateListMultiRegionEndpoints = core.createPaginator(SESv2Client, ListMultiRegionEndpointsCommand, "NextToken", "NextToken", "PageSize");

const paginateListRecommendations = core.createPaginator(SESv2Client, ListRecommendationsCommand, "NextToken", "NextToken", "PageSize");

const paginateListReputationEntities = core.createPaginator(SESv2Client, ListReputationEntitiesCommand, "NextToken", "NextToken", "PageSize");

const paginateListResourceTenants = core.createPaginator(SESv2Client, ListResourceTenantsCommand, "NextToken", "NextToken", "PageSize");

const paginateListSuppressedDestinations = core.createPaginator(SESv2Client, ListSuppressedDestinationsCommand, "NextToken", "NextToken", "PageSize");

const paginateListTenantResources = core.createPaginator(SESv2Client, ListTenantResourcesCommand, "NextToken", "NextToken", "PageSize");

const paginateListTenants = core.createPaginator(SESv2Client, ListTenantsCommand, "NextToken", "NextToken", "PageSize");

const ContactLanguage = {
    EN: "EN",
    JA: "JA",
};
const MailType = {
    MARKETING: "MARKETING",
    TRANSACTIONAL: "TRANSACTIONAL",
};
const ReviewStatus = {
    DENIED: "DENIED",
    FAILED: "FAILED",
    GRANTED: "GRANTED",
    PENDING: "PENDING",
};
const AttachmentContentDisposition = {
    ATTACHMENT: "ATTACHMENT",
    INLINE: "INLINE",
};
const AttachmentContentTransferEncoding = {
    BASE64: "BASE64",
    QUOTED_PRINTABLE: "QUOTED_PRINTABLE",
    SEVEN_BIT: "SEVEN_BIT",
};
const MetricDimensionName = {
    CONFIGURATION_SET: "CONFIGURATION_SET",
    EMAIL_IDENTITY: "EMAIL_IDENTITY",
    ISP: "ISP",
};
const Metric = {
    CLICK: "CLICK",
    COMPLAINT: "COMPLAINT",
    DELIVERY: "DELIVERY",
    DELIVERY_CLICK: "DELIVERY_CLICK",
    DELIVERY_COMPLAINT: "DELIVERY_COMPLAINT",
    DELIVERY_OPEN: "DELIVERY_OPEN",
    OPEN: "OPEN",
    PERMANENT_BOUNCE: "PERMANENT_BOUNCE",
    SEND: "SEND",
    TRANSIENT_BOUNCE: "TRANSIENT_BOUNCE",
};
const MetricNamespace = {
    VDM: "VDM",
};
const QueryErrorCode = {
    ACCESS_DENIED: "ACCESS_DENIED",
    INTERNAL_FAILURE: "INTERNAL_FAILURE",
};
const BehaviorOnMxFailure = {
    REJECT_MESSAGE: "REJECT_MESSAGE",
    USE_DEFAULT_VALUE: "USE_DEFAULT_VALUE",
};
const BounceType = {
    PERMANENT: "PERMANENT",
    TRANSIENT: "TRANSIENT",
    UNDETERMINED: "UNDETERMINED",
};
const BulkEmailStatus = {
    ACCOUNT_DAILY_QUOTA_EXCEEDED: "ACCOUNT_DAILY_QUOTA_EXCEEDED",
    ACCOUNT_SENDING_PAUSED: "ACCOUNT_SENDING_PAUSED",
    ACCOUNT_SUSPENDED: "ACCOUNT_SUSPENDED",
    ACCOUNT_THROTTLED: "ACCOUNT_THROTTLED",
    CONFIGURATION_SET_NOT_FOUND: "CONFIGURATION_SET_NOT_FOUND",
    CONFIGURATION_SET_SENDING_PAUSED: "CONFIGURATION_SET_SENDING_PAUSED",
    FAILED: "FAILED",
    INVALID_PARAMETER: "INVALID_PARAMETER",
    INVALID_SENDING_POOL_NAME: "INVALID_SENDING_POOL_NAME",
    MAIL_FROM_DOMAIN_NOT_VERIFIED: "MAIL_FROM_DOMAIN_NOT_VERIFIED",
    MESSAGE_REJECTED: "MESSAGE_REJECTED",
    SUCCESS: "SUCCESS",
    TEMPLATE_NOT_FOUND: "TEMPLATE_NOT_FOUND",
    TRANSIENT_FAILURE: "TRANSIENT_FAILURE",
};
const DimensionValueSource = {
    EMAIL_HEADER: "EMAIL_HEADER",
    LINK_TAG: "LINK_TAG",
    MESSAGE_TAG: "MESSAGE_TAG",
};
const SubscriptionStatus = {
    OPT_IN: "OPT_IN",
    OPT_OUT: "OPT_OUT",
};
const ContactListImportAction = {
    DELETE: "DELETE",
    PUT: "PUT",
};
const TlsPolicy = {
    OPTIONAL: "OPTIONAL",
    REQUIRE: "REQUIRE",
};
const SuppressionListReason = {
    BOUNCE: "BOUNCE",
    COMPLAINT: "COMPLAINT",
};
const FeatureStatus = {
    DISABLED: "DISABLED",
    ENABLED: "ENABLED",
};
const SuppressionConfidenceVerdictThreshold = {
    HIGH: "HIGH",
    MANAGED: "MANAGED",
    MEDIUM: "MEDIUM",
};
const HttpsPolicy = {
    OPTIONAL: "OPTIONAL",
    REQUIRE: "REQUIRE",
    REQUIRE_OPEN_ONLY: "REQUIRE_OPEN_ONLY",
};
const EventType = {
    BOUNCE: "BOUNCE",
    CLICK: "CLICK",
    COMPLAINT: "COMPLAINT",
    DELIVERY: "DELIVERY",
    DELIVERY_DELAY: "DELIVERY_DELAY",
    OPEN: "OPEN",
    REJECT: "REJECT",
    RENDERING_FAILURE: "RENDERING_FAILURE",
    SEND: "SEND",
    SUBSCRIPTION: "SUBSCRIPTION",
};
const ScalingMode = {
    MANAGED: "MANAGED",
    STANDARD: "STANDARD",
};
const DeliverabilityTestStatus = {
    COMPLETED: "COMPLETED",
    IN_PROGRESS: "IN_PROGRESS",
};
const DkimSigningAttributesOrigin = {
    AWS_SES: "AWS_SES",
    AWS_SES_AF_SOUTH_1: "AWS_SES_AF_SOUTH_1",
    AWS_SES_AP_NORTHEAST_1: "AWS_SES_AP_NORTHEAST_1",
    AWS_SES_AP_NORTHEAST_2: "AWS_SES_AP_NORTHEAST_2",
    AWS_SES_AP_NORTHEAST_3: "AWS_SES_AP_NORTHEAST_3",
    AWS_SES_AP_SOUTHEAST_1: "AWS_SES_AP_SOUTHEAST_1",
    AWS_SES_AP_SOUTHEAST_2: "AWS_SES_AP_SOUTHEAST_2",
    AWS_SES_AP_SOUTHEAST_3: "AWS_SES_AP_SOUTHEAST_3",
    AWS_SES_AP_SOUTHEAST_5: "AWS_SES_AP_SOUTHEAST_5",
    AWS_SES_AP_SOUTH_1: "AWS_SES_AP_SOUTH_1",
    AWS_SES_AP_SOUTH_2: "AWS_SES_AP_SOUTH_2",
    AWS_SES_CA_CENTRAL_1: "AWS_SES_CA_CENTRAL_1",
    AWS_SES_CA_WEST_1: "AWS_SES_CA_WEST_1",
    AWS_SES_EU_CENTRAL_1: "AWS_SES_EU_CENTRAL_1",
    AWS_SES_EU_CENTRAL_2: "AWS_SES_EU_CENTRAL_2",
    AWS_SES_EU_NORTH_1: "AWS_SES_EU_NORTH_1",
    AWS_SES_EU_SOUTH_1: "AWS_SES_EU_SOUTH_1",
    AWS_SES_EU_WEST_1: "AWS_SES_EU_WEST_1",
    AWS_SES_EU_WEST_2: "AWS_SES_EU_WEST_2",
    AWS_SES_EU_WEST_3: "AWS_SES_EU_WEST_3",
    AWS_SES_IL_CENTRAL_1: "AWS_SES_IL_CENTRAL_1",
    AWS_SES_ME_CENTRAL_1: "AWS_SES_ME_CENTRAL_1",
    AWS_SES_ME_SOUTH_1: "AWS_SES_ME_SOUTH_1",
    AWS_SES_SA_EAST_1: "AWS_SES_SA_EAST_1",
    AWS_SES_US_EAST_1: "AWS_SES_US_EAST_1",
    AWS_SES_US_EAST_2: "AWS_SES_US_EAST_2",
    AWS_SES_US_WEST_1: "AWS_SES_US_WEST_1",
    AWS_SES_US_WEST_2: "AWS_SES_US_WEST_2",
    EXTERNAL: "EXTERNAL",
};
const DkimSigningKeyLength = {
    RSA_1024_BIT: "RSA_1024_BIT",
    RSA_2048_BIT: "RSA_2048_BIT",
};
const DkimStatus = {
    FAILED: "FAILED",
    NOT_STARTED: "NOT_STARTED",
    PENDING: "PENDING",
    SUCCESS: "SUCCESS",
    TEMPORARY_FAILURE: "TEMPORARY_FAILURE",
};
const IdentityType = {
    DOMAIN: "DOMAIN",
    EMAIL_ADDRESS: "EMAIL_ADDRESS",
    MANAGED_DOMAIN: "MANAGED_DOMAIN",
};
const DeliveryEventType = {
    COMPLAINT: "COMPLAINT",
    DELIVERY: "DELIVERY",
    PERMANENT_BOUNCE: "PERMANENT_BOUNCE",
    SEND: "SEND",
    TRANSIENT_BOUNCE: "TRANSIENT_BOUNCE",
    UNDETERMINED_BOUNCE: "UNDETERMINED_BOUNCE",
};
const EngagementEventType = {
    CLICK: "CLICK",
    OPEN: "OPEN",
};
const MetricAggregation = {
    RATE: "RATE",
    VOLUME: "VOLUME",
};
const DataFormat = {
    CSV: "CSV",
    JSON: "JSON",
};
const SuppressionListImportAction = {
    DELETE: "DELETE",
    PUT: "PUT",
};
const Status = {
    CREATING: "CREATING",
    DELETING: "DELETING",
    FAILED: "FAILED",
    READY: "READY",
};
const SendingStatus = {
    DISABLED: "DISABLED",
    ENABLED: "ENABLED",
    REINSTATED: "REINSTATED",
};
const WarmupStatus = {
    DONE: "DONE",
    IN_PROGRESS: "IN_PROGRESS",
    NOT_APPLICABLE: "NOT_APPLICABLE",
};
const DeliverabilityDashboardAccountStatus = {
    ACTIVE: "ACTIVE",
    DISABLED: "DISABLED",
    PENDING_EXPIRATION: "PENDING_EXPIRATION",
};
const EmailAddressInsightsConfidenceVerdict = {
    HIGH: "HIGH",
    LOW: "LOW",
    MEDIUM: "MEDIUM",
};
const ExportSourceType = {
    MESSAGE_INSIGHTS: "MESSAGE_INSIGHTS",
    METRICS_DATA: "METRICS_DATA",
};
const JobStatus = {
    CANCELLED: "CANCELLED",
    COMPLETED: "COMPLETED",
    CREATED: "CREATED",
    FAILED: "FAILED",
    PROCESSING: "PROCESSING",
};
const MailFromDomainStatus = {
    FAILED: "FAILED",
    PENDING: "PENDING",
    SUCCESS: "SUCCESS",
    TEMPORARY_FAILURE: "TEMPORARY_FAILURE",
};
const VerificationError = {
    DNS_SERVER_ERROR: "DNS_SERVER_ERROR",
    HOST_NOT_FOUND: "HOST_NOT_FOUND",
    INVALID_VALUE: "INVALID_VALUE",
    REPLICATION_ACCESS_DENIED: "REPLICATION_ACCESS_DENIED",
    REPLICATION_PRIMARY_BYO_DKIM_NOT_SUPPORTED: "REPLICATION_PRIMARY_BYO_DKIM_NOT_SUPPORTED",
    REPLICATION_PRIMARY_INVALID_REGION: "REPLICATION_PRIMARY_INVALID_REGION",
    REPLICATION_PRIMARY_NOT_FOUND: "REPLICATION_PRIMARY_NOT_FOUND",
    REPLICATION_REPLICA_AS_PRIMARY_NOT_SUPPORTED: "REPLICATION_REPLICA_AS_PRIMARY_NOT_SUPPORTED",
    SERVICE_ERROR: "SERVICE_ERROR",
    TYPE_NOT_FOUND: "TYPE_NOT_FOUND",
};
const VerificationStatus = {
    FAILED: "FAILED",
    NOT_STARTED: "NOT_STARTED",
    PENDING: "PENDING",
    SUCCESS: "SUCCESS",
    TEMPORARY_FAILURE: "TEMPORARY_FAILURE",
};
const ReputationEntityType = {
    RESOURCE: "RESOURCE",
};
const RecommendationImpact = {
    HIGH: "HIGH",
    LOW: "LOW",
};
const ImportDestinationType = {
    CONTACT_LIST: "CONTACT_LIST",
    SUPPRESSION_LIST: "SUPPRESSION_LIST",
};
const ListRecommendationsFilterKey = {
    IMPACT: "IMPACT",
    RESOURCE_ARN: "RESOURCE_ARN",
    STATUS: "STATUS",
    TYPE: "TYPE",
};
const RecommendationStatus = {
    FIXED: "FIXED",
    OPEN: "OPEN",
};
const RecommendationType = {
    BIMI: "BIMI",
    BOUNCE: "BOUNCE",
    COMPLAINT: "COMPLAINT",
    DKIM: "DKIM",
    DMARC: "DMARC",
    FEEDBACK_3P: "FEEDBACK_3P",
    IP_LISTING: "IP_LISTING",
    SPF: "SPF",
};
const ReputationEntityFilterKey = {
    ENTITY_REFERENCE_PREFIX: "ENTITY_REFERENCE_PREFIX",
    ENTITY_TYPE: "ENTITY_TYPE",
    REPUTATION_IMPACT: "REPUTATION_IMPACT",
    STATUS: "SENDING_STATUS",
};
const ListTenantResourcesFilterKey = {
    RESOURCE_TYPE: "RESOURCE_TYPE",
};
const ResourceType = {
    CONFIGURATION_SET: "CONFIGURATION_SET",
    EMAIL_IDENTITY: "EMAIL_IDENTITY",
    EMAIL_TEMPLATE: "EMAIL_TEMPLATE",
};

Object.defineProperty(exports, "$Command", {
    enumerable: true,
    get: function () { return smithyClient.Command; }
});
Object.defineProperty(exports, "__Client", {
    enumerable: true,
    get: function () { return smithyClient.Client; }
});
exports.AccountDetails$ = AccountDetails$;
exports.AccountSuspendedException = AccountSuspendedException;
exports.AccountSuspendedException$ = AccountSuspendedException$;
exports.AlreadyExistsException = AlreadyExistsException;
exports.AlreadyExistsException$ = AlreadyExistsException$;
exports.ArchivingOptions$ = ArchivingOptions$;
exports.Attachment$ = Attachment$;
exports.AttachmentContentDisposition = AttachmentContentDisposition;
exports.AttachmentContentTransferEncoding = AttachmentContentTransferEncoding;
exports.BadRequestException = BadRequestException;
exports.BadRequestException$ = BadRequestException$;
exports.BatchGetMetricData$ = BatchGetMetricData$;
exports.BatchGetMetricDataCommand = BatchGetMetricDataCommand;
exports.BatchGetMetricDataQuery$ = BatchGetMetricDataQuery$;
exports.BatchGetMetricDataRequest$ = BatchGetMetricDataRequest$;
exports.BatchGetMetricDataResponse$ = BatchGetMetricDataResponse$;
exports.BehaviorOnMxFailure = BehaviorOnMxFailure;
exports.BlacklistEntry$ = BlacklistEntry$;
exports.Body$ = Body$;
exports.Bounce$ = Bounce$;
exports.BounceType = BounceType;
exports.BulkEmailContent$ = BulkEmailContent$;
exports.BulkEmailEntry$ = BulkEmailEntry$;
exports.BulkEmailEntryResult$ = BulkEmailEntryResult$;
exports.BulkEmailStatus = BulkEmailStatus;
exports.CancelExportJob$ = CancelExportJob$;
exports.CancelExportJobCommand = CancelExportJobCommand;
exports.CancelExportJobRequest$ = CancelExportJobRequest$;
exports.CancelExportJobResponse$ = CancelExportJobResponse$;
exports.CloudWatchDestination$ = CloudWatchDestination$;
exports.CloudWatchDimensionConfiguration$ = CloudWatchDimensionConfiguration$;
exports.Complaint$ = Complaint$;
exports.ConcurrentModificationException = ConcurrentModificationException;
exports.ConcurrentModificationException$ = ConcurrentModificationException$;
exports.ConflictException = ConflictException;
exports.ConflictException$ = ConflictException$;
exports.Contact$ = Contact$;
exports.ContactLanguage = ContactLanguage;
exports.ContactList$ = ContactList$;
exports.ContactListDestination$ = ContactListDestination$;
exports.ContactListImportAction = ContactListImportAction;
exports.Content$ = Content$;
exports.CreateConfigurationSet$ = CreateConfigurationSet$;
exports.CreateConfigurationSetCommand = CreateConfigurationSetCommand;
exports.CreateConfigurationSetEventDestination$ = CreateConfigurationSetEventDestination$;
exports.CreateConfigurationSetEventDestinationCommand = CreateConfigurationSetEventDestinationCommand;
exports.CreateConfigurationSetEventDestinationRequest$ = CreateConfigurationSetEventDestinationRequest$;
exports.CreateConfigurationSetEventDestinationResponse$ = CreateConfigurationSetEventDestinationResponse$;
exports.CreateConfigurationSetRequest$ = CreateConfigurationSetRequest$;
exports.CreateConfigurationSetResponse$ = CreateConfigurationSetResponse$;
exports.CreateContact$ = CreateContact$;
exports.CreateContactCommand = CreateContactCommand;
exports.CreateContactList$ = CreateContactList$;
exports.CreateContactListCommand = CreateContactListCommand;
exports.CreateContactListRequest$ = CreateContactListRequest$;
exports.CreateContactListResponse$ = CreateContactListResponse$;
exports.CreateContactRequest$ = CreateContactRequest$;
exports.CreateContactResponse$ = CreateContactResponse$;
exports.CreateCustomVerificationEmailTemplate$ = CreateCustomVerificationEmailTemplate$;
exports.CreateCustomVerificationEmailTemplateCommand = CreateCustomVerificationEmailTemplateCommand;
exports.CreateCustomVerificationEmailTemplateRequest$ = CreateCustomVerificationEmailTemplateRequest$;
exports.CreateCustomVerificationEmailTemplateResponse$ = CreateCustomVerificationEmailTemplateResponse$;
exports.CreateDedicatedIpPool$ = CreateDedicatedIpPool$;
exports.CreateDedicatedIpPoolCommand = CreateDedicatedIpPoolCommand;
exports.CreateDedicatedIpPoolRequest$ = CreateDedicatedIpPoolRequest$;
exports.CreateDedicatedIpPoolResponse$ = CreateDedicatedIpPoolResponse$;
exports.CreateDeliverabilityTestReport$ = CreateDeliverabilityTestReport$;
exports.CreateDeliverabilityTestReportCommand = CreateDeliverabilityTestReportCommand;
exports.CreateDeliverabilityTestReportRequest$ = CreateDeliverabilityTestReportRequest$;
exports.CreateDeliverabilityTestReportResponse$ = CreateDeliverabilityTestReportResponse$;
exports.CreateEmailIdentity$ = CreateEmailIdentity$;
exports.CreateEmailIdentityCommand = CreateEmailIdentityCommand;
exports.CreateEmailIdentityPolicy$ = CreateEmailIdentityPolicy$;
exports.CreateEmailIdentityPolicyCommand = CreateEmailIdentityPolicyCommand;
exports.CreateEmailIdentityPolicyRequest$ = CreateEmailIdentityPolicyRequest$;
exports.CreateEmailIdentityPolicyResponse$ = CreateEmailIdentityPolicyResponse$;
exports.CreateEmailIdentityRequest$ = CreateEmailIdentityRequest$;
exports.CreateEmailIdentityResponse$ = CreateEmailIdentityResponse$;
exports.CreateEmailTemplate$ = CreateEmailTemplate$;
exports.CreateEmailTemplateCommand = CreateEmailTemplateCommand;
exports.CreateEmailTemplateRequest$ = CreateEmailTemplateRequest$;
exports.CreateEmailTemplateResponse$ = CreateEmailTemplateResponse$;
exports.CreateExportJob$ = CreateExportJob$;
exports.CreateExportJobCommand = CreateExportJobCommand;
exports.CreateExportJobRequest$ = CreateExportJobRequest$;
exports.CreateExportJobResponse$ = CreateExportJobResponse$;
exports.CreateImportJob$ = CreateImportJob$;
exports.CreateImportJobCommand = CreateImportJobCommand;
exports.CreateImportJobRequest$ = CreateImportJobRequest$;
exports.CreateImportJobResponse$ = CreateImportJobResponse$;
exports.CreateMultiRegionEndpoint$ = CreateMultiRegionEndpoint$;
exports.CreateMultiRegionEndpointCommand = CreateMultiRegionEndpointCommand;
exports.CreateMultiRegionEndpointRequest$ = CreateMultiRegionEndpointRequest$;
exports.CreateMultiRegionEndpointResponse$ = CreateMultiRegionEndpointResponse$;
exports.CreateTenant$ = CreateTenant$;
exports.CreateTenantCommand = CreateTenantCommand;
exports.CreateTenantRequest$ = CreateTenantRequest$;
exports.CreateTenantResourceAssociation$ = CreateTenantResourceAssociation$;
exports.CreateTenantResourceAssociationCommand = CreateTenantResourceAssociationCommand;
exports.CreateTenantResourceAssociationRequest$ = CreateTenantResourceAssociationRequest$;
exports.CreateTenantResourceAssociationResponse$ = CreateTenantResourceAssociationResponse$;
exports.CreateTenantResponse$ = CreateTenantResponse$;
exports.CustomVerificationEmailTemplateMetadata$ = CustomVerificationEmailTemplateMetadata$;
exports.DailyVolume$ = DailyVolume$;
exports.DashboardAttributes$ = DashboardAttributes$;
exports.DashboardOptions$ = DashboardOptions$;
exports.DataFormat = DataFormat;
exports.DedicatedIp$ = DedicatedIp$;
exports.DedicatedIpPool$ = DedicatedIpPool$;
exports.DeleteConfigurationSet$ = DeleteConfigurationSet$;
exports.DeleteConfigurationSetCommand = DeleteConfigurationSetCommand;
exports.DeleteConfigurationSetEventDestination$ = DeleteConfigurationSetEventDestination$;
exports.DeleteConfigurationSetEventDestinationCommand = DeleteConfigurationSetEventDestinationCommand;
exports.DeleteConfigurationSetEventDestinationRequest$ = DeleteConfigurationSetEventDestinationRequest$;
exports.DeleteConfigurationSetEventDestinationResponse$ = DeleteConfigurationSetEventDestinationResponse$;
exports.DeleteConfigurationSetRequest$ = DeleteConfigurationSetRequest$;
exports.DeleteConfigurationSetResponse$ = DeleteConfigurationSetResponse$;
exports.DeleteContact$ = DeleteContact$;
exports.DeleteContactCommand = DeleteContactCommand;
exports.DeleteContactList$ = DeleteContactList$;
exports.DeleteContactListCommand = DeleteContactListCommand;
exports.DeleteContactListRequest$ = DeleteContactListRequest$;
exports.DeleteContactListResponse$ = DeleteContactListResponse$;
exports.DeleteContactRequest$ = DeleteContactRequest$;
exports.DeleteContactResponse$ = DeleteContactResponse$;
exports.DeleteCustomVerificationEmailTemplate$ = DeleteCustomVerificationEmailTemplate$;
exports.DeleteCustomVerificationEmailTemplateCommand = DeleteCustomVerificationEmailTemplateCommand;
exports.DeleteCustomVerificationEmailTemplateRequest$ = DeleteCustomVerificationEmailTemplateRequest$;
exports.DeleteCustomVerificationEmailTemplateResponse$ = DeleteCustomVerificationEmailTemplateResponse$;
exports.DeleteDedicatedIpPool$ = DeleteDedicatedIpPool$;
exports.DeleteDedicatedIpPoolCommand = DeleteDedicatedIpPoolCommand;
exports.DeleteDedicatedIpPoolRequest$ = DeleteDedicatedIpPoolRequest$;
exports.DeleteDedicatedIpPoolResponse$ = DeleteDedicatedIpPoolResponse$;
exports.DeleteEmailIdentity$ = DeleteEmailIdentity$;
exports.DeleteEmailIdentityCommand = DeleteEmailIdentityCommand;
exports.DeleteEmailIdentityPolicy$ = DeleteEmailIdentityPolicy$;
exports.DeleteEmailIdentityPolicyCommand = DeleteEmailIdentityPolicyCommand;
exports.DeleteEmailIdentityPolicyRequest$ = DeleteEmailIdentityPolicyRequest$;
exports.DeleteEmailIdentityPolicyResponse$ = DeleteEmailIdentityPolicyResponse$;
exports.DeleteEmailIdentityRequest$ = DeleteEmailIdentityRequest$;
exports.DeleteEmailIdentityResponse$ = DeleteEmailIdentityResponse$;
exports.DeleteEmailTemplate$ = DeleteEmailTemplate$;
exports.DeleteEmailTemplateCommand = DeleteEmailTemplateCommand;
exports.DeleteEmailTemplateRequest$ = DeleteEmailTemplateRequest$;
exports.DeleteEmailTemplateResponse$ = DeleteEmailTemplateResponse$;
exports.DeleteMultiRegionEndpoint$ = DeleteMultiRegionEndpoint$;
exports.DeleteMultiRegionEndpointCommand = DeleteMultiRegionEndpointCommand;
exports.DeleteMultiRegionEndpointRequest$ = DeleteMultiRegionEndpointRequest$;
exports.DeleteMultiRegionEndpointResponse$ = DeleteMultiRegionEndpointResponse$;
exports.DeleteSuppressedDestination$ = DeleteSuppressedDestination$;
exports.DeleteSuppressedDestinationCommand = DeleteSuppressedDestinationCommand;
exports.DeleteSuppressedDestinationRequest$ = DeleteSuppressedDestinationRequest$;
exports.DeleteSuppressedDestinationResponse$ = DeleteSuppressedDestinationResponse$;
exports.DeleteTenant$ = DeleteTenant$;
exports.DeleteTenantCommand = DeleteTenantCommand;
exports.DeleteTenantRequest$ = DeleteTenantRequest$;
exports.DeleteTenantResourceAssociation$ = DeleteTenantResourceAssociation$;
exports.DeleteTenantResourceAssociationCommand = DeleteTenantResourceAssociationCommand;
exports.DeleteTenantResourceAssociationRequest$ = DeleteTenantResourceAssociationRequest$;
exports.DeleteTenantResourceAssociationResponse$ = DeleteTenantResourceAssociationResponse$;
exports.DeleteTenantResponse$ = DeleteTenantResponse$;
exports.DeliverabilityDashboardAccountStatus = DeliverabilityDashboardAccountStatus;
exports.DeliverabilityTestReport$ = DeliverabilityTestReport$;
exports.DeliverabilityTestStatus = DeliverabilityTestStatus;
exports.DeliveryEventType = DeliveryEventType;
exports.DeliveryOptions$ = DeliveryOptions$;
exports.Destination$ = Destination$;
exports.Details$ = Details$;
exports.DimensionValueSource = DimensionValueSource;
exports.DkimAttributes$ = DkimAttributes$;
exports.DkimSigningAttributes$ = DkimSigningAttributes$;
exports.DkimSigningAttributesOrigin = DkimSigningAttributesOrigin;
exports.DkimSigningKeyLength = DkimSigningKeyLength;
exports.DkimStatus = DkimStatus;
exports.DomainDeliverabilityCampaign$ = DomainDeliverabilityCampaign$;
exports.DomainDeliverabilityTrackingOption$ = DomainDeliverabilityTrackingOption$;
exports.DomainIspPlacement$ = DomainIspPlacement$;
exports.EmailAddressInsightsConfidenceVerdict = EmailAddressInsightsConfidenceVerdict;
exports.EmailAddressInsightsMailboxEvaluations$ = EmailAddressInsightsMailboxEvaluations$;
exports.EmailAddressInsightsVerdict$ = EmailAddressInsightsVerdict$;
exports.EmailContent$ = EmailContent$;
exports.EmailInsights$ = EmailInsights$;
exports.EmailTemplateContent$ = EmailTemplateContent$;
exports.EmailTemplateMetadata$ = EmailTemplateMetadata$;
exports.EngagementEventType = EngagementEventType;
exports.EventBridgeDestination$ = EventBridgeDestination$;
exports.EventDestination$ = EventDestination$;
exports.EventDestinationDefinition$ = EventDestinationDefinition$;
exports.EventDetails$ = EventDetails$;
exports.EventType = EventType;
exports.ExportDataSource$ = ExportDataSource$;
exports.ExportDestination$ = ExportDestination$;
exports.ExportJobSummary$ = ExportJobSummary$;
exports.ExportMetric$ = ExportMetric$;
exports.ExportSourceType = ExportSourceType;
exports.ExportStatistics$ = ExportStatistics$;
exports.FailureInfo$ = FailureInfo$;
exports.FeatureStatus = FeatureStatus;
exports.GetAccount$ = GetAccount$;
exports.GetAccountCommand = GetAccountCommand;
exports.GetAccountRequest$ = GetAccountRequest$;
exports.GetAccountResponse$ = GetAccountResponse$;
exports.GetBlacklistReports$ = GetBlacklistReports$;
exports.GetBlacklistReportsCommand = GetBlacklistReportsCommand;
exports.GetBlacklistReportsRequest$ = GetBlacklistReportsRequest$;
exports.GetBlacklistReportsResponse$ = GetBlacklistReportsResponse$;
exports.GetConfigurationSet$ = GetConfigurationSet$;
exports.GetConfigurationSetCommand = GetConfigurationSetCommand;
exports.GetConfigurationSetEventDestinations$ = GetConfigurationSetEventDestinations$;
exports.GetConfigurationSetEventDestinationsCommand = GetConfigurationSetEventDestinationsCommand;
exports.GetConfigurationSetEventDestinationsRequest$ = GetConfigurationSetEventDestinationsRequest$;
exports.GetConfigurationSetEventDestinationsResponse$ = GetConfigurationSetEventDestinationsResponse$;
exports.GetConfigurationSetRequest$ = GetConfigurationSetRequest$;
exports.GetConfigurationSetResponse$ = GetConfigurationSetResponse$;
exports.GetContact$ = GetContact$;
exports.GetContactCommand = GetContactCommand;
exports.GetContactList$ = GetContactList$;
exports.GetContactListCommand = GetContactListCommand;
exports.GetContactListRequest$ = GetContactListRequest$;
exports.GetContactListResponse$ = GetContactListResponse$;
exports.GetContactRequest$ = GetContactRequest$;
exports.GetContactResponse$ = GetContactResponse$;
exports.GetCustomVerificationEmailTemplate$ = GetCustomVerificationEmailTemplate$;
exports.GetCustomVerificationEmailTemplateCommand = GetCustomVerificationEmailTemplateCommand;
exports.GetCustomVerificationEmailTemplateRequest$ = GetCustomVerificationEmailTemplateRequest$;
exports.GetCustomVerificationEmailTemplateResponse$ = GetCustomVerificationEmailTemplateResponse$;
exports.GetDedicatedIp$ = GetDedicatedIp$;
exports.GetDedicatedIpCommand = GetDedicatedIpCommand;
exports.GetDedicatedIpPool$ = GetDedicatedIpPool$;
exports.GetDedicatedIpPoolCommand = GetDedicatedIpPoolCommand;
exports.GetDedicatedIpPoolRequest$ = GetDedicatedIpPoolRequest$;
exports.GetDedicatedIpPoolResponse$ = GetDedicatedIpPoolResponse$;
exports.GetDedicatedIpRequest$ = GetDedicatedIpRequest$;
exports.GetDedicatedIpResponse$ = GetDedicatedIpResponse$;
exports.GetDedicatedIps$ = GetDedicatedIps$;
exports.GetDedicatedIpsCommand = GetDedicatedIpsCommand;
exports.GetDedicatedIpsRequest$ = GetDedicatedIpsRequest$;
exports.GetDedicatedIpsResponse$ = GetDedicatedIpsResponse$;
exports.GetDeliverabilityDashboardOptions$ = GetDeliverabilityDashboardOptions$;
exports.GetDeliverabilityDashboardOptionsCommand = GetDeliverabilityDashboardOptionsCommand;
exports.GetDeliverabilityDashboardOptionsRequest$ = GetDeliverabilityDashboardOptionsRequest$;
exports.GetDeliverabilityDashboardOptionsResponse$ = GetDeliverabilityDashboardOptionsResponse$;
exports.GetDeliverabilityTestReport$ = GetDeliverabilityTestReport$;
exports.GetDeliverabilityTestReportCommand = GetDeliverabilityTestReportCommand;
exports.GetDeliverabilityTestReportRequest$ = GetDeliverabilityTestReportRequest$;
exports.GetDeliverabilityTestReportResponse$ = GetDeliverabilityTestReportResponse$;
exports.GetDomainDeliverabilityCampaign$ = GetDomainDeliverabilityCampaign$;
exports.GetDomainDeliverabilityCampaignCommand = GetDomainDeliverabilityCampaignCommand;
exports.GetDomainDeliverabilityCampaignRequest$ = GetDomainDeliverabilityCampaignRequest$;
exports.GetDomainDeliverabilityCampaignResponse$ = GetDomainDeliverabilityCampaignResponse$;
exports.GetDomainStatisticsReport$ = GetDomainStatisticsReport$;
exports.GetDomainStatisticsReportCommand = GetDomainStatisticsReportCommand;
exports.GetDomainStatisticsReportRequest$ = GetDomainStatisticsReportRequest$;
exports.GetDomainStatisticsReportResponse$ = GetDomainStatisticsReportResponse$;
exports.GetEmailAddressInsights$ = GetEmailAddressInsights$;
exports.GetEmailAddressInsightsCommand = GetEmailAddressInsightsCommand;
exports.GetEmailAddressInsightsRequest$ = GetEmailAddressInsightsRequest$;
exports.GetEmailAddressInsightsResponse$ = GetEmailAddressInsightsResponse$;
exports.GetEmailIdentity$ = GetEmailIdentity$;
exports.GetEmailIdentityCommand = GetEmailIdentityCommand;
exports.GetEmailIdentityPolicies$ = GetEmailIdentityPolicies$;
exports.GetEmailIdentityPoliciesCommand = GetEmailIdentityPoliciesCommand;
exports.GetEmailIdentityPoliciesRequest$ = GetEmailIdentityPoliciesRequest$;
exports.GetEmailIdentityPoliciesResponse$ = GetEmailIdentityPoliciesResponse$;
exports.GetEmailIdentityRequest$ = GetEmailIdentityRequest$;
exports.GetEmailIdentityResponse$ = GetEmailIdentityResponse$;
exports.GetEmailTemplate$ = GetEmailTemplate$;
exports.GetEmailTemplateCommand = GetEmailTemplateCommand;
exports.GetEmailTemplateRequest$ = GetEmailTemplateRequest$;
exports.GetEmailTemplateResponse$ = GetEmailTemplateResponse$;
exports.GetExportJob$ = GetExportJob$;
exports.GetExportJobCommand = GetExportJobCommand;
exports.GetExportJobRequest$ = GetExportJobRequest$;
exports.GetExportJobResponse$ = GetExportJobResponse$;
exports.GetImportJob$ = GetImportJob$;
exports.GetImportJobCommand = GetImportJobCommand;
exports.GetImportJobRequest$ = GetImportJobRequest$;
exports.GetImportJobResponse$ = GetImportJobResponse$;
exports.GetMessageInsights$ = GetMessageInsights$;
exports.GetMessageInsightsCommand = GetMessageInsightsCommand;
exports.GetMessageInsightsRequest$ = GetMessageInsightsRequest$;
exports.GetMessageInsightsResponse$ = GetMessageInsightsResponse$;
exports.GetMultiRegionEndpoint$ = GetMultiRegionEndpoint$;
exports.GetMultiRegionEndpointCommand = GetMultiRegionEndpointCommand;
exports.GetMultiRegionEndpointRequest$ = GetMultiRegionEndpointRequest$;
exports.GetMultiRegionEndpointResponse$ = GetMultiRegionEndpointResponse$;
exports.GetReputationEntity$ = GetReputationEntity$;
exports.GetReputationEntityCommand = GetReputationEntityCommand;
exports.GetReputationEntityRequest$ = GetReputationEntityRequest$;
exports.GetReputationEntityResponse$ = GetReputationEntityResponse$;
exports.GetSuppressedDestination$ = GetSuppressedDestination$;
exports.GetSuppressedDestinationCommand = GetSuppressedDestinationCommand;
exports.GetSuppressedDestinationRequest$ = GetSuppressedDestinationRequest$;
exports.GetSuppressedDestinationResponse$ = GetSuppressedDestinationResponse$;
exports.GetTenant$ = GetTenant$;
exports.GetTenantCommand = GetTenantCommand;
exports.GetTenantRequest$ = GetTenantRequest$;
exports.GetTenantResponse$ = GetTenantResponse$;
exports.GuardianAttributes$ = GuardianAttributes$;
exports.GuardianOptions$ = GuardianOptions$;
exports.HttpsPolicy = HttpsPolicy;
exports.IdentityInfo$ = IdentityInfo$;
exports.IdentityType = IdentityType;
exports.ImportDataSource$ = ImportDataSource$;
exports.ImportDestination$ = ImportDestination$;
exports.ImportDestinationType = ImportDestinationType;
exports.ImportJobSummary$ = ImportJobSummary$;
exports.InboxPlacementTrackingOption$ = InboxPlacementTrackingOption$;
exports.InsightsEvent$ = InsightsEvent$;
exports.InternalServiceErrorException = InternalServiceErrorException;
exports.InternalServiceErrorException$ = InternalServiceErrorException$;
exports.InvalidNextTokenException = InvalidNextTokenException;
exports.InvalidNextTokenException$ = InvalidNextTokenException$;
exports.IspPlacement$ = IspPlacement$;
exports.JobStatus = JobStatus;
exports.KinesisFirehoseDestination$ = KinesisFirehoseDestination$;
exports.LimitExceededException = LimitExceededException;
exports.LimitExceededException$ = LimitExceededException$;
exports.ListConfigurationSets$ = ListConfigurationSets$;
exports.ListConfigurationSetsCommand = ListConfigurationSetsCommand;
exports.ListConfigurationSetsRequest$ = ListConfigurationSetsRequest$;
exports.ListConfigurationSetsResponse$ = ListConfigurationSetsResponse$;
exports.ListContactLists$ = ListContactLists$;
exports.ListContactListsCommand = ListContactListsCommand;
exports.ListContactListsRequest$ = ListContactListsRequest$;
exports.ListContactListsResponse$ = ListContactListsResponse$;
exports.ListContacts$ = ListContacts$;
exports.ListContactsCommand = ListContactsCommand;
exports.ListContactsFilter$ = ListContactsFilter$;
exports.ListContactsRequest$ = ListContactsRequest$;
exports.ListContactsResponse$ = ListContactsResponse$;
exports.ListCustomVerificationEmailTemplates$ = ListCustomVerificationEmailTemplates$;
exports.ListCustomVerificationEmailTemplatesCommand = ListCustomVerificationEmailTemplatesCommand;
exports.ListCustomVerificationEmailTemplatesRequest$ = ListCustomVerificationEmailTemplatesRequest$;
exports.ListCustomVerificationEmailTemplatesResponse$ = ListCustomVerificationEmailTemplatesResponse$;
exports.ListDedicatedIpPools$ = ListDedicatedIpPools$;
exports.ListDedicatedIpPoolsCommand = ListDedicatedIpPoolsCommand;
exports.ListDedicatedIpPoolsRequest$ = ListDedicatedIpPoolsRequest$;
exports.ListDedicatedIpPoolsResponse$ = ListDedicatedIpPoolsResponse$;
exports.ListDeliverabilityTestReports$ = ListDeliverabilityTestReports$;
exports.ListDeliverabilityTestReportsCommand = ListDeliverabilityTestReportsCommand;
exports.ListDeliverabilityTestReportsRequest$ = ListDeliverabilityTestReportsRequest$;
exports.ListDeliverabilityTestReportsResponse$ = ListDeliverabilityTestReportsResponse$;
exports.ListDomainDeliverabilityCampaigns$ = ListDomainDeliverabilityCampaigns$;
exports.ListDomainDeliverabilityCampaignsCommand = ListDomainDeliverabilityCampaignsCommand;
exports.ListDomainDeliverabilityCampaignsRequest$ = ListDomainDeliverabilityCampaignsRequest$;
exports.ListDomainDeliverabilityCampaignsResponse$ = ListDomainDeliverabilityCampaignsResponse$;
exports.ListEmailIdentities$ = ListEmailIdentities$;
exports.ListEmailIdentitiesCommand = ListEmailIdentitiesCommand;
exports.ListEmailIdentitiesRequest$ = ListEmailIdentitiesRequest$;
exports.ListEmailIdentitiesResponse$ = ListEmailIdentitiesResponse$;
exports.ListEmailTemplates$ = ListEmailTemplates$;
exports.ListEmailTemplatesCommand = ListEmailTemplatesCommand;
exports.ListEmailTemplatesRequest$ = ListEmailTemplatesRequest$;
exports.ListEmailTemplatesResponse$ = ListEmailTemplatesResponse$;
exports.ListExportJobs$ = ListExportJobs$;
exports.ListExportJobsCommand = ListExportJobsCommand;
exports.ListExportJobsRequest$ = ListExportJobsRequest$;
exports.ListExportJobsResponse$ = ListExportJobsResponse$;
exports.ListImportJobs$ = ListImportJobs$;
exports.ListImportJobsCommand = ListImportJobsCommand;
exports.ListImportJobsRequest$ = ListImportJobsRequest$;
exports.ListImportJobsResponse$ = ListImportJobsResponse$;
exports.ListManagementOptions$ = ListManagementOptions$;
exports.ListMultiRegionEndpoints$ = ListMultiRegionEndpoints$;
exports.ListMultiRegionEndpointsCommand = ListMultiRegionEndpointsCommand;
exports.ListMultiRegionEndpointsRequest$ = ListMultiRegionEndpointsRequest$;
exports.ListMultiRegionEndpointsResponse$ = ListMultiRegionEndpointsResponse$;
exports.ListRecommendations$ = ListRecommendations$;
exports.ListRecommendationsCommand = ListRecommendationsCommand;
exports.ListRecommendationsFilterKey = ListRecommendationsFilterKey;
exports.ListRecommendationsRequest$ = ListRecommendationsRequest$;
exports.ListRecommendationsResponse$ = ListRecommendationsResponse$;
exports.ListReputationEntities$ = ListReputationEntities$;
exports.ListReputationEntitiesCommand = ListReputationEntitiesCommand;
exports.ListReputationEntitiesRequest$ = ListReputationEntitiesRequest$;
exports.ListReputationEntitiesResponse$ = ListReputationEntitiesResponse$;
exports.ListResourceTenants$ = ListResourceTenants$;
exports.ListResourceTenantsCommand = ListResourceTenantsCommand;
exports.ListResourceTenantsRequest$ = ListResourceTenantsRequest$;
exports.ListResourceTenantsResponse$ = ListResourceTenantsResponse$;
exports.ListSuppressedDestinations$ = ListSuppressedDestinations$;
exports.ListSuppressedDestinationsCommand = ListSuppressedDestinationsCommand;
exports.ListSuppressedDestinationsRequest$ = ListSuppressedDestinationsRequest$;
exports.ListSuppressedDestinationsResponse$ = ListSuppressedDestinationsResponse$;
exports.ListTagsForResource$ = ListTagsForResource$;
exports.ListTagsForResourceCommand = ListTagsForResourceCommand;
exports.ListTagsForResourceRequest$ = ListTagsForResourceRequest$;
exports.ListTagsForResourceResponse$ = ListTagsForResourceResponse$;
exports.ListTenantResources$ = ListTenantResources$;
exports.ListTenantResourcesCommand = ListTenantResourcesCommand;
exports.ListTenantResourcesFilterKey = ListTenantResourcesFilterKey;
exports.ListTenantResourcesRequest$ = ListTenantResourcesRequest$;
exports.ListTenantResourcesResponse$ = ListTenantResourcesResponse$;
exports.ListTenants$ = ListTenants$;
exports.ListTenantsCommand = ListTenantsCommand;
exports.ListTenantsRequest$ = ListTenantsRequest$;
exports.ListTenantsResponse$ = ListTenantsResponse$;
exports.MailFromAttributes$ = MailFromAttributes$;
exports.MailFromDomainNotVerifiedException = MailFromDomainNotVerifiedException;
exports.MailFromDomainNotVerifiedException$ = MailFromDomainNotVerifiedException$;
exports.MailFromDomainStatus = MailFromDomainStatus;
exports.MailType = MailType;
exports.MailboxValidation$ = MailboxValidation$;
exports.Message$ = Message$;
exports.MessageHeader$ = MessageHeader$;
exports.MessageInsightsDataSource$ = MessageInsightsDataSource$;
exports.MessageInsightsFilters$ = MessageInsightsFilters$;
exports.MessageRejected = MessageRejected;
exports.MessageRejected$ = MessageRejected$;
exports.MessageTag$ = MessageTag$;
exports.Metric = Metric;
exports.MetricAggregation = MetricAggregation;
exports.MetricDataError$ = MetricDataError$;
exports.MetricDataResult$ = MetricDataResult$;
exports.MetricDimensionName = MetricDimensionName;
exports.MetricNamespace = MetricNamespace;
exports.MetricsDataSource$ = MetricsDataSource$;
exports.MultiRegionEndpoint$ = MultiRegionEndpoint$;
exports.NotFoundException = NotFoundException;
exports.NotFoundException$ = NotFoundException$;
exports.OverallVolume$ = OverallVolume$;
exports.PinpointDestination$ = PinpointDestination$;
exports.PlacementStatistics$ = PlacementStatistics$;
exports.PutAccountDedicatedIpWarmupAttributes$ = PutAccountDedicatedIpWarmupAttributes$;
exports.PutAccountDedicatedIpWarmupAttributesCommand = PutAccountDedicatedIpWarmupAttributesCommand;
exports.PutAccountDedicatedIpWarmupAttributesRequest$ = PutAccountDedicatedIpWarmupAttributesRequest$;
exports.PutAccountDedicatedIpWarmupAttributesResponse$ = PutAccountDedicatedIpWarmupAttributesResponse$;
exports.PutAccountDetails$ = PutAccountDetails$;
exports.PutAccountDetailsCommand = PutAccountDetailsCommand;
exports.PutAccountDetailsRequest$ = PutAccountDetailsRequest$;
exports.PutAccountDetailsResponse$ = PutAccountDetailsResponse$;
exports.PutAccountSendingAttributes$ = PutAccountSendingAttributes$;
exports.PutAccountSendingAttributesCommand = PutAccountSendingAttributesCommand;
exports.PutAccountSendingAttributesRequest$ = PutAccountSendingAttributesRequest$;
exports.PutAccountSendingAttributesResponse$ = PutAccountSendingAttributesResponse$;
exports.PutAccountSuppressionAttributes$ = PutAccountSuppressionAttributes$;
exports.PutAccountSuppressionAttributesCommand = PutAccountSuppressionAttributesCommand;
exports.PutAccountSuppressionAttributesRequest$ = PutAccountSuppressionAttributesRequest$;
exports.PutAccountSuppressionAttributesResponse$ = PutAccountSuppressionAttributesResponse$;
exports.PutAccountVdmAttributes$ = PutAccountVdmAttributes$;
exports.PutAccountVdmAttributesCommand = PutAccountVdmAttributesCommand;
exports.PutAccountVdmAttributesRequest$ = PutAccountVdmAttributesRequest$;
exports.PutAccountVdmAttributesResponse$ = PutAccountVdmAttributesResponse$;
exports.PutConfigurationSetArchivingOptions$ = PutConfigurationSetArchivingOptions$;
exports.PutConfigurationSetArchivingOptionsCommand = PutConfigurationSetArchivingOptionsCommand;
exports.PutConfigurationSetArchivingOptionsRequest$ = PutConfigurationSetArchivingOptionsRequest$;
exports.PutConfigurationSetArchivingOptionsResponse$ = PutConfigurationSetArchivingOptionsResponse$;
exports.PutConfigurationSetDeliveryOptions$ = PutConfigurationSetDeliveryOptions$;
exports.PutConfigurationSetDeliveryOptionsCommand = PutConfigurationSetDeliveryOptionsCommand;
exports.PutConfigurationSetDeliveryOptionsRequest$ = PutConfigurationSetDeliveryOptionsRequest$;
exports.PutConfigurationSetDeliveryOptionsResponse$ = PutConfigurationSetDeliveryOptionsResponse$;
exports.PutConfigurationSetReputationOptions$ = PutConfigurationSetReputationOptions$;
exports.PutConfigurationSetReputationOptionsCommand = PutConfigurationSetReputationOptionsCommand;
exports.PutConfigurationSetReputationOptionsRequest$ = PutConfigurationSetReputationOptionsRequest$;
exports.PutConfigurationSetReputationOptionsResponse$ = PutConfigurationSetReputationOptionsResponse$;
exports.PutConfigurationSetSendingOptions$ = PutConfigurationSetSendingOptions$;
exports.PutConfigurationSetSendingOptionsCommand = PutConfigurationSetSendingOptionsCommand;
exports.PutConfigurationSetSendingOptionsRequest$ = PutConfigurationSetSendingOptionsRequest$;
exports.PutConfigurationSetSendingOptionsResponse$ = PutConfigurationSetSendingOptionsResponse$;
exports.PutConfigurationSetSuppressionOptions$ = PutConfigurationSetSuppressionOptions$;
exports.PutConfigurationSetSuppressionOptionsCommand = PutConfigurationSetSuppressionOptionsCommand;
exports.PutConfigurationSetSuppressionOptionsRequest$ = PutConfigurationSetSuppressionOptionsRequest$;
exports.PutConfigurationSetSuppressionOptionsResponse$ = PutConfigurationSetSuppressionOptionsResponse$;
exports.PutConfigurationSetTrackingOptions$ = PutConfigurationSetTrackingOptions$;
exports.PutConfigurationSetTrackingOptionsCommand = PutConfigurationSetTrackingOptionsCommand;
exports.PutConfigurationSetTrackingOptionsRequest$ = PutConfigurationSetTrackingOptionsRequest$;
exports.PutConfigurationSetTrackingOptionsResponse$ = PutConfigurationSetTrackingOptionsResponse$;
exports.PutConfigurationSetVdmOptions$ = PutConfigurationSetVdmOptions$;
exports.PutConfigurationSetVdmOptionsCommand = PutConfigurationSetVdmOptionsCommand;
exports.PutConfigurationSetVdmOptionsRequest$ = PutConfigurationSetVdmOptionsRequest$;
exports.PutConfigurationSetVdmOptionsResponse$ = PutConfigurationSetVdmOptionsResponse$;
exports.PutDedicatedIpInPool$ = PutDedicatedIpInPool$;
exports.PutDedicatedIpInPoolCommand = PutDedicatedIpInPoolCommand;
exports.PutDedicatedIpInPoolRequest$ = PutDedicatedIpInPoolRequest$;
exports.PutDedicatedIpInPoolResponse$ = PutDedicatedIpInPoolResponse$;
exports.PutDedicatedIpPoolScalingAttributes$ = PutDedicatedIpPoolScalingAttributes$;
exports.PutDedicatedIpPoolScalingAttributesCommand = PutDedicatedIpPoolScalingAttributesCommand;
exports.PutDedicatedIpPoolScalingAttributesRequest$ = PutDedicatedIpPoolScalingAttributesRequest$;
exports.PutDedicatedIpPoolScalingAttributesResponse$ = PutDedicatedIpPoolScalingAttributesResponse$;
exports.PutDedicatedIpWarmupAttributes$ = PutDedicatedIpWarmupAttributes$;
exports.PutDedicatedIpWarmupAttributesCommand = PutDedicatedIpWarmupAttributesCommand;
exports.PutDedicatedIpWarmupAttributesRequest$ = PutDedicatedIpWarmupAttributesRequest$;
exports.PutDedicatedIpWarmupAttributesResponse$ = PutDedicatedIpWarmupAttributesResponse$;
exports.PutDeliverabilityDashboardOption$ = PutDeliverabilityDashboardOption$;
exports.PutDeliverabilityDashboardOptionCommand = PutDeliverabilityDashboardOptionCommand;
exports.PutDeliverabilityDashboardOptionRequest$ = PutDeliverabilityDashboardOptionRequest$;
exports.PutDeliverabilityDashboardOptionResponse$ = PutDeliverabilityDashboardOptionResponse$;
exports.PutEmailIdentityConfigurationSetAttributes$ = PutEmailIdentityConfigurationSetAttributes$;
exports.PutEmailIdentityConfigurationSetAttributesCommand = PutEmailIdentityConfigurationSetAttributesCommand;
exports.PutEmailIdentityConfigurationSetAttributesRequest$ = PutEmailIdentityConfigurationSetAttributesRequest$;
exports.PutEmailIdentityConfigurationSetAttributesResponse$ = PutEmailIdentityConfigurationSetAttributesResponse$;
exports.PutEmailIdentityDkimAttributes$ = PutEmailIdentityDkimAttributes$;
exports.PutEmailIdentityDkimAttributesCommand = PutEmailIdentityDkimAttributesCommand;
exports.PutEmailIdentityDkimAttributesRequest$ = PutEmailIdentityDkimAttributesRequest$;
exports.PutEmailIdentityDkimAttributesResponse$ = PutEmailIdentityDkimAttributesResponse$;
exports.PutEmailIdentityDkimSigningAttributes$ = PutEmailIdentityDkimSigningAttributes$;
exports.PutEmailIdentityDkimSigningAttributesCommand = PutEmailIdentityDkimSigningAttributesCommand;
exports.PutEmailIdentityDkimSigningAttributesRequest$ = PutEmailIdentityDkimSigningAttributesRequest$;
exports.PutEmailIdentityDkimSigningAttributesResponse$ = PutEmailIdentityDkimSigningAttributesResponse$;
exports.PutEmailIdentityFeedbackAttributes$ = PutEmailIdentityFeedbackAttributes$;
exports.PutEmailIdentityFeedbackAttributesCommand = PutEmailIdentityFeedbackAttributesCommand;
exports.PutEmailIdentityFeedbackAttributesRequest$ = PutEmailIdentityFeedbackAttributesRequest$;
exports.PutEmailIdentityFeedbackAttributesResponse$ = PutEmailIdentityFeedbackAttributesResponse$;
exports.PutEmailIdentityMailFromAttributes$ = PutEmailIdentityMailFromAttributes$;
exports.PutEmailIdentityMailFromAttributesCommand = PutEmailIdentityMailFromAttributesCommand;
exports.PutEmailIdentityMailFromAttributesRequest$ = PutEmailIdentityMailFromAttributesRequest$;
exports.PutEmailIdentityMailFromAttributesResponse$ = PutEmailIdentityMailFromAttributesResponse$;
exports.PutSuppressedDestination$ = PutSuppressedDestination$;
exports.PutSuppressedDestinationCommand = PutSuppressedDestinationCommand;
exports.PutSuppressedDestinationRequest$ = PutSuppressedDestinationRequest$;
exports.PutSuppressedDestinationResponse$ = PutSuppressedDestinationResponse$;
exports.QueryErrorCode = QueryErrorCode;
exports.RawMessage$ = RawMessage$;
exports.Recommendation$ = Recommendation$;
exports.RecommendationImpact = RecommendationImpact;
exports.RecommendationStatus = RecommendationStatus;
exports.RecommendationType = RecommendationType;
exports.ReplacementEmailContent$ = ReplacementEmailContent$;
exports.ReplacementTemplate$ = ReplacementTemplate$;
exports.ReputationEntity$ = ReputationEntity$;
exports.ReputationEntityFilterKey = ReputationEntityFilterKey;
exports.ReputationEntityType = ReputationEntityType;
exports.ReputationOptions$ = ReputationOptions$;
exports.ResourceTenantMetadata$ = ResourceTenantMetadata$;
exports.ResourceType = ResourceType;
exports.ReviewDetails$ = ReviewDetails$;
exports.ReviewStatus = ReviewStatus;
exports.Route$ = Route$;
exports.RouteDetails$ = RouteDetails$;
exports.SESv2 = SESv2;
exports.SESv2Client = SESv2Client;
exports.SESv2ServiceException = SESv2ServiceException;
exports.SESv2ServiceException$ = SESv2ServiceException$;
exports.SOARecord$ = SOARecord$;
exports.ScalingMode = ScalingMode;
exports.SendBulkEmail$ = SendBulkEmail$;
exports.SendBulkEmailCommand = SendBulkEmailCommand;
exports.SendBulkEmailRequest$ = SendBulkEmailRequest$;
exports.SendBulkEmailResponse$ = SendBulkEmailResponse$;
exports.SendCustomVerificationEmail$ = SendCustomVerificationEmail$;
exports.SendCustomVerificationEmailCommand = SendCustomVerificationEmailCommand;
exports.SendCustomVerificationEmailRequest$ = SendCustomVerificationEmailRequest$;
exports.SendCustomVerificationEmailResponse$ = SendCustomVerificationEmailResponse$;
exports.SendEmail$ = SendEmail$;
exports.SendEmailCommand = SendEmailCommand;
exports.SendEmailRequest$ = SendEmailRequest$;
exports.SendEmailResponse$ = SendEmailResponse$;
exports.SendQuota$ = SendQuota$;
exports.SendingOptions$ = SendingOptions$;
exports.SendingPausedException = SendingPausedException;
exports.SendingPausedException$ = SendingPausedException$;
exports.SendingStatus = SendingStatus;
exports.SnsDestination$ = SnsDestination$;
exports.Status = Status;
exports.StatusRecord$ = StatusRecord$;
exports.SubscriptionStatus = SubscriptionStatus;
exports.SuppressedDestination$ = SuppressedDestination$;
exports.SuppressedDestinationAttributes$ = SuppressedDestinationAttributes$;
exports.SuppressedDestinationSummary$ = SuppressedDestinationSummary$;
exports.SuppressionAttributes$ = SuppressionAttributes$;
exports.SuppressionConditionThreshold$ = SuppressionConditionThreshold$;
exports.SuppressionConfidenceThreshold$ = SuppressionConfidenceThreshold$;
exports.SuppressionConfidenceVerdictThreshold = SuppressionConfidenceVerdictThreshold;
exports.SuppressionListDestination$ = SuppressionListDestination$;
exports.SuppressionListImportAction = SuppressionListImportAction;
exports.SuppressionListReason = SuppressionListReason;
exports.SuppressionOptions$ = SuppressionOptions$;
exports.SuppressionValidationAttributes$ = SuppressionValidationAttributes$;
exports.SuppressionValidationOptions$ = SuppressionValidationOptions$;
exports.Tag$ = Tag$;
exports.TagResource$ = TagResource$;
exports.TagResourceCommand = TagResourceCommand;
exports.TagResourceRequest$ = TagResourceRequest$;
exports.TagResourceResponse$ = TagResourceResponse$;
exports.Template$ = Template$;
exports.Tenant$ = Tenant$;
exports.TenantInfo$ = TenantInfo$;
exports.TenantResource$ = TenantResource$;
exports.TestRenderEmailTemplate$ = TestRenderEmailTemplate$;
exports.TestRenderEmailTemplateCommand = TestRenderEmailTemplateCommand;
exports.TestRenderEmailTemplateRequest$ = TestRenderEmailTemplateRequest$;
exports.TestRenderEmailTemplateResponse$ = TestRenderEmailTemplateResponse$;
exports.TlsPolicy = TlsPolicy;
exports.TooManyRequestsException = TooManyRequestsException;
exports.TooManyRequestsException$ = TooManyRequestsException$;
exports.Topic$ = Topic$;
exports.TopicFilter$ = TopicFilter$;
exports.TopicPreference$ = TopicPreference$;
exports.TrackingOptions$ = TrackingOptions$;
exports.UntagResource$ = UntagResource$;
exports.UntagResourceCommand = UntagResourceCommand;
exports.UntagResourceRequest$ = UntagResourceRequest$;
exports.UntagResourceResponse$ = UntagResourceResponse$;
exports.UpdateConfigurationSetEventDestination$ = UpdateConfigurationSetEventDestination$;
exports.UpdateConfigurationSetEventDestinationCommand = UpdateConfigurationSetEventDestinationCommand;
exports.UpdateConfigurationSetEventDestinationRequest$ = UpdateConfigurationSetEventDestinationRequest$;
exports.UpdateConfigurationSetEventDestinationResponse$ = UpdateConfigurationSetEventDestinationResponse$;
exports.UpdateContact$ = UpdateContact$;
exports.UpdateContactCommand = UpdateContactCommand;
exports.UpdateContactList$ = UpdateContactList$;
exports.UpdateContactListCommand = UpdateContactListCommand;
exports.UpdateContactListRequest$ = UpdateContactListRequest$;
exports.UpdateContactListResponse$ = UpdateContactListResponse$;
exports.UpdateContactRequest$ = UpdateContactRequest$;
exports.UpdateContactResponse$ = UpdateContactResponse$;
exports.UpdateCustomVerificationEmailTemplate$ = UpdateCustomVerificationEmailTemplate$;
exports.UpdateCustomVerificationEmailTemplateCommand = UpdateCustomVerificationEmailTemplateCommand;
exports.UpdateCustomVerificationEmailTemplateRequest$ = UpdateCustomVerificationEmailTemplateRequest$;
exports.UpdateCustomVerificationEmailTemplateResponse$ = UpdateCustomVerificationEmailTemplateResponse$;
exports.UpdateEmailIdentityPolicy$ = UpdateEmailIdentityPolicy$;
exports.UpdateEmailIdentityPolicyCommand = UpdateEmailIdentityPolicyCommand;
exports.UpdateEmailIdentityPolicyRequest$ = UpdateEmailIdentityPolicyRequest$;
exports.UpdateEmailIdentityPolicyResponse$ = UpdateEmailIdentityPolicyResponse$;
exports.UpdateEmailTemplate$ = UpdateEmailTemplate$;
exports.UpdateEmailTemplateCommand = UpdateEmailTemplateCommand;
exports.UpdateEmailTemplateRequest$ = UpdateEmailTemplateRequest$;
exports.UpdateEmailTemplateResponse$ = UpdateEmailTemplateResponse$;
exports.UpdateReputationEntityCustomerManagedStatus$ = UpdateReputationEntityCustomerManagedStatus$;
exports.UpdateReputationEntityCustomerManagedStatusCommand = UpdateReputationEntityCustomerManagedStatusCommand;
exports.UpdateReputationEntityCustomerManagedStatusRequest$ = UpdateReputationEntityCustomerManagedStatusRequest$;
exports.UpdateReputationEntityCustomerManagedStatusResponse$ = UpdateReputationEntityCustomerManagedStatusResponse$;
exports.UpdateReputationEntityPolicy$ = UpdateReputationEntityPolicy$;
exports.UpdateReputationEntityPolicyCommand = UpdateReputationEntityPolicyCommand;
exports.UpdateReputationEntityPolicyRequest$ = UpdateReputationEntityPolicyRequest$;
exports.UpdateReputationEntityPolicyResponse$ = UpdateReputationEntityPolicyResponse$;
exports.VdmAttributes$ = VdmAttributes$;
exports.VdmOptions$ = VdmOptions$;
exports.VerificationError = VerificationError;
exports.VerificationInfo$ = VerificationInfo$;
exports.VerificationStatus = VerificationStatus;
exports.VolumeStatistics$ = VolumeStatistics$;
exports.WarmupStatus = WarmupStatus;
exports.paginateGetDedicatedIps = paginateGetDedicatedIps;
exports.paginateListConfigurationSets = paginateListConfigurationSets;
exports.paginateListContactLists = paginateListContactLists;
exports.paginateListContacts = paginateListContacts;
exports.paginateListCustomVerificationEmailTemplates = paginateListCustomVerificationEmailTemplates;
exports.paginateListDedicatedIpPools = paginateListDedicatedIpPools;
exports.paginateListDeliverabilityTestReports = paginateListDeliverabilityTestReports;
exports.paginateListDomainDeliverabilityCampaigns = paginateListDomainDeliverabilityCampaigns;
exports.paginateListEmailIdentities = paginateListEmailIdentities;
exports.paginateListEmailTemplates = paginateListEmailTemplates;
exports.paginateListExportJobs = paginateListExportJobs;
exports.paginateListImportJobs = paginateListImportJobs;
exports.paginateListMultiRegionEndpoints = paginateListMultiRegionEndpoints;
exports.paginateListRecommendations = paginateListRecommendations;
exports.paginateListReputationEntities = paginateListReputationEntities;
exports.paginateListResourceTenants = paginateListResourceTenants;
exports.paginateListSuppressedDestinations = paginateListSuppressedDestinations;
exports.paginateListTenantResources = paginateListTenantResources;
exports.paginateListTenants = paginateListTenants;
