import { color } from "../../../core/util/Color";
import { ColorSet } from "../../../core/util/ColorSet";
import { Dropdown } from "./Dropdown";
import * as $array from "../../../core/util/Array";
import * as $utils from "../../../core/util/Utils";
/**
 * A dropdown used for color picker control.
 *
 * Should not be used as standalone control.
 */
export class DropdownColors extends Dropdown {
    // private _itemDisposers: Array<IDisposer> = [];
    _afterNew() {
        super._afterNew();
        this._setupKeyboardNav();
        this._root.addDisposer(this);
    }
    _initElements() {
        super._initElements();
        // Create container
        const container = this.getPrivate("container");
        // Create list
        const list = document.createElement("ul");
        list.className = "am5stock-control-colors";
        container.appendChild(list);
        this.setPrivate("list", list);
        this._initItems();
    }
    _initItems() {
        const list = this.getPrivate("list");
        list.innerHTML = "";
        let cs = this.get("colors");
        if (!cs) {
            cs = ColorSet.new(this._root, {});
        }
        const colors = cs.get("colors", []);
        $array.each(colors, (item) => {
            this.addItem(item);
        });
        this._initOpacity();
        this._maybeMakeAccessible();
    }
    _beforeChanged() {
        super._beforeChanged();
        if (this.isDirty("colors") || this.isDirty("useOpacity") || this.isPrivateDirty("color") || this.isPrivateDirty("opacity")) {
            this._initItems();
        }
        if (this.isDirty("control")) {
            this._maybeMakeAccessible();
        }
    }
    _dispose() {
        super._dispose();
    }
    addItem(color) {
        const currentColor = this.getPrivate("color") ? this.getPrivate("color").hex : 0;
        const list = this.getPrivate("list");
        const item = document.createElement("li");
        item.className = "am5stock-control-color";
        if (currentColor == color.hex) {
            item.className += " am5stock-control-active";
        }
        item.style.background = color.toCSS();
        list.appendChild(item);
        // Add click event
        this._disposers.push($utils.addEventListener(item, "click", (_ev) => {
            this.hide();
            this.events.dispatch("invoked", {
                type: "invoked",
                color: color,
                target: this
            });
        }));
    }
    _initOpacity() {
        if (this.get("useOpacity")) {
            const currentOpacity = this.getPrivate("opacity", 1);
            const list = this.getPrivate("list");
            const hr = document.createElement("hr");
            list.appendChild(hr);
            for (let opacity = 100; opacity >= 0; opacity -= 25) {
                const fill = color(0x000000);
                const item = document.createElement("li");
                item.innerHTML = opacity + "%";
                item.className = "am5stock-control-opacity am5stock-control-opacity-" + opacity;
                if (currentOpacity * 100 == opacity) {
                    item.className += " am5stock-control-active";
                }
                item.style.background = fill.toCSS(opacity / 100);
                list.appendChild(item);
                // Add click event
                this._disposers.push($utils.addEventListener(item, "click", (_ev) => {
                    this.hide();
                    this.events.dispatch("invokedOpacity", {
                        type: "invokedOpacity",
                        opacity: opacity / 100,
                        target: this
                    });
                }));
            }
        }
    }
    _setupKeyboardNav() {
        if ($utils.supports("keyboardevents")) {
            const button = this.get("control").getPrivate("button");
            this._disposers.push($utils.addEventListener(document, "keydown", (ev) => {
                if (this.isAccessible()) {
                    if (document.activeElement && (document.activeElement === button || $utils.contains(button, document.activeElement))) {
                        if (ev.keyCode == 13) {
                            // ENTER
                            if (document.activeElement !== button) {
                                document.activeElement.click();
                            }
                        }
                        else if ([37, 38, 39, 40].indexOf(ev.keyCode) !== -1) {
                            const dir = ev.keyCode == 37 || ev.keyCode == 38 ? -1 : 1;
                            const items = this._getActiveItems();
                            const selected = button.querySelectorAll(".am5stock-control-color:focus, .am5stock-control-opacity:focus");
                            let index = -1;
                            if (selected.length > 0) {
                                items.forEach((item, key) => {
                                    if (item === selected.item(0)) {
                                        index = key;
                                    }
                                });
                            }
                            index += dir;
                            if (index < 0) {
                                index = items.length - 1;
                            }
                            else if (index >= items.length) {
                                index = 0;
                            }
                            $utils.focus(items.item(index));
                        }
                    }
                }
            }));
        }
    }
    _maybeMakeAccessible() {
        super._maybeMakeAccessible();
        if (this.isAccessible()) {
            //const tabindex = this._root.tabindex.toString();
            //const list = this.getPrivate("list")!;
            const items = this._getActiveItems();
            items.forEach((item, key) => {
                item.setAttribute("tabindex", "-1");
                item.setAttribute("aria-label", this.root.language.translateAny("Selection") + " #" + (key + 1));
            });
        }
    }
    _getActiveItems() {
        return this.getPrivate("list").querySelectorAll(".am5stock-control-color, .am5stock-control-opacity");
    }
}
Object.defineProperty(DropdownColors, "className", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: "DropdownColors"
});
Object.defineProperty(DropdownColors, "classNames", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: Dropdown.classNames.concat([DropdownColors.className])
});
//# sourceMappingURL=DropdownColors.js.map