import type { Pattern } from "../render/patterns/Pattern";
import type { Color } from "./Color";
import { LinePattern, ILinePatternSettings } from "../render/patterns/LinePattern";
import { RectanglePattern, IRectanglePatternSettings } from "../render/patterns/RectanglePattern";
import { CirclePattern, ICirclePatternSettings } from "../render/patterns/CirclePattern";
import { Entity, IEntitySettings, IEntityPrivate } from "./Entity";
export interface IPatternSetSettings extends IEntitySettings {
    /**
     * List of colors in the set.
     */
    patterns?: Pattern[];
    /**
     * A step size when using `next()`.
     *
     * E.g. setting to `2` will make it return every second pattern in the list.
     *
     * @default 1
     */
    step?: number;
    /**
     * A base color to use for all patterns.
     *
     * @see {@link https://www.amcharts.com/docs/v5/concepts/colors-gradients-and-patterns/patterns/#Colors} for more info
     */
    color?: Color;
    /**
     * Start iterating patterns from specific index.
     */
    startIndex?: number;
}
export interface IPatternSetPrivate extends IEntityPrivate {
    /**
     * Current step.
     */
    currentStep?: number;
}
/**
 * An object which holds list of [[Pattern]] objects and can serve them up in
 * an interative way.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/colors-gradients-and-patterns/patterns/#Pattern_sets} for more info
 * @since 5.10.0
 */
export declare class PatternSet extends Entity {
    static className: string;
    static classNames: Array<string>;
    _settings: IPatternSetSettings;
    _privateSettings: IPatternSetPrivate;
    protected _afterNew(): void;
    _beforeChanged(): void;
    /**
     * Returns a [[Pattern]] at specific index.
     *
     * @param   index  Index
     * @return         Color
     */
    getIndex(index: number): Pattern;
    /**
     * Returns next [[Color]] in the list.
     *
     * If the list is out of colors, new ones are generated dynamically.
     */
    next(): Pattern;
    /**
     * Resets counter to the start of the list, so the next call for `next()` will
     * return the first pattern.
     */
    reset(): void;
    /**
     * Returns a [[LinePattern].
     *
     * @param   settings  Pattern settings
     * @return            Pattern object
     */
    getLinePattern(settings: ILinePatternSettings): LinePattern;
    /**
     * Returns a [[RectanglePattern].
     *
     * @param   settings  Pattern settings
     * @return            Pattern object
     */
    getRectanglePattern(settings: IRectanglePatternSettings): RectanglePattern;
    /**
     * Returns a [[CirclePattern].
     *
     * @param   settings  Pattern settings
     * @return            Pattern object
     */
    getCirclePattern(settings: ICirclePatternSettings): CirclePattern;
}
//# sourceMappingURL=PatternSet.d.ts.map