import { SimpleLineSeries } from "./SimpleLineSeries";
import { ListTemplate } from "../../../core/util/List";
import { Triangle } from "../../../core/render/Triangle";
import { Template } from "../../../core/util/Template";
import { Bullet } from "../../../core/render/Bullet";
import * as $math from "../../../core/util/Math";
import * as $array from "../../../core/util/Array";
import * as $object from "../../../core/util/Object";
/**
 * Used for Line Arrow drawing tool in [[StockChart]].
 *
 * @since 5.10.5
 */
export class LineArrowSeries extends SimpleLineSeries {
    constructor() {
        super(...arguments);
        Object.defineProperty(this, "_arrows", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: []
        });
        Object.defineProperty(this, "arrows", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: this.addDisposer(new ListTemplate(Template.new({}), () => Triangle._new(this._root, { themeTags: ["arrow"] }, [this.arrows.template])))
        });
    }
    _afterNew() {
        super._afterNew();
        this.lines.template.set("forceHidden", true);
        this.bullets.push((_root, _series, dataItem) => {
            const dataContext = dataItem.dataContext;
            if (dataContext.corner == "p2") {
                const index = dataContext.index;
                const di = this._di[index]["e"];
                let color = this.get("strokeColor", this.get("stroke"));
                if (di) {
                    const dc = di.dataContext;
                    if (dc) {
                        const strokeTemplate = dc.stroke;
                        if (strokeTemplate) {
                            color = strokeTemplate.get("stroke");
                        }
                    }
                }
                const bullets = dataItem.bullets;
                if (bullets) {
                    const bullet = bullets[0];
                    if (bullet) {
                        const container = bullet.get("sprite");
                        if (container) {
                            const child = container.children.getIndex(0);
                            if (child) {
                                child.set("scale", .1);
                            }
                        }
                    }
                }
                const arrow = this.makeArrow();
                arrow.setAll({
                    forceInactive: true,
                    fill: color,
                    userData: dataContext.index
                });
                return Bullet.new(this._root, {
                    sprite: arrow,
                    locationX: undefined
                });
            }
        });
    }
    clearDrawings() {
        super.clearDrawings();
        this.arrows.clear();
    }
    _changed() {
        super._changed();
        this.arrows.each((arrow) => {
            const index = arrow.get("userData");
            const di = this._di[index];
            if (di) {
                const di1 = this._di[index]["p1"];
                const di2 = this._di[index]["p2"];
                const p1 = di1.get("point");
                const p2 = di2.get("point");
                if (p1 && p2) {
                    // rotate arrow
                    const angle = $math.getAngle(p1, p2) + 90;
                    arrow.set("rotation", angle);
                    const w = arrow.width();
                    const strokeIndex = this._getStrokeIndex(index);
                    const stroke = this.strokes.getIndex(strokeIndex);
                    if (stroke) {
                        arrow.set("scale", (w + stroke.get("strokeWidth", 2) * 2) / w);
                    }
                }
            }
        });
    }
    _applySettings(index, settings) {
        super._applySettings(index, settings);
        let template;
        $array.each(this.dataItems, (dataItem) => {
            const dataContext = dataItem.dataContext;
            if (dataContext.index == index) {
                if (dataContext.settings) {
                    template = dataContext.settings;
                }
            }
        });
        const settings2 = $object.copy(settings);
        if (settings.stroke != undefined) {
            settings2.fill = settings.stroke;
        }
        else {
            delete settings2.fill;
        }
        if (settings.strokeOpacity != undefined) {
            settings2.fillOpacity = settings.strokeOpacity;
        }
        else {
            delete settings2.fillOpacity;
        }
        settings2.strokeOpacity = 0;
        const arrow = this._getSprite(this.arrows, index);
        if (arrow) {
            const defaultState = arrow.states.lookup("default");
            $object.each(settings2, (key, value) => {
                arrow.set(key, value);
                defaultState.set(key, value);
                if (template) {
                    template.set(key, value);
                }
            });
        }
    }
    /**
     * @ignore
     */
    makeArrow() {
        const arrow = this.arrows.make();
        this.arrows.push(arrow);
        return arrow;
    }
    _updateLine(index, p11, p22, p1, p2) {
        super._updateLine(index, p11, p22, p1, p2);
    }
}
Object.defineProperty(LineArrowSeries, "className", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: "LineArrowSeries"
});
Object.defineProperty(LineArrowSeries, "classNames", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: SimpleLineSeries.classNames.concat([LineArrowSeries.className])
});
//# sourceMappingURL=LineArrowSeries.js.map