import { string } from 'alga-js'
import refreshTokenHelper from "../../../utils/refreshTokenHelper"
import forceLogoutHelper from "../../../utils/forceLogoutHelper"
import errorHandlingHelper from "../../../utils/errorHandlingHelper"
import fetchHelper from "../../../utils/fetchHelper"

const handleFunc = async (event: any, authToken: any = null) => {
  let data: any = {}
  const token = authToken ?? await getTokenHelper(event)
  const query = getQuery(event)

  // Build filter for fetching storage usage records
  // Use pagination to get all records - fetch in batches
  const pageSize = 1000
  let allRecords: any[] = []
  let page = 0
  let hasMore = true

  // Add date filter if provided
  let filterStr = ''
  if (query.dateFrom && query.dateTo) {
    filterStr = string.urlEncode(`DateAcct ge '${query.dateFrom}' and DateAcct le '${query.dateTo}'`)
  } else if (query.dateFrom) {
    filterStr = string.urlEncode(`DateAcct ge '${query.dateFrom}'`)
  } else if (query.dateTo) {
    filterStr = string.urlEncode(`DateAcct le '${query.dateTo}'`)
  }

  // Fetch all records with pagination
  while (hasMore) {
    const skip = page * pageSize
    let url = `models/cust_storage_usage?$top=${pageSize}&$skip=${skip}&$orderby=${string.urlEncode('DateAcct desc')}`
    if (filterStr) {
      url += `&$filter=${filterStr}`
    }

    const res: any = await fetchHelper(event, url, 'GET', token, null)

    if (res?.records && res.records.length > 0) {
      allRecords = [...allRecords, ...res.records]
      hasMore = res.records.length === pageSize
      page++
    } else {
      hasMore = false
    }

    // Safety limit - don't fetch more than 10 pages (10000 records)
    if (page >= 10) {
      hasMore = false
    }
  }

  // Group by DateAcct and AD_Org_ID for chart data
  const chartDataByOrg: Record<string, Record<string, number>> = {}
  const chartDataTotal: Record<string, number> = {}

  allRecords.forEach((record: any) => {
    const date = record.DateAcct
    const orgName = record.AD_Org_ID?.identifier || 'Unknown'
    const volume = Number(record.Volume) || 0

    // Group by org
    if (!chartDataByOrg[orgName]) {
      chartDataByOrg[orgName] = {}
    }
    if (!chartDataByOrg[orgName][date]) {
      chartDataByOrg[orgName][date] = 0
    }
    chartDataByOrg[orgName][date] += volume

    // Total
    if (!chartDataTotal[date]) {
      chartDataTotal[date] = 0
    }
    chartDataTotal[date] += volume
  })

  // Convert to array format for chart
  const dates = [...new Set(allRecords.map((r: any) => r.DateAcct))].sort()
  const organizations = Object.keys(chartDataByOrg)

  // Format chart data - each entry has date and volume per org
  const chartData = dates.map(date => {
    const entry: Record<string, any> = {
      date: date,
      total: Number((chartDataTotal[date] || 0).toFixed(3))
    }
    organizations.forEach(org => {
      entry[org] = Number((chartDataByOrg[org]?.[date] || 0).toFixed(3))
    })
    return entry
  })

  data = {
    records: allRecords,
    chartData: chartData,
    organizations: organizations,
    totalRecords: allRecords.length
  }

  return data
}

export default defineEventHandler(async (event) => {
  let data: any = {}

  try {
    data = await handleFunc(event)
  } catch (err: any) {
    try {
      let authToken: any = await refreshTokenHelper(event)
      data = await handleFunc(event, authToken)
    } catch (error: any) {
      data = errorHandlingHelper(err?.data ?? err, error?.data ?? error)
      forceLogoutHelper(event, data)
    }
  }

  return data
})
