/**
 * Generate PWA icons from SVG
 * Uses sharp library to convert SVG to PNG at different sizes
 *
 * Run: node scripts/generate-icons.js
 */

import fs from 'fs'
import path from 'path'
import { fileURLToPath } from 'url'

const __filename = fileURLToPath(import.meta.url)
const __dirname = path.dirname(__filename)

// Icon sizes needed for PWA
const sizes = [72, 96, 128, 144, 152, 192, 384, 512]

const svgPath = path.join(__dirname, '../public/icon.svg')
const publicDir = path.join(__dirname, '../public')

async function main() {
  // Read SVG content
  const svgContent = fs.readFileSync(svgPath, 'utf8')

  console.log('📱 Generating PWA icons...')

  // Try to use sharp if available, otherwise create placeholder
  let sharp
  try {
    sharp = (await import('sharp')).default
  } catch (e) {
    console.warn('⚠️  Sharp not installed. Creating placeholder icons.')
    console.log('   To generate proper icons, run: npm install --save-dev sharp')
    console.log('   Then run this script again: node scripts/generate-icons.js')

    // Create simple placeholder text files
    sizes.forEach(size => {
      const filename = `icon-${size}x${size}.png`
      const filepath = path.join(publicDir, filename)

      // Create a simple placeholder (not a real PNG, but prevents 404 errors)
      fs.writeFileSync(filepath, `Placeholder for ${size}x${size} icon`)
      console.log(`   Created placeholder: ${filename}`)
    })

    console.log('\n✅ Placeholder icons created')
    console.log('   Install sharp and regenerate for proper PNG icons')
    process.exit(0)
  }

  // Generate icons using sharp
  for (const size of sizes) {
    const filename = `icon-${size}x${size}.png`
    const filepath = path.join(publicDir, filename)

    try {
      await sharp(Buffer.from(svgContent))
        .resize(size, size)
        .png()
        .toFile(filepath)

      console.log(`✅ Generated: ${filename}`)
    } catch (error) {
      console.error(`❌ Failed to generate ${filename}:`, error.message)
    }
  }

  console.log('\n🎉 All icons generated successfully!')
}

main().catch(error => {
  console.error('❌ Error generating icons:', error)
  process.exit(1)
})
