import { DateFormatter, createCalendar, toCalendar, CalendarDateTime, CalendarDate, Time, getLocalTimeZone, parseZonedDateTime, parseDateTime, parseDate, ZonedDateTime, getDayOfWeek } from '@internationalized/date';

function getDefaultDate(props) {
  const { defaultValue, defaultPlaceholder, granularity = "day", locale = "en" } = props;
  if (Array.isArray(defaultValue) && defaultValue.length)
    return defaultValue.at(-1).copy();
  if (defaultValue && !Array.isArray(defaultValue))
    return defaultValue.copy();
  if (defaultPlaceholder)
    return defaultPlaceholder.copy();
  const date = /* @__PURE__ */ new Date();
  const year = date.getFullYear();
  const month = date.getMonth() + 1;
  const day = date.getDate();
  const calendarDateTimeGranularities = ["hour", "minute", "second"];
  const defaultFormatter = new DateFormatter(locale);
  const calendar = createCalendar(defaultFormatter.resolvedOptions().calendar);
  if (calendarDateTimeGranularities.includes(granularity ?? "day"))
    return toCalendar(new CalendarDateTime(year, month, day, 0, 0, 0), calendar);
  return toCalendar(new CalendarDate(year, month, day), calendar);
}
function getDefaultTime(props) {
  const { defaultValue, defaultPlaceholder } = props;
  if (defaultValue) {
    return defaultValue.copy();
  }
  if (defaultPlaceholder) {
    return defaultPlaceholder.copy();
  }
  return new Time(0, 0, 0);
}

function parseStringToDateValue(dateStr, referenceVal) {
  let dateValue;
  if (isZonedDateTime(referenceVal))
    dateValue = parseZonedDateTime(dateStr);
  else if (isCalendarDateTime(referenceVal))
    dateValue = parseDateTime(dateStr);
  else
    dateValue = parseDate(dateStr);
  return dateValue.calendar !== referenceVal.calendar ? toCalendar(dateValue, referenceVal.calendar) : dateValue;
}
function toDate(dateValue, tz = getLocalTimeZone()) {
  if (isZonedDateTime(dateValue))
    return dateValue.toDate();
  else
    return dateValue.toDate(tz);
}
function isCalendarDateTime(dateValue) {
  return dateValue instanceof CalendarDateTime;
}
function isZonedDateTime(dateValue) {
  return dateValue instanceof ZonedDateTime;
}
function hasTime(dateValue) {
  return isCalendarDateTime(dateValue) || isZonedDateTime(dateValue);
}
function getDaysInMonth(date) {
  if (date instanceof Date) {
    const year = date.getFullYear();
    const month = date.getMonth() + 1;
    return new Date(year, month, 0).getDate();
  } else {
    return date.set({ day: 100 }).day;
  }
}
function isBefore(dateToCompare, referenceDate) {
  return dateToCompare.compare(referenceDate) < 0;
}
function isAfter(dateToCompare, referenceDate) {
  return dateToCompare.compare(referenceDate) > 0;
}
function isBeforeOrSame(dateToCompare, referenceDate) {
  return dateToCompare.compare(referenceDate) <= 0;
}
function isAfterOrSame(dateToCompare, referenceDate) {
  return dateToCompare.compare(referenceDate) >= 0;
}
function isBetweenInclusive(date, start, end) {
  return isAfterOrSame(date, start) && isBeforeOrSame(date, end);
}
function isBetween(date, start, end) {
  return isAfter(date, start) && isBefore(date, end);
}
function getLastFirstDayOfWeek(date, firstDayOfWeek, locale) {
  const day = getDayOfWeek(date, locale);
  if (firstDayOfWeek > day)
    return date.subtract({ days: day + 7 - firstDayOfWeek });
  if (firstDayOfWeek === day)
    return date;
  return date.subtract({ days: day - firstDayOfWeek });
}
function getNextLastDayOfWeek(date, firstDayOfWeek, locale) {
  const day = getDayOfWeek(date, locale);
  const lastDayOfWeek = firstDayOfWeek === 0 ? 6 : firstDayOfWeek - 1;
  if (day === lastDayOfWeek)
    return date;
  if (day > lastDayOfWeek)
    return date.add({ days: 7 - day + lastDayOfWeek });
  return date.add({ days: lastDayOfWeek - day });
}
function areAllDaysBetweenValid(start, end, isUnavailable, isDisabled, isHighlightable) {
  if (isUnavailable === void 0 && isDisabled === void 0 && isHighlightable === void 0)
    return true;
  let dCurrent = start.add({ days: 1 });
  if ((isDisabled?.(dCurrent) || isUnavailable?.(dCurrent)) && !isHighlightable?.(dCurrent)) {
    return false;
  }
  const dEnd = end;
  while (dCurrent.compare(dEnd) < 0) {
    dCurrent = dCurrent.add({ days: 1 });
    if ((isDisabled?.(dCurrent) || isUnavailable?.(dCurrent)) && !isHighlightable?.(dCurrent)) {
      return false;
    }
  }
  return true;
}

export { isZonedDateTime as a, isBefore as b, isAfter as c, isBeforeOrSame as d, isAfterOrSame as e, isBetweenInclusive as f, getDaysInMonth as g, hasTime as h, isCalendarDateTime as i, isBetween as j, getLastFirstDayOfWeek as k, getNextLastDayOfWeek as l, areAllDaysBetweenValid as m, getDefaultDate as n, getDefaultTime as o, parseStringToDateValue as p, toDate as t };
//# sourceMappingURL=comparators.js.map
