var __defProp = Object.defineProperty;
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};

// packages/ag-charts-core/src/globals/logger.ts
var logger_exports = {};
__export(logger_exports, {
  error: () => error,
  errorOnce: () => errorOnce,
  log: () => log,
  logGroup: () => logGroup,
  reset: () => reset,
  table: () => table,
  warn: () => warn,
  warnOnce: () => warnOnce
});
var doOnceCache = /* @__PURE__ */ new Set();
function log(...logContent) {
  console.log(...logContent);
}
function warn(message, ...logContent) {
  console.warn(`AG Charts - ${message}`, ...logContent);
}
function error(message, ...logContent) {
  if (typeof message === "object") {
    console.error(`AG Charts error`, message, ...logContent);
  } else {
    console.error(`AG Charts - ${message}`, ...logContent);
  }
}
function table(...logContent) {
  console.table(...logContent);
}
function warnOnce(message, ...logContent) {
  const cacheKey = `Logger.warn: ${message}`;
  if (doOnceCache.has(cacheKey))
    return;
  warn(message, ...logContent);
  doOnceCache.add(cacheKey);
}
function errorOnce(message, ...logContent) {
  const cacheKey = `Logger.error: ${message}`;
  if (doOnceCache.has(cacheKey))
    return;
  error(message, ...logContent);
  doOnceCache.add(cacheKey);
}
function reset() {
  doOnceCache.clear();
}
function logGroup(name, cb) {
  console.groupCollapsed(name);
  try {
    return cb();
  } finally {
    console.groupEnd();
  }
}

// packages/ag-charts-core/src/interfaces/moduleDefinition.ts
var ModuleType = /* @__PURE__ */ ((ModuleType2) => {
  ModuleType2["Chart"] = "chart";
  ModuleType2["Series"] = "series";
  ModuleType2["Feature"] = "feature";
  return ModuleType2;
})(ModuleType || {});

// packages/ag-charts-core/src/globals/moduleRegistry.ts
var ModuleRegistry = class {
  static [Symbol.iterator]() {
    return this.registeredModules.values();
  }
  static register(definition) {
    const existingDefinition = this.registeredModules.get(definition.name);
    if (existingDefinition && (existingDefinition.enterprise || !definition.enterprise)) {
      throw new Error(`AG Charts - Module '${definition.name}' already registered`);
    }
    this.registeredModules.set(definition.name, definition);
  }
  static registerMany(definitions) {
    for (const definition of definitions) {
      this.register(definition);
    }
  }
  static reset() {
    this.registeredModules.clear();
  }
  static detectChartDefinition(options) {
    return this.detectDefinition("chart" /* Chart */, options);
  }
  static detectSeriesDefinition(options) {
    return this.detectDefinition("series" /* Series */, options);
  }
  static detectDefinition(moduleType, options) {
    for (const definition of this.registeredModules.values()) {
      if (definition.type === moduleType && definition.detect(options)) {
        return definition;
      }
    }
    throw new Error(
      `AG Charts - Unknown ${moduleType} type; Check options are correctly structured and series types are specified`
    );
  }
};
ModuleRegistry.registeredModules = /* @__PURE__ */ new Map();

// packages/ag-charts-core/src/classes/eventEmitter.ts
var EventEmitter = class {
  constructor() {
    this.events = /* @__PURE__ */ new Map();
  }
  /**
   * Registers an event listener.
   * @param eventName The event name to listen for.
   * @param listener The callback to be invoked on the event.
   * @returns A function to unregister the listener.
   */
  on(eventName, listener) {
    if (!this.events.has(eventName)) {
      this.events.set(eventName, /* @__PURE__ */ new Set());
    }
    this.events.get(eventName)?.add(listener);
    return () => this.off(eventName, listener);
  }
  /**
   * Unregisters an event listener.
   * @param eventName The event name to stop listening for.
   * @param listener The callback to be removed.
   */
  off(eventName, listener) {
    const eventListeners = this.events.get(eventName);
    if (eventListeners) {
      eventListeners.delete(listener);
      if (eventListeners.size === 0) {
        this.events.delete(eventName);
      }
    }
  }
  /**
   * Emits an event to all registered listeners.
   * @param eventName The name of the event to emit.
   * @param event The event payload.
   */
  emit(eventName, event) {
    this.events.get(eventName)?.forEach((callback2) => callback2(event));
  }
  /**
   * Clears all listeners for a specific event or all events if no event name is provided.
   * @param eventName (Optional) The name of the event to clear listeners for. If not provided, all listeners for all events are cleared.
   */
  clear(eventName) {
    if (eventName) {
      this.events.delete(eventName);
    } else {
      this.events.clear();
    }
  }
};

// packages/ag-charts-core/src/utils/arrays.ts
function toArray(value) {
  if (typeof value === "undefined") {
    return [];
  }
  return Array.isArray(value) ? value : [value];
}
function unique(array2) {
  return Array.from(new Set(array2));
}
function groupBy(array2, iteratee) {
  return array2.reduce((result, item) => {
    const groupKey = iteratee(item);
    result[groupKey] ?? (result[groupKey] = []);
    result[groupKey].push(item);
    return result;
  }, {});
}
function arraysEqual(a, b) {
  if (a == null || b == null || a.length !== b.length) {
    return false;
  }
  for (let i = 0; i < a.length; i++) {
    if (Array.isArray(a[i]) && Array.isArray(b[i])) {
      if (!arraysEqual(a[i], b[i])) {
        return false;
      }
    } else if (a[i] !== b[i]) {
      return false;
    }
  }
  return true;
}
function circularSliceArray(data, size, offset = 0) {
  if (data.length === 0) {
    return [];
  }
  const result = [];
  for (let i = 0; i < size; i++) {
    result.push(data.at((i + offset) % data.length));
  }
  return result;
}
function sortBasedOnArray(baseArray, orderArray) {
  const orderMap = /* @__PURE__ */ new Map();
  orderArray.forEach((item, index) => {
    orderMap.set(item, index);
  });
  return baseArray.sort((a, b) => {
    const indexA = orderMap.get(a) ?? Infinity;
    const indexB = orderMap.get(b) ?? Infinity;
    return indexA - indexB;
  });
}

// packages/ag-charts-core/src/utils/binarySearch.ts
function findMaxIndex(min, max, iteratee) {
  if (min > max)
    return;
  let found;
  while (max >= min) {
    const index = Math.floor((max + min) / 2);
    const value = iteratee(index);
    if (value) {
      found = index;
      min = index + 1;
    } else {
      max = index - 1;
    }
  }
  return found;
}
function findMinIndex(min, max, iteratee) {
  if (min > max)
    return;
  let found;
  while (max >= min) {
    const index = Math.floor((max + min) / 2);
    const value = iteratee(index);
    if (value) {
      found = index;
      max = index - 1;
    } else {
      min = index + 1;
    }
  }
  return found;
}
function findMaxValue(min, max, iteratee) {
  if (min > max)
    return;
  let found;
  while (max >= min) {
    const index = Math.floor((max + min) / 2);
    const value = iteratee(index);
    if (value == null) {
      max = index - 1;
    } else {
      found = value;
      min = index + 1;
    }
  }
  return found;
}
function findMinValue(min, max, iteratee) {
  if (min > max)
    return;
  let found;
  while (max >= min) {
    const index = Math.floor((max + min) / 2);
    const value = iteratee(index);
    if (value == null) {
      min = index + 1;
    } else {
      found = value;
      max = index - 1;
    }
  }
  return found;
}

// packages/ag-charts-core/src/utils/diff.ts
function diffArrays(previous, current) {
  const size = Math.max(previous.length, current.length);
  const added = /* @__PURE__ */ new Set();
  const removed = /* @__PURE__ */ new Set();
  for (let i = 0; i < size; i++) {
    const prev = previous[i];
    const curr = current[i];
    if (prev === curr)
      continue;
    if (removed.has(curr)) {
      removed.delete(curr);
    } else if (curr) {
      added.add(curr);
    }
    if (added.has(prev)) {
      added.delete(prev);
    } else if (prev) {
      removed.add(prev);
    }
  }
  return { changed: added.size > 0 || removed.size > 0, added, removed };
}

// packages/ag-charts-core/src/utils/functions.ts
function debounce(callback2, waitMs = 0, options) {
  const { leading = false, trailing = true, maxWait = Infinity } = options ?? {};
  let timerId;
  let startTime;
  if (maxWait < waitMs) {
    throw new Error("Value of maxWait cannot be lower than waitMs.");
  }
  function debounceCallback(...args) {
    if (leading && !startTime) {
      startTime = Date.now();
      timerId = setTimeout(() => startTime = null, waitMs);
      callback2(...args);
      return;
    }
    let adjustedWaitMs = waitMs;
    if (maxWait !== Infinity && startTime) {
      const elapsedTime = Date.now() - startTime;
      if (waitMs > maxWait - elapsedTime) {
        adjustedWaitMs = maxWait - elapsedTime;
      }
    }
    clearTimeout(timerId);
    startTime ?? (startTime = Date.now());
    timerId = setTimeout(() => {
      startTime = null;
      if (trailing) {
        callback2(...args);
      }
    }, adjustedWaitMs);
  }
  return Object.assign(debounceCallback, {
    cancel() {
      clearTimeout(timerId);
      startTime = null;
    }
  });
}
function throttle(callback2, waitMs, options) {
  const { leading = true, trailing = true } = options ?? {};
  let timerId;
  let lastArgs;
  let shouldWait = false;
  function timeoutHandler() {
    if (trailing && lastArgs) {
      timerId = setTimeout(timeoutHandler, waitMs);
      callback2(...lastArgs);
    } else {
      shouldWait = false;
    }
    lastArgs = null;
  }
  function throttleCallback(...args) {
    if (shouldWait) {
      lastArgs = args;
    } else {
      shouldWait = true;
      timerId = setTimeout(timeoutHandler, waitMs);
      if (leading) {
        callback2(...args);
      } else {
        lastArgs = args;
      }
    }
  }
  return Object.assign(throttleCallback, {
    cancel() {
      clearTimeout(timerId);
      shouldWait = false;
      lastArgs = null;
    }
  });
}

// packages/ag-charts-core/src/utils/iterators.ts
function* iterate(...iterators) {
  for (const iterator of iterators) {
    yield* iterator;
  }
}
function toIterable(value) {
  return value != null && typeof value === "object" && Symbol.iterator in value ? value : [value];
}
function first(iterable) {
  for (const value of iterable) {
    return value;
  }
  throw new Error("AG Charts - no first() value found");
}

// packages/ag-charts-core/src/utils/strings.ts
function joinFormatted(values, conjunction = "and", format = String, maxItems = Infinity) {
  if (values.length === 1) {
    return format(values[0]);
  }
  values = values.map(format);
  const lastValue = values.pop();
  if (values.length >= maxItems) {
    const remainingCount = values.length - (maxItems - 1);
    return `${values.slice(0, maxItems - 1).join(", ")}, and ${remainingCount} more ${conjunction} ${lastValue}`;
  }
  return `${values.join(", ")} ${conjunction} ${lastValue}`;
}
function stringifyValue(value, maxLength = Infinity) {
  switch (typeof value) {
    case "undefined":
      return "undefined";
    case "number":
      if (isNaN(value)) {
        return "NaN";
      } else if (value === Infinity) {
        return "Infinity";
      } else if (value === -Infinity) {
        return "-Infinity";
      }
    default:
      value = JSON.stringify(value);
      if (value.length > maxLength) {
        return `${value.slice(0, maxLength)}... (+${value.length - maxLength} characters)`;
      }
      return value;
  }
}
function countLines(text) {
  let count = 1;
  for (let i = 0; i < text.length; i++) {
    if (text.charCodeAt(i) === 10) {
      count++;
    }
  }
  return count;
}

// packages/ag-charts-core/src/utils/typeGuards.ts
function isDefined(val) {
  return val != null;
}
function isArray(value) {
  return Array.isArray(value);
}
function isBoolean(value) {
  return typeof value === "boolean";
}
function isDate(value) {
  return value instanceof Date;
}
function isValidDate(value) {
  return isDate(value) && !isNaN(Number(value));
}
function isRegExp(value) {
  return value instanceof RegExp;
}
function isFunction(value) {
  return typeof value === "function";
}
function isObject(value) {
  return typeof value === "object" && value !== null && !isArray(value);
}
function isObjectLike(value) {
  return isArray(value) || isPlainObject(value);
}
function isPlainObject(value) {
  return typeof value === "object" && value !== null && Object.getPrototypeOf(value) === Object.prototype;
}
function isString(value) {
  return typeof value === "string";
}
function isNumber(value) {
  return typeof value === "number";
}
function isFiniteNumber(value) {
  return Number.isFinite(value);
}
function isHtmlElement(value) {
  return typeof window !== "undefined" && value instanceof HTMLElement;
}
function isEnumKey(enumObject, enumKey) {
  return isString(enumKey) && Object.keys(enumObject).includes(enumKey);
}
function isEnumValue(enumObject, enumValue) {
  return Object.values(enumObject).includes(enumValue);
}
function isSymbol(value) {
  return typeof value === "symbol";
}

// packages/ag-charts-core/src/utils/validation.ts
var descriptionSymbol = Symbol("description");
var requiredSymbol = Symbol("required");
function validate(options, optionsDefs2, path = "") {
  if (!isObject(options)) {
    return {
      valid: null,
      errors: [{ path, value: options, message: validateMessage(path, options, "an object") }]
    };
  }
  const optionsKeys = new Set(Object.keys(options));
  const errors = [];
  const valid = {};
  function extendPath(key) {
    if (isArray(optionsDefs2)) {
      return `${path}[${key}]`;
    }
    return path ? `${path}.${key}` : key;
  }
  for (const [key, validatorOrDefs] of Object.entries(optionsDefs2)) {
    optionsKeys.delete(key);
    const value = options[key];
    if (!validatorOrDefs[requiredSymbol] && typeof value === "undefined")
      continue;
    if (isFunction(validatorOrDefs)) {
      if (validatorOrDefs(value, options)) {
        valid[key] = value;
      } else {
        errors.push({ key, path, value, message: validateMessage(extendPath(key), value, validatorOrDefs) });
      }
    } else {
      const nestedResult = validate(value, validatorOrDefs, extendPath(key));
      valid[key] = nestedResult.valid;
      errors.push(...nestedResult.errors);
    }
  }
  for (const key of optionsKeys) {
    errors.push({
      key,
      path,
      unknown: true,
      message: `Unknown option \`${extendPath(key)}\`, ignoring.`
    });
  }
  return { valid, errors };
}
function isValid(options, optionsDefs2, path) {
  const { errors } = validate(options, optionsDefs2, path);
  return errors.length === 0;
}
function validateMessage(path, value, validatorOrDefs) {
  const description = isString(validatorOrDefs) ? validatorOrDefs : validatorOrDefs[descriptionSymbol];
  const expecting = description ? `; expecting ${description}` : "";
  const prefix = path ? `Option \`${path}\`` : "Value";
  return `${prefix} cannot be set to \`${stringifyValue(value)}\`${expecting}, ignoring.`;
}
function attachDescription(validator, description) {
  return Object.assign((value, context) => validator(value, context), {
    [descriptionSymbol]: description
  });
}
function required(validatorOrDefs) {
  return Object.assign(
    isFunction(validatorOrDefs) ? (value) => validatorOrDefs(value) : optionsDefs(validatorOrDefs),
    { [requiredSymbol]: true }
  );
}
var optionsDefs = (defs, description = "an object") => attachDescription(
  (value) => isObject(value) && Object.entries(defs).every(([key, validatorOrDefs]) => {
    const validator = isFunction(validatorOrDefs) ? validatorOrDefs : optionsDefs(validatorOrDefs);
    return validator(value[key], value);
  }),
  description
);
var and = (...validators) => attachDescription(
  (value, context) => validators.every((validator) => validator(value, context)),
  validators.map((v) => v[descriptionSymbol]).filter(Boolean).join(" and ")
);
var or = (...validators) => attachDescription(
  (value, context) => validators.some((validator) => validator(value, context)),
  validators.map((v) => v[descriptionSymbol]).filter(Boolean).join(" or ")
);
var array = attachDescription(isArray, "an array");
var boolean = attachDescription(isBoolean, "a boolean");
var callback = attachDescription(isFunction, "a function");
var number = attachDescription(isFiniteNumber, "a number");
var object = attachDescription(isObject, "an object");
var string = attachDescription(isString, "a string");
var date = attachDescription(
  (value) => isDate(value) || (isFiniteNumber(value) || isString(value)) && isValidDate(new Date(value)),
  "a date"
);
var numberMin = (min, inclusive = true) => attachDescription(
  (value) => isFiniteNumber(value) && (value > min || inclusive && value === min),
  `a number greater than ${inclusive ? "or equal to " : ""}${min}`
);
var numberRange = (min, max) => attachDescription(
  (value) => isFiniteNumber(value) && value >= min && value <= max,
  `a number between ${min} and ${max} inclusive`
);
var positiveNumber = numberMin(0);
var ratio = numberRange(0, 1);
var isComparable = (value) => isFiniteNumber(value) || isValidDate(value);
var lessThan = (otherField) => attachDescription(
  (value, context) => !isComparable(value) || !isComparable(context[otherField]) || value < context[otherField],
  `to be less than ${otherField}`
);
function union(...allowed) {
  if (isObject(allowed[0])) {
    allowed = Object.values(allowed[0]);
  }
  const keywords = joinFormatted(allowed, "or", (value) => `'${value}'`, 6);
  return attachDescription((value) => allowed.includes(value), `a keyword such as ${keywords}`);
}
var constant = (allowed) => attachDescription((value) => allowed === value, `the value ${JSON.stringify(allowed)}`);
var instanceOf = (instanceType, description) => attachDescription((value) => value instanceof instanceType, description ?? `an instance of ${instanceType.name}`);
var arrayOf = (validator, description) => attachDescription(
  (value, context) => isArray(value) && value.every((v) => validator(v, context)),
  description ?? `${validator[descriptionSymbol]} array`
);
export {
  EventEmitter,
  logger_exports as Logger,
  ModuleRegistry,
  ModuleType,
  and,
  array,
  arrayOf,
  arraysEqual,
  attachDescription,
  boolean,
  callback,
  circularSliceArray,
  constant,
  countLines,
  date,
  debounce,
  diffArrays,
  findMaxIndex,
  findMaxValue,
  findMinIndex,
  findMinValue,
  first,
  groupBy,
  instanceOf,
  isArray,
  isBoolean,
  isDate,
  isDefined,
  isEnumKey,
  isEnumValue,
  isFiniteNumber,
  isFunction,
  isHtmlElement,
  isNumber,
  isObject,
  isObjectLike,
  isPlainObject,
  isRegExp,
  isString,
  isSymbol,
  isValid,
  isValidDate,
  iterate,
  joinFormatted,
  lessThan,
  number,
  object,
  optionsDefs,
  or,
  positiveNumber,
  ratio,
  required,
  sortBasedOnArray,
  string,
  stringifyValue,
  throttle,
  toArray,
  toIterable,
  union,
  unique,
  validate
};
