var __defProp = Object.defineProperty;
var __getOwnPropDesc = Object.getOwnPropertyDescriptor;
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};
var __decorateClass = (decorators, target, key, kind) => {
  var result = kind > 1 ? void 0 : kind ? __getOwnPropDesc(target, key) : target;
  for (var i = decorators.length - 1, decorator; i >= 0; i--)
    if (decorator = decorators[i])
      result = (kind ? decorator(target, key, result) : decorator(result)) || result;
  if (kind && result)
    __defProp(target, key, result);
  return result;
};

// packages/ag-charts-core/src/globals/logger.ts
var logger_exports = {};
__export(logger_exports, {
  error: () => error,
  errorOnce: () => errorOnce,
  log: () => log,
  logGroup: () => logGroup,
  reset: () => reset,
  table: () => table,
  warn: () => warn,
  warnOnce: () => warnOnce
});
var doOnceCache = /* @__PURE__ */ new Set();
function log(...logContent) {
  console.log(...logContent);
}
function warn(message, ...logContent) {
  console.warn(`AG Charts - ${message}`, ...logContent);
}
function error(message, ...logContent) {
  if (typeof message === "object") {
    console.error(`AG Charts error`, message, ...logContent);
  } else {
    console.error(`AG Charts - ${message}`, ...logContent);
  }
}
function table(...logContent) {
  console.table(...logContent);
}
function warnOnce(message, ...logContent) {
  const cacheKey = `Logger.warn: ${message}`;
  if (doOnceCache.has(cacheKey))
    return;
  warn(message, ...logContent);
  doOnceCache.add(cacheKey);
}
function errorOnce(message, ...logContent) {
  const cacheKey = `Logger.error: ${message}`;
  if (doOnceCache.has(cacheKey))
    return;
  error(message, ...logContent);
  doOnceCache.add(cacheKey);
}
function reset() {
  doOnceCache.clear();
}
function logGroup(name, cb) {
  console.groupCollapsed(name);
  try {
    return cb();
  } finally {
    console.groupEnd();
  }
}

// packages/ag-charts-core/src/globals/moduleRegistry.ts
var ModuleRegistry = class {
  static [Symbol.iterator]() {
    return this.registeredModules.values();
  }
  static register(definition) {
    const existingDefinition = this.registeredModules.get(definition.name);
    if (existingDefinition && (existingDefinition.enterprise || !definition.enterprise)) {
      throw new Error(`AG Charts - Module '${definition.name}' already registered`);
    }
    this.registeredModules.set(definition.name, definition);
  }
  static registerMany(definitions) {
    for (const definition of definitions) {
      this.register(definition);
    }
  }
  static reset() {
    this.registeredModules.clear();
  }
  static detectChartDefinition(options) {
    return this.detectDefinition("chart" /* Chart */, options);
  }
  static detectSeriesDefinition(options) {
    return this.detectDefinition("series" /* Series */, options);
  }
  static detectDefinition(moduleType, options) {
    for (const definition of this.registeredModules.values()) {
      if (definition.type === moduleType && definition.detect(options)) {
        return definition;
      }
    }
    throw new Error(
      `AG Charts - Unknown ${moduleType} type; Check options are correctly structured and series types are specified`
    );
  }
};
ModuleRegistry.registeredModules = /* @__PURE__ */ new Map();

// packages/ag-charts-core/src/classes/eventEmitter.ts
var EventEmitter = class {
  constructor() {
    this.events = /* @__PURE__ */ new Map();
  }
  /**
   * Registers an event listener.
   * @param eventName The event name to listen for.
   * @param listener The callback to be invoked on the event.
   * @returns A function to unregister the listener.
   */
  on(eventName, listener) {
    if (!this.events.has(eventName)) {
      this.events.set(eventName, /* @__PURE__ */ new Set());
    }
    this.events.get(eventName)?.add(listener);
    return () => this.off(eventName, listener);
  }
  /**
   * Unregisters an event listener.
   * @param eventName The event name to stop listening for.
   * @param listener The callback to be removed.
   */
  off(eventName, listener) {
    const eventListeners = this.events.get(eventName);
    if (eventListeners) {
      eventListeners.delete(listener);
      if (eventListeners.size === 0) {
        this.events.delete(eventName);
      }
    }
  }
  /**
   * Emits an event to all registered listeners.
   * @param eventName The name of the event to emit.
   * @param event The event payload.
   */
  emit(eventName, event) {
    this.events.get(eventName)?.forEach((callback2) => callback2(event));
  }
  /**
   * Clears all listeners for a specific event or all events if no event name is provided.
   * @param eventName (Optional) The name of the event to clear listeners for. If not provided, all listeners for all events are cleared.
   */
  clear(eventName) {
    if (eventName) {
      this.events.delete(eventName);
    } else {
      this.events.clear();
    }
  }
};

// packages/ag-charts-core/src/utils/arrays.ts
function toArray(value) {
  if (typeof value === "undefined") {
    return [];
  }
  return Array.isArray(value) ? value : [value];
}
function groupBy(array2, iteratee) {
  return array2.reduce((result, item) => {
    const groupKey = iteratee(item);
    result[groupKey] ?? (result[groupKey] = []);
    result[groupKey].push(item);
    return result;
  }, {});
}
function arraysEqual(a, b) {
  if (a == null || b == null || a.length !== b.length) {
    return false;
  }
  for (let i = 0; i < a.length; i++) {
    if (Array.isArray(a[i]) && Array.isArray(b[i])) {
      if (!arraysEqual(a[i], b[i])) {
        return false;
      }
    } else if (a[i] !== b[i]) {
      return false;
    }
  }
  return true;
}

// packages/ag-charts-core/src/utils/binarySearch.ts
function findMaxIndex(min, max, iteratee) {
  if (min > max)
    return;
  let found;
  while (max >= min) {
    const index = Math.floor((max + min) / 2);
    const value = iteratee(index);
    if (value) {
      found = index;
      min = index + 1;
    } else {
      max = index - 1;
    }
  }
  return found;
}
function findMinIndex(min, max, iteratee) {
  if (min > max)
    return;
  let found;
  while (max >= min) {
    const index = Math.floor((max + min) / 2);
    const value = iteratee(index);
    if (value) {
      found = index;
      max = index - 1;
    } else {
      min = index + 1;
    }
  }
  return found;
}
function findMinValue(min, max, iteratee) {
  if (min > max)
    return;
  let found;
  while (max >= min) {
    const index = Math.floor((max + min) / 2);
    const value = iteratee(index);
    if (value == null) {
      min = index + 1;
    } else {
      found = value;
      max = index - 1;
    }
  }
  return found;
}

// packages/ag-charts-core/src/utils/diff.ts
function diffArrays(previous, current) {
  const size = Math.max(previous.length, current.length);
  const added = /* @__PURE__ */ new Set();
  const removed = /* @__PURE__ */ new Set();
  for (let i = 0; i < size; i++) {
    const prev = previous[i];
    const curr = current[i];
    if (prev === curr)
      continue;
    if (removed.has(curr)) {
      removed.delete(curr);
    } else if (curr) {
      added.add(curr);
    }
    if (added.has(prev)) {
      added.delete(prev);
    } else if (prev) {
      removed.add(prev);
    }
  }
  return { changed: added.size > 0 || removed.size > 0, added, removed };
}

// packages/ag-charts-core/src/utils/functions.ts
function throttle(callback2, waitMs, options) {
  const { leading = true, trailing = true } = options ?? {};
  let timerId;
  let lastArgs;
  let shouldWait = false;
  function timeoutHandler() {
    if (trailing && lastArgs) {
      timerId = setTimeout(timeoutHandler, waitMs);
      callback2(...lastArgs);
    } else {
      shouldWait = false;
    }
    lastArgs = null;
  }
  function throttleCallback(...args) {
    if (shouldWait) {
      lastArgs = args;
    } else {
      shouldWait = true;
      timerId = setTimeout(timeoutHandler, waitMs);
      if (leading) {
        callback2(...args);
      } else {
        lastArgs = args;
      }
    }
  }
  return Object.assign(throttleCallback, {
    cancel() {
      clearTimeout(timerId);
      shouldWait = false;
      lastArgs = null;
    }
  });
}

// packages/ag-charts-core/src/utils/iterators.ts
function* iterate(...iterators) {
  for (const iterator of iterators) {
    yield* iterator;
  }
}
function toIterable(value) {
  return value != null && typeof value === "object" && Symbol.iterator in value ? value : [value];
}
function first(iterable) {
  for (const value of iterable) {
    return value;
  }
  throw new Error("AG Charts - no first() value found");
}

// packages/ag-charts-core/src/utils/strings.ts
function joinFormatted(values, conjunction = "and", format = String, maxItems = Infinity) {
  if (values.length === 1) {
    return format(values[0]);
  }
  values = values.map(format);
  const lastValue = values.pop();
  if (values.length >= maxItems) {
    const remainingCount = values.length - (maxItems - 1);
    return `${values.slice(0, maxItems - 1).join(", ")}, and ${remainingCount} more ${conjunction} ${lastValue}`;
  }
  return `${values.join(", ")} ${conjunction} ${lastValue}`;
}
function stringifyValue(value, maxLength = Infinity) {
  switch (typeof value) {
    case "undefined":
      return "undefined";
    case "number":
      if (isNaN(value)) {
        return "NaN";
      } else if (value === Infinity) {
        return "Infinity";
      } else if (value === -Infinity) {
        return "-Infinity";
      }
    default:
      value = JSON.stringify(value);
      if (value.length > maxLength) {
        return `${value.slice(0, maxLength)}... (+${value.length - maxLength} characters)`;
      }
      return value;
  }
}

// packages/ag-charts-core/src/utils/typeGuards.ts
function isArray(value) {
  return Array.isArray(value);
}
function isBoolean(value) {
  return typeof value === "boolean";
}
function isDate(value) {
  return value instanceof Date;
}
function isValidDate(value) {
  return isDate(value) && !isNaN(Number(value));
}
function isRegExp(value) {
  return value instanceof RegExp;
}
function isFunction(value) {
  return typeof value === "function";
}
function isObject(value) {
  return typeof value === "object" && value !== null && !isArray(value);
}
function isPlainObject(value) {
  return typeof value === "object" && value !== null && Object.getPrototypeOf(value) === Object.prototype;
}
function isString(value) {
  return typeof value === "string";
}
function isNumber(value) {
  return typeof value === "number";
}
function isFiniteNumber(value) {
  return Number.isFinite(value);
}
function isHtmlElement(value) {
  return typeof window !== "undefined" && value instanceof HTMLElement;
}

// packages/ag-charts-core/src/utils/validation.ts
var descriptionSymbol = Symbol("description");
var requiredSymbol = Symbol("required");
function validate(options, optionsDefs, path = "") {
  if (!isObject(options)) {
    return {
      valid: null,
      errors: [{ path, value: options, message: validateMessage(path, options, "an object") }]
    };
  }
  const optionsKeys = new Set(Object.keys(options));
  const errors = [];
  const valid = {};
  function extendPath(key) {
    if (isArray(optionsDefs)) {
      return `${path}[${key}]`;
    }
    return path ? `${path}.${key}` : key;
  }
  for (const [key, validatorOrDefs] of Object.entries(optionsDefs)) {
    optionsKeys.delete(key);
    const value = options[key];
    if (!validatorOrDefs[requiredSymbol] && typeof value === "undefined")
      continue;
    if (isFunction(validatorOrDefs)) {
      if (validatorOrDefs(value, options)) {
        valid[key] = value;
      } else {
        errors.push({ key, path, value, message: validateMessage(extendPath(key), value, validatorOrDefs) });
      }
    } else {
      const nestedResult = validate(value, validatorOrDefs, extendPath(key));
      valid[key] = nestedResult.valid;
      errors.push(...nestedResult.errors);
    }
  }
  for (const key of optionsKeys) {
    errors.push({
      key,
      path,
      unknown: true,
      message: `Unknown option \`${extendPath(key)}\`, ignoring.`
    });
  }
  return { valid, errors };
}
function validateMessage(path, value, validatorOrDefs) {
  const description = isString(validatorOrDefs) ? validatorOrDefs : validatorOrDefs[descriptionSymbol];
  const expecting = description ? `; expecting ${description}` : "";
  const prefix = path ? `Option \`${path}\`` : "Value";
  return `${prefix} cannot be set to \`${stringifyValue(value)}\`${expecting}, ignoring.`;
}
function attachDescription(validator, description) {
  return Object.assign((value, context) => validator(value, context), {
    [descriptionSymbol]: description
  });
}
var and = (...validators) => attachDescription(
  (value, context) => validators.every((validator) => validator(value, context)),
  validators.map((v) => v[descriptionSymbol]).filter(Boolean).join(" and ")
);
var or = (...validators) => attachDescription(
  (value, context) => validators.some((validator) => validator(value, context)),
  validators.map((v) => v[descriptionSymbol]).filter(Boolean).join(" or ")
);
var array = attachDescription(isArray, "an array");
var boolean = attachDescription(isBoolean, "a boolean");
var callback = attachDescription(isFunction, "a function");
var number = attachDescription(isFiniteNumber, "a number");
var object = attachDescription(isObject, "an object");
var string = attachDescription(isString, "a string");
var date = attachDescription(
  (value) => isDate(value) || (isFiniteNumber(value) || isString(value)) && isValidDate(new Date(value)),
  "a date"
);
var numberMin = (min, inclusive = true) => attachDescription(
  (value) => isFiniteNumber(value) && (value > min || inclusive && value === min),
  `a number greater than ${inclusive ? "or equal to " : ""}${min}`
);
var numberRange = (min, max) => attachDescription(
  (value) => isFiniteNumber(value) && value >= min && value <= max,
  `a number between ${min} and ${max} inclusive`
);
var positiveNumber = numberMin(0);
var ratio = numberRange(0, 1);
var isComparable = (value) => isFiniteNumber(value) || isValidDate(value);
var lessThan = (otherField) => attachDescription(
  (value, context) => !isComparable(value) || !isComparable(context[otherField]) || value < context[otherField],
  `to be less than ${otherField}`
);
function union(...allowed) {
  if (isObject(allowed[0])) {
    allowed = Object.values(allowed[0]);
  }
  const keywords = joinFormatted(allowed, "or", (value) => `'${value}'`, 6);
  return attachDescription((value) => allowed.includes(value), `a keyword such as ${keywords}`);
}
var arrayOf = (validator, description) => attachDescription(
  (value, context) => isArray(value) && value.every((v) => validator(v, context)),
  description ?? `${validator[descriptionSymbol]} array`
);

// packages/ag-charts-community/src/core/globalsProxy.ts
var verifiedGlobals = {};
if (typeof window !== "undefined") {
  verifiedGlobals.window = window;
} else if (typeof global !== "undefined") {
  verifiedGlobals.window = global.window;
}
if (typeof document !== "undefined") {
  verifiedGlobals.document = document;
} else if (typeof global !== "undefined") {
  verifiedGlobals.document = global.document;
}
function getDocument(propertyName) {
  return propertyName ? verifiedGlobals.document?.[propertyName] : verifiedGlobals.document;
}
function getWindow(propertyName) {
  return propertyName ? verifiedGlobals.window?.[propertyName] : verifiedGlobals.window;
}

// packages/ag-charts-community/src/core/domElements.ts
function createElement(tagName, className, style) {
  const element2 = getDocument().createElement(tagName);
  if (typeof className === "object") {
    style = className;
    className = void 0;
  }
  if (className) {
    for (const name of className.split(" ")) {
      element2.classList.add(name);
    }
  }
  if (style) {
    Object.assign(element2.style, style);
  }
  return element2;
}
function createSvgElement(elementName) {
  return getDocument().createElementNS("http://www.w3.org/2000/svg", elementName);
}

// packages/ag-charts-community/src/core/domDownload.ts
function downloadUrl(dataUrl, fileName) {
  const body = getDocument("body");
  const element2 = createElement("a", { display: "none" });
  element2.href = dataUrl;
  element2.download = fileName;
  body.appendChild(element2);
  element2.click();
  setTimeout(() => body.removeChild(element2));
}

// packages/ag-charts-community/src/util/id.ts
var ID_MAP = /* @__PURE__ */ new Map();
function createId(instance) {
  const constructor = instance.constructor;
  const className = Object.hasOwn(constructor, "className") ? constructor.className : constructor.name;
  if (!className) {
    throw new Error(`The ${constructor} is missing the 'className' property.`);
  }
  const nextId = (ID_MAP.get(className) ?? 0) + 1;
  ID_MAP.set(className, nextId);
  return `${className}-${nextId}`;
}

// packages/ag-charts-community/src/util/bboxinterface.ts
var BBoxValues = { containsPoint, isEmpty, normalize };
function containsPoint(bbox, x, y) {
  return x >= bbox.x && x <= bbox.x + bbox.width && y >= bbox.y && y <= bbox.y + bbox.height;
}
function isEmpty(bbox) {
  return bbox == null || bbox.height === 0 || bbox.width === 0 || isNaN(bbox.height) || isNaN(bbox.width);
}
function normalize(bbox) {
  let { x, y, width: width2, height: height2 } = bbox;
  if ((width2 == null || width2 > 0) && (height2 == null || height2 > 0))
    return bbox;
  if (x != null && width2 != null && width2 < 0) {
    width2 = -width2;
    x = x - width2;
  }
  if (y != null && height2 != null && height2 < 0) {
    height2 = -height2;
    y = y - height2;
  }
  return { x, y, width: width2, height: height2 };
}

// packages/ag-charts-community/src/util/interpolating.ts
var interpolate = Symbol("interpolate");
var isInterpolating = (x) => x[interpolate] != null;

// packages/ag-charts-community/src/util/nearest.ts
function nearestSquared(x, y, objects, maxDistanceSquared = Infinity) {
  const result = { nearest: void 0, distanceSquared: maxDistanceSquared };
  for (const obj of objects) {
    const thisDistance = obj.distanceSquared(x, y);
    if (thisDistance === 0) {
      return { nearest: obj, distanceSquared: 0 };
    } else if (thisDistance < result.distanceSquared) {
      result.nearest = obj;
      result.distanceSquared = thisDistance;
    }
  }
  return result;
}

// packages/ag-charts-community/src/util/number.ts
function clamp(min, value, max) {
  return Math.min(max, Math.max(min, value));
}
function clampArray(value, array2) {
  const [min, max] = findMinMax(array2);
  return clamp(min, value, max);
}
function findMinMax(array2) {
  if (array2.length === 0)
    return [];
  const result = [Infinity, -Infinity];
  for (const val of array2) {
    if (val < result[0])
      result[0] = val;
    if (val > result[1])
      result[1] = val;
  }
  return result;
}
function findRangeExtent(array2) {
  const [min, max] = findMinMax(array2);
  return max - min;
}
function isNumberEqual(a, b, epsilon = 1e-10) {
  return a === b || Math.abs(a - b) < epsilon;
}
function isNegative(value) {
  return Math.sign(value) === -1 || Object.is(value, -0);
}
function isInteger(value) {
  return value % 1 === 0;
}
function countFractionDigits(value) {
  if (Math.floor(value) === value)
    return 0;
  let valueString = String(value);
  let exponent = 0;
  if (value < 1e-6 || value >= 1e21) {
    let exponentString;
    [valueString, exponentString] = valueString.split("e");
    if (exponentString != null) {
      exponent = Number(exponentString);
    }
  }
  const decimalPlaces2 = valueString.split(".")[1]?.length ?? 0;
  return Math.max(decimalPlaces2 - exponent, 0);
}

// packages/ag-charts-community/src/scene/bbox.ts
var _BBox = class _BBox {
  constructor(x, y, width2, height2) {
    this.x = x;
    this.y = y;
    this.width = width2;
    this.height = height2;
  }
  static fromDOMRect({ x, y, width: width2, height: height2 }) {
    return new _BBox(x, y, width2, height2);
  }
  static merge(boxes) {
    let left = Infinity;
    let top = Infinity;
    let right = -Infinity;
    let bottom = -Infinity;
    for (const box of boxes) {
      if (box.x < left) {
        left = box.x;
      }
      if (box.y < top) {
        top = box.y;
      }
      if (box.x + box.width > right) {
        right = box.x + box.width;
      }
      if (box.y + box.height > bottom) {
        bottom = box.y + box.height;
      }
    }
    return new _BBox(left, top, right - left, bottom - top);
  }
  static nearestBox(x, y, boxes) {
    return nearestSquared(x, y, boxes);
  }
  toDOMRect() {
    return {
      x: this.x,
      y: this.y,
      width: this.width,
      height: this.height,
      top: this.y,
      left: this.x,
      right: this.x + this.width,
      bottom: this.y + this.height,
      toJSON() {
        return {};
      }
    };
  }
  clone() {
    const { x, y, width: width2, height: height2 } = this;
    return new _BBox(x, y, width2, height2);
  }
  equals(other) {
    return this.x === other.x && this.y === other.y && this.width === other.width && this.height === other.height;
  }
  containsPoint(x, y) {
    return BBoxValues.containsPoint(this, x, y);
  }
  intersection(other) {
    if (!this.collidesBBox(other))
      return;
    const newX1 = clamp(other.x, this.x, other.x + other.width);
    const newY1 = clamp(other.y, this.y, other.y + other.height);
    const newX2 = clamp(other.x, this.x + this.width, other.x + other.width);
    const newY2 = clamp(other.y, this.y + this.height, other.y + other.height);
    return new _BBox(newX1, newY1, newX2 - newX1, newY2 - newY1);
  }
  collidesBBox(other) {
    return this.x < other.x + other.width && this.x + this.width > other.x && this.y < other.y + other.height && this.y + this.height > other.y;
  }
  computeCenter() {
    return { x: this.x + this.width / 2, y: this.y + this.height / 2 };
  }
  isFinite() {
    return Number.isFinite(this.x) && Number.isFinite(this.y) && Number.isFinite(this.width) && Number.isFinite(this.height);
  }
  distanceSquared(x, y) {
    if (this.containsPoint(x, y)) {
      return 0;
    }
    const dx = x - clamp(this.x, x, this.x + this.width);
    const dy = y - clamp(this.y, y, this.y + this.height);
    return dx * dx + dy * dy;
  }
  shrink(amount, position) {
    if (typeof amount === "number") {
      this.applyMargin(amount, position);
    } else {
      for (const [key, value] of Object.entries(amount)) {
        this.applyMargin(value, key);
      }
    }
    if (this.width < 0) {
      this.width = 0;
    }
    if (this.height < 0) {
      this.height = 0;
    }
    return this;
  }
  grow(amount, position) {
    if (typeof amount === "number") {
      this.applyMargin(-amount, position);
    } else {
      for (const [key, value] of Object.entries(amount)) {
        this.applyMargin(-value, key);
      }
    }
    return this;
  }
  applyMargin(value, position) {
    switch (position) {
      case "top":
        this.y += value;
      case "bottom":
        this.height -= value;
        break;
      case "left":
        this.x += value;
      case "right":
        this.width -= value;
        break;
      case "vertical":
        this.y += value;
        this.height -= value * 2;
        break;
      case "horizontal":
        this.x += value;
        this.width -= value * 2;
        break;
      case void 0:
        this.x += value;
        this.y += value;
        this.width -= value * 2;
        this.height -= value * 2;
        break;
    }
  }
  translate(x, y) {
    this.x += x;
    this.y += y;
    return this;
  }
  [interpolate](other, d) {
    return new _BBox(
      this.x * (1 - d) + other.x * d,
      this.y * (1 - d) + other.y * d,
      this.width * (1 - d) + other.width * d,
      this.height * (1 - d) + other.height * d
    );
  }
};
_BBox.zero = Object.freeze(new _BBox(0, 0, 0, 0));
_BBox.NaN = Object.freeze(new _BBox(NaN, NaN, NaN, NaN));
var BBox = _BBox;

// packages/ag-charts-community/src/scene/changeDetectable.ts
function SceneChangeDetection(opts) {
  return function(target, key) {
    const privateKey = `__${key}`;
    if (target[key]) {
      return;
    }
    prepareGetSet(target, key, privateKey, opts);
  };
}
function prepareGetSet(target, key, privateKey, opts) {
  const { type = "normal", changeCb, convertor, checkDirtyOnAssignment = false } = opts ?? {};
  const requiredOpts = { type, changeCb, checkDirtyOnAssignment, convertor };
  let setter;
  switch (type) {
    case "normal":
      setter = buildNormalSetter(privateKey, requiredOpts);
      break;
    case "transform":
      setter = buildTransformSetter(privateKey);
      break;
    case "path":
      setter = buildPathSetter(privateKey);
      break;
  }
  setter = buildCheckDirtyChain(
    buildChangeCallbackChain(buildConvertorChain(setter, requiredOpts), requiredOpts),
    requiredOpts
  );
  const getter = function() {
    return this[privateKey];
  };
  Object.defineProperty(target, key, {
    set: setter,
    get: getter,
    enumerable: true,
    configurable: true
  });
}
function buildConvertorChain(setterFn, opts) {
  const { convertor } = opts;
  if (convertor) {
    return function(value) {
      setterFn.call(this, convertor(value));
    };
  }
  return setterFn;
}
var NO_CHANGE = Symbol("no-change");
function buildChangeCallbackChain(setterFn, opts) {
  const { changeCb } = opts;
  if (changeCb) {
    return function(value) {
      const change = setterFn.call(this, value);
      if (change !== NO_CHANGE) {
        changeCb.call(this, this);
      }
      return change;
    };
  }
  return setterFn;
}
function buildCheckDirtyChain(setterFn, opts) {
  const { checkDirtyOnAssignment } = opts;
  if (checkDirtyOnAssignment) {
    return function(value) {
      const change = setterFn.call(this, value);
      if (value?._dirty === true) {
        this.markDirty();
      }
      return change;
    };
  }
  return setterFn;
}
function buildNormalSetter(privateKey, opts) {
  const { changeCb } = opts;
  return function(value) {
    const oldValue = this[privateKey];
    if (value !== oldValue) {
      this[privateKey] = value;
      this.markDirty();
      changeCb?.(this);
      return value;
    }
    return NO_CHANGE;
  };
}
function buildTransformSetter(privateKey) {
  return function(value) {
    const oldValue = this[privateKey];
    if (value !== oldValue) {
      this[privateKey] = value;
      this.markDirtyTransform();
      return value;
    }
    return NO_CHANGE;
  };
}
function buildPathSetter(privateKey) {
  return function(value) {
    const oldValue = this[privateKey];
    if (value !== oldValue) {
      this[privateKey] = value;
      if (!this._dirtyPath) {
        this._dirtyPath = true;
        this.markDirty();
      }
      return value;
    }
    return NO_CHANGE;
  };
}

// packages/ag-charts-community/src/scene/node.ts
var _Node = class _Node {
  constructor(options) {
    /** Unique number to allow creation order to be easily determined. */
    this.serialNumber = _Node._nextSerialNumber++;
    this.childNodeCounts = { groups: 0, nonGroups: 0, thisComplexity: 0, complexity: 0 };
    /** Unique node ID in the form `ClassName-NaturalNumber`. */
    this.id = createId(this);
    this.pointerEvents = 0 /* All */;
    this._dirty = true;
    this.dirtyZIndex = false;
    /**
     * To simplify the type system (especially in Selections) we don't have the `Parent` node
     * (one that has children). Instead, we mimic HTML DOM, where any node can have children.
     * But we still need to distinguish regular leaf nodes from container leafs somehow.
     */
    this.isContainerNode = false;
    this.visible = true;
    this.zIndex = 0;
    this.name = options?.name;
    this.tag = options?.tag ?? NaN;
    this.zIndex = options?.zIndex ?? 0;
  }
  static toSVG(node, width2, height2) {
    const svg = node?.toSVG();
    if (svg == null || !svg.elements.length && !svg.defs?.length)
      return;
    const root = createSvgElement("svg");
    root.setAttribute("width", String(width2));
    root.setAttribute("height", String(height2));
    root.setAttribute("viewBox", `0 0 ${width2} ${height2}`);
    if (svg.defs?.length) {
      const defs = createSvgElement("defs");
      defs.append(...svg.defs);
      root.append(defs);
    }
    root.append(...svg.elements);
    return root.outerHTML;
  }
  static *extractBBoxes(nodes, skipInvisible) {
    for (const n of nodes) {
      if (!skipInvisible || n.visible && !n.transitionOut) {
        const bbox = n.getBBox();
        if (bbox)
          yield bbox;
      }
    }
  }
  /**
   * Some arbitrary data bound to the node.
   */
  get datum() {
    return this._datum ?? this.parentNode?.datum;
  }
  set datum(datum) {
    if (this._datum !== datum) {
      this._previousDatum = this._datum;
      this._datum = datum;
    }
  }
  get previousDatum() {
    return this._previousDatum;
  }
  get layerManager() {
    return this._layerManager;
  }
  get dirty() {
    return this._dirty;
  }
  /** Perform any pre-rendering initialization. */
  preRender(renderCtx, thisComplexity = 1) {
    this.childNodeCounts.groups = 0;
    this.childNodeCounts.nonGroups = 1;
    this.childNodeCounts.complexity = thisComplexity;
    this.childNodeCounts.thisComplexity = thisComplexity;
    for (const child of this.children()) {
      const childCounts = child.preRender(renderCtx);
      this.childNodeCounts.groups += childCounts.groups;
      this.childNodeCounts.nonGroups += childCounts.nonGroups;
      this.childNodeCounts.complexity += childCounts.complexity;
    }
    return this.childNodeCounts;
  }
  render(renderCtx) {
    const { stats } = renderCtx;
    this._dirty = false;
    if (renderCtx.debugNodeSearch) {
      const idOrName = this.name ?? this.id;
      if (renderCtx.debugNodeSearch.some((v) => typeof v === "string" ? v === idOrName : v.test(idOrName))) {
        renderCtx.debugNodes[this.name ?? this.id] = this;
      }
    }
    if (stats) {
      stats.nodesRendered++;
      stats.opsPerformed += this.childNodeCounts.thisComplexity;
    }
  }
  _setLayerManager(value) {
    this._layerManager = value;
    this._debug = value?.debug;
    for (const child of this.children()) {
      child._setLayerManager(value);
    }
  }
  sortChildren(compareFn) {
    this.dirtyZIndex = false;
    if (!this.childNodes)
      return;
    const sortedChildren = [...this.childNodes].sort(compareFn);
    this.childNodes.clear();
    for (const child of sortedChildren) {
      this.childNodes.add(child);
    }
  }
  *traverseUp(includeSelf) {
    let node = this;
    if (includeSelf) {
      yield node;
    }
    while (node = node.parentNode) {
      yield node;
    }
  }
  *children() {
    if (!this.childNodes)
      return;
    for (const child of this.childNodes) {
      yield child;
    }
  }
  *descendants() {
    for (const child of this.children()) {
      yield child;
      yield* child.descendants();
    }
  }
  /**
   * Checks if the node is a leaf (has no children).
   */
  isLeaf() {
    return !this.childNodes?.size;
  }
  /**
   * Checks if the node is the root (has no parent).
   */
  isRoot() {
    return !this.parentNode;
  }
  /**
   * Appends one or more new node instances to this parent.
   * If one needs to:
   * - move a child to the end of the list of children
   * - move a child from one parent to another (including parents in other scenes)
   * one should use the {@link insertBefore} method instead.
   * @param nodes A node or nodes to append.
   */
  append(nodes) {
    this.childNodes ?? (this.childNodes = /* @__PURE__ */ new Set());
    for (const node of toIterable(nodes)) {
      node.parentNode?.removeChild(node);
      this.childNodes.add(node);
      node.parentNode = this;
      node._setLayerManager(this.layerManager);
    }
    this.invalidateCachedBBox();
    this.dirtyZIndex = true;
    this.markDirty();
  }
  appendChild(node) {
    this.append(node);
    return node;
  }
  removeChild(node) {
    if (!this.childNodes?.delete(node)) {
      return false;
    }
    delete node.parentNode;
    node._setLayerManager();
    this.invalidateCachedBBox();
    this.dirtyZIndex = true;
    this.markDirty();
    return true;
  }
  remove() {
    return this.parentNode?.removeChild(this) ?? false;
  }
  clear() {
    for (const child of this.children()) {
      delete child.parentNode;
      child._setLayerManager();
    }
    this.childNodes?.clear();
    this.invalidateCachedBBox();
  }
  destroy() {
    this.parentNode?.removeChild(this);
  }
  setProperties(styles, pickKeys) {
    if (pickKeys) {
      for (const key of pickKeys) {
        this[key] = styles[key];
      }
    } else {
      Object.assign(this, styles);
    }
    return this;
  }
  containsPoint(_x, _y) {
    return false;
  }
  /**
   * Hit testing method.
   * Recursively checks if the given point is inside this node or any of its children.
   * Returns the first matching node or `undefined`.
   * Nodes that render later (show on top) are hit tested first.
   */
  pickNode(x, y, _localCoords = false) {
    if (!this.visible || this.pointerEvents === 1 /* None */ || !this.containsPoint(x, y)) {
      return;
    }
    const children = [...this.children()];
    if (children.length > 1e3) {
      for (let i = children.length - 1; i >= 0; i--) {
        const child = children[i];
        const containsPoint2 = child.containsPoint(x, y);
        const hit = containsPoint2 ? child.pickNode(x, y) : void 0;
        if (hit) {
          return hit;
        }
      }
    } else if (children.length) {
      for (let i = children.length - 1; i >= 0; i--) {
        const hit = children[i].pickNode(x, y);
        if (hit) {
          return hit;
        }
      }
    } else if (!this.isContainerNode) {
      return this;
    }
  }
  invalidateCachedBBox() {
    if (this.cachedBBox != null) {
      this.cachedBBox = void 0;
      this.parentNode?.invalidateCachedBBox();
    }
  }
  getBBox() {
    if (this.cachedBBox == null) {
      this.cachedBBox = Object.freeze(this.computeBBox());
    }
    return this.cachedBBox;
  }
  computeBBox() {
    return;
  }
  markDirty() {
    const { _dirty } = this;
    const noParentCachedBBox = this.cachedBBox == null;
    if (noParentCachedBBox && _dirty)
      return;
    this.invalidateCachedBBox();
    this._dirty = true;
    if (this.parentNode) {
      this.parentNode.markDirty();
    }
  }
  markClean() {
    if (!this._dirty)
      return;
    this._dirty = false;
    for (const child of this.children()) {
      child.markClean();
    }
  }
  onZIndexChange() {
    const { parentNode } = this;
    if (parentNode) {
      parentNode.dirtyZIndex = true;
    }
  }
  toSVG() {
    return;
  }
};
_Node._nextSerialNumber = 0;
__decorateClass([
  SceneChangeDetection()
], _Node.prototype, "visible", 2);
__decorateClass([
  SceneChangeDetection({
    changeCb: (target) => target.onZIndexChange()
  })
], _Node.prototype, "zIndex", 2);
var Node = _Node;

// packages/ag-charts-community/src/util/color.ts
var lerp = (x, y, t) => x * (1 - t) + y * t;
var srgbToLinear = (value) => {
  const sign = value < 0 ? -1 : 1;
  const abs = Math.abs(value);
  if (abs <= 0.04045)
    return value / 12.92;
  return sign * ((abs + 0.055) / 1.055) ** 2.4;
};
var srgbFromLinear = (value) => {
  const sign = value < 0 ? -1 : 1;
  const abs = Math.abs(value);
  if (abs > 31308e-7) {
    return sign * (1.055 * abs ** (1 / 2.4) - 0.055);
  }
  return 12.92 * value;
};
var _Color = class _Color {
  /**
   * Every color component should be in the [0, 1] range.
   * Some easing functions (such as elastic easing) can overshoot the target value by some amount.
   * So, when animating colors, if the source or target color components are already near
   * or at the edge of the allowed [0, 1] range, it is possible for the intermediate color
   * component value to end up outside of that range mid-animation. For this reason the constructor
   * performs range checking/constraining.
   * @param r Red component.
   * @param g Green component.
   * @param b Blue component.
   * @param a Alpha (opacity) component.
   */
  constructor(r, g, b, a = 1) {
    this.r = clamp(0, r || 0, 1);
    this.g = clamp(0, g || 0, 1);
    this.b = clamp(0, b || 0, 1);
    this.a = clamp(0, a || 0, 1);
  }
  /**
   * A color string can be in one of the following formats to be valid:
   * - #rgb
   * - #rrggbb
   * - rgb(r, g, b)
   * - rgba(r, g, b, a)
   * - CSS color name such as 'white', 'orange', 'cyan', etc.
   */
  static validColorString(str) {
    if (str.indexOf("#") >= 0) {
      return !!_Color.parseHex(str);
    }
    if (str.indexOf("rgb") >= 0) {
      return !!_Color.stringToRgba(str);
    }
    return _Color.nameToHex.has(str.toLowerCase());
  }
  /**
   * The given string can be in one of the following formats:
   * - #rgb
   * - #rrggbb
   * - rgb(r, g, b)
   * - rgba(r, g, b, a)
   * - CSS color name such as 'white', 'orange', 'cyan', etc.
   * @param str
   */
  static fromString(str) {
    if (str.indexOf("#") >= 0) {
      return _Color.fromHexString(str);
    }
    const hex = _Color.nameToHex.get(str.toLowerCase());
    if (hex) {
      return _Color.fromHexString(hex);
    }
    if (str.indexOf("rgb") >= 0) {
      return _Color.fromRgbaString(str);
    }
    throw new Error(`Invalid color string: '${str}'`);
  }
  // See https://drafts.csswg.org/css-color/#hex-notation
  static parseHex(input) {
    input = input.replace(/ /g, "").slice(1);
    let parts;
    switch (input.length) {
      case 6:
      case 8:
        parts = [];
        for (let i = 0; i < input.length; i += 2) {
          parts.push(parseInt(`${input[i]}${input[i + 1]}`, 16));
        }
        break;
      case 3:
      case 4:
        parts = input.split("").map((p) => parseInt(p, 16)).map((p) => p + p * 16);
        break;
    }
    if (parts?.length >= 3 && parts.every((p) => p >= 0)) {
      if (parts.length === 3) {
        parts.push(255);
      }
      return parts;
    }
  }
  static fromHexString(str) {
    const values = _Color.parseHex(str);
    if (values) {
      const [r, g, b, a] = values;
      return new _Color(r / 255, g / 255, b / 255, a / 255);
    }
    throw new Error(`Malformed hexadecimal color string: '${str}'`);
  }
  static stringToRgba(str) {
    let po = -1;
    let pc = -1;
    for (let i = 0; i < str.length; i++) {
      const c = str[i];
      if (po === -1 && c === "(") {
        po = i;
      } else if (c === ")") {
        pc = i;
        break;
      }
    }
    if (po === -1 || pc === -1)
      return;
    const contents = str.substring(po + 1, pc);
    const parts = contents.split(",");
    const rgba = [];
    for (let i = 0; i < parts.length; i++) {
      const part = parts[i];
      let value = parseFloat(part);
      if (!Number.isFinite(value)) {
        return;
      }
      if (part.indexOf("%") >= 0) {
        value = clamp(0, value, 100);
        value /= 100;
      } else if (i === 3) {
        value = clamp(0, value, 1);
      } else {
        value = clamp(0, value, 255);
        value /= 255;
      }
      rgba.push(value);
    }
    return rgba;
  }
  static fromRgbaString(str) {
    const rgba = _Color.stringToRgba(str);
    if (rgba) {
      if (rgba.length === 3) {
        return new _Color(rgba[0], rgba[1], rgba[2]);
      } else if (rgba.length === 4) {
        return new _Color(rgba[0], rgba[1], rgba[2], rgba[3]);
      }
    }
    throw new Error(`Malformed rgb/rgba color string: '${str}'`);
  }
  static fromArray(arr) {
    if (arr.length === 4) {
      return new _Color(arr[0], arr[1], arr[2], arr[3]);
    }
    if (arr.length === 3) {
      return new _Color(arr[0], arr[1], arr[2]);
    }
    throw new Error("The given array should contain 3 or 4 color components (numbers).");
  }
  static fromHSB(h, s, b, alpha = 1) {
    const rgb = _Color.HSBtoRGB(h, s, b);
    return new _Color(rgb[0], rgb[1], rgb[2], alpha);
  }
  static fromHSL(h, s, l, alpha = 1) {
    const rgb = _Color.HSLtoRGB(h, s, l);
    return new _Color(rgb[0], rgb[1], rgb[2], alpha);
  }
  static fromOKLCH(l, c, h, alpha = 1) {
    const rgb = _Color.OKLCHtoRGB(l, c, h);
    return new _Color(rgb[0], rgb[1], rgb[2], alpha);
  }
  static padHex(str) {
    return str.length === 1 ? "0" + str : str;
  }
  toHexString() {
    let hex = "#" + _Color.padHex(Math.round(this.r * 255).toString(16)) + _Color.padHex(Math.round(this.g * 255).toString(16)) + _Color.padHex(Math.round(this.b * 255).toString(16));
    if (this.a < 1) {
      hex += _Color.padHex(Math.round(this.a * 255).toString(16));
    }
    return hex;
  }
  toRgbaString(fractionDigits = 3) {
    const components = [Math.round(this.r * 255), Math.round(this.g * 255), Math.round(this.b * 255)];
    const k = Math.pow(10, fractionDigits);
    if (this.a !== 1) {
      components.push(Math.round(this.a * k) / k);
      return `rgba(${components.join(", ")})`;
    }
    return `rgb(${components.join(", ")})`;
  }
  toString() {
    if (this.a === 1) {
      return this.toHexString();
    }
    return this.toRgbaString();
  }
  toHSB() {
    return _Color.RGBtoHSB(this.r, this.g, this.b);
  }
  static RGBtoOKLCH(r, g, b) {
    const LSRGB0 = srgbToLinear(r);
    const LSRGB1 = srgbToLinear(g);
    const LSRGB2 = srgbToLinear(b);
    const LMS0 = Math.cbrt(0.4122214708 * LSRGB0 + 0.5363325363 * LSRGB1 + 0.0514459929 * LSRGB2);
    const LMS1 = Math.cbrt(0.2119034982 * LSRGB0 + 0.6806995451 * LSRGB1 + 0.1073969566 * LSRGB2);
    const LMS2 = Math.cbrt(0.0883024619 * LSRGB0 + 0.2817188376 * LSRGB1 + 0.6299787005 * LSRGB2);
    const OKLAB0 = 0.2104542553 * LMS0 + 0.793617785 * LMS1 - 0.0040720468 * LMS2;
    const OKLAB1 = 1.9779984951 * LMS0 - 2.428592205 * LMS1 + 0.4505937099 * LMS2;
    const OKLAB2 = 0.0259040371 * LMS0 + 0.7827717662 * LMS1 - 0.808675766 * LMS2;
    const hue = Math.atan2(OKLAB2, OKLAB1) * 180 / Math.PI;
    const OKLCH0 = OKLAB0;
    const OKLCH1 = Math.hypot(OKLAB1, OKLAB2);
    const OKLCH2 = hue >= 0 ? hue : hue + 360;
    return [OKLCH0, OKLCH1, OKLCH2];
  }
  static OKLCHtoRGB(l, c, h) {
    const OKLAB0 = l;
    const OKLAB1 = c * Math.cos(h * Math.PI / 180);
    const OKLAB2 = c * Math.sin(h * Math.PI / 180);
    const LMS0 = (OKLAB0 + 0.3963377774 * OKLAB1 + 0.2158037573 * OKLAB2) ** 3;
    const LMS1 = (OKLAB0 - 0.1055613458 * OKLAB1 - 0.0638541728 * OKLAB2) ** 3;
    const LMS2 = (OKLAB0 - 0.0894841775 * OKLAB1 - 1.291485548 * OKLAB2) ** 3;
    const LSRGB0 = 4.0767416621 * LMS0 - 3.3077115913 * LMS1 + 0.2309699292 * LMS2;
    const LSRGB1 = -1.2684380046 * LMS0 + 2.6097574011 * LMS1 - 0.3413193965 * LMS2;
    const LSRGB2 = -0.0041960863 * LMS0 - 0.7034186147 * LMS1 + 1.707614701 * LMS2;
    const SRGB0 = srgbFromLinear(LSRGB0);
    const SRGB1 = srgbFromLinear(LSRGB1);
    const SRGB2 = srgbFromLinear(LSRGB2);
    return [SRGB0, SRGB1, SRGB2];
  }
  static RGBtoHSL(r, g, b) {
    const min = Math.min(r, g, b);
    const max = Math.max(r, g, b);
    const l = (max + min) / 2;
    let h;
    let s;
    if (max === min) {
      h = 0;
      s = 0;
    } else {
      const delta2 = max - min;
      s = l > 0.5 ? delta2 / (2 - max - min) : delta2 / (max + min);
      if (max === r) {
        h = (g - b) / delta2 + (g < b ? 6 : 0);
      } else if (max === g) {
        h = (b - r) / delta2 + 2;
      } else {
        h = (r - g) / delta2 + 4;
      }
      h *= 360 / 6;
    }
    return [h, s, l];
  }
  static HSLtoRGB(h, s, l) {
    h = (h % 360 + 360) % 360;
    if (s === 0) {
      return [l, l, l];
    }
    const q = l < 0.5 ? l * (1 + s) : l + s - l * s;
    const p = 2 * l - q;
    function hueToRgb(t) {
      if (t < 0)
        t += 1;
      if (t > 1)
        t -= 1;
      if (t < 1 / 6)
        return p + (q - p) * 6 * t;
      if (t < 1 / 2)
        return q;
      if (t < 2 / 3)
        return p + (q - p) * (2 / 3 - t) * 6;
      return p;
    }
    const r = hueToRgb(h / 360 + 1 / 3);
    const g = hueToRgb(h / 360);
    const b = hueToRgb(h / 360 - 1 / 3);
    return [r, g, b];
  }
  /**
   * Converts the given RGB triple to an array of HSB (HSV) components.
   */
  static RGBtoHSB(r, g, b) {
    const min = Math.min(r, g, b);
    const max = Math.max(r, g, b);
    const S = max === 0 ? 0 : (max - min) / max;
    let H = 0;
    if (min !== max) {
      const delta2 = max - min;
      const rc = (max - r) / delta2;
      const gc = (max - g) / delta2;
      const bc = (max - b) / delta2;
      if (r === max) {
        H = bc - gc;
      } else if (g === max) {
        H = 2 + rc - bc;
      } else {
        H = 4 + gc - rc;
      }
      H /= 6;
      if (H < 0) {
        H = H + 1;
      }
    }
    return [H * 360, S, max];
  }
  /**
   * Converts the given HSB (HSV) triple to an array of RGB components.
   */
  static HSBtoRGB(H, S, B) {
    H = (H % 360 + 360) % 360 / 360;
    let r = 0;
    let g = 0;
    let b = 0;
    if (S === 0) {
      r = g = b = B;
    } else {
      const h = (H - Math.floor(H)) * 6;
      const f = h - Math.floor(h);
      const p = B * (1 - S);
      const q = B * (1 - S * f);
      const t = B * (1 - S * (1 - f));
      switch (h >> 0) {
        case 0:
          r = B;
          g = t;
          b = p;
          break;
        case 1:
          r = q;
          g = B;
          b = p;
          break;
        case 2:
          r = p;
          g = B;
          b = t;
          break;
        case 3:
          r = p;
          g = q;
          b = B;
          break;
        case 4:
          r = t;
          g = p;
          b = B;
          break;
        case 5:
          r = B;
          g = p;
          b = q;
          break;
      }
    }
    return [r, g, b];
  }
  static mix(c0, c1, t) {
    return new _Color(lerp(c0.r, c1.r, t), lerp(c0.g, c1.g, t), lerp(c0.b, c1.b, t), lerp(c0.a, c1.a, t));
  }
};
/**
 * CSS Color Module Level 4:
 * https://drafts.csswg.org/css-color/#named-colors
 */
_Color.nameToHex = /* @__PURE__ */ new Map([
  ["aliceblue", "#F0F8FF"],
  ["antiquewhite", "#FAEBD7"],
  ["aqua", "#00FFFF"],
  ["aquamarine", "#7FFFD4"],
  ["azure", "#F0FFFF"],
  ["beige", "#F5F5DC"],
  ["bisque", "#FFE4C4"],
  ["black", "#000000"],
  ["blanchedalmond", "#FFEBCD"],
  ["blue", "#0000FF"],
  ["blueviolet", "#8A2BE2"],
  ["brown", "#A52A2A"],
  ["burlywood", "#DEB887"],
  ["cadetblue", "#5F9EA0"],
  ["chartreuse", "#7FFF00"],
  ["chocolate", "#D2691E"],
  ["coral", "#FF7F50"],
  ["cornflowerblue", "#6495ED"],
  ["cornsilk", "#FFF8DC"],
  ["crimson", "#DC143C"],
  ["cyan", "#00FFFF"],
  ["darkblue", "#00008B"],
  ["darkcyan", "#008B8B"],
  ["darkgoldenrod", "#B8860B"],
  ["darkgray", "#A9A9A9"],
  ["darkgreen", "#006400"],
  ["darkgrey", "#A9A9A9"],
  ["darkkhaki", "#BDB76B"],
  ["darkmagenta", "#8B008B"],
  ["darkolivegreen", "#556B2F"],
  ["darkorange", "#FF8C00"],
  ["darkorchid", "#9932CC"],
  ["darkred", "#8B0000"],
  ["darksalmon", "#E9967A"],
  ["darkseagreen", "#8FBC8F"],
  ["darkslateblue", "#483D8B"],
  ["darkslategray", "#2F4F4F"],
  ["darkslategrey", "#2F4F4F"],
  ["darkturquoise", "#00CED1"],
  ["darkviolet", "#9400D3"],
  ["deeppink", "#FF1493"],
  ["deepskyblue", "#00BFFF"],
  ["dimgray", "#696969"],
  ["dimgrey", "#696969"],
  ["dodgerblue", "#1E90FF"],
  ["firebrick", "#B22222"],
  ["floralwhite", "#FFFAF0"],
  ["forestgreen", "#228B22"],
  ["fuchsia", "#FF00FF"],
  ["gainsboro", "#DCDCDC"],
  ["ghostwhite", "#F8F8FF"],
  ["gold", "#FFD700"],
  ["goldenrod", "#DAA520"],
  ["gray", "#808080"],
  ["green", "#008000"],
  ["greenyellow", "#ADFF2F"],
  ["grey", "#808080"],
  ["honeydew", "#F0FFF0"],
  ["hotpink", "#FF69B4"],
  ["indianred", "#CD5C5C"],
  ["indigo", "#4B0082"],
  ["ivory", "#FFFFF0"],
  ["khaki", "#F0E68C"],
  ["lavender", "#E6E6FA"],
  ["lavenderblush", "#FFF0F5"],
  ["lawngreen", "#7CFC00"],
  ["lemonchiffon", "#FFFACD"],
  ["lightblue", "#ADD8E6"],
  ["lightcoral", "#F08080"],
  ["lightcyan", "#E0FFFF"],
  ["lightgoldenrodyellow", "#FAFAD2"],
  ["lightgray", "#D3D3D3"],
  ["lightgreen", "#90EE90"],
  ["lightgrey", "#D3D3D3"],
  ["lightpink", "#FFB6C1"],
  ["lightsalmon", "#FFA07A"],
  ["lightseagreen", "#20B2AA"],
  ["lightskyblue", "#87CEFA"],
  ["lightslategray", "#778899"],
  ["lightslategrey", "#778899"],
  ["lightsteelblue", "#B0C4DE"],
  ["lightyellow", "#FFFFE0"],
  ["lime", "#00FF00"],
  ["limegreen", "#32CD32"],
  ["linen", "#FAF0E6"],
  ["magenta", "#FF00FF"],
  ["maroon", "#800000"],
  ["mediumaquamarine", "#66CDAA"],
  ["mediumblue", "#0000CD"],
  ["mediumorchid", "#BA55D3"],
  ["mediumpurple", "#9370DB"],
  ["mediumseagreen", "#3CB371"],
  ["mediumslateblue", "#7B68EE"],
  ["mediumspringgreen", "#00FA9A"],
  ["mediumturquoise", "#48D1CC"],
  ["mediumvioletred", "#C71585"],
  ["midnightblue", "#191970"],
  ["mintcream", "#F5FFFA"],
  ["mistyrose", "#FFE4E1"],
  ["moccasin", "#FFE4B5"],
  ["navajowhite", "#FFDEAD"],
  ["navy", "#000080"],
  ["oldlace", "#FDF5E6"],
  ["olive", "#808000"],
  ["olivedrab", "#6B8E23"],
  ["orange", "#FFA500"],
  ["orangered", "#FF4500"],
  ["orchid", "#DA70D6"],
  ["palegoldenrod", "#EEE8AA"],
  ["palegreen", "#98FB98"],
  ["paleturquoise", "#AFEEEE"],
  ["palevioletred", "#DB7093"],
  ["papayawhip", "#FFEFD5"],
  ["peachpuff", "#FFDAB9"],
  ["peru", "#CD853F"],
  ["pink", "#FFC0CB"],
  ["plum", "#DDA0DD"],
  ["powderblue", "#B0E0E6"],
  ["purple", "#800080"],
  ["rebeccapurple", "#663399"],
  ["red", "#FF0000"],
  ["rosybrown", "#BC8F8F"],
  ["royalblue", "#4169E1"],
  ["saddlebrown", "#8B4513"],
  ["salmon", "#FA8072"],
  ["sandybrown", "#F4A460"],
  ["seagreen", "#2E8B57"],
  ["seashell", "#FFF5EE"],
  ["sienna", "#A0522D"],
  ["silver", "#C0C0C0"],
  ["skyblue", "#87CEEB"],
  ["slateblue", "#6A5ACD"],
  ["slategray", "#708090"],
  ["slategrey", "#708090"],
  ["snow", "#FFFAFA"],
  ["springgreen", "#00FF7F"],
  ["steelblue", "#4682B4"],
  ["tan", "#D2B48C"],
  ["teal", "#008080"],
  ["thistle", "#D8BFD8"],
  ["tomato", "#FF6347"],
  ["transparent", "#00000000"],
  ["turquoise", "#40E0D0"],
  ["violet", "#EE82EE"],
  ["wheat", "#F5DEB3"],
  ["white", "#FFFFFF"],
  ["whitesmoke", "#F5F5F5"],
  ["yellow", "#FFFF00"],
  ["yellowgreen", "#9ACD32"]
]);
var Color = _Color;

// packages/ag-charts-community/src/util/interpolate.ts
function interpolateNumber(a, b) {
  return (d) => Number(a) * (1 - d) + Number(b) * d;
}
function interpolateColor(a, b) {
  if (typeof a === "string") {
    try {
      a = Color.fromString(a);
    } catch {
      a = Color.fromArray([0, 0, 0]);
    }
  }
  if (typeof b === "string") {
    try {
      b = Color.fromString(b);
    } catch {
      b = Color.fromArray([0, 0, 0]);
    }
  }
  return (d) => Color.mix(a, b, d).toRgbaString();
}

// packages/ag-charts-community/src/util/decorator.ts
var BREAK_TRANSFORM_CHAIN = Symbol("BREAK");
var CONFIG_KEY = "__decorator_config";
function initialiseConfig(target, propertyKeyOrSymbol) {
  if (Object.getOwnPropertyDescriptor(target, CONFIG_KEY) == null) {
    Object.defineProperty(target, CONFIG_KEY, { value: {} });
  }
  const config = target[CONFIG_KEY];
  const propertyKey = propertyKeyOrSymbol.toString();
  if (typeof config[propertyKey] !== "undefined") {
    return config[propertyKey];
  }
  const valuesMap = /* @__PURE__ */ new WeakMap();
  config[propertyKey] = { setters: [], getters: [], observers: [], valuesMap };
  const descriptor = Object.getOwnPropertyDescriptor(target, propertyKeyOrSymbol);
  const prevSet = descriptor?.set;
  const prevGet = descriptor?.get;
  const getter = function() {
    let value = prevGet ? prevGet.call(this) : valuesMap.get(this);
    for (const transformFn of config[propertyKey].getters) {
      value = transformFn(this, propertyKeyOrSymbol, value);
      if (value === BREAK_TRANSFORM_CHAIN) {
        return;
      }
    }
    return value;
  };
  const setter = function(value) {
    const { setters, observers } = config[propertyKey];
    let oldValue;
    if (setters.some((f) => f.length > 2)) {
      oldValue = prevGet ? prevGet.call(this) : valuesMap.get(this);
    }
    for (const transformFn of setters) {
      value = transformFn(this, propertyKeyOrSymbol, value, oldValue);
      if (value === BREAK_TRANSFORM_CHAIN) {
        return;
      }
    }
    if (prevSet) {
      prevSet.call(this, value);
    } else {
      valuesMap.set(this, value);
    }
    for (const observerFn of observers) {
      observerFn(this, value, oldValue);
    }
  };
  Object.defineProperty(target, propertyKeyOrSymbol, {
    set: setter,
    get: getter,
    enumerable: true,
    configurable: false
  });
  return config[propertyKey];
}
function addTransformToInstanceProperty(setTransform, getTransform, configMetadata) {
  return (target, propertyKeyOrSymbol) => {
    const config = initialiseConfig(target, propertyKeyOrSymbol);
    config.setters.push(setTransform);
    if (getTransform) {
      config.getters.unshift(getTransform);
    }
    if (configMetadata) {
      Object.assign(config, configMetadata);
    }
  };
}
function addObserverToInstanceProperty(setObserver) {
  return (target, propertyKeyOrSymbol) => {
    initialiseConfig(target, propertyKeyOrSymbol).observers.push(setObserver);
  };
}
function isDecoratedObject(target) {
  return typeof target !== "undefined" && CONFIG_KEY in target;
}
function listDecoratedProperties(target) {
  const targets = /* @__PURE__ */ new Set();
  while (isDecoratedObject(target)) {
    targets.add(target?.[CONFIG_KEY]);
    target = Object.getPrototypeOf(target);
  }
  return Array.from(targets).flatMap((configMap) => Object.keys(configMap));
}
function extractDecoratedProperties(target) {
  return listDecoratedProperties(target).reduce((result, key) => {
    result[key] = target[key] ?? null;
    return result;
  }, {});
}
function extractDecoratedPropertyMetadata(target, propertyKeyOrSymbol) {
  const propertyKey = propertyKeyOrSymbol.toString();
  while (isDecoratedObject(target)) {
    const config = target[CONFIG_KEY];
    if (Object.hasOwn(config, propertyKey)) {
      return config[propertyKey];
    }
    target = Object.getPrototypeOf(target);
  }
}

// packages/ag-charts-community/src/util/object.ts
function objectsEqual(a, b) {
  if (Array.isArray(a)) {
    if (!Array.isArray(b))
      return false;
    if (a.length !== b.length)
      return false;
    return a.every((av, i) => objectsEqual(av, b[i]));
  } else if (isPlainObject(a)) {
    if (!isPlainObject(b))
      return false;
    return objectsEqualWith(a, b, objectsEqual);
  }
  return a === b;
}
function objectsEqualWith(a, b, cmp2) {
  for (const key of Object.keys(b)) {
    if (!(key in a))
      return false;
  }
  for (const key of Object.keys(a)) {
    if (!(key in b))
      return false;
    if (!cmp2(a[key], b[key]))
      return false;
  }
  return true;
}
function mergeDefaults(...sources) {
  const target = {};
  for (const source of sources) {
    if (!isObject(source))
      continue;
    const keys = isDecoratedObject(source) ? listDecoratedProperties(source) : Object.keys(source);
    for (const key of keys) {
      if (isPlainObject(target[key]) && isPlainObject(source[key])) {
        target[key] = mergeDefaults(target[key], source[key]);
      } else {
        target[key] ?? (target[key] = source[key]);
      }
    }
  }
  return target;
}
function without(object2, keys) {
  const clone2 = { ...object2 };
  for (const key of keys) {
    delete clone2[key];
  }
  return clone2;
}
function getPath(object2, path) {
  const pathArray = isArray(path) ? path : path.split(".");
  return pathArray.reduce((value, pathKey) => value[pathKey], object2);
}
var SKIP_JS_BUILTINS = /* @__PURE__ */ new Set(["__proto__", "constructor", "prototype"]);
function setPath(object2, path, newValue) {
  const pathArray = isArray(path) ? path.slice() : path.split(".");
  const lastKey = pathArray.pop();
  if (pathArray.some((p) => SKIP_JS_BUILTINS.has(p)))
    return;
  const lastObject = pathArray.reduce((value, pathKey) => value[pathKey], object2);
  lastObject[lastKey] = newValue;
  return lastObject[lastKey];
}
function partialAssign(keysToCopy, target, source) {
  if (source === void 0) {
    return target;
  }
  for (const key of keysToCopy) {
    const value = source[key];
    if (value !== void 0) {
      target[key] = value;
    }
  }
  return target;
}

// packages/ag-charts-community/src/motion/easing.ts
var linear = (n) => n;
var easeOut = (n) => Math.sin(n * Math.PI / 2);

// packages/ag-charts-community/src/motion/animation.ts
var QUICK_TRANSITION = 0.2;
var PHASE_ORDER = ["initial", "remove", "update", "add", "trailing", "end", "none"];
var PHASE_METADATA = {
  initial: {
    animationDuration: 1,
    animationDelay: 0
  },
  add: {
    animationDuration: 0.25,
    animationDelay: 0.75
  },
  remove: {
    animationDuration: 0.25,
    animationDelay: 0
  },
  update: {
    animationDuration: 0.5,
    animationDelay: 0.25
  },
  trailing: {
    animationDuration: QUICK_TRANSITION,
    animationDelay: 1,
    skipIfNoEarlierAnimations: true
  },
  end: {
    animationDelay: 1 + QUICK_TRANSITION,
    animationDuration: 0,
    skipIfNoEarlierAnimations: true
  },
  none: {
    animationDuration: 0,
    animationDelay: 0
  }
};
function isNodeArray(array2) {
  return array2.every((n) => n instanceof Node);
}
function deconstructSelectionsOrNodes(selectionsOrNodes) {
  return isNodeArray(selectionsOrNodes) ? { nodes: selectionsOrNodes, selections: [] } : { nodes: [], selections: selectionsOrNodes };
}
function animationValuesEqual(a, b) {
  if (a === b) {
    return true;
  } else if (Array.isArray(a) && Array.isArray(b)) {
    return a.length === b.length && a.every((v, i) => animationValuesEqual(v, b[i]));
  } else if (isInterpolating(a) && isInterpolating(b)) {
    return a.equals(b);
  } else if (isPlainObject(a) && isPlainObject(b)) {
    return objectsEqualWith(a, b, animationValuesEqual);
  }
  return false;
}
var Animation = class {
  constructor(opts) {
    this.isComplete = false;
    this.elapsed = 0;
    this.iteration = 0;
    this.isPlaying = false;
    this.isReverse = false;
    this.id = opts.id;
    this.groupId = opts.groupId;
    this.autoplay = opts.autoplay ?? true;
    this.ease = opts.ease ?? linear;
    this.phase = opts.phase;
    const durationProportion = opts.duration ?? PHASE_METADATA[this.phase].animationDuration;
    this.duration = durationProportion * opts.defaultDuration;
    this.delay = (opts.delay ?? 0) * opts.defaultDuration;
    this.onComplete = opts.onComplete;
    this.onPlay = opts.onPlay;
    this.onStop = opts.onStop;
    this.onUpdate = opts.onUpdate;
    this.interpolate = this.createInterpolator(opts.from, opts.to);
    this.from = opts.from;
    if (opts.skip === true) {
      this.onUpdate?.(opts.to, false, this);
      this.onStop?.(this);
      this.onComplete?.(this);
      this.isComplete = true;
    }
    if (opts.collapsable !== false) {
      this.duration = this.checkCollapse(opts, this.duration);
    }
  }
  checkCollapse(opts, calculatedDuration) {
    return animationValuesEqual(opts.from, opts.to) ? 0 : calculatedDuration;
  }
  play(initialUpdate = false) {
    if (this.isPlaying || this.isComplete)
      return;
    this.isPlaying = true;
    this.onPlay?.(this);
    if (!this.autoplay)
      return;
    this.autoplay = false;
    if (!initialUpdate)
      return;
    this.onUpdate?.(this.from, true, this);
  }
  stop() {
    this.isPlaying = false;
    if (!this.isComplete) {
      this.isComplete = true;
      this.onStop?.(this);
    }
  }
  update(time2) {
    if (this.isComplete)
      return time2;
    if (!this.isPlaying && this.autoplay) {
      this.play(true);
    }
    const previousElapsed = this.elapsed;
    this.elapsed += time2;
    if (this.delay > this.elapsed)
      return 0;
    const value = this.interpolate(this.isReverse ? 1 - this.delta : this.delta);
    this.onUpdate?.(value, false, this);
    const totalDuration = this.delay + this.duration;
    if (this.elapsed >= totalDuration) {
      this.stop();
      this.isComplete = true;
      this.onComplete?.(this);
      return time2 - (totalDuration - previousElapsed);
    }
    return 0;
  }
  get delta() {
    return this.ease(clamp(0, (this.elapsed - this.delay) / this.duration, 1));
  }
  createInterpolator(from2, to) {
    if (typeof to !== "object" || isInterpolating(to)) {
      return this.interpolateValue(from2, to);
    }
    const interpolatorEntries = [];
    for (const key of Object.keys(to)) {
      const interpolator = this.interpolateValue(from2[key], to[key]);
      if (interpolator != null) {
        interpolatorEntries.push([key, interpolator]);
      }
    }
    return (d) => {
      const result = {};
      for (const [key, interpolator] of interpolatorEntries) {
        result[key] = interpolator(d);
      }
      return result;
    };
  }
  interpolateValue(a, b) {
    if (a == null || b == null) {
      return;
    } else if (isInterpolating(a)) {
      return (d) => a[interpolate](b, d);
    }
    try {
      switch (typeof a) {
        case "number":
          return interpolateNumber(a, b);
        case "string":
          return interpolateColor(a, b);
        case "boolean":
          if (a === b) {
            return () => a;
          }
          break;
      }
    } catch {
    }
    throw new Error(`Unable to interpolate values: ${a}, ${b}`);
  }
};

// packages/ag-charts-community/src/motion/fromToMotion.ts
var NODE_UPDATE_STATE_TO_PHASE_MAPPING = {
  added: "add",
  updated: "update",
  removed: "remove",
  unknown: "initial",
  "no-op": "none"
};
function fromToMotion(groupId, subId, animationManager, selectionsOrNodes, fns, getDatumId, diff) {
  const { fromFn, toFn, applyFn = (node, props) => node.setProperties(props) } = fns;
  const { nodes, selections } = deconstructSelectionsOrNodes(selectionsOrNodes);
  const processNodes = (liveNodes, subNodes) => {
    let prevFromProps;
    let liveNodeIndex = 0;
    let nodeIndex = 0;
    for (const node of subNodes) {
      const isLive = liveNodes[liveNodeIndex] === node;
      const ctx = {
        last: nodeIndex >= subNodes.length - 1,
        lastLive: liveNodeIndex >= liveNodes.length - 1,
        prev: subNodes[nodeIndex - 1],
        prevFromProps,
        prevLive: liveNodes[liveNodeIndex - 1],
        next: subNodes[nodeIndex + 1],
        nextLive: liveNodes[liveNodeIndex + (isLive ? 1 : 0)]
      };
      const animationId = `${groupId}_${subId}_${node.id}`;
      animationManager.stopByAnimationId(animationId);
      let status = "unknown";
      if (!isLive) {
        status = "removed";
      } else if (getDatumId && diff) {
        status = calculateStatus(node, node.datum, getDatumId, diff);
      }
      node.transitionOut = status === "removed";
      const { phase, start: start2, finish, delay, duration, ...from2 } = fromFn(node, node.datum, status, ctx);
      const {
        phase: toPhase,
        start: toStart,
        finish: toFinish,
        delay: toDelay,
        duration: toDuration,
        ...to
      } = toFn(node, node.datum, status, ctx);
      const collapsable = finish == null;
      animationManager.animate({
        id: animationId,
        groupId,
        phase: phase ?? toPhase ?? "update",
        duration: duration ?? toDuration,
        delay: delay ?? toDelay,
        from: from2,
        to,
        ease: easeOut,
        collapsable,
        onPlay: () => {
          const startProps = { ...start2, ...toStart, ...from2 };
          applyFn(node, startProps, "start");
        },
        onUpdate(props) {
          applyFn(node, props, "update");
        },
        onStop: () => {
          const endProps = {
            ...start2,
            ...toStart,
            ...from2,
            ...to,
            ...finish,
            ...toFinish
          };
          applyFn(node, endProps, "end");
        }
      });
      if (isLive) {
        liveNodeIndex++;
      }
      nodeIndex++;
      prevFromProps = from2;
    }
  };
  let selectionIndex = 0;
  for (const selection of selections) {
    const selectionNodes = selection.nodes();
    const liveNodes = selectionNodes.filter((n) => !selection.isGarbage(n));
    processNodes(liveNodes, selectionNodes);
    animationManager.animate({
      id: `${groupId}_${subId}_selection_${selectionIndex}`,
      groupId,
      phase: "end",
      from: 0,
      to: 1,
      ease: easeOut,
      onStop() {
        selection.cleanup();
      }
    });
    selectionIndex++;
  }
  processNodes(nodes, nodes);
}
function staticFromToMotion(groupId, subId, animationManager, selectionsOrNodes, from2, to, extraOpts) {
  const { nodes, selections } = deconstructSelectionsOrNodes(selectionsOrNodes);
  const { start: start2, finish, phase } = extraOpts;
  animationManager.animate({
    id: `${groupId}_${subId}`,
    groupId,
    phase: phase ?? "update",
    from: from2,
    to,
    ease: easeOut,
    onPlay: () => {
      if (!start2)
        return;
      for (const node of nodes) {
        node.setProperties(start2);
      }
      for (const selection of selections) {
        for (const node of selection.nodes()) {
          node.setProperties(start2);
        }
      }
    },
    onUpdate(props) {
      for (const node of nodes) {
        node.setProperties(props);
      }
      for (const selection of selections) {
        for (const node of selection.nodes()) {
          node.setProperties(props);
        }
      }
    },
    onStop: () => {
      for (const node of nodes) {
        node.setProperties({ ...to, ...finish });
      }
      for (const selection of selections) {
        for (const node of selection.nodes()) {
          node.setProperties({ ...to, ...finish });
        }
        selection.cleanup();
      }
    }
  });
}
function calculateStatus(node, datum, getDatumId, diff) {
  const id = getDatumId(node, datum);
  if (diff.added.has(id)) {
    return "added";
  }
  if (diff.removed.has(id)) {
    return "removed";
  }
  return "updated";
}

// packages/ag-charts-community/src/scale/abstractScale.ts
var AbstractScale = class {
  ticks(_ticks, _domain, _visibleRange) {
    return void 0;
  }
  niceDomain(_ticks, domain = this.domain) {
    return domain;
  }
  tickFormatter(_params) {
    return void 0;
  }
  datumFormatter(_params) {
    return void 0;
  }
  get bandwidth() {
    return void 0;
  }
  get step() {
    return void 0;
  }
  get inset() {
    return void 0;
  }
};

// packages/ag-charts-community/src/scale/continuousScale.ts
var _ContinuousScale = class _ContinuousScale extends AbstractScale {
  constructor(domain = [], range2 = []) {
    super();
    this.domain = domain;
    this.range = range2;
    this.defaultClamp = false;
  }
  static is(value) {
    return value instanceof _ContinuousScale;
  }
  normalizeDomains(...domains) {
    let min;
    let minValue = Infinity;
    let max;
    let maxValue = -Infinity;
    for (const domain of domains) {
      for (const d of domain) {
        const value = d.valueOf();
        if (value < minValue) {
          minValue = value;
          min = d;
        }
        if (value > maxValue) {
          maxValue = value;
          max = d;
        }
      }
    }
    if (min != null && max != null) {
      const domain = [min, max];
      return { domain, animatable: true };
    } else {
      return { domain: [], animatable: false };
    }
  }
  transform(x) {
    return x;
  }
  transformInvert(x) {
    return x;
  }
  calcBandwidth(smallestInterval = 1) {
    const { domain } = this;
    const rangeDistance = this.getPixelRange();
    if (domain.length === 0)
      return rangeDistance;
    const intervals = Math.abs(domain[1].valueOf() - domain[0].valueOf()) / smallestInterval + 1;
    const maxBands = Math.floor(rangeDistance);
    const bands = Math.min(intervals, maxBands);
    return rangeDistance / Math.max(1, bands);
  }
  convert(value, clamp2 = this.defaultClamp) {
    const { domain } = this;
    if (!domain || domain.length < 2) {
      return NaN;
    }
    const d0 = Number(this.transform(domain[0]));
    const d1 = Number(this.transform(domain[1]));
    const x = Number(this.transform(value));
    const { range: range2 } = this;
    const [r0, r1] = range2;
    if (clamp2) {
      const [start2, stop] = findMinMax([d0, d1]);
      if (x < start2) {
        return r0;
      } else if (x > stop) {
        return r1;
      }
    }
    if (d0 === d1) {
      return (r0 + r1) / 2;
    } else if (x === d0) {
      return r0;
    } else if (x === d1) {
      return r1;
    }
    return r0 + (x - d0) / (d1 - d0) * (r1 - r0);
  }
  invert(x, _nearest) {
    const domain = this.domain.map((d2) => this.transform(d2));
    const [d0, d1] = domain;
    const { range: range2 } = this;
    const [r0, r1] = range2;
    let d;
    if (r0 === r1) {
      d = this.toDomain((Number(d0) + Number(d1)) / 2);
    } else {
      d = this.toDomain(Number(d0) + (x - r0) / (r1 - r0) * (Number(d1) - Number(d0)));
    }
    return this.transformInvert(d);
  }
  getPixelRange() {
    const [a, b] = this.range;
    return Math.abs(b - a);
  }
};
_ContinuousScale.defaultTickCount = 5;
var ContinuousScale = _ContinuousScale;

// packages/ag-charts-community/src/util/time/interval.ts
var TimeInterval = class {
  constructor(_encode, _decode, _rangeCallback) {
    this._encode = _encode;
    this._decode = _decode;
    this._rangeCallback = _rangeCallback;
  }
  /**
   * Returns a new date representing the latest interval boundary date before or equal to date.
   * For example, `day.floor(date)` typically returns 12:00 AM local time on the given date.
   * @param date
   */
  floor(date2) {
    const d = new Date(date2);
    const e = this._encode(d);
    return this._decode(e);
  }
  /**
   * Returns a new date representing the earliest interval boundary date after or equal to date.
   * @param date
   */
  ceil(date2) {
    const d = new Date(Number(date2) - 1);
    const e = this._encode(d);
    return this._decode(e + 1);
  }
  /**
   * Returns an array of dates representing every interval boundary after or equal to start (inclusive) and before stop (exclusive).
   * @param start Range start.
   * @param stop Range end.
   * @param extend If specified, the requested range will be extended to the closest "nice" values.
   */
  range(start2, stop, { extend = false, visibleRange = [0, 1] } = {}) {
    let reversed = false;
    if (start2.getTime() > stop.getTime()) {
      [start2, stop] = [stop, start2];
      reversed = true;
    }
    const rangeCallback = this._rangeCallback?.(start2, stop);
    const e0 = this._encode(extend ? this.floor(start2) : this.ceil(start2));
    const e1 = this._encode(extend ? this.ceil(stop) : this.floor(stop));
    if (e1 < e0) {
      return [];
    }
    const de = e1 - e0;
    let startIndex;
    let endIndex;
    if (reversed) {
      startIndex = Math.ceil(e0 + (1 - visibleRange[1]) * de);
      endIndex = Math.floor(e0 + (1 - visibleRange[0]) * de);
    } else {
      startIndex = Math.floor(e0 + visibleRange[0] * de);
      endIndex = Math.ceil(e0 + visibleRange[1] * de);
    }
    const range2 = [];
    for (let e = startIndex; e <= endIndex; e += 1) {
      const d = this._decode(e);
      range2.push(d);
    }
    rangeCallback?.();
    return range2;
  }
};
var CountableTimeInterval = class extends TimeInterval {
  getOffset(snapTo, step) {
    return Math.floor(this._encode(new Date(snapTo))) % step;
  }
  /**
   * Returns a filtered view of this interval representing every step'th date.
   * It can be a number of minutes, hours, days etc.
   * Must be a positive integer.
   * @param step
   */
  every(step, options) {
    let offset4 = 0;
    let rangeCallback;
    const unsafeStep = step;
    step = Math.max(1, Math.round(step));
    if (unsafeStep !== step) {
      logger_exports.warnOnce(`interval step of [${unsafeStep}] rounded to [${step}].`);
    }
    const { snapTo = "start" } = options ?? {};
    if (typeof snapTo === "string") {
      const initialOffset = offset4;
      rangeCallback = (start2, stop) => {
        const s = snapTo === "start" ? start2 : stop;
        offset4 = this.getOffset(s, step);
        return () => offset4 = initialOffset;
      };
    } else if (typeof snapTo === "number") {
      offset4 = this.getOffset(new Date(snapTo), step);
    } else if (snapTo instanceof Date) {
      offset4 = this.getOffset(snapTo, step);
    }
    const encode8 = (date2) => Math.floor((this._encode(date2) - offset4) / step);
    const decode8 = (encoded) => this._decode(encoded * step + offset4);
    return new TimeInterval(encode8, decode8, rangeCallback);
  }
};

// packages/ag-charts-community/src/util/time/millisecond.ts
function encode(date2) {
  return date2.getTime();
}
function decode(encoded) {
  return new Date(encoded);
}
var millisecond = new CountableTimeInterval(encode, decode);

// packages/ag-charts-community/src/util/time/duration.ts
var durationSecond = 1e3;
var durationMinute = durationSecond * 60;
var durationHour = durationMinute * 60;
var durationDay = durationHour * 24;
var durationWeek = durationDay * 7;
var durationMonth = durationDay * 30;
var durationYear = durationDay * 365;

// packages/ag-charts-community/src/util/time/second.ts
var offset = (/* @__PURE__ */ new Date()).getTimezoneOffset() * durationMinute;
function encode2(date2) {
  return Math.floor((date2.getTime() - offset) / durationSecond);
}
function decode2(encoded) {
  return new Date(offset + encoded * durationSecond);
}
var second = new CountableTimeInterval(encode2, decode2);

// packages/ag-charts-community/src/util/time/minute.ts
var offset2 = (/* @__PURE__ */ new Date()).getTimezoneOffset() * durationMinute;
function encode3(date2) {
  return Math.floor((date2.getTime() - offset2) / durationMinute);
}
function decode3(encoded) {
  return new Date(offset2 + encoded * durationMinute);
}
var minute = new CountableTimeInterval(encode3, decode3);

// packages/ag-charts-community/src/util/time/hour.ts
var offset3 = (/* @__PURE__ */ new Date()).getTimezoneOffset() * durationMinute;
function encode4(date2) {
  return Math.floor((date2.getTime() - offset3) / durationHour);
}
function decode4(encoded) {
  return new Date(offset3 + encoded * durationHour);
}
var hour = new CountableTimeInterval(encode4, decode4);

// packages/ag-charts-community/src/util/time/day.ts
function encode5(date2) {
  const tzOffsetMs = date2.getTimezoneOffset() * durationMinute;
  return Math.floor((date2.getTime() - tzOffsetMs) / durationDay);
}
function decode5(encoded) {
  const d = new Date(1970, 0, 1);
  d.setDate(d.getDate() + encoded);
  return d;
}
var day = new CountableTimeInterval(encode5, decode5);

// packages/ag-charts-community/src/util/time/week.ts
function weekday(weekStart) {
  const thursday2 = 4;
  const dayShift = (7 + weekStart - thursday2) % 7;
  function encode8(date2) {
    const tzOffsetMs = date2.getTimezoneOffset() * durationMinute;
    return Math.floor((date2.getTime() - tzOffsetMs) / durationWeek - dayShift / 7);
  }
  function decode8(encoded) {
    const d = new Date(1970, 0, 1);
    d.setDate(d.getDate() + encoded * 7 + dayShift);
    return d;
  }
  return new CountableTimeInterval(encode8, decode8);
}
var sunday = weekday(0);
var monday = weekday(1);
var tuesday = weekday(2);
var wednesday = weekday(3);
var thursday = weekday(4);
var friday = weekday(5);
var saturday = weekday(6);

// packages/ag-charts-community/src/util/time/month.ts
function encode6(date2) {
  return date2.getFullYear() * 12 + date2.getMonth();
}
function decode6(encoded) {
  const year2 = Math.floor(encoded / 12);
  const month2 = encoded - year2 * 12;
  return new Date(year2, month2, 1);
}
var month = new CountableTimeInterval(encode6, decode6);

// packages/ag-charts-community/src/util/time/year.ts
function encode7(date2) {
  return date2.getFullYear();
}
function decode7(encoded) {
  const d = /* @__PURE__ */ new Date();
  d.setFullYear(encoded);
  d.setMonth(0, 1);
  d.setHours(0, 0, 0, 0);
  return d;
}
var year = new CountableTimeInterval(encode7, decode7);

// packages/ag-charts-community/src/util/timeFormat.ts
var CONSTANTS = {
  periods: ["AM", "PM"],
  days: ["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"],
  shortDays: ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"],
  months: [
    "January",
    "February",
    "March",
    "April",
    "May",
    "June",
    "July",
    "August",
    "September",
    "October",
    "November",
    "December"
  ],
  shortMonths: ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"]
};
function dayOfYear(date2, startOfYear = new Date(date2.getFullYear(), 0, 1)) {
  const startOffset = date2.getTimezoneOffset() - startOfYear.getTimezoneOffset();
  const timeDiff = date2.getTime() - startOfYear.getTime() + startOffset * 6e4;
  const timeOneDay = 36e5 * 24;
  return Math.floor(timeDiff / timeOneDay);
}
function weekOfYear(date2, startDay) {
  const startOfYear = new Date(date2.getFullYear(), 0, 1);
  const startOfYearDay = startOfYear.getDay();
  const firstWeekStartOffset = (startDay - startOfYearDay + 7) % 7;
  const startOffset = new Date(date2.getFullYear(), 0, firstWeekStartOffset + 1);
  if (startOffset <= date2) {
    return Math.floor(dayOfYear(date2, startOffset) / 7) + 1;
  }
  return 0;
}
var SUNDAY = 0;
var MONDAY = 1;
var THURSDAY = 4;
function isoWeekOfYear(date2, year2 = date2.getFullYear()) {
  const firstOfYear = new Date(year2, 0, 1);
  const firstOfYearDay = firstOfYear.getDay();
  const firstThursdayOffset = (THURSDAY - firstOfYearDay + 7) % 7;
  const startOffset = new Date(year2, 0, firstThursdayOffset - (THURSDAY - MONDAY) + 1);
  if (startOffset <= date2) {
    return Math.floor(dayOfYear(date2, startOffset) / 7) + 1;
  }
  return isoWeekOfYear(date2, year2 - 1);
}
function timezone(date2) {
  const offset4 = date2.getTimezoneOffset();
  const unsignedOffset = Math.abs(offset4);
  const sign = offset4 > 0 ? "-" : "+";
  return `${sign}${pad(Math.floor(unsignedOffset / 60), 2, "0")}${pad(Math.floor(unsignedOffset % 60), 2, "0")}`;
}
var FORMATTERS = {
  a: (d) => CONSTANTS.shortDays[d.getDay()],
  A: (d) => CONSTANTS.days[d.getDay()],
  b: (d) => CONSTANTS.shortMonths[d.getMonth()],
  B: (d) => CONSTANTS.months[d.getMonth()],
  c: "%x, %X",
  d: (d, p) => pad(d.getDate(), 2, p ?? "0"),
  e: "%_d",
  f: (d, p) => pad(d.getMilliseconds() * 1e3, 6, p ?? "0"),
  H: (d, p) => pad(d.getHours(), 2, p ?? "0"),
  I: (d, p) => {
    const hours = d.getHours() % 12;
    return hours === 0 ? "12" : pad(hours, 2, p ?? "0");
  },
  j: (d, p) => pad(dayOfYear(d) + 1, 3, p ?? "0"),
  m: (d, p) => pad(d.getMonth() + 1, 2, p ?? "0"),
  M: (d, p) => pad(d.getMinutes(), 2, p ?? "0"),
  L: (d, p) => pad(d.getMilliseconds(), 3, p ?? "0"),
  p: (d) => d.getHours() < 12 ? "AM" : "PM",
  Q: (d) => String(d.getTime()),
  s: (d) => String(Math.floor(d.getTime() / 1e3)),
  S: (d, p) => pad(d.getSeconds(), 2, p ?? "0"),
  u: (d) => {
    let day2 = d.getDay();
    if (day2 < 1)
      day2 += 7;
    return String(day2 % 7);
  },
  U: (d, p) => pad(weekOfYear(d, SUNDAY), 2, p ?? "0"),
  V: (d, p) => pad(isoWeekOfYear(d), 2, p ?? "0"),
  w: (d, p) => pad(d.getDay(), 2, p ?? "0"),
  W: (d, p) => pad(weekOfYear(d, MONDAY), 2, p ?? "0"),
  x: "%-m/%-d/%Y",
  X: "%-I:%M:%S %p",
  y: (d, p) => pad(d.getFullYear() % 100, 2, p ?? "0"),
  Y: (d, p) => pad(d.getFullYear(), 4, p ?? "0"),
  Z: (d) => timezone(d),
  "%": () => "%"
};
var PADS = {
  _: " ",
  "0": "0",
  "-": ""
};
function pad(value, size, padChar) {
  const output = String(Math.floor(value));
  if (output.length >= size) {
    return output;
  }
  return `${padChar.repeat(size - output.length)}${output}`;
}
function buildFormatter(formatString) {
  const formatParts = [];
  while (formatString.length > 0) {
    let nextEscapeIdx = formatString.indexOf("%");
    if (nextEscapeIdx !== 0) {
      const literalPart = nextEscapeIdx > 0 ? formatString.substring(0, nextEscapeIdx) : formatString;
      formatParts.push(literalPart);
    }
    if (nextEscapeIdx < 0)
      break;
    const maybePadSpecifier = formatString[nextEscapeIdx + 1];
    const maybePad = PADS[maybePadSpecifier];
    if (maybePad != null) {
      nextEscapeIdx++;
    }
    const maybeFormatterSpecifier = formatString[nextEscapeIdx + 1];
    const maybeFormatter = FORMATTERS[maybeFormatterSpecifier];
    if (typeof maybeFormatter === "function") {
      formatParts.push([maybeFormatter, maybePad]);
    } else if (typeof maybeFormatter === "string") {
      const formatter = buildFormatter(maybeFormatter);
      formatParts.push([formatter, maybePad]);
    } else {
      formatParts.push(`${maybePad ?? ""}${maybeFormatterSpecifier}`);
    }
    formatString = formatString.substring(nextEscapeIdx + 2);
  }
  return (dateTime) => {
    const dateTimeAsDate = typeof dateTime === "number" ? new Date(dateTime) : dateTime;
    return formatParts.map((c) => typeof c === "string" ? c : c[0](dateTimeAsDate, c[1])).join("");
  };
}

// packages/ag-charts-community/src/util/timeFormatDefaults.ts
function dateToNumber(value) {
  return value instanceof Date ? value.getTime() : value;
}
function defaultTimeTickFormat(ticks, domain, formatOffset) {
  const formatString = calculateDefaultTimeTickFormat(ticks, domain, formatOffset);
  const formatter = buildFormatter(formatString);
  return (date2) => formatter(date2);
}
function calculateDefaultTimeTickFormat(ticks, domain, formatOffset = 0) {
  let minInterval = Infinity;
  for (let i = 1; i < ticks.length; i++) {
    minInterval = Math.min(minInterval, Math.abs(ticks[i].valueOf() - ticks[i - 1].valueOf()));
  }
  const [d0, d1] = domain.length === 0 ? [0, 0] : findMinMax([domain[0].valueOf(), domain[domain.length - 1].valueOf()]);
  const startYear = new Date(d0).getFullYear();
  const stopYear = new Date(d1).getFullYear();
  const yearChange = stopYear - startYear > 0;
  const timeFormat = isFinite(minInterval) ? getIntervalLowestGranularityFormat(minInterval, ticks) : getLowestGranularityFormat(ticks[0]);
  return formatStringBuilder(Math.max(timeFormat - formatOffset, 0), yearChange, ticks);
}
function getIntervalLowestGranularityFormat(value, ticks) {
  if (value < durationSecond) {
    return 0 /* MILLISECOND */;
  } else if (value < durationMinute) {
    return 1 /* SECOND */;
  } else if (value < durationHour) {
    return 2 /* MINUTE */;
  } else if (value < durationDay) {
    return 3 /* HOUR */;
  } else if (value < durationWeek) {
    return 4 /* WEEK_DAY */;
  } else if (value < durationDay * 28 || value < durationDay * 31 && hasDuplicateMonth(ticks)) {
    return 5 /* SHORT_MONTH */;
  } else if (value < durationYear) {
    return 6 /* MONTH */;
  }
  return 7 /* YEAR */;
}
function getLowestGranularityFormat(value) {
  if (second.floor(value) < value) {
    return 0 /* MILLISECOND */;
  } else if (minute.floor(value) < value) {
    return 1 /* SECOND */;
  } else if (hour.floor(value) < value) {
    return 2 /* MINUTE */;
  } else if (day.floor(value) < value) {
    return 3 /* HOUR */;
  } else if (month.floor(value) < value) {
    if (sunday.floor(value) < value) {
      return 4 /* WEEK_DAY */;
    }
    return 5 /* SHORT_MONTH */;
  } else if (year.floor(value) < value) {
    return 6 /* MONTH */;
  }
  return 7 /* YEAR */;
}
function hasDuplicateMonth(ticks) {
  let prevMonth = new Date(ticks[0]).getMonth();
  for (let i = 1; i < ticks.length; i++) {
    const tickMonth = new Date(ticks[i]).getMonth();
    if (prevMonth === tickMonth) {
      return true;
    }
    prevMonth = tickMonth;
  }
  return false;
}
function formatStringBuilder(defaultTimeFormat, yearChange, ticks) {
  const firstTick = dateToNumber(ticks[0]);
  const lastTick = dateToNumber(ticks.at(-1));
  const extent2 = Math.abs(lastTick - firstTick);
  const activeYear = yearChange || defaultTimeFormat === 7 /* YEAR */;
  const activeDate = extent2 === 0;
  const parts = [
    ["hour", 6 * durationHour, 14 * durationDay, 3 /* HOUR */, "%I %p"],
    ["hour", durationMinute, 6 * durationHour, 3 /* HOUR */, "%I:%M"],
    ["second", 1e3, 6 * durationHour, 1 /* SECOND */, ":%S"],
    ["ms", 0, 6 * durationHour, 0 /* MILLISECOND */, ".%L"],
    ["am/pm", durationMinute, 6 * durationHour, 3 /* HOUR */, "%p"],
    " ",
    ["day", durationDay, durationWeek, 4 /* WEEK_DAY */, "%a"],
    ["month", activeDate ? 0 : durationWeek, 52 * durationWeek, 5 /* SHORT_MONTH */, "%b %d"],
    ["month", 5 * durationWeek, 10 * durationYear, 6 /* MONTH */, "%B"],
    " ",
    ["year", activeYear ? 0 : durationYear, Infinity, 7 /* YEAR */, "%Y"]
  ];
  const formatParts = parts.filter((v) => {
    if (typeof v === "string") {
      return true;
    }
    const [_, min, max, format] = v;
    return format >= defaultTimeFormat && min <= extent2 && extent2 < max;
  }).reduce(
    (r, next) => {
      if (typeof next === "string") {
        r.result.push(next);
      } else if (!r.used.has(next[0])) {
        r.result.push(next);
        r.used.add(next[0]);
      }
      return r;
    },
    { result: [], used: /* @__PURE__ */ new Set() }
  ).result;
  const firstFormat = formatParts.findIndex((v) => typeof v !== "string");
  const lastFormat = formatParts.findLastIndex((v) => typeof v !== "string");
  return formatParts.slice(firstFormat, lastFormat + 1).map((v) => typeof v === "string" ? v : v[4]).join("").replaceAll(/\s+/g, " ").trim();
}

// packages/ag-charts-community/src/scale/invalidating.ts
var Invalidating = (target, propertyKey) => {
  const mappedProperty = Symbol(String(propertyKey));
  target[mappedProperty] = void 0;
  Object.defineProperty(target, propertyKey, {
    get() {
      return this[mappedProperty];
    },
    set(newValue) {
      const oldValue = this[mappedProperty];
      if (oldValue !== newValue) {
        this[mappedProperty] = newValue;
        this.invalid = true;
      }
    },
    enumerable: true,
    configurable: false
  });
};

// packages/ag-charts-community/src/scale/scaleUtil.ts
function filterVisibleTicks(ticks, reversed, visibleRange) {
  if (visibleRange == null || visibleRange[0] === 0 && visibleRange[1] === 1)
    return ticks;
  const vt0 = clamp(0, Math.floor(visibleRange[0] * ticks.length), ticks.length);
  const vt1 = clamp(0, Math.ceil(visibleRange[1] * ticks.length), ticks.length);
  const t0 = reversed ? ticks.length - vt1 : vt0;
  const t1 = reversed ? ticks.length - vt0 : vt1;
  return ticks.slice(t0, t1);
}

// packages/ag-charts-community/src/scale/bandScale.ts
var _BandScale = class _BandScale extends AbstractScale {
  constructor() {
    super(...arguments);
    this.invalid = true;
    this.range = [0, 1];
    this.round = false;
    this.interval = void 0;
    this._bandwidth = 1;
    this._step = 1;
    this._inset = 1;
    this._rawBandwidth = 1;
    /**
     * The ratio of the range that is reserved for space between bands.
     */
    this._paddingInner = 0;
    /**
     * The ratio of the range that is reserved for space before the first
     * and after the last band.
     */
    this._paddingOuter = 0;
  }
  static is(value) {
    return value instanceof _BandScale;
  }
  get bandwidth() {
    this.refresh();
    return this._bandwidth;
  }
  get step() {
    this.refresh();
    return this._step;
  }
  get inset() {
    this.refresh();
    return this._inset;
  }
  get rawBandwidth() {
    this.refresh();
    return this._rawBandwidth;
  }
  set padding(value) {
    value = clamp(0, value, 1);
    this._paddingInner = value;
    this._paddingOuter = value;
  }
  get padding() {
    return this._paddingInner;
  }
  set paddingInner(value) {
    this.invalid = true;
    this._paddingInner = clamp(0, value, 1);
  }
  get paddingInner() {
    return this._paddingInner;
  }
  set paddingOuter(value) {
    this.invalid = true;
    this._paddingOuter = clamp(0, value, 1);
  }
  get paddingOuter() {
    return this._paddingOuter;
  }
  refresh() {
    if (!this.invalid)
      return;
    this.invalid = false;
    this.update();
    if (this.invalid) {
      logger_exports.warnOnce("Expected update to not invalidate scale");
    }
  }
  ticks(_params, domain = this.domain, visibleRange) {
    return filterVisibleTicks(domain, false, visibleRange);
  }
  convert(d, _clamp) {
    this.refresh();
    const i = this.getIndex(d);
    if (i == null || i < 0 || i >= this.domain.length) {
      return NaN;
    }
    return this.ordinalRange(i);
  }
  invertNearestIndex(position) {
    this.refresh();
    const { domain } = this;
    if (domain.length === 0)
      return -1;
    let low = 0;
    let high = domain.length - 1;
    let closestDistance = Infinity;
    let closestIndex = 0;
    while (low <= high) {
      const mid = (high + low) / 2 | 0;
      const p = this.ordinalRange(mid);
      const distance = Math.abs(p - position);
      if (distance === 0)
        return mid;
      if (distance < closestDistance) {
        closestDistance = distance;
        closestIndex = mid;
      }
      if (p < position) {
        low = mid + 1;
      } else {
        high = mid - 1;
      }
    }
    return closestIndex;
  }
  update() {
    const count = this.domain.length;
    if (count === 0)
      return;
    const [r0, r1] = this.range;
    let { _paddingInner: paddingInner } = this;
    const { _paddingOuter: paddingOuter, round: round3 } = this;
    const rangeDistance = r1 - r0;
    let rawStep;
    if (count === 1) {
      paddingInner = 0;
      rawStep = rangeDistance * (1 - paddingOuter * 2);
    } else {
      rawStep = rangeDistance / Math.max(1, count - paddingInner + paddingOuter * 2);
    }
    const step = round3 ? Math.floor(rawStep) : rawStep;
    let inset = r0 + (rangeDistance - step * (count - paddingInner)) / 2;
    let bandwidth = step * (1 - paddingInner);
    if (round3) {
      inset = Math.round(inset);
      bandwidth = Math.round(bandwidth);
    }
    this._step = step;
    this._inset = inset;
    this._bandwidth = bandwidth;
    this._rawBandwidth = rawStep * (1 - paddingInner);
  }
  ordinalRange(i) {
    const { _inset: inset, _step: step, range: range2 } = this;
    const min = Math.min(range2[0], range2[1]);
    const max = Math.max(range2[0], range2[1]);
    return clamp(min, inset + step * i, max);
  }
};
__decorateClass([
  Invalidating
], _BandScale.prototype, "range", 2);
__decorateClass([
  Invalidating
], _BandScale.prototype, "round", 2);
__decorateClass([
  Invalidating
], _BandScale.prototype, "interval", 2);
var BandScale = _BandScale;

// packages/ag-charts-community/src/scale/categoryScale.ts
var CategoryScale = class extends BandScale {
  constructor() {
    super(...arguments);
    this.type = "band";
    /**
     * Maps datum to its index in the {@link domain} array.
     * Used to check for duplicate data (not allowed).
     */
    this.index = void 0;
    /**
     * Contains unique data only.
     */
    this._domain = [];
  }
  set domain(values) {
    if (this._domain === values)
      return;
    this.invalid = true;
    this._domain = values;
    this.index = void 0;
  }
  get domain() {
    return this._domain;
  }
  normalizeDomains(...domains) {
    let normalizedDomain = void 0;
    const seenDomains = /* @__PURE__ */ new Set();
    let animatable = true;
    for (const domain of domains) {
      if (seenDomains.has(domain))
        continue;
      seenDomains.add(domain);
      if (normalizedDomain == null) {
        normalizedDomain = deduplicateCategories(domain);
      } else {
        animatable && (animatable = domainOrderedToNormalizedDomain(domain, normalizedDomain));
        normalizedDomain = deduplicateCategories([...normalizedDomain, ...domain]);
      }
    }
    normalizedDomain ?? (normalizedDomain = []);
    return { domain: normalizedDomain, animatable };
  }
  toDomain(_value) {
    return void 0;
  }
  invert(position, nearest = false) {
    this.refresh();
    const offset4 = nearest ? this.bandwidth / 2 : 0;
    const index = this.invertNearestIndex(Math.max(0, position - offset4));
    const matches = nearest || position === this.ordinalRange(index);
    return matches ? this.domain[index] : void 0;
  }
  getIndex(value) {
    let { index } = this;
    if (index == null) {
      const { domain } = this;
      index = /* @__PURE__ */ new Map();
      for (let i = 0; i < domain.length; i++) {
        index.set(dateToNumber(domain[i]), i);
      }
      this.index = index;
    }
    return index.get(dateToNumber(value));
  }
};
function deduplicateCategories(d) {
  let domain;
  const uniqueValues = /* @__PURE__ */ new Set();
  for (const value of d) {
    const key = dateToNumber(value);
    const lastSize = uniqueValues.size;
    uniqueValues.add(key);
    const isUniqueValue = uniqueValues.size !== lastSize;
    if (isUniqueValue) {
      domain?.push(value);
    } else {
      domain ?? (domain = d.slice(0, uniqueValues.size));
    }
  }
  return domain ?? d;
}
function domainOrderedToNormalizedDomain(domain, normalizedDomain) {
  let normalizedIndex = -1;
  for (const value of domain) {
    const normalizedNextIndex = normalizedDomain.indexOf(value);
    if (normalizedNextIndex === -1) {
      normalizedIndex = Infinity;
    } else if (normalizedNextIndex <= normalizedIndex) {
      return false;
    } else {
      normalizedIndex = normalizedNextIndex;
    }
  }
  return true;
}

// packages/ag-charts-community/src/util/properties.ts
var BaseProperties = class {
  set(properties) {
    const { className = this.constructor.name } = this.constructor;
    if (typeof properties !== "object") {
      logger_exports.warn(`unable to set ${className} - expecting a properties object`);
      return this;
    }
    const keys = new Set(Object.keys(properties));
    for (const propertyKey of listDecoratedProperties(this)) {
      if (keys.has(propertyKey)) {
        const value = properties[propertyKey];
        const self = this;
        if (isProperties(self[propertyKey])) {
          if (self[propertyKey] instanceof PropertiesArray) {
            const array2 = self[propertyKey].reset(value);
            if (array2 != null) {
              self[propertyKey] = array2;
            } else {
              logger_exports.warn(`unable to set [${propertyKey}] - expecting a properties array`);
            }
          } else {
            self[propertyKey].set(value);
          }
        } else {
          self[propertyKey] = value;
        }
        keys.delete(propertyKey);
      }
    }
    for (const unknownKey of keys) {
      logger_exports.warn(`unable to set [${unknownKey}] in ${className} - property is unknown`);
    }
    return this;
  }
  isValid(warningPrefix) {
    return listDecoratedProperties(this).every((propertyKey) => {
      const { optional } = extractDecoratedPropertyMetadata(this, propertyKey);
      const valid = optional === true || typeof this[propertyKey] !== "undefined";
      if (!valid) {
        logger_exports.warnOnce(`${warningPrefix ?? ""}[${propertyKey}] is required.`);
      }
      return valid;
    });
  }
  toJson() {
    return listDecoratedProperties(this).reduce((object2, propertyKey) => {
      const propertyValue = this[propertyKey];
      object2[propertyKey] = isProperties(propertyValue) ? propertyValue.toJson() : propertyValue;
      return object2;
    }, {});
  }
};
var PropertiesArray = class _PropertiesArray extends Array {
  constructor(itemFactory, ...properties) {
    super(properties.length);
    const isConstructor = (value2) => Boolean(value2?.prototype?.constructor?.name);
    const value = isConstructor(itemFactory) ? (params) => new itemFactory().set(params) : itemFactory;
    Object.defineProperty(this, "itemFactory", { value, enumerable: false, configurable: false });
    this.set(properties);
  }
  set(properties) {
    if (isArray(properties)) {
      this.length = properties.length;
      for (let i = 0; i < properties.length; i++) {
        this[i] = this.itemFactory(properties[i]);
      }
    }
    return this;
  }
  reset(properties) {
    if (Array.isArray(properties)) {
      return new _PropertiesArray(this.itemFactory, ...properties);
    }
  }
  toJson() {
    return this.map((value) => value?.toJson?.() ?? value);
  }
};
function isProperties(value) {
  return value instanceof BaseProperties || value instanceof PropertiesArray;
}

// packages/ag-charts-community/src/util/validation.ts
function Validate(predicate, options = {}) {
  const { optional = false, property: overrideProperty } = options;
  return addTransformToInstanceProperty(
    (target, property, value) => {
      const context = { ...options, target, property };
      if (optional && typeof value === "undefined" || predicate(value, context)) {
        if (isProperties(target[property]) && !isProperties(value)) {
          target[property].set(value);
          return target[property];
        }
        return value;
      }
      const cleanKey = overrideProperty ?? String(property).replace(/^_*/, "");
      const targetName = target.constructor.className ?? target.constructor.name.replace(/Properties$/, "");
      const valueString = stringifyValue(value, 50);
      logger_exports.warn(
        `Property [${cleanKey}] of [${targetName}] cannot be set to [${valueString}]${predicate.message ? `; expecting ${getPredicateMessage(predicate, context)}` : ""}, ignoring.`
      );
      return BREAK_TRANSFORM_CHAIN;
    },
    void 0,
    { optional }
  );
}
var AND = (...predicates) => {
  const messages = [];
  return predicateWithMessage(
    (value, ctx) => {
      messages.length = 0;
      return predicates.every((predicate) => {
        const isValid = predicate(value, ctx);
        if (!isValid) {
          messages.push(getPredicateMessage(predicate, ctx));
        }
        return isValid;
      });
    },
    () => messages.filter(Boolean).join(" AND ")
  );
};
var OR = (...predicates) => predicateWithMessage(
  (value, ctx) => predicates.some((predicate) => predicate(value, ctx)),
  (ctx) => predicates.map(getPredicateMessageMapper(ctx)).filter(Boolean).join(" OR ")
);
var OBJECT = attachObjectRestrictions(
  predicateWithMessage(
    (value, ctx) => isProperties(value) || isObject(value) && isProperties(ctx.target[ctx.property]),
    "a properties object"
  )
);
var PLAIN_OBJECT = attachObjectRestrictions(predicateWithMessage((value) => isObject(value), "an object"));
var BOOLEAN = predicateWithMessage(isBoolean, "a boolean");
var FUNCTION = predicateWithMessage(isFunction, "a function");
var STRING = predicateWithMessage(isString, "a string");
var NUMBER = attachNumberRestrictions(predicateWithMessage(isFiniteNumber, "a number"));
var REAL_NUMBER = predicateWithMessage((value) => isNumber(value) && !isNaN(value), "a real number");
var NAN = predicateWithMessage((value) => isNumber(value) && isNaN(value), "NaN");
var POSITIVE_NUMBER = NUMBER.restrict({ min: 0 });
var RATIO = NUMBER.restrict({ min: 0, max: 1 });
var NUMBER_OR_NAN = OR(NUMBER, NAN);
var ARRAY = attachArrayRestrictions(predicateWithMessage(isArray, "an array"));
var ARRAY_OF = (predicate, message) => predicateWithMessage(
  (value, ctx) => isArray(value) && value.every((item) => predicate(item, ctx)),
  (ctx) => {
    const arrayMessage = getPredicateMessage(ARRAY, ctx) ?? "";
    if (typeof message === "function") {
      return `${arrayMessage} of ${message(ctx)}`;
    }
    return message ? `${arrayMessage} of ${message}` : arrayMessage;
  }
);
var isComparable2 = (value) => isFiniteNumber(value) || isValidDate(value);
var LESS_THAN = (otherField) => predicateWithMessage(
  (v, ctx) => !isComparable2(v) || !isComparable2(ctx.target[otherField]) || v < ctx.target[otherField],
  `to be less than ${otherField}`
);
var GREATER_THAN = (otherField) => predicateWithMessage(
  (v, ctx) => !isComparable2(v) || !isComparable2(ctx.target[otherField]) || v > ctx.target[otherField],
  `to be greater than ${otherField}`
);
var DATE = predicateWithMessage(isValidDate, "Date object");
var DATE_OR_DATETIME_MS = OR(DATE, POSITIVE_NUMBER);
var colorMessage = `A color string can be in one of the following formats to be valid: #rgb, #rrggbb, rgb(r, g, b), rgba(r, g, b, a) or a CSS color name such as 'white', 'orange', 'cyan', etc`;
var COLOR_STRING = predicateWithMessage(
  (v) => isString(v) && Color.validColorString(v),
  `color String. ${colorMessage}`
);
var COLOR_GRADIENT = attachObjectRestrictions(
  predicateWithMessage((value) => isObject(value) && value.type === "gradient", "a color gradient object")
);
var COLOR_STRING_ARRAY = predicateWithMessage(ARRAY_OF(COLOR_STRING), `color strings. ${colorMessage}`);
var BOOLEAN_ARRAY = ARRAY_OF(BOOLEAN, "boolean values");
var NUMBER_ARRAY = ARRAY_OF(NUMBER, "numbers");
var STRING_ARRAY = ARRAY_OF(STRING, "strings");
var DATE_ARRAY = predicateWithMessage(ARRAY_OF(DATE), "Date objects");
var OBJECT_ARRAY = predicateWithMessage(ARRAY_OF(OBJECT), "objects");
var LINE_CAP = UNION(["butt", "round", "square"], "a line cap");
var LINE_JOIN = UNION(["round", "bevel", "miter"], "a line join");
var LINE_STYLE = UNION(["solid", "dashed", "dotted"], "a line style");
var LINE_DASH = predicateWithMessage(
  ARRAY_OF(POSITIVE_NUMBER),
  "numbers specifying the length in pixels of alternating dashes and gaps, for example, [6, 3] means dashes with a length of 6 pixels with gaps between of 3 pixels."
);
var POSITION = UNION(["top", "right", "bottom", "left"], "a position");
var FONT_STYLE = UNION(["normal", "italic", "oblique"], "a font style");
var FONT_WEIGHT = OR(
  UNION(["normal", "bold", "bolder", "lighter"], "a font weight"),
  NUMBER.restrict({ min: 1, max: 1e3 })
);
var TEXT_WRAP = UNION(["never", "always", "hyphenate", "on-space"], "a text wrap strategy");
var TEXT_ALIGN = UNION(["left", "center", "right"], "a text align");
var VERTICAL_ALIGN = UNION(["top", "middle", "bottom"], "a vertical align");
var OVERFLOW_STRATEGY = UNION(["ellipsis", "hide"], "an overflow strategy");
var DIRECTION = UNION(["horizontal", "vertical"], "a direction");
var PLACEMENT = UNION(["inside", "outside"], "a placement");
var INTERACTION_RANGE = OR(UNION(["exact", "nearest"], "interaction range"), NUMBER);
var LABEL_PLACEMENT = UNION(["top", "bottom", "left", "right"]);
function UNION(options, message = "a") {
  return predicateWithMessage(
    (v, ctx) => {
      const option = options.find((o) => {
        const value = typeof o === "string" ? o : o.value;
        return v === value;
      });
      if (option == null)
        return false;
      if (typeof option !== "string" && (option.deprecated === true || option.deprecatedTo != null)) {
        const messages = [`Property [%s] with value '${option.value}' is deprecated.`];
        if (option.deprecatedTo) {
          messages.push(`Use ${option.deprecatedTo} instead.`);
        }
        logger_exports.warnOnce(messages.join(" "), ctx.property);
      }
      return true;
    },
    `${message} keyword such as ${joinUnionOptions(options)}`
  );
}
var MIN_SPACING = OR(AND(NUMBER.restrict({ min: 1 }), LESS_THAN("maxSpacing")), NAN);
var MAX_SPACING = OR(AND(NUMBER.restrict({ min: 1 }), GREATER_THAN("minSpacing")), NAN);
function predicateWithMessage(predicate, message) {
  predicate.message = message;
  return predicate;
}
function joinUnionOptions(options) {
  const values = options.filter((option) => typeof option === "string" || option.undocumented !== true).map((option) => `'${typeof option === "string" ? option : option.value}'`);
  if (values.length === 1) {
    return values[0];
  }
  const lastValue = values.pop();
  return `${values.join(", ")} or ${lastValue}`;
}
function getPredicateMessage(predicate, ctx) {
  return isFunction(predicate.message) ? predicate.message(ctx) : predicate.message;
}
function getPredicateMessageMapper(ctx) {
  return (predicate) => getPredicateMessage(predicate, ctx);
}
function attachArrayRestrictions(predicate) {
  return Object.assign(predicate, {
    restrict({ length, minLength } = {}) {
      let message = "an array";
      if (isNumber(minLength) && minLength > 0) {
        message = "a non-empty array";
      } else if (isNumber(length)) {
        message = `an array of length ${length}`;
      }
      return predicateWithMessage(
        (value) => isArray(value) && (isNumber(length) ? value.length === length : true) && (isNumber(minLength) ? value.length >= minLength : true),
        message
      );
    }
  });
}
function attachNumberRestrictions(predicate) {
  return Object.assign(predicate, {
    restrict({ min, max } = {}) {
      const message = ["a number"];
      const hasMin = isNumber(min);
      const hasMax = isNumber(max);
      if (hasMin && hasMax) {
        message.push(`between ${min} and ${max} inclusive`);
      } else if (hasMin) {
        message.push(`greater than or equal to ${min}`);
      } else if (hasMax) {
        message.push(`less than or equal to ${max}`);
      }
      return predicateWithMessage(
        (value) => isFiniteNumber(value) && (hasMin ? value >= min : true) && (hasMax ? value <= max : true),
        message.join(" ")
      );
    }
  });
}
function attachObjectRestrictions(predicate) {
  return Object.assign(predicate, {
    restrict(objectType) {
      return predicateWithMessage(
        (value) => value instanceof objectType,
        (ctx) => getPredicateMessage(predicate, ctx) ?? `an instance of ${objectType.name}`
      );
    }
  });
}

// packages/ag-charts-community/src/motion/resetMotion.ts
function resetMotion(selectionsOrNodes, propsFn) {
  const { nodes, selections } = deconstructSelectionsOrNodes(selectionsOrNodes);
  for (const selection of selections) {
    for (const node of selection.nodes()) {
      const from2 = propsFn(node, node.datum);
      node.setProperties(from2);
    }
    selection.cleanup();
  }
  for (const node of nodes) {
    const from2 = propsFn(node, node.datum);
    node.setProperties(from2);
  }
}

// packages/ag-charts-community/src/util/debug.ts
var LONG_TIME_PERIOD_THRESHOLD = 2e3;
var timeOfLastLog = Date.now();
var logTimeGap = () => {
  const timeSinceLastLog = Date.now() - timeOfLastLog;
  if (timeSinceLastLog > LONG_TIME_PERIOD_THRESHOLD) {
    const prettyDuration = (Math.floor(timeSinceLastLog / 100) / 10).toFixed(1);
    logger_exports.log(`**** ${prettyDuration}s since last log message ****`);
  }
  timeOfLastLog = Date.now();
};
var Debug = {
  create(...debugSelectors) {
    const resultFn = (...logContent) => {
      if (Debug.check(...debugSelectors)) {
        if (typeof logContent[0] === "function") {
          logContent = toArray(logContent[0]());
        }
        logTimeGap();
        logger_exports.log(...logContent);
      }
    };
    return Object.assign(resultFn, {
      check: () => Debug.check(...debugSelectors),
      group: (name, cb) => {
        if (Debug.check(...debugSelectors)) {
          return logger_exports.logGroup(name, cb);
        }
        return cb();
      }
    });
  },
  check(...debugSelectors) {
    if (debugSelectors.length === 0) {
      debugSelectors.push(true);
    }
    const chartDebug = toArray(getWindow("agChartsDebug"));
    return chartDebug.some((selector) => debugSelectors.includes(selector));
  }
};

// packages/ag-charts-community/src/scene/canvas/canvasUtil.ts
function clearContext({
  context,
  pixelRatio,
  width: width2,
  height: height2
}) {
  context.save();
  context.resetTransform();
  context.clearRect(0, 0, Math.ceil(width2 * pixelRatio), Math.ceil(height2 * pixelRatio));
  context.restore();
}
function debugContext(ctx) {
  if (Debug.check("canvas")) {
    const save = ctx.save.bind(ctx);
    const restore = ctx.restore.bind(ctx);
    let depth = 0;
    Object.assign(ctx, {
      save() {
        save();
        depth++;
      },
      restore() {
        if (depth === 0) {
          throw new Error("AG Charts - Unable to restore() past depth 0");
        }
        restore();
        depth--;
      },
      verifyDepthZero() {
        if (depth !== 0) {
          throw new Error(`AG Charts - Save/restore depth is non-zero: ${depth}`);
        }
      }
    });
  }
}

// packages/ag-charts-community/src/scene/canvas/hdpiOffscreenCanvas.ts
function canvasDimensions(width2, height2, pixelRatio) {
  return [Math.round(width2 * pixelRatio), Math.round(height2 * pixelRatio)];
}
var HdpiOffscreenCanvas = class {
  constructor(options) {
    const { width: width2, height: height2, pixelRatio, willReadFrequently = false } = options;
    this.width = width2;
    this.height = height2;
    this.pixelRatio = pixelRatio;
    const [canvasWidth, canvasHeight] = canvasDimensions(width2, height2, pixelRatio);
    this.canvas = new OffscreenCanvas(canvasWidth, canvasHeight);
    this.context = this.canvas.getContext("2d", { willReadFrequently });
    this.context.setTransform(pixelRatio, 0, 0, pixelRatio, 0, 0);
    debugContext(this.context);
  }
  drawImage(context, dx = 0, dy = 0) {
    return context.drawImage(this.canvas, dx, dy);
  }
  transferToImageBitmap() {
    return this.canvas.transferToImageBitmap();
  }
  resize(width2, height2, pixelRatio) {
    if (!(width2 > 0 && height2 > 0))
      return;
    const { canvas, context } = this;
    if (width2 !== this.width || height2 !== this.height || pixelRatio !== this.pixelRatio) {
      const [canvasWidth, canvasHeight] = canvasDimensions(width2, height2, pixelRatio);
      canvas.width = canvasWidth;
      canvas.height = canvasHeight;
    }
    context.setTransform(pixelRatio, 0, 0, pixelRatio, 0, 0);
    this.width = width2;
    this.height = height2;
    this.pixelRatio = pixelRatio;
  }
  clear() {
    clearContext(this);
  }
  destroy() {
    this.canvas.width = 0;
    this.canvas.height = 0;
    this.context.clearRect(0, 0, 0, 0);
    this.canvas = null;
    this.context = null;
    Object.freeze(this);
  }
};

// packages/ag-charts-community/src/scale/colorScale.ts
var convertColorStringToOklcha = (v) => {
  const color = Color.fromString(v);
  const [l, c, h] = Color.RGBtoOKLCH(color.r, color.g, color.b);
  return { l, c, h, a: color.a };
};
var delta = 1e-6;
var isAchromatic = (x) => x.c < delta || x.l < delta || x.l > 1 - delta;
var interpolateOklch = (x, y, d) => {
  d = clamp(0, d, 1);
  let h;
  if (isAchromatic(x)) {
    h = y.h;
  } else if (isAchromatic(y)) {
    h = x.h;
  } else {
    const xH = x.h;
    let yH = y.h;
    const deltaH = y.h - x.h;
    if (deltaH > 180) {
      yH -= 360;
    } else if (deltaH < -180) {
      yH += 360;
    }
    h = xH * (1 - d) + yH * d;
  }
  const c = x.c * (1 - d) + y.c * d;
  const l = x.l * (1 - d) + y.l * d;
  const a = x.a * (1 - d) + y.a * d;
  return Color.fromOKLCH(l, c, h, a);
};
var ColorScale = class extends AbstractScale {
  constructor() {
    super(...arguments);
    this.type = "color";
    this.invalid = true;
    this.domain = [0, 1];
    this.range = ["red", "blue"];
    this.parsedRange = this.range.map(convertColorStringToOklcha);
  }
  update() {
    const { domain, range: range2 } = this;
    if (domain.length < 2) {
      logger_exports.warnOnce("`colorDomain` should have at least 2 values.");
      if (domain.length === 0) {
        domain.push(0, 1);
      } else if (domain.length === 1) {
        domain.push(domain[0] + 1);
      }
    }
    for (let i = 1; i < domain.length; i++) {
      const a = domain[i - 1];
      const b = domain[i];
      if (a >= b) {
        logger_exports.warnOnce("`colorDomain` values should be supplied in ascending order.");
        domain.sort((a2, b2) => a2 - b2);
        break;
      }
    }
    if (range2.length < domain.length) {
      for (let i = range2.length; i < domain.length; i++) {
        range2.push(range2.length > 0 ? range2[0] : "black");
      }
    }
    this.parsedRange = this.range.map(convertColorStringToOklcha);
  }
  normalizeDomains(...domains) {
    return { domain: domains.flat(), animatable: true };
  }
  toDomain() {
    return;
  }
  convert(x) {
    this.refresh();
    const { domain, range: range2, parsedRange } = this;
    const d0 = domain[0];
    const d1 = domain.at(-1);
    const r0 = range2[0];
    const r1 = range2[range2.length - 1];
    if (x <= d0) {
      return r0;
    }
    if (x >= d1) {
      return r1;
    }
    let index;
    let q;
    if (domain.length === 2) {
      const t = (x - d0) / (d1 - d0);
      const step = 1 / (range2.length - 1);
      index = range2.length <= 2 ? 0 : Math.min(Math.floor(t * (range2.length - 1)), range2.length - 2);
      q = (t - index * step) / step;
    } else {
      for (index = 0; index < domain.length - 2; index++) {
        if (x < domain[index + 1]) {
          break;
        }
      }
      const a = domain[index];
      const b = domain[index + 1];
      q = (x - a) / (b - a);
    }
    const c0 = parsedRange[index];
    const c1 = parsedRange[index + 1];
    return interpolateOklch(c0, c1, q).toRgbaString();
  }
  invert() {
    return;
  }
  refresh() {
    if (!this.invalid)
      return;
    this.invalid = false;
    this.update();
    if (this.invalid) {
      logger_exports.warnOnce("Expected update to not invalidate scale");
    }
  }
};
__decorateClass([
  Invalidating
], ColorScale.prototype, "domain", 2);
__decorateClass([
  Invalidating
], ColorScale.prototype, "range", 2);

// packages/ag-charts-community/src/scene/gradient/gradient.ts
var Gradient = class {
  constructor(colorSpace, stops = [], bbox) {
    this.colorSpace = colorSpace;
    this.stops = stops;
    this.bbox = bbox;
    this._cache = void 0;
  }
  createGradient(ctx, shapeBbox) {
    const bbox = this.bbox ?? shapeBbox;
    if (this._cache != null && this._cache.ctx === ctx && this._cache.bbox.equals(bbox)) {
      return this._cache.gradient;
    }
    const { stops, colorSpace } = this;
    if (stops.length === 0)
      return;
    if (stops.length === 1)
      return stops[0].color;
    let gradient = this.createCanvasGradient(ctx, bbox);
    if (gradient == null)
      return;
    const isOkLch = colorSpace === "oklch";
    const step = 0.05;
    let c0 = stops[0];
    gradient.addColorStop(c0.offset, c0.color);
    for (let i = 1; i < stops.length; i += 1) {
      const c1 = stops[i];
      if (isOkLch) {
        const scale = new ColorScale();
        scale.domain = [c0.offset, c1.offset];
        scale.range = [c0.color, c1.color];
        for (let offset4 = c0.offset + step; offset4 < c1.offset; offset4 += step) {
          gradient.addColorStop(offset4, scale.convert(offset4));
        }
      }
      gradient.addColorStop(c1.offset, c1.color);
      c0 = c1;
    }
    if ("createPattern" in gradient) {
      gradient = gradient.createPattern();
    }
    this._cache = { ctx, bbox, gradient };
    return gradient;
  }
};

// packages/ag-charts-community/src/util/angle.ts
var twoPi = Math.PI * 2;
var halfPi = Math.PI / 2;
function normalizeAngle360(radians) {
  radians %= twoPi;
  radians += twoPi;
  radians %= twoPi;
  return radians;
}
function isBetweenAngles(targetAngle, startAngle, endAngle) {
  const t = normalizeAngle360(targetAngle);
  const a0 = normalizeAngle360(startAngle);
  const a1 = normalizeAngle360(endAngle);
  if (a0 < a1) {
    return a0 <= t && t <= a1;
  } else if (a0 > a1) {
    return a0 <= t || t <= a1;
  } else {
    return true;
  }
}
function toRadians(degrees) {
  return degrees / 180 * Math.PI;
}

// packages/ag-charts-community/src/scene/gradient/linearGradient.ts
var LinearGradient = class extends Gradient {
  constructor(colorSpace, stops, angle = 0, bbox) {
    super(colorSpace, stops, bbox);
    this.angle = angle;
  }
  createCanvasGradient(ctx, bbox) {
    const angleOffset = 90;
    const { angle } = this;
    const radians = normalizeAngle360(toRadians(angle + angleOffset));
    const cos = Math.cos(radians);
    const sin = Math.sin(radians);
    const w = bbox.width;
    const h = bbox.height;
    const cx = bbox.x + w * 0.5;
    const cy = bbox.y + h * 0.5;
    const diagonal = Math.sqrt(h * h + w * w) / 2;
    const diagonalAngle = Math.atan2(h, w);
    let quarteredAngle;
    if (radians < Math.PI / 2) {
      quarteredAngle = radians;
    } else if (radians < Math.PI) {
      quarteredAngle = Math.PI - radians;
    } else if (radians < 1.5 * Math.PI) {
      quarteredAngle = radians - Math.PI;
    } else {
      quarteredAngle = 2 * Math.PI - radians;
    }
    const l = diagonal * Math.abs(Math.cos(quarteredAngle - diagonalAngle));
    return ctx.createLinearGradient(cx + cos * l, cy + sin * l, cx - cos * l, cy - sin * l);
  }
};

// packages/ag-charts-community/src/scene/gradient/stops.ts
var StopProperties = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.color = "black";
  }
};
__decorateClass([
  Validate(NUMBER, { optional: true })
], StopProperties.prototype, "stop", 2);
__decorateClass([
  Validate(COLOR_STRING, { optional: true })
], StopProperties.prototype, "color", 2);
function stopsAreAscending(fills) {
  let currentStop;
  for (const { stop } of fills) {
    if (stop == null) {
      continue;
    } else if (currentStop != null && stop < currentStop) {
      return false;
    }
    currentStop = stop;
  }
  return true;
}
function discreteColorStops(colorStops) {
  return colorStops.flatMap((colorStop, i) => {
    const { offset: offset4 } = colorStop;
    const nextColor = colorStops.at(i + 1)?.color;
    return nextColor != null ? [colorStop, { offset: offset4, color: nextColor }] : [colorStop];
  });
}
function getDefaultColorStops(defaultColorStops, fillMode) {
  const stopOffset = fillMode === "discrete" ? 1 : 0;
  const colorStops = defaultColorStops.map(
    (color, index, { length }) => ({
      offset: (index + stopOffset) / (length - 1 + stopOffset),
      color
    })
  );
  return fillMode === "discrete" ? discreteColorStops(colorStops) : colorStops;
}
function getColorStops(fills, defaultColorStops, domain, fillMode = "continuous") {
  if (fills.length === 0) {
    return getDefaultColorStops(defaultColorStops, fillMode);
  } else if (!stopsAreAscending(fills)) {
    logger_exports.warnOnce(`[fills] must have the stops defined in ascending order`);
    return [];
  }
  const d0 = Math.min(...domain);
  const d1 = Math.max(...domain);
  const isDiscrete = fillMode === "discrete";
  const offsets = new Float64Array(fills.length);
  let previousDefinedStopIndex = 0;
  let nextDefinedStopIndex = -1;
  for (let i = 0; i < fills.length; i += 1) {
    const colorStop = fills[i];
    if (i >= nextDefinedStopIndex) {
      nextDefinedStopIndex = fills.length - 1;
      for (let j = i + 1; j < fills.length; j += 1) {
        if (fills[j].stop != null) {
          nextDefinedStopIndex = j;
          break;
        }
      }
    }
    let { stop } = colorStop;
    if (stop == null) {
      const stop0 = fills[previousDefinedStopIndex].stop;
      const stop1 = fills[nextDefinedStopIndex].stop;
      const value0 = stop0 ?? d0;
      const value1 = stop1 ?? d1;
      const stopOffset = isDiscrete && stop0 == null ? 1 : 0;
      stop = value0 + (value1 - value0) * (i - previousDefinedStopIndex + stopOffset) / (nextDefinedStopIndex - previousDefinedStopIndex + stopOffset);
    } else {
      previousDefinedStopIndex = i;
    }
    offsets[i] = Math.max(0, Math.min(1, (stop - d0) / (d1 - d0)));
  }
  let lastDefinedColor = fills.find((c) => c.color != null)?.color;
  let colorScale;
  const colorStops = fills.map(({ color }, i) => {
    const offset4 = offsets[i];
    if (color != null) {
      lastDefinedColor = color;
    } else if (lastDefinedColor != null) {
      color = lastDefinedColor;
    } else {
      if (colorScale == null) {
        colorScale = new ColorScale();
        colorScale.domain = [0, 1];
        colorScale.range = defaultColorStops;
      }
      color = colorScale.convert(offset4);
    }
    return { offset: offset4, color };
  });
  return fillMode === "discrete" ? discreteColorStops(colorStops) : colorStops;
}

// packages/ag-charts-community/src/scene/util/fill.ts
function isGradientFill(fill) {
  return fill !== null && isObject(fill) && fill.type == "gradient";
}

// packages/ag-charts-community/src/scene/util/pixel.ts
function align(pixelRatio, start2, length) {
  const alignedStart = Math.round(start2 * pixelRatio) / pixelRatio;
  if (length == null) {
    return alignedStart;
  } else if (length === 0) {
    return 0;
  } else if (length < 1) {
    return Math.ceil(length * pixelRatio) / pixelRatio;
  }
  return Math.round((length + start2) * pixelRatio) / pixelRatio - alignedStart;
}
function alignBefore(pixelRatio, start2) {
  return Math.floor(start2 * pixelRatio) / pixelRatio;
}

// packages/ag-charts-community/src/scene/shape/shape.ts
var LINEAR_GRADIENT_REGEXP = /^linear-gradient\((-?[\d.]+)deg,(.*?)\)$/i;
var _Shape = class _Shape extends Node {
  constructor() {
    super(...arguments);
    this.fillOpacity = 1;
    this.strokeOpacity = 1;
    this.fill = _Shape.defaultStyles.fill;
    this.stroke = _Shape.defaultStyles.stroke;
    this.strokeWidth = _Shape.defaultStyles.strokeWidth;
    this.lineDash = _Shape.defaultStyles.lineDash;
    this.lineDashOffset = _Shape.defaultStyles.lineDashOffset;
    this.lineCap = _Shape.defaultStyles.lineCap;
    this.lineJoin = _Shape.defaultStyles.lineJoin;
    this.miterLimit = void 0;
    this.opacity = _Shape.defaultStyles.opacity;
    this.fillShadow = _Shape.defaultStyles.fillShadow;
    this.gradientFillOptions = {
      domain: [0, 1],
      defaultColorRange: _Shape.defaultStyles.defaultColorRange
    };
  }
  /**
   * Restores the default styles introduced by this subclass.
   */
  restoreOwnStyles() {
    const { defaultStyles } = this.constructor;
    Object.assign(this, defaultStyles);
  }
  getGradient(pattern) {
    let linearGradientMatch;
    if (pattern instanceof Gradient) {
      return pattern;
    } else if (typeof pattern === "string" && pattern?.startsWith("linear-gradient") && (linearGradientMatch = LINEAR_GRADIENT_REGEXP.exec(pattern))) {
      const angle = parseFloat(linearGradientMatch[1]);
      const colors = [];
      const colorsPart = linearGradientMatch[2];
      const colorRegex = /(#[0-9a-f]+)|(rgba?\(.+?\))|([a-z]+)/gi;
      let c;
      while (c = colorRegex.exec(colorsPart)) {
        colors.push(c[0]);
      }
      return new LinearGradient(
        "rgb",
        colors.map((color, index) => ({ color, offset: index / (colors.length - 1) })),
        angle
      );
    } else if (isGradientFill(pattern)) {
      return this.createLinearGradient(pattern);
    }
    return void 0;
  }
  createLinearGradient(fill) {
    const { colorStops = [], direction } = fill;
    const isHorizontal = direction === "horizontal";
    const { domain, defaultColorRange = [] } = this.gradientFillOptions;
    const stops = getColorStops(colorStops, defaultColorRange, domain);
    return new LinearGradient("oklch", stops, isHorizontal ? 0 : 90);
  }
  onFillChange() {
    this.fillGradient = this.getGradient(this.fill);
  }
  onStrokeChange() {
    this.strokeGradient = this.getGradient(this.stroke);
  }
  /**
   * Returns a device-pixel aligned coordinate (or length if length is supplied).
   *
   * NOTE: Not suitable for strokes, since the stroke needs to be offset to the middle
   * of a device pixel.
   */
  align(start2, length) {
    return align(this.layerManager?.canvas?.pixelRatio ?? 1, start2, length);
  }
  fillStroke(ctx, path) {
    this.renderFill(ctx, path);
    this.renderStroke(ctx, path);
  }
  renderFill(ctx, path) {
    if (this.fill) {
      const { globalAlpha } = ctx;
      this.applyFill(ctx);
      this.applyFillAlpha(ctx);
      this.applyShadow(ctx);
      this.executeFill(ctx, path);
      ctx.globalAlpha = globalAlpha;
    }
    ctx.shadowColor = "rgba(0, 0, 0, 0)";
  }
  executeFill(ctx, path) {
    if (path) {
      ctx.fill(path);
    } else {
      ctx.fill();
    }
  }
  applyFill(ctx) {
    const bbox = this.gradientFillOptions.bbox ?? this.getBBox();
    ctx.fillStyle = this.fillGradient?.createGradient(ctx, bbox) ?? (typeof this.fill === "string" ? this.fill : void 0) ?? "black";
  }
  applyStroke(ctx) {
    ctx.strokeStyle = this.strokeGradient?.createGradient(ctx, this.getBBox()) ?? (typeof this.stroke === "string" ? this.stroke : void 0) ?? "black";
  }
  applyFillAlpha(ctx) {
    ctx.globalAlpha *= this.opacity * this.fillOpacity;
  }
  applyShadow(ctx) {
    const pixelRatio = this.layerManager?.canvas.pixelRatio ?? 1;
    const fillShadow = this.fillShadow;
    if (fillShadow?.enabled) {
      ctx.shadowColor = fillShadow.color;
      ctx.shadowOffsetX = fillShadow.xOffset * pixelRatio;
      ctx.shadowOffsetY = fillShadow.yOffset * pixelRatio;
      ctx.shadowBlur = fillShadow.blur * pixelRatio;
    }
  }
  renderStroke(ctx, path) {
    if (this.stroke && this.strokeWidth) {
      const { globalAlpha } = ctx;
      this.applyStroke(ctx);
      ctx.globalAlpha *= this.opacity * this.strokeOpacity;
      ctx.lineWidth = this.strokeWidth;
      if (this.lineDash) {
        ctx.setLineDash(this.lineDash);
      }
      if (this.lineDashOffset) {
        ctx.lineDashOffset = this.lineDashOffset;
      }
      if (this.lineCap) {
        ctx.lineCap = this.lineCap;
      }
      if (this.lineJoin) {
        ctx.lineJoin = this.lineJoin;
      }
      if (this.miterLimit != null) {
        ctx.miterLimit = this.miterLimit;
      }
      this.executeStroke(ctx, path);
      ctx.globalAlpha = globalAlpha;
    }
  }
  executeStroke(ctx, path) {
    if (path) {
      ctx.stroke(path);
    } else {
      ctx.stroke();
    }
  }
  containsPoint(x, y) {
    return this.isPointInPath(x, y);
  }
  applySvgFillAttributes(element2) {
    const { fill, fillOpacity } = this;
    element2.setAttribute("fill", typeof fill === "string" ? fill : "none");
    element2.setAttribute("fill-opacity", String(fillOpacity));
  }
  applySvgStrokeAttributes(element2) {
    const { stroke, strokeOpacity, strokeWidth, lineDash, lineDashOffset } = this;
    if (stroke != null) {
      element2.setAttribute("stroke", typeof stroke === "string" ? stroke : "none");
      element2.setAttribute("stroke-opacity", String(strokeOpacity));
      element2.setAttribute("stroke-width", String(strokeWidth));
    }
    if (lineDash?.some((d) => d !== 0) === true) {
      const svgLineDash = lineDash.length % 2 === 1 ? [...lineDash, ...lineDash] : lineDash;
      element2.setAttribute("stroke-dasharray", svgLineDash.join(" "));
      element2.setAttribute("stroke-dashoffset", String(lineDashOffset));
    }
  }
};
/**
 * Defaults for style properties. Note that properties that affect the position
 * and shape of the node are not considered style properties, for example:
 * `x`, `y`, `width`, `height`, `radius`, `rotation`, etc.
 * Can be used to reset to the original styling after some custom styling
 * has been applied (using the `restoreOwnStyles` method).
 * These static defaults are meant to be inherited by subclasses.
 */
_Shape.defaultStyles = {
  fill: "black",
  stroke: void 0,
  strokeWidth: 0,
  lineDash: void 0,
  lineDashOffset: 0,
  lineCap: void 0,
  lineJoin: void 0,
  opacity: 1,
  fillShadow: void 0,
  defaultColorRange: ["#5090dc", "#ef5452"]
};
__decorateClass([
  SceneChangeDetection()
], _Shape.prototype, "fillOpacity", 2);
__decorateClass([
  SceneChangeDetection()
], _Shape.prototype, "strokeOpacity", 2);
__decorateClass([
  SceneChangeDetection({ changeCb: (s) => s.onFillChange() })
], _Shape.prototype, "fill", 2);
__decorateClass([
  SceneChangeDetection({ changeCb: (s) => s.onStrokeChange() })
], _Shape.prototype, "stroke", 2);
__decorateClass([
  SceneChangeDetection()
], _Shape.prototype, "strokeWidth", 2);
__decorateClass([
  SceneChangeDetection()
], _Shape.prototype, "lineDash", 2);
__decorateClass([
  SceneChangeDetection()
], _Shape.prototype, "lineDashOffset", 2);
__decorateClass([
  SceneChangeDetection()
], _Shape.prototype, "lineCap", 2);
__decorateClass([
  SceneChangeDetection()
], _Shape.prototype, "lineJoin", 2);
__decorateClass([
  SceneChangeDetection()
], _Shape.prototype, "miterLimit", 2);
__decorateClass([
  SceneChangeDetection({ convertor: (v) => clamp(0, v, 1) })
], _Shape.prototype, "opacity", 2);
__decorateClass([
  SceneChangeDetection({ checkDirtyOnAssignment: true })
], _Shape.prototype, "fillShadow", 2);
__decorateClass([
  SceneChangeDetection({ changeCb: (s) => s.onFillChange() })
], _Shape.prototype, "gradientFillOptions", 2);
var Shape = _Shape;

// packages/ag-charts-community/src/scene/matrix.ts
var IDENTITY_MATRIX_ELEMENTS = [1, 0, 0, 1, 0, 0];
var Matrix = class _Matrix {
  get e() {
    return [...this.elements];
  }
  constructor(elements = IDENTITY_MATRIX_ELEMENTS) {
    this.elements = [...elements];
  }
  setElements(elements) {
    const e = this.elements;
    e[0] = elements[0];
    e[1] = elements[1];
    e[2] = elements[2];
    e[3] = elements[3];
    e[4] = elements[4];
    e[5] = elements[5];
    return this;
  }
  get identity() {
    const e = this.elements;
    return isNumberEqual(e[0], 1) && isNumberEqual(e[1], 0) && isNumberEqual(e[2], 0) && isNumberEqual(e[3], 1) && isNumberEqual(e[4], 0) && isNumberEqual(e[5], 0);
  }
  /**
   * Performs the AxB matrix multiplication and saves the result
   * to `C`, if given, or to `A` otherwise.
   */
  AxB(A, B, C2) {
    const a = A[0] * B[0] + A[2] * B[1], b = A[1] * B[0] + A[3] * B[1], c = A[0] * B[2] + A[2] * B[3], d = A[1] * B[2] + A[3] * B[3], e = A[0] * B[4] + A[2] * B[5] + A[4], f = A[1] * B[4] + A[3] * B[5] + A[5];
    C2 = C2 ?? A;
    C2[0] = a;
    C2[1] = b;
    C2[2] = c;
    C2[3] = d;
    C2[4] = e;
    C2[5] = f;
  }
  /**
   * The `other` matrix gets post-multiplied to the current matrix.
   * Returns the current matrix.
   * @param other
   */
  multiplySelf(other) {
    this.AxB(this.elements, other.elements);
    return this;
  }
  /**
   * The `other` matrix gets post-multiplied to the current matrix.
   * Returns a new matrix.
   * @param other
   */
  multiply(other) {
    const elements = new Array(6);
    if (other instanceof _Matrix) {
      this.AxB(this.elements, other.elements, elements);
    } else {
      this.AxB(this.elements, [other.a, other.b, other.c, other.d, other.e, other.f], elements);
    }
    return new _Matrix(elements);
  }
  preMultiplySelf(other) {
    this.AxB(other.elements, this.elements, this.elements);
    return this;
  }
  /**
   * Returns the inverse of this matrix as a new matrix.
   */
  inverse() {
    const el = this.elements;
    let a = el[0], b = el[1], c = el[2], d = el[3];
    const e = el[4], f = el[5];
    const rD = 1 / (a * d - b * c);
    a *= rD;
    b *= rD;
    c *= rD;
    d *= rD;
    return new _Matrix([d, -b, -c, a, c * f - d * e, b * e - a * f]);
  }
  invertSelf() {
    const el = this.elements;
    let a = el[0], b = el[1], c = el[2], d = el[3];
    const e = el[4], f = el[5];
    const rD = 1 / (a * d - b * c);
    a *= rD;
    b *= rD;
    c *= rD;
    d *= rD;
    el[0] = d;
    el[1] = -b;
    el[2] = -c;
    el[3] = a;
    el[4] = c * f - d * e;
    el[5] = b * e - a * f;
    return this;
  }
  transformPoint(x, y) {
    const e = this.elements;
    return {
      x: x * e[0] + y * e[2] + e[4],
      y: x * e[1] + y * e[3] + e[5]
    };
  }
  transformBBox(bbox, target) {
    const el = this.elements;
    const xx = el[0];
    const xy = el[1];
    const yx = el[2];
    const yy = el[3];
    const h_w = bbox.width * 0.5;
    const h_h = bbox.height * 0.5;
    const cx = bbox.x + h_w;
    const cy = bbox.y + h_h;
    const w = Math.abs(h_w * xx) + Math.abs(h_h * yx);
    const h = Math.abs(h_w * xy) + Math.abs(h_h * yy);
    target ?? (target = new BBox(0, 0, 0, 0));
    target.x = cx * xx + cy * yx + el[4] - w;
    target.y = cx * xy + cy * yy + el[5] - h;
    target.width = w + w;
    target.height = h + h;
    return target;
  }
  toContext(ctx) {
    if (this.identity) {
      return;
    }
    const e = this.elements;
    ctx.transform(e[0], e[1], e[2], e[3], e[4], e[5]);
  }
  static updateTransformMatrix(matrix, scalingX, scalingY, rotation, translationX, translationY, opts) {
    const sx = scalingX;
    const sy = scalingY;
    let scx;
    let scy;
    if (sx === 1 && sy === 1) {
      scx = 0;
      scy = 0;
    } else {
      scx = opts?.scalingCenterX ?? 0;
      scy = opts?.scalingCenterY ?? 0;
    }
    const r = rotation;
    const cos = Math.cos(r);
    const sin = Math.sin(r);
    let rcx;
    let rcy;
    if (r === 0) {
      rcx = 0;
      rcy = 0;
    } else {
      rcx = opts?.rotationCenterX ?? 0;
      rcy = opts?.rotationCenterY ?? 0;
    }
    const tx = translationX;
    const ty = translationY;
    const tx4 = scx * (1 - sx) - rcx;
    const ty4 = scy * (1 - sy) - rcy;
    matrix.setElements([
      cos * sx,
      sin * sx,
      -sin * sy,
      cos * sy,
      cos * tx4 - sin * ty4 + rcx + tx,
      sin * tx4 + cos * ty4 + rcy + ty
    ]);
    return matrix;
  }
};

// packages/ag-charts-community/src/scene/transformable.ts
function isMatrixTransform(node) {
  return isMatrixTransformType(node.constructor);
}
var MATRIX_TRANSFORM_TYPE = Symbol("isMatrixTransform");
function isMatrixTransformType(cstr) {
  return cstr[MATRIX_TRANSFORM_TYPE] === true;
}
function MatrixTransform(Parent) {
  var _a, _b;
  const ParentNode = Parent;
  if (isMatrixTransformType(Parent)) {
    return Parent;
  }
  const TRANSFORM_MATRIX = Symbol("matrix_combined_transform");
  class MatrixTransformInternal extends ParentNode {
    constructor() {
      super(...arguments);
      this[_b] = new Matrix();
      this._dirtyTransform = true;
    }
    markDirtyTransform() {
      this._dirtyTransform = true;
      super.markDirty();
    }
    updateMatrix(_matrix) {
    }
    computeTransformMatrix() {
      if (!this._dirtyTransform)
        return;
      this[TRANSFORM_MATRIX].setElements(IDENTITY_MATRIX_ELEMENTS);
      this.updateMatrix(this[TRANSFORM_MATRIX]);
      this._dirtyTransform = false;
    }
    toParent(bbox) {
      this.computeTransformMatrix();
      if (this[TRANSFORM_MATRIX].identity)
        return bbox.clone();
      return this[TRANSFORM_MATRIX].transformBBox(bbox);
    }
    toParentPoint(x, y) {
      this.computeTransformMatrix();
      if (this[TRANSFORM_MATRIX].identity)
        return { x, y };
      return this[TRANSFORM_MATRIX].transformPoint(x, y);
    }
    fromParent(bbox) {
      this.computeTransformMatrix();
      if (this[TRANSFORM_MATRIX].identity)
        return bbox.clone();
      return this[TRANSFORM_MATRIX].inverse().transformBBox(bbox);
    }
    fromParentPoint(x, y) {
      this.computeTransformMatrix();
      if (this[TRANSFORM_MATRIX].identity)
        return { x, y };
      return this[TRANSFORM_MATRIX].inverse().transformPoint(x, y);
    }
    computeBBox() {
      const bbox = super.computeBBox();
      if (!bbox)
        return bbox;
      return this.toParent(bbox);
    }
    computeBBoxWithoutTransforms() {
      return super.computeBBox();
    }
    pickNode(x, y, localCoords = false) {
      if (!localCoords) {
        ({ x, y } = this.fromParentPoint(x, y));
      }
      return super.pickNode(x, y);
    }
    render(renderCtx) {
      this.computeTransformMatrix();
      const { ctx } = renderCtx;
      const matrix = this[TRANSFORM_MATRIX];
      let performRestore = false;
      if (!matrix.identity) {
        ctx.save();
        performRestore = true;
        matrix.toContext(ctx);
      }
      super.render(renderCtx);
      if (performRestore) {
        ctx.restore();
      }
    }
    toSVG() {
      this.computeTransformMatrix();
      const svg = super.toSVG();
      const matrix = this[TRANSFORM_MATRIX];
      if (matrix.identity || svg == null)
        return svg;
      const g = createSvgElement("g");
      g.append(...svg.elements);
      const [a, b, c, d, e, f] = matrix.e;
      g.setAttribute("transform", `matrix(${a} ${b} ${c} ${d} ${e} ${f})`);
      return {
        elements: [g],
        defs: svg.defs
      };
    }
  }
  _a = MATRIX_TRANSFORM_TYPE, _b = TRANSFORM_MATRIX;
  MatrixTransformInternal[_a] = true;
  return MatrixTransformInternal;
}
function Rotatable(Parent) {
  var _a;
  const ParentNode = Parent;
  const ROTATABLE_MATRIX = Symbol("matrix_rotation");
  class RotatableInternal extends MatrixTransform(ParentNode) {
    constructor() {
      super(...arguments);
      this[_a] = new Matrix();
      this.rotationCenterX = null;
      this.rotationCenterY = null;
      this.rotation = 0;
    }
    updateMatrix(matrix) {
      super.updateMatrix(matrix);
      const { rotation, rotationCenterX, rotationCenterY } = this;
      if (rotation === 0)
        return;
      Matrix.updateTransformMatrix(this[ROTATABLE_MATRIX], 1, 1, rotation, 0, 0, {
        rotationCenterX,
        rotationCenterY
      });
      matrix.multiplySelf(this[ROTATABLE_MATRIX]);
    }
  }
  _a = ROTATABLE_MATRIX;
  __decorateClass([
    SceneChangeDetection({ type: "transform" })
  ], RotatableInternal.prototype, "rotationCenterX", 2);
  __decorateClass([
    SceneChangeDetection({ type: "transform" })
  ], RotatableInternal.prototype, "rotationCenterY", 2);
  __decorateClass([
    SceneChangeDetection({ type: "transform" })
  ], RotatableInternal.prototype, "rotation", 2);
  return RotatableInternal;
}
function Scalable(Parent) {
  var _a;
  const ParentNode = Parent;
  const SCALABLE_MATRIX = Symbol("matrix_scale");
  class ScalableInternal extends MatrixTransform(ParentNode) {
    constructor() {
      super(...arguments);
      this[_a] = new Matrix();
      this.scalingX = 1;
      this.scalingY = 1;
      this.scalingCenterX = null;
      this.scalingCenterY = null;
    }
    updateMatrix(matrix) {
      super.updateMatrix(matrix);
      const { scalingX, scalingY, scalingCenterX, scalingCenterY } = this;
      if (scalingX === 1 && scalingY === 1)
        return;
      Matrix.updateTransformMatrix(this[SCALABLE_MATRIX], scalingX, scalingY, 0, 0, 0, {
        scalingCenterX,
        scalingCenterY
      });
      matrix.multiplySelf(this[SCALABLE_MATRIX]);
    }
  }
  _a = SCALABLE_MATRIX;
  __decorateClass([
    SceneChangeDetection({ type: "transform" })
  ], ScalableInternal.prototype, "scalingX", 2);
  __decorateClass([
    SceneChangeDetection({ type: "transform" })
  ], ScalableInternal.prototype, "scalingY", 2);
  __decorateClass([
    SceneChangeDetection({ type: "transform" })
  ], ScalableInternal.prototype, "scalingCenterX", 2);
  __decorateClass([
    SceneChangeDetection({ type: "transform" })
  ], ScalableInternal.prototype, "scalingCenterY", 2);
  return ScalableInternal;
}
function Translatable(Parent) {
  var _a;
  const ParentNode = Parent;
  const TRANSLATABLE_MATRIX = Symbol("matrix_translation");
  class TranslatableInternal extends MatrixTransform(ParentNode) {
    constructor() {
      super(...arguments);
      this[_a] = new Matrix();
      this.translationX = 0;
      this.translationY = 0;
    }
    updateMatrix(matrix) {
      super.updateMatrix(matrix);
      const { translationX, translationY } = this;
      if (translationX === 0 && translationY === 0)
        return;
      Matrix.updateTransformMatrix(this[TRANSLATABLE_MATRIX], 1, 1, 0, translationX, translationY);
      matrix.multiplySelf(this[TRANSLATABLE_MATRIX]);
    }
  }
  _a = TRANSLATABLE_MATRIX;
  __decorateClass([
    SceneChangeDetection({ type: "transform" })
  ], TranslatableInternal.prototype, "translationX", 2);
  __decorateClass([
    SceneChangeDetection({ type: "transform" })
  ], TranslatableInternal.prototype, "translationY", 2);
  return TranslatableInternal;
}
var Transformable = class {
  /**
   * Converts a BBox from canvas coordinate space into the coordinate space of the given Node.
   */
  static fromCanvas(node, bbox) {
    const parents = [];
    for (const parent of node.traverseUp()) {
      if (isMatrixTransform(parent)) {
        parents.unshift(parent);
      }
    }
    for (const parent of parents) {
      bbox = parent.fromParent(bbox);
    }
    if (isMatrixTransform(node)) {
      bbox = node.fromParent(bbox);
    }
    return bbox;
  }
  /**
   * Converts a Nodes BBox (or an arbitrary BBox if supplied) from local Node coordinate space
   * into the Canvas coordinate space.
   */
  static toCanvas(node, bbox) {
    if (bbox == null) {
      bbox = node.getBBox();
    } else if (isMatrixTransform(node)) {
      bbox = node.toParent(bbox);
    }
    for (const parent of node.traverseUp()) {
      if (isMatrixTransform(parent)) {
        bbox = parent.toParent(bbox);
      }
    }
    return bbox;
  }
  /**
   * Converts a point from canvas coordinate space into the coordinate space of the given Node.
   */
  static fromCanvasPoint(node, x, y) {
    const parents = [];
    for (const parent of node.traverseUp()) {
      if (isMatrixTransform(parent)) {
        parents.unshift(parent);
      }
    }
    for (const parent of parents) {
      ({ x, y } = parent.fromParentPoint(x, y));
    }
    if (isMatrixTransform(node)) {
      ({ x, y } = node.fromParentPoint(x, y));
    }
    return { x, y };
  }
  /**
   * Converts a point from a Nodes local coordinate space into the Canvas coordinate space.
   */
  static toCanvasPoint(node, x, y) {
    if (isMatrixTransform(node)) {
      ({ x, y } = node.toParentPoint(x, y));
    }
    for (const parent of node.traverseUp()) {
      if (isMatrixTransform(parent)) {
        ({ x, y } = parent.toParentPoint(x, y));
      }
    }
    return { x, y };
  }
};

// packages/ag-charts-community/src/scene/zIndex.ts
var cmp = (a, b) => Math.sign(a - b);
function compareZIndex(a, b) {
  if (typeof a === "number" && typeof b === "number") {
    return cmp(a, b);
  }
  const aArray = typeof a === "number" ? [a] : a;
  const bArray = typeof b === "number" ? [b] : b;
  const length = Math.min(aArray.length, bArray.length);
  for (let i = 0; i < length; i += 1) {
    const diff = cmp(aArray[i], bArray[i]);
    if (diff !== 0)
      return diff;
  }
  return cmp(aArray.length, bArray.length);
}

// packages/ag-charts-community/src/scene/group.ts
var sharedOffscreenCanvas;
var _Group = class _Group extends Node {
  // optimizeForInfrequentRedraws: true
  constructor(opts) {
    super(opts);
    this.opacity = 1;
    this.renderToOffscreenCanvas = false;
    this.optimizeForInfrequentRedraws = false;
    // Used when renderToOffscreenCanvas: true
    this.layer = void 0;
    // optimizeForInfrequentRedraws: false
    this.image = void 0;
    this._lastWidth = NaN;
    this._lastHeight = NaN;
    this._lastDevicePixelRatio = NaN;
    this.isContainerNode = true;
    this.renderToOffscreenCanvas = opts?.renderToOffscreenCanvas === true;
  }
  static is(value) {
    return value instanceof _Group;
  }
  static computeChildrenBBox(nodes, skipInvisible = true) {
    return BBox.merge(Node.extractBBoxes(nodes, skipInvisible));
  }
  static compareChildren(a, b) {
    return compareZIndex(a.zIndex, b.zIndex) || a.serialNumber - b.serialNumber;
  }
  // We consider a group to be boundless, thus any point belongs to it.
  containsPoint(_x, _y) {
    return true;
  }
  computeBBox() {
    return _Group.computeChildrenBBox(this.children());
  }
  computeSafeClippingBBox(pixelRatio) {
    const bbox = this.computeBBox();
    if (!bbox.isFinite())
      return;
    let strokeWidth = 0;
    const strokeMiterAmount = 4;
    for (const child of this.descendants()) {
      if (child instanceof Shape) {
        strokeWidth = Math.max(strokeWidth, child.strokeWidth);
      }
    }
    const padding = Math.max(
      // Account for anti-aliasing artefacts
      1,
      // Account for strokes (incl. miters) - this may not be the best place to include this
      strokeWidth / 2 * strokeMiterAmount
    );
    const { x: originX, y: originY } = Transformable.toCanvasPoint(this, 0, 0);
    const x = alignBefore(pixelRatio, originX + bbox.x - padding) - originX;
    const y = alignBefore(pixelRatio, originY + bbox.y - padding) - originY;
    const width2 = Math.ceil(bbox.x + bbox.width - x + padding);
    const height2 = Math.ceil(bbox.y + bbox.height - y + padding);
    return new BBox(x, y, width2, height2);
  }
  prepareSharedCanvas(width2, height2, pixelRatio) {
    if (sharedOffscreenCanvas == null || sharedOffscreenCanvas.pixelRatio !== pixelRatio) {
      sharedOffscreenCanvas = new HdpiOffscreenCanvas({ width: width2, height: height2, pixelRatio });
    } else {
      sharedOffscreenCanvas.resize(width2, height2, pixelRatio);
    }
    return sharedOffscreenCanvas;
  }
  isDirty(renderCtx) {
    const { width: width2, height: height2, devicePixelRatio } = renderCtx;
    const { dirty, dirtyZIndex, layer } = this;
    const layerResized = layer != null && (this._lastWidth !== width2 || this._lastHeight !== height2);
    const pixelRatioChanged = this._lastDevicePixelRatio !== devicePixelRatio;
    this._lastWidth = width2;
    this._lastHeight = height2;
    this._lastDevicePixelRatio = devicePixelRatio;
    if (dirty || dirtyZIndex || layerResized || pixelRatioChanged)
      return true;
    for (const child of this.children()) {
      if (child.dirty)
        return true;
    }
    return false;
  }
  preRender(renderCtx) {
    const counts = super.preRender(renderCtx, 0);
    counts.groups += 1;
    counts.nonGroups -= 1;
    if (this.renderToOffscreenCanvas && !this.optimizeForInfrequentRedraws && counts.nonGroups > 0 && this.getVisibility()) {
      this.layer ?? (this.layer = this._layerManager?.addLayer({ name: this.name }));
    } else if (this.layer != null) {
      this._layerManager?.removeLayer(this.layer);
      this.layer = void 0;
    }
    return counts;
  }
  render(renderCtx) {
    const { layer, renderToOffscreenCanvas } = this;
    const childRenderCtx = { ...renderCtx };
    if (!renderToOffscreenCanvas) {
      this.renderInContext(childRenderCtx);
      super.render(childRenderCtx);
      return;
    }
    const { ctx, stats, devicePixelRatio: pixelRatio } = renderCtx;
    let { image } = this;
    if (this.isDirty(renderCtx)) {
      image?.bitmap.close();
      image = void 0;
      const bbox = layer ? void 0 : this.computeSafeClippingBBox(pixelRatio);
      const renderOffscreen = (offscreenCanvas, ...transform) => {
        const offscreenCtx = offscreenCanvas.context;
        childRenderCtx.ctx = offscreenCtx;
        offscreenCanvas.clear();
        offscreenCtx.save();
        offscreenCtx.setTransform(...transform);
        offscreenCtx.globalAlpha = 1;
        this.renderInContext(childRenderCtx);
        offscreenCtx.restore();
        offscreenCtx.verifyDepthZero?.();
      };
      if (layer) {
        renderOffscreen(layer, ctx.getTransform());
      } else if (bbox) {
        const { x, y, width: width2, height: height2 } = bbox;
        const canvas = this.prepareSharedCanvas(width2, height2, pixelRatio);
        renderOffscreen(canvas, pixelRatio, 0, 0, pixelRatio, -x * pixelRatio, -y * pixelRatio);
        image = { bitmap: canvas.transferToImageBitmap(), x, y, width: width2, height: height2 };
      } else if (this.dirtyZIndex) {
        this.sortChildren(_Group.compareChildren);
      }
      this.image = image;
      if (stats)
        stats.layersRendered++;
    } else {
      this.skipRender(childRenderCtx);
      if (stats)
        stats.layersSkipped++;
    }
    const { globalAlpha } = ctx;
    ctx.globalAlpha = globalAlpha * this.opacity;
    if (layer) {
      ctx.save();
      ctx.resetTransform();
      layer.drawImage(ctx);
      ctx.restore();
    } else if (image) {
      const { bitmap, x, y, width: width2, height: height2 } = image;
      ctx.drawImage(bitmap, 0, 0, width2 * pixelRatio, height2 * pixelRatio, x, y, width2, height2);
    }
    ctx.globalAlpha = globalAlpha;
    super.render(childRenderCtx);
  }
  skipRender(childRenderCtx) {
    const { stats } = childRenderCtx;
    for (const child of this.children()) {
      child.markClean();
      if (stats) {
        stats.nodesSkipped += this.childNodeCounts.groups + this.childNodeCounts.nonGroups;
        stats.opsSkipped += this.childNodeCounts.complexity;
      }
    }
  }
  applyClip(ctx, clipRect) {
    const { x, y, width: width2, height: height2 } = clipRect;
    ctx.beginPath();
    ctx.rect(x, y, width2, height2);
    ctx.clip();
  }
  renderInContext(childRenderCtx) {
    const { ctx, stats } = childRenderCtx;
    if (this.dirtyZIndex) {
      this.sortChildren(_Group.compareChildren);
    }
    ctx.save();
    ctx.globalAlpha *= this.opacity;
    if (this.clipRect != null) {
      this.applyClip(ctx, this.clipRect);
      childRenderCtx.clipBBox = Transformable.toCanvas(this, this.clipRect);
    }
    for (const child of this.children()) {
      if (!child.visible) {
        child.markClean();
        if (stats) {
          stats.nodesSkipped += child.childNodeCounts.nonGroups + child.childNodeCounts.groups;
          stats.opsSkipped += child.childNodeCounts.complexity;
        }
        continue;
      }
      ctx.save();
      child.render(childRenderCtx);
      ctx.restore();
    }
    ctx.restore();
  }
  /**
   * Transforms bbox given in the canvas coordinate space to bbox in this group's coordinate space and
   * sets this group's clipRect to the transformed bbox.
   * @param bbox clipRect bbox in the canvas coordinate space.
   */
  setClipRect(bbox) {
    this.clipRect = bbox ? Transformable.fromCanvas(this, bbox) : void 0;
  }
  /**
   * Set the clip rect within the canvas coordinate space.
   * @param bbox clipRect bbox in the canvas coordinate space.
   */
  setClipRectCanvasSpace(bbox) {
    this.clipRect = bbox;
  }
  _setLayerManager(layersManager) {
    if (this.layer) {
      this._layerManager?.removeLayer(this.layer);
      this.layer = void 0;
    }
    super._setLayerManager(layersManager);
  }
  getVisibility() {
    for (const node of this.traverseUp(true)) {
      if (!node.visible) {
        return false;
      }
    }
    return true;
  }
  toSVG() {
    if (!this.visible)
      return;
    const defs = [];
    const elements = [];
    for (const child of this.children()) {
      const svg = child.toSVG();
      if (svg != null) {
        elements.push(...svg.elements);
        if (svg.defs != null) {
          defs.push(...svg.defs);
        }
      }
    }
    return { elements, defs };
  }
};
_Group.className = "Group";
__decorateClass([
  SceneChangeDetection({ convertor: (v) => clamp(0, v, 1) })
], _Group.prototype, "opacity", 2);
var Group = _Group;
var ScalableGroup = class extends Scalable(Group) {
};
var RotatableGroup = class extends Rotatable(Group) {
};
var TranslatableGroup = class extends Translatable(Group) {
};
var TransformableGroup = class extends Rotatable(Translatable(Group)) {
};

// packages/ag-charts-community/src/util/canvas.util.ts
function createCanvasContext(width2 = 0, height2 = 0) {
  return new OffscreenCanvas(width2, height2).getContext("2d");
}

// packages/ag-charts-community/src/util/lruCache.ts
var LRUCache = class {
  constructor(maxCacheSize = 5) {
    this.maxCacheSize = maxCacheSize;
    this.store = /* @__PURE__ */ new Map();
  }
  get(key) {
    if (!this.store.has(key))
      return void 0;
    const hit = this.store.get(key);
    this.store.delete(key);
    this.store.set(key, hit);
    return hit;
  }
  has(key) {
    return this.store.has(key);
  }
  set(key, value) {
    this.store.set(key, value);
    if (this.store.size > this.maxCacheSize) {
      const iterator = this.store.keys();
      let evictCount = this.store.size - this.maxCacheSize;
      while (evictCount > 0) {
        const evictKeyIterator = iterator.next();
        if (!evictKeyIterator.done) {
          this.store.delete(evictKeyIterator.value);
        }
        evictCount--;
      }
    }
    return value;
  }
  clear() {
    this.store.clear();
  }
};

// packages/ag-charts-community/src/util/textMeasurer.ts
var CachedTextMeasurerPool = class {
  // Measures the dimensions of the provided text, handling multiline if needed.
  static measureText(text, options) {
    const textMeasurer = this.getMeasurer(options);
    return textMeasurer.measureText(text);
  }
  static measureLines(text, options) {
    const textMeasurer = this.getMeasurer(options);
    return textMeasurer.measureLines(text);
  }
  // Gets a TextMeasurer instance, configuring text alignment and baseline if provided.
  static getMeasurer(options) {
    const font = typeof options.font === "string" ? options.font : TextUtils.toFontString(options.font);
    const key = `${font}-${options.textAlign ?? "start"}-${options.textBaseline ?? "alphabetic"}`;
    return this.instanceMap.get(key) ?? this.createFontMeasurer(font, options, key);
  }
  // Creates or retrieves a TextMeasurer instance for a specific font.
  static createFontMeasurer(font, options, key) {
    const ctx = createCanvasContext();
    ctx.font = font;
    ctx.textAlign = options.textAlign ?? "start";
    ctx.textBaseline = options.textBaseline ?? "alphabetic";
    const measurer = new CachedTextMeasurer(ctx, options);
    this.instanceMap.set(key, measurer);
    return measurer;
  }
};
CachedTextMeasurerPool.instanceMap = new LRUCache(10);
var CachedTextMeasurer = class {
  constructor(ctx, options) {
    this.ctx = ctx;
    // cached text measurements
    this.measureMap = new LRUCache(100);
    if (options.textAlign) {
      ctx.textAlign = options.textAlign;
    }
    if (options.textBaseline) {
      ctx.textBaseline = options.textBaseline;
    }
    ctx.font = typeof options.font === "string" ? options.font : TextUtils.toFontString(options.font);
    this.textMeasurer = new SimpleTextMeasurer(
      (t) => this.cachedCtxMeasureText(t),
      options.textBaseline ?? "alphabetic"
    );
  }
  textWidth(text, estimate) {
    return this.textMeasurer.textWidth(text, estimate);
  }
  measureText(text) {
    return this.textMeasurer.measureText(text);
  }
  measureLines(text) {
    return this.textMeasurer.measureLines(text);
  }
  cachedCtxMeasureText(text) {
    if (!this.measureMap.has(text)) {
      const rawResult = this.ctx.measureText(text);
      this.measureMap.set(text, {
        actualBoundingBoxAscent: rawResult.actualBoundingBoxAscent,
        emHeightAscent: rawResult.emHeightAscent,
        emHeightDescent: rawResult.emHeightDescent,
        actualBoundingBoxDescent: rawResult.actualBoundingBoxDescent,
        actualBoundingBoxLeft: rawResult.actualBoundingBoxLeft,
        actualBoundingBoxRight: rawResult.actualBoundingBoxRight,
        alphabeticBaseline: rawResult.alphabeticBaseline,
        fontBoundingBoxAscent: rawResult.fontBoundingBoxAscent,
        fontBoundingBoxDescent: rawResult.fontBoundingBoxDescent,
        hangingBaseline: rawResult.hangingBaseline,
        ideographicBaseline: rawResult.ideographicBaseline,
        width: rawResult.width
      });
    }
    return this.measureMap.get(text);
  }
};
var TextUtils = class {
  static toFontString({ fontSize = 10, fontStyle, fontWeight, fontFamily, lineHeight }) {
    let fontString = "";
    if (fontStyle) {
      fontString += `${fontStyle} `;
    }
    if (fontWeight) {
      fontString += `${fontWeight} `;
    }
    fontString += `${fontSize}px`;
    if (lineHeight) {
      fontString += `/${lineHeight}px`;
    }
    fontString += ` ${fontFamily}`;
    return fontString.trim();
  }
  static getLineHeight(fontSize) {
    return Math.ceil(fontSize * this.defaultLineHeight);
  }
  // Determines vertical offset modifier based on text baseline.
  static getVerticalModifier(textBaseline) {
    switch (textBaseline) {
      case "hanging":
      case "top":
        return 0;
      case "middle":
        return 0.5;
      case "alphabetic":
      case "bottom":
      case "ideographic":
      default:
        return 1;
    }
  }
};
TextUtils.EllipsisChar = "\u2026";
// Representation for text clipping.
TextUtils.defaultLineHeight = 1.15;
// Normally between 1.1 and 1.2
TextUtils.lineSplitter = /\r?\n/g;
var SimpleTextMeasurer = class {
  constructor(measureTextFn, textBaseline = "alphabetic") {
    this.measureTextFn = measureTextFn;
    this.textBaseline = textBaseline;
    // local chars width cache per TextMeasurer
    this.charMap = /* @__PURE__ */ new Map();
  }
  // Measures metrics for a single line of text.
  getMetrics(text) {
    const m = this.measureTextFn(text);
    m.fontBoundingBoxAscent ?? (m.fontBoundingBoxAscent = m.emHeightAscent);
    m.fontBoundingBoxDescent ?? (m.fontBoundingBoxDescent = m.emHeightDescent);
    return {
      width: m.width,
      height: m.actualBoundingBoxAscent + m.actualBoundingBoxDescent,
      lineHeight: m.fontBoundingBoxAscent + m.fontBoundingBoxDescent,
      offsetTop: m.actualBoundingBoxAscent,
      offsetLeft: m.actualBoundingBoxLeft
    };
  }
  // Calculates aggregated metrics for multiline text.
  getMultilineMetrics(lines) {
    let width2 = 0;
    let height2 = 0;
    let offsetTop = 0;
    let offsetLeft = 0;
    let baselineDistance = 0;
    const verticalModifier = TextUtils.getVerticalModifier(this.textBaseline);
    const lineMetrics = [];
    let index = 0;
    const length = lines.length;
    for (const line of lines) {
      const m = this.measureTextFn(line);
      m.fontBoundingBoxAscent ?? (m.fontBoundingBoxAscent = m.emHeightAscent);
      m.fontBoundingBoxDescent ?? (m.fontBoundingBoxDescent = m.emHeightDescent);
      if (width2 < m.width) {
        width2 = m.width;
      }
      if (offsetLeft < m.actualBoundingBoxLeft) {
        offsetLeft = m.actualBoundingBoxLeft;
      }
      if (index === 0) {
        height2 += m.actualBoundingBoxAscent;
        offsetTop += m.actualBoundingBoxAscent;
      } else {
        baselineDistance += m.fontBoundingBoxAscent;
      }
      if (index === length - 1) {
        height2 += m.actualBoundingBoxDescent;
      } else {
        baselineDistance += m.fontBoundingBoxDescent;
      }
      lineMetrics.push({
        text: line,
        width: m.width,
        height: m.actualBoundingBoxAscent + m.actualBoundingBoxDescent,
        lineHeight: m.fontBoundingBoxAscent + m.fontBoundingBoxDescent,
        offsetTop: m.actualBoundingBoxAscent,
        offsetLeft: m.actualBoundingBoxLeft
      });
      index++;
    }
    height2 += baselineDistance;
    offsetTop += baselineDistance * verticalModifier;
    return { width: width2, height: height2, offsetTop, offsetLeft, lineMetrics };
  }
  textWidth(text, estimate) {
    if (estimate) {
      let estimatedWidth = 0;
      for (let i = 0; i < text.length; i++) {
        estimatedWidth += this.textWidth(text.charAt(i));
      }
      return estimatedWidth;
    }
    if (text.length > 1) {
      return this.measureTextFn(text).width;
    }
    return this.charMap.get(text) ?? this.charWidth(text);
  }
  measureText(text) {
    return this.getMetrics(text);
  }
  // Measures the dimensions of the provided text, handling multiline if needed.
  measureLines(text) {
    const lines = typeof text === "string" ? text.split(TextUtils.lineSplitter) : text;
    return this.getMultilineMetrics(lines);
  }
  charWidth(char) {
    const { width: width2 } = this.measureTextFn(char);
    this.charMap.set(char, width2);
    return width2;
  }
};

// packages/ag-charts-community/src/scene/shape/text.ts
var _Text = class _Text extends Shape {
  constructor() {
    super(...arguments);
    this.x = 0;
    this.y = 0;
    this.lines = [];
    this.text = void 0;
    this.fontSize = 10;
    this.fontFamily = "sans-serif";
    this.textAlign = _Text.defaultStyles.textAlign;
    this.textBaseline = _Text.defaultStyles.textBaseline;
  }
  onTextChange() {
    this.lines = this.text?.split("\n").map((s) => s.trim()) ?? [];
  }
  static computeBBox(lines, x, y, opts) {
    const { offsetTop, offsetLeft, width: width2, height: height2 } = CachedTextMeasurerPool.measureLines(lines, opts);
    return new BBox(x - offsetLeft, y - offsetTop, width2, height2);
  }
  computeBBox() {
    const { x, y, lines, textBaseline, textAlign } = this;
    return _Text.computeBBox(lines, x, y, { font: this, textBaseline, textAlign });
  }
  isPointInPath(x, y) {
    const bbox = this.getBBox();
    return bbox ? bbox.containsPoint(x, y) : false;
  }
  render(renderCtx) {
    const { ctx, stats } = renderCtx;
    if (!this.lines.length || !this.layerManager) {
      if (stats)
        stats.nodesSkipped += 1;
      return super.render(renderCtx);
    }
    const { fill, stroke, strokeWidth } = this;
    const { pixelRatio } = this.layerManager.canvas;
    ctx.font = TextUtils.toFontString(this);
    ctx.textAlign = this.textAlign;
    ctx.textBaseline = this.textBaseline;
    if (fill) {
      this.applyFill(ctx);
      ctx.globalAlpha *= this.opacity * this.fillOpacity;
      const { fillShadow } = this;
      if (fillShadow?.enabled) {
        ctx.shadowColor = fillShadow.color;
        ctx.shadowOffsetX = fillShadow.xOffset * pixelRatio;
        ctx.shadowOffsetY = fillShadow.yOffset * pixelRatio;
        ctx.shadowBlur = fillShadow.blur * pixelRatio;
      }
      this.renderLines((line, x, y) => ctx.fillText(line, x, y));
    }
    if (stroke && strokeWidth) {
      this.applyStroke(ctx);
      ctx.lineWidth = strokeWidth;
      ctx.globalAlpha *= this.opacity * this.strokeOpacity;
      const { lineDash, lineDashOffset, lineCap, lineJoin } = this;
      if (lineDash) {
        ctx.setLineDash(lineDash);
      }
      if (lineDashOffset) {
        ctx.lineDashOffset = lineDashOffset;
      }
      if (lineCap) {
        ctx.lineCap = lineCap;
      }
      if (lineJoin) {
        ctx.lineJoin = lineJoin;
      }
      this.renderLines((line, x, y) => ctx.strokeText(line, x, y));
    }
    super.render(renderCtx);
  }
  renderLines(renderCallback) {
    const { lines, x, y } = this;
    const lineHeight = this.lineHeight ?? TextUtils.getLineHeight(this.fontSize);
    let offsetY = (lineHeight - lineHeight * lines.length) * TextUtils.getVerticalModifier(this.textBaseline);
    for (const line of lines) {
      renderCallback(line, x, y + offsetY);
      offsetY += lineHeight;
    }
  }
  setFont(props) {
    this.fontFamily = props.fontFamily;
    this.fontSize = props.fontSize;
    this.fontStyle = props.fontStyle;
    this.fontWeight = props.fontWeight;
  }
  setAlign(props) {
    this.textAlign = props.textAlign;
    this.textBaseline = props.textBaseline;
  }
  toSVG() {
    if (!this.visible || !this.text)
      return;
    const element2 = createSvgElement("text");
    this.applySvgFillAttributes(element2);
    element2.setAttribute("font-family", this.fontFamily?.split(",")[0] ?? "");
    element2.setAttribute("font-size", String(this.fontSize));
    element2.setAttribute("font-style", this.fontStyle ?? "");
    element2.setAttribute("font-weight", String(this.fontWeight ?? ""));
    element2.setAttribute(
      "text-anchor",
      {
        center: "middle",
        left: "start",
        right: "end",
        start: "start",
        end: "end"
      }[this.textAlign ?? "start"]
    );
    element2.setAttribute(
      "alignment-baseline",
      {
        alphabetic: "alphabetic",
        top: "top",
        bottom: "bottom",
        hanging: "hanging",
        middle: "middle",
        ideographic: "ideographic"
      }[this.textBaseline ?? "alphabetic"]
    );
    element2.setAttribute("x", String(this.x));
    element2.setAttribute("y", String(this.y));
    element2.textContent = this.text ?? "";
    return { elements: [element2] };
  }
};
_Text.className = "Text";
_Text.defaultStyles = {
  ...Shape.defaultStyles,
  textAlign: "start",
  fontStyle: void 0,
  fontWeight: void 0,
  fontSize: 10,
  fontFamily: "sans-serif",
  textBaseline: "alphabetic"
};
__decorateClass([
  SceneChangeDetection()
], _Text.prototype, "x", 2);
__decorateClass([
  SceneChangeDetection()
], _Text.prototype, "y", 2);
__decorateClass([
  SceneChangeDetection({ changeCb: (o) => o.onTextChange() })
], _Text.prototype, "text", 2);
__decorateClass([
  SceneChangeDetection()
], _Text.prototype, "fontStyle", 2);
__decorateClass([
  SceneChangeDetection()
], _Text.prototype, "fontWeight", 2);
__decorateClass([
  SceneChangeDetection()
], _Text.prototype, "fontSize", 2);
__decorateClass([
  SceneChangeDetection()
], _Text.prototype, "fontFamily", 2);
__decorateClass([
  SceneChangeDetection()
], _Text.prototype, "textAlign", 2);
__decorateClass([
  SceneChangeDetection()
], _Text.prototype, "textBaseline", 2);
__decorateClass([
  SceneChangeDetection()
], _Text.prototype, "lineHeight", 2);
var Text = _Text;
var RotatableText = class extends Rotatable(Text) {
};
var TransformableText = class extends Rotatable(Translatable(Text)) {
};

// packages/ag-charts-community/src/util/stateMachine.ts
var debugColor = "color: green";
var debugQuietColor = "color: grey";
function applyProperties(parentState, childState) {
  const childProperties = listDecoratedProperties(childState);
  if (childProperties.length === 0)
    return;
  const properties = extractDecoratedProperties(parentState);
  for (const property of childProperties) {
    if (property in properties) {
      childState[property] = properties[property];
    }
  }
}
var AbstractStateMachine = class {
  transitionRoot(event, data) {
    if (this.parent) {
      this.parent.transitionRoot(event, data);
    } else {
      this.transition(event, data);
    }
  }
};
var _StateMachine = class _StateMachine extends AbstractStateMachine {
  constructor(defaultState, states, enterEach) {
    super();
    this.defaultState = defaultState;
    this.states = states;
    this.enterEach = enterEach;
    this.debug = Debug.create(true, "animation");
    this.state = defaultState;
    this.debug(`%c${this.constructor.name} | init -> ${defaultState}`, debugColor);
  }
  // TODO: handle events which do not require data without requiring `undefined` to be passed as as parameter, while
  // also still requiring data to be passed to those events which do require it.
  transition(event, data) {
    const shouldTransitionSelf = this.transitionChild(event, data);
    if (!shouldTransitionSelf || this.state === _StateMachine.child || this.state === _StateMachine.parent) {
      return;
    }
    const currentState = this.state;
    const currentStateConfig = this.states[this.state];
    let destination = currentStateConfig[event];
    const debugPrefix = `%c${this.constructor.name} | ${this.state} -> ${event} ->`;
    if (Array.isArray(destination)) {
      destination = destination.find((transition) => {
        if (!transition.guard)
          return true;
        const valid = transition.guard(data);
        if (!valid) {
          this.debug(`${debugPrefix} (guarded)`, transition.target, debugQuietColor);
        }
        return valid;
      });
    } else if (typeof destination === "object" && !(destination instanceof _StateMachine) && destination.guard && !destination.guard(data)) {
      this.debug(`${debugPrefix} (guarded)`, destination.target, debugQuietColor);
      return;
    }
    if (!destination) {
      this.debug(`${debugPrefix} ${this.state}`, debugQuietColor);
      return;
    }
    const destinationState = this.getDestinationState(destination);
    const exitFn = destinationState === this.state ? void 0 : currentStateConfig.onExit;
    this.debug(`${debugPrefix} ${destinationState}`, debugColor);
    this.state = destinationState;
    if (typeof destination === "function") {
      destination(data);
    } else if (typeof destination === "object" && !(destination instanceof _StateMachine)) {
      destination.action?.(data);
    }
    exitFn?.();
    this.enterEach?.(currentState, destinationState);
    if (destinationState !== currentState && destinationState !== _StateMachine.child && destinationState !== _StateMachine.parent) {
      this.states[destinationState].onEnter?.(currentState, data);
    }
  }
  transitionAsync(event, data) {
    setTimeout(() => {
      this.transition(event, data);
    }, 0);
  }
  is(value) {
    if (this.state === _StateMachine.child && this.childState) {
      return this.childState.is(value);
    }
    return this.state === value;
  }
  resetHierarchy() {
    this.debug(
      `%c${this.constructor.name} | ${this.state} -> [resetHierarchy] -> ${this.defaultState}`,
      "color: green"
    );
    this.state = this.defaultState;
  }
  transitionChild(event, data) {
    if (this.state !== _StateMachine.child || !this.childState)
      return true;
    applyProperties(this, this.childState);
    this.childState.transition(event, data);
    if (!this.childState.is(_StateMachine.parent))
      return true;
    this.debug(`%c${this.constructor.name} | ${this.state} -> ${event} -> ${this.defaultState}`, debugColor);
    this.state = this.defaultState;
    this.states[this.state].onEnter?.();
    this.childState.resetHierarchy();
    return false;
  }
  getDestinationState(destination) {
    let state = this.state;
    if (typeof destination === "string") {
      state = destination;
    } else if (destination instanceof _StateMachine) {
      this.childState = destination;
      this.childState.parent = this;
      state = _StateMachine.child;
    } else if (typeof destination === "object") {
      if (destination.target instanceof _StateMachine) {
        this.childState = destination.target;
        this.childState.parent = this;
        state = _StateMachine.child;
      } else if (destination.target != null) {
        state = destination.target;
      }
    }
    return state;
  }
};
_StateMachine.child = "__child";
_StateMachine.parent = "__parent";
var StateMachine = _StateMachine;

// packages/ag-charts-community/src/util/proxy.ts
function ProxyProperty(proxyPath, configMetadata) {
  const pathArray = isArray(proxyPath) ? proxyPath : proxyPath.split(".");
  if (pathArray.length === 1) {
    const [property] = pathArray;
    return addTransformToInstanceProperty(
      (target, _, value) => target[property] = value,
      (target) => target[property],
      configMetadata
    );
  }
  return addTransformToInstanceProperty(
    (target, _, value) => setPath(target, pathArray, value),
    (target) => getPath(target, pathArray),
    configMetadata
  );
}
function ProxyPropertyOnWrite(childName, childProperty) {
  return addTransformToInstanceProperty((target, key, value) => target[childName][childProperty ?? key] = value);
}
function ActionOnSet(opts) {
  const { newValue: newValueFn, oldValue: oldValueFn, changeValue: changeValueFn } = opts;
  return addTransformToInstanceProperty((target, _, newValue, oldValue) => {
    if (newValue !== oldValue) {
      if (oldValue !== void 0) {
        oldValueFn?.call(target, oldValue);
      }
      if (newValue !== void 0) {
        newValueFn?.call(target, newValue);
      }
      changeValueFn?.call(target, newValue, oldValue);
    }
    return newValue;
  });
}
function ObserveChanges(observerFn) {
  return addObserverToInstanceProperty(observerFn);
}

// packages/ag-charts-community/src/util/textWrapper.ts
var TextWrapper = class {
  static wrapText(text, options) {
    return this.wrapLines(text, options).join("\n");
  }
  static wrapLines(text, options) {
    const clippedResult = this.textWrap(text, options);
    if (options.overflow === "hide" && clippedResult.some((l) => l.endsWith(TextUtils.EllipsisChar))) {
      return [];
    }
    return clippedResult;
  }
  static appendEllipsis(text) {
    return text.replace(/[.,]{1,5}$/, "") + TextUtils.EllipsisChar;
  }
  static truncateLine(text, measurer, maxWidth, ellipsisForce) {
    const ellipsisWidth = measurer.textWidth(TextUtils.EllipsisChar);
    let estimatedWidth = 0;
    let i = 0;
    for (; i < text.length; i++) {
      const charWidth = measurer.textWidth(text.charAt(i));
      if (estimatedWidth + charWidth > maxWidth)
        break;
      estimatedWidth += charWidth;
    }
    if (text.length === i && (!ellipsisForce || estimatedWidth + ellipsisWidth <= maxWidth)) {
      return ellipsisForce ? text + TextUtils.EllipsisChar : text;
    }
    text = text.slice(0, i).trimEnd();
    while (text.length && measurer.textWidth(text) + ellipsisWidth > maxWidth) {
      text = text.slice(0, -1).trimEnd();
    }
    return text + TextUtils.EllipsisChar;
  }
  static textWrap(text, options) {
    const lines = text.split(TextUtils.lineSplitter);
    const measurer = CachedTextMeasurerPool.getMeasurer(options);
    if (options.textWrap === "never") {
      return lines.map((line) => this.truncateLine(line.trimEnd(), measurer, options.maxWidth));
    }
    const result = [];
    const wrapHyphenate = options.textWrap === "hyphenate";
    const wrapOnSpace = options.textWrap == null || options.textWrap === "on-space";
    for (const untrimmedLine of lines) {
      let line = untrimmedLine.trimEnd();
      if (line === "") {
        result.push(line);
        continue;
      }
      let i = 0;
      let estimatedWidth = 0;
      let lastSpaceIndex = 0;
      while (i < line.length) {
        const char = line.charAt(i);
        estimatedWidth += measurer.textWidth(char);
        if (char === " ") {
          lastSpaceIndex = i;
        }
        if (estimatedWidth > options.maxWidth) {
          if (i === 0)
            break;
          const actualWidth = measurer.textWidth(line.slice(0, i + 1));
          if (actualWidth <= options.maxWidth) {
            estimatedWidth = actualWidth;
            i++;
            continue;
          }
          if (lastSpaceIndex) {
            const nextWord = this.getWordAt(line, lastSpaceIndex + 1);
            const textWidth = measurer.textWidth(nextWord);
            if (textWidth <= options.maxWidth) {
              result.push(line.slice(0, lastSpaceIndex).trimEnd());
              line = line.slice(lastSpaceIndex).trimStart();
              i = 0;
              estimatedWidth = 0;
              lastSpaceIndex = 0;
              continue;
            } else if (wrapOnSpace && textWidth > options.maxWidth) {
              result.push(
                line.slice(0, lastSpaceIndex).trimEnd(),
                this.truncateLine(
                  line.slice(lastSpaceIndex).trimStart(),
                  measurer,
                  options.maxWidth,
                  true
                )
              );
            }
          } else if (wrapOnSpace) {
            result.push(this.truncateLine(line, measurer, options.maxWidth, true));
          }
          if (wrapOnSpace) {
            line = "";
            break;
          }
          const postfix = wrapHyphenate ? "-" : "";
          let newLine = line.slice(0, i).trim();
          while (newLine.length && measurer.textWidth(newLine + postfix) > options.maxWidth) {
            newLine = newLine.slice(0, -1).trimEnd();
          }
          result.push(newLine + postfix);
          if (!newLine.length) {
            line = "";
            break;
          }
          line = line.slice(newLine.length).trimStart();
          i = -1;
          estimatedWidth = 0;
          lastSpaceIndex = 0;
        }
        i++;
      }
      if (line) {
        result.push(line);
      }
    }
    this.avoidOrphans(result, measurer, options);
    return this.clipLines(result, measurer, options);
  }
  static getWordAt(text, position) {
    const nextSpaceIndex = text.indexOf(" ", position);
    return nextSpaceIndex === -1 ? text.slice(position) : text.slice(position, nextSpaceIndex);
  }
  static clipLines(lines, measurer, options) {
    if (!options.maxHeight) {
      return lines;
    }
    const { height: height2, lineMetrics } = measurer.measureLines(lines);
    if (height2 <= options.maxHeight) {
      return lines;
    }
    for (let i = 0, cumulativeHeight = 0; i < lineMetrics.length; i++) {
      const { lineHeight } = lineMetrics[i];
      cumulativeHeight += lineHeight;
      if (cumulativeHeight > options.maxHeight) {
        if (options.overflow === "hide") {
          return [];
        }
        const clippedResults = lines.slice(0, i || 1);
        const lastLine = clippedResults.pop();
        return clippedResults.concat(this.truncateLine(lastLine, measurer, options.maxWidth, true));
      }
    }
    return lines;
  }
  static avoidOrphans(lines, measurer, options) {
    if (options.avoidOrphans === false || lines.length < 2)
      return;
    const { length } = lines;
    const lastLine = lines[length - 1];
    const beforeLast = lines[length - 2];
    if (beforeLast.length < lastLine.length)
      return;
    const lastSpaceIndex = beforeLast.lastIndexOf(" ");
    if (lastSpaceIndex === -1 || lastSpaceIndex === beforeLast.indexOf(" ") || lastLine.includes(" "))
      return;
    const lastWord = beforeLast.slice(lastSpaceIndex + 1);
    if (measurer.textWidth(lastLine + lastWord) <= options.maxWidth) {
      lines[length - 2] = beforeLast.slice(0, lastSpaceIndex);
      lines[length - 1] = lastWord + " " + lastLine;
    }
  }
};

// packages/ag-charts-community/src/chart/caption.ts
var Caption = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.id = createId(this);
    this.node = new RotatableText({ zIndex: 1 }).setProperties({
      textAlign: "center",
      pointerEvents: 1 /* None */
    });
    this.enabled = false;
    this.textAlign = "center";
    this.fontSize = 10;
    this.fontFamily = "sans-serif";
    this.wrapping = "always";
    this.padding = 0;
    this.layoutStyle = "block";
    this.truncated = false;
  }
  registerInteraction(moduleCtx, where) {
    return moduleCtx.layoutManager.addListener("layout:complete", () => this.updateA11yText(moduleCtx, where));
  }
  computeTextWrap(containerWidth, containerHeight) {
    const { text, padding, wrapping } = this;
    const maxWidth = Math.min(this.maxWidth ?? Infinity, containerWidth) - padding * 2;
    const maxHeight = this.maxHeight ?? containerHeight - padding * 2;
    if (!isFinite(maxWidth) && !isFinite(maxHeight)) {
      this.node.text = text;
      return;
    }
    const wrappedText = TextWrapper.wrapText(text ?? "", { maxWidth, maxHeight, font: this, textWrap: wrapping });
    this.node.text = wrappedText;
    this.truncated = wrappedText.includes(TextUtils.EllipsisChar);
  }
  updateA11yText(moduleCtx, where) {
    const { proxyInteractionService } = moduleCtx;
    if (this.enabled && this.text) {
      const bbox = Transformable.toCanvas(this.node);
      if (bbox) {
        const { id: domManagerId } = this;
        this.proxyText ?? (this.proxyText = proxyInteractionService.createProxyElement({ type: "text", domManagerId, where }));
        this.proxyText.textContent = this.text;
        this.proxyText.setBounds(bbox);
        this.proxyText.addListener("mousemove", (ev) => this.handleMouseMove(moduleCtx, ev));
        this.proxyText.addListener("mouseleave", (ev) => this.handleMouseLeave(moduleCtx, ev));
      }
    } else {
      this.proxyText?.destroy();
      this.proxyText = void 0;
    }
  }
  handleMouseMove(moduleCtx, event) {
    if (event != null && this.enabled && this.node.visible && this.truncated) {
      const { x, y } = Transformable.toCanvas(this.node);
      const canvasX = event.sourceEvent.offsetX + x;
      const canvasY = event.sourceEvent.offsetY + y;
      const lastPointerEvent = { type: "pointermove", canvasX, canvasY };
      moduleCtx.tooltipManager.updateTooltip(
        this.id,
        { canvasX, canvasY, lastPointerEvent, showArrow: false },
        { type: "structured", title: this.text }
      );
    }
  }
  handleMouseLeave(moduleCtx, _event) {
    moduleCtx.tooltipManager.removeTooltip(this.id);
  }
};
Caption.SMALL_PADDING = 10;
Caption.LARGE_PADDING = 20;
__decorateClass([
  Validate(BOOLEAN),
  ProxyPropertyOnWrite("node", "visible")
], Caption.prototype, "enabled", 2);
__decorateClass([
  Validate(STRING, { optional: true }),
  ProxyPropertyOnWrite("node")
], Caption.prototype, "text", 2);
__decorateClass([
  Validate(TEXT_ALIGN, { optional: true }),
  ProxyPropertyOnWrite("node")
], Caption.prototype, "textAlign", 2);
__decorateClass([
  Validate(FONT_STYLE, { optional: true }),
  ProxyPropertyOnWrite("node")
], Caption.prototype, "fontStyle", 2);
__decorateClass([
  Validate(FONT_WEIGHT, { optional: true }),
  ProxyPropertyOnWrite("node")
], Caption.prototype, "fontWeight", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER),
  ProxyPropertyOnWrite("node")
], Caption.prototype, "fontSize", 2);
__decorateClass([
  Validate(STRING),
  ProxyPropertyOnWrite("node")
], Caption.prototype, "fontFamily", 2);
__decorateClass([
  Validate(COLOR_STRING, { optional: true }),
  ProxyPropertyOnWrite("node", "fill")
], Caption.prototype, "color", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER, { optional: true })
], Caption.prototype, "spacing", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER, { optional: true })
], Caption.prototype, "maxWidth", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER, { optional: true })
], Caption.prototype, "maxHeight", 2);
__decorateClass([
  Validate(TEXT_WRAP)
], Caption.prototype, "wrapping", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], Caption.prototype, "padding", 2);
__decorateClass([
  Validate(STRING)
], Caption.prototype, "layoutStyle", 2);

// packages/ag-charts-community/src/chart/chartAxisDirection.ts
var ChartAxisDirection = /* @__PURE__ */ ((ChartAxisDirection2) => {
  ChartAxisDirection2["X"] = "x";
  ChartAxisDirection2["Y"] = "y";
  return ChartAxisDirection2;
})(ChartAxisDirection || {});

// packages/ag-charts-community/src/module/moduleMap.ts
var ModuleMap = class {
  constructor() {
    this.moduleMap = /* @__PURE__ */ new Map();
  }
  *modules() {
    for (const m of this.moduleMap.values()) {
      yield m.moduleInstance;
    }
  }
  addModule(module, moduleFactory) {
    if (this.moduleMap.has(module.optionsKey)) {
      throw new Error(`AG Charts - module already initialised: ${module.optionsKey}`);
    }
    this.moduleMap.set(module.optionsKey, { module, moduleInstance: moduleFactory(module) });
  }
  removeModule(module) {
    const moduleKey = isString(module) ? module : module.optionsKey;
    this.moduleMap.get(moduleKey)?.moduleInstance.destroy();
    this.moduleMap.delete(moduleKey);
  }
  getModule(module) {
    return this.moduleMap.get(isString(module) ? module : module.optionsKey)?.moduleInstance;
  }
  isEnabled(module) {
    return this.moduleMap.has(isString(module) ? module : module.optionsKey);
  }
  mapModules(callback2) {
    return Array.from(this.moduleMap.values(), (m, i) => callback2(m.moduleInstance, i));
  }
  destroy() {
    for (const moduleKey of this.moduleMap.keys()) {
      this.moduleMap.get(moduleKey)?.moduleInstance.destroy();
    }
    this.moduleMap.clear();
  }
};

// packages/ag-charts-community/src/util/date.ts
function compareDates(a, b) {
  return a.valueOf() - b.valueOf();
}
function deduplicateSortedArray(values) {
  let v0 = NaN;
  const out = [];
  for (const v of values) {
    const v1 = v.valueOf();
    if (v0 !== v1)
      out.push(v);
    v0 = v1;
  }
  return out;
}
function sortAndUniqueDates(values) {
  const sortedValues = values.slice().sort(compareDates);
  return datesSortOrder(sortedValues) == null ? deduplicateSortedArray(sortedValues) : sortedValues;
}
function datesSortOrder(d) {
  if (d.length === 0)
    return 1;
  const sign = Number(d[d.length - 1]) > Number(d[0]) ? 1 : -1;
  let v0 = -Infinity * sign;
  for (const v of d) {
    const v1 = v.valueOf();
    if (Math.sign(v1 - v0) !== sign)
      return;
    v0 = v1;
  }
  return sign;
}

// packages/ag-charts-community/src/util/numberFormat.ts
function parseFormat(format) {
  let prefix;
  let suffix;
  const surrounded = surroundedRegEx.exec(format);
  if (surrounded) {
    [, prefix, format, suffix] = surrounded;
  }
  const match = formatRegEx.exec(format);
  if (!match) {
    throw new Error(`The number formatter is invalid: ${format}`);
  }
  const [, fill, align2, sign, symbol, zero, width2, comma, precision, trim, type] = match;
  return {
    fill,
    align: align2,
    sign,
    symbol,
    zero,
    width: parseInt(width2),
    comma,
    precision: parseInt(precision),
    trim: Boolean(trim),
    type,
    prefix,
    suffix
  };
}
function numberFormat(format) {
  const options = typeof format === "string" ? parseFormat(format) : format;
  const { fill, align: align2, sign = "-", symbol, zero, width: width2, comma, type, prefix = "", suffix = "", precision } = options;
  let { trim } = options;
  const precisionIsNaN = precision == null || isNaN(precision);
  let formatBody;
  if (!type) {
    formatBody = decimalTypes["g"];
    trim = true;
  } else if (type in decimalTypes && type in integerTypes) {
    formatBody = precisionIsNaN ? integerTypes[type] : decimalTypes[type];
  } else if (type in decimalTypes) {
    formatBody = decimalTypes[type];
  } else if (type in integerTypes) {
    formatBody = integerTypes[type];
  } else {
    throw new Error(`The number formatter type is invalid: ${type}`);
  }
  let formatterPrecision;
  if (precision == null || precisionIsNaN) {
    formatterPrecision = type ? 6 : 12;
  } else {
    formatterPrecision = precision;
  }
  return (n) => {
    let result = formatBody(n, formatterPrecision);
    if (trim) {
      result = removeTrailingZeros(result);
    }
    if (comma) {
      result = insertSeparator(result, comma);
    }
    result = addSign(n, result, sign);
    if (symbol && symbol !== "#") {
      result = `${symbol}${result}`;
    }
    if (symbol === "#" && type === "x") {
      result = `0x${result}`;
    }
    if (type === "s") {
      result = `${result}${getSIPrefix(n)}`;
    }
    if (type === "%" || type === "p") {
      result = `${result}%`;
    }
    if (width2 != null && !isNaN(width2)) {
      result = addPadding(result, width2, fill ?? zero, align2);
    }
    result = `${prefix}${result}${suffix}`;
    return result;
  };
}
var formatRegEx = /^(?:(.)?([<>=^]))?([+\-( ])?([$€£¥₣₹#])?(0)?(\d+)?(,)?(?:\.(\d+))?(~)?([%a-z])?$/i;
var surroundedRegEx = /^((?:[^#]|#[^{])*)#{([^}]+)}(.*)$/;
var integerTypes = {
  b: (n) => absFloor(n).toString(2),
  c: (n) => String.fromCharCode(n),
  d: (n) => Math.round(Math.abs(n)).toFixed(0),
  o: (n) => absFloor(n).toString(8),
  x: (n) => absFloor(n).toString(16),
  X: (n) => integerTypes.x(n).toUpperCase(),
  n: (n) => integerTypes.d(n),
  "%": (n) => `${absFloor(n * 100).toFixed(0)}`
};
var decimalTypes = {
  e: (n, f) => Math.abs(n).toExponential(f),
  E: (n, f) => decimalTypes.e(n, f).toUpperCase(),
  f: (n, f) => Math.abs(n).toFixed(f),
  F: (n, f) => decimalTypes.f(n, f).toUpperCase(),
  g: (n, f) => {
    if (n === 0) {
      return "0";
    }
    const a = Math.abs(n);
    const p = Math.floor(Math.log10(a));
    if (p >= -4 && p < f) {
      return a.toFixed(f - 1 - p);
    }
    return a.toExponential(f - 1);
  },
  G: (n, f) => decimalTypes.g(n, f).toUpperCase(),
  n: (n, f) => decimalTypes.g(n, f),
  p: (n, f) => decimalTypes.r(n * 100, f),
  r: (n, f) => {
    if (n === 0) {
      return "0";
    }
    const a = Math.abs(n);
    const p = Math.floor(Math.log10(a));
    const q = p - (f - 1);
    if (q <= 0) {
      return a.toFixed(-q);
    }
    const x = 10 ** q;
    return (Math.round(a / x) * x).toFixed();
  },
  s: (n, f) => {
    const p = getSIPrefixPower(n);
    return decimalTypes.r(n / 10 ** p, f);
  },
  "%": (n, f) => decimalTypes.f(n * 100, f)
};
var minSIPrefix = -24;
var maxSIPrefix = 24;
var siPrefixes = {
  [minSIPrefix]: "y",
  [-21]: "z",
  [-18]: "a",
  [-15]: "f",
  [-12]: "p",
  [-9]: "n",
  [-6]: "\xB5",
  [-3]: "m",
  [0]: "",
  [3]: "k",
  [6]: "M",
  [9]: "G",
  [12]: "T",
  [15]: "P",
  [18]: "E",
  [21]: "Z",
  [maxSIPrefix]: "Y"
};
var minusSign = "\u2212";
function absFloor(n) {
  return Math.floor(Math.abs(n));
}
function removeTrailingZeros(numString) {
  return numString.replace(/\.0+$/, "").replace(/(\.[1-9])0+$/, "$1");
}
function insertSeparator(numString, separator) {
  let dotIndex = numString.indexOf(".");
  if (dotIndex < 0) {
    dotIndex = numString.length;
  }
  const integerChars = numString.substring(0, dotIndex).split("");
  const fractionalPart = numString.substring(dotIndex);
  for (let i = integerChars.length - 3; i > 0; i -= 3) {
    integerChars.splice(i, 0, separator);
  }
  return `${integerChars.join("")}${fractionalPart}`;
}
function getSIPrefix(n) {
  return siPrefixes[getSIPrefixPower(n)];
}
function getSIPrefixPower(n) {
  return clamp(minSIPrefix, n ? Math.floor(Math.log10(Math.abs(n)) / 3) * 3 : 0, maxSIPrefix);
}
function addSign(num, numString, signType = "") {
  if (signType === "(") {
    return num >= 0 ? numString : `(${numString})`;
  }
  const plusSign = signType === "+" ? "+" : "";
  return `${num >= 0 ? plusSign : minusSign}${numString}`;
}
function addPadding(numString, width2, fill = " ", align2 = ">") {
  let result = numString;
  if (align2 === ">" || !align2) {
    result = result.padStart(width2, fill);
  } else if (align2 === "<") {
    result = result.padEnd(width2, fill);
  } else if (align2 === "^") {
    const padWidth = Math.max(0, width2 - result.length);
    const padLeft = Math.ceil(padWidth / 2);
    const padRight = Math.floor(padWidth / 2);
    result = result.padStart(padLeft + result.length, fill);
    result = result.padEnd(padRight + result.length, fill);
  }
  return result;
}

// packages/ag-charts-community/src/util/ticks.ts
var tInterval = (timeInterval, baseDuration, step) => ({
  duration: baseDuration * step,
  timeInterval,
  step
});
var TickIntervals = [
  tInterval(second, durationSecond, 1),
  tInterval(second, durationSecond, 5),
  tInterval(second, durationSecond, 15),
  tInterval(second, durationSecond, 30),
  tInterval(minute, durationMinute, 1),
  tInterval(minute, durationMinute, 5),
  tInterval(minute, durationMinute, 15),
  tInterval(minute, durationMinute, 30),
  tInterval(hour, durationHour, 1),
  tInterval(hour, durationHour, 3),
  tInterval(hour, durationHour, 6),
  tInterval(hour, durationHour, 12),
  tInterval(day, durationDay, 1),
  tInterval(day, durationDay, 2),
  tInterval(sunday, durationWeek, 1),
  tInterval(sunday, durationWeek, 2),
  tInterval(sunday, durationWeek, 3),
  tInterval(month, durationMonth, 1),
  tInterval(month, durationMonth, 2),
  tInterval(month, durationMonth, 3),
  tInterval(month, durationMonth, 4),
  tInterval(month, durationMonth, 6),
  tInterval(year, durationYear, 1)
];
var TickMultipliers = [1, 2, 5, 10];
function isCloseToInteger(n, delta2) {
  return Math.abs(Math.round(n) - n) < delta2;
}
function createTicks(start2, stop, count, minCount, maxCount, visibleRange) {
  if (count < 2) {
    return [start2, stop];
  }
  const step = tickStep(start2, stop, count, minCount, maxCount);
  if (!Number.isFinite(step)) {
    return [];
  }
  if (!isCloseToInteger(start2 / step, 1e-12)) {
    start2 = Math.ceil(start2 / step) * step;
  }
  if (!isCloseToInteger(stop / step, 1e-12)) {
    stop = Math.floor(stop / step) * step;
  }
  if (visibleRange != null && visibleRange[0] !== 0 && visibleRange[1] !== 1) {
    const rangeExtent = stop - start2;
    const adjustedStart = start2 + rangeExtent * visibleRange[0];
    const adjustedEnd = stop - rangeExtent * (1 - visibleRange[1]);
    return range(adjustedStart - adjustedStart % step, adjustedEnd + adjustedEnd % step, step);
  }
  return range(start2, stop, step);
}
function getTickInterval(start2, stop, count, minCount, maxCount, targetInterval) {
  const target = targetInterval ?? Math.abs(stop - start2) / Math.max(count, 1);
  let i = 0;
  for (const tickInterval of TickIntervals) {
    if (target <= tickInterval.duration)
      break;
    i++;
  }
  if (i === 0) {
    const step2 = Math.max(tickStep(start2, stop, count, minCount, maxCount), 1);
    return millisecond.every(step2);
  } else if (i === TickIntervals.length) {
    const step2 = targetInterval == null ? tickStep(start2 / durationYear, stop / durationYear, count, minCount, maxCount) : 1;
    return year.every(step2);
  }
  const i0 = TickIntervals[i - 1];
  const i1 = TickIntervals[i];
  const { timeInterval, step } = target - i0.duration < i1.duration - target ? i0 : i1;
  return timeInterval.every(step);
}
function tickStep(start2, end2, count, minCount = 0, maxCount = Infinity) {
  if (start2 === end2) {
    return clamp(1, minCount, maxCount);
  } else if (count < 1) {
    return NaN;
  }
  const extent2 = Math.abs(end2 - start2);
  const step = 10 ** Math.floor(Math.log10(extent2 / count));
  let m = NaN, minDiff = Infinity, isInBounds = false;
  for (const multiplier of TickMultipliers) {
    const c = Math.ceil(extent2 / (multiplier * step));
    const validBounds = c >= minCount && c <= maxCount;
    if (isInBounds && !validBounds)
      continue;
    const diffCount = Math.abs(c - count);
    if (minDiff > diffCount || isInBounds !== validBounds) {
      isInBounds || (isInBounds = validBounds);
      minDiff = diffCount;
      m = multiplier;
    }
  }
  return m * step;
}
function decimalPlaces(decimal) {
  for (let i = decimal.length - 1; i >= 0; i -= 1) {
    if (decimal[i] !== "0") {
      return i + 1;
    }
  }
  return 0;
}
function tickFormat(ticks, format) {
  const options = parseFormat(format ?? ",f");
  if (options.precision == null || isNaN(options.precision)) {
    if (!options.type || "eEFgGnprs".includes(options.type)) {
      options.precision = Math.max(
        ...ticks.map((x) => {
          if (!Number.isFinite(x))
            return 0;
          const [integer, decimal] = x.toExponential((options.type ? 6 : 12) - 1).split(/[.e]/g);
          return (integer !== "1" && integer !== "-1" ? 1 : 0) + decimalPlaces(decimal) + 1;
        })
      );
    } else if ("f%".includes(options.type)) {
      options.precision = Math.max(
        ...ticks.map((x) => {
          if (!Number.isFinite(x) || x === 0)
            return 0;
          const l = Math.floor(Math.log10(Math.abs(x)));
          const digits = options.type ? 6 : 12;
          const decimal = x.toExponential(digits - 1).split(/[.e]/g)[1];
          const decimalLength = decimalPlaces(decimal);
          return Math.max(0, decimalLength - l);
        })
      );
    }
  }
  const formatter = numberFormat(options);
  return (n) => formatter(Number(n));
}
function range(start2, end2, step) {
  if (!Number.isFinite(step) || step <= 0) {
    return [];
  }
  const f = 10 ** countFractionDigits(step);
  const d0 = Math.min(start2, end2);
  const d1 = Math.max(start2, end2);
  const out = [];
  for (let i = 0; ; i += 1) {
    const p = Math.round((d0 + step * i) * f) / f;
    if (p > d1)
      break;
    out.push(p);
  }
  return out;
}
function isDenseInterval(count, availableRange) {
  if (count >= availableRange) {
    logger_exports.warnOnce(
      `the configured interval results in more than 1 item per pixel, ignoring. Supply a larger interval or omit this configuration`
    );
    return true;
  }
  return false;
}
function niceTicksDomain(start2, end2) {
  const extent2 = Math.abs(end2 - start2);
  const step = 10 ** Math.floor(Math.log10(extent2));
  let minError = Infinity, ticks = [start2, end2];
  for (const multiplier of TickMultipliers) {
    const m = multiplier * step;
    const d0 = Math.floor(start2 / m) * m;
    const d1 = Math.ceil(end2 / m) * m;
    const error2 = 1 - extent2 / Math.abs(d1 - d0);
    if (minError > error2) {
      minError = error2;
      ticks = [d0, d1];
    }
  }
  return ticks;
}
function estimateTickCount(rangeExtent, zoomExtent, minSpacing, maxSpacing, defaultTickCount, defaultMinSpacing) {
  defaultMinSpacing = Math.max(defaultMinSpacing, rangeExtent / (defaultTickCount + 1));
  if (isNaN(minSpacing)) {
    minSpacing = defaultMinSpacing;
  }
  if (isNaN(maxSpacing)) {
    maxSpacing = rangeExtent;
  }
  if (minSpacing > maxSpacing) {
    if (minSpacing === defaultMinSpacing) {
      minSpacing = maxSpacing;
    } else {
      maxSpacing = minSpacing;
    }
  }
  const maxTickCount = Math.max(1, Math.floor(rangeExtent / (zoomExtent * minSpacing)));
  const minTickCount = Math.min(maxTickCount, Math.ceil(rangeExtent / (zoomExtent * maxSpacing)));
  const tickCount = clamp(minTickCount, Math.floor(defaultTickCount / zoomExtent), maxTickCount);
  return { minTickCount, maxTickCount, tickCount };
}

// packages/ag-charts-community/src/scale/timeScale.ts
var TimeScale = class extends ContinuousScale {
  constructor() {
    super([], [0, 1]);
    this.type = "time";
  }
  toDomain(d) {
    return new Date(d);
  }
  convert(value, clamp2) {
    if (!(value instanceof Date))
      value = new Date(value);
    return super.convert(value, clamp2);
  }
  invert(value) {
    return new Date(super.invert(value));
  }
  niceDomain(ticks, domain = this.domain) {
    if (domain.length < 2)
      return [];
    const maxAttempts = 4;
    let [d0, d1] = domain;
    for (let i = 0; i < maxAttempts; i++) {
      const [n0, n1] = updateNiceDomainIteration(d0, d1, ticks);
      if (dateToNumber(d0) === dateToNumber(n0) && dateToNumber(d1) === dateToNumber(n1)) {
        break;
      }
      d0 = n0;
      d1 = n1;
    }
    return [d0, d1];
  }
  /**
   * Returns uniformly-spaced dates that represent the scale's domain.
   */
  ticks(params, domain = this.domain, visibleRange = [0, 1]) {
    const { nice, interval, tickCount = ContinuousScale.defaultTickCount, minTickCount, maxTickCount } = params;
    if (domain.length < 2)
      return [];
    const timestamps = domain.map(dateToNumber);
    const start2 = timestamps[0];
    const stop = timestamps[timestamps.length - 1];
    if (interval != null) {
      const availableRange = this.getPixelRange();
      return getDateTicksForInterval({ start: start2, stop, interval, availableRange, visibleRange }) ?? getDefaultDateTicks({ start: start2, stop, tickCount, minTickCount, maxTickCount, visibleRange });
    } else if (nice && tickCount === 2) {
      return domain;
    } else if (nice && tickCount === 1) {
      return domain.slice(0, 1);
    }
    return getDefaultDateTicks({ start: start2, stop, tickCount, minTickCount, maxTickCount, visibleRange });
  }
  _tickFormatter({ domain, ticks, specifier }, formatOffset) {
    return specifier != null ? buildFormatter(specifier) : defaultTimeTickFormat(ticks, domain, formatOffset);
  }
  /**
   * Returns a time format function suitable for displaying tick values.
   *
   * @param ticks Optional array of tick values for custom formatting.
   * @param domain Optional array representing the [min, max] values of the time axis.
   * @param specifier Optional format specifier string for custom date formatting (e.g., `%Y`, `%m`, `%d`).
   * @param formatOffset Optional number for applying an offset to the format (e.g., timezone shifts).
   * @returns A function that formats a `Date` object into a string based on the provided specifier or default format.
   */
  tickFormatter(params) {
    return this._tickFormatter(params);
  }
  datumFormatter(params) {
    return this._tickFormatter(params, 1);
  }
};
function getDefaultDateTicks({
  start: start2,
  stop,
  tickCount,
  minTickCount,
  maxTickCount,
  visibleRange
}) {
  const t = getTickInterval(start2, stop, tickCount, minTickCount, maxTickCount);
  return t ? t.range(new Date(start2), new Date(stop), { visibleRange }) : [];
}
function getDateTicksForInterval({
  start: start2,
  stop,
  interval,
  availableRange,
  visibleRange
}) {
  if (!interval) {
    return [];
  }
  if (interval instanceof TimeInterval) {
    const ticks2 = interval.range(new Date(start2), new Date(stop), { visibleRange });
    if (isDenseInterval(ticks2.length, availableRange)) {
      return;
    }
    return ticks2;
  }
  const absInterval = Math.abs(interval);
  if (isDenseInterval(Math.abs(stop - start2) / absInterval, availableRange))
    return;
  const timeInterval = TickIntervals.findLast((tickInterval) => absInterval % tickInterval.duration === 0);
  if (timeInterval) {
    const i = timeInterval.timeInterval.every(absInterval / (timeInterval.duration / timeInterval.step));
    return i.range(new Date(start2), new Date(stop), { visibleRange });
  }
  let date2 = new Date(Math.min(start2, stop));
  const stopDate = new Date(Math.max(start2, stop));
  const ticks = [];
  while (date2 <= stopDate) {
    ticks.push(date2);
    date2 = new Date(date2);
    date2.setMilliseconds(date2.getMilliseconds() + absInterval);
  }
  return ticks;
}
function updateNiceDomainIteration(d0, d1, ticks) {
  const { interval } = ticks;
  const start2 = Math.min(dateToNumber(d0), dateToNumber(d1));
  const stop = Math.max(dateToNumber(d0), dateToNumber(d1));
  const isReversed = d0 > d1;
  let i;
  if (interval instanceof TimeInterval) {
    i = interval;
  } else {
    const tickCount = typeof interval === "number" ? (stop - start2) / Math.max(interval, 1) : ticks.tickCount ?? ContinuousScale.defaultTickCount;
    i = getTickInterval(start2, stop, tickCount, ticks.minTickCount, ticks.maxTickCount);
  }
  if (i) {
    const intervalRange = i.range(new Date(start2), new Date(stop), { extend: true });
    const domain = isReversed ? [...intervalRange].reverse() : intervalRange;
    const n0 = domain[0];
    const n1 = domain.at(-1);
    return [n0, n1];
  } else {
    return [d0, d1];
  }
}

// packages/ag-charts-community/src/scale/ordinalTimeScale.ts
var OrdinalTimeScale = class _OrdinalTimeScale extends BandScale {
  constructor() {
    super(...arguments);
    this.type = "ordinal-time";
    this._domain = [];
    this.isReversed = false;
  }
  static is(value) {
    return value instanceof _OrdinalTimeScale;
  }
  set domain(domain) {
    if (domain === this._domain)
      return;
    this.invalid = true;
    this._domain = domain;
    this.isReversed = domain.length > 0 && domain[0] > domain[domain.length - 1];
    this.sortedTimestamps = void 0;
    this.precomputedSteps = void 0;
  }
  get domain() {
    return this._domain;
  }
  toDomain(value) {
    return new Date(value);
  }
  normalizeDomains(...domains) {
    const sortedDomains = domains.filter((domain2) => domain2.length > 0).map((domain2) => {
      let sortOrder = datesSortOrder(domain2);
      if (sortOrder == null) {
        domain2 = sortAndUniqueDates(domain2.slice());
        sortOrder = 1;
      }
      return { domain: domain2, sortOrder };
    });
    if (sortedDomains.length === 0)
      return { domain: [], animatable: false };
    if (sortedDomains.length === 1)
      return { domain: sortedDomains[0].domain, animatable: true };
    let domain = sortedDomains.flatMap((s) => s.domain);
    domain = sortAndUniqueDates(domain);
    if (sortedDomains.every((s) => s.sortOrder === -1)) {
      domain.reverse();
    }
    return { domain, animatable: true };
  }
  ticks({ interval, maxTickCount }, domain = this.domain, visibleRange = [0, 1]) {
    if (!domain.length) {
      return [];
    }
    this.refresh();
    const { isReversed } = this;
    if (interval == null) {
      return getDefaultTicks(domain, maxTickCount, isReversed, visibleRange);
    }
    const [t0, t1] = [domain[0].valueOf(), domain.at(-1).valueOf()];
    const start2 = Math.min(t0, t1);
    const stop = Math.max(t0, t1);
    const [r0, r1] = this.range;
    const availableRange = Math.abs(r1 - r0);
    const ticks = getDateTicksForInterval({ start: start2, stop, interval, availableRange, visibleRange }) ?? [];
    let lastIndex = -1;
    return ticks.filter((tick) => {
      const index = this.findInterval(tick.valueOf());
      const duplicated = index === lastIndex;
      lastIndex = index;
      return !duplicated;
    });
  }
  getSortedTimestamps() {
    let { sortedTimestamps } = this;
    if (sortedTimestamps == null) {
      sortedTimestamps = this.domain.map(dateToNumber);
      if (this.isReversed)
        sortedTimestamps.reverse();
      this.sortedTimestamps = sortedTimestamps;
    }
    return sortedTimestamps;
  }
  getPrecomputedSteps() {
    const { domain } = this;
    let { precomputedSteps } = this;
    const computedStepCount = domain.length < 1e4 ? domain.length : Math.ceil(domain.length / 16);
    if (precomputedSteps != null || computedStepCount <= 1)
      return precomputedSteps;
    const sortedTimestamps = this.getSortedTimestamps();
    precomputedSteps = new Int32Array(computedStepCount);
    const d0 = sortedTimestamps[0];
    const d1 = sortedTimestamps[sortedTimestamps.length - 1];
    const dRange = d1 - d0;
    const low = 0;
    const high = sortedTimestamps.length - 1;
    for (let i = 0; i < precomputedSteps.length; i += 1) {
      precomputedSteps[i] = this.findIntervalInRange(d0 + i / computedStepCount * dRange, low, high);
    }
    this.precomputedSteps = precomputedSteps;
  }
  findIntervalInRange(target, low, high) {
    const sortedTimestamps = this.getSortedTimestamps();
    while (low <= high) {
      const mid = (low + high) / 2 | 0;
      if (sortedTimestamps[mid] === target) {
        return mid;
      } else if (sortedTimestamps[mid] < target) {
        low = mid + 1;
      } else {
        high = mid - 1;
      }
    }
    return low;
  }
  findInterval(target) {
    const precomputedSteps = this.getPrecomputedSteps();
    let low;
    let high;
    if (precomputedSteps == null) {
      low = 0;
      high = this.domain.length - 1;
    } else {
      const sortedTimestamps = this.getSortedTimestamps();
      const d0 = sortedTimestamps[0];
      const d1 = sortedTimestamps[sortedTimestamps.length - 1];
      const i = Math.min(
        (target - d0) / (d1 - d0) * precomputedSteps.length | 0,
        precomputedSteps.length - 1 | 0
      );
      low = precomputedSteps[i];
      high = i < precomputedSteps.length - 2 ? precomputedSteps[i + 1] : sortedTimestamps.length - 1;
    }
    return this.findIntervalInRange(target, low, high);
  }
  /**
   * Returns a time format function suitable for displaying tick values.
   * @param specifier If the specifier string is provided, this method is equivalent to
   * the {@link TimeLocaleObject.format} method.
   * If no specifier is provided, this method returns the default time format function.
   */
  tickFormatter({ domain, ticks, specifier }) {
    return specifier != null ? buildFormatter(specifier) : defaultTimeTickFormat(ticks, domain);
  }
  datumFormatter(params) {
    return this.tickFormatter(params);
  }
  invert(position, nearest = false) {
    this.refresh();
    const { domain } = this;
    if (nearest) {
      const index = this.invertNearestIndex(position - this.bandwidth / 2);
      return index != null ? domain[index] : void 0;
    }
    const closest = findMinValue(0, domain.length - 1, (i) => {
      const p = this.ordinalRange(i);
      return p >= position ? domain[i] : void 0;
    });
    return closest ?? domain[0];
  }
  getIndex(value) {
    const sortedTimestamps = this.getSortedTimestamps();
    const n = Number(value);
    if (n < sortedTimestamps[0]) {
      return void 0;
    }
    let i = this.findInterval(n);
    if (this.isReversed) {
      i = this.domain.length - i - 1;
    }
    return i;
  }
};
function getDefaultTicks(domain, maxTickCount, isReversed, visibleRange) {
  const ticks = [];
  const tickEvery = Math.ceil(domain.length / maxTickCount);
  const tickOffset = Math.floor(tickEvery / 2);
  const startIndex = Math.floor(visibleRange[0] * domain.length);
  const endIndex = Math.ceil(visibleRange[1] * domain.length);
  for (let index = startIndex; index < endIndex; index += 1) {
    const tickIndex = isReversed ? domain.length - 1 - index : index;
    if (tickEvery <= 0 || (tickIndex + tickOffset) % tickEvery === 0) {
      ticks.push(domain[index]);
    }
  }
  if (isReversed) {
    ticks.reverse();
  }
  return ticks;
}

// packages/ag-charts-community/src/scene/selection.ts
var Selection = class _Selection {
  constructor(parentNode, classOrFactory, autoCleanup = true) {
    this.parentNode = parentNode;
    this.autoCleanup = autoCleanup;
    this.garbageBin = /* @__PURE__ */ new Set();
    this._nodesMap = /* @__PURE__ */ new Map();
    this._nodes = [];
    this.data = [];
    this.debug = Debug.create(true, "scene", "scene:selections");
    this.nodeFactory = Object.prototype.isPrototypeOf.call(Node, classOrFactory) ? () => new classOrFactory() : classOrFactory;
  }
  static select(parent, classOrFactory, garbageCollection = true) {
    return new _Selection(parent, classOrFactory, garbageCollection);
  }
  static selectAll(parent, predicate) {
    const results = [];
    const traverse = (node) => {
      if (predicate(node)) {
        results.push(node);
      }
      for (const child of node.children()) {
        traverse(child);
      }
    };
    traverse(parent);
    return results;
  }
  static selectByClass(node, ...Classes) {
    return _Selection.selectAll(node, (n) => Classes.some((C2) => n instanceof C2));
  }
  static selectByTag(node, tag) {
    return _Selection.selectAll(node, (n) => n.tag === tag);
  }
  createNode(datum, initializer, idx) {
    const node = this.nodeFactory(datum);
    node.datum = datum;
    initializer?.(node);
    if (idx == null) {
      this._nodes.push(node);
    } else {
      this._nodes.splice(idx, 0, node);
    }
    this.parentNode.appendChild(node);
    return node;
  }
  /**
   * Update the data in a selection. If an `getDatumId()` function is provided, maintain a list of ids related to
   * the nodes. Otherwise, take the more efficient route of simply creating and destroying nodes at the end
   * of the array.
   */
  update(data, initializer, getDatumId) {
    if (this.garbageBin.size > 0) {
      this.debug(`Selection - update() called with pending garbage`, data);
    }
    if (getDatumId) {
      const dataMap = new Map(
        data.map((datum, idx) => [getDatumId(datum), [datum, idx]])
      );
      for (const [node, datumId] of this._nodesMap.entries()) {
        if (dataMap.has(datumId)) {
          const [newDatum] = dataMap.get(datumId);
          node.datum = newDatum;
          this.garbageBin.delete(node);
          dataMap.delete(datumId);
        } else {
          this.garbageBin.add(node);
        }
      }
      for (const [datumId, [datum, idx]] of dataMap.entries()) {
        this._nodesMap.set(this.createNode(datum, initializer, idx), datumId);
      }
    } else {
      const maxLength = Math.max(data.length, this.data.length);
      for (let i = 0; i < maxLength; i++) {
        if (i >= data.length) {
          this.garbageBin.add(this._nodes[i]);
        } else if (i >= this._nodes.length) {
          this.createNode(data[i], initializer);
        } else {
          this._nodes[i].datum = data[i];
          this.garbageBin.delete(this._nodes[i]);
        }
      }
    }
    this.data = data.slice();
    if (this.autoCleanup) {
      this.cleanup();
    }
    return this;
  }
  cleanup() {
    if (this.garbageBin.size === 0) {
      return this;
    }
    this._nodes = this._nodes.filter((node) => {
      if (this.garbageBin.has(node)) {
        this._nodesMap.delete(node);
        this.garbageBin.delete(node);
        node.destroy();
        return false;
      }
      return true;
    });
    return this;
  }
  clear() {
    this.update([]);
    return this;
  }
  isGarbage(node) {
    return this.garbageBin.has(node);
  }
  each(iterate2) {
    for (const entry of this._nodes.entries()) {
      iterate2(entry[1], entry[1].datum, entry[0]);
    }
    return this;
  }
  *[Symbol.iterator]() {
    for (let index = 0; index < this._nodes.length; index++) {
      const node = this._nodes[index];
      yield { node, datum: node.datum, index };
    }
  }
  select(predicate) {
    return _Selection.selectAll(this.parentNode, predicate);
  }
  selectByClass(Class) {
    return _Selection.selectByClass(this.parentNode, Class);
  }
  selectByTag(tag) {
    return _Selection.selectByTag(this.parentNode, tag);
  }
  nodes() {
    return this._nodes;
  }
  at(index) {
    return this._nodes.at(index);
  }
  get length() {
    return this._nodes.length;
  }
};

// packages/ag-charts-community/src/util/distance.ts
function pointsDistanceSquared(x1, y1, x2, y2) {
  const dx = x1 - x2;
  const dy = y1 - y2;
  return dx * dx + dy * dy;
}
function lineDistanceSquared(x, y, x1, y1, x2, y2, best) {
  if (x1 === x2 && y1 === y2) {
    return Math.min(best, pointsDistanceSquared(x, y, x1, y1));
  }
  const dx = x2 - x1;
  const dy = y2 - y1;
  const t = Math.max(0, Math.min(1, ((x - x1) * dx + (y - y1) * dy) / (dx * dx + dy * dy)));
  const ix = x1 + t * dx;
  const iy = y1 + t * dy;
  return Math.min(best, pointsDistanceSquared(x, y, ix, iy));
}
function arcDistanceSquared(x, y, cx, cy, radius, startAngle, endAngle, counterClockwise, best) {
  if (counterClockwise) {
    [endAngle, startAngle] = [startAngle, endAngle];
  }
  const angle = Math.atan2(y - cy, x - cx);
  if (!isBetweenAngles(angle, startAngle, endAngle)) {
    const startX = cx + Math.cos(startAngle) * radius;
    const startY = cy + Math.sin(startAngle) * radius;
    const endX = cx + Math.cos(startAngle) * radius;
    const endY = cy + Math.sin(startAngle) * radius;
    return Math.min(best, pointsDistanceSquared(x, y, startX, startY), pointsDistanceSquared(x, y, endX, endY));
  }
  const distToArc = radius - Math.sqrt(pointsDistanceSquared(x, y, cx, cy));
  return Math.min(best, distToArc * distToArc);
}

// packages/ag-charts-community/src/scene/shape/line.ts
var Line = class extends Shape {
  constructor(opts = {}) {
    super(opts);
    this.x1 = 0;
    this.y1 = 0;
    this.x2 = 0;
    this.y2 = 0;
    this.restoreOwnStyles();
  }
  set x(value) {
    this.x1 = value;
    this.x2 = value;
  }
  set y(value) {
    this.y1 = value;
    this.y2 = value;
  }
  get midPoint() {
    return { x: (this.x1 + this.x2) / 2, y: (this.y1 + this.y2) / 2 };
  }
  computeBBox() {
    return new BBox(
      Math.min(this.x1, this.x2),
      Math.min(this.y1, this.y2),
      Math.abs(this.x2 - this.x1),
      Math.abs(this.y2 - this.y1)
    );
  }
  isPointInPath(x, y) {
    if (this.x1 === this.x2 || this.y1 === this.y2) {
      return this.getBBox().clone().grow(this.strokeWidth / 2).containsPoint(x, y);
    }
    return false;
  }
  distanceSquared(px, py) {
    const { x1, y1, x2, y2 } = this;
    return lineDistanceSquared(px, py, x1, y1, x2, y2, Infinity);
  }
  render(renderCtx) {
    const { ctx, devicePixelRatio } = renderCtx;
    let { x1, y1, x2, y2 } = this;
    if (x1 === x2) {
      const { strokeWidth } = this;
      const x = Math.round(x1 * devicePixelRatio) / devicePixelRatio + Math.trunc(strokeWidth * devicePixelRatio) % 2 / (devicePixelRatio * 2);
      x1 = x;
      x2 = x;
    } else if (y1 === y2) {
      const { strokeWidth } = this;
      const y = Math.round(y1 * devicePixelRatio) / devicePixelRatio + Math.trunc(strokeWidth * devicePixelRatio) % 2 / (devicePixelRatio * 2);
      y1 = y;
      y2 = y;
    }
    ctx.beginPath();
    ctx.moveTo(x1, y1);
    ctx.lineTo(x2, y2);
    this.fillStroke(ctx);
    this.fillShadow?.markClean();
    super.render(renderCtx);
  }
  toSVG() {
    if (!this.visible)
      return;
    const element2 = createSvgElement("line");
    element2.setAttribute("x1", String(this.x1));
    element2.setAttribute("y1", String(this.y1));
    element2.setAttribute("x2", String(this.x2));
    element2.setAttribute("y2", String(this.y2));
    this.applySvgStrokeAttributes(element2);
    return {
      elements: [element2]
    };
  }
};
Line.className = "Line";
Line.defaultStyles = { ...Shape.defaultStyles, fill: void 0, strokeWidth: 1 };
__decorateClass([
  SceneChangeDetection()
], Line.prototype, "x1", 2);
__decorateClass([
  SceneChangeDetection()
], Line.prototype, "y1", 2);
__decorateClass([
  SceneChangeDetection()
], Line.prototype, "x2", 2);
__decorateClass([
  SceneChangeDetection()
], Line.prototype, "y2", 2);

// packages/ag-charts-community/src/util/format.util.ts
var defaultNumberFormatter = new Intl.NumberFormat("en-US", { maximumFractionDigits: 2, useGrouping: false });
var percentFormatter = new Intl.NumberFormat("en-US", { style: "percent" });
function formatValue(value, maximumFractionDigits = 2) {
  if (typeof value === "number") {
    return formatNumber(value, maximumFractionDigits);
  }
  return String(value ?? "");
}
function formatPercent(value) {
  return percentFormatter.format(value);
}
function formatNumber(value, maximumFractionDigits) {
  if (maximumFractionDigits === 2) {
    return defaultNumberFormatter.format(value);
  }
  return new Intl.NumberFormat("en-US", { maximumFractionDigits, useGrouping: false }).format(value);
}

// packages/ag-charts-community/src/scene/shape/range.ts
var Range = class extends Shape {
  constructor(opts = {}) {
    super(opts);
    this.x1 = 0;
    this.y1 = 0;
    this.x2 = 0;
    this.y2 = 0;
    this.startLine = false;
    this.endLine = false;
    this.isRange = false;
    this.restoreOwnStyles();
  }
  computeBBox() {
    return new BBox(this.x1, this.y1, this.x2 - this.x1, this.y2 - this.y1);
  }
  isPointInPath(_x, _y) {
    return false;
  }
  render(renderCtx) {
    const { ctx } = renderCtx;
    let { x1, y1, x2, y2 } = this;
    x1 = this.align(x1);
    y1 = this.align(y1);
    x2 = this.align(x2);
    y2 = this.align(y2);
    const { fill, opacity, isRange } = this;
    const fillActive = !!(isRange && fill);
    if (fillActive) {
      const { fillOpacity } = this;
      this.applyFill(ctx);
      ctx.globalAlpha = opacity * fillOpacity;
      ctx.beginPath();
      ctx.moveTo(x1, y1);
      ctx.lineTo(x2, y1);
      ctx.lineTo(x2, y2);
      ctx.lineTo(x1, y2);
      ctx.closePath();
      ctx.fill();
    }
    const { stroke, strokeWidth, startLine, endLine } = this;
    const strokeActive = !!((startLine || endLine) && stroke && strokeWidth);
    if (strokeActive) {
      const { strokeOpacity, lineDash, lineDashOffset, lineCap, lineJoin } = this;
      this.applyStroke(ctx);
      ctx.globalAlpha = opacity * strokeOpacity;
      ctx.lineWidth = strokeWidth;
      if (lineDash) {
        ctx.setLineDash(lineDash);
      }
      if (lineDashOffset) {
        ctx.lineDashOffset = lineDashOffset;
      }
      if (lineCap) {
        ctx.lineCap = lineCap;
      }
      if (lineJoin) {
        ctx.lineJoin = lineJoin;
      }
      ctx.beginPath();
      if (startLine) {
        ctx.moveTo(x1, y1);
        ctx.lineTo(x2, y1);
      }
      if (endLine) {
        ctx.moveTo(x2, y2);
        ctx.lineTo(x1, y2);
      }
      ctx.stroke();
    }
    this.fillShadow?.markClean();
    super.render(renderCtx);
  }
};
Range.className = "Range";
Range.defaultStyles = {
  ...Shape.defaultStyles,
  strokeWidth: 1
};
__decorateClass([
  SceneChangeDetection()
], Range.prototype, "x1", 2);
__decorateClass([
  SceneChangeDetection()
], Range.prototype, "y1", 2);
__decorateClass([
  SceneChangeDetection()
], Range.prototype, "x2", 2);
__decorateClass([
  SceneChangeDetection()
], Range.prototype, "y2", 2);
__decorateClass([
  SceneChangeDetection()
], Range.prototype, "startLine", 2);
__decorateClass([
  SceneChangeDetection()
], Range.prototype, "endLine", 2);
__decorateClass([
  SceneChangeDetection()
], Range.prototype, "isRange", 2);

// packages/ag-charts-community/src/chart/label.ts
var Label = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.enabled = true;
    this.fontSize = 12;
    this.fontFamily = "Verdana, sans-serif";
  }
  getFont() {
    return TextUtils.toFontString(this);
  }
};
__decorateClass([
  Validate(BOOLEAN)
], Label.prototype, "enabled", 2);
__decorateClass([
  Validate(COLOR_STRING, { optional: true })
], Label.prototype, "color", 2);
__decorateClass([
  Validate(FONT_STYLE, { optional: true })
], Label.prototype, "fontStyle", 2);
__decorateClass([
  Validate(FONT_WEIGHT, { optional: true })
], Label.prototype, "fontWeight", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], Label.prototype, "fontSize", 2);
__decorateClass([
  Validate(STRING)
], Label.prototype, "fontFamily", 2);
__decorateClass([
  Validate(FUNCTION, { optional: true })
], Label.prototype, "formatter", 2);
function calculateLabelRotation(opts) {
  const { parallelFlipRotation = 0, regularFlipRotation = 0 } = opts;
  const configuredRotation = opts.rotation ? normalizeAngle360(toRadians(opts.rotation)) : 0;
  const parallelFlipFlag = !configuredRotation && parallelFlipRotation >= 0 && parallelFlipRotation <= Math.PI ? -1 : 1;
  const regularFlipFlag = !configuredRotation && regularFlipRotation >= 0 && regularFlipRotation <= Math.PI ? -1 : 1;
  let defaultRotation = 0;
  if (opts.parallel) {
    defaultRotation = parallelFlipFlag * Math.PI / 2;
  } else if (regularFlipFlag === -1) {
    defaultRotation = Math.PI;
  }
  return { configuredRotation, defaultRotation, parallelFlipFlag, regularFlipFlag };
}
function getLabelSpacing(minSpacing, rotated) {
  if (!isNaN(minSpacing)) {
    return minSpacing;
  }
  return rotated ? 0 : 10;
}
function getTextBaseline(parallel, labelRotation, sideFlag, parallelFlipFlag) {
  if (parallel && !labelRotation) {
    return sideFlag * parallelFlipFlag === -1 ? "hanging" : "bottom";
  }
  return "middle";
}
function getTextAlign(parallel, labelRotation, labelAutoRotation, sideFlag, regularFlipFlag) {
  const labelRotated = labelRotation > 0 && labelRotation <= Math.PI;
  const labelAutoRotated = labelAutoRotation > 0 && labelAutoRotation <= Math.PI;
  const alignFlag = labelRotated || labelAutoRotated ? -1 : 1;
  if (parallel) {
    if (labelRotation || labelAutoRotation) {
      if (sideFlag * alignFlag === -1) {
        return "end";
      }
    } else {
      return "center";
    }
  } else if (sideFlag * regularFlipFlag === -1) {
    return "end";
  }
  return "start";
}
function createLabelData(tickData, labelX, labelMatrix, textMeasurer) {
  const labelData = [];
  for (const { tickLabel: text, translationY } of tickData) {
    if (!text)
      continue;
    const { width: width2, height: height2 } = textMeasurer.measureLines(text);
    const bbox = new BBox(labelX, translationY, width2, height2);
    const translatedBBox = new BBox(labelX, translationY, 0, 0);
    labelMatrix.transformBBox(translatedBBox, bbox);
    const { x, y } = bbox;
    labelData.push({
      point: { x, y },
      label: { text, width: width2, height: height2 }
    });
  }
  return labelData;
}

// packages/ag-charts-community/src/util/value.ts
function isNumberObject(value) {
  return value != null && Object.hasOwn(value, "valueOf") && isFiniteNumber(value.valueOf());
}
function isContinuous(value) {
  return isFiniteNumber(value) || isValidDate(value) || isNumberObject(value);
}
function checkDatum(value, isContinuousScale) {
  return value != null && (!isContinuousScale || isContinuous(value));
}

// packages/ag-charts-community/src/chart/crossline/crossLine.ts
var MATCHING_CROSSLINE_TYPE = (property) => {
  return property === "value" ? predicateWithMessage(
    (_, ctx) => ctx.target["type"] === "line",
    (ctx) => ctx.target["type"] === "range" ? `crossLine type 'range' to have a 'range' property instead of 'value'` : `crossLine property 'type' to be 'line'`
  ) : predicateWithMessage(
    (_, ctx) => ctx.target["type"] === "range",
    (ctx) => ctx.target.type === "line" ? `crossLine type 'line' to have a 'value' property instead of 'range'` : `crossLine property 'type' to be 'range'`
  );
};
var validateCrossLineValues = (type, value, range2, scale, visibilityCheck) => {
  const lineCrossLine = type === "line" && value !== void 0;
  const rangeCrossLine = type === "range" && range2 !== void 0;
  if (!lineCrossLine && !rangeCrossLine) {
    return true;
  }
  const [start2, end2] = range2 ?? [value, void 0];
  const isContinuous2 = ContinuousScale.is(scale) || OrdinalTimeScale.is(scale);
  const validStart = checkDatum(start2, isContinuous2) && !isNaN(scale.convert(start2));
  const validEnd = checkDatum(end2, isContinuous2) && !isNaN(scale.convert(end2));
  if (lineCrossLine && validStart || rangeCrossLine && validStart && validEnd) {
    return visibilityCheck?.() ?? true;
  }
  const message = [`Expecting crossLine`];
  if (rangeCrossLine) {
    if (!validStart) {
      message.push(`range start ${stringifyValue(start2)}`);
    }
    if (!validEnd) {
      message.push(`${validStart ? "" : "and "}range end ${stringifyValue(end2)}`);
    }
  } else {
    message.push(`value ${stringifyValue(start2)}`);
  }
  message.push(`to match the axis scale domain.`);
  logger_exports.warnOnce(message.join(" "));
  return false;
};

// packages/ag-charts-community/src/chart/crossline/crossLineLabelPosition.ts
var horizontalCrosslineTranslationDirections = {
  top: { xTranslationDirection: 0, yTranslationDirection: -1 },
  bottom: { xTranslationDirection: 0, yTranslationDirection: 1 },
  left: { xTranslationDirection: -1, yTranslationDirection: 0 },
  right: { xTranslationDirection: 1, yTranslationDirection: 0 },
  "top-left": { xTranslationDirection: 1, yTranslationDirection: -1 },
  "top-right": { xTranslationDirection: -1, yTranslationDirection: -1 },
  "bottom-left": { xTranslationDirection: 1, yTranslationDirection: 1 },
  "bottom-right": { xTranslationDirection: -1, yTranslationDirection: 1 },
  inside: { xTranslationDirection: 0, yTranslationDirection: 0 },
  "inside-left": { xTranslationDirection: 1, yTranslationDirection: 0 },
  "inside-right": { xTranslationDirection: -1, yTranslationDirection: 0 },
  "inside-top": { xTranslationDirection: 0, yTranslationDirection: 1 },
  "inside-bottom": { xTranslationDirection: 0, yTranslationDirection: -1 },
  "inside-top-left": { xTranslationDirection: 1, yTranslationDirection: 1 },
  "inside-bottom-left": { xTranslationDirection: 1, yTranslationDirection: -1 },
  "inside-top-right": { xTranslationDirection: -1, yTranslationDirection: 1 },
  "inside-bottom-right": { xTranslationDirection: -1, yTranslationDirection: -1 }
};
var verticalCrossLineTranslationDirections = {
  top: { xTranslationDirection: 1, yTranslationDirection: 0 },
  bottom: { xTranslationDirection: -1, yTranslationDirection: 0 },
  left: { xTranslationDirection: 0, yTranslationDirection: -1 },
  right: { xTranslationDirection: 0, yTranslationDirection: 1 },
  "top-left": { xTranslationDirection: -1, yTranslationDirection: -1 },
  "top-right": { xTranslationDirection: -1, yTranslationDirection: 1 },
  "bottom-left": { xTranslationDirection: 1, yTranslationDirection: -1 },
  "bottom-right": { xTranslationDirection: 1, yTranslationDirection: 1 },
  inside: { xTranslationDirection: 0, yTranslationDirection: 0 },
  "inside-left": { xTranslationDirection: 0, yTranslationDirection: 1 },
  "inside-right": { xTranslationDirection: 0, yTranslationDirection: -1 },
  "inside-top": { xTranslationDirection: -1, yTranslationDirection: 0 },
  "inside-bottom": { xTranslationDirection: 1, yTranslationDirection: 0 },
  "inside-top-left": { xTranslationDirection: -1, yTranslationDirection: 1 },
  "inside-bottom-left": { xTranslationDirection: 1, yTranslationDirection: 1 },
  "inside-top-right": { xTranslationDirection: -1, yTranslationDirection: -1 },
  "inside-bottom-right": { xTranslationDirection: 1, yTranslationDirection: -1 }
};
function calculateLabelTranslation({
  yDirection,
  padding = 0,
  position = "top",
  bbox
}) {
  const crossLineTranslationDirections = yDirection ? horizontalCrosslineTranslationDirections : verticalCrossLineTranslationDirections;
  const { xTranslationDirection, yTranslationDirection } = crossLineTranslationDirections[position];
  const xTranslation = xTranslationDirection * (padding + bbox.width / 2);
  const yTranslation = yTranslationDirection * (padding + bbox.height / 2);
  return {
    xTranslation,
    yTranslation
  };
}
function calculateLabelChartPadding({
  yDirection,
  bbox,
  padding = 0,
  position = "top"
}) {
  const chartPadding = {};
  if (position.startsWith("inside"))
    return chartPadding;
  if (position === "top" && !yDirection) {
    chartPadding.top = padding + bbox.height;
  } else if (position === "bottom" && !yDirection) {
    chartPadding.bottom = padding + bbox.height;
  } else if (position === "left" && yDirection) {
    chartPadding.left = padding + bbox.width;
  } else if (position === "right" && yDirection) {
    chartPadding.right = padding + bbox.width;
  }
  return chartPadding;
}
var POSITION_TOP_COORDINATES = ({ direction, xEnd, yStart, yEnd }) => {
  if (direction === "y" /* Y */) {
    return { x: xEnd / 2, y: yStart };
  } else {
    return { x: xEnd, y: isNaN(yEnd) ? yStart : (yStart + yEnd) / 2 };
  }
};
var POSITION_LEFT_COORDINATES = ({ direction, xStart, xEnd, yStart, yEnd }) => {
  if (direction === "y" /* Y */) {
    return { x: xStart, y: isNaN(yEnd) ? yStart : (yStart + yEnd) / 2 };
  } else {
    return { x: xEnd / 2, y: yStart };
  }
};
var POSITION_RIGHT_COORDINATES = ({ direction, xEnd, yStart, yEnd }) => {
  if (direction === "y" /* Y */) {
    return { x: xEnd, y: isNaN(yEnd) ? yStart : (yStart + yEnd) / 2 };
  } else {
    return { x: xEnd / 2, y: isNaN(yEnd) ? yStart : yEnd };
  }
};
var POSITION_BOTTOM_COORDINATES = ({ direction, xStart, xEnd, yStart, yEnd }) => {
  if (direction === "y" /* Y */) {
    return { x: xEnd / 2, y: isNaN(yEnd) ? yStart : yEnd };
  } else {
    return { x: xStart, y: isNaN(yEnd) ? yStart : (yStart + yEnd) / 2 };
  }
};
var POSITION_INSIDE_COORDINATES = ({ xEnd, yStart, yEnd }) => {
  return { x: xEnd / 2, y: isNaN(yEnd) ? yStart : (yStart + yEnd) / 2 };
};
var POSITION_TOP_LEFT_COORDINATES = ({ direction, xStart, xEnd, yStart }) => {
  if (direction === "y" /* Y */) {
    return { x: xStart / 2, y: yStart };
  } else {
    return { x: xEnd, y: yStart };
  }
};
var POSITION_BOTTOM_LEFT_COORDINATES = ({ direction, xStart, yStart, yEnd }) => {
  if (direction === "y" /* Y */) {
    return { x: xStart, y: isNaN(yEnd) ? yStart : yEnd };
  } else {
    return { x: xStart, y: yStart };
  }
};
var POSITION_TOP_RIGHT_COORDINATES = ({ direction, xEnd, yStart, yEnd }) => {
  if (direction === "y" /* Y */) {
    return { x: xEnd, y: yStart };
  } else {
    return { x: xEnd, y: isNaN(yEnd) ? yStart : yEnd };
  }
};
var POSITION_BOTTOM_RIGHT_COORDINATES = ({ direction, xStart, xEnd, yStart, yEnd }) => {
  if (direction === "y" /* Y */) {
    return { x: xEnd, y: isNaN(yEnd) ? yStart : yEnd };
  } else {
    return { x: xStart, y: isNaN(yEnd) ? yStart : yEnd };
  }
};
var labelDirectionHandling = {
  top: { c: POSITION_TOP_COORDINATES },
  bottom: { c: POSITION_BOTTOM_COORDINATES },
  left: { c: POSITION_LEFT_COORDINATES },
  right: { c: POSITION_RIGHT_COORDINATES },
  "top-left": { c: POSITION_TOP_LEFT_COORDINATES },
  "top-right": { c: POSITION_TOP_RIGHT_COORDINATES },
  "bottom-left": { c: POSITION_BOTTOM_LEFT_COORDINATES },
  "bottom-right": { c: POSITION_BOTTOM_RIGHT_COORDINATES },
  inside: { c: POSITION_INSIDE_COORDINATES },
  "inside-left": { c: POSITION_LEFT_COORDINATES },
  "inside-right": { c: POSITION_RIGHT_COORDINATES },
  "inside-top": { c: POSITION_TOP_COORDINATES },
  "inside-bottom": { c: POSITION_BOTTOM_COORDINATES },
  "inside-top-left": { c: POSITION_TOP_LEFT_COORDINATES },
  "inside-bottom-left": { c: POSITION_BOTTOM_LEFT_COORDINATES },
  "inside-top-right": { c: POSITION_TOP_RIGHT_COORDINATES },
  "inside-bottom-right": { c: POSITION_BOTTOM_RIGHT_COORDINATES }
};

// packages/ag-charts-community/src/chart/crossline/cartesianCrossLine.ts
var CROSSLINE_LABEL_POSITION = UNION(
  [
    "top",
    "left",
    "right",
    "bottom",
    "top-left",
    "top-right",
    "bottom-left",
    "bottom-right",
    "inside",
    "inside-left",
    "inside-right",
    "inside-top",
    "inside-bottom",
    "inside-top-left",
    "inside-bottom-left",
    "inside-top-right",
    "inside-bottom-right"
  ],
  "crossLine label position"
);
var CartesianCrossLineLabel = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.fontSize = 14;
    this.fontFamily = "Verdana, sans-serif";
    this.padding = 5;
    this.color = "rgba(87, 87, 87, 1)";
  }
};
__decorateClass([
  Validate(BOOLEAN, { optional: true })
], CartesianCrossLineLabel.prototype, "enabled", 2);
__decorateClass([
  Validate(STRING, { optional: true })
], CartesianCrossLineLabel.prototype, "text", 2);
__decorateClass([
  Validate(FONT_STYLE, { optional: true })
], CartesianCrossLineLabel.prototype, "fontStyle", 2);
__decorateClass([
  Validate(FONT_WEIGHT, { optional: true })
], CartesianCrossLineLabel.prototype, "fontWeight", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], CartesianCrossLineLabel.prototype, "fontSize", 2);
__decorateClass([
  Validate(STRING)
], CartesianCrossLineLabel.prototype, "fontFamily", 2);
__decorateClass([
  Validate(NUMBER)
], CartesianCrossLineLabel.prototype, "padding", 2);
__decorateClass([
  Validate(COLOR_STRING, { optional: true })
], CartesianCrossLineLabel.prototype, "color", 2);
__decorateClass([
  Validate(CROSSLINE_LABEL_POSITION, { optional: true })
], CartesianCrossLineLabel.prototype, "position", 2);
__decorateClass([
  Validate(NUMBER, { optional: true })
], CartesianCrossLineLabel.prototype, "rotation", 2);
__decorateClass([
  Validate(BOOLEAN, { optional: true })
], CartesianCrossLineLabel.prototype, "parallel", 2);
var CartesianCrossLine = class extends BaseProperties {
  constructor() {
    super();
    this.id = createId(this);
    this.label = new CartesianCrossLineLabel();
    this.scale = void 0;
    this.clippedRange = [-Infinity, Infinity];
    this.gridLength = 0;
    this.sideFlag = -1;
    this.parallelFlipRotation = 0;
    this.regularFlipRotation = 0;
    this.direction = "x" /* X */;
    this.rangeGroup = new Group({ name: this.id });
    this.lineGroup = new Group({ name: this.id });
    this.labelGroup = new Group({ name: this.id });
    this.crossLineRange = new Range();
    this.crossLineLabel = new TransformableText();
    this.labelPoint = void 0;
    this.data = [];
    this.startLine = false;
    this.endLine = false;
    this.isRange = false;
    this._isRange = void 0;
    this.lineGroup.append(this.crossLineRange);
    this.labelGroup.append(this.crossLineLabel);
    this.crossLineRange.pointerEvents = 1 /* None */;
  }
  update(visible) {
    const { enabled, data, type, value, range: range2, scale } = this;
    if (!type || !scale || !enabled || !visible || !validateCrossLineValues(type, value, range2, scale) || data.length === 0) {
      this.rangeGroup.visible = false;
      this.lineGroup.visible = false;
      this.labelGroup.visible = false;
      return;
    }
    this.rangeGroup.visible = visible;
    this.lineGroup.visible = visible;
    this.labelGroup.visible = visible;
    this.updateNodes();
    const { isRange } = this;
    if (isRange !== this._isRange) {
      if (isRange) {
        this.rangeGroup.appendChild(this.crossLineRange);
      } else {
        this.lineGroup.appendChild(this.crossLineRange);
      }
    }
    this._isRange = isRange;
  }
  calculateLayout(visible, reversedAxis) {
    if (!visible)
      return;
    const {
      scale,
      gridLength,
      sideFlag,
      direction,
      label: { position = "top" },
      clippedRange,
      strokeWidth = 0
    } = this;
    this.data = [];
    if (!scale)
      return;
    const bandwidth = scale.bandwidth ?? 0;
    const step = scale.step ?? 0;
    const padding = (reversedAxis ? -1 : 1) * (scale instanceof BandScale ? (step - bandwidth) / 2 : 0);
    const [xStart, xEnd] = [0, sideFlag * gridLength];
    let [yStart, yEnd] = this.getRange();
    const ordinalTimeScalePadding = yEnd === void 0 && OrdinalTimeScale.is(scale) ? bandwidth / 2 + padding : 0;
    let [clampedYStart, clampedYEnd] = [
      Number(scale.convert(yStart, true)) - padding + ordinalTimeScalePadding,
      scale.convert(yEnd, true) + bandwidth + padding
    ];
    clampedYStart = clampArray(clampedYStart, clippedRange);
    clampedYEnd = clampArray(clampedYEnd, clippedRange);
    [yStart, yEnd] = [Number(scale.convert(yStart)) + ordinalTimeScalePadding, scale.convert(yEnd) + bandwidth];
    const validRange = (yStart === clampedYStart || yEnd === clampedYEnd || clampedYStart !== clampedYEnd) && Math.abs(clampedYEnd - clampedYStart) > 0;
    if (validRange && clampedYStart > clampedYEnd) {
      [clampedYStart, clampedYEnd] = [clampedYEnd, clampedYStart];
      [yStart, yEnd] = [yEnd, yStart];
    }
    if (yStart - padding >= clampedYStart)
      yStart -= padding;
    if (yEnd + padding <= clampedYEnd)
      yEnd += padding;
    this.isRange = validRange;
    this.startLine = strokeWidth > 0 && yStart >= clampedYStart && yStart <= clampedYStart + padding;
    this.endLine = strokeWidth > 0 && yEnd >= clampedYEnd - bandwidth - padding && yEnd <= clampedYEnd;
    if (!validRange && !this.startLine && !this.endLine)
      return;
    this.data = [clampedYStart, clampedYEnd];
    if (!this.label.enabled)
      return;
    const { c = POSITION_TOP_COORDINATES } = labelDirectionHandling[position] ?? {};
    const { x: labelX, y: labelY } = c({
      direction,
      xStart,
      xEnd,
      yStart: clampedYStart,
      yEnd: clampedYEnd
    });
    this.labelPoint = {
      x: labelX,
      y: labelY
    };
  }
  updateNodes() {
    this.updateRangeNode();
    if (this.label.enabled) {
      this.updateLabel();
      this.positionLabel();
    }
  }
  updateRangeNode() {
    const {
      crossLineRange,
      sideFlag,
      gridLength,
      data,
      startLine,
      endLine,
      isRange,
      fill,
      fillOpacity,
      stroke,
      strokeWidth,
      lineDash
    } = this;
    crossLineRange.x1 = 0;
    crossLineRange.x2 = sideFlag * gridLength;
    crossLineRange.y1 = data[0];
    crossLineRange.y2 = data[1];
    crossLineRange.startLine = startLine;
    crossLineRange.endLine = endLine;
    crossLineRange.isRange = isRange;
    crossLineRange.fill = fill;
    crossLineRange.fillOpacity = fillOpacity ?? 1;
    crossLineRange.stroke = stroke;
    crossLineRange.strokeWidth = strokeWidth ?? 1;
    crossLineRange.strokeOpacity = this.strokeOpacity ?? 1;
    crossLineRange.lineDash = lineDash;
  }
  updateLabel() {
    const { crossLineLabel, label } = this;
    if (!label.text)
      return;
    crossLineLabel.fontStyle = label.fontStyle;
    crossLineLabel.fontWeight = label.fontWeight;
    crossLineLabel.fontSize = label.fontSize;
    crossLineLabel.fontFamily = label.fontFamily;
    crossLineLabel.fill = label.color;
    crossLineLabel.text = label.text;
  }
  positionLabel() {
    const {
      crossLineLabel,
      labelPoint: { x = void 0, y = void 0 } = {},
      label: { parallel, rotation, position = "top", padding = 0 },
      direction,
      parallelFlipRotation,
      regularFlipRotation
    } = this;
    if (x === void 0 || y === void 0)
      return;
    const { defaultRotation, configuredRotation } = calculateLabelRotation({
      rotation,
      parallel,
      regularFlipRotation,
      parallelFlipRotation
    });
    crossLineLabel.rotation = defaultRotation + configuredRotation;
    crossLineLabel.textBaseline = "middle";
    crossLineLabel.textAlign = "center";
    const bbox = crossLineLabel.getBBox();
    if (!bbox)
      return;
    const yDirection = direction === "y" /* Y */;
    const { xTranslation, yTranslation } = calculateLabelTranslation({
      yDirection,
      padding,
      position,
      bbox
    });
    crossLineLabel.translationX = x + xTranslation;
    crossLineLabel.translationY = y + yTranslation;
  }
  getRange() {
    const { value, range: range2, scale } = this;
    const isContinuous2 = ContinuousScale.is(scale) || OrdinalTimeScale.is(scale);
    const start2 = range2?.[0] ?? value;
    let end2 = range2?.[1];
    if (!isContinuous2 && end2 === void 0) {
      end2 = start2;
    }
    if (isContinuous2 && start2 === end2) {
      end2 = void 0;
    }
    return [start2, end2];
  }
  computeLabelBBox() {
    const { label } = this;
    if (!label.enabled)
      return;
    const tempText = new TransformableText();
    tempText.fontFamily = label.fontFamily;
    tempText.fontSize = label.fontSize;
    tempText.fontStyle = label.fontStyle;
    tempText.fontWeight = label.fontWeight;
    tempText.text = label.text;
    const {
      labelPoint: { x = void 0, y = void 0 } = {},
      label: { parallel, rotation, position = "top", padding = 0 },
      direction,
      parallelFlipRotation,
      regularFlipRotation
    } = this;
    if (x === void 0 || y === void 0)
      return;
    const { configuredRotation } = calculateLabelRotation({
      rotation,
      parallel,
      regularFlipRotation,
      parallelFlipRotation
    });
    tempText.rotation = configuredRotation;
    tempText.textBaseline = "middle";
    tempText.textAlign = "center";
    const bbox = tempText.getBBox();
    if (!bbox)
      return;
    const yDirection = direction === "y" /* Y */;
    const { xTranslation, yTranslation } = calculateLabelTranslation({
      yDirection,
      padding,
      position,
      bbox
    });
    tempText.x = x + xTranslation;
    tempText.y = y + yTranslation;
    return tempText.getBBox();
  }
  calculatePadding(padding) {
    const {
      isRange,
      startLine,
      endLine,
      direction,
      label: { padding: labelPadding = 0, position = "top" }
    } = this;
    if (!isRange && !startLine && !endLine)
      return;
    const crossLineLabelBBox = this.computeLabelBBox();
    if (crossLineLabelBBox?.x == null || crossLineLabelBBox?.y == null)
      return;
    const chartPadding = calculateLabelChartPadding({
      yDirection: direction === "y" /* Y */,
      padding: labelPadding,
      position,
      bbox: crossLineLabelBBox
    });
    padding.left = Math.max(padding.left ?? 0, chartPadding.left ?? 0);
    padding.right = Math.max(padding.right ?? 0, chartPadding.right ?? 0);
    padding.top = Math.max(padding.top ?? 0, chartPadding.top ?? 0);
    padding.bottom = Math.max(padding.bottom ?? 0, chartPadding.bottom ?? 0);
  }
};
CartesianCrossLine.className = "CrossLine";
__decorateClass([
  Validate(BOOLEAN, { optional: true })
], CartesianCrossLine.prototype, "enabled", 2);
__decorateClass([
  Validate(UNION(["range", "line"], "a crossLine type"), { optional: true })
], CartesianCrossLine.prototype, "type", 2);
__decorateClass([
  Validate(AND(MATCHING_CROSSLINE_TYPE("range"), ARRAY.restrict({ length: 2 })), {
    optional: true
  })
], CartesianCrossLine.prototype, "range", 2);
__decorateClass([
  Validate(MATCHING_CROSSLINE_TYPE("value"), { optional: true })
], CartesianCrossLine.prototype, "value", 2);
__decorateClass([
  Validate(COLOR_STRING, { optional: true })
], CartesianCrossLine.prototype, "fill", 2);
__decorateClass([
  Validate(RATIO, { optional: true })
], CartesianCrossLine.prototype, "fillOpacity", 2);
__decorateClass([
  Validate(COLOR_STRING, { optional: true })
], CartesianCrossLine.prototype, "stroke", 2);
__decorateClass([
  Validate(NUMBER, { optional: true })
], CartesianCrossLine.prototype, "strokeWidth", 2);
__decorateClass([
  Validate(RATIO, { optional: true })
], CartesianCrossLine.prototype, "strokeOpacity", 2);
__decorateClass([
  Validate(LINE_DASH, { optional: true })
], CartesianCrossLine.prototype, "lineDash", 2);
__decorateClass([
  Validate(OBJECT)
], CartesianCrossLine.prototype, "label", 2);

// packages/ag-charts-community/src/chart/axis/axisGridLine.ts
var GRID_STYLE_KEYS = ["stroke", "lineDash"];
var GRID_STYLE = ARRAY_OF(
  (value) => isObject(value) && Object.keys(value).every((key) => GRID_STYLE_KEYS.includes(key)),
  "objects with gridline style properties such as 'stroke' or 'lineDash'"
);
var AxisGridLine = class {
  constructor() {
    this.enabled = true;
    this.width = 1;
    this.style = [
      {
        stroke: void 0,
        lineDash: []
      }
    ];
  }
};
__decorateClass([
  Validate(BOOLEAN)
], AxisGridLine.prototype, "enabled", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], AxisGridLine.prototype, "width", 2);
__decorateClass([
  Validate(GRID_STYLE)
], AxisGridLine.prototype, "style", 2);

// packages/ag-charts-community/src/util/default.ts
function Default(defaultValue, replaces = [void 0]) {
  return addTransformToInstanceProperty((_, __, v) => {
    if (replaces.includes(v)) {
      return isFunction(defaultValue) ? defaultValue(v) : defaultValue;
    }
    return v;
  });
}

// packages/ag-charts-community/src/chart/axis/axisInterval.ts
var TICK_INTERVAL = predicateWithMessage(
  (value) => isFiniteNumber(value) && value > 0 || value instanceof TimeInterval,
  `a non-zero positive Number value or, for a time axis, a Time Interval such as 'agCharts.time.month'`
);
var AxisInterval = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.minSpacing = NaN;
    this.maxSpacing = NaN;
  }
};
__decorateClass([
  Validate(TICK_INTERVAL, { optional: true })
], AxisInterval.prototype, "step", 2);
__decorateClass([
  Validate(ARRAY, { optional: true })
], AxisInterval.prototype, "values", 2);
__decorateClass([
  Validate(MIN_SPACING),
  Default(NaN)
], AxisInterval.prototype, "minSpacing", 2);
__decorateClass([
  Validate(MAX_SPACING),
  Default(NaN)
], AxisInterval.prototype, "maxSpacing", 2);

// packages/ag-charts-community/src/chart/axis/axisLabel.ts
var AxisLabel = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.enabled = true;
    this.fontSize = 12;
    this.fontFamily = "Verdana, sans-serif";
    this.spacing = 5;
    this.minSpacing = NaN;
    this.color = "#575757";
    this.avoidCollisions = true;
    this.mirrored = false;
    this.parallel = false;
  }
  /**
   * The side of the axis line to position the labels on.
   * -1 = left (default)
   * 1 = right
   */
  getSideFlag() {
    return this.mirrored ? 1 : -1;
  }
};
__decorateClass([
  Validate(BOOLEAN)
], AxisLabel.prototype, "enabled", 2);
__decorateClass([
  Validate(FONT_STYLE, { optional: true })
], AxisLabel.prototype, "fontStyle", 2);
__decorateClass([
  Validate(FONT_WEIGHT, { optional: true })
], AxisLabel.prototype, "fontWeight", 2);
__decorateClass([
  Validate(NUMBER.restrict({ min: 1 }))
], AxisLabel.prototype, "fontSize", 2);
__decorateClass([
  Validate(STRING)
], AxisLabel.prototype, "fontFamily", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], AxisLabel.prototype, "spacing", 2);
__decorateClass([
  Validate(NUMBER_OR_NAN),
  Default(NaN)
], AxisLabel.prototype, "minSpacing", 2);
__decorateClass([
  Validate(COLOR_STRING, { optional: true })
], AxisLabel.prototype, "color", 2);
__decorateClass([
  Validate(NUMBER, { optional: true })
], AxisLabel.prototype, "rotation", 2);
__decorateClass([
  Validate(BOOLEAN)
], AxisLabel.prototype, "avoidCollisions", 2);
__decorateClass([
  Validate(BOOLEAN)
], AxisLabel.prototype, "mirrored", 2);
__decorateClass([
  Validate(BOOLEAN)
], AxisLabel.prototype, "parallel", 2);
__decorateClass([
  Validate(FUNCTION, { optional: true })
], AxisLabel.prototype, "itemStyler", 2);
__decorateClass([
  Validate(FUNCTION, { optional: true })
], AxisLabel.prototype, "formatter", 2);
__decorateClass([
  Validate(STRING, { optional: true })
], AxisLabel.prototype, "format", 2);

// packages/ag-charts-community/src/chart/axis/axisLine.ts
var AxisLine = class {
  constructor() {
    this.enabled = true;
    this.width = 1;
    this.stroke = void 0;
  }
};
__decorateClass([
  Validate(BOOLEAN)
], AxisLine.prototype, "enabled", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], AxisLine.prototype, "width", 2);
__decorateClass([
  Validate(COLOR_STRING, { optional: true })
], AxisLine.prototype, "stroke", 2);

// packages/ag-charts-community/src/chart/axis/axisTick.ts
var AxisTick = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.enabled = true;
    this.width = 1;
    this.size = 6;
  }
};
__decorateClass([
  Validate(BOOLEAN)
], AxisTick.prototype, "enabled", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], AxisTick.prototype, "width", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], AxisTick.prototype, "size", 2);
__decorateClass([
  Validate(COLOR_STRING, { optional: true })
], AxisTick.prototype, "stroke", 2);

// packages/ag-charts-community/src/chart/axis/axisTitle.ts
var AxisTitle = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.caption = new Caption();
    this.enabled = false;
    this.spacing = Caption.SMALL_PADDING;
    this.fontSize = 10;
    this.fontFamily = "sans-serif";
    this.wrapping = "always";
  }
};
__decorateClass([
  Validate(BOOLEAN)
], AxisTitle.prototype, "enabled", 2);
__decorateClass([
  Validate(STRING, { optional: true })
], AxisTitle.prototype, "text", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER, { optional: true })
], AxisTitle.prototype, "spacing", 2);
__decorateClass([
  Validate(FONT_STYLE, { optional: true })
], AxisTitle.prototype, "fontStyle", 2);
__decorateClass([
  Validate(FONT_WEIGHT, { optional: true })
], AxisTitle.prototype, "fontWeight", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], AxisTitle.prototype, "fontSize", 2);
__decorateClass([
  Validate(STRING)
], AxisTitle.prototype, "fontFamily", 2);
__decorateClass([
  Validate(COLOR_STRING, { optional: true })
], AxisTitle.prototype, "color", 2);
__decorateClass([
  Validate(TEXT_WRAP)
], AxisTitle.prototype, "wrapping", 2);
__decorateClass([
  Validate(FUNCTION, { optional: true })
], AxisTitle.prototype, "formatter", 2);

// packages/ag-charts-community/src/chart/axis/axisUtil.ts
function prepareAxisAnimationContext(axis) {
  const [requestedRangeMin, requestedRangeMax] = findMinMax(axis.range);
  const min = Math.floor(requestedRangeMin);
  const max = Math.ceil(requestedRangeMax);
  return { min, max, visible: min !== max };
}
var fullCircle = Math.PI * 2;
var halfCircle = fullCircle / 2;
function normaliseEndRotation(start2, end2) {
  const directDistance = Math.abs(end2 - start2);
  if (directDistance < halfCircle) {
    return end2;
  } else if (start2 > end2) {
    return end2 + fullCircle;
  }
  return end2 - fullCircle;
}
function prepareAxisAnimationFunctions(ctx) {
  const outOfBounds = (y, range2) => {
    const [min = ctx.min, max = ctx.max] = findMinMax(range2 ?? []);
    return y < min || y > max;
  };
  const tick = {
    fromFn(node, datum, status) {
      let y = node.y1 + node.translationY;
      let opacity = node.opacity;
      if (status === "added" || outOfBounds(node.datum.translationY, node.datum.range)) {
        y = datum.translationY;
        opacity = 0;
      }
      return { y: 0, translationY: y, opacity, phase: NODE_UPDATE_STATE_TO_PHASE_MAPPING[status] };
    },
    toFn(_node, datum, status) {
      const y = datum.translationY;
      let opacity = 1;
      if (status === "removed") {
        opacity = 0;
      }
      return {
        y: 0,
        translationY: y,
        opacity,
        finish: {
          // Set explicit y after animation so it's pixel aligned
          y,
          translationY: 0
        }
      };
    },
    applyFn(node, props) {
      node.setProperties(props);
      node.visible = !outOfBounds(node.y);
    }
  };
  const label = {
    fromFn(node, newDatum, status) {
      const datum = node.previousDatum ?? newDatum;
      const x = datum.x;
      const y = datum.y;
      const rotationCenterX = datum.rotationCenterX;
      let translationY = Math.round(node.translationY);
      let rotation = datum.rotation;
      let opacity = node.opacity;
      if (status === "removed" || outOfBounds(datum.y, datum.range)) {
        rotation = newDatum.rotation;
      } else if (status === "added" || outOfBounds(node.datum.y, node.datum.range)) {
        translationY = Math.round(datum.translationY);
        opacity = 0;
        rotation = newDatum.rotation;
      }
      return {
        x,
        y,
        rotationCenterX,
        translationY,
        rotation,
        opacity,
        phase: NODE_UPDATE_STATE_TO_PHASE_MAPPING[status]
      };
    },
    toFn(node, datum, status) {
      const x = datum.x;
      const y = datum.y;
      const rotationCenterX = datum.rotationCenterX;
      const translationY = Math.round(datum.translationY);
      let rotation = 0;
      let opacity = 1;
      if (status === "added") {
        rotation = datum.rotation;
      } else if (status === "removed") {
        opacity = 0;
        rotation = datum.rotation;
      } else {
        rotation = normaliseEndRotation(node.previousDatum?.rotation ?? datum.rotation, datum.rotation);
      }
      return { x, y, rotationCenterX, translationY, rotation, opacity, finish: { rotation: datum.rotation } };
    }
  };
  const line = {
    fromFn(node, datum) {
      return {
        ...node.previousDatum ?? datum,
        phase: NODE_UPDATE_STATE_TO_PHASE_MAPPING["updated"]
      };
    },
    toFn(_node, datum) {
      return { ...datum };
    }
  };
  const group = {
    fromFn(node, _datum) {
      const { rotation, translationX, translationY } = node;
      return {
        rotation,
        translationX,
        translationY,
        phase: NODE_UPDATE_STATE_TO_PHASE_MAPPING["updated"]
      };
    },
    toFn(_node, datum) {
      const { rotation, translationX, translationY } = datum;
      return {
        rotation,
        translationX,
        translationY
      };
    }
  };
  return { tick, line, label, group };
}
function resetAxisGroupFn() {
  return (_node, datum) => {
    return {
      rotation: datum.rotation,
      rotationCenterX: datum.rotationCenterX,
      rotationCenterY: datum.rotationCenterY,
      translationX: datum.translationX,
      translationY: datum.translationY
    };
  };
}
function resetAxisSelectionFn(ctx) {
  const { visible: rangeVisible, min, max } = ctx;
  return (_node, datum) => {
    const y = datum.translationY;
    const visible = rangeVisible && y >= min && y <= max;
    return {
      y,
      translationY: 0,
      opacity: 1,
      visible
    };
  };
}
function resetAxisLabelSelectionFn() {
  return (_node, datum) => {
    return {
      x: datum.x,
      y: datum.y,
      translationY: datum.translationY,
      rotation: datum.rotation,
      rotationCenterX: datum.rotationCenterX
    };
  };
}
function resetAxisLineSelectionFn() {
  return (_node, datum) => {
    return { ...datum };
  };
}

// packages/ag-charts-community/src/chart/axis/axis.ts
var TranslatableLine = class extends Translatable(Line) {
};
var _Axis = class _Axis {
  constructor(moduleCtx, scale) {
    this.moduleCtx = moduleCtx;
    this.scale = scale;
    this.id = createId(this);
    this.nice = true;
    this.reverse = false;
    this.keys = [];
    this.interval = new AxisInterval();
    this.dataDomain = { domain: [], clipped: false };
    this.layoutConstraints = {
      stacked: true,
      align: "start",
      width: 100,
      unit: "percent"
    };
    this.boundSeries = [];
    this.includeInvisibleDomains = false;
    this.interactionEnabled = true;
    this.axisGroup = new TransformableGroup({ name: `${this.id}-axis` });
    // Order is important to apply the correct z-index.
    this.tickLineGroup = this.axisGroup.appendChild(
      new Group({ name: `${this.id}-Axis-tick-lines`, zIndex: 0 /* TickLines */ })
    );
    this.tickLabelGroup = this.axisGroup.appendChild(
      new Group({ name: `${this.id}-Axis-tick-labels`, zIndex: 2 /* TickLabels */ })
    );
    this.labelGroup = new Group({
      name: `${this.id}-Labels`,
      zIndex: 9 /* SERIES_ANNOTATION */
    });
    this.gridGroup = new TransformableGroup({ name: `${this.id}-Axis-grid`, zIndex: 1 /* AXIS_GRID */ });
    this.gridLineGroup = this.gridGroup.appendChild(new Group({ name: `${this.id}-gridLines` }));
    this.crossLineRangeGroup = new TransformableGroup({
      name: `${this.id}-CrossLines-Range`,
      zIndex: 4 /* SERIES_CROSSLINE_RANGE */
    });
    this.crossLineLineGroup = new TransformableGroup({
      name: `${this.id}-CrossLines-Line`,
      zIndex: 8 /* SERIES_CROSSLINE_LINE */
    });
    this.crossLineLabelGroup = new TransformableGroup({
      name: `${this.id}-CrossLines-Label`,
      zIndex: 13 /* SERIES_LABEL */
    });
    this.tickLineGroupSelection = Selection.select(
      this.tickLineGroup,
      TranslatableLine,
      false
    );
    this.tickLabelGroupSelection = Selection.select(
      this.tickLabelGroup,
      TransformableText,
      false
    );
    this.gridLineGroupSelection = Selection.select(
      this.gridLineGroup,
      TranslatableLine,
      false
    );
    this._crossLines = [];
    this.line = new AxisLine();
    this.tick = new AxisTick();
    this.gridLine = new AxisGridLine();
    this.label = this.createLabel();
    this.defaultTickMinSpacing = _Axis.defaultTickMinSpacing;
    this.translation = { x: 0, y: 0 };
    this.rotation = 0;
    // axis rotation angle in degrees
    this.layout = {
      label: {
        fractionDigits: 0,
        spacing: this.label.spacing,
        format: this.label.format
      }
    };
    this.axisContext = void 0;
    this.labelFormatter = void 0;
    this.datumFormatter = void 0;
    this.scaleFormatterParams = void 0;
    this.destroyFns = [];
    this.range = [0, 1];
    this.visibleRange = [0, 1];
    this.title = new AxisTitle();
    this.gridLength = 0;
    /**
     * The distance between the grid ticks and the axis ticks.
     */
    this.gridPadding = 0;
    /**
     * Is used to avoid collisions between axis labels and series.
     */
    this.seriesAreaPadding = 0;
    this.animatable = true;
    this._scaleNiceDomainInputDomain = void 0;
    this._scaleNiceDomainRangeExtent = NaN;
    this.moduleMap = new ModuleMap();
    this.range = this.scale.range.slice();
    this.crossLines.forEach((crossLine) => this.initCrossLine(crossLine));
  }
  get type() {
    return this.constructor.type ?? "";
  }
  get labelNodes() {
    return this.tickLabelGroupSelection.nodes();
  }
  set crossLines(value) {
    const { CrossLineConstructor } = this.constructor;
    this._crossLines.forEach((crossLine) => this.detachCrossLine(crossLine));
    this._crossLines = value.map((crossLine) => {
      const instance = new CrossLineConstructor();
      instance.set(crossLine);
      return instance;
    });
    this._crossLines.forEach((crossLine) => {
      this.attachCrossLine(crossLine);
      this.initCrossLine(crossLine);
    });
  }
  get crossLines() {
    return this._crossLines;
  }
  resetAnimation(_phase) {
  }
  attachCrossLine(crossLine) {
    this.crossLineRangeGroup.appendChild(crossLine.rangeGroup);
    this.crossLineLineGroup.appendChild(crossLine.lineGroup);
    this.crossLineLabelGroup.appendChild(crossLine.labelGroup);
  }
  detachCrossLine(crossLine) {
    this.crossLineRangeGroup.removeChild(crossLine.rangeGroup);
    this.crossLineLineGroup.removeChild(crossLine.lineGroup);
    this.crossLineLabelGroup.removeChild(crossLine.labelGroup);
  }
  destroy() {
    this.moduleMap.destroy();
    this.destroyFns.forEach((f) => f());
  }
  updateScale() {
    const { range: rr, visibleRange: vr, scale } = this;
    const span = (rr[1] - rr[0]) / (vr[1] - vr[0]);
    const shift = span * vr[0];
    const start2 = rr[0] - shift;
    scale.range = [start2, start2 + span];
    this.crossLines.forEach((crossLine) => {
      crossLine.clippedRange = [rr[0], rr[1]];
    });
  }
  setCrossLinesVisible(visible) {
    this.crossLineRangeGroup.visible = visible;
    this.crossLineLineGroup.visible = visible;
    this.crossLineLabelGroup.visible = visible;
  }
  attachAxis(groups) {
    groups.gridNode.appendChild(this.gridGroup);
    groups.axisNode.appendChild(this.axisGroup);
    groups.labelNode.appendChild(this.labelGroup);
    groups.crossLineRangeNode.appendChild(this.crossLineRangeGroup);
    groups.crossLineLineNode.appendChild(this.crossLineLineGroup);
    groups.crossLineLabelNode.appendChild(this.crossLineLabelGroup);
  }
  detachAxis(groups) {
    groups.gridNode.removeChild(this.gridGroup);
    groups.axisNode.removeChild(this.axisGroup);
    groups.labelNode.removeChild(this.labelGroup);
    groups.crossLineRangeNode.removeChild(this.crossLineRangeGroup);
    groups.crossLineLineNode.removeChild(this.crossLineLineGroup);
    groups.crossLineLabelNode.removeChild(this.crossLineLabelGroup);
  }
  attachLabel(axisLabelNode) {
    this.labelGroup.append(axisLabelNode);
  }
  /**
   * Checks if a point or an object is in range.
   * @param value A point (or object's starting point).
   * @param tolerance Expands the range on both ends by this amount.
   */
  inRange(value, tolerance = 0) {
    const [min, max] = findMinMax(this.range);
    return value >= min - tolerance && value <= max + tolerance;
  }
  defaultDatumFormatter(datum, fractionDigits) {
    return formatValue(datum, fractionDigits + 1);
  }
  defaultLabelFormatter(datum, fractionDigits) {
    return formatValue(datum, fractionDigits);
  }
  onGridLengthChange(value, prevValue) {
    if (prevValue ^ value) {
      this.onGridVisibilityChange();
    }
    this.crossLines.forEach((crossLine) => this.initCrossLine(crossLine));
  }
  onGridVisibilityChange() {
    this.gridLineGroupSelection.clear();
  }
  createLabel() {
    return new AxisLabel();
  }
  /**
   * Creates/removes/updates the scene graph nodes that constitute the axis.
   */
  update() {
    this.updatePosition();
    this.updateSelections();
    this.tickLineGroup.visible = this.tick.enabled;
    this.gridLineGroup.visible = this.gridLine.enabled;
    this.tickLabelGroup.visible = this.label.enabled;
    this.updateLabels();
    this.updateGridLines();
    this.updateTickLines();
    this.updateCrossLines();
  }
  getAxisLineCoordinates() {
    const [min, max] = findMinMax(this.range);
    return { x: 0, y1: min, y2: max };
  }
  getLabelStyles(params, additionalStyles) {
    const { label } = this;
    const defaultStyle = {
      color: label.color,
      spacing: label.spacing,
      fontFamily: label.fontFamily,
      fontSize: label.fontSize,
      fontStyle: label.fontStyle,
      fontWeight: label.fontWeight
    };
    let stylerOutput;
    if (label.itemStyler) {
      stylerOutput = this.moduleCtx.callbackCache.call(label.itemStyler, {
        ...params,
        ...defaultStyle
      });
    }
    const {
      color: fill,
      fontFamily,
      fontSize,
      fontStyle,
      fontWeight,
      spacing
    } = mergeDefaults(stylerOutput, additionalStyles, defaultStyle);
    return { fill, fontFamily, fontSize, fontStyle, fontWeight, spacing };
  }
  getTickSize() {
    return this.tick.enabled ? this.tick.size : 0;
  }
  setTitleProps(caption, params) {
    const { title } = this;
    if (!title.enabled) {
      caption.enabled = false;
      caption.node.visible = false;
      return;
    }
    caption.enabled = true;
    caption.color = title.color;
    caption.fontFamily = title.fontFamily;
    caption.fontSize = title.fontSize;
    caption.fontStyle = title.fontStyle;
    caption.fontWeight = title.fontWeight;
    caption.wrapping = title.wrapping;
    const titleNode = caption.node;
    const padding = (title.spacing ?? 0) + params.spacing;
    const sideFlag = this.label.getSideFlag();
    const parallelFlipRotation = normalizeAngle360(this.rotation);
    const titleRotationFlag = sideFlag === -1 && parallelFlipRotation > Math.PI && parallelFlipRotation < Math.PI * 2 ? -1 : 1;
    const rotation = titleRotationFlag * sideFlag * Math.PI / 2;
    const textBaseline = titleRotationFlag === 1 ? "bottom" : "top";
    const { range: range2 } = this;
    const x = Math.floor(titleRotationFlag * sideFlag * (range2[0] + range2[1]) / 2);
    const y = sideFlag === -1 ? Math.floor(titleRotationFlag * -padding) : Math.floor(-padding);
    const { callbackCache } = this.moduleCtx;
    const { formatter = (p) => p.defaultValue } = title;
    const text = callbackCache.call(formatter, this.getTitleFormatterParams());
    caption.text = text;
    titleNode.setProperties({ visible: true, text, textBaseline, x, y, rotation });
  }
  processData() {
    const { includeInvisibleDomains, boundSeries, direction } = this;
    const visibleSeries = includeInvisibleDomains ? boundSeries : boundSeries.filter((s) => s.isEnabled());
    const domains = visibleSeries.map((series) => series.getDomain(direction));
    this.setDomains(...domains);
  }
  setDomains(...domains) {
    let domain;
    let animatable;
    if (domains.length > 0) {
      ({ domain, animatable } = this.scale.normalizeDomains(...domains));
    } else {
      domain = [];
      animatable = true;
    }
    this.dataDomain = this.normaliseDataDomain(domain);
    if (this.reverse) {
      this.dataDomain.domain.reverse();
    }
    this.animatable = animatable;
  }
  calculateLayout(initialPrimaryTickCount) {
    const { scale, label, visibleRange, nice } = this;
    this.updateScale();
    const rangeExtent = findRangeExtent(this.range);
    const domain = this.dataDomain.domain;
    let tickLayoutDomain;
    if (visibleRange[0] === 0 && visibleRange[1] === 1) {
      tickLayoutDomain = void 0;
    } else if (!nice) {
      tickLayoutDomain = domain;
    } else if (this._scaleNiceDomainInputDomain === domain && this._scaleNiceDomainRangeExtent === rangeExtent) {
      tickLayoutDomain = this.scale.domain;
    } else {
      tickLayoutDomain = this.calculateTickLayout(domain, 0 /* TickAndDomain */, [0, 1]).niceDomain;
    }
    let niceMode;
    if (!nice) {
      niceMode = 2 /* Off */;
    } else if (tickLayoutDomain == null) {
      niceMode = 0 /* TickAndDomain */;
    } else {
      niceMode = 1 /* TicksOnly */;
    }
    const { niceDomain, primaryTickCount, ticks, tickDomain, fractionDigits, bbox } = this.calculateTickLayout(
      tickLayoutDomain ?? domain,
      niceMode,
      visibleRange,
      initialPrimaryTickCount
    );
    this.scale.domain = niceDomain;
    this._scaleNiceDomainInputDomain = nice ? domain : void 0;
    this._scaleNiceDomainRangeExtent = nice ? rangeExtent : NaN;
    const specifier = label.format;
    this.labelFormatter = scale.tickFormatter({ domain: tickDomain, specifier, ticks, fractionDigits }) ?? ((value) => this.defaultLabelFormatter(value, fractionDigits));
    this.datumFormatter = scale.datumFormatter({ domain: tickDomain, specifier, ticks, fractionDigits }) ?? ((value) => this.defaultDatumFormatter(value, fractionDigits));
    this.scaleFormatterParams = { domain: tickDomain, ticks, fractionDigits };
    this.layout.label = {
      fractionDigits,
      spacing: this.label.spacing,
      format: this.label.format
    };
    const sideFlag = label.getSideFlag();
    const anySeriesActive = this.isAnySeriesActive();
    const { rotation, parallelFlipRotation, regularFlipRotation } = this.calculateRotations();
    this.crossLines.forEach((crossLine) => {
      var _a;
      crossLine.sideFlag = -sideFlag;
      crossLine.direction = rotation === -Math.PI / 2 ? "x" /* X */ : "y" /* Y */;
      if (crossLine instanceof CartesianCrossLine) {
        (_a = crossLine.label).parallel ?? (_a.parallel = label.parallel);
      }
      crossLine.parallelFlipRotation = parallelFlipRotation;
      crossLine.regularFlipRotation = regularFlipRotation;
      crossLine.calculateLayout?.(anySeriesActive, this.reverse);
    });
    return { primaryTickCount, bbox };
  }
  getTransformBox(bbox) {
    const matrix = new Matrix();
    const { rotation, translationX, translationY } = this.getAxisTransform();
    Matrix.updateTransformMatrix(matrix, 1, 1, rotation, translationX, translationY);
    return matrix.transformBBox(bbox);
  }
  calculateRotations() {
    const rotation = toRadians(this.rotation);
    const parallelFlipRotation = normalizeAngle360(rotation);
    const regularFlipRotation = normalizeAngle360(rotation - Math.PI / 2);
    return { rotation, parallelFlipRotation, regularFlipRotation };
  }
  updateCrossLines() {
    const anySeriesActive = this.isAnySeriesActive();
    this.crossLines.forEach((crossLine) => {
      crossLine.update(anySeriesActive);
    });
  }
  updateTickLines() {
    const { tick, label } = this;
    const sideFlag = label.getSideFlag();
    this.tickLineGroupSelection.each((line, datum) => {
      line.strokeWidth = datum.tickWidth ?? tick.width;
      line.stroke = datum.tickStroke ?? tick.stroke;
      line.x1 = sideFlag * (datum.tickSize ?? this.getTickSize());
      line.x2 = 0;
    });
  }
  getAxisTransform() {
    return {
      rotation: toRadians(this.rotation),
      translationX: Math.floor(this.translation.x),
      translationY: Math.floor(this.translation.y)
    };
  }
  updatePosition() {
    const { crossLineRangeGroup, crossLineLineGroup, crossLineLabelGroup, gridGroup, translation } = this;
    const { rotation } = this.calculateRotations();
    const translationX = Math.floor(translation.x);
    const translationY = Math.floor(translation.y);
    crossLineRangeGroup.setProperties({ rotation, translationX, translationY });
    crossLineLineGroup.setProperties({ rotation, translationX, translationY });
    crossLineLabelGroup.setProperties({ rotation, translationX, translationY });
    gridGroup.setProperties({ rotation, translationX, translationY });
  }
  updateGridLines() {
    const sideFlag = this.label.getSideFlag();
    const {
      gridLine: { style, width: width2 },
      gridPadding,
      gridLength
    } = this;
    if (gridLength === 0 || style.length === 0) {
      return;
    }
    this.gridLineGroupSelection.each((line, _, index) => {
      const { stroke, lineDash } = style[index % style.length];
      line.setProperties({
        x1: gridPadding,
        x2: -sideFlag * gridLength + gridPadding,
        stroke,
        strokeWidth: width2,
        lineDash
      });
    });
  }
  // For formatting (nice rounded) tick values.
  formatTick(value, index, fractionDigits, defaultFormatter) {
    const {
      labelFormatter,
      label: { formatter },
      moduleCtx: { callbackCache }
    } = this;
    let result;
    if (formatter) {
      result = callbackCache.call(formatter, { value, index, fractionDigits });
    } else if (defaultFormatter) {
      result = defaultFormatter(value);
    } else if (labelFormatter) {
      result = labelFormatter(value);
    }
    return String(result ?? value);
  }
  // For formatting arbitrary values between the ticks.
  formatDatum(value) {
    const {
      label: { formatter },
      moduleCtx: { callbackCache },
      datumFormatter: valueFormatter = this.labelFormatter
    } = this;
    let result;
    if (formatter) {
      result = callbackCache.call(formatter, { value, index: NaN });
    } else if (valueFormatter) {
      result = callbackCache.call(valueFormatter, value);
    } else if (isArray(value)) {
      result = value.filter(Boolean).join(" - ");
    }
    return String(result ?? value);
  }
  getScaleValueFormatter(format) {
    const { scaleFormatterParams } = this;
    let formatter;
    try {
      if (format != null && scaleFormatterParams != null) {
        formatter = this.scale.tickFormatter({ ...scaleFormatterParams, specifier: format });
      }
    } catch {
      logger_exports.warnOnce(`the format string ${format} is invalid, ignoring.`);
    }
    formatter ?? (formatter = (value) => this.formatDatum(value));
    return formatter;
  }
  getBBox() {
    return this.axisGroup.getBBox();
  }
  initCrossLine(crossLine) {
    crossLine.scale = this.scale;
    crossLine.gridLength = this.gridLength;
  }
  isAnySeriesActive() {
    return this.boundSeries.some((s) => this.includeInvisibleDomains || s.isEnabled());
  }
  clipTickLines(x, y, width2, height2) {
    this.tickLineGroup.setClipRect(new BBox(x, y, width2, height2));
  }
  clipGrid(x, y, width2, height2) {
    this.gridGroup.setClipRect(new BBox(x, y, width2, height2));
  }
  getTitleFormatterParams() {
    const { direction } = this;
    const boundSeries = [];
    for (const series of this.boundSeries) {
      const keys = series.getKeys(direction);
      const names = series.getNames(direction);
      for (let idx = 0; idx < keys.length; idx++) {
        boundSeries.push({ key: keys[idx], name: names[idx] });
      }
    }
    return { direction, boundSeries, defaultValue: this.title?.text };
  }
  normaliseDataDomain(d) {
    return { domain: [...d], clipped: false };
  }
  getLayoutState() {
    return {
      id: this.id,
      rect: this.getBBox(),
      gridPadding: this.gridPadding,
      seriesAreaPadding: this.seriesAreaPadding,
      tickSize: this.getTickSize(),
      direction: this.direction,
      domain: this.dataDomain.domain,
      scale: this.scale,
      ...this.layout
    };
  }
  getModuleMap() {
    return this.moduleMap;
  }
  createModuleContext() {
    this.axisContext ?? (this.axisContext = this.createAxisContext());
    return { ...this.moduleCtx, parent: this.axisContext };
  }
  createAxisContext() {
    const { scale } = this;
    return {
      axisId: this.id,
      scale: this.scale,
      direction: this.direction,
      continuous: ContinuousScale.is(scale) || OrdinalTimeScale.is(scale),
      getCanvasBounds: () => {
        return Transformable.toCanvas(this.axisGroup);
      },
      seriesKeyProperties: () => this.boundSeries.reduce((keys, series) => {
        const seriesKeys = series.getKeyProperties(this.direction);
        seriesKeys.forEach((key) => keys.add(key));
        return keys;
      }, /* @__PURE__ */ new Set()),
      seriesIds: () => this.boundSeries.map((series) => series.id),
      scaleValueFormatter: (specifier) => this.getScaleValueFormatter(specifier),
      scaleInvert: (val) => scale.invert(val, true),
      scaleInvertNearest: (val) => scale.invert(val, true),
      attachLabel: (node) => this.attachLabel(node),
      inRange: (x, tolerance) => this.inRange(x, tolerance)
    };
  }
  isReversed() {
    return this.reverse;
  }
};
_Axis.defaultTickMinSpacing = 50;
_Axis.CrossLineConstructor = CartesianCrossLine;
__decorateClass([
  Validate(BOOLEAN)
], _Axis.prototype, "nice", 2);
__decorateClass([
  Validate(BOOLEAN)
], _Axis.prototype, "reverse", 2);
__decorateClass([
  Validate(STRING_ARRAY)
], _Axis.prototype, "keys", 2);
__decorateClass([
  Validate(OBJECT)
], _Axis.prototype, "interval", 2);
__decorateClass([
  Validate(OBJECT)
], _Axis.prototype, "title", 2);
__decorateClass([
  ObserveChanges((target, value, oldValue) => target.onGridLengthChange(value, oldValue))
], _Axis.prototype, "gridLength", 2);
var Axis = _Axis;

// packages/ag-charts-community/src/scene/util/labelPlacement.ts
function circleRectOverlap(c, unitCenter, x, y, w, h) {
  if (c.size === 0) {
    return false;
  }
  let cx = c.x;
  let cy = c.y;
  if (unitCenter != null) {
    cx -= (unitCenter.x - 0.5) * c.size;
    cy -= (unitCenter.y - 0.5) * c.size;
  }
  let edgeX = cx;
  if (cx < x) {
    edgeX = x;
  } else if (cx > x + w) {
    edgeX = x + w;
  }
  let edgeY = cy;
  if (cy < y) {
    edgeY = y;
  } else if (cy > y + h) {
    edgeY = y + h;
  }
  const dx = cx - edgeX;
  const dy = cy - edgeY;
  const d = Math.sqrt(dx * dx + dy * dy);
  return d <= c.size * 0.5;
}
function rectRectOverlap(r1, x2, y2, w2, h2) {
  const xOverlap = r1.x + r1.width > x2 && r1.x < x2 + w2;
  const yOverlap = r1.y + r1.height > y2 && r1.y < y2 + h2;
  return xOverlap && yOverlap;
}
function rectContainsRect(r1, r2x, r2y, r2w, r2h) {
  return r2x + r2w < r1.x + r1.width && r2x > r1.x && r2y > r1.y && r2y + r2h < r1.y + r1.height;
}
function isPointLabelDatum(x) {
  return x != null && typeof x.point === "object" && typeof x.label === "object";
}
var labelPlacements = {
  top: { x: 0, y: -1 },
  bottom: { x: 0, y: 1 },
  left: { x: -1, y: 0 },
  right: { x: 1, y: 0 },
  "top-left": { x: -1, y: -1 },
  "top-right": { x: 1, y: -1 },
  "bottom-left": { x: -1, y: 1 },
  "bottom-right": { x: 1, y: 1 }
};
function placeLabels(data, bounds, padding = 5) {
  const result = /* @__PURE__ */ new Map();
  const previousResults = [];
  const sortedDataClone = new Map(
    [...data.entries()].map(([k, d]) => [k, d.toSorted((a, b) => b.point.size - a.point.size)])
  );
  const dataValues = [...sortedDataClone.values()].flat();
  for (const [seriesId, datums] of sortedDataClone.entries()) {
    const labels = [];
    if (!datums[0]?.label)
      continue;
    for (let index = 0, ln = datums.length; index < ln; index++) {
      const d = datums[index];
      const { point, label, anchor } = d;
      const { text, width: width2, height: height2 } = label;
      const r = point.size * 0.5;
      let dx = 0;
      let dy = 0;
      if (r > 0 && d.placement != null) {
        const placement = labelPlacements[d.placement];
        dx = (width2 * 0.5 + r + padding) * placement.x;
        dy = (height2 * 0.5 + r + padding) * placement.y;
      }
      const x = point.x - width2 * 0.5 + dx - ((anchor?.x ?? 0.5) - 0.5) * point.size;
      const y = point.y - height2 * 0.5 + dy - ((anchor?.y ?? 0.5) - 0.5) * point.size;
      const withinBounds = !bounds || rectContainsRect(bounds, x, y, width2, height2);
      if (!withinBounds)
        continue;
      const overlapPoints = dataValues.some(
        (dataDatum) => circleRectOverlap(dataDatum.point, dataDatum.anchor, x, y, width2, height2)
      );
      if (overlapPoints)
        continue;
      const overlapLabels = previousResults.some((pr) => rectRectOverlap(pr, x, y, width2, height2));
      if (overlapLabels)
        continue;
      const resultDatum = { index, text, x, y, width: width2, height: height2, datum: d };
      labels.push(resultDatum);
      previousResults.push(resultDatum);
    }
    result.set(seriesId, labels);
  }
  return result;
}
function axisLabelsOverlap(data, padding = 0) {
  const result = [];
  for (let index = 0; index < data.length; index++) {
    const datum = data[index];
    const {
      point: { x, y },
      label: { text }
    } = datum;
    let { width: width2, height: height2 } = datum.label;
    width2 += padding;
    height2 += padding;
    if (result.some((l) => rectRectOverlap(l, x, y, width2, height2))) {
      return true;
    }
    result.push({ index, text, x, y, width: width2, height: height2, datum });
  }
  return false;
}

// packages/ag-charts-community/src/util/secondaryAxisTicks.ts
function calculateNiceSecondaryAxis(domain, primaryTickCount, reverse) {
  let [start2, stop] = findMinMax(domain);
  start2 = calculateNiceStart(Math.floor(start2), stop, primaryTickCount);
  const step = getTickStep(start2, stop, primaryTickCount);
  const segments = primaryTickCount - 1;
  stop = start2 + segments * step;
  const d = reverse ? [stop, start2] : [start2, stop];
  const ticks = getTicks(start2, step, primaryTickCount);
  return { domain: d, ticks };
}
function calculateNiceStart(a, b, count) {
  const rawStep = Math.abs(b - a) / (count - 1);
  const order = Math.floor(Math.log10(rawStep));
  const magnitude = Math.pow(10, order);
  return Math.floor(a / magnitude) * magnitude;
}
function getTicks(start2, step, count) {
  const stepPower = Math.floor(Math.log10(step));
  const fractionDigits = step > 0 && step < 1 ? Math.abs(stepPower) : 0;
  const f = Math.pow(10, fractionDigits);
  const ticks = [];
  for (let i = 0; i < count; i++) {
    const tick = start2 + step * i;
    ticks[i] = Math.round(tick * f) / f;
  }
  return ticks;
}
function getTickStep(start2, stop, count) {
  const segments = count - 1;
  const rawStep = (stop - start2) / segments;
  return calculateNextNiceStep(rawStep);
}
function calculateNextNiceStep(rawStep) {
  const order = Math.floor(Math.log10(rawStep));
  const magnitude = Math.pow(10, order);
  const step = rawStep / magnitude * 10;
  if (step > 0 && step <= 1) {
    return magnitude / 10;
  }
  if (step > 1 && step <= 2) {
    return 2 * magnitude / 10;
  }
  if (step > 1 && step <= 5) {
    return 5 * magnitude / 10;
  }
  if (step > 5 && step <= 10) {
    return 10 * magnitude / 10;
  }
  if (step > 10 && step <= 20) {
    return 20 * magnitude / 10;
  }
  if (step > 20 && step <= 40) {
    return 40 * magnitude / 10;
  }
  if (step > 40 && step <= 50) {
    return 50 * magnitude / 10;
  }
  if (step > 50 && step <= 100) {
    return 100 * magnitude / 10;
  }
  return step;
}

// packages/ag-charts-community/src/util/tempUtils.ts
function createIdsGenerator() {
  const idsCounter = /* @__PURE__ */ new Map();
  return (name) => {
    const counter = idsCounter.get(name);
    if (counter) {
      idsCounter.set(name, counter + 1);
      return `${name}_${counter}`;
    }
    idsCounter.set(name, 1);
    return name;
  };
}

// packages/ag-charts-community/src/chart/axis/axisTickGenerator.ts
var AxisTickGenerator = class {
  constructor(axis) {
    this.axis = axis;
  }
  estimateTickCount(visibleRange, minSpacing, maxSpacing) {
    return estimateTickCount(
      findRangeExtent(this.axis.range),
      findRangeExtent(visibleRange),
      minSpacing,
      maxSpacing,
      ContinuousScale.defaultTickCount,
      this.axis.defaultTickMinSpacing
    );
  }
  filterTicks(ticks, tickCount) {
    const { minSpacing, maxSpacing } = this.axis.interval;
    const tickSpacing = !isNaN(minSpacing) || !isNaN(maxSpacing);
    const keepEvery = tickSpacing ? Math.ceil(ticks.length / tickCount) : 2;
    const offset4 = ticks.length % keepEvery ? -1 : 0;
    return ticks.filter((_, i) => (i + offset4) % keepEvery === 0);
  }
  generateTicks({
    domain,
    primaryTickCount,
    visibleRange,
    niceMode,
    parallelFlipRotation,
    regularFlipRotation,
    labelX,
    sideFlag
  }) {
    const {
      scale,
      label,
      interval: { minSpacing, maxSpacing }
    } = this.axis;
    const { parallel, rotation, fontFamily, fontSize, fontStyle, fontWeight } = label;
    const secondaryAxis = primaryTickCount !== void 0;
    const { defaultRotation, configuredRotation, parallelFlipFlag, regularFlipFlag } = calculateLabelRotation({
      rotation,
      parallel,
      regularFlipRotation,
      parallelFlipRotation
    });
    const initialRotation = configuredRotation + defaultRotation;
    const labelMatrix = new Matrix();
    const { maxTickCount } = this.estimateTickCount(visibleRange, minSpacing, maxSpacing);
    const continuous = ContinuousScale.is(scale) || OrdinalTimeScale.is(scale);
    const maxIterations = !continuous || isNaN(maxTickCount) ? 10 : maxTickCount;
    let textAlign = getTextAlign(parallel, configuredRotation, 0, sideFlag, regularFlipFlag);
    const textBaseline = getTextBaseline(parallel, configuredRotation, sideFlag, parallelFlipFlag);
    const font = TextUtils.toFontString({ fontFamily, fontSize, fontStyle, fontWeight });
    const textMeasurer = CachedTextMeasurerPool.getMeasurer({ font });
    const textProps = {
      fontFamily,
      fontSize,
      fontStyle,
      fontWeight,
      textBaseline,
      textAlign
    };
    const checkLabelOverlap = label.enabled && label.avoidCollisions;
    const getLabelOverlap = ({ ticks }, iterationRotation) => {
      if (!checkLabelOverlap)
        return false;
      const rotated = configuredRotation !== 0 || iterationRotation !== 0;
      const labelRotation = initialRotation + iterationRotation;
      const labelSpacing = getLabelSpacing(label.minSpacing, rotated);
      Matrix.updateTransformMatrix(labelMatrix, 1, 1, labelRotation, 0, 0);
      const labelData = createLabelData(ticks, labelX, labelMatrix, textMeasurer);
      return axisLabelsOverlap(labelData, labelSpacing);
    };
    let tickData = {
      tickDomain: [],
      ticks: [],
      rawTicks: [],
      fractionDigits: 0,
      niceDomain: void 0
    };
    let index = 0;
    let autoRotation = 0;
    let labelOverlap = true;
    let terminate = false;
    while (!terminate && labelOverlap && index <= maxIterations) {
      autoRotation = 0;
      for (const strategy of this.getTickStrategies({ domain, niceMode, secondaryAxis, index })) {
        ({ tickData, index, autoRotation, terminate } = strategy({
          index,
          tickData,
          textProps,
          terminate,
          primaryTickCount,
          visibleRange,
          // Lazily generate as only one strategy actually uses this, and it's expensive to compute
          get labelOverlap() {
            return getLabelOverlap(tickData, autoRotation);
          }
        }));
      }
      labelOverlap = getLabelOverlap(tickData, autoRotation);
    }
    textAlign = getTextAlign(parallel, configuredRotation, autoRotation, sideFlag, regularFlipFlag);
    const combinedRotation = defaultRotation + configuredRotation + autoRotation;
    if (!secondaryAxis && tickData.rawTicks.length > 0) {
      primaryTickCount = tickData.rawTicks.length;
    }
    return { tickData, primaryTickCount, combinedRotation, textBaseline, textAlign };
  }
  getTickStrategies({
    domain,
    niceMode,
    index: iteration,
    secondaryAxis
  }) {
    const { scale, label, interval } = this.axis;
    const { minSpacing } = interval;
    const continuous = ContinuousScale.is(scale) || OrdinalTimeScale.is(scale);
    const avoidLabelCollisions = label.enabled && label.avoidCollisions;
    const filterTicks = !continuous && iteration !== 0 && avoidLabelCollisions;
    const autoRotate = label.autoRotate === true && label.rotation === void 0;
    const strategies = [];
    let tickGenerationType;
    if (interval.values) {
      tickGenerationType = 3 /* VALUES */;
    } else if (secondaryAxis) {
      tickGenerationType = 1 /* CREATE_SECONDARY */;
    } else if (filterTicks) {
      tickGenerationType = 2 /* FILTER */;
    } else {
      tickGenerationType = 0 /* CREATE */;
    }
    const tickGenerationStrategy = ({
      index,
      tickData,
      primaryTickCount,
      visibleRange,
      terminate
    }) => this.createTickData(
      domain,
      niceMode,
      visibleRange,
      primaryTickCount,
      tickGenerationType,
      index,
      tickData,
      terminate
    );
    strategies.push(tickGenerationStrategy);
    if (!continuous && !isNaN(minSpacing)) {
      const tickFilterStrategy = ({
        index,
        tickData,
        primaryTickCount,
        visibleRange,
        terminate
      }) => this.createTickData(
        domain,
        niceMode,
        visibleRange,
        primaryTickCount,
        2 /* FILTER */,
        index,
        tickData,
        terminate
      );
      strategies.push(tickFilterStrategy);
    }
    if (avoidLabelCollisions && autoRotate) {
      const autoRotateStrategy = ({ index, tickData, labelOverlap, terminate }) => ({
        index,
        tickData,
        autoRotation: labelOverlap ? normalizeAngle360(toRadians(label.autoRotateAngle ?? 0)) : 0,
        terminate
      });
      strategies.push(autoRotateStrategy);
    }
    return strategies;
  }
  createTickData(domain, niceMode, visibleRange, primaryTickCount, tickGenerationType, index, tickData, terminate) {
    const { scale, interval } = this.axis;
    const { step, values, minSpacing, maxSpacing } = interval;
    const { maxTickCount, minTickCount, tickCount } = this.estimateTickCount(visibleRange, minSpacing, maxSpacing);
    const continuous = ContinuousScale.is(scale) || OrdinalTimeScale.is(scale);
    const maxIterations = !continuous || isNaN(maxTickCount) ? 10 : maxTickCount;
    const countTicks = (i) => continuous ? Math.max(tickCount - i, minTickCount) : maxTickCount;
    const regenerateTicks = step == null && values == null && countTicks(index) > minTickCount && (continuous || tickGenerationType === 2 /* FILTER */);
    while (index <= maxIterations) {
      const previousTicks = tickData.rawTicks;
      tickData = this.getTicks({
        domain,
        niceMode,
        visibleRange,
        tickGenerationType,
        previousTicks,
        minTickCount,
        maxTickCount,
        primaryTickCount,
        tickCount: countTicks(index)
      });
      index++;
      if (!regenerateTicks || !arraysEqual(tickData.rawTicks, previousTicks))
        break;
    }
    terminate || (terminate = step != null || values != null);
    return { tickData, index, autoRotation: 0, terminate };
  }
  getTicks({
    domain,
    niceMode,
    visibleRange,
    tickGenerationType,
    previousTicks,
    tickCount,
    minTickCount,
    maxTickCount,
    primaryTickCount
  }) {
    const { axis } = this;
    const { label, range: range2, scale, interval } = axis;
    const idGenerator = createIdsGenerator();
    const domainParams = {
      nice: niceMode === 0 /* TickAndDomain */,
      interval: interval.step,
      tickCount,
      minTickCount,
      maxTickCount
    };
    const tickParams = {
      ...domainParams,
      nice: niceMode === 0 /* TickAndDomain */ || niceMode === 1 /* TicksOnly */
    };
    let niceDomain = niceMode === 0 /* TickAndDomain */ ? scale.niceDomain(domainParams, domain) : domain;
    let tickDomain = niceDomain;
    let rawTicks;
    switch (tickGenerationType) {
      case 3 /* VALUES */:
        tickDomain = interval.values;
        rawTicks = interval.values;
        if (ContinuousScale.is(scale)) {
          const [d0, d1] = findMinMax(niceDomain.map(Number));
          rawTicks = rawTicks.filter((value) => Number(value) >= d0 && Number(value) <= d1).sort((a, b) => Number(a) - Number(b));
        }
        break;
      case 1 /* CREATE_SECONDARY */:
        if (ContinuousScale.is(scale)) {
          const secondaryAxisTicks = calculateNiceSecondaryAxis(
            domain.map(Number),
            primaryTickCount ?? 0,
            axis.reverse
          );
          rawTicks = secondaryAxisTicks.ticks;
          niceDomain = secondaryAxisTicks.domain.map((d) => scale.toDomain(d));
        } else {
          rawTicks = scale.ticks(tickParams, niceDomain, visibleRange) ?? [];
        }
        break;
      case 2 /* FILTER */:
        rawTicks = this.filterTicks(previousTicks, tickCount);
        break;
      default:
        rawTicks = scale.ticks(tickParams, niceDomain, visibleRange) ?? [];
    }
    const fractionDigits = rawTicks.reduce(
      (max, tick) => Math.max(max, typeof tick === "number" ? countFractionDigits(tick) : 0),
      0
    );
    const formatParams = {
      domain: tickDomain,
      ticks: rawTicks,
      fractionDigits,
      specifier: label.format
    };
    const labelFormatter = scale.tickFormatter(formatParams);
    const scaleDomain = scale.domain;
    scale.domain = niceDomain;
    const halfBandwidth = (scale.bandwidth ?? 0) / 2;
    const ticks = [];
    for (let i = 0; i < rawTicks.length; i++) {
      const tick = rawTicks[i];
      const translationY = scale.convert(tick) + halfBandwidth;
      if (range2.length > 0 && !axis.inRange(translationY, 1e-3))
        continue;
      const tickLabel = label.enabled ? axis.formatTick(tick, i, fractionDigits, labelFormatter) : "";
      ticks.push({ tick, tickId: idGenerator(tickLabel), tickLabel, translationY: Math.floor(translationY) });
    }
    scale.domain = scaleDomain;
    return {
      tickDomain,
      rawTicks,
      fractionDigits,
      ticks,
      niceDomain
    };
  }
};

// packages/ag-charts-community/src/chart/axis/cartesianAxisLabel.ts
var CartesianAxisLabel = class extends AxisLabel {
  constructor() {
    super(...arguments);
    this.autoRotateAngle = 335;
  }
};
__decorateClass([
  Validate(BOOLEAN, { optional: true })
], CartesianAxisLabel.prototype, "autoRotate", 2);
__decorateClass([
  Validate(NUMBER)
], CartesianAxisLabel.prototype, "autoRotateAngle", 2);

// packages/ag-charts-community/src/chart/axis/cartesianAxis.ts
var _CartesianAxis = class _CartesianAxis extends Axis {
  constructor(moduleCtx, scale) {
    super(moduleCtx, scale);
    this.lineNode = this.axisGroup.appendChild(
      new TranslatableLine({
        name: `${this.id}-Axis-line`,
        zIndex: 1 /* AxisLine */
      })
    );
    this.tempText = new TransformableText();
    this.tempCaption = new Caption();
    this.tickGenerator = new AxisTickGenerator(this);
    this.generatedTicks = void 0;
    this.animationManager = moduleCtx.animationManager;
    this.animationState = new StateMachine("empty", {
      empty: {
        update: {
          target: "ready",
          action: () => this.resetSelectionNodes()
        },
        reset: "empty"
      },
      ready: {
        update: (data) => this.animateReadyUpdate(data),
        resize: () => this.resetSelectionNodes(),
        reset: "empty"
      }
    });
    this.axisGroup.appendChild(this.title.caption.node);
    let previousSize = void 0;
    this.destroyFns.push(
      moduleCtx.layoutManager.addListener("layout:complete", (e) => {
        const size = [e.chart.width, e.chart.height];
        if (previousSize != null && !arraysEqual(size, previousSize)) {
          this.animationState.transition("resize");
        }
        previousSize = size;
      }),
      this.title.caption.registerInteraction(this.moduleCtx, "afterend")
    );
  }
  static is(value) {
    return value instanceof _CartesianAxis;
  }
  resetAnimation(phase) {
    if (phase === "initial") {
      this.animationState.transition("reset");
    }
  }
  get direction() {
    return this.position === "top" || this.position === "bottom" ? "x" /* X */ : "y" /* Y */;
  }
  createAxisContext() {
    return { ...super.createAxisContext(), position: this.position };
  }
  createLabel() {
    return new CartesianAxisLabel();
  }
  updateDirection() {
    switch (this.position) {
      case "top":
        this.rotation = -90;
        this.label.mirrored = true;
        this.label.parallel = true;
        break;
      case "right":
        this.rotation = 0;
        this.label.mirrored = true;
        this.label.parallel = false;
        break;
      case "bottom":
        this.rotation = -90;
        this.label.mirrored = false;
        this.label.parallel = true;
        break;
      case "left":
        this.rotation = 0;
        this.label.mirrored = false;
        this.label.parallel = false;
        break;
    }
    if (this.axisContext) {
      this.axisContext.position = this.position;
      this.axisContext.direction = this.direction;
    }
  }
  calculateLayout(primaryTickCount) {
    this.updateDirection();
    return super.calculateLayout(primaryTickCount);
  }
  calculateTickLayout(domain, niceMode, visibleRange, initialPrimaryTickCount) {
    const sideFlag = this.label.getSideFlag();
    const { parallelFlipRotation, regularFlipRotation } = this.calculateRotations();
    const labelX = sideFlag * (this.getTickSize() + this.label.spacing + this.seriesAreaPadding);
    const tickGenerationResult = this.tickGenerator.generateTicks({
      domain,
      niceMode,
      visibleRange,
      primaryTickCount: initialPrimaryTickCount,
      parallelFlipRotation,
      regularFlipRotation,
      labelX,
      sideFlag
    });
    const { tickData, primaryTickCount = initialPrimaryTickCount } = tickGenerationResult;
    const { ticks, tickDomain, rawTicks, fractionDigits, niceDomain = domain } = tickData;
    const labels = ticks.map((d) => this.getTickLabelProps(d, tickGenerationResult));
    const bbox = this.tickBBox(ticks, labels);
    this.generatedTicks = { ticks, labels };
    return { ticks: rawTicks, tickDomain, niceDomain, primaryTickCount, fractionDigits, bbox };
  }
  update() {
    this.updateDirection();
    const previousTicksIds = Array.from(this.tickLabelGroupSelection.nodes(), (node) => node.datum.tickId);
    super.update();
    if (!this.animatable) {
      this.moduleCtx.animationManager.skipCurrentBatch();
    }
    if (this.generatedTicks) {
      const { ticks } = this.generatedTicks;
      if (this.animationManager.isSkipped()) {
        this.resetSelectionNodes();
      } else {
        const tickIds = ticks.map((datum) => datum.tickId);
        const diff = diffArrays(previousTicksIds, tickIds);
        this.animationState.transition("update", diff);
      }
    }
    const { enabled, stroke, width: width2 } = this.line;
    this.lineNode.setProperties({ stroke, strokeWidth: enabled ? width2 : 0 });
    this.updateTitle(!this.generatedTicks?.ticks.length);
  }
  updatePosition() {
    super.updatePosition();
    this.axisGroup.datum = this.getAxisTransform();
  }
  tickBBox(ticks, labels) {
    const sideFlag = this.label.getSideFlag();
    const boxes = [];
    const { x, y1, y2 } = this.getAxisLineCoordinates();
    const lineBox = new BBox(
      x + Math.min(sideFlag * this.seriesAreaPadding, 0),
      y1,
      this.seriesAreaPadding,
      y2 - y1
    );
    boxes.push(lineBox);
    if (this.tick.enabled) {
      for (const datum of ticks) {
        const { x1, x2, y } = this.getTickLineCoordinates(datum);
        const tickLineBox = new BBox(x1, y, x2 - x1, 0);
        boxes.push(tickLineBox);
      }
    }
    const { tempText } = this;
    if (this.label.enabled) {
      for (const datum of labels) {
        if (!datum.visible)
          continue;
        tempText.setProperties({
          ...datum,
          translationY: Math.round(datum.translationY)
        });
        const box = tempText.getBBox();
        if (box) {
          boxes.push(box);
        }
      }
    }
    if (this.title?.enabled) {
      const spacing = BBox.merge(boxes).width;
      this.setTitleProps(this.tempCaption, { spacing });
      const titleBox = this.tempCaption.node.getBBox();
      if (titleBox) {
        boxes.push(titleBox);
      }
    }
    const bbox = BBox.merge(boxes);
    return this.getTransformBox(bbox);
  }
  getTickLabelProps(datum, tickGenerationResult) {
    const { combinedRotation, textBaseline, textAlign } = tickGenerationResult;
    const { range: range2 } = this.scale;
    const text = datum.tickLabel;
    const sideFlag = this.label.getSideFlag();
    const labelX = sideFlag * (this.getTickSize() + this.label.spacing + this.seriesAreaPadding);
    const visible = text !== "" && text != null;
    return {
      ...this.getLabelStyles({ value: datum.tickLabel }),
      tickId: datum.tickId,
      rotation: combinedRotation,
      rotationCenterX: labelX,
      translationY: datum.translationY,
      text,
      textAlign,
      textBaseline,
      visible,
      x: labelX,
      y: 0,
      range: range2
    };
  }
  getTickLineCoordinates(datum) {
    const sideFlag = this.label.getSideFlag();
    const x = sideFlag * this.getTickSize();
    const x1 = Math.min(0, x);
    const x2 = x1 + Math.abs(x);
    const y = datum.translationY;
    return { x1, x2, y };
  }
  updateSelections() {
    if (!this.generatedTicks)
      return;
    const lineData = this.getAxisLineCoordinates();
    const { ticks, labels } = this.generatedTicks;
    const getDatumId = (datum) => datum.tickId;
    this.lineNode.datum = lineData;
    this.gridLineGroupSelection.update(this.gridLength ? ticks : [], void 0, getDatumId);
    this.tickLineGroupSelection.update(ticks, void 0, getDatumId);
    this.tickLabelGroupSelection.update(labels, void 0, getDatumId);
  }
  updateTitle(noVisibleTicks, spacing) {
    const { title, tickLineGroup, tickLabelGroup, lineNode } = this;
    if (title.enabled && !noVisibleTicks && spacing == null) {
      const tickBBox = Group.computeChildrenBBox([tickLineGroup, tickLabelGroup, lineNode]);
      spacing = tickBBox.width + (tickLabelGroup.visible ? 0 : this.seriesAreaPadding);
    }
    spacing ?? (spacing = 0);
    this.setTitleProps(title.caption, { spacing });
  }
  updateLabels() {
    if (!this.label.enabled)
      return;
    this.tickLabelGroupSelection.each((node, datum) => {
      node.fill = datum.fill;
      node.fontFamily = datum.fontFamily;
      node.fontSize = datum.fontSize;
      node.fontStyle = datum.fontStyle;
      node.fontWeight = datum.fontWeight;
      node.text = datum.text;
      node.textBaseline = datum.textBaseline;
      node.textAlign = datum.textAlign ?? "center";
    });
  }
  animateReadyUpdate(diff) {
    const { animationManager } = this.moduleCtx;
    const selectionCtx = prepareAxisAnimationContext(this);
    const fns = prepareAxisAnimationFunctions(selectionCtx);
    fromToMotion(this.id, "axis-group", animationManager, [this.axisGroup], fns.group);
    fromToMotion(this.id, "line", animationManager, [this.lineNode], fns.line);
    fromToMotion(
      this.id,
      "line-paths",
      animationManager,
      [this.gridLineGroupSelection, this.tickLineGroupSelection],
      fns.tick,
      (_, d) => d.tickId,
      diff
    );
    fromToMotion(
      this.id,
      "tick-labels",
      animationManager,
      [this.tickLabelGroupSelection],
      fns.label,
      (_, d) => d.tickId,
      diff
    );
  }
  resetSelectionNodes() {
    const selectionCtx = prepareAxisAnimationContext(this);
    resetMotion([this.axisGroup], resetAxisGroupFn());
    resetMotion([this.gridLineGroupSelection, this.tickLineGroupSelection], resetAxisSelectionFn(selectionCtx));
    resetMotion([this.tickLabelGroupSelection], resetAxisLabelSelectionFn());
    resetMotion([this.lineNode], resetAxisLineSelectionFn());
  }
};
__decorateClass([
  Validate(POSITIVE_NUMBER, { optional: true })
], _CartesianAxis.prototype, "thickness", 2);
__decorateClass([
  Validate(POSITION)
], _CartesianAxis.prototype, "position", 2);
var CartesianAxis = _CartesianAxis;

// packages/ag-charts-community/src/chart/axis/categoryAxis.ts
var _CategoryAxis = class _CategoryAxis extends CartesianAxis {
  constructor(moduleCtx, scale = new CategoryScale()) {
    super(moduleCtx, scale);
    this.groupPaddingInner = 0.1;
    this.includeInvisibleDomains = true;
  }
  static is(value) {
    return value instanceof _CategoryAxis;
  }
  normaliseDataDomain(domain) {
    return { domain, clipped: false };
  }
  updateScale() {
    super.updateScale();
    let { paddingInner, paddingOuter } = this;
    if (!isFiniteNumber(paddingInner) || !isFiniteNumber(paddingOuter)) {
      const padding = this.reduceBandScalePadding();
      paddingInner ?? (paddingInner = padding.inner);
      paddingOuter ?? (paddingOuter = padding.outer);
    }
    this.scale.paddingInner = paddingInner ?? 0;
    this.scale.paddingOuter = paddingOuter ?? 0;
  }
  reduceBandScalePadding() {
    return this.boundSeries.reduce(
      (result, series) => {
        const padding = series.getBandScalePadding?.();
        if (padding) {
          if (result.inner > padding.inner) {
            result.inner = padding.inner;
          }
          if (result.outer < padding.outer) {
            result.outer = padding.outer;
          }
        }
        return result;
      },
      { inner: Infinity, outer: -Infinity }
    );
  }
};
_CategoryAxis.className = "CategoryAxis";
_CategoryAxis.type = "category";
__decorateClass([
  Validate(RATIO)
], _CategoryAxis.prototype, "groupPaddingInner", 2);
__decorateClass([
  Validate(RATIO, { optional: true })
], _CategoryAxis.prototype, "paddingInner", 2);
__decorateClass([
  Validate(RATIO, { optional: true })
], _CategoryAxis.prototype, "paddingOuter", 2);
var CategoryAxis = _CategoryAxis;

// packages/ag-charts-community/src/module/module.ts
var ModuleRegistry2 = class {
  constructor() {
    this.modules = [];
    this.dependencies = /* @__PURE__ */ new Map();
    this.modulesByOptionKey = /* @__PURE__ */ new Map();
  }
  register(...modules) {
    for (const module of modules) {
      this.registerDependencies(module);
      const otherModule = this.modules.find(
        (other) => module.type === other.type && ("optionsKey" in module && "optionsKey" in other ? module.optionsKey === other.optionsKey : true) && module.identifier === other.identifier
      );
      if (otherModule) {
        if (module.packageType === "enterprise" && otherModule.packageType === "community") {
          const index = this.modules.indexOf(otherModule);
          this.modules.splice(index, 1, module);
          if ("optionsKey" in module) {
            this.modulesByOptionKey.set(module.optionsKey, module);
          }
        }
      } else {
        this.modules.push(module);
        if ("optionsKey" in module) {
          this.modulesByOptionKey.set(module.optionsKey, module);
        }
      }
    }
  }
  hasEnterpriseModules() {
    return this.modules.some((m) => m.packageType === "enterprise");
  }
  *byType(...types) {
    const yielded = /* @__PURE__ */ new Set();
    const modulesByType = this.modules.filter((module) => types.includes(module.type));
    const calculateDependencies = (module) => {
      const deps = this.dependencies.get(module);
      return deps?.flatMap(calculateDependencies).concat(deps) ?? [];
    };
    const unresolvable = [];
    for (const module of modulesByType) {
      const uniqueKey = "optionsKey" in module ? module.optionsKey : module.contextKey;
      if (yielded.has(uniqueKey))
        continue;
      for (const dependency of calculateDependencies(uniqueKey)) {
        if (yielded.has(dependency))
          continue;
        const dependencyModule = this.modulesByOptionKey.get(dependency);
        if (!dependencyModule) {
          unresolvable.push(dependency);
          continue;
        }
        if (!types.includes(dependencyModule.type))
          continue;
        yield dependencyModule;
        yielded.add(dependency);
      }
      yield module;
      yielded.add(uniqueKey);
    }
    if (unresolvable.length > 0) {
      throw new Error(`Could not resolve module dependencies: ${unresolvable}`);
    }
  }
  registerDependencies(module) {
    if (module.dependencies == null || module.dependencies.length === 0)
      return;
    const uniqueKey = "optionsKey" in module ? module.optionsKey : module.contextKey;
    this.dependencies.set(uniqueKey, module.dependencies);
  }
};
var moduleRegistry = new ModuleRegistry2();

// packages/ag-charts-community/src/util/async.ts
var AsyncAwaitQueue = class {
  constructor() {
    this.queue = [];
  }
  await(timeout = 50) {
    return new Promise((resolve) => {
      const successFn = () => {
        clearTimeout(timeoutHandle);
        resolve(true);
      };
      const timeoutFn = () => {
        const queueIndex = this.queue.indexOf(successFn);
        if (queueIndex < 0)
          return;
        this.queue.splice(queueIndex, 1);
        resolve(false);
      };
      const timeoutHandle = setTimeout(timeoutFn, timeout);
      this.queue.push(successFn);
    });
  }
  notify() {
    this.queue.splice(0).forEach((cb) => cb());
  }
};
function pause() {
  return new Promise((resolve) => {
    setTimeout(resolve, 0);
  });
}

// packages/ag-charts-community/src/util/dom.ts
function setElementBBox(element2, bbox) {
  if (!element2)
    return;
  bbox = BBoxValues.normalize(bbox);
  if (bbox.width == null) {
    element2.style.removeProperty("width");
  } else {
    element2.style.width = `${bbox.width}px`;
  }
  if (bbox.height == null) {
    element2.style.removeProperty("height");
  } else {
    element2.style.height = `${bbox.height}px`;
  }
  if (bbox.x == null) {
    element2.style.removeProperty("left");
  } else {
    element2.style.left = `${bbox.x}px`;
  }
  if (bbox.y == null) {
    element2.style.removeProperty("top");
  } else {
    element2.style.top = `${bbox.y}px`;
  }
}
function getElementBBox(element2) {
  const width2 = parseFloat(element2.style.width) || element2.offsetWidth;
  const height2 = parseFloat(element2.style.height) || element2.offsetHeight;
  const x = parseFloat(element2.style.left) || element2.offsetLeft;
  const y = parseFloat(element2.style.top) || element2.offsetTop;
  return { x, y, width: width2, height: height2 };
}
function isInputPending() {
  const navigator = getWindow("navigator");
  if ("scheduling" in navigator) {
    const scheduling = navigator.scheduling;
    if ("isInputPending" in scheduling) {
      return scheduling.isInputPending({ includeContinuous: true });
    }
  }
  return false;
}

// packages/ag-charts-community/src/util/json.ts
var CLASS_INSTANCE_TYPE = "class-instance";
function jsonDiff(source, target, skip) {
  if (isArray(target)) {
    if (!isArray(source) || source.length !== target.length || target.some((v, i) => jsonDiff(source[i], v) != null)) {
      return target;
    }
  } else if (isPlainObject(target)) {
    if (!isPlainObject(source)) {
      return target;
    }
    const result = {};
    const allKeys = /* @__PURE__ */ new Set([
      ...Object.keys(source),
      ...Object.keys(target)
    ]);
    for (const key of allKeys) {
      if (source[key] === target[key] || skip?.includes(key)) {
        continue;
      }
      if (typeof source[key] === typeof target[key]) {
        const diff = jsonDiff(source[key], target[key]);
        if (diff !== null) {
          result[key] = diff;
        }
      } else {
        result[key] = target[key];
      }
    }
    return Object.keys(result).length ? result : null;
  } else if (source !== target) {
    return target;
  }
  return null;
}
function deepClone(source, shallow) {
  if (isArray(source)) {
    return source.map((item) => deepClone(item, shallow));
  }
  if (isPlainObject(source)) {
    return clonePlainObject(source, shallow);
  }
  if (source instanceof Map) {
    return new Map(deepClone(Array.from(source)));
  }
  return shallowClone(source);
}
function clonePlainObject(source, shallow) {
  const target = {};
  for (const key of Object.keys(source)) {
    target[key] = shallow?.has(key) ? shallowClone(source[key]) : deepClone(source[key], shallow);
  }
  return target;
}
function shallowClone(source) {
  if (isArray(source)) {
    return source.slice(0);
  }
  if (isPlainObject(source)) {
    return { ...source };
  }
  if (isDate(source)) {
    return new Date(source);
  }
  if (isRegExp(source)) {
    return new RegExp(source.source, source.flags);
  }
  return source;
}
function jsonApply(target, source, params = {}) {
  const { path, matcherPath = path?.replace(/(\[[0-9+]+])/i, "[]"), skip = [] } = params;
  if (target == null) {
    throw new Error(`AG Charts - target is uninitialised: ${path ?? "<root>"}`);
  }
  if (source == null) {
    return target;
  }
  if (isProperties(target)) {
    return target.set(source);
  }
  const targetAny = target;
  const targetType = classify(target);
  for (const property of Object.keys(source)) {
    if (SKIP_JS_BUILTINS.has(property))
      continue;
    const propertyMatcherPath = `${matcherPath ? matcherPath + "." : ""}${property}`;
    if (skip.includes(propertyMatcherPath))
      continue;
    const newValue = source[property];
    const propertyPath = `${path ? path + "." : ""}${property}`;
    const targetClass = targetAny.constructor;
    const currentValue = targetAny[property];
    try {
      const currentValueType = classify(currentValue);
      const newValueType = classify(newValue);
      if (targetType === CLASS_INSTANCE_TYPE && !(property in target)) {
        if (newValue === void 0)
          continue;
        logger_exports.warn(`unable to set [${propertyPath}] in ${targetClass?.name} - property is unknown`);
        continue;
      }
      if (currentValueType != null && newValueType != null && newValueType !== currentValueType && (currentValueType !== CLASS_INSTANCE_TYPE || newValueType !== "object")) {
        logger_exports.warn(
          `unable to set [${propertyPath}] in ${targetClass?.name} - can't apply type of [${newValueType}], allowed types are: [${currentValueType}]`
        );
        continue;
      }
      if (isProperties(currentValue)) {
        targetAny[property].set(newValue);
      } else if (newValueType === "object") {
        if (currentValue == null) {
          logger_exports.warn(`unable to set [${propertyPath}] in ${targetClass?.name} - property is unknown`);
          continue;
        }
        jsonApply(currentValue, newValue, {
          ...params,
          path: propertyPath,
          matcherPath: propertyMatcherPath
        });
      } else {
        targetAny[property] = newValue;
      }
    } catch (error2) {
      logger_exports.warn(`unable to set [${propertyPath}] in [${targetClass?.name}]; nested error is: ${error2.message}`);
    }
  }
  return target;
}
function classify(value) {
  if (value == null) {
    return null;
  }
  if (isHtmlElement(value) || isDate(value)) {
    return "primitive";
  }
  if (isArray(value)) {
    return "array";
  }
  if (isObject(value)) {
    return isPlainObject(value) ? "object" : CLASS_INSTANCE_TYPE;
  }
  if (isFunction(value)) {
    return "function";
  }
  return "primitive";
}
var Operation = /* @__PURE__ */ ((Operation2) => {
  Operation2["Ref"] = "$ref";
  Operation2["Path"] = "$path";
  Operation2["If"] = "$if";
  Operation2["Eq"] = "$eq";
  Operation2["Not"] = "$not";
  Operation2["Or"] = "$or";
  Operation2["And"] = "$and";
  Operation2["Mul"] = "$mul";
  Operation2["Round"] = "$round";
  Operation2["Rem"] = "$rem";
  Operation2["Mix"] = "$mix";
  Operation2["ForegroundBackgroundMix"] = "$foregroundBackgroundMix";
  Operation2["ForegroundBackgroundAccentMix"] = "$foregroundBackgroundAccentMix";
  return Operation2;
})(Operation || {});
var operationKeys = new Set(Object.values(Operation));

// packages/ag-charts-community/src/util/mutex.ts
var Mutex = class {
  constructor() {
    this.available = true;
    this.acquireQueue = [];
  }
  acquire(cb) {
    return new Promise((resolve) => {
      this.acquireQueue.push([cb, resolve]);
      if (this.available) {
        this.dispatchNext().catch((e) => logger_exports.errorOnce(e));
      }
    });
  }
  async acquireImmediately(cb) {
    if (!this.available) {
      return false;
    }
    await this.acquire(cb);
    return true;
  }
  async waitForClearAcquireQueue() {
    return this.acquire(() => Promise.resolve(void 0));
  }
  async dispatchNext() {
    this.available = false;
    let [next, done] = this.acquireQueue.shift() ?? [];
    while (next) {
      try {
        await next();
        done?.();
      } catch (error2) {
        logger_exports.error("mutex callback error", error2);
        done?.();
      }
      [next, done] = this.acquireQueue.shift() ?? [];
    }
    this.available = true;
  }
};

// packages/ag-charts-community/src/util/observable.ts
var Observable = class {
  constructor() {
    this.eventListeners = /* @__PURE__ */ new Map();
  }
  addEventListener(eventType, listener) {
    if (typeof listener !== "function") {
      throw new Error("AG Charts - listener must be a Function");
    }
    const eventTypeListeners = this.eventListeners.get(eventType);
    if (eventTypeListeners) {
      eventTypeListeners.add(listener);
    } else {
      this.eventListeners.set(eventType, /* @__PURE__ */ new Set([listener]));
    }
  }
  removeEventListener(type, listener) {
    this.eventListeners.get(type)?.delete(listener);
    if (this.eventListeners.size === 0) {
      this.eventListeners.delete(type);
    }
  }
  hasEventListener(type) {
    return this.eventListeners.has(type);
  }
  clearEventListeners() {
    this.eventListeners.clear();
  }
  fireEvent(event) {
    this.eventListeners.get(event.type)?.forEach((listener) => listener(event));
  }
};

// packages/ag-charts-community/src/util/padding.ts
var Padding = class extends BaseProperties {
  constructor(top = 0, right = top, bottom = top, left = right) {
    super();
    this.top = top;
    this.right = right;
    this.bottom = bottom;
    this.left = left;
  }
};
__decorateClass([
  Validate(POSITIVE_NUMBER)
], Padding.prototype, "top", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], Padding.prototype, "right", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], Padding.prototype, "bottom", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], Padding.prototype, "left", 2);

// packages/ag-charts-community/src/util/render.ts
function debouncedAnimationFrame(cb) {
  return buildScheduler((innerCb, _delayMs) => getWindow().requestAnimationFrame(innerCb), cb);
}
function debouncedCallback(cb) {
  return buildScheduler((innerCb, delayMs = 0) => {
    if (delayMs === 0) {
      queueMicrotask(innerCb);
    } else {
      setTimeout(innerCb, delayMs);
    }
  }, cb);
}
function buildScheduler(scheduleFn, cb) {
  let scheduleCount = 0;
  let promiseRunning = false;
  let awaitingPromise;
  let awaitingDone;
  const busy = () => {
    return promiseRunning;
  };
  const done = () => {
    promiseRunning = false;
    awaitingDone?.();
    awaitingDone = void 0;
    awaitingPromise = void 0;
    if (scheduleCount > 0) {
      scheduleFn(scheduleCb);
    }
  };
  const scheduleCb = () => {
    const count = scheduleCount;
    scheduleCount = 0;
    promiseRunning = true;
    const maybePromise = cb({ count });
    if (!maybePromise) {
      done();
      return;
    }
    maybePromise.then(done, done);
  };
  return {
    schedule(delayMs) {
      if (scheduleCount === 0 && !busy()) {
        scheduleFn(scheduleCb, delayMs);
      }
      scheduleCount++;
    },
    async await() {
      if (!busy()) {
        return;
      }
      if (awaitingPromise == null) {
        awaitingPromise = new Promise((resolve) => {
          awaitingDone = resolve;
        });
      }
      while (busy()) {
        await awaitingPromise;
      }
    }
  };
}

// packages/ag-charts-community/src/util/attributeUtil.ts
function booleanParser(value) {
  return value === "true";
}
function numberParser(value) {
  return Number(value);
}
function stringParser(value) {
  return value;
}
var AttributeTypeParsers = {
  role: stringParser,
  "aria-checked": booleanParser,
  "aria-controls": stringParser,
  "aria-describedby": stringParser,
  "aria-disabled": booleanParser,
  "aria-expanded": booleanParser,
  "aria-haspopup": booleanParser,
  "aria-hidden": booleanParser,
  "aria-label": stringParser,
  "aria-labelledby": stringParser,
  "aria-live": stringParser,
  "aria-orientation": stringParser,
  "aria-selected": booleanParser,
  "data-preventdefault": booleanParser,
  class: stringParser,
  id: stringParser,
  tabindex: numberParser,
  title: stringParser,
  placeholder: stringParser
};
function setAttribute(e, qualifiedName, value) {
  if (value == null || value === "" || value === "") {
    e?.removeAttribute(qualifiedName);
  } else {
    e?.setAttribute(qualifiedName, value.toString());
  }
}
function setAttributes(e, attrs) {
  if (attrs == null)
    return;
  for (const [key, value] of Object.entries(attrs)) {
    if (key === "class")
      continue;
    setAttribute(e, key, value);
  }
}
function getAttribute(e, qualifiedName, defaultValue) {
  if (!(e instanceof HTMLElement))
    return void 0;
  const value = e.getAttribute(qualifiedName);
  if (value === null)
    return defaultValue;
  return AttributeTypeParsers[qualifiedName]?.(value) ?? void 0;
}
function setElementStyle(e, property, value) {
  if (e == null)
    return;
  if (value == null) {
    e.style.removeProperty(property);
  } else {
    e.style.setProperty(property, value.toString());
  }
}
function setElementStyles(e, styles) {
  for (const [key, value] of Object.entries(styles)) {
    setElementStyle(e, key, value);
  }
}

// packages/ag-charts-community/src/widget/widgetEvents.ts
var WIDGET_HTML_EVENTS = [
  "blur",
  "change",
  "contextmenu",
  "focus",
  "keydown",
  "keyup",
  "click",
  "dblclick",
  "mouseenter",
  "mousemove",
  "mouseleave",
  "wheel",
  "touchstart",
  "touchmove",
  "touchend",
  "touchcancel"
];
function allocMouseEvent(type, sourceEvent, current) {
  const { offsetX, offsetY, clientX, clientY } = sourceEvent;
  const { currentX, currentY } = WidgetEventUtil.calcCurrentXY(current, sourceEvent);
  return { type, offsetX, offsetY, clientX, clientY, currentX, currentY, sourceEvent };
}
function allocTouchEvent(type, sourceEvent, _current) {
  return { type, sourceEvent };
}
var WidgetAllocators = {
  blur: (sourceEvent) => {
    return { type: "blur", sourceEvent };
  },
  change: (sourceEvent) => {
    return { type: "change", sourceEvent };
  },
  contextmenu: (sourceEvent, current) => {
    return allocMouseEvent("contextmenu", sourceEvent, current);
  },
  focus: (sourceEvent) => {
    return { type: "focus", sourceEvent };
  },
  keydown: (sourceEvent) => {
    return { type: "keydown", sourceEvent };
  },
  keyup: (sourceEvent) => {
    return { type: "keyup", sourceEvent };
  },
  click: (sourceEvent, current) => {
    return allocMouseEvent("click", sourceEvent, current);
  },
  dblclick: (sourceEvent, current) => {
    return allocMouseEvent("dblclick", sourceEvent, current);
  },
  mouseenter: (sourceEvent, current) => {
    return allocMouseEvent("mouseenter", sourceEvent, current);
  },
  mousemove: (sourceEvent, current) => {
    return allocMouseEvent("mousemove", sourceEvent, current);
  },
  mouseleave: (sourceEvent, current) => {
    return allocMouseEvent("mouseleave", sourceEvent, current);
  },
  wheel: (sourceEvent) => {
    const { offsetX, offsetY, clientX, clientY } = sourceEvent;
    const factor = sourceEvent.deltaMode === 0 ? 0.01 : 1;
    const deltaX = sourceEvent.deltaX * factor;
    const deltaY = sourceEvent.deltaY * factor;
    return { type: "wheel", offsetX, offsetY, clientX, clientY, deltaX, deltaY, sourceEvent };
  },
  touchstart: (sourceEvent, current) => {
    return allocTouchEvent("touchstart", sourceEvent, current);
  },
  touchmove: (sourceEvent, current) => {
    return allocTouchEvent("touchmove", sourceEvent, current);
  },
  touchend: (sourceEvent, current) => {
    return allocTouchEvent("touchend", sourceEvent, current);
  },
  touchcancel: (sourceEvent, current) => {
    return allocTouchEvent("touchcancel", sourceEvent, current);
  }
};
var WidgetEventUtil = class {
  static alloc(type, sourceEvent, current) {
    return WidgetAllocators[type](sourceEvent, current);
  }
  static isHTMLEvent(type) {
    const htmlTypes = WIDGET_HTML_EVENTS;
    return htmlTypes.includes(type);
  }
  static calcCurrentXY(current, event) {
    const currentRect = current.getBoundingClientRect();
    return { currentX: event.clientX - currentRect.x, currentY: event.clientY - currentRect.y };
  }
};

// packages/ag-charts-community/src/widget/widgetListenerHTML.ts
var WidgetListenerHTML = class {
  constructor() {
    this.widgetListeners = {};
    this.sourceListeners = {};
  }
  initSourceHandler(type, handler) {
    this.sourceListeners ?? (this.sourceListeners = {});
    this.sourceListeners[type] = handler;
  }
  lazyGetWidgetListeners(type, target) {
    var _a;
    if (!(type in (this.sourceListeners ?? {}))) {
      const sourceHandler = (sourceEvent) => {
        const widgetEvent = WidgetEventUtil.alloc(type, sourceEvent, target.getElement());
        for (const widgetListener of this.widgetListeners?.[type] ?? []) {
          widgetListener(widgetEvent, target);
        }
      };
      const opts = {};
      if (type.startsWith("touch"))
        opts.passive = false;
      this.initSourceHandler(type, sourceHandler);
      target.getElement().addEventListener(type, sourceHandler, opts);
    }
    this.widgetListeners ?? (this.widgetListeners = {});
    (_a = this.widgetListeners)[type] ?? (_a[type] = []);
    return this.widgetListeners[type];
  }
  add(type, target, handler) {
    const listeners = this.lazyGetWidgetListeners(type, target);
    listeners.push(handler);
  }
  remove(type, target, handler) {
    const listeners = this.lazyGetWidgetListeners(type, target);
    const index = listeners.indexOf(handler);
    if (index > -1)
      listeners.splice(index, 1);
  }
  destroy(target) {
    for (const [key, sourceHandler] of Object.entries(this.sourceListeners ?? {})) {
      const type = key;
      target.getElement().removeEventListener(type, sourceHandler);
    }
    this.widgetListeners = void 0;
    this.sourceListeners = void 0;
  }
};

// packages/ag-charts-community/src/widget/mouseDragger.ts
var MouseDragger = class {
  constructor(glob, self, myCallbacks, downEvent) {
    this.glob = glob;
    this.self = self;
    this.window = getWindow();
    this.mousegeneral = (generalEvent) => {
      generalEvent.stopPropagation();
      generalEvent.stopImmediatePropagation();
    };
    this.mousemove = (moveEvent) => {
      moveEvent.stopPropagation();
      moveEvent.stopImmediatePropagation();
      this.glob.globalMouseDragCallbacks?.mousemove(moveEvent);
    };
    this.mouseup = (upEvent) => {
      if (upEvent.button === 0) {
        upEvent.stopPropagation();
        upEvent.stopImmediatePropagation();
        this.glob.globalMouseDragCallbacks?.mouseup(upEvent);
        this.destroy();
      }
    };
    const { window: window2, mousegeneral, mousemove, mouseup } = this;
    window2.addEventListener("mousedown", mousegeneral, { capture: true });
    window2.addEventListener("mouseenter", mousegeneral, { capture: true });
    window2.addEventListener("mouseleave", mousegeneral, { capture: true });
    window2.addEventListener("mouseout", mousegeneral, { capture: true });
    window2.addEventListener("mouseover", mousegeneral, { capture: true });
    window2.addEventListener("mousemove", mousemove, { capture: true });
    window2.addEventListener("mouseup", mouseup, { capture: true });
    self.mouseDragger = this;
    glob.globalMouseDragCallbacks = myCallbacks;
    glob.globalMouseDragCallbacks.mousedown(downEvent);
  }
  destroy() {
    const { window: window2, mousegeneral, mousemove, mouseup } = this;
    window2.removeEventListener("mousedown", mousegeneral, { capture: true });
    window2.removeEventListener("mouseenter", mousegeneral, { capture: true });
    window2.removeEventListener("mouseleave", mousegeneral, { capture: true });
    window2.removeEventListener("mouseout", mousegeneral, { capture: true });
    window2.removeEventListener("mouseover", mousegeneral, { capture: true });
    window2.removeEventListener("mousemove", mousemove, { capture: true });
    window2.removeEventListener("mouseup", mouseup, { capture: true });
    this.glob.globalMouseDragCallbacks = void 0;
    this.self.mouseDragger = void 0;
  }
};
function startMouseDrag(glob, self, myCallbacks, downEvent) {
  if (glob.globalMouseDragCallbacks != null)
    return void 0;
  return new MouseDragger(glob, self, myCallbacks, downEvent);
}

// packages/ag-charts-community/src/widget/touchDragger.ts
var LONG_TAP_DURATION_MS = 500;
var LONG_TAP_INTERRUPT_MIN_TOUCHMOVE_PXPX = 100;
function deltaClientSquared(a, b) {
  const dx = a.clientX - b.clientX;
  const dy = a.clientY - b.clientY;
  return dx * dx + dy * dy;
}
var gIsInLongTap = false;
var TouchDragger = class {
  constructor(glob, self, myCallbacks, initialTouch, target) {
    this.glob = glob;
    this.self = self;
    this.initialTouch = initialTouch;
    this.target = target;
    this.longTapInterrupted = false;
    this.longtap = () => {
      const { target, initialTouch } = this;
      if (!this.longTapInterrupted) {
        target.dispatchEvent(new TouchEvent("touchcancel", { touches: [initialTouch], bubbles: true }));
        gIsInLongTap = true;
        const longTapMove = (e) => {
          e.preventDefault();
        };
        const longTapEnd = (e) => {
          gIsInLongTap = false;
          e.preventDefault();
          target.removeEventListener("touchmove", longTapMove);
          target.removeEventListener("touchend", longTapEnd);
          target.removeEventListener("touchcancel", longTapEnd);
        };
        target.addEventListener("touchmove", longTapMove, { passive: false });
        target.addEventListener("touchend", longTapEnd, { passive: false });
        target.addEventListener("touchcancel", longTapEnd, { passive: false });
        const { clientX, clientY } = initialTouch;
        const contextMenuEvent = new PointerEvent("contextmenu", {
          bubbles: true,
          cancelable: true,
          view: getWindow(),
          clientX,
          clientY,
          pointerType: "touch"
        });
        target.dispatchEvent(contextMenuEvent);
      }
    };
    this.touchmove = (moveEvent) => {
      const { glob, self, initialTouch } = this;
      const touch = this.findInitialFinger(moveEvent.targetTouches);
      if (touch != null) {
        this.longTapInterrupted = this.longTapInterrupted || deltaClientSquared(initialTouch, touch) > LONG_TAP_INTERRUPT_MIN_TOUCHMOVE_PXPX;
        if (self.dragTouchEnabled && touch != null) {
          glob.globalTouchDragCallbacks?.touchmove(moveEvent, touch);
        }
      }
    };
    this.touchend = (endEvent) => {
      this.longTapInterrupted = true;
      const touch = this.findInitialFinger(endEvent.changedTouches, endEvent.touches);
      if (touch != null) {
        this.glob.globalTouchDragCallbacks?.touchend(endEvent, touch);
      }
      this.destroy();
    };
    this.longtapTimer = setTimeout(this.longtap, LONG_TAP_DURATION_MS);
    const { touchmove, touchend } = this;
    target.addEventListener("touchmove", touchmove, { passive: false });
    target.addEventListener("touchstart", touchend, { passive: false });
    target.addEventListener("touchend", touchend, { passive: false });
    target.addEventListener("touchcancel", touchend, { passive: false });
    self.touchDragger = this;
    glob.globalTouchDragCallbacks = myCallbacks;
  }
  destroy() {
    const { longtapTimer, touchmove, touchend } = this;
    clearTimeout(longtapTimer);
    this.target.removeEventListener("touchstart", touchend);
    this.target.removeEventListener("touchmove", touchmove);
    this.target.removeEventListener("touchend", touchend);
    this.target.removeEventListener("touchcancel", touchend);
    this.glob.globalTouchDragCallbacks = void 0;
    this.self.touchDragger = void 0;
  }
  findInitialFinger(...touchLists) {
    const touches = touchLists.map((touchList) => Array.from(touchList)).flat();
    return Array.from(touches).find((v) => v.identifier === this.initialTouch.identifier);
  }
};
function startOneFingerTouch(glob, self, myCallbacks, initialTouch, target) {
  if (glob.globalTouchDragCallbacks != null || gIsInLongTap)
    return void 0;
  return new TouchDragger(glob, self, myCallbacks, initialTouch, target);
}

// packages/ag-charts-community/src/widget/widgetListenerInternal.ts
function makeMouseDrag(current, type, origin2, sourceEvent) {
  const { currentX, currentY } = WidgetEventUtil.calcCurrentXY(current.getElement(), sourceEvent);
  const originDeltaX = sourceEvent.pageX - origin2.pageX;
  const originDeltaY = sourceEvent.pageY - origin2.pageY;
  return {
    type,
    device: "mouse",
    offsetX: origin2.offsetX + originDeltaX,
    offsetY: origin2.offsetY + originDeltaY,
    clientX: sourceEvent.clientX,
    clientY: sourceEvent.clientY,
    currentX,
    currentY,
    originDeltaX,
    originDeltaY,
    sourceEvent
  };
}
function getTouchOffsets(current, { pageX, pageY }) {
  const { x, y } = current.getElement().getBoundingClientRect();
  return { offsetX: pageX - x, offsetY: pageY - y };
}
function makeTouchDrag(current, type, origin2, sourceEvent, touch) {
  const { currentX, currentY } = WidgetEventUtil.calcCurrentXY(current.getElement(), touch);
  const originDeltaX = touch.pageX - origin2.pageX;
  const originDeltaY = touch.pageY - origin2.pageY;
  return {
    type,
    device: "touch",
    offsetX: origin2.offsetX + originDeltaX,
    offsetY: origin2.offsetY + originDeltaY,
    clientX: touch.clientX,
    clientY: touch.clientY,
    currentX,
    currentY,
    originDeltaX,
    originDeltaY,
    sourceEvent
  };
}
var GlobalCallbacks = {};
var WidgetListenerInternal = class {
  constructor(dispatchCallback) {
    this.dispatchCallback = dispatchCallback;
    this.dragTouchEnabled = true;
  }
  destroy() {
    this.dragTriggerRemover?.();
    this.dragTriggerRemover = void 0;
    this.dragStartListeners = void 0;
    this.dragMoveListeners = void 0;
    this.dragEndListeners = void 0;
    this.mouseDragger?.destroy();
    this.touchDragger?.destroy();
  }
  add(type, target, handler) {
    switch (type) {
      case "drag-start": {
        this.dragStartListeners ?? (this.dragStartListeners = []);
        this.dragStartListeners.push(handler);
        this.registerDragTrigger(target);
        break;
      }
      case "drag-move": {
        this.dragMoveListeners ?? (this.dragMoveListeners = []);
        this.dragMoveListeners.push(handler);
        this.registerDragTrigger(target);
        break;
      }
      case "drag-end": {
        this.dragEndListeners ?? (this.dragEndListeners = []);
        this.dragEndListeners.push(handler);
        this.registerDragTrigger(target);
        break;
      }
    }
  }
  remove(type, _target, handler) {
    switch (type) {
      case "drag-start":
        return this.removeHandler(this.dragStartListeners, handler);
      case "drag-move":
        return this.removeHandler(this.dragMoveListeners, handler);
      case "drag-end":
        return this.removeHandler(this.dragEndListeners, handler);
    }
  }
  removeHandler(array2, handler) {
    const index = array2?.indexOf(handler);
    if (index !== void 0)
      array2?.splice(index, 1);
  }
  registerDragTrigger(target) {
    if (this.dragTriggerRemover == null) {
      const mouseTrigger = (event) => this.triggerMouseDrag(target, event);
      const touchTrigger = (event) => this.triggerTouchDrag(target, event);
      target.getElement().addEventListener("mousedown", mouseTrigger);
      target.getElement().addEventListener("touchstart", touchTrigger, { passive: false });
      this.dragTriggerRemover = () => {
        target.getElement().removeEventListener("mousedown", mouseTrigger);
        target.getElement().removeEventListener("touchstart", touchTrigger);
      };
    }
  }
  triggerMouseDrag(current, downEvent) {
    if (downEvent.button === 0) {
      this.startMouseDrag(current, downEvent);
    }
  }
  startMouseDrag(current, initialDownEvent) {
    const origin2 = { pageX: NaN, pageY: NaN, offsetX: NaN, offsetY: NaN };
    partialAssign(["pageX", "pageY", "offsetX", "offsetY"], origin2, initialDownEvent);
    const dragCallbacks = {
      mousedown: (downEvent) => {
        const dragStartEvent = makeMouseDrag(current, "drag-start", origin2, downEvent);
        this.dispatch("drag-start", current, dragStartEvent);
      },
      mousemove: (moveEvent) => {
        const dragMoveEvent = makeMouseDrag(current, "drag-move", origin2, moveEvent);
        this.dispatch("drag-move", current, dragMoveEvent);
      },
      mouseup: (upEvent) => {
        const dragEndEvent = makeMouseDrag(current, "drag-end", origin2, upEvent);
        this.dispatch("drag-end", current, dragEndEvent);
        this.endDrag(current, dragEndEvent);
      }
    };
    this.mouseDragger = startMouseDrag(GlobalCallbacks, this, dragCallbacks, initialDownEvent);
  }
  endDrag(target, { sourceEvent, clientX, clientY }) {
    const elem = target.getElement();
    const rect = elem.getBoundingClientRect();
    if (!BBoxValues.containsPoint(rect, clientX, clientY)) {
      elem.dispatchEvent(new MouseEvent("mouseleave", sourceEvent));
      sourceEvent.target?.dispatchEvent(new MouseEvent("mouseenter", sourceEvent));
    }
  }
  triggerTouchDrag(current, startEvent) {
    const touch = startEvent.targetTouches.item(0);
    if (startEvent.targetTouches.length === 1 && touch != null) {
      this.startOneFingerTouch(current, startEvent, touch);
    }
  }
  startOneFingerTouch(current, initialEvent, initialTouch) {
    const origin2 = { pageX: NaN, pageY: NaN, ...getTouchOffsets(current, initialTouch) };
    partialAssign(["pageX", "pageY"], origin2, initialTouch);
    const dragCallbacks = {
      touchmove: (moveEvent, touch) => {
        const dragMoveEvent = makeTouchDrag(current, "drag-move", origin2, moveEvent, touch);
        this.dispatch("drag-move", current, dragMoveEvent);
      },
      touchend: (cancelEvent, touch) => {
        const dragMoveEvent = makeTouchDrag(current, "drag-end", origin2, cancelEvent, touch);
        this.dispatch("drag-end", current, dragMoveEvent);
      }
    };
    const target = current.getElement();
    this.touchDragger = startOneFingerTouch(GlobalCallbacks, this, dragCallbacks, initialTouch, target);
    const dragStartEvent = makeTouchDrag(current, "drag-start", origin2, initialEvent, initialTouch);
    this.dispatch("drag-start", current, dragStartEvent);
  }
  dispatch(type, current, event) {
    switch (type) {
      case "drag-start":
        this.dragStartListeners?.forEach((handler) => handler(event, current));
        break;
      case "drag-move":
        this.dragMoveListeners?.forEach((handler) => handler(event, current));
        break;
      case "drag-end":
        this.dragEndListeners?.forEach((handler) => handler(event, current));
        break;
    }
    this.dispatchCallback(type, event);
  }
};

// packages/ag-charts-community/src/widget/widget.ts
var WidgetBounds = class {
  constructor(elem) {
    this.elem = elem;
  }
  setBounds(bounds) {
    setElementBBox(this.elemContainer ?? this.elem, bounds);
  }
  getBounds() {
    return getElementBBox(this.elemContainer ?? this.elem);
  }
  static setElementContainer(widget, elemContainer) {
    const currentBounds = widget.getBounds();
    setElementBBox(elemContainer, currentBounds);
    setElementStyles(widget.elem, { width: "100%", height: "100%" });
    widget.elem.remove();
    widget.elemContainer = elemContainer;
    widget.elemContainer.replaceChildren(widget.elem);
  }
};
var Widget = class extends WidgetBounds {
  constructor(elem) {
    super(elem);
    this.elem = elem;
    this.index = NaN;
    this.children = [];
  }
  getElement() {
    return this.elem;
  }
  getBoundingClientRect() {
    return this.elem.getBoundingClientRect();
  }
  get clientWidth() {
    return this.elem.clientWidth;
  }
  get clientHeight() {
    return this.elem.clientHeight;
  }
  destroy() {
    this.parent?.removeChild(this);
    this.children.forEach((child) => {
      child.parent = void 0;
      child.destroy();
    });
    this.children.length = 0;
    this.destructor();
    this.elem.remove();
    this.elemContainer?.remove();
    this.internalListener?.destroy();
    this.htmlListener?.destroy(this);
  }
  setHidden(hidden) {
    setElementStyle(this.elem, "display", hidden ? "none" : void 0);
  }
  isHidden() {
    return getWindow()?.getComputedStyle?.(this.elem).display === "none";
  }
  setCursor(cursor) {
    setElementStyle(this.elem, "cursor", cursor);
  }
  setTextContent(textContent) {
    this.elem.textContent = textContent ?? null;
  }
  setAriaDescribedBy(ariaDescribedBy) {
    setAttribute(this.elem, "aria-describedby", ariaDescribedBy);
  }
  setAriaHidden(ariaHidden) {
    setAttribute(this.elem, "aria-hidden", ariaHidden);
  }
  setAriaLabel(ariaLabel) {
    setAttribute(this.elem, "aria-label", ariaLabel);
  }
  setInnerHTML(html) {
    this.elem.innerHTML = html;
  }
  isDisabled() {
    return getAttribute(this.elem, "aria-disabled", false);
  }
  parseFloat(s) {
    return s === "" ? 0 : parseFloat(s);
  }
  cssLeft() {
    return this.parseFloat(this.elem.style.left);
  }
  cssTop() {
    return this.parseFloat(this.elem.style.top);
  }
  cssWidth() {
    return this.parseFloat(this.elem.style.width);
  }
  cssHeight() {
    return this.parseFloat(this.elem.style.height);
  }
  focus() {
    this.elem.focus();
  }
  setPreventsDefault(preventDefault) {
    setAttribute(this.elem, "data-preventdefault", preventDefault);
  }
  setTabIndex(tabIndex) {
    setAttribute(this.elem, "tabindex", tabIndex);
  }
  addChild(child) {
    this.addChildToDOM(child, this.getBefore(child));
    this.children.push(child);
    child.index = this.children.length - 1;
    child.parent = this;
    this.onChildAdded(child);
  }
  removeChild(child) {
    const i = this.children.findIndex((value) => value === child);
    this.children.splice(i, 1);
    this.removeChildFromDOM(child);
    this.onChildRemoved(child);
  }
  moveChild(child, domIndex) {
    if (child.domIndex === domIndex)
      return;
    child.domIndex = domIndex;
    this.removeChildFromDOM(child);
    this.addChildToDOM(child, this.getBefore(child));
  }
  addClass(...tokens) {
    this.elem.classList.add(...tokens);
  }
  removeClass(...tokens) {
    this.elem.classList.remove(...tokens);
  }
  toggleClass(token, force) {
    this.elem.classList.toggle(token, force);
  }
  appendOrInsert(child, before) {
    if (before) {
      before.getElement().insertAdjacentElement("beforebegin", child);
    } else {
      this.elem.appendChild(child);
    }
  }
  addChildToDOM(child, before) {
    this.appendOrInsert(child.getElement(), before);
  }
  removeChildFromDOM(child) {
    this.elem.removeChild(child.getElement());
  }
  onChildAdded(_child) {
  }
  onChildRemoved(_child) {
  }
  getBefore({ domIndex }) {
    if (domIndex === void 0)
      return void 0;
    return this.children.filter((child) => child.domIndex !== void 0 && child.domIndex > domIndex).reduce((prev, curr) => !prev || curr.domIndex < prev.domIndex ? curr : prev, void 0);
  }
  addListener(type, listener) {
    if (WidgetEventUtil.isHTMLEvent(type)) {
      this.htmlListener ?? (this.htmlListener = new WidgetListenerHTML());
      this.htmlListener.add(type, this, listener);
    } else {
      this.internalListener ?? (this.internalListener = new WidgetListenerInternal(this.onDispatch.bind(this)));
      this.internalListener.add(type, this, listener);
    }
    return () => this.removeListener(type, listener);
  }
  removeListener(type, listener) {
    if (WidgetEventUtil.isHTMLEvent(type)) {
      this.htmlListener?.remove(type, this, listener);
    } else if (this.htmlListener != null) {
      this.internalListener?.remove(type, this, listener);
    }
  }
  setDragTouchEnabled(dragTouchEnabled) {
    this.internalListener ?? (this.internalListener = new WidgetListenerInternal(this.onDispatch.bind(this)));
    this.internalListener.dragTouchEnabled = dragTouchEnabled;
  }
  onDispatch(type, event) {
    if (!event.sourceEvent.bubbles)
      return;
    let { parent } = this;
    while (parent != null) {
      const { internalListener } = parent;
      if (internalListener != null) {
        const parentEvent = { ...event, ...WidgetEventUtil.calcCurrentXY(parent.getElement(), event) };
        internalListener.dispatch(type, parent, parentEvent);
      }
      parent = parent.parent;
    }
  }
  static addWindowEvent(_type, listener) {
    const pagehideHandler = (event) => {
      if (event.persisted) {
        return;
      }
      listener();
    };
    getWindow().addEventListener("pagehide", pagehideHandler);
    return () => getWindow().removeEventListener("pagehide", pagehideHandler);
  }
};

// packages/ag-charts-community/src/chart/layout/layoutManager.ts
var LayoutElement = /* @__PURE__ */ ((LayoutElement2) => {
  LayoutElement2[LayoutElement2["Caption"] = 0] = "Caption";
  LayoutElement2[LayoutElement2["Legend"] = 1] = "Legend";
  LayoutElement2[LayoutElement2["ToolbarLeft"] = 2] = "ToolbarLeft";
  LayoutElement2[LayoutElement2["ToolbarBottom"] = 3] = "ToolbarBottom";
  LayoutElement2[LayoutElement2["Navigator"] = 4] = "Navigator";
  LayoutElement2[LayoutElement2["Overlay"] = 5] = "Overlay";
  return LayoutElement2;
})(LayoutElement || {});
var LayoutManager = class {
  constructor() {
    this.events = new EventEmitter();
    this.elements = /* @__PURE__ */ new Map();
  }
  addListener(eventName, listener) {
    return this.events.on(eventName, listener);
  }
  registerElement(element2, listener) {
    if (this.elements.has(element2)) {
      this.elements.get(element2).add(listener);
    } else {
      this.elements.set(element2, /* @__PURE__ */ new Set([listener]));
    }
    return () => this.elements.get(element2)?.delete(listener);
  }
  createContext(width2, height2) {
    const context = new LayoutContext(width2, height2);
    for (const element2 of Object.values(LayoutElement)) {
      if (typeof element2 !== "number")
        continue;
      this.elements.get(element2)?.forEach((listener) => listener(context));
    }
    return context;
  }
  emitLayoutComplete(context, options) {
    const eventType = "layout:complete";
    const { width: width2, height: height2 } = context;
    this.events.emit(eventType, {
      type: eventType,
      axes: options.axes ?? [],
      chart: { width: width2, height: height2 },
      clipSeries: options.clipSeries ?? false,
      series: options.series
    });
  }
};
var LayoutContext = class {
  constructor(width2, height2) {
    this.width = width2;
    this.height = height2;
    this.layoutBox = new BBox(0, 0, width2, height2);
  }
};

// packages/ag-charts-community/src/chart/chartCaptions.ts
var ChartCaptions = class {
  constructor() {
    this.title = new Caption();
    this.subtitle = new Caption();
    this.footnote = new Caption();
  }
  positionCaptions(ctx) {
    const { title, subtitle, footnote } = this;
    const maxHeight = ctx.layoutBox.height / 10;
    if (title.enabled) {
      const { spacing = subtitle.enabled ? Caption.SMALL_PADDING : Caption.LARGE_PADDING } = title;
      this.positionCaption("top", title, ctx.layoutBox, maxHeight);
      this.shrinkLayoutByCaption("top", title, ctx.layoutBox, spacing);
    }
    if (subtitle.enabled) {
      this.positionCaption("top", subtitle, ctx.layoutBox, maxHeight);
      this.shrinkLayoutByCaption("top", subtitle, ctx.layoutBox, subtitle.spacing);
    }
    if (footnote.enabled) {
      this.positionCaption("bottom", footnote, ctx.layoutBox, maxHeight);
      this.shrinkLayoutByCaption("bottom", footnote, ctx.layoutBox, footnote.spacing);
    }
  }
  positionAbsoluteCaptions(ctx) {
    const { title, subtitle, footnote } = this;
    const { rect } = ctx.series;
    for (const caption of [title, subtitle, footnote]) {
      if (caption.layoutStyle !== "overlay")
        continue;
      if (caption.textAlign === "left") {
        caption.node.x = rect.x + caption.padding;
      } else if (caption.textAlign === "right") {
        const bbox = caption.node.getBBox();
        caption.node.x = rect.x + rect.width - bbox.width - caption.padding;
      }
    }
  }
  computeX(align2, layoutBox) {
    if (align2 === "left") {
      return layoutBox.x;
    } else if (align2 === "right") {
      return layoutBox.x + layoutBox.width;
    }
    return layoutBox.x + layoutBox.width / 2;
  }
  positionCaption(vAlign, caption, layoutBox, maxHeight) {
    const containerHeight = Math.max(TextUtils.getLineHeight(caption.fontSize), maxHeight);
    caption.node.x = this.computeX(caption.textAlign, layoutBox) + caption.padding;
    caption.node.y = layoutBox.y + (vAlign === "top" ? 0 : layoutBox.height) + caption.padding;
    caption.node.textBaseline = vAlign;
    caption.computeTextWrap(layoutBox.width, containerHeight);
  }
  shrinkLayoutByCaption(vAlign, caption, layoutBox, spacing = 0) {
    if (caption.layoutStyle === "block") {
      const bbox = caption.node.getBBox();
      layoutBox.shrink(
        vAlign === "top" ? Math.ceil(bbox.y - layoutBox.y + bbox.height + spacing) : Math.ceil(layoutBox.y + layoutBox.height - bbox.y + spacing),
        vAlign
      );
    }
  }
};
__decorateClass([
  Validate(OBJECT)
], ChartCaptions.prototype, "title", 2);
__decorateClass([
  Validate(OBJECT)
], ChartCaptions.prototype, "subtitle", 2);
__decorateClass([
  Validate(OBJECT)
], ChartCaptions.prototype, "footnote", 2);

// packages/ag-charts-community/src/api/preset/chartTypeOriginator.ts
var chartTypes = [
  "candlestick",
  "hollow-candlestick",
  "ohlc",
  "line",
  "step-line",
  "hlc",
  "high-low"
];
var ChartTypeOriginator = class {
  constructor(chartService) {
    this.chartService = chartService;
    this.mementoOriginatorKey = "chartType";
  }
  createMemento() {
    let chartType = this.chartService.publicApi?.getOptions()?.chartType;
    if (chartType == null)
      chartType = "candlestick";
    return chartType;
  }
  guardMemento(blob) {
    return blob == null || chartTypes.includes(blob);
  }
  restoreMemento(_version, _mementoVersion, memento) {
    if (memento == null)
      return;
    const options = { chartType: memento };
    this.chartService.publicApi?.updateDelta(options).catch((e) => logger_exports.error("error restoring state", e));
  }
};

// packages/ag-charts-community/src/util/destroy.ts
var DestroyFns = class {
  constructor() {
    this.destroyFns = [];
  }
  destroy() {
    this.destroyFns.forEach((fn) => fn());
    this.destroyFns.length = 0;
  }
  setFns(destroyFns) {
    this.destroy();
    this.destroyFns = destroyFns;
  }
  push(...destroyFns) {
    this.destroyFns.push(...destroyFns);
  }
};

// packages/ag-charts-community/src/version.ts
var VERSION = "11.1.1";

// packages/ag-charts-community/src/api/state/historyManager.ts
var NOT_FOUND = Symbol("previous-memento-not-found");
var HistoryManager = class {
  constructor(chartEventManager) {
    this.history = [];
    this.historyIndex = -1;
    this.originators = /* @__PURE__ */ new Map();
    this.clearState = /* @__PURE__ */ new Map();
    this.maxHistoryLength = 100;
    this.debug = Debug.create(true, "history");
    this.destroyFns = new DestroyFns();
    this.destroyFns.setFns([
      chartEventManager.addListener("series-undo", this.undo.bind(this)),
      chartEventManager.addListener("series-redo", this.redo.bind(this))
    ]);
  }
  destroy() {
    this.destroyFns.destroy();
  }
  addMementoOriginator(originator) {
    this.originators.set(originator.mementoOriginatorKey, originator);
    this.clearState.set(originator.mementoOriginatorKey, originator.createMemento());
    this.debugEvent("History add originator:", originator.mementoOriginatorKey);
  }
  clear() {
    this.debug(`History clear:`, Object.keys(this.originators));
    this.history = [];
    this.historyIndex = -1;
    for (const [mementoOriginatorKey, originator] of this.originators.entries()) {
      this.clearState.set(mementoOriginatorKey, originator.createMemento());
    }
  }
  record(label, ...originators) {
    if (this.historyIndex < this.history.length - 1) {
      this.history = this.history.slice(0, this.historyIndex + 1);
    }
    if (this.history.length > this.maxHistoryLength) {
      this.history = this.history.slice(-this.maxHistoryLength);
    }
    const mementos = /* @__PURE__ */ new Map();
    for (const originator of originators) {
      if (!this.originators.has(originator.mementoOriginatorKey)) {
        throw new Error(
          `Originator [${originator.mementoOriginatorKey}] has not been added to the HistoryManager.`
        );
      }
      mementos.set(originator.mementoOriginatorKey, originator.createMemento());
    }
    this.history.push({ label, mementos });
    this.historyIndex = this.history.length - 1;
    this.debugEvent(`History record: [${label}]`);
  }
  undo() {
    const undoAction = this.history[this.historyIndex];
    if (!undoAction)
      return;
    for (const mementoOriginatorKey of undoAction.mementos.keys()) {
      const previousMemento = this.findPreviousMemento(mementoOriginatorKey);
      if (previousMemento === NOT_FOUND) {
        throw new Error(`Could not find previous memento for [${mementoOriginatorKey}].`);
      }
      this.restoreMemento(mementoOriginatorKey, previousMemento);
    }
    this.historyIndex -= 1;
    this.debugEvent(`History undo: [${undoAction.label}]`);
  }
  redo() {
    const redoAction = this.history[this.historyIndex + 1];
    if (!redoAction)
      return;
    for (const [mementoOriginatorKey, memento] of redoAction.mementos.entries()) {
      this.restoreMemento(mementoOriginatorKey, memento);
    }
    this.historyIndex += 1;
    this.debugEvent(`History redo: [${redoAction.label}]`);
  }
  findPreviousMemento(mementoOriginatorKey) {
    for (let i = this.historyIndex - 1; i >= 0; i--) {
      if (this.history[i].mementos.has(mementoOriginatorKey)) {
        return this.history[i].mementos.get(mementoOriginatorKey);
      }
    }
    if (this.clearState.has(mementoOriginatorKey)) {
      return this.clearState.get(mementoOriginatorKey);
    }
    return NOT_FOUND;
  }
  restoreMemento(mementoOriginatorKey, memento) {
    this.originators.get(mementoOriginatorKey)?.restoreMemento(VERSION, VERSION, memento);
  }
  debugEvent(...logContent) {
    this.debug(
      ...logContent,
      this.history.map((action, index) => index === this.historyIndex ? `** ${action.label} **` : action.label)
    );
  }
};

// packages/ag-charts-community/src/api/state/memento.ts
var MementoCaretaker = class _MementoCaretaker {
  constructor(version) {
    this.version = version.split("-")[0];
  }
  save(...originators) {
    const packet = { version: this.version };
    for (const originator of Object.values(originators)) {
      packet[originator.mementoOriginatorKey] = this.encode(originator, originator.createMemento());
    }
    return packet;
  }
  restore(blob, ...originators) {
    if (typeof blob !== "object") {
      logger_exports.warnOnce(`Could not restore data of type [${typeof blob}], expecting an object, ignoring.`);
      return;
    }
    if (blob == null) {
      logger_exports.warnOnce(`Could not restore data of type [null], expecting an object, ignoring.`);
      return;
    }
    if (!("version" in blob) || typeof blob.version !== "string") {
      logger_exports.warnOnce(`Could not restore data, missing [version] string in object, ignoring.`);
      return;
    }
    for (const originator of originators) {
      const memento = this.decode(originator, blob[originator.mementoOriginatorKey]);
      const messages = [];
      if (!originator.guardMemento(memento, messages)) {
        const messagesString = messages.length > 0 ? `

${messages.join("\n\n")}

` : "";
        logger_exports.warnOnce(
          `Could not restore [${originator.mementoOriginatorKey}] data, value was invalid, ignoring.${messagesString}`,
          memento
        );
        return;
      }
      originator.restoreMemento(this.version, blob.version, memento);
    }
  }
  /**
   * Encode a memento as a serializable object, encoding any non-serializble types.
   */
  encode(originator, memento) {
    try {
      return JSON.parse(JSON.stringify(memento, _MementoCaretaker.encodeTypes));
    } catch (error2) {
      throw new Error(`Failed to encode [${originator.mementoOriginatorKey}] value [${error2}].`, {
        cause: error2
      });
    }
  }
  /**
   * Decode an encoded memento, decoding any non-serializable types.
   */
  decode(originator, encoded) {
    if (encoded == null)
      return encoded;
    try {
      return JSON.parse(JSON.stringify(encoded), _MementoCaretaker.decodeTypes);
    } catch (error2) {
      throw new Error(`Failed to decode [${originator.mementoOriginatorKey}] value [${error2}].`, {
        cause: error2
      });
    }
  }
  static encodeTypes(key, value) {
    if (isDate(this[key])) {
      return { __type: "date", value: this[key].toISOString() };
    }
    return value;
  }
  static decodeTypes(key, value) {
    if (isObject(this[key]) && "__type" in this[key] && this[key].__type === "date") {
      return new Date(this[key].value);
    }
    return value;
  }
};

// packages/ag-charts-community/src/api/state/stateManager.ts
var StateManager = class {
  constructor() {
    this.caretaker = new MementoCaretaker(VERSION);
    this.state = /* @__PURE__ */ new Map();
  }
  setState(originator, value) {
    if (objectsEqual(this.state.get(originator.mementoOriginatorKey), value)) {
      return;
    }
    this.setStateAndRestore(originator, value);
  }
  setStateAndRestore(originator, value) {
    this.state.set(originator.mementoOriginatorKey, value);
    this.restoreState(originator);
  }
  restoreState(originator) {
    const { caretaker, state } = this;
    if (!state.has(originator.mementoOriginatorKey))
      return;
    const value = state.get(originator.mementoOriginatorKey);
    caretaker.restore({ version: caretaker.version, [originator.mementoOriginatorKey]: value }, originator);
  }
};

// packages/ag-charts-community/src/styles.css
var styles_default = '.ag-charts-wrapper,.ag-charts-wrapper:after,.ag-charts-wrapper:before,.ag-charts-wrapper *,.ag-charts-wrapper *:after,.ag-charts-wrapper *:before{box-sizing:border-box}.ag-charts-wrapper{--align-items: center;--justify-content: center;position:relative;user-select:none;-webkit-user-select:none;-webkit-tap-highlight-color:rgba(0,0,0,0)}.ag-charts-wrapper--safe-horizontal{--justify-content: flex-start}.ag-charts-wrapper--safe-vertical{--align-items: flex-start}.ag-charts-tab-guard{width:0%;height:0%;position:absolute;pointer-events:none}.ag-charts-canvas-center{width:100%;height:100%;position:absolute;touch-action:auto;pointer-events:auto;display:flex;align-items:var(--align-items);justify-content:var(--justify-content)}.ag-charts-canvas-container,.ag-charts-canvas{position:relative;user-select:none;-webkit-user-select:none}.ag-charts-canvas-container>*,.ag-charts-canvas>*{display:block;pointer-events:none}.ag-charts-series-area{outline:none;pointer-events:auto;position:absolute}.ag-charts-swapchain{outline:none;opacity:0;pointer-events:none;position:absolute;width:100%;height:100%}.ag-charts-swapchain:focus-visible{opacity:1}.ag-charts-canvas-proxy,.ag-charts-canvas-overlay{inset:0;pointer-events:none;position:absolute;user-select:none;-webkit-user-select:none}.ag-charts-canvas-overlay>*{position:absolute;pointer-events:auto}.ag-charts-theme-default,.ag-charts-theme-default-dark{--ag-charts-accent-color: #2196f3;--ag-charts-background-color: #fff;--ag-charts-border-color: #dddddd;--ag-charts-foreground-color: #464646;--ag-charts-font-family: Verdana, sans-serif;--ag-charts-font-size: 12px;--ag-charts-font-weight: 400;--ag-charts-padding: 20px;--ag-charts-shadow-color: #00000080;--ag-charts-subtle-text-color: #8c8c8c;--ag-charts-text-color: #464646;--ag-charts-chrome-background-color: #fafafa;--ag-charts-chrome-font-family: Verdana, sans-serif;--ag-charts-chrome-font-size: 12px;--ag-charts-chrome-font-weight: 400;--ag-charts-chrome-subtle-text-color: #8c8c8c;--ag-charts-chrome-text-color: #181d1f;--ag-charts-input-background-color: #fff;--ag-charts-input-text-color: #464646;--ag-charts-crosshair-label-background-color: #fff;--ag-charts-crosshair-label-text-color: #464646;--ag-charts-spacing: 4px;--ag-charts-border-radius: 4px;--ag-charts-shadow: 0 2px 8px 0 color-mix(in srgb, black 8%, transparent);--ag-charts-focus-shadow: 0 0 0 3px var(--ag-charts-accent-color);--ag-charts-focus-color: color-mix(in srgb, var(--ag-charts-background-color), var(--ag-charts-accent-color) 12%);--ag-charts-input-border-color: var(--ag-charts-border-color);--ag-charts-input-border-radius: var(--ag-charts-border-radius);--ag-charts-input-focus-border-color: var(--ag-charts-accent-color);--ag-charts-input-focus-text-color: var(--ag-charts-accent-color);--ag-charts-input-disabled-background-color: color-mix( in srgb, var(--ag-charts-chrome-background-color), var(--ag-charts-foreground-color) 6% );--ag-charts-input-disabled-border-color: var(--ag-charts-border-color);--ag-charts-input-disabled-text-color: color-mix( in srgb, var(--ag-charts-chrome-background-color), var(--ag-charts-input-text-color) 50% );--ag-charts-input-placeholder-text-color: color-mix( in srgb, var(--ag-charts-input-background-color), var(--ag-charts-input-text-color) 60% );--ag-charts-button-background-color: var(--ag-charts-background-color);--ag-charts-button-border-color: var(--ag-charts-border-color);--ag-charts-button-border-radius: var(--ag-charts-border-radius);--ag-charts-button-text-color: var(--ag-charts-text-color);--ag-charts-button-focus-background-color: color-mix( in srgb, var(--ag-charts-button-background-color), var(--ag-charts-accent-color) 12% );--ag-charts-button-focus-border-color: var(--ag-charts-accent-color);--ag-charts-button-focus-text-color: var(--ag-charts-accent-color);--ag-charts-button-disabled-background-color: color-mix( in srgb, var(--ag-charts-chrome-background-color), var(--ag-charts-foreground-color) 6% );--ag-charts-button-disabled-border-color: var(--ag-charts-border-color);--ag-charts-button-disabled-text-color: color-mix( in srgb, var(--ag-charts-chrome-background-color), var(--ag-charts-chrome-text-color) 50% );--ag-charts-checkbox-background-color: color-mix( in srgb, var(--ag-charts-background-color), var(--ag-charts-foreground-color) 35% );--ag-charts-checkbox-checked-background-color: var(--ag-charts-accent-color);--ag-charts-chrome-font-size-small: var(--ag-charts-chrome-font-size);--ag-charts-chrome-font-size-medium: calc(var(--ag-charts-chrome-font-size) * (13 / 12));--ag-charts-chrome-font-size-large: calc(var(--ag-charts-chrome-font-size) * (14 / 12));--ag-charts-border: solid 1px var(--ag-charts-border-color);--ag-charts-focus-border: solid 1px var(--ag-charts-accent-color);--ag-charts-focus-border-shadow: 0 0 0 3px color-mix(in srgb, transparent, var(--ag-charts-accent-color) 20%);--ag-charts-layer-menu: 6;--ag-charts-layer-ui-overlay: 5;--ag-charts-layer-tooltip: 4;--ag-charts-layer-toolbar: 3;--ag-charts-layer-crosshair: 2;--ag-charts-layer-annotations: 1}.ag-charts-theme-default-dark{--ag-charts-shadow: 0 2px 12px 0 color-mix(in srgb, black 33.3%, transparent);--ag-charts-focus-color: color-mix(in srgb, var(--ag-charts-background-color), var(--ag-charts-accent-color) 22%)}.ag-chart-canvas-wrapper .ag-charts-theme-default{--ag-charts-border-radius: var(--ag-border-radius, 4px);--ag-charts-border: var(--ag-borders-critical, solid 1px) var(--ag-charts-border-color);--ag-charts-focus-shadow: var(--ag-focus-shadow, 0 0 0 3px var(--ag-charts-accent-color));--ag-charts-focus-border-shadow: var( --ag-focus-shadow, 0 0 0 3px color-mix(in srgb, transparent, var(--ag-charts-accent-color) 20%) )}.ag-charts-icon{display:block;width:20px;height:20px;speak:none;speak:never;mask:var(--icon) center / contain no-repeat;background-color:currentColor;transition:background-color .25s ease-in-out}.ag-charts-icon-align-center{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsPSIjMDAwIiBkPSJNNyAxMGg2djFIN3pNNCA3aDEydjFINHptMSA2aDEwdjFINXoiLz48L3N2Zz4=)}.ag-charts-icon-align-left{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsPSIjMDAwIiBkPSJNNCAxMGg2djFINHptMC0zaDEydjFINHptMCA2aDEwdjFINHoiLz48L3N2Zz4=)}.ag-charts-icon-align-right{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsPSIjMDAwIiBkPSJNMTAgMTBoNnYxaC02ek00IDdoMTJ2MUg0em0yIDZoMTB2MUg2eiIvPjwvc3ZnPg==)}.ag-charts-icon-arrow-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsLXJ1bGU9ImV2ZW5vZGQiIGNsaXAtcnVsZT0iZXZlbm9kZCIgZD0iTTE1LjI5MyA0LjVIMTIuNXYtMUgxN3Y0aC0xVjUuMjA3bC05LjY0NiA5LjY0Ny0uNzA4LS43MDh6IiBmaWxsPSIjMDAwIi8+PHBhdGggZmlsbC1ydWxlPSJldmVub2RkIiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik03IDE2YTIuNSAyLjUgMCAxIDEtNSAwIDIuNSAyLjUgMCAwIDEgNSAwbS0yLjUgMS41YTEuNSAxLjUgMCAxIDAgMC0zIDEuNSAxLjUgMCAwIDAgMCAzIiBmaWxsPSIjMDAwIi8+PC9zdmc+)}.ag-charts-icon-arrow-down-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2RkIiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik02IDhMMS41IDhMMTAgMThMMTguNSA4TDE0IDhMMTQgM0w2IDNMNiA4Wk03IDRMNyA5SDMuNjYyNDRMMTAgMTYuNDU2TDE2LjMzNzYgOUwxMyA5TDEzIDRMNyA0WiIgZmlsbD0iYmxhY2siLz4KPC9zdmc+Cg==)}.ag-charts-icon-arrow-up-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2RkIiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik0xNCAxMkgxOC41TDEwIDJMMS41IDEySDZMNi4wMDAwMiAxN0gxNFYxMlpNMTMgMTZWMTFIMTYuMzM3NkwxMCAzLjU0NDA1TDMuNjYyNDQgMTFIN0w3LjAwMDAyIDE2SDEzWiIgZmlsbD0iYmxhY2siLz4KPC9zdmc+Cg==)}.ag-charts-icon-callout-annotation{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMyA0LjVBMS41IDEuNSAwIDAgMSA0LjUgM2gxMUExLjUgMS41IDAgMCAxIDE3IDQuNXY4YTEuNSAxLjUgMCAwIDEtMS41IDEuNWgtNC41MTRhMjYgMjYgMCAwIDAtMi4wMTcgMS41NGwtLjMxNC4yNmMtLjU1LjQ1Ny0xLjExNS45MjYtMS43NiAxLjQtLjY2OS40OTEtMS41NjItLjAxMi0xLjU2Mi0uOFYxNEg0LjVBMS41IDEuNSAwIDAgMSAzIDEyLjV6TTQuNSA0YS41LjUgMCAwIDAtLjUuNXY4YS41LjUgMCAwIDAgLjUuNWgxLjgzM3YzLjM3MmEzNiAzNiAwIDAgMCAxLjY3OC0xLjMzOGwuMzItLjI2NWEyNiAyNiAwIDAgMSAyLjIyNS0xLjY4NWwuMTI2LS4wODRIMTUuNWEuNS41IDAgMCAwIC41LS41di04YS41LjUgMCAwIDAtLjUtLjV6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=)}.ag-charts-icon-candlestick-series{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNyAxdjNoMnYxMkg3djNINnYtM0g0VjRoMlYxek01IDVoM3YxMEg1ek0xMSAxNFY2aDJWMy4yNWgxVjZoMnY4aC0ydjIuNzVoLTFWMTR6bTEtN2gzdjZoLTN6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=)}.ag-charts-icon-close{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBkPSJtNSA1IDEwIDEwTTUgMTUgMTUgNSIgc3Ryb2tlPSIjMDAwIi8+PC9zdmc+)}.ag-charts-icon-comment-annotation{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNy41MTMgMy45OTVhNi41IDYuNSAwIDAgMSA2LjA5OCAxMS40MWMtLjU4OC4zOTMtMS4yMTcuNTM2LTEuODI5LjU4NWExMyAxMyAwIDAgMS0xLjI3LjAxN0EyNyAyNyAwIDAgMCAxMCAxNkg0LjVhLjUuNSAwIDAgMS0uNS0uNVYxMHEwLS4yNDctLjAwNy0uNTEzYy0uMDA4LS40MTYtLjAxNi0uODU3LjAxNy0xLjI2OS4wNS0uNjEyLjE5Mi0xLjI0LjU4NS0xLjgzYTYuNSA2LjUgMCAwIDEgMi45MTgtMi4zOTNtMy41Ni42MWE1LjUgNS41IDAgMCAwLTUuNjQ2IDIuMzRjLS4yNjYuMzk3LS4zNzkuODQyLS40MiAxLjM1NC0uMDMuMzYtLjAyMi43MTgtLjAxNSAxLjEwOFE1IDkuNjg5IDUgMTB2NWg1cS4zMTEuMDAxLjU5My4wMDhjLjM5LjAwNy43NDcuMDE1IDEuMTA4LS4wMTUuNTEyLS4wNDEuOTU3LS4xNTQgMS4zNTUtLjQyYTUuNSA1LjUgMCAwIDAtMS45ODMtOS45NjciIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==)}.ag-charts-icon-crosshair-add-line{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZD0iTTEwIDUuNWEuNS41IDAgMCAxIC41LjV2My41aDMuODc1YS41LjUgMCAwIDEgMCAxSDEwLjV2NC4yNWEuNS41IDAgMSAxLTEgMFYxMC41SDUuNjI1YS41LjUgMCAxIDEgMC0xSDkuNVY2YS41LjUgMCAwIDEgLjUtLjUiLz48L3N2Zz4=)}.ag-charts-icon-date-range-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBkPSJNMiAyaDF2MTZIMnptMTUgMGgxdjE2aC0xeiIgZmlsbD0iIzE4MUQxRiIvPjxwYXRoIGQ9Ik0xMy4xNTcgMTFINXYtMWg3Ljc5M0wxMSA4LjIwN2wuNzA3LS43MDcgMy4xODIgMy4xODItMy4xODIgMy4xODItLjcwNy0uNzA3eiIgZmlsbD0iIzAwMCIvPjwvc3ZnPg==)}.ag-charts-icon-date-price-range-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBkPSJNMyAySDJ2MTZoMXptMy41MDcgNC44OUw4LjUgNC44OTVWMTBINXYxaDMuNXY3aDF2LTdoNS4wODhsLTEuOTU3IDEuOTU3LjcwNy43MDcgMy4xODItMy4xODJMMTMuMzM4IDcuM2wtLjcwNy43MDdMMTQuNjI0IDEwSDkuNVY0LjkzMmwxLjk1NyAxLjk1Ny43MDctLjcwN0w4Ljk4MiAzIDUuOCA2LjE4MnoiIGZpbGw9IiMxODFEMUYiLz48L3N2Zz4=)}.ag-charts-icon-delete{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZD0iTTguNDk2IDguOTk2QS41LjUgMCAwIDEgOSA5LjQ5MnY0YS41LjUgMCAxIDEtMSAuMDA4di00YS41LjUgMCAwIDEgLjQ5Ni0uNTA0TTEyIDkuNWEuNS41IDAgMCAwLTEgMHY0YS41LjUgMCAwIDAgMSAweiIvPjxwYXRoIGZpbGw9IiMxMzE3MjIiIGZpbGwtcnVsZT0iZXZlbm9kZCIgZD0iTTYgNVYzLjVBMi41IDIuNSAwIDAgMSA4LjUgMWgzQTIuNSAyLjUgMCAwIDEgMTQgMy41VjVoMi44MzNhLjUuNSAwIDAgMSAwIDFIMTV2MTAuMjVjMCAuNDE1LS4wNjYuODYzLS4zIDEuMjIxLS4yNTcuMzk0LS42NzIuNjEyLTEuMi42MTJoLTdjLS41MjggMC0uOTQzLS4yMTgtMS4yLS42MTItLjIzNC0uMzU4LS4zLS44MDYtLjMtMS4yMjFWNkgzLjMzM2EuNS41IDAgMCAxIDAtMXptMS0xLjVBMS41IDEuNSAwIDAgMSA4LjUgMmgzQTEuNSAxLjUgMCAwIDEgMTMgMy41VjVIN3pNNiAxNi4yNVY2aDh2MTAuMjVjMCAuMzM1LS4wNTkuNTU0LS4xMzguNjc1LS4wNTUuMDg1LS4xNC4xNTgtLjM2Mi4xNThoLTdjLS4yMjIgMC0uMzA3LS4wNzMtLjM2Mi0uMTU4LS4wOC0uMTIxLS4xMzgtLjM0LS4xMzgtLjY3NSIgY2xpcC1ydWxlPSJldmVub2RkIi8+PC9zdmc+)}.ag-charts-icon-disjoint-channel,.ag-charts-icon-disjoint-channel-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMTkuMDI4IDE3LjQ2YTIuMjUgMi4yNSAwIDAgMC00LjA5Mi0xLjg1bC05LjUxMS0yLjM3OGEyLjI1IDIuMjUgMCAxIDAtLjIyNS45NzRsOS40NzUgMi4zNjlhMi4yNTEgMi4yNTEgMCAwIDAgNC4zNTMuODg2bS0xLjY2Mi0xLjk2NWExLjI1IDEuMjUgMCAxIDEtLjg4NSAyLjMzOCAxLjI1IDEuMjUgMCAwIDEgLjg4NS0yLjMzOE00LjM0MyAxMy42NjlhMS4yNSAxLjI1IDAgMSAwLTIuMzM4LS44ODUgMS4yNSAxLjI1IDAgMCAwIDIuMzM4Ljg4NU0zLjk3IDguNzY5YTIuMjUgMi4yNSAwIDAgMCAxLjQ1NS0yLjExbDkuNTExLTIuMzc4YTIuMjUgMi4yNSAwIDEgMC0uMjYtLjk2NUw1LjIgNS42ODVhMi4yNSAyLjI1IDAgMSAwLTEuMjMgMy4wODRtLjM3My0yLjU0N2ExLjI1IDEuMjUgMCAxIDEtMi4zMzguODg1IDEuMjUgMS4yNSAwIDAgMSAyLjMzOC0uODg1bTEzLjc1LTMuNDM4YTEuMjUgMS4yNSAwIDEgMS0yLjMzOC44ODUgMS4yNSAxLjI1IDAgMCAxIDIuMzM4LS44ODUiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==)}.ag-charts-icon-drag-handle{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48Y2lyY2xlIGN4PSI1Ljc1IiBjeT0iNy43NSIgcj0iLjc1IiBmaWxsPSIjMDAwIiBmaWxsLW9wYWNpdHk9Ii41Ii8+PGNpcmNsZSBjeD0iOS43NSIgY3k9IjcuNzUiIHI9Ii43NSIgZmlsbD0iIzAwMCIgZmlsbC1vcGFjaXR5PSIuNSIvPjxjaXJjbGUgY3g9IjEzLjc1IiBjeT0iNy43NSIgcj0iLjc1IiBmaWxsPSIjMDAwIiBmaWxsLW9wYWNpdHk9Ii41Ii8+PGNpcmNsZSBjeD0iMTMuNzUiIGN5PSIxMS43NSIgcj0iLjc1IiBmaWxsPSIjMDAwIiBmaWxsLW9wYWNpdHk9Ii41Ii8+PGNpcmNsZSBjeD0iOS43NSIgY3k9IjExLjc1IiByPSIuNzUiIGZpbGw9IiMwMDAiIGZpbGwtb3BhY2l0eT0iLjUiLz48Y2lyY2xlIGN4PSI1Ljc1IiBjeT0iMTEuNzUiIHI9Ii43NSIgZmlsbD0iIzAwMCIgZmlsbC1vcGFjaXR5PSIuNSIvPjwvc3ZnPg==)}.ag-charts-icon-fibonacci-retracement-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsPSIjMDAwIiBkPSJNMiA1aDEydjFIMnoiLz48Y2lyY2xlIGN4PSIxNS43NSIgY3k9IjUuNSIgcj0iMS43NSIgc3Ryb2tlPSIjMDAwIi8+PGNpcmNsZSBjeD0iNC4yNSIgY3k9IjE0LjUiIHI9IjEuNzUiIHN0cm9rZT0iIzAwMCIvPjxwYXRoIGZpbGw9IiMwMDAiIGQ9Ik0xOCAxNUg2di0xaDEyem0wLTQuNUgydi0xaDE2eiIvPjwvc3ZnPg==)}.ag-charts-icon-fibonacci-retracement-trend-based-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsPSIjMDAwIiBkPSJtNC45OTYgMTIuNjc0IDMuMjkxLTUuNzQzLjg2OC40OTctMy4yOTEgNS43NDN6Ii8+PGNpcmNsZSBjeD0iOS43NSIgY3k9IjUuNSIgcj0iMS43NSIgc3Ryb2tlPSIjMDAwIi8+PGNpcmNsZSBjeD0iNC4zNTEiIGN5PSIxNC41IiByPSIxLjc1IiBzdHJva2U9IiMwMDAiLz48cGF0aCBmaWxsPSIjMDAwIiBkPSJNMTggNmgtN1Y1aDd6bTAgNC41aC03di0xaDd6bTAgNC41SDZ2LTFoMTJ6Ii8+PC9zdmc+)}.ag-charts-icon-fill-color{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJtOC4wNzEgNC4wNi0uOTI0LS45MjQuNzA3LS43MDcgNy4yODggNy4yODgtNC45NSA0Ljk1YTMuNSAzLjUgMCAwIDEtNC45NSAwbC0xLjQxNC0xLjQxNGEzLjUgMy41IDAgMCAxIDAtNC45NXptLjcwNy43MDhMNC41MzYgOS4wMWEyLjUgMi41IDAgMCAwIDAgMy41MzZMNS45NSAxMy45NmEyLjUgMi41IDAgMCAwIDMuNTM1IDBsNC4yNDMtNC4yNDN6bTYuOSA3LjIwMi0uMzQ1LjM2My0uMzQ0LS4zNjNhLjUuNSAwIDAgMSAuNjg4IDBtLS4zNDUgMS4wOGE4IDggMCAwIDAtLjI4LjMyMyA0LjMgNC4zIDAgMCAwLS40MDkuNTgyYy0uMTEzLjIwMS0uMTQ0LjMyNi0uMTQ0LjM3OGEuODMzLjgzMyAwIDAgMCAxLjY2NyAwYzAtLjA1Mi0uMDMxLS4xNzctLjE0NC0uMzc4YTQuMyA0LjMgMCAwIDAtLjQxLS41ODIgOCA4IDAgMCAwLS4yOC0uMzIybS0uMzQ0LTEuMDguMzQ0LjM2My4zNDQtLjM2My4wMDIuMDAyLjAwNC4wMDQuMDEzLjAxMmE2IDYgMCAwIDEgLjIwNi4yMDhjLjEzMS4xMzYuMzA4LjMyNy40ODUuNTQ1LjE3Ni4yMTUuMzYzLjQ2Ny41MDcuNzI0LjEzNy4yNDMuMjczLjU1My4yNzMuODY4YTEuODMzIDEuODMzIDAgMSAxLTMuNjY3IDBjMC0uMzE1LjEzNi0uNjI1LjI3My0uODY4LjE0NC0uMjU3LjMzLS41MDkuNTA3LS43MjRhOSA5IDAgMCAxIC42NDUtLjcwOGwuMDQ2LS4wNDUuMDEzLS4wMTIuMDA0LS4wMDR6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=)}.ag-charts-icon-hollow-candlestick-series{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZmlsbC1vcGFjaXR5PSIuMTUiIGQ9Ik01IDVoM3YxMEg1eiIvPjxwYXRoIGZpbGw9IiMxMzE3MjIiIGZpbGwtcnVsZT0iZXZlbm9kZCIgZD0iTTcgMXYzaDJ2MTJIN3YzSDZ2LTNINFY0aDJWMXpNNSA1aDN2MTBINXptNyAyaDN2NmgtM3ptLTEgN1Y2aDJWMy4yNWgxVjZoMnY4aC0ydjIuNzVoLTFWMTR6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=)}.ag-charts-icon-horizontal-line,.ag-charts-icon-horizontal-line-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNLjUgOS41aDcuMzA2YTIuMjUgMi4yNSAwIDAgMSA0LjM4OCAwSDE5LjV2MWgtNy4zMDZhMi4yNSAyLjI1IDAgMCAxLTQuMzg4IDBILjV6bTkuNSAxLjc1YTEuMjUgMS4yNSAwIDEgMCAwLTIuNSAxLjI1IDEuMjUgMCAwIDAgMCAyLjUiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==)}.ag-charts-icon-line-color{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMTQuMjQyIDIuNzIyYy0uNjEyIDAtMS4yLjI0My0xLjYzMi42NzVsLTEuMzQzIDEuMzQ0YS41LjUgMCAwIDAtLjExMi4xMTJMNC4wNSAxMS45NTljLS4yMDcuMjA3LS4zNi40Ni0uNDQ2Ljc0di4wMDFsLS42OSAyLjc2N3YuMDAyYS44Mi44MiAwIDAgMCAxLjAyMiAxLjAyMWguMDAybDIuNjM0LS44MjJjLjI4LS4wODUuNTM0LS4yMzcuNzQtLjQ0M2w3LjEwNy03LjEwOGEuNS41IDAgMCAwIC4xMTItLjExMmwxLjM0My0xLjM0M2EyLjMwOCAyLjMwOCAwIDAgMC0xLjYzMi0zLjk0TTE0LjEyMiA3bDEuMDQ0LTEuMDQ1YTEuMzA4IDEuMzA4IDAgMSAwLTEuODQ5LTEuODVMMTIuMjcxIDUuMTV6bS0yLjU1OC0xLjE0Mi02LjgwNyA2LjgwOWEuOC44IDAgMCAwLS4xOTYuMzI1bC0uNzUgMi40NjggMi40Ny0uNzQ5YS44LjggMCAwIDAgLjMyNS0uMTk0bDYuODA4LTYuODF6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=)}.ag-charts-icon-line-series{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJtMTcuMzYyIDQuODczLTQuNTk0IDYuNjU0LTQuODUtMy4zMTctNC4yNTEgNi45NzctLjg1NC0uNTJMNy42MTIgNi43OWw0Ljg5OSAzLjM1IDQuMDI4LTUuODM2eiIgY2xpcC1ydWxlPSJldmVub2RkIi8+PC9zdmc+)}.ag-charts-icon-line-style-dashed{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsPSIjMDAwIiBkPSJNMiA5aDR2MUgyem0xMiAwaDR2MWgtNHpNOCA5aDR2MUg4eiIvPjwvc3ZnPg==)}.ag-charts-icon-line-style-dotted{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48Y2lyY2xlIGN4PSIyLjUiIGN5PSI5LjUiIHI9Ii41IiBmaWxsPSIjMDAwIi8+PGNpcmNsZSBjeD0iNC41IiBjeT0iOS41IiByPSIuNSIgZmlsbD0iIzAwMCIvPjxjaXJjbGUgY3g9IjYuNSIgY3k9IjkuNSIgcj0iLjUiIGZpbGw9IiMwMDAiLz48Y2lyY2xlIGN4PSI4LjUiIGN5PSI5LjUiIHI9Ii41IiBmaWxsPSIjMDAwIi8+PGNpcmNsZSBjeD0iMTAuNSIgY3k9IjkuNSIgcj0iLjUiIGZpbGw9IiMwMDAiLz48Y2lyY2xlIGN4PSIxMi41IiBjeT0iOS41IiByPSIuNSIgZmlsbD0iIzAwMCIvPjxjaXJjbGUgY3g9IjE0LjUiIGN5PSI5LjUiIHI9Ii41IiBmaWxsPSIjMDAwIi8+PGNpcmNsZSBjeD0iMTYuNSIgY3k9IjkuNSIgcj0iLjUiIGZpbGw9IiMwMDAiLz48L3N2Zz4=)}.ag-charts-icon-line-style-solid{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsPSIjMDAwIiBkPSJNMiA5aDE2djFIMnoiLz48L3N2Zz4=)}.ag-charts-icon-line-with-markers-series{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJtMTguMTk4IDQuODg4LTMuNTU2IDQuOTE4YTIuMjUgMi4yNSAwIDEgMS0zLjg2Ni43NWwtMS40MzItLjlhMi4yNCAyLjI0IDAgMCAxLTIuMDA5LjQzNWwtMy44MjggNi40MjgtLjg2LS41MTJMNi40NSA5LjYyM2EyLjI1IDIuMjUgMCAxIDEgMy41MS0uNzYxbDEuMzI5LjgzNWEyLjI0IDIuMjQgMCAwIDEgMi41NTctLjQ5N2wzLjU0Mi00Ljg5OHptLTQuOTYgNS4xNTNhMS4yNSAxLjI1IDAgMSAwLS42NCAyLjQxOSAxLjI1IDEuMjUgMCAwIDAgLjY0LTIuNDE5TTkuMSA4LjMyMXEuMDY2LS4xOTIuMDY3LS40MDRhMS4yNSAxLjI1IDAgMSAwLS4wNjcuNDA0IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=)}.ag-charts-icon-lock,.ag-charts-icon-locked{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMTAuMjA3IDMuNzY0YTIuODk0IDIuODk0IDAgMCAwLTIuODk1IDIuODk0VjloNS43ODlWNi42NThhMi44OTQgMi44OTQgMCAwIDAtMi44OTUtMi44OTRNMTQuMSA5VjYuNjU4YTMuODk0IDMuODk0IDAgMSAwLTcuNzg5IDB2Mi4zNDlBMi41IDIuNSAwIDAgMCA0IDExLjV2M0EyLjUgMi41IDAgMCAwIDYuNSAxN2g4YTIuNSAyLjUgMCAwIDAgMi41LTIuNXYtM0EyLjUgMi41IDAgMCAwIDE0LjUgOXpNNi41IDEwQTEuNSAxLjUgMCAwIDAgNSAxMS41djNBMS41IDEuNSAwIDAgMCA2LjUgMTZoOGExLjUgMS41IDAgMCAwIDEuNS0xLjV2LTNhMS41IDEuNSAwIDAgMC0xLjUtMS41eiIgY2xpcC1ydWxlPSJldmVub2RkIi8+PC9zdmc+)}.ag-charts-icon-measurer-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsLXJ1bGU9ImV2ZW5vZGQiIGNsaXAtcnVsZT0iZXZlbm9kZCIgZD0ibTQuNDYxIDEyLjcxIDEuNTMyLTEuNTMxIDEuNDE0IDEuNDE0LjcwNy0uNzA3TDYuNyAxMC40NzJsMS41MzItMS41MzMgMiAyIC43MDctLjcwNy0yLTIgNi4wMS02LjAxIDIuODMgMi44MjhMNS4wNSAxNy43NzggMi4yMjIgMTQuOTVsMS41MzItMS41MzIgMS40MTQgMS40MTQuNzA3LS43MDd6TS44MDggMTQuOTVsLjcwNy0uNzA3TDE0LjI0MyAxLjUxNWwuNzA3LS43MDcuNzA3LjcwNyAyLjgyOCAyLjgyOC43MDcuNzA3LS43MDcuNzA3TDUuNzU3IDE4LjQ4NWwtLjcwNy43MDctLjcwNy0uNzA3LTIuODI4LTIuODI4em0xMS4wNzgtNi44MzVMMTAuNDcgNi43bC43MDctLjcwNyAxLjQxNSAxLjQxNHptLjgyNC0zLjY1NCAxIDEgLjcwOC0uNzA3LTEtMXoiIGZpbGw9IiMxODFEMUYiLz48L3N2Zz4=)}.ag-charts-icon-note-annotation{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMyA0LjVBMS41IDEuNSAwIDAgMSA0LjUgM2gxMUExLjUgMS41IDAgMCAxIDE3IDQuNXY4YTEuNSAxLjUgMCAwIDEtMS41IDEuNWgtMy4yMWwtMS40NjkgMi41N2ExIDEgMCAwIDEtMS42ODIuMDg1TDcuMjQzIDE0SDQuNUExLjUgMS41IDAgMCAxIDMgMTIuNXpNNC41IDRhLjUuNSAwIDAgMC0uNS41djhhLjUuNSAwIDAgMCAuNS41aDMuMjU3bDIuMTk2IDMuMDc0TDExLjcxIDEzaDMuNzlhLjUuNSAwIDAgMCAuNS0uNXYtOGEuNS41IDAgMCAwLS41LS41eiIgY2xpcC1ydWxlPSJldmVub2RkIi8+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNi41IDYuNUEuNS41IDAgMCAxIDcgNmg2YS41LjUgMCAwIDEgMCAxSDdhLjUuNSAwIDAgMS0uNS0uNU02LjUgOS41QS41LjUgMCAwIDEgNyA5aDZhLjUuNSAwIDAgMSAwIDFIN2EuNS41IDAgMCAxLS41LS41IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=)}.ag-charts-icon-ohlc-series{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZD0iTTEzIDExaC0zdi0xaDNWM2gxdjJoNHYxaC00djExaC0xek02IDE3di0yaDN2LTFINlY0SDV2MUgydjFoM3YxMXoiLz48L3N2Zz4=)}.ag-charts-icon-pan-end{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZD0ibTYuNjQ2IDEzLjgxMy0uMzUzLjM1NC43MDcuNzA3LjM1NC0uMzU0ek0xMS4xNjYgMTBsLjM1NC4zNTQuMzU0LS4zNTQtLjM1NC0uMzU0ek03LjM1NSA1LjQ4IDcgNS4xMjZsLS43MDcuNzA3LjM1My4zNTR6bTAgOS4wNCA0LjE2Ni00LjE2Ni0uNzA3LS43MDgtNC4xNjcgNC4xNjd6bTQuMTY2LTQuODc0TDcuMzU0IDUuNDhsLS43MDguNzA3IDQuMTY3IDQuMTY3ek0xMy4wODMgNXYxMGgxVjV6Ii8+PC9zdmc+)}.ag-charts-icon-pan-left{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMTIuNzkgNS44MzMgOC42MjUgMTBsNC4xNjYgNC4xNjctLjcwNy43MDdMNy4yMSAxMGw0Ljg3My00Ljg3NHoiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==)}.ag-charts-icon-pan-right{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNy4yMSAxNC4xNjcgMTEuMzc2IDEwIDcuMjEgNS44MzNsLjcwNy0uNzA3TDEyLjc5IDEwbC00Ljg3MyA0Ljg3NHoiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==)}.ag-charts-icon-pan-start{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZD0iTTYgNXYxMGgxVjV6TTkuNjI0IDEwbDQuMTY2LTQuMTY3LS43MDctLjcwN0w4LjIxIDEwbDQuODc0IDQuODc0LjcwNy0uNzA3eiIvPjwvc3ZnPg==)}.ag-charts-icon-parallel-channel,.ag-charts-icon-parallel-channel-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMTcuNzIgNS4zMzFBMi4yNSAyLjI1IDAgMSAwIDE0LjcwNSAzLjZsLTkuNDkgNC41NjJhMi4yNSAyLjI1IDAgMSAwIC4yMDkgMS4wMWw5LjY2Mi00LjY0NmEyLjI1IDIuMjUgMCAwIDAgMi42MzQuODA1bS4zNzMtMi41NDdhMS4yNSAxLjI1IDAgMSAxLTIuMzM4Ljg4NSAxLjI1IDEuMjUgMCAwIDEgMi4zMzgtLjg4NU00LjM0MyA4LjY3YTEuMjUgMS4yNSAwIDEgMS0yLjMzOC44ODUgMS4yNSAxLjI1IDAgMCAxIDIuMzM4LS44ODVNNS4zMDcgMTYuNzI4YTIuMjUgMi4yNSAwIDEgMS0uNTI1LS44NThsOS45MjMtNC43N2EyLjI1IDIuMjUgMCAxIDEgLjM4MS45MjZ6bS0uOTY0LjI3NGExLjI1IDEuMjUgMCAxIDEtMi4zMzguODg1IDEuMjUgMS4yNSAwIDAgMSAyLjMzOC0uODg1bTEzLjAyMy01LjEwNmExLjI1IDEuMjUgMCAxIDAtLjg4NS0yLjMzOSAxLjI1IDEuMjUgMCAwIDAgLjg4NSAyLjMzOSIgY2xpcC1ydWxlPSJldmVub2RkIi8+PC9zdmc+)}.ag-charts-icon-position-bottom{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsPSIjMDAwIiBmaWxsLW9wYWNpdHk9Ii4yNSIgZD0iTTMgMTBoMTR2MUgzem0zLTNoOHYxSDZ6Ii8+PHBhdGggZmlsbD0iIzAwMCIgZD0iTTYgMTNoOHYxSDZ6Ii8+PC9zdmc+)}.ag-charts-icon-position-center{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsPSIjMDAwIiBkPSJNMyAxMGgxNHYxSDN6Ii8+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1vcGFjaXR5PSIuMjUiIGQ9Ik02IDdoOHYxSDZ6bTAgNmg4djFINnoiLz48L3N2Zz4=)}.ag-charts-icon-position-top{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsPSIjMDAwIiBmaWxsLW9wYWNpdHk9Ii4yNSIgZD0iTTMgMTBoMTR2MUgzeiIvPjxwYXRoIGZpbGw9IiMwMDAiIGQ9Ik02IDdoOHYxSDZ6Ii8+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1vcGFjaXR5PSIuMjUiIGQ9Ik02IDEzaDh2MUg2eiIvPjwvc3ZnPg==)}.ag-charts-icon-price-label-annotation{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNC41IDNBMS41IDEuNSAwIDAgMCAzIDQuNVYxM2ExLjUgMS41IDAgMCAwIDEuNSAxLjVoLjgzM3YuMDU3Yy4yNDItLjI5OS41OTctLjUwMyAxLS41NDhWMTMuNUg0LjVBLjUuNSAwIDAgMSA0IDEzVjQuNWEuNS41IDAgMCAxIC41LS41aDExYS41LjUgMCAwIDEgLjUuNXY4YS41LjUgMCAwIDEtLjUuNWgtNC44MThsLS4xMjYuMDg0YTI2IDI2IDAgMCAwLTIuMjI1IDEuNjg1bC0uMzIuMjY1LS4wNjguMDU2YTEuNSAxLjUgMCAwIDEtMi42MDkgMS4zNTRjLjAzMy43NjMuOTA1IDEuMjM4IDEuNTYuNzU2LjY0Ni0uNDc0IDEuMjEtLjk0MyAxLjc2MS0xLjRsLjMxMy0uMjZBMjYgMjYgMCAwIDEgMTAuOTg2IDE0SDE1LjVhMS41IDEuNSAwIDAgMCAxLjUtMS41di04QTEuNSAxLjUgMCAwIDAgMTUuNSAzeiIgY2xpcC1ydWxlPSJldmVub2RkIi8+PHBhdGggZmlsbD0iIzEzMTcyMiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNOC43MTYgMTQuODE1YTIuMjUgMi4yNSAwIDEgMS00LjIxIDEuNTkzIDIuMjUgMi4yNSAwIDAgMSA0LjIxLTEuNTkzbS0xLjY2MiAxLjk2NmExLjI1IDEuMjUgMCAxIDAtLjg4NS0yLjMzOSAxLjI1IDEuMjUgMCAwIDAgLjg4NSAyLjMzOSIgY2xpcC1ydWxlPSJldmVub2RkIi8+PC9zdmc+)}.ag-charts-icon-price-range-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBkPSJNOS4wNDYgMTVWNS44NzdoLjk0MlYxNXoiIGZpbGw9IiMxODFEMUYiLz48cGF0aCBkPSJNOS4wNDYgMTVWNS44NzdoLjk0MlYxNXoiIGZpbGw9IiMxODFEMUYiLz48cGF0aCBkPSJNOS41IDYuMjI4IDcuMTY3IDguMzc2IDYuNSA3Ljc2MiA5LjUgNWwzIDIuNzYyLS42NjcuNjE0eiIgZmlsbD0iIzAwMCIvPjxwYXRoIGQ9Ik0yIDE4di0xaDE2djF6TTIgM1YyaDE2djF6IiBmaWxsPSIjMTgxRDFGIi8+PC9zdmc+)}.ag-charts-icon-reset{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMTIuMDQgNC40NDVhNS44MSA1LjgxIDAgMCAwLTcuMjU3IDIuNDUzLjUuNSAwIDAgMS0uODY1LS41MDJBNi44MSA2LjgxIDAgMSAxIDMgOS44MTNhLjUuNSAwIDAgMSAxIDAgNS44MSA1LjgxIDAgMSAwIDguMDQtNS4zNjgiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjxwYXRoIGZpbGw9IiMwMDAiIGZpbGwtcnVsZT0iZXZlbm9kZCIgZD0iTTQuMjg5IDMuMDAyYS41LjUgMCAwIDEgLjUuNXYyLjY1NWgyLjY1NWEuNS41IDAgMCAxIDAgMUg0LjI5YS41LjUgMCAwIDEtLjUtLjVWMy41MDJhLjUuNSAwIDAgMSAuNS0uNSIgY2xpcC1ydWxlPSJldmVub2RkIi8+PC9zdmc+)}.ag-charts-icon-settings{--icon: url(data:image/svg+xml;base64,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)}.ag-charts-icon-step-line-series{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzE4MUQxRiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNiA0aDV2OGgzVjhoNXYxaC00djRoLTVWNUg3djEwSDJ2LTFoNHoiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==)}.ag-charts-icon-text-annotation{--icon: url(data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2RkIiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik00IDRIMTZWN0gxNVY1SDEwLjVWMTVIMTRWMTZINlYxNUg5LjVWNUg1VjdINFY0WiIgZmlsbD0iYmxhY2siLz4KPC9zdmc+Cg==)}.ag-charts-icon-trend-line,.ag-charts-icon-trend-line-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNS4zMTQgMTAuOTM4YTIuMjUgMi4yNSAwIDEgMSAuMDEtMWg5LjM1MmEyLjI1IDIuMjUgMCAxIDEgLjAxIDF6bS0yLjE4OS43MjlhMS4yNSAxLjI1IDAgMSAwIDAtMi41IDEuMjUgMS4yNSAwIDAgMCAwIDIuNW0xMy43NSAwYTEuMjUgMS4yNSAwIDEgMCAwLTIuNSAxLjI1IDEuMjUgMCAwIDAgMCAyLjUiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==)}.ag-charts-icon-unlock,.ag-charts-icon-unlocked{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMTAuNjUxIDMuNWEyLjg5NCAyLjg5NCAwIDAgMC0yLjg5NCAyLjg5NFY5SDE0LjVhMi41IDIuNSAwIDAgMSAyLjUgMi41djNhMi41IDIuNSAwIDAgMS0yLjUgMi41aC04QTIuNSAyLjUgMCAwIDEgNCAxNC41di0zQTIuNSAyLjUgMCAwIDEgNi41IDloLjI1N1Y2LjM5NGEzLjg5NCAzLjg5NCAwIDEgMSA3Ljc4OSAwIC41LjUgMCAwIDEtMSAwQTIuODk0IDIuODk0IDAgMCAwIDEwLjY1IDMuNU02LjUgMTBBMS41IDEuNSAwIDAgMCA1IDExLjV2M0ExLjUgMS41IDAgMCAwIDYuNSAxNmg4YTEuNSAxLjUgMCAwIDAgMS41LTEuNXYtM2ExLjUgMS41IDAgMCAwLTEuNS0xLjV6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=)}.ag-charts-icon-vertical-line,.ag-charts-icon-vertical-line-drawing{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMTAuNSA3LjgwNmEyLjI1IDIuMjUgMCAwIDEgMCA0LjM4OFYxOS41aC0xdi03LjMwNmEyLjI1IDIuMjUgMCAwIDEgMC00LjM4OFYuNWgxem0tLjUuOTQ0YTEuMjUgMS4yNSAwIDEgMSAwIDIuNSAxLjI1IDEuMjUgMCAwIDEgMC0yLjUiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==)}.ag-charts-icon-zoom-in{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZD0iTTEwIDUuNWEuNS41IDAgMCAxIC41LjV2My41aDMuODc1YS41LjUgMCAwIDEgMCAxSDEwLjV2NC4yNWEuNS41IDAgMSAxLTEgMFYxMC41SDUuNjI1YS41LjUgMCAxIDEgMC0xSDkuNVY2YS41LjUgMCAwIDEgLjUtLjUiLz48L3N2Zz4=)}.ag-charts-icon-zoom-out{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzAwMCIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNS41IDEwYS41LjUgMCAwIDEgLjUtLjVoOGEuNS41IDAgMCAxIDAgMUg2YS41LjUgMCAwIDEtLjUtLjUiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==)}.ag-charts-icon-high-low-series{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNyA0aDJ2MTJINFY0aDNNNSA1aDN2MTBINXpNMTEgMTRWNmg1djhoLTVtMS03aDN2NmgtM3oiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==)}.ag-charts-icon-hlc-series{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iIzEzMTcyMiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJtMTguMTYzIDEuODM3LTUuMzM0IDExLjYyMUw2Ljk1NyA4LjEybC00LjE5OSA5LjYyMi0uOTE2LS40IDQuNzU2LTEwLjlMMTIuNDkgMTEuOCAxNy4yNTQgMS40MnoiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjxwYXRoIGZpbGw9IiMwMDAiIGZpbGwtb3BhY2l0eT0iLjQiIGZpbGwtcnVsZT0iZXZlbm9kZCIgZD0iTTUuODI1IDIuNzA0LjU1IDEzLjc4NWwuOTAyLjQzIDQuNzI0LTkuOTE5IDYuMDM0IDUuMDI5IDMuMjU1LTguMTQtLjkyOC0uMzctMi43NDUgNi44NnptNy44NTIgMTQuNjM2IDUuNzgtMTMuMTM5LS45MTUtLjQwMi01LjIxOSAxMS44Ni02LjAwNS01LjUwNC0zLjI3OCA3LjY0OC45Mi4zOTQgMi43MjItNi4zNTJ6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=)}.ag-charts-icon-zoom-in-alt{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgdmlld0JveD0iMCAwIDI0IDI0IiBmaWxsPSJub25lIiBzdHJva2U9ImN1cnJlbnRDb2xvciIgc3Ryb2tlLXdpZHRoPSIyIiBzdHJva2UtbGluZWNhcD0icm91bmQiIHN0cm9rZS1saW5lam9pbj0icm91bmQiIGNsYXNzPSJsdWNpZGUgbHVjaWRlLXpvb20taW4iPjxjaXJjbGUgY3g9IjExIiBjeT0iMTEiIHI9IjgiLz48bGluZSB4MT0iMjEiIHgyPSIxNi42NSIgeTE9IjIxIiB5Mj0iMTYuNjUiLz48bGluZSB4MT0iMTEiIHgyPSIxMSIgeTE9IjgiIHkyPSIxNCIvPjxsaW5lIHgxPSI4IiB4Mj0iMTQiIHkxPSIxMSIgeTI9IjExIi8+PC9zdmc+)}.ag-charts-icon-zoom-out-alt{--icon: url(data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgdmlld0JveD0iMCAwIDI0IDI0IiBmaWxsPSJub25lIiBzdHJva2U9ImN1cnJlbnRDb2xvciIgc3Ryb2tlLXdpZHRoPSIyIiBzdHJva2UtbGluZWNhcD0icm91bmQiIHN0cm9rZS1saW5lam9pbj0icm91bmQiIGNsYXNzPSJsdWNpZGUgbHVjaWRlLXpvb20tb3V0Ij48Y2lyY2xlIGN4PSIxMSIgY3k9IjExIiByPSI4Ii8+PGxpbmUgeDE9IjIxIiB4Mj0iMTYuNjUiIHkxPSIyMSIgeTI9IjE2LjY1Ii8+PGxpbmUgeDE9IjgiIHgyPSIxNCIgeTE9IjExIiB5Mj0iMTEiLz48L3N2Zz4=)}.ag-charts-input{--input-layer-active: 1;--input-layer-focus: 2;--input-padding: calc(var(--ag-charts-spacing) * 2);--input-padding-large: calc(var(--ag-charts-spacing) * 2.5);color:var(--ag-charts-input-text-color);font-family:var(--ag-charts-chrome-font-family);font-size:var(--ag-charts-chrome-font-size-large);transition-duration:.25s;transition-property:none;transition-timing-function:ease-out}.ag-charts-input:focus-visible{outline:var(--ag-charts-focus-border);box-shadow:var(--ag-charts-focus-border-shadow);z-index:var(--input-layer-focus)}.ag-charts-button{background:var(--ag-charts-button-background-color);border:1px solid var(--ag-charts-border-color);border-radius:var(--ag-charts-button-border-radius);color:var(--ag-charts-button-text-color);cursor:pointer;padding:var(--input-padding);transition-property:background,border-color}.ag-charts-button:hover{background:var(--ag-charts-focus-color)}.ag-charts-button:has(.ag-charts-icon){padding:2px}.ag-charts-checkbox{--checkbox-transition-duration: .1s;appearance:none;background:var(--ag-charts-checkbox-background-color);border-radius:calc(var(--ag-charts-border-radius) * 9);cursor:pointer;height:18px;margin:0;transition-duration:var(--checkbox-transition-duration);transition-property:margin;width:29px}.ag-charts-checkbox:before{display:block;background:var(--ag-charts-input-background-color);border-radius:calc(var(--ag-charts-border-radius) * 7);content:" ";height:14px;margin:2px;transition-duration:var(--checkbox-transition-duration);transition-property:margin;transition-timing-function:var(--ag-charts-input-transition-easing);width:14px}.ag-charts-checkbox:checked{background:var(--ag-charts-checkbox-checked-background-color)}.ag-charts-checkbox:checked:before{margin-left:13px}.ag-charts-select{background:var(--ag-charts-input-background-color);border:1px solid var(--ag-charts-input-border-color);border-radius:var(--ag-charts-input-border-radius);padding:3px 2px 4px;font-size:inherit}.ag-charts-textarea{--textarea-line-height: 1.38;background:var(--ag-charts-input-background-color);border:1px solid var(--ag-charts-border-color);border-radius:var(--ag-charts-input-border-radius);line-height:var(--textarea-line-height);font-family:var(--ag-charts-chrome-font-family);font-size:var(--ag-charts-chrome-font-size-large);padding:var(--input-padding-large) var(--input-padding)}.ag-charts-textarea::placeholder{color:var(--ag-charts-input-placeholder-text-color)}.ag-charts-proxy-container{pointer-events:none;position:absolute}.ag-charts-proxy-elem{-webkit-appearance:none;appearance:none;background:none;border:none;color:#0000;overflow:hidden;pointer-events:auto;position:absolute}.ag-charts-proxy-elem::-moz-range-thumb,.ag-charts-proxy-elem::-moz-range-track{opacity:0}.ag-charts-proxy-elem::-webkit-slider-runnable-track,.ag-charts-proxy-elem::-webkit-slider-thumb{opacity:0}.ag-charts-proxy-elem:focus-visible{outline:var(--ag-charts-focus-border);box-shadow:var(--ag-charts-focus-border-shadow)}.ag-charts-focus-indicator{position:absolute;display:block;pointer-events:none;user-select:none;-webkit-user-select:none;width:100%;height:100%}.ag-charts-focus-indicator>div{position:absolute;outline:solid 1px var(--ag-charts-chrome-background-color);box-shadow:var(--ag-charts-focus-shadow)}.ag-charts-focus-indicator>svg{width:100%;height:100%;fill:none;overflow:visible}.ag-charts-focus-svg-outer-path{stroke:var(--ag-charts-chrome-background-color);stroke-width:4px}.ag-charts-focus-svg-inner-path{stroke:var(--ag-charts-accent-color);stroke-width:2px}.ag-charts-overlay{color:#181d1f;pointer-events:none}.ag-charts-overlay.ag-charts-dark-overlay{color:#fff}.ag-charts-overlay--loading{color:#8c8c8c}.ag-charts-overlay__loading-background{background:#fff;pointer-events:none}.ag-charts-overlay.ag-charts-dark-overlay .ag-charts-overlay__loading-background{background:#192232}.ag-charts-tooltip{--tooltip-arrow-size: 8px;position:fixed;inset:unset;margin:0;padding:0;overflow:visible;width:max-content;max-width:100%;font-family:var(--ag-charts-chrome-font-family);font-size:var(--ag-charts-chrome-font-size);font-weight:var(--ag-charts-chrome-font-weight);color:var(--ag-charts-chrome-text-color);background:var(--ag-charts-chrome-background-color);border:var(--ag-charts-border);border-radius:var(--ag-charts-border-radius);box-shadow:var(--ag-charts-shadow)}.ag-charts-tooltip--compact .ag-charts-tooltip-content{padding:3px 6px}.ag-charts-tooltip--arrow:before{content:"";position:absolute;top:100%;left:50%;margin:0 auto;transform:translate(-50%) translateY(calc(var(--tooltip-arrow-size) * -.5)) rotate(45deg);display:block;width:var(--tooltip-arrow-size);height:var(--tooltip-arrow-size);border:inherit;border-top:none;border-left:none;clip-path:polygon(0% 100%,100% 0%,100% 100%)}.ag-charts-tooltip--arrow:after{--tooltip-inner-size: calc(var(--tooltip-arrow-size) - 2px);content:"";position:absolute;top:100%;left:50%;margin:0 auto;transform:translate(-50%) translateY(calc(var(--tooltip-inner-size) * -.5)) rotate(45deg);display:block;width:var(--tooltip-inner-size);height:var(--tooltip-inner-size);border:inherit;border-color:transparent;background:inherit;clip-path:polygon(-10% 100%,100% -10%,100% 100%)}.ag-charts-tooltip--no-interaction{pointer-events:none;user-select:none;-webkit-user-select:none}.ag-charts-tooltip--wrap-always{overflow-wrap:break-word;word-break:break-word;hyphens:none}.ag-charts-tooltip--wrap-hyphenate{overflow-wrap:break-word;word-break:break-word;hyphens:auto}.ag-charts-tooltip--wrap-on-space{overflow-wrap:normal;word-break:normal}.ag-charts-tooltip--wrap-never{white-space:nowrap}.ag-charts-tooltip-heading,.ag-charts-tooltip-title,.ag-charts-tooltip-label,.ag-charts-tooltip-value{overflow:hidden;text-overflow:ellipsis}.ag-charts-tooltip-content{display:grid;grid:auto-flow minmax(1em,auto) / 1fr;padding:8px 12px;gap:8px}.ag-charts-tooltip-content:has(.ag-charts-tooltip-symbol){grid:auto-flow minmax(1em,auto) / auto 1fr}.ag-charts-tooltip-heading{grid-column:1 / -1}.ag-charts-tooltip-symbol{grid-column:1 / 2;align-self:center}.ag-charts-tooltip-symbol svg{display:block}.ag-charts-tooltip-title{grid-column:-2 / -1}.ag-charts-tooltip-row{grid-column:1 / -1;display:flex;gap:16px;align-items:baseline;justify-content:space-between;overflow:hidden}.ag-charts-tooltip-row--inline{grid-column:-2 / -1}.ag-charts-tooltip-label{flex:1;min-width:0}.ag-charts-tooltip-value{min-width:0}.ag-charts-popover{position:absolute;border:var(--ag-charts-border);border-radius:var(--ag-charts-border-radius);background:var(--ag-charts-chrome-background-color);color:var(--ag-charts-chrome-text-color);font-family:var(--ag-charts-chrome-font-family);font-size:var(--ag-charts-chrome-font-size);font-weight:var(--ag-charts-chrome-font-weight);box-shadow:var(--ag-charts-shadow);z-index:var(--ag-charts-layer-ui-overlay)}.ag-charts-menu{--item-padding: 6px 12px;--icon-color: var(--ag-charts-chrome-text-color);display:grid;grid:auto-flow auto / 1fr;column-gap:12px;font-size:var(--ag-charts-chrome-font-size)}.ag-charts-menu:has(.ag-charts-menu__icon,.ag-charts-menu__row--stroke-width-visible){grid:auto-flow auto / auto 1fr}.ag-charts-menu__row--stroke-width-visible:before{content:"";height:var(--strokeWidth);width:12px;background:var(--icon-color)}.ag-charts-menu__row--stroke-width-visible[aria-disabled=true]:before{filter:grayscale(1);opacity:.5}.ag-charts-menu__row{display:grid;grid-column:1 / -1;grid-template-columns:subgrid;align-items:center;padding:var(--item-padding)}.ag-charts-menu__row:not(.ag-charts-menu__row--active){cursor:pointer}.ag-charts-menu__row:first-child{border-top-left-radius:inherit;border-top-right-radius:inherit}.ag-charts-menu__row:last-child{border-bottom-left-radius:inherit;border-bottom-right-radius:inherit}.ag-charts-menu__row:focus{background:var(--ag-charts-focus-color)}.ag-charts-menu__row:focus-visible{outline:var(--ag-charts-focus-border);box-shadow:var(--ag-charts-focus-border-shadow);z-index:var(--ag-charts-layer-menu)}.ag-charts-menu__row--active{--icon-color: var(--ag-charts-accent-color);background:var(--ag-charts-focus-color);color:var(--ag-charts-accent-color)}.ag-charts-menu__label{grid-column:-1 / span 1}.ag-charts-toolbar{--toolbar-gap: calc(var(--ag-charts-spacing) * 2);--toolbar-size: 34px;--toolbar-button-padding: 6px;align-items:center;display:flex;flex-wrap:nowrap;position:absolute}.ag-charts-toolbar__button{align-items:center;background:var(--ag-charts-chrome-background-color);border:var(--ag-charts-border);color:var(--ag-charts-chrome-text-color);cursor:pointer;display:flex;font-family:var(--ag-charts-chrome-font-family);font-size:var(--ag-charts-chrome-font-size-medium);font-weight:var(--ag-charts-chrome-font-weight);justify-content:center;min-height:var(--toolbar-size);min-width:var(--toolbar-size);padding:var(--toolbar-button-padding);position:relative;transition:background-color .25s ease-in-out,border-color .25s ease-in-out,color .25s ease-in-out;white-space:nowrap}.ag-charts-toolbar__button:hover{background:var(--ag-charts-focus-color);z-index:1}.ag-charts-toolbar__button:focus-visible{outline:var(--ag-charts-focus-border);box-shadow:var(--ag-charts-focus-border-shadow);z-index:calc(var(--ag-charts-layer-ui-overlay) + 1)}.ag-charts-toolbar__button--active{background:var(--ag-charts-focus-color);border-color:var(--ag-charts-accent-color);color:var(--ag-charts-accent-color);z-index:2}.ag-charts-toolbar__button[aria-disabled=true]{background:var(--ag-charts-button-disabled-background-color);color:var(--ag-charts-button-disabled-text-color);cursor:default}.ag-charts-toolbar--horizontal{flex-direction:row;.ag-charts-toolbar__button{margin:0 0 0 -1px}.ag-charts-toolbar__button--first{border-bottom-left-radius:var(--ag-charts-border-radius);border-top-left-radius:var(--ag-charts-border-radius);margin:0}.ag-charts-toolbar__button--last{border-bottom-right-radius:var(--ag-charts-border-radius);border-top-right-radius:var(--ag-charts-border-radius)}}.ag-charts-toolbar--vertical{flex-direction:column;.ag-charts-toolbar__button{margin:-1px 0 0;max-width:100%}.ag-charts-toolbar__button--first{border-top-left-radius:var(--ag-charts-border-radius);border-top-right-radius:var(--ag-charts-border-radius);margin:0}.ag-charts-toolbar__button--last{border-bottom-left-radius:var(--ag-charts-border-radius);border-bottom-right-radius:var(--ag-charts-border-radius)}}.ag-charts-toolbar__icon+.ag-charts-toolbar__label{margin-left:var(--toolbar-gap)}.ag-charts-toolbar__icon,.ag-charts-toolbar__label{pointer-events:none}.ag-charts-floating-toolbar{border:none;display:flex;.ag-charts-toolbar{align-items:unset;position:unset}}.ag-charts-floating-toolbar__drag-handle{align-items:center;background:var(--ag-charts-chrome-background-color);border:var(--ag-charts-border);border-bottom-left-radius:var(--ag-charts-border-radius);border-top-left-radius:var(--ag-charts-border-radius);cursor:grab;display:flex;justify-content:center;min-width:24px;padding-left:0;padding-right:0}.ag-charts-floating-toolbar__drag-handle--dragging{cursor:grabbing}\n';

// packages/ag-charts-community/src/util/listeners.ts
var Listeners = class {
  constructor() {
    this.registeredListeners = /* @__PURE__ */ new Map();
  }
  addListener(eventType, handler) {
    const record = { symbol: Symbol(eventType), handler };
    if (this.registeredListeners.has(eventType)) {
      this.registeredListeners.get(eventType).push(record);
    } else {
      this.registeredListeners.set(eventType, [record]);
    }
    return () => this.removeListener(record.symbol);
  }
  removeListener(eventSymbol) {
    for (const [type, listeners] of this.registeredListeners.entries()) {
      const matchIndex = listeners.findIndex((listener) => listener.symbol === eventSymbol);
      if (matchIndex >= 0) {
        listeners.splice(matchIndex, 1);
        if (listeners.length === 0) {
          this.registeredListeners.delete(type);
        }
        break;
      }
    }
  }
  dispatch(eventType, ...params) {
    for (const listener of this.getListenersByType(eventType)) {
      try {
        listener.handler(...params);
      } catch (e) {
        logger_exports.errorOnce(String(e));
      }
    }
  }
  getListenersByType(eventType) {
    return this.registeredListeners.get(eventType) ?? [];
  }
  destroy() {
    this.registeredListeners.clear();
  }
};

// packages/ag-charts-community/src/util/baseManager.ts
var BaseManager = class {
  constructor() {
    this.listeners = new Listeners();
    this.destroyFns = [];
  }
  addListener(type, handler) {
    return this.listeners.addListener(type, handler);
  }
  destroy() {
    this.listeners.destroy();
    this.destroyFns.forEach((fn) => fn());
  }
};

// packages/ag-charts-community/src/util/guardedElement.ts
var GuardedElement = class _GuardedElement {
  constructor(element2, topTabGuard, bottomTabGuard) {
    this.element = element2;
    this.topTabGuard = topTabGuard;
    this.bottomTabGuard = bottomTabGuard;
    this.destroyFns = [];
    this.guardTabIndex = 0;
    this.hasFocus = false;
    this.initTabGuard(this.topTabGuard, (el) => this.onTab(el, false));
    this.initTabGuard(this.bottomTabGuard, (el) => this.onTab(el, true));
    this.element.addEventListener("focus", () => this.onFocus(), { capture: true });
    this.element.addEventListener("blur", (ev) => this.onBlur(ev), { capture: true });
  }
  set tabIndex(index) {
    this.guardTabIndex = index;
    if (this.guardTabIndex === 0) {
      this.setGuardIndices(void 0);
    } else if (!this.hasFocus) {
      this.setGuardIndices(this.guardTabIndex);
    }
  }
  destroy() {
    for (const fn of this.destroyFns)
      fn();
    this.destroyFns.length = 0;
  }
  initTabGuard(guard, handler) {
    const handlerBinding = () => handler(guard);
    guard.addEventListener("focus", handlerBinding);
    this.destroyFns.push(() => guard.removeEventListener("focus", handlerBinding));
  }
  setGuardIndices(index) {
    const tabindex = index;
    setAttribute(this.topTabGuard, "tabindex", tabindex);
    setAttribute(this.bottomTabGuard, "tabindex", tabindex);
  }
  onFocus() {
    this.hasFocus = true;
    if (this.guardTabIndex !== 0) {
      this.setGuardIndices(0);
    }
  }
  onBlur({ relatedTarget }) {
    const { topTabGuard: top, bottomTabGuard: bot } = this;
    this.hasFocus = false;
    if (this.guardTabIndex !== 0 && relatedTarget !== top && relatedTarget !== bot) {
      this.setGuardIndices(this.guardTabIndex);
    }
  }
  onTab(guard, reverse) {
    if (this.guardTabIndex !== 0) {
      let focusTarget;
      if (guard.tabIndex === 0) {
        focusTarget = this.findExitTarget(!reverse);
        this.setGuardIndices(this.guardTabIndex);
      } else {
        focusTarget = this.findEnterTarget(reverse);
      }
      focusTarget?.focus();
    }
  }
  static queryFocusable(element2, selectors) {
    const myWindow = getWindow();
    return Array.from(element2.querySelectorAll(selectors)).filter((e) => {
      if (e instanceof HTMLElement) {
        const style = myWindow.getComputedStyle(e);
        return style.display !== "none" && style.visibility !== "none";
      }
      return false;
    });
  }
  findEnterTarget(reverse) {
    const focusables = _GuardedElement.queryFocusable(this.element, '[tabindex="0"]');
    const index = reverse ? focusables.length - 1 : 0;
    return focusables[index];
  }
  findExitTarget(reverse) {
    const focusables = _GuardedElement.queryFocusable(getDocument(), "[tabindex]").filter((e) => e.tabIndex > 0).sort((a, b) => a.tabIndex - b.tabIndex);
    const { before, after } = _GuardedElement.findBeforeAndAfter(focusables, this.guardTabIndex);
    return reverse ? before : after;
  }
  static findBeforeAndAfter(elements, targetTabIndex) {
    let left = 0;
    let right = elements.length - 1;
    let before = void 0;
    let after = void 0;
    while (left <= right) {
      const mid = Math.floor((left + right) / 2);
      const currentTabIndex = elements[mid].tabIndex;
      if (currentTabIndex === targetTabIndex) {
        before = elements[mid - 1] || void 0;
        after = elements[mid + 1] || void 0;
        break;
      } else if (currentTabIndex < targetTabIndex) {
        before = elements[mid];
        left = mid + 1;
      } else {
        after = elements[mid];
        right = mid - 1;
      }
    }
    return { before, after };
  }
};

// packages/ag-charts-community/src/util/keynavUtil.ts
function hasNoModifiers(event) {
  return !(event.shiftKey || event.altKey || event.ctrlKey || event.metaKey);
}
function matchesKey(event, key, ...morekeys) {
  return hasNoModifiers(event) && (event.key === key || morekeys.some((altkey) => event.key === altkey));
}
var PREV_NEXT_KEYS = {
  horizontal: { nextKey: "ArrowRight", prevKey: "ArrowLeft" },
  vertical: { nextKey: "ArrowDown", prevKey: "ArrowUp" }
};
function stopPageScrolling(element2) {
  const handler = (event) => {
    if (event.defaultPrevented)
      return;
    const shouldPrevent = getAttribute(event.target, "data-preventdefault", true);
    if (shouldPrevent && matchesKey(event, "ArrowRight", "ArrowLeft", "ArrowDown", "ArrowUp")) {
      event.preventDefault();
    }
  };
  element2.addEventListener("keydown", handler);
  return () => element2.removeEventListener("keydown", handler);
}

// packages/ag-charts-community/src/util/pixelRatioObserver.ts
var PixelRatioObserver = class {
  constructor(callback2) {
    this.callback = callback2;
    this.devicePixelRatio = getWindow("devicePixelRatio") ?? 1;
    this.devicePixelRatioMediaQuery = void 0;
    this.devicePixelRatioListener = (e) => {
      if (e.matches)
        return;
      this.devicePixelRatio = getWindow("devicePixelRatio") ?? 1;
      this.unregisterDevicePixelRatioListener();
      this.registerDevicePixelRatioListener();
      this.callback(this.pixelRatio);
    };
  }
  get pixelRatio() {
    return this.devicePixelRatio;
  }
  observe() {
    this.registerDevicePixelRatioListener();
  }
  disconnect() {
    this.unregisterDevicePixelRatioListener();
  }
  unregisterDevicePixelRatioListener() {
    this.devicePixelRatioMediaQuery?.removeEventListener("change", this.devicePixelRatioListener);
    this.devicePixelRatioMediaQuery = void 0;
  }
  registerDevicePixelRatioListener() {
    const devicePixelRatioMediaQuery = getWindow("matchMedia")?.(`(resolution: ${this.pixelRatio}dppx)`);
    devicePixelRatioMediaQuery?.addEventListener("change", this.devicePixelRatioListener);
    this.devicePixelRatioMediaQuery = devicePixelRatioMediaQuery;
  }
};

// packages/ag-charts-community/src/util/sizeMonitor.ts
var SizeMonitor = class {
  constructor() {
    this.elements = /* @__PURE__ */ new Map();
    this.documentReady = false;
    this.queuedObserveRequests = [];
    this.onLoad = () => {
      this.documentReady = true;
      this.queuedObserveRequests.forEach(([el, cb]) => this.observe(el, cb));
      this.queuedObserveRequests = [];
      this.observeWindow();
    };
    if (typeof ResizeObserver !== "undefined") {
      this.resizeObserver = new ResizeObserver((entries) => {
        for (const {
          target,
          contentRect: { width: width2, height: height2 }
        } of entries) {
          const entry = this.elements.get(target);
          this.checkSize(entry, target, width2, height2);
        }
      });
    }
    let animationFrame;
    this.pixelRatioObserver = new PixelRatioObserver(() => {
      clearTimeout(animationFrame);
      animationFrame = setTimeout(() => this.checkPixelRatio(), 0);
    });
    this.documentReady = getDocument("readyState") === "complete";
    if (this.documentReady) {
      this.observeWindow();
    } else {
      getWindow()?.addEventListener("load", this.onLoad);
    }
  }
  destroy() {
    getWindow()?.removeEventListener("load", this.onLoad);
    this.resizeObserver?.disconnect();
    this.resizeObserver = void 0;
    this.pixelRatioObserver?.disconnect();
    this.pixelRatioObserver = void 0;
  }
  observeWindow() {
    this.pixelRatioObserver?.observe();
  }
  checkPixelRatio() {
    const pixelRatio = this.pixelRatioObserver?.pixelRatio ?? 1;
    for (const [element2, entry] of this.elements) {
      if (entry.size != null && entry.size.pixelRatio !== pixelRatio) {
        const { width: width2, height: height2 } = entry.size;
        entry.size = { width: width2, height: height2, pixelRatio };
        entry.cb(entry.size, element2);
      }
    }
  }
  checkSize(entry, element2, width2, height2) {
    if (!entry)
      return;
    if (width2 !== entry.size?.width || height2 !== entry.size?.height) {
      const pixelRatio = this.pixelRatioObserver?.pixelRatio ?? 1;
      entry.size = { width: width2, height: height2, pixelRatio };
      entry.cb(entry.size, element2);
    }
  }
  // Only a single callback is supported.
  observe(element2, cb) {
    if (!this.documentReady) {
      this.queuedObserveRequests.push([element2, cb]);
      return;
    }
    if (this.elements.has(element2)) {
      this.removeFromQueue(element2);
    } else {
      this.resizeObserver?.observe(element2);
    }
    const entry = { cb };
    this.elements.set(element2, entry);
  }
  unobserve(element2) {
    this.resizeObserver?.unobserve(element2);
    this.elements.delete(element2);
    this.removeFromQueue(element2);
    if (!this.elements.size) {
      this.destroy();
    }
  }
  removeFromQueue(element2) {
    this.queuedObserveRequests = this.queuedObserveRequests.filter(([el]) => el !== element2);
  }
};

// packages/ag-charts-community/src/util/stateTracker.ts
var StateTracker = class extends Map {
  constructor(defaultValue, defaultState) {
    super();
    this.defaultValue = defaultValue;
    this.defaultState = defaultState;
  }
  set(key, value) {
    this.delete(key);
    if (typeof value !== "undefined") {
      super.set(key, value);
    }
    return this;
  }
  stateId() {
    return Array.from(this.keys()).pop() ?? this.defaultState;
  }
  stateValue() {
    return Array.from(this.values()).pop() ?? this.defaultValue;
  }
};

// packages/ag-charts-community/src/dom/domLayout.html
var domLayout_default = '<div role="presentation" class="ag-charts-wrapper ag-charts-styles" data-ag-charts><div role="presentation" class="ag-charts-canvas-center"><div role="presentation" class="ag-charts-canvas-container"><div role="presentation" class="ag-charts-canvas" aria-hidden="true"></div><div role="figure" class="ag-charts-canvas-proxy"><div role="presentation" class="ag-charts-series-area"></div></div><div role="presentation" class="ag-charts-canvas-overlay"></div></div></div></div>';

// packages/ag-charts-community/src/dom/domManager.ts
var DOM_ELEMENT_CLASSES = [
  "styles",
  "canvas",
  "canvas-center",
  "canvas-container",
  "canvas-overlay",
  "canvas-proxy",
  "series-area"
];
var CONTAINER_MODIFIERS = {
  safeHorizontal: "ag-charts-wrapper--safe-horizontal",
  safeVertical: "ag-charts-wrapper--safe-vertical"
};
var domElementConfig = /* @__PURE__ */ new Map([
  ["styles", { childElementType: "style" }],
  ["canvas", { childElementType: "canvas" }],
  ["canvas-proxy", { childElementType: "div" }],
  ["canvas-overlay", { childElementType: "div" }],
  ["canvas-center", { childElementType: "div" }],
  ["series-area", { childElementType: "div" }]
]);
function setupObserver(element2, cb) {
  if (typeof IntersectionObserver === "undefined")
    return;
  const observer = new IntersectionObserver(
    (entries) => {
      for (const entry of entries) {
        if (entry.target === element2) {
          cb(entry.intersectionRatio);
        }
      }
    },
    { root: element2 }
  );
  observer.observe(element2);
  return observer;
}
var NULL_DOMRECT = {
  x: 0,
  y: 0,
  width: 0,
  height: 0,
  top: 0,
  bottom: 0,
  left: 0,
  right: 0,
  toJSON() {
    return NULL_DOMRECT;
  }
};
function createTabGuardElement(guardedElem, where) {
  const div = createElement("div");
  div.className = "ag-charts-tab-guard";
  guardedElem.insertAdjacentElement(where, div);
  return div;
}
var DOMManager = class extends BaseManager {
  constructor(container, styleContainer) {
    super();
    this.styles = /* @__PURE__ */ new Map();
    this.container = void 0;
    this.containerSize = void 0;
    this.sizeMonitor = new SizeMonitor();
    this.cursorState = new StateTracker("default");
    this.minWidth = 0;
    this.minHeight = 0;
    const templateEl = createElement("div");
    templateEl.innerHTML = domLayout_default;
    this.element = templateEl.children.item(0);
    this.styleRootElement = styleContainer;
    this.rootElements = DOM_ELEMENT_CLASSES.reduce(
      (r, c) => {
        const cssClass = `ag-charts-${c}`;
        const el = this.element.classList.contains(cssClass) ? this.element : this.element.querySelector(`.${cssClass}`);
        if (!el)
          throw new Error(`AG Charts - unable to find DOM element ${cssClass}`);
        r[c] = { element: el, children: /* @__PURE__ */ new Map(), listeners: [] };
        return r;
      },
      {}
    );
    let hidden = false;
    this.observer = setupObserver(this.element, (intersectionRatio) => {
      if (intersectionRatio === 0 && !hidden) {
        this.listeners.dispatch("hidden", { type: "hidden" });
      }
      hidden = intersectionRatio === 0;
    });
    this.setSizeOptions();
    this.updateContainerSize();
    this.addStyles("ag-charts-community", styles_default);
    if (container) {
      this.setContainer(container);
    }
    this.destroyFns.push(stopPageScrolling(this.element));
    const guardedElement = this.element.querySelector(".ag-charts-canvas-center");
    if (guardedElement == null)
      throw new Error("Error initializing tab guards");
    const topGuard = createTabGuardElement(guardedElement, "beforebegin");
    const botGuard = createTabGuardElement(guardedElement, "afterend");
    this.tabGuards = new GuardedElement(guardedElement, topGuard, botGuard);
  }
  destroy() {
    super.destroy();
    this.observer?.unobserve(this.element);
    if (this.container) {
      this.sizeMonitor.unobserve(this.container);
    }
    Object.values(this.rootElements).forEach((el) => {
      el.children.forEach((c) => c.remove());
      el.element.remove();
    });
    this.element.remove();
  }
  setSizeOptions(minWidth = 300, minHeight = 300, optionsWidth, optionsHeight) {
    const { style } = this.element;
    style.width = `${optionsWidth ?? minWidth}px`;
    style.height = `${optionsHeight ?? minHeight}px`;
    this.minWidth = optionsWidth ?? minWidth;
    this.minHeight = optionsHeight ?? minHeight;
    this.updateContainerClassName();
  }
  updateContainerSize() {
    const { style: centerStyle } = this.rootElements["canvas-center"].element;
    centerStyle.visibility = this.containerSize == null ? "hidden" : "";
    if (this.containerSize) {
      centerStyle.width = `${this.containerSize.width ?? 0}px`;
      centerStyle.height = `${this.containerSize.height ?? 0}px`;
    } else {
      centerStyle.width = "";
      centerStyle.height = "";
    }
    this.updateContainerClassName();
  }
  setTabGuardIndex(tabIndex) {
    this.tabGuards.tabIndex = tabIndex;
  }
  setContainer(newContainer) {
    if (newContainer === this.container)
      return;
    if (this.container) {
      this.container.removeChild(this.element);
      this.sizeMonitor.unobserve(this.container);
    }
    const isShadowDom = this.getShadowDocumentRoot(newContainer) != null;
    if (!isShadowDom) {
      for (const id of this.rootElements["styles"].children.keys()) {
        this.removeChild("styles", id);
      }
    }
    this.container = newContainer;
    for (const [id, styles] of this.styles) {
      this.addStyles(id, styles);
    }
    newContainer.appendChild(this.element);
    this.sizeMonitor.observe(newContainer, (size) => {
      this.containerSize = size;
      this.updateContainerSize();
      this.listeners.dispatch("resize", { type: "resize" });
    });
    this.listeners.dispatch("container-changed", { type: "container-changed" });
  }
  setThemeClass(themeClassName) {
    const themeClassNamePrefix = "ag-charts-theme-";
    this.element.classList.forEach((className) => {
      if (className.startsWith(themeClassNamePrefix) && className !== themeClassName) {
        this.element.classList.remove(className);
      }
    });
    this.element.classList.add(themeClassName);
  }
  setThemeParameters(params) {
    const keysMap = {
      accentColor: "accent-color",
      axisColor: "axis-color",
      backgroundColor: "background-color",
      borderColor: "border-color",
      foregroundColor: "foreground-color",
      fontFamily: "font-family",
      fontSize: "font-size",
      fontWeight: "font-weight",
      gridLineColor: "grid-line-color",
      padding: "padding",
      subtleTextColor: "subtle-text-color",
      textColor: "text-color",
      chromeBackgroundColor: "chrome-background-color",
      chromeFontFamily: "chrome-font-family",
      chromeFontSize: "chrome-font-size",
      chromeFontWeight: "chrome-font-weight",
      chromeSubtleTextColor: "chrome-subtle-text-color",
      chromeTextColor: "chrome-text-color",
      inputBackgroundColor: "input-background-color",
      inputTextColor: "input-text-color",
      crosshairLabelBackgroundColor: "crosshair-label-background-color",
      crosshairLabelTextColor: "crosshair-label-text-color"
    };
    const lengthKeys = ["fontSize", "chromeFontSize"];
    for (const [key, value] of Object.entries(params)) {
      let formattedValue = `${value}`;
      if (lengthKeys.includes(key)) {
        formattedValue = `${value}px`;
      }
      this.element.style.setProperty(`--ag-charts-${keysMap[key]}`, formattedValue);
    }
  }
  updateCanvasLabel(ariaLabel) {
    setAttribute(this.rootElements["canvas-proxy"].element, "aria-label", ariaLabel);
  }
  getEventElement(defaultElem, eventType) {
    const events = ["focus", "blur", "keydown", "keyup"];
    return events.includes(eventType) ? this.rootElements["series-area"].element : defaultElem;
  }
  addEventListener(type, listener, options) {
    this.getEventElement(this.element, type).addEventListener(type, listener, options);
  }
  removeEventListener(type, listener, options) {
    this.getEventElement(this.element, type).removeEventListener(type, listener, options);
  }
  /** Get the main chart area client bound rect. */
  getBoundingClientRect() {
    return this.rootElements["canvas"].element.getBoundingClientRect();
  }
  /**
   * Get the client bounding rect for overlay elements that might float outside the bounds of the
   * main chart area.
   */
  getOverlayClientRect() {
    const window2 = getWindow();
    const windowBBox = new BBox(0, 0, window2.innerWidth, window2.innerHeight);
    const containerBBox = this.getRawOverlayClientRect();
    return windowBBox.intersection(containerBBox)?.toDOMRect() ?? NULL_DOMRECT;
  }
  getRawOverlayClientRect() {
    let element2 = this.element;
    while (element2 != null) {
      const styleMap = element2.computedStyleMap?.();
      const overflowY = styleMap?.get("overflow-y")?.toString();
      if (overflowY === "auto" || overflowY === "scroll") {
        return BBox.fromDOMRect(element2.getBoundingClientRect());
      }
      element2 = element2.parentElement;
    }
    const docRoot = this.getShadowDocumentRoot();
    if (docRoot != null)
      return BBox.fromDOMRect(docRoot.getBoundingClientRect());
    const { innerWidth, innerHeight } = getWindow();
    return new BBox(0, 0, innerWidth, innerHeight);
  }
  getShadowDocumentRoot(current = this.container) {
    const docRoot = current?.ownerDocument?.body ?? getDocument("body");
    while (current != null) {
      if (current === docRoot) {
        return void 0;
      }
      if (current.parentNode instanceof DocumentFragment) {
        return current;
      }
      current = current.parentNode;
    }
    return this.container;
  }
  getParent(domElementClass) {
    return this.rootElements[domElementClass].element;
  }
  getChildBoundingClientRect(type) {
    const { children } = this.rootElements[type];
    const childRects = [];
    for (const child of children.values()) {
      childRects.push(BBox.fromDOMRect(child.getBoundingClientRect()));
    }
    return BBox.merge(childRects);
  }
  isManagedChildDOMElement(el, domElementClass, id) {
    const { children } = this.rootElements[domElementClass];
    const search = children?.get(id);
    return search != null && el.contains(search);
  }
  contains(element2, domElementClass) {
    if (domElementClass == null)
      return this.element.contains(element2);
    return this.rootElements[domElementClass].element.contains(element2);
  }
  addStyles(id, styles) {
    const dataAttribute = "data-ag-charts";
    this.styles.set(id, styles);
    if (this.container == null)
      return;
    const checkId = (el) => {
      return el.getAttribute(dataAttribute) === id;
    };
    const addStyleElement = (el) => {
      const metaElements = /* @__PURE__ */ new Set(["TITLE", "META"]);
      let skippingMetaElements = true;
      let insertAfterEl;
      for (const child of el.children) {
        if (skippingMetaElements && metaElements.has(child.tagName)) {
          insertAfterEl = child;
          continue;
        }
        skippingMetaElements = false;
        if (checkId(child))
          return;
        if (child.hasAttribute(dataAttribute)) {
          insertAfterEl = child;
        }
      }
      const styleEl = createElement("style");
      if (insertAfterEl != null) {
        el.insertBefore(styleEl, insertAfterEl.nextSibling);
      } else {
        el.prepend(styleEl);
      }
      return styleEl;
    };
    let styleElement;
    if (this.styleRootElement) {
      styleElement = addStyleElement(this.styleRootElement);
    } else {
      const documentRoot = this.getShadowDocumentRoot(this.container);
      if (documentRoot != null) {
        styleElement = this.addChild("styles", id);
      } else {
        styleElement = addStyleElement(getDocument("head"));
      }
    }
    if (styleElement == null || checkId(styleElement))
      return;
    styleElement.setAttribute(dataAttribute, id);
    styleElement.innerHTML = styles;
  }
  removeStyles(id) {
    this.removeChild("styles", id);
  }
  updateCursor(callerId, style) {
    this.cursorState.set(callerId, style);
    this.element.style.cursor = this.cursorState.stateValue();
  }
  getCursor() {
    return this.element.style.cursor;
  }
  addChild(domElementClass, id, child, insert) {
    const { element: element2, children, listeners } = this.rootElements[domElementClass];
    if (!children) {
      throw new Error("AG Charts - unable to create DOM elements after destroy()");
    }
    if (children.has(id)) {
      return children.get(id);
    }
    const { childElementType = "div" } = domElementConfig.get(domElementClass) ?? {};
    if (child && child.tagName.toLowerCase() !== childElementType.toLowerCase()) {
      throw new Error("AG Charts - mismatching DOM element type");
    }
    const newChild = child ?? createElement(childElementType);
    for (const [type, fn, opts] of listeners) {
      newChild.addEventListener(type, fn, opts);
    }
    children.set(id, newChild);
    if (insert) {
      const queryResult = element2.querySelector(insert.query);
      if (queryResult == null) {
        throw new Error(`AG Charts - addChild query failed ${insert.query}`);
      }
      queryResult.insertAdjacentElement(insert.where, newChild);
    } else {
      element2?.appendChild(newChild);
    }
    return newChild;
  }
  removeChild(domElementClass, id) {
    const { children } = this.rootElements[domElementClass];
    if (!children)
      return;
    children.get(id)?.remove();
    children.delete(id);
  }
  incrementDataCounter(name) {
    const { dataset } = this.element;
    dataset[name] ?? (dataset[name] = "0");
    dataset[name] = String(Number(dataset[name]) + 1);
  }
  setDataBoolean(name, value) {
    this.element.dataset[name] = String(value);
  }
  updateContainerClassName() {
    const { element: element2, containerSize, minWidth, minHeight } = this;
    element2.classList.toggle(CONTAINER_MODIFIERS.safeHorizontal, minWidth >= (containerSize?.width ?? Infinity));
    element2.classList.toggle(CONTAINER_MODIFIERS.safeVertical, minHeight >= (containerSize?.height ?? Infinity));
  }
};

// packages/ag-charts-community/src/widget/boundedTextWidget.ts
var BoundedTextWidget = class extends Widget {
  constructor() {
    super(createElement("div"));
    this.textElement = createSvgElement("text");
    this.textElement.role = "presentation";
    this.svgElement = createSvgElement("svg");
    this.svgElement.appendChild(this.textElement);
    this.svgElement.style.width = "100%";
    this.svgElement.style.opacity = "0";
    this.svgElement.role = "presentation";
    this.elem.appendChild(this.svgElement);
    this.elem.role = "presentation";
  }
  set textContent(text) {
    this.textElement.textContent = text;
    const bboxCalculator = this.textElement;
    const bbox = bboxCalculator.getBBox?.();
    if (bbox) {
      this.svgElement.setAttribute("viewBox", `${bbox.x} ${bbox.y} ${bbox.width} ${bbox.height}`);
    }
  }
  get textContent() {
    return this.textElement.textContent;
  }
  destructor() {
  }
};

// packages/ag-charts-community/src/widget/buttonWidget.ts
var ButtonWidget = class extends Widget {
  constructor() {
    super(createElement("button"));
    this.setEnabled(true);
  }
  destructor() {
  }
  setEnabled(enabled) {
    setAttribute(this.elem, "aria-disabled", !enabled);
    setElementStyle(this.elem, "pointer-events", enabled ? void 0 : "none");
  }
  addListener(type, listener) {
    return super.addListener(type, (ev, current) => {
      if ((type === "click" || type === "dblclick") && this.isDisabled())
        return;
      listener(ev, current);
    });
  }
};

// packages/ag-charts-community/src/widget/groupWidget.ts
var GroupWidget = class extends Widget {
  constructor() {
    super(createElement("div"));
    setAttribute(this.elem, "role", "group");
  }
  destructor() {
  }
};

// packages/ag-charts-community/src/widget/rovingTabContainerWidget.ts
var RovingTabContainerWidget = class extends Widget {
  constructor(initialOrientation, role) {
    super(createElement("div"));
    this.focusedChildIndex = 0;
    this.onChildFocus = (_event, child) => {
      const oldFocus = this.children[this.focusedChildIndex];
      this.focusedChildIndex = child.index;
      oldFocus?.setTabIndex(-1);
      child.setTabIndex(0);
    };
    this.onChildKeyDown = (event, child) => {
      const rovingOrientation = this.orientation;
      const [primaryKeys, secondaryKeys] = rovingOrientation === "both" ? [PREV_NEXT_KEYS["horizontal"], PREV_NEXT_KEYS["vertical"]] : [PREV_NEXT_KEYS[rovingOrientation], void 0];
      let targetIndex = -1;
      if (hasNoModifiers(event.sourceEvent)) {
        const key = event.sourceEvent.key;
        if (key === primaryKeys.nextKey || key === secondaryKeys?.nextKey) {
          targetIndex = child.index + 1;
        } else if (key === primaryKeys.prevKey || key === secondaryKeys?.prevKey) {
          targetIndex = child.index - 1;
        }
      }
      this.children[targetIndex]?.focus();
    };
    this.orientation = initialOrientation;
    setAttribute(this.elem, "role", role);
  }
  get orientation() {
    return getAttribute(this.elem, "aria-orientation") ?? "both";
  }
  set orientation(orientation) {
    setAttribute(this.elem, "aria-orientation", orientation !== "both" ? orientation : void 0);
  }
  focus() {
    this.children[this.focusedChildIndex]?.focus();
  }
  onChildAdded(child) {
    child.addListener("focus", this.onChildFocus);
    child.addListener("keydown", this.onChildKeyDown);
    child.setTabIndex(this.children.length === 1 ? 0 : -1);
  }
  onChildRemoved(child) {
    child.removeListener("focus", this.onChildFocus);
    child.removeListener("keydown", this.onChildKeyDown);
  }
};

// packages/ag-charts-community/src/widget/listWidget.ts
var ListWidget = class extends RovingTabContainerWidget {
  constructor() {
    super("both", "list");
    this.setHidden(true);
  }
  destructor() {
    this.children.forEach((c) => c.getElement().parentElement.remove());
  }
  addChildToDOM(child, before) {
    const listItem = createElement("div");
    setAttribute(listItem, "role", "listitem");
    setElementStyle(listItem, "position", "absolute");
    Widget.setElementContainer(child, listItem);
    this.appendOrInsert(listItem, before);
    this.setHidden(false);
  }
  removeChildFromDOM(child) {
    child.getElement().parentElement.remove();
    this.setHidden(this.children.length === 0);
  }
  setHidden(hidden) {
    if (this.children.length === 0) {
      hidden = true;
    }
    super.setHidden(hidden);
  }
};

// packages/ag-charts-community/src/widget/nativeWidget.ts
var NativeWidget = class extends Widget {
  constructor(elem) {
    super(elem);
  }
  destructor() {
  }
};

// packages/ag-charts-community/src/widget/sliderWidget.ts
var _SliderWidget = class _SliderWidget extends Widget {
  constructor() {
    super(createElement("input"));
    this._step = _SliderWidget.STEP_ONE;
    this.orientation = "both";
  }
  get step() {
    return this._step;
  }
  set step(step) {
    this._step = step;
    this.getElement().step = step.attributeValue;
  }
  get keyboardStep() {
    return this._keyboardStep?.step ?? this._step;
  }
  set keyboardStep(step) {
    if (step === this._keyboardStep?.step)
      return;
    if (this._keyboardStep !== void 0) {
      this.removeListener("keydown", this._keyboardStep.onKeyDown);
      this.removeListener("keyup", this._keyboardStep.onKeyUp);
      this.removeListener("blur", this._keyboardStep.onBlur);
      this._keyboardStep = void 0;
    }
    if (step !== void 0) {
      const onKeyDown = () => this.getElement().step = step.attributeValue;
      const resetStep = () => this.getElement().step = this._step.attributeValue;
      this._keyboardStep = { step, onKeyDown, onKeyUp: resetStep, onBlur: resetStep };
      this.addListener("keydown", this._keyboardStep.onKeyDown);
      this.addListener("keyup", this._keyboardStep.onKeyUp);
      this.addListener("blur", this._keyboardStep.onBlur);
    }
  }
  get orientation() {
    return getAttribute(this.elem, "aria-orientation") ?? "both";
  }
  set orientation(orientation) {
    setAttribute(this.elem, "aria-orientation", orientation !== "both" ? orientation : void 0);
    _SliderWidget.registerDefaultPreventers(this, orientation);
  }
  destructor() {
  }
  clampValueRatio(clampMin, clampMax) {
    const ratio2 = this.getValueRatio();
    const clampedRatio = clamp(clampMin, ratio2, clampMax);
    if (clampedRatio !== ratio2) {
      this.setValueRatio(clampedRatio);
    }
    return clampedRatio;
  }
  setValueRatio(ratio2, opts) {
    const { divider } = this.step;
    const value = Math.round(ratio2 * 1e4) / divider;
    const { ariaValueText = formatPercent(value / divider) } = opts ?? {};
    const elem = this.getElement();
    elem.value = `${value}`;
    elem.ariaValueText = ariaValueText;
  }
  getValueRatio() {
    return parseFloat(this.getElement().value) / this.step.divider;
  }
  static registerDefaultPreventers(target, orientation) {
    if (orientation === "both") {
      target.removeListener("keydown", _SliderWidget.onKeyDown);
    } else {
      target.addListener("keydown", _SliderWidget.onKeyDown);
    }
  }
  static onKeyDown(ev, current) {
    let ignoredKeys = [];
    const { orientation } = current;
    if (orientation === "horizontal") {
      ignoredKeys = ["ArrowUp", "ArrowDown"];
    } else if (orientation === "vertical") {
      ignoredKeys = ["ArrowLeft", "ArrowRight"];
    }
    if (ignoredKeys.includes(ev.sourceEvent.code)) {
      ev.sourceEvent.preventDefault();
    }
  }
};
_SliderWidget.STEP_ONE = { attributeValue: "1", divider: 1 };
_SliderWidget.STEP_HUNDRETH = { attributeValue: "0.01", divider: 100 };
var SliderWidget = _SliderWidget;

// packages/ag-charts-community/src/widget/switchWidget.ts
var SwitchWidget = class extends ButtonWidget {
  constructor() {
    super();
    setAttribute(this.elem, "role", "switch");
    this.setChecked(false);
  }
  setChecked(checked) {
    setAttribute(this.elem, "aria-checked", checked);
  }
};

// packages/ag-charts-community/src/widget/toolbarWidget.ts
var ToolbarWidget = class extends RovingTabContainerWidget {
  constructor(orientation = "horizontal") {
    super(orientation, "toolbar");
  }
  destructor() {
  }
};

// packages/ag-charts-community/src/dom/proxyInteractionService.ts
function checkType(type, meta) {
  return meta.params?.type === type;
}
function allocateResult(type) {
  if ("button" === type) {
    return new ButtonWidget();
  } else if ("slider" === type) {
    return new SliderWidget();
  } else if ("toolbar" === type) {
    return new ToolbarWidget();
  } else if ("group" === type) {
    return new GroupWidget();
  } else if ("list" === type) {
    return new ListWidget();
  } else if ("region" === type) {
    return new NativeWidget(createElement("div"));
  } else if ("text" === type) {
    return new BoundedTextWidget();
  } else if ("listswitch" === type) {
    return new SwitchWidget();
  } else {
    throw Error("AG Charts - error allocating meta");
  }
}
function allocateMeta(params) {
  const meta = { params, result: void 0 };
  meta.result = allocateResult(meta.params.type);
  return meta;
}
var ProxyInteractionService = class {
  constructor(localeManager, domManager) {
    this.localeManager = localeManager;
    this.domManager = domManager;
    this.destroyFns = [];
  }
  destroy() {
    this.destroyFns.forEach((fn) => fn());
  }
  addLocalisation(fn) {
    fn();
    this.destroyFns.push(this.localeManager.addListener("locale-changed", fn));
  }
  createProxyContainer(args) {
    const meta = allocateMeta(args);
    const { params, result } = meta;
    const div = result.getElement();
    this.domManager.addChild("canvas-proxy", params.domManagerId, div);
    div.classList.add(...params.classList, "ag-charts-proxy-container");
    div.role = params.type;
    if ("ariaOrientation" in params) {
      div.ariaOrientation = params.ariaOrientation;
    }
    if (checkType("toolbar", meta)) {
      meta.result.orientation = meta.params.orientation;
    }
    this.addLocalisation(() => {
      div.ariaLabel = this.localeManager.t(params.ariaLabel.id, params.ariaLabel.params);
    });
    return result;
  }
  createProxyElement(args) {
    const meta = allocateMeta(args);
    if (checkType("button", meta)) {
      const { params, result } = meta;
      const button = result.getElement();
      this.initInteract(params, result);
      if (typeof params.textContent === "string") {
        button.textContent = params.textContent;
      } else {
        const { textContent } = params;
        this.addLocalisation(() => {
          button.textContent = this.localeManager.t(textContent.id, textContent.params);
        });
      }
      this.setParent(meta.params, meta.result);
    }
    if (checkType("slider", meta)) {
      const { params, result } = meta;
      const slider = result.getElement();
      this.initInteract(params, result);
      slider.type = "range";
      slider.role = "presentation";
      slider.style.margin = "0px";
      this.addLocalisation(() => {
        slider.ariaLabel = this.localeManager.t(params.ariaLabel.id, params.ariaLabel.params);
      });
      this.setParent(meta.params, meta.result);
    }
    if (checkType("text", meta)) {
      const { params, result } = meta;
      this.initElement(params, result);
      this.setParent(meta.params, meta.result);
    }
    if (checkType("listswitch", meta)) {
      const { params, result: button } = meta;
      this.initInteract(params, button);
      button.setTextContent(params.textContent);
      button.setChecked(params.ariaChecked);
      button.setAriaDescribedBy(params.ariaDescribedBy);
      this.setParent(meta.params, meta.result);
    }
    if (checkType("region", meta)) {
      const { params, result } = meta;
      const region = result.getElement();
      this.initInteract(params, result);
      region.role = "region";
      this.setParent(meta.params, meta.result);
    }
    return meta.result;
  }
  initElement(params, widget) {
    const element2 = widget.getElement();
    setAttribute(element2, "id", params.id);
    setElementStyle(element2, "cursor", params.cursor);
    element2.classList.toggle("ag-charts-proxy-elem", true);
    return element2;
  }
  initInteract(params, widget) {
    const { tabIndex, domIndex } = params;
    const element2 = this.initElement(params, widget);
    if (tabIndex !== void 0) {
      element2.tabIndex = tabIndex;
    }
    if (domIndex !== void 0) {
      widget.domIndex = domIndex;
    }
  }
  setParent(params, element2) {
    if ("parent" in params) {
      params.parent?.addChild(element2);
    } else {
      const insert = { where: params.where, query: ".ag-charts-series-area" };
      this.domManager.addChild("canvas-proxy", params.domManagerId, element2.getElement(), insert);
    }
  }
};

// packages/ag-charts-locale/src/en-US.ts
var AG_CHARTS_LOCALE_EN_US = {
  // Screen reader announcement when focusing an item in the chart
  ariaAnnounceHoverDatum: "${datum}",
  // Screen reader announcement when focusing a chart
  ariaAnnounceChart: "chart, ${seriesCount}[number] series",
  // Screen reader announcement when focusing a hierarchy chart
  ariaAnnounceHierarchyChart: "hierarchy chart, ${caption}",
  // Screen reader announcement when focusing a gauge chart
  ariaAnnounceGaugeChart: "gauge chart, ${caption}",
  // Screen reader announcement when focusing an item in a treemap or sunburst chart
  ariaAnnounceHierarchyDatum: "level ${level}[number], ${count}[number] children, ${description}",
  // Screen reader announcement when focusing a link in a Sankey or chord chart
  ariaAnnounceFlowProportionLink: "link ${index} of ${count}, from ${from} to ${to}, ${sizeName} ${size}",
  // Screen reader announcement when focusing a node in a Sankey or chord chart
  ariaAnnounceFlowProportionNode: "node ${index} of ${count}, ${description}",
  // Screen reader description for legend items
  ariaDescriptionLegendItem: "Press Space or Enter to toggle visibility",
  // Screen reader for the '+' horizontal line button on the Y-axis
  ariaLabelAddHorizontalLine: "Add Horizontal Line",
  // Screen reader text for annotation-options toolbar
  ariaLabelAnnotationOptionsToolbar: "Annotation Options",
  // Screen reader text for annotation-settings dialog
  ariaLabelAnnotationSettingsDialog: "Annotation Settings",
  // Screen reader text for the color picker dialog
  ariaLabelColorPicker: "Color picker",
  // Screen reader text for the financial charts toolbar
  ariaLabelFinancialCharts: "Financial Charts",
  // Screen reader text for the legend toolbar
  ariaLabelLegend: "Legend",
  // Screen reader text for the legend pagination button
  ariaLabelLegendPagination: "Legend Pagination",
  // Screen reader text for the previous legend page button
  ariaLabelLegendPagePrevious: "Previous Legend Page",
  // Screen reader text for the next legend page button
  ariaLabelLegendPageNext: "Next Legend Page",
  // Screen reader text for the an item in the legend
  ariaLabelLegendItem: "${label}, Legend item ${index}[number] of ${count}[number]",
  // Screen reader text for the an unknown item in the legend
  ariaLabelLegendItemUnknown: "Unknown legend item",
  // Screen reader text for the navigator element
  ariaLabelNavigator: "Navigator",
  // Screen reader text for an accessibility control that changes the position of the navigator's range
  ariaLabelNavigatorRange: "Range",
  // Screen reader text for an accessibility control that changes the start of the navigator's range
  ariaLabelNavigatorMinimum: "Minimum",
  // Screen reader text for an accessibility control that changes the end of the navigator's range
  ariaLabelNavigatorMaximum: "Maximum",
  // Screen reader text for ranges toolbar
  ariaLabelRangesToolbar: "Ranges",
  // Screen reader text for the settings dialog tab-bar
  ariaLabelSettingsTabBar: "Settings",
  // Screen reader text for zoom toolbar
  ariaLabelZoomToolbar: "Zoom",
  // Screen reader text for the value of the navigator's range
  ariaValuePanRange: "${min}[percent] to ${max}[percent]",
  // Alt-text for the solid line dash style menu item icon
  iconAltTextLineStyleSolid: "Solid",
  // Alt-text for the long-dashed line dash style menu item icon
  iconAltTextLineStyleDashed: "Long-dashed",
  // Alt-text for the short-dashed line dash style menu item icon
  iconAltTextLineStyleDotted: "Short-dashed",
  // Alt-text for the 'position-top' icon
  iconAltTextPositionTop: "Top",
  // Alt-text for the 'position-center' icon
  iconAltTextPositionCenter: "Center",
  // Alt-text for the 'position-bottom' icon
  iconAltTextPositionBottom: "Bottom",
  // Alt-text for the 'position-left' icon
  iconAltTextAlignLeft: "Left",
  // Alt-text for the 'align-center' icon
  iconAltTextAlignCenter: "Center",
  // Alt-text for the 'position-right' icon
  iconAltTextAlignRight: "Right",
  // Alt-text for the 'close' icon
  iconAltTextClose: "Close",
  // Default text for the 'loading data' overlay
  overlayLoadingData: "Loading data...",
  // Default text for the 'no data' overlay
  overlayNoData: "No data to display",
  // Default text for the 'no visible series' overlay
  overlayNoVisibleSeries: "No visible series",
  // Default text for the 'unsupported browser' overlay
  overlayUnsupportedBrowser: "Incompatible browser version. Please upgrade your browser.",
  // Text for frequency label in Histogram Series tooltip
  seriesHistogramTooltipFrequency: "Frequency",
  // Text for sum label in Histogram Series tooltip
  seriesHistogramTooltipSum: "${yName} (sum)",
  // Text for sum label in Histogram Series tooltip
  seriesHistogramTooltipCount: "${yName} (count)",
  // Text for sum label in Histogram Series tooltip
  seriesHistogramTooltipMean: "${yName} (mean)",
  // Text for the series type toolbar's chart type button
  toolbarSeriesTypeDropdown: "Chart Type",
  // Text for the series type toolbar's OHLC chart type button
  toolbarSeriesTypeOHLC: "OHLC",
  // Text for the series type toolbar's HLC chart type button
  toolbarSeriesTypeHLC: "HLC",
  // Text for the series type toolbar's high low chart type button
  toolbarSeriesTypeHighLow: "High Low",
  // Text for the series type toolbar's candles chart type button
  toolbarSeriesTypeCandles: "Candles",
  // Text for the series type toolbar's hollow candles chart type button
  toolbarSeriesTypeHollowCandles: "Hollow Candles",
  // Text for the series type toolbar's line chart type button
  toolbarSeriesTypeLine: "Line",
  // Text for the series type toolbar's line with markers chart type button
  toolbarSeriesTypeLineWithMarkers: "Line with Markers",
  // Text for the series type toolbar's line with step line chart type button
  toolbarSeriesTypeStepLine: "Step Line",
  // Text for the annotation toolbar's trend line button
  toolbarAnnotationsTrendLine: "Trend Line",
  // Text for the annotation toolbar's Fibonacci Retracement button
  toolbarAnnotationsFibonacciRetracement: "Fib Retracement",
  // Text for the annotation toolbar's Fibonacci Retracement Trend Based button
  toolbarAnnotationsFibonacciRetracementTrendBased: "Fib Trend Based",
  // Text for the annotation toolbar's horizontal line button
  toolbarAnnotationsHorizontalLine: "Horizontal Line",
  // Text for the annotation toolbar's vertical line button
  toolbarAnnotationsVerticalLine: "Vertical Line",
  // Text for the annotation toolbar's parallel channel button
  toolbarAnnotationsParallelChannel: "Parallel Channel",
  // Text for the annotation toolbar's disjoint channel button
  toolbarAnnotationsDisjointChannel: "Disjoint Channel",
  // Text for the annotation toolbar's clear all button
  toolbarAnnotationsClearAll: "Clear All",
  // Text for the annotation toolbar's fill color picker annotation button
  toolbarAnnotationsFillColor: "Fill Color",
  // Text for the annotation toolbar's line color picker annotation button
  toolbarAnnotationsLineColor: "Line Color",
  // Text for the annotation toolbar's line style type button
  toolbarAnnotationsLineStyle: "Line Style",
  // Text for the annotation toolbar's line stroke width button
  toolbarAnnotationsLineStrokeWidth: "Line Stroke Width",
  // Text for the annotation toolbar's settings annotation button
  toolbarAnnotationsSettings: "Settings",
  // Text for the annotation toolbar's text color picker annotation button
  toolbarAnnotationsTextColor: "Text Color",
  // Text for the annotation toolbar's text size picker annotation button
  toolbarAnnotationsTextSize: "Text Size",
  // Text for the annotation toolbar's lock annotation button
  toolbarAnnotationsLock: "Lock",
  // Text for the annotation toolbar's unlock annotation button
  toolbarAnnotationsUnlock: "Unlock",
  // Text for the annotation toolbar's delete annotation button
  toolbarAnnotationsDelete: "Delete",
  // Text for the annotation toolbar's drag handle
  toolbarAnnotationsDragHandle: "Drag Toolbar",
  // Text for the annotation toolbar's line drawings menu button
  toolbarAnnotationsLineAnnotations: "Trend Lines",
  // Text for the annotation toolbar's Fibonacci drawings menu button
  toolbarAnnotationsFibonacciAnnotations: "Fibonacci",
  // Text for the annotation toolbar's text annotations menu button
  toolbarAnnotationsTextAnnotations: "Text Annotations",
  // Text for the annotation toolbar's shapes menu button
  toolbarAnnotationsShapeAnnotations: "Arrows",
  // Text for the annotation toolbar's measurers menu button
  toolbarAnnotationsMeasurerAnnotations: "Measurers",
  // Text for the annotation toolbar's callout button
  toolbarAnnotationsCallout: "Callout",
  // Text for the annotation toolbar's comment button
  toolbarAnnotationsComment: "Comment",
  // Text for the annotation toolbar's note button
  toolbarAnnotationsNote: "Note",
  // Text for the annotation toolbar's text button
  toolbarAnnotationsText: "Text",
  // Text for the annotation toolbar's arrow button
  toolbarAnnotationsArrow: "Arrow",
  // Text for the annotation toolbar's arrow up button
  toolbarAnnotationsArrowUp: "Arrow Up",
  // Text for the annotation toolbar's arrow down button
  toolbarAnnotationsArrowDown: "Arrow Down",
  // Text for the annotation toolbar's date range button
  toolbarAnnotationsDateRange: "Date Range",
  // Text for the annotation toolbar's price range button
  toolbarAnnotationsPriceRange: "Price Range",
  // Text for the annotation toolbar's date and price range button
  toolbarAnnotationsDatePriceRange: "Date and Price",
  // Text for the annotation toolbar's quick date and price range button
  toolbarAnnotationsQuickDatePriceRange: "Measure",
  // Text for the range toolbar's 1 month button
  toolbarRange1Month: "1M",
  // Aria label for the range toolbar's 1 month button
  toolbarRange1MonthAria: "1 month",
  // Text for the range toolbar's 3 month button
  toolbarRange3Months: "3M",
  // Aria label for the range toolbar's 3 month button
  toolbarRange3MonthsAria: "3 months",
  // Text for the range toolbar's 6 month button
  toolbarRange6Months: "6M",
  // Aria label for the range toolbar's 6 month button
  toolbarRange6MonthsAria: "6 months",
  // Text for the range toolbar's year to date button
  toolbarRangeYearToDate: "YTD",
  // Aria label for the range toolbar's year to date month button
  toolbarRangeYearToDateAria: "Year to date",
  // Text for the range toolbar's 1 year button
  toolbarRange1Year: "1Y",
  // Aria label for the range toolbar's 1 year button
  toolbarRange1YearAria: "1 year",
  // Text for the range toolbar's full range button
  toolbarRangeAll: "All",
  // Aria label for the range toolbar's full range button
  toolbarRangeAllAria: "All",
  // Text for the zoom toolbar's zoom out button
  toolbarZoomZoomOut: "Zoom out",
  // Text for the zoom toolbar's zoom in button
  toolbarZoomZoomIn: "Zoom in",
  // Text for the zoom toolbar's pan left button
  toolbarZoomPanLeft: "Pan left",
  // Text for the zoom toolbar's pan right button
  toolbarZoomPanRight: "Pan right",
  // Text for the zoom toolbar's pan to the start button
  toolbarZoomPanStart: "Pan to the start",
  // Text for the zoom toolbar's pan to the end button
  toolbarZoomPanEnd: "Pan to the end",
  // Text for the zoom toolbar's pan reset button
  toolbarZoomReset: "Reset the zoom",
  // Text for the context menu's download button
  contextMenuDownload: "Download",
  // Text for the context menu's toggle series visibility button
  contextMenuToggleSeriesVisibility: "Toggle Visibility",
  // Text for the context menu's toggle other series visibility button
  contextMenuToggleOtherSeries: "Toggle Other Series",
  // Text for the context menu's zoom to point button
  contextMenuZoomToCursor: "Zoom to here",
  // Text for the context menu's pan to point button
  contextMenuPanToCursor: "Pan to here",
  // Text for the annotation dialog's header channel tab label
  dialogHeaderChannel: "Channel",
  // Text for the annotation dialog's header line tab label
  dialogHeaderLine: "Line",
  // Text for the annotation dialog's header fibonacci retracement line tab label
  dialogHeaderFibonacciRange: "Fib Retracement",
  // Text for the annotation dialog's header date range tab label
  dialogHeaderDateRange: "Date Range",
  // Text for the annotation dialog's header price range tab label
  dialogHeaderPriceRange: "Price Range",
  // Text for the annotation dialog's header date and price range tab label
  dialogHeaderDatePriceRange: "Date and Price",
  // Text for the annotation dialog's header text tab label
  dialogHeaderText: "Text",
  // Text for the annotation dialog's text alignment radio label
  dialogInputAlign: "Align",
  // Text for the annotation dialog's color picker label
  dialogInputColorPicker: "Color",
  // Text for the annotation dialog's color picker alt text
  dialogInputColorPickerAltText: "Text Color",
  // Text for the annotation dialog's fill color picker label
  dialogInputFillColorPicker: "Fill",
  // Text for the annotation dialog's fill color picker alt text
  dialogInputFillColorPickerAltText: "Fill Color",
  // Text for the annotation dialog's extend channel start checkbox
  dialogInputExtendChannelStart: "Extend channel start",
  // Text for the annotation dialog's extend channel end checkbox
  dialogInputExtendChannelEnd: "Extend channel end",
  // Text for the annotation dialog's extend line start checkbox
  dialogInputExtendLineStart: "Extend line start",
  // Text for the annotation dialog's extend line end checkbox
  dialogInputExtendLineEnd: "Extend line end",
  // Text for the annotation dialog's extend above checkbox
  dialogInputExtendAbove: "Extend above",
  // Text for the annotation dialog's extend below checkbox
  dialogInputExtendBelow: "Extend below",
  // Text for the annotation dialog's extend left checkbox
  dialogInputExtendLeft: "Extend left",
  // Text for the annotation dialog's extend right checkbox
  dialogInputExtendRight: "Extend right",
  // Text for the annotation dialog's reverse checkbox
  dialogInputReverse: "Reverse",
  // Text for the annotation dialog's show fill checkbox
  dialogInputShowFill: "Show Fill",
  // Text for the annotation dialog's font size select box label
  dialogInputFontSize: "Size",
  // Text for the annotation dialog's font size select box alt text
  dialogInputFontSizeAltText: "Font Size",
  // Text for the annotation dialog's line style radio label
  dialogInputLineStyle: "Dash",
  // Text for the annotation dialog's text position radio label
  dialogInputPosition: "Position",
  // Text for the annotation dialog's stroke width label
  dialogInputStrokeWidth: "Weight",
  // Text for the annotation dialog's stroke width label
  dialogInputStrokeWidthAltText: "Line Weight",
  // Text for the annotation dialog's Fibonacci bands label
  dialogInputFibonacciBands: "Bands",
  // Text for the annotation dialog's Fibonacci bands label
  dialogInputFibonacciBandsAltText: "Fibonacci Bands",
  // Text for text area input placeholders
  inputTextareaPlaceholder: "Add Text",
  // Text for the measurer statistics date range bars value
  measurerDateRangeBars: "${value}[number] bars",
  // Text for the measurer statistics price range value
  measurerPriceRangeValue: "${value}[number]",
  // Text for the measurer statistics price range percentage
  measurerPriceRangePercent: "${value}[percent]",
  // Text for the measurer statistics volume value
  measurerVolume: "Vol ${value}"
};

// packages/ag-charts-community/src/locale/defaultMessageFormatter.ts
var messageRegExp = /\$\{(\w+)}(?:\[(\w+)])?/gi;
var formatters = {
  number: new Intl.NumberFormat("en-US"),
  percent: new Intl.NumberFormat("en-US", { style: "percent", minimumFractionDigits: 2, maximumFractionDigits: 2 }),
  date: new Intl.DateTimeFormat("en-US", { dateStyle: "full" }),
  time: new Intl.DateTimeFormat("en-US", { timeStyle: "full" }),
  datetime: new Intl.DateTimeFormat("en-US", { dateStyle: "full", timeStyle: "full" })
};
var defaultMessageFormatter = ({ defaultValue, variables }) => {
  return defaultValue?.replaceAll(messageRegExp, (_, match, format) => {
    const value = variables[match];
    const formatter = format != null ? formatters[format] : null;
    if (format != null && formatter == null) {
      logger_exports.warnOnce(`Format style [${format}] is not supported`);
    }
    if (formatter != null) {
      return formatter.format(value);
    } else if (typeof value === "number") {
      return formatters.number.format(value);
    } else if (value instanceof Date) {
      return formatters.datetime.format(value);
    }
    return String(value);
  });
};

// packages/ag-charts-community/src/locale/localeManager.ts
var LocaleManager = class extends Listeners {
  constructor() {
    super(...arguments);
    this.localeText = void 0;
    this.getLocaleText = void 0;
  }
  setLocaleText(localeText) {
    if (this.localeText !== localeText) {
      this.localeText = localeText;
      this.dispatch("locale-changed");
    }
  }
  setLocaleTextFormatter(getLocaleText) {
    this.getLocaleText = getLocaleText;
    if (this.getLocaleText !== getLocaleText) {
      this.getLocaleText = getLocaleText;
      this.dispatch("locale-changed");
    }
  }
  t(key, variables = {}) {
    const { localeText = AG_CHARTS_LOCALE_EN_US, getLocaleText } = this;
    const defaultValue = localeText[key];
    return getLocaleText?.({ key, defaultValue, variables }) ?? defaultMessageFormatter({ key, defaultValue, variables }) ?? key;
  }
};

// packages/ag-charts-community/src/scene/canvas/hdpiCanvas.ts
var HdpiCanvas = class {
  constructor(options) {
    this.enabled = true;
    this.width = 600;
    this.height = 300;
    const { width: width2, height: height2, canvasElement, willReadFrequently = false } = options;
    this.pixelRatio = options.pixelRatio ?? getWindow("devicePixelRatio") ?? 1;
    this.element = canvasElement ?? createElement("canvas");
    this.element.style.display = "block";
    this.element.style.width = (width2 ?? this.width) + "px";
    this.element.style.height = (height2 ?? this.height) + "px";
    this.element.width = Math.round((width2 ?? this.width) * this.pixelRatio);
    this.element.height = Math.round((height2 ?? this.height) * this.pixelRatio);
    this.context = this.element.getContext("2d", { willReadFrequently });
    this.onEnabledChange();
    this.resize(width2 ?? 0, height2 ?? 0, this.pixelRatio);
    debugContext(this.context);
  }
  drawImage(context, dx = 0, dy = 0) {
    return context.drawImage(this.context.canvas, dx, dy);
  }
  toDataURL(type) {
    return this.element.toDataURL(type);
  }
  resize(width2, height2, pixelRatio) {
    if (!(width2 > 0 && height2 > 0))
      return;
    const { element: element2, context } = this;
    element2.width = Math.round(width2 * pixelRatio);
    element2.height = Math.round(height2 * pixelRatio);
    context.setTransform(pixelRatio, 0, 0, pixelRatio, 0, 0);
    element2.style.width = width2 + "px";
    element2.style.height = height2 + "px";
    this.width = width2;
    this.height = height2;
    this.pixelRatio = pixelRatio;
  }
  clear() {
    clearContext(this);
  }
  destroy() {
    this.element.remove();
    this.element.width = 0;
    this.element.height = 0;
    this.context.clearRect(0, 0, 0, 0);
    Object.freeze(this);
  }
  onEnabledChange() {
    if (this.element) {
      this.element.style.display = this.enabled ? "" : "none";
    }
  }
};
__decorateClass([
  ObserveChanges((target) => target.onEnabledChange())
], HdpiCanvas.prototype, "enabled", 2);

// packages/ag-charts-community/src/scene/layersManager.ts
var LayersManager = class {
  constructor(canvas) {
    this.canvas = canvas;
    this.debug = Debug.create(true, "scene");
    this.layersMap = /* @__PURE__ */ new Map();
    this.nextLayerId = 0;
  }
  get size() {
    return this.layersMap.size;
  }
  resize(width2, height2, pixelRatio) {
    this.canvas.resize(width2, height2, pixelRatio);
    this.layersMap.forEach(({ canvas }) => canvas.resize(width2, height2, pixelRatio));
  }
  addLayer(opts) {
    const { width: width2, height: height2, pixelRatio } = this.canvas;
    const { name } = opts;
    const canvas = new HdpiOffscreenCanvas({ width: width2, height: height2, pixelRatio });
    this.layersMap.set(canvas, {
      id: this.nextLayerId++,
      name,
      canvas
    });
    this.debug("Scene.addLayer() - layers", this.layersMap);
    return canvas;
  }
  removeLayer(canvas) {
    if (this.layersMap.has(canvas)) {
      this.layersMap.delete(canvas);
      canvas.destroy();
      this.debug("Scene.removeLayer() -  layers", this.layersMap);
    }
  }
  clear() {
    for (const layer of this.layersMap.values()) {
      layer.canvas.destroy();
    }
    this.layersMap.clear();
  }
};

// packages/ag-charts-community/src/scene/sceneDebug.ts
function formatBytes(value) {
  for (const unit of ["B", "KB", "MB", "GB"]) {
    if (value < 1536) {
      return `${value.toFixed(1)}${unit}`;
    }
    value /= 1024;
  }
  return `${value.toFixed(1)}TB}`;
}
function memoryUsage() {
  if (!("memory" in performance))
    return;
  const { totalJSHeapSize, usedJSHeapSize, jsHeapSizeLimit } = performance.memory;
  const result = [];
  for (const amount of [usedJSHeapSize, totalJSHeapSize, jsHeapSizeLimit]) {
    if (typeof amount !== "number")
      continue;
    result.push(formatBytes(amount));
  }
  return `Heap ${result.join(" / ")}`;
}
function debugStats(layersManager, debugSplitTimes, ctx, renderCtxStats, extraDebugStats = {}, seriesRect = BBox.zero) {
  if (!Debug.check("scene:stats" /* SCENE_STATS */, "scene:stats:verbose" /* SCENE_STATS_VERBOSE */))
    return;
  const {
    layersRendered = 0,
    layersSkipped = 0,
    nodesRendered = 0,
    nodesSkipped = 0,
    opsPerformed = 0,
    opsSkipped = 0
  } = renderCtxStats ?? {};
  const end2 = performance.now();
  const { start: start2, ...durations } = debugSplitTimes;
  const splits = Object.entries(durations).map(([n, t]) => {
    return time(n, t);
  }).filter((v) => v != null).join(" + ");
  const extras = Object.entries(extraDebugStats).map(([k, v]) => `${k}: ${JSON.stringify(v)}`).join(" ; ");
  const detailedStats = Debug.check("scene:stats:verbose" /* SCENE_STATS_VERBOSE */);
  const memUsage = memoryUsage();
  const stats = [
    `${time("\u23F1\uFE0F", start2, end2)} (${splits})`,
    `${extras}`,
    `Layers: ${detailedStats ? pct(layersRendered, layersSkipped) : layersManager.size}`,
    detailedStats ? `Nodes: ${pct(nodesRendered, nodesSkipped)}` : null,
    detailedStats ? `Ops: ${pct(opsPerformed, opsSkipped)}` : null,
    detailedStats && memUsage ? memUsage : null
  ].filter(isString);
  const measurer = new SimpleTextMeasurer((t) => ctx.measureText(t));
  const statsSize = new Map(stats.map((t) => [t, measurer.measureLines(t)]));
  const width2 = Math.max(...Array.from(statsSize.values(), (s) => s.width));
  const height2 = accumulate(statsSize.values(), (s) => s.height);
  const x = 2 + seriesRect.x;
  ctx.save();
  ctx.fillStyle = "white";
  ctx.fillRect(x, 0, width2, height2);
  ctx.fillStyle = "black";
  let y = 0;
  for (const [stat, size] of statsSize.entries()) {
    y += size.height;
    ctx.fillText(stat, x, y);
  }
  ctx.restore();
}
function prepareSceneNodeHighlight(ctx) {
  const config = toArray(getWindow("agChartsSceneDebug"));
  const result = [];
  for (const name of config) {
    if (name === "layout") {
      result.push("seriesRoot", "legend", "root", /.*Axis-\d+-axis.*/);
    } else {
      result.push(name);
    }
  }
  ctx.debugNodeSearch = result;
}
function debugSceneNodeHighlight(ctx, debugNodes) {
  ctx.save();
  for (const [name, node] of Object.entries(debugNodes)) {
    const bbox = Transformable.toCanvas(node);
    if (!bbox) {
      logger_exports.log(`Scene.render() - no bbox for debugged node [${name}].`);
      continue;
    }
    ctx.globalAlpha = 0.8;
    ctx.strokeStyle = "red";
    ctx.lineWidth = 1;
    ctx.strokeRect(bbox.x, bbox.y, bbox.width, bbox.height);
    ctx.fillStyle = "red";
    ctx.strokeStyle = "white";
    ctx.font = "16px sans-serif";
    ctx.textBaseline = "top";
    ctx.textAlign = "left";
    ctx.lineWidth = 2;
    ctx.strokeText(name, bbox.x, bbox.y, bbox.width);
    ctx.fillText(name, bbox.x, bbox.y, bbox.width);
  }
  ctx.restore();
}
var skippedProperties = /* @__PURE__ */ new Set();
var allowedProperties = /* @__PURE__ */ new Set([
  "gradient",
  // '_datum',
  "zIndex",
  "clipRect",
  "cachedBBox",
  "childNodeCounts",
  "path",
  "__zIndex",
  "name",
  "__scalingCenterX",
  "__scalingCenterY",
  "__rotationCenterX",
  "__rotationCenterY",
  "_previousDatum",
  "__fill",
  "__lineDash",
  "borderPath",
  "borderClipPath",
  "_clipPath"
]);
function nodeProps(node) {
  const { ...allProps } = node;
  for (const prop of Object.keys(allProps)) {
    if (allowedProperties.has(prop))
      continue;
    if (typeof allProps[prop] === "number")
      continue;
    if (typeof allProps[prop] === "string")
      continue;
    if (typeof allProps[prop] === "boolean")
      continue;
    skippedProperties.add(prop);
    delete allProps[prop];
  }
  return allProps;
}
function buildTree(node, mode) {
  if (!Debug.check(true, "scene" /* SCENE */)) {
    return {};
  }
  let order = 0;
  return {
    node: mode === "json" ? nodeProps(node) : node,
    name: node.name ?? node.id,
    dirty: node.dirty,
    ...Array.from(node.children(), (c) => buildTree(c, mode)).reduce(
      (result, childTree) => {
        let { name: treeNodeName } = childTree;
        const {
          node: { visible, opacity, zIndex, translationX, translationY, rotation, scalingX, scalingY },
          node: childNode
        } = childTree;
        if (!visible || opacity <= 0) {
          treeNodeName = `(${treeNodeName})`;
        }
        if (Group.is(childNode) && childNode.renderToOffscreenCanvas) {
          treeNodeName = `*${treeNodeName}*`;
        }
        const zIndexString = Array.isArray(zIndex) ? `(${zIndex.join(", ")})` : zIndex;
        const key = [
          `${(order++).toString().padStart(3, "0")}|`,
          `${treeNodeName ?? "<unknown>"}`,
          `z: ${zIndexString}`,
          translationX && `x: ${translationX}`,
          translationY && `y: ${translationY}`,
          rotation && `r: ${rotation}`,
          scalingX != null && scalingX !== 1 && `sx: ${scalingX}`,
          scalingY != null && scalingY !== 1 && `sy: ${scalingY}`
        ].filter((v) => !!v).join(" ");
        let selectedKey = key;
        let index = 1;
        while (result[selectedKey] != null && index < 100) {
          selectedKey = `${key} (${index++})`;
        }
        result[selectedKey] = childTree;
        return result;
      },
      {}
    )
  };
}
function buildDirtyTree(node) {
  if (!node.dirty) {
    return { dirtyTree: {}, paths: [] };
  }
  const childrenDirtyTree = Array.from(node.children(), (c) => buildDirtyTree(c)).filter((c) => c.paths.length > 0);
  const name = Group.is(node) ? node.name ?? node.id : node.id;
  const paths = childrenDirtyTree.length ? childrenDirtyTree.flatMap((c) => c.paths).map((p) => `${name}.${p}`) : [name];
  return {
    dirtyTree: {
      name,
      node,
      dirty: node.dirty,
      ...childrenDirtyTree.map((c) => c.dirtyTree).filter((t) => t.dirty != null).reduce((result, childTree) => {
        result[childTree.name ?? "<unknown>"] = childTree;
        return result;
      }, {})
    },
    paths
  };
}
function pct(rendered, skipped) {
  const total = rendered + skipped;
  return `${rendered} / ${total} (${Math.round(100 * rendered / total)}%)`;
}
function time(name, start2, end2) {
  const duration = end2 != null ? end2 - start2 : start2;
  return `${name}: ${Math.round(duration * 100) / 100}ms`;
}
function accumulate(iterator, mapper) {
  let sum = 0;
  for (const item of iterator) {
    sum += mapper(item);
  }
  return sum;
}

// packages/ag-charts-community/src/scene/scene.ts
var Scene = class {
  constructor(canvasOptions) {
    this.debug = Debug.create(true, "scene" /* SCENE */);
    this.id = createId(this);
    this.root = null;
    this.pendingSize = null;
    this.isDirty = false;
    this.canvas = new HdpiCanvas(canvasOptions);
    this.layersManager = new LayersManager(this.canvas);
  }
  get width() {
    return this.pendingSize?.[0] ?? this.canvas.width;
  }
  get height() {
    return this.pendingSize?.[1] ?? this.canvas.height;
  }
  get pixelRatio() {
    return this.pendingSize?.[2] ?? this.canvas.pixelRatio;
  }
  /** @deprecated v10.2.0 Only used by AG Grid Sparklines */
  setContainer(value) {
    const { element: element2 } = this.canvas;
    element2.parentElement?.removeChild(element2);
    value.appendChild(element2);
    return this;
  }
  setRoot(node) {
    if (this.root === node) {
      return this;
    }
    this.isDirty = true;
    this.root?._setLayerManager();
    this.root = node;
    if (node) {
      node.visible = true;
      node._setLayerManager(this.layersManager);
    }
    return this;
  }
  clear() {
    this.canvas.clear();
  }
  attachNode(node) {
    this.appendChild(node);
    return () => this.removeChild(node);
  }
  appendChild(node) {
    this.root?.appendChild(node);
    return this;
  }
  removeChild(node) {
    this.root?.removeChild(node);
    return this;
  }
  download(fileName, fileFormat) {
    downloadUrl(this.canvas.toDataURL(fileFormat), fileName?.trim() ?? "image");
  }
  /** NOTE: Integrated Charts undocumented image download method. */
  getDataURL(fileFormat) {
    return this.canvas.toDataURL(fileFormat);
  }
  resize(width2, height2, pixelRatio) {
    width2 = Math.round(width2);
    height2 = Math.round(height2);
    pixelRatio ?? (pixelRatio = this.pixelRatio);
    if (width2 > 0 && height2 > 0 && (width2 !== this.width || height2 !== this.height || pixelRatio !== this.pixelRatio)) {
      this.pendingSize = [width2, height2, pixelRatio];
      this.isDirty = true;
      return true;
    }
    return false;
  }
  render(opts) {
    const { debugSplitTimes = { start: performance.now() }, extraDebugStats, seriesRect } = opts ?? {};
    const {
      canvas,
      canvas: { context: ctx } = {},
      root,
      pendingSize,
      width: width2,
      height: height2,
      pixelRatio: devicePixelRatio
    } = this;
    if (!ctx) {
      return;
    }
    const renderStartTime = performance.now();
    if (pendingSize) {
      this.layersManager.resize(...pendingSize);
      this.pendingSize = null;
    }
    if (root && !root.visible) {
      this.isDirty = false;
      return;
    }
    if (root?.dirty === false && !this.isDirty) {
      if (this.debug.check()) {
        this.debug("Scene.render() - no-op", {
          tree: buildTree(root, "console")
        });
      }
      debugStats(this.layersManager, debugSplitTimes, ctx, void 0, extraDebugStats, seriesRect);
      return;
    }
    const renderCtx = {
      ctx,
      width: width2,
      height: height2,
      devicePixelRatio,
      debugNodes: {}
    };
    if (Debug.check("scene:stats:verbose" /* SCENE_STATS_VERBOSE */)) {
      renderCtx.stats = {
        layersRendered: 0,
        layersSkipped: 0,
        nodesRendered: 0,
        nodesSkipped: 0,
        opsPerformed: 0,
        opsSkipped: 0
      };
    }
    prepareSceneNodeHighlight(renderCtx);
    let canvasCleared = false;
    if (root?.dirty !== false) {
      canvasCleared = true;
      canvas.clear();
    }
    if (root && Debug.check("scene:dirtyTree" /* SCENE_DIRTY_TREE */)) {
      const { dirtyTree, paths } = buildDirtyTree(root);
      Debug.create("scene:dirtyTree" /* SCENE_DIRTY_TREE */)("Scene.render() - dirtyTree", { dirtyTree, paths });
    }
    if (root && canvasCleared) {
      if (root.visible) {
        root.preRender(renderCtx);
      }
      if (this.debug.check()) {
        const tree = buildTree(root, "console");
        this.debug("Scene.render() - before", {
          canvasCleared,
          tree
        });
      }
      if (root.visible) {
        ctx.save();
        root.render(renderCtx);
        ctx.restore();
      }
    }
    debugSplitTimes["\u270D\uFE0F"] = performance.now() - renderStartTime;
    ctx.verifyDepthZero?.();
    this.isDirty = false;
    debugStats(this.layersManager, debugSplitTimes, ctx, renderCtx.stats, extraDebugStats, seriesRect);
    debugSceneNodeHighlight(ctx, renderCtx.debugNodes);
    if (root && this.debug.check()) {
      this.debug("Scene.render() - after", {
        tree: buildTree(root, "console"),
        canvasCleared
      });
    }
  }
  toSVG() {
    const { root, width: width2, height: height2 } = this;
    if (root == null)
      return;
    return Node.toSVG(root, width2, height2);
  }
  /** Alternative to destroy() that preserves re-usable resources. */
  strip() {
    const { context, pixelRatio } = this.canvas;
    context.setTransform(pixelRatio, 0, 0, pixelRatio, 0, 0);
    this.layersManager.clear();
    this.setRoot(null);
    this.isDirty = false;
  }
  destroy() {
    this.strip();
    this.canvas.destroy();
    Object.assign(this, { canvas: void 0 });
  }
};
Scene.className = "Scene";

// packages/ag-charts-community/src/util/callbackCache.ts
var CallbackCache = class {
  constructor() {
    this.cache = /* @__PURE__ */ new WeakMap();
  }
  call(fn, ...params) {
    let serialisedParams;
    let paramCache = this.cache.get(fn);
    try {
      serialisedParams = JSON.stringify(params);
    } catch {
      return this.invoke(fn, params, paramCache);
    }
    if (paramCache == null) {
      paramCache = /* @__PURE__ */ new Map();
      this.cache.set(fn, paramCache);
    }
    if (!paramCache.has(serialisedParams)) {
      return this.invoke(fn, params, paramCache, serialisedParams);
    }
    return paramCache.get(serialisedParams);
  }
  invoke(fn, params, paramCache, serialisedParams) {
    try {
      const result = fn(...params);
      if (paramCache && serialisedParams != null) {
        paramCache.set(serialisedParams, result);
      }
      return result;
    } catch (e) {
      logger_exports.warnOnce(`User callback errored, ignoring`, e);
      return;
    }
  }
  invalidateCache() {
    this.cache = /* @__PURE__ */ new WeakMap();
  }
};

// packages/ag-charts-community/src/chart/annotation/annotationManager.ts
var AnnotationManager = class extends BaseManager {
  constructor(annotationRoot) {
    super();
    this.annotationRoot = annotationRoot;
    this.mementoOriginatorKey = "annotations";
    this.annotations = [];
  }
  createMemento() {
    return this.annotations;
  }
  guardMemento(blob) {
    return blob == null || isArray(blob);
  }
  restoreMemento(_version, _mementoVersion, memento) {
    this.annotations = this.cleanData(memento ?? []).map((annotation) => {
      const annotationTheme = this.getAnnotationTypeStyles(annotation.type);
      return mergeDefaults(annotation, annotationTheme);
    });
    this.listeners.dispatch("restore-annotations", {
      type: "restore-annotations",
      annotations: this.annotations
    });
  }
  updateData(annotations) {
    this.annotations = this.cleanData(annotations ?? []);
  }
  attachNode(node) {
    this.annotationRoot.append(node);
    return () => {
      this.annotationRoot?.removeChild(node);
      return this;
    };
  }
  setAnnotationStyles(styles) {
    this.styles = styles;
  }
  getAnnotationTypeStyles(type) {
    return this.styles?.[type];
  }
  cleanData(annotations) {
    for (const annotation of annotations) {
      if ("textAlign" in annotation)
        delete annotation.textAlign;
    }
    return annotations;
  }
};

// packages/ag-charts-community/src/chart/axis/axisManager.ts
var AxisManager = class {
  constructor(sceneRoot) {
    this.sceneRoot = sceneRoot;
    this.axes = /* @__PURE__ */ new Map();
    this.axisGridGroup = new Group({ name: "Axes-Grids", zIndex: 1 /* AXIS_GRID */ });
    this.axisGroup = new Group({ name: "Axes", zIndex: 2 /* AXIS */ });
    this.axisLabelGroup = new Group({ name: "Axes-Labels", zIndex: 13 /* SERIES_LABEL */ });
    this.axisCrosslineRangeGroup = new Group({
      name: "Axes-Crosslines-Range",
      zIndex: 4 /* SERIES_CROSSLINE_RANGE */
    });
    this.axisCrosslineLineGroup = new Group({
      name: "Axes-Crosslines-Line",
      zIndex: 8 /* SERIES_CROSSLINE_LINE */
    });
    this.axisCrosslineLabelGroup = new Group({
      name: "Axes-Crosslines-Label",
      zIndex: 13 /* SERIES_LABEL */
    });
    this.sceneRoot.appendChild(this.axisGroup);
    this.sceneRoot.appendChild(this.axisGridGroup);
    this.sceneRoot.appendChild(this.axisLabelGroup);
    this.sceneRoot.appendChild(this.axisCrosslineRangeGroup);
    this.sceneRoot.appendChild(this.axisCrosslineLineGroup);
    this.sceneRoot.appendChild(this.axisCrosslineLabelGroup);
  }
  updateAxes(oldAxes, newAxes) {
    const axisNodes = {
      axisNode: this.axisGroup,
      gridNode: this.axisGridGroup,
      labelNode: this.axisLabelGroup,
      crossLineRangeNode: this.axisCrosslineRangeGroup,
      crossLineLineNode: this.axisCrosslineLineGroup,
      crossLineLabelNode: this.axisCrosslineLabelGroup
    };
    for (const axis of oldAxes) {
      if (newAxes.includes(axis))
        continue;
      axis.detachAxis(axisNodes);
      axis.destroy();
    }
    for (const axis of newAxes) {
      if (oldAxes?.includes(axis))
        continue;
      axis.attachAxis(axisNodes);
    }
    this.axes.clear();
    for (const axis of newAxes) {
      const ctx = axis.createAxisContext();
      if (this.axes.has(ctx.direction)) {
        this.axes.get(ctx.direction)?.push(ctx);
      } else {
        this.axes.set(ctx.direction, [ctx]);
      }
    }
  }
  getAxisContext(direction) {
    return this.axes.get(direction) ?? [];
  }
  destroy() {
    this.axes.clear();
    this.sceneRoot.removeChild(this.axisGroup);
    this.sceneRoot.removeChild(this.axisGridGroup);
  }
};

// packages/ag-charts-community/src/chart/data/dataService.ts
var DataService = class extends Listeners {
  constructor(animationManager) {
    super();
    this.animationManager = animationManager;
    this.dispatchOnlyLatest = true;
    this.dispatchThrottle = 0;
    this.requestThrottle = 300;
    this.isLoadingInitialData = false;
    this.isLoadingData = false;
    this.freshRequests = [];
    this.requestCounter = 0;
    this.debug = Debug.create(true, "data-model", "data-source");
    this.throttledFetch = this.createThrottledFetch(this.requestThrottle);
    this.throttledDispatch = this.createThrottledDispatch(this.dispatchThrottle);
  }
  updateCallback(dataSourceCallback) {
    if (typeof dataSourceCallback !== "function")
      return;
    this.debug("DataService - updated data source callback");
    this.dataSourceCallback = dataSourceCallback;
    this.isLoadingInitialData = true;
    this.animationManager.skip();
    this.dispatch("data-source-change");
  }
  clearCallback() {
    this.dataSourceCallback = void 0;
  }
  load(params) {
    this.isLoadingData = true;
    this.throttledFetch(params);
  }
  isLazy() {
    return this.dataSourceCallback != null;
  }
  isLoading() {
    return this.isLazy() && (this.isLoadingInitialData || this.isLoadingData);
  }
  createThrottledFetch(requestThrottle) {
    return throttle(
      (params) => this.fetch(params).catch((e) => logger_exports.error("callback failed", e)),
      requestThrottle,
      {
        leading: false,
        trailing: true
      }
    );
  }
  createThrottledDispatch(dispatchThrottle) {
    return throttle(
      (id, data) => {
        this.debug(`DataService - dispatching 'data-load' | ${id}`);
        this.dispatch("data-load", { type: "data-load", data });
      },
      dispatchThrottle,
      {
        leading: true,
        trailing: true
      }
    );
  }
  async fetch(params) {
    if (!this.dataSourceCallback) {
      throw new Error("DataService - [dataSource.getData] callback not initialised");
    }
    const start2 = performance.now();
    const id = this.requestCounter++;
    this.debug(`DataService - requesting | ${id}`);
    this.freshRequests.push(id);
    let response;
    try {
      response = await this.dataSourceCallback(params);
      this.debug(`DataService - response | ${performance.now() - start2}ms | ${id}`);
    } catch (error2) {
      this.debug(`DataService - request failed | ${id}`);
      logger_exports.errorOnce(`DataService - request failed | [${error2}]`);
    }
    this.isLoadingInitialData = false;
    const requestIndex = this.freshRequests.findIndex((rid) => rid === id);
    if (requestIndex === -1 || this.dispatchOnlyLatest && requestIndex !== this.freshRequests.length - 1) {
      this.debug(`DataService - discarding stale request | ${id}`);
      return;
    }
    this.freshRequests = this.freshRequests.slice(requestIndex + 1);
    if (this.freshRequests.length === 0) {
      this.isLoadingData = false;
    }
    if (Array.isArray(response)) {
      this.throttledDispatch(id, response);
    } else {
      this.dispatch("data-error");
    }
  }
};
__decorateClass([
  ActionOnSet({
    newValue(dispatchThrottle) {
      this.throttledDispatch = this.createThrottledDispatch(dispatchThrottle);
    }
  })
], DataService.prototype, "dispatchThrottle", 2);
__decorateClass([
  ActionOnSet({
    newValue(requestThrottle) {
      this.throttledFetch = this.createThrottledFetch(requestThrottle);
    }
  })
], DataService.prototype, "requestThrottle", 2);

// packages/ag-charts-community/src/chart/interaction/animationBatch.ts
var AnimationBatch = class {
  constructor(maxAnimationTime) {
    this.maxAnimationTime = maxAnimationTime;
    this.debug = Debug.create(true, "animation");
    this.controllers = /* @__PURE__ */ new Map();
    this.stoppedCbs = /* @__PURE__ */ new Set();
    this.currentPhase = 0;
    this.phases = new Map(PHASE_ORDER.map((p) => [p, []]));
    this.skipAnimations = false;
    this.animationTimeConsumed = 0;
    /** Guard against premature animation execution. */
    this.isReady = false;
  }
  get size() {
    return this.controllers.size;
  }
  get consumedTimeMs() {
    return this.animationTimeConsumed;
  }
  isActive() {
    return this.controllers.size > 0;
  }
  getActiveControllers() {
    return this.phases.get(PHASE_ORDER[this.currentPhase]) ?? [];
  }
  checkOverlappingId(id) {
    if (id != null && this.controllers.has(id)) {
      this.controllers.get(id).stop();
      this.debug(`Skipping animation batch due to update of existing animation: ${id}`);
      this.skip();
    }
  }
  addAnimation(animation) {
    if (animation.isComplete)
      return;
    const animationPhaseIdx = PHASE_ORDER.indexOf(animation.phase);
    if (animationPhaseIdx < this.currentPhase) {
      this.debug(`Skipping animation due to being for an earlier phase`, animation.id);
      animation.stop();
      return;
    }
    this.controllers.set(animation.id, animation);
    this.phases.get(animation.phase)?.push(animation);
  }
  removeAnimation(animation) {
    this.controllers.delete(animation.id);
    const phase = this.phases.get(animation.phase);
    const index = phase?.indexOf(animation);
    if (index != null && index >= 0) {
      phase?.splice(index, 1);
    }
  }
  progress(deltaTime) {
    if (!this.isReady)
      return;
    let unusedTime = deltaTime === 0 ? 0.01 : deltaTime;
    const refresh = () => {
      const phase2 = PHASE_ORDER[this.currentPhase];
      return {
        phaseControllers: [...this.getActiveControllers()],
        phase: phase2,
        phaseMeta: PHASE_METADATA[phase2]
      };
    };
    let { phase, phaseControllers, phaseMeta } = refresh();
    const arePhasesComplete = () => PHASE_ORDER[this.currentPhase] == null;
    const progressPhase = () => {
      ({ phase, phaseControllers, phaseMeta } = refresh());
      while (!arePhasesComplete() && phaseControllers.length === 0) {
        this.currentPhase++;
        ({ phase, phaseControllers, phaseMeta } = refresh());
        this.debug(`AnimationBatch - phase changing to ${phase}`, { unusedTime }, phaseControllers);
      }
    };
    const total = this.controllers.size;
    this.debug(`AnimationBatch - ${deltaTime}ms; phase ${phase} with ${phaseControllers?.length} of ${total}`);
    do {
      const phaseDeltaTime = unusedTime;
      const skipPhase = phaseMeta.skipIfNoEarlierAnimations && this.animationTimeConsumed === 0;
      let completeCount = 0;
      for (const controller of phaseControllers) {
        if (skipPhase) {
          controller.stop();
        } else {
          unusedTime = Math.min(controller.update(phaseDeltaTime), unusedTime);
        }
        if (controller.isComplete) {
          completeCount++;
          this.removeAnimation(controller);
        }
      }
      this.animationTimeConsumed += phaseDeltaTime - unusedTime;
      this.debug(`AnimationBatch - updated ${phaseControllers.length} controllers; ${completeCount} completed`);
      this.debug(`AnimationBatch - animationTimeConsumed: ${this.animationTimeConsumed}`);
      progressPhase();
    } while (unusedTime > 0 && !arePhasesComplete());
    if (this.animationTimeConsumed > this.maxAnimationTime) {
      this.debug(`Animation batch exceeded max animation time, skipping`, [...this.controllers]);
      this.stop();
    }
  }
  ready() {
    if (this.isReady)
      return;
    this.isReady = true;
    this.debug(`AnimationBatch - ready; skipped: ${this.skipAnimations}`, [...this.controllers]);
    let skipAll = true;
    for (const [, controller] of this.controllers) {
      if (controller.duration > 0 && PHASE_METADATA[controller.phase].skipIfNoEarlierAnimations !== true) {
        skipAll = false;
        break;
      }
    }
    if (!skipAll) {
      for (const [, controller] of this.controllers) {
        if (controller.autoplay) {
          controller.play(true);
        }
      }
    }
  }
  skip(skip = true) {
    if (this.skipAnimations === false && skip === true) {
      for (const controller of this.controllers.values()) {
        controller.stop();
      }
      this.controllers.clear();
    }
    this.skipAnimations = skip;
  }
  play() {
    for (const controller of this.controllers.values()) {
      controller.play();
    }
  }
  stop() {
    for (const controller of this.controllers.values()) {
      try {
        controller.stop();
        this.removeAnimation(controller);
      } catch (error2) {
        logger_exports.error("Error during animation stop", error2);
      }
    }
    this.dispatchStopped();
  }
  stopByAnimationId(id) {
    if (id != null && this.controllers.has(id)) {
      const controller = this.controllers.get(id);
      if (controller) {
        controller.stop();
        this.removeAnimation(controller);
      }
    }
  }
  stopByAnimationGroupId(id) {
    for (const controller of this.controllers.values()) {
      if (controller.groupId === id) {
        this.stopByAnimationId(controller.id);
      }
    }
  }
  dispatchStopped() {
    this.stoppedCbs.forEach((cb) => cb());
    this.stoppedCbs.clear();
  }
  isSkipped() {
    return this.skipAnimations;
  }
  destroy() {
    this.stop();
    this.controllers.clear();
  }
};

// packages/ag-charts-community/src/chart/interaction/interactionManager.ts
var InteractionManager = class {
  constructor() {
    this.stateQueue = 32 /* Default */ | 2 /* Animation */;
  }
  pushState(state) {
    this.stateQueue |= state;
  }
  popState(state) {
    this.stateQueue &= ~state;
  }
  isState(allowedStates) {
    return !!(this.stateQueue & -this.stateQueue & allowedStates);
  }
};

// packages/ag-charts-community/src/chart/interaction/animationManager.ts
function validAnimationDuration(testee) {
  if (testee == null)
    return true;
  return !isNaN(testee) && testee >= 0 && testee <= 2;
}
var AnimationManager = class {
  constructor(interactionManager, chartUpdateMutex) {
    this.interactionManager = interactionManager;
    this.chartUpdateMutex = chartUpdateMutex;
    this.defaultDuration = 1e3;
    this.batch = new AnimationBatch(this.defaultDuration * 1.5);
    this.debug = Debug.create(true, "animation");
    this.events = new EventEmitter();
    this.rafAvailable = typeof requestAnimationFrame !== "undefined";
    this.isPlaying = true;
    this.requestId = null;
    this.skipAnimations = true;
    this.currentAnonymousAnimationId = 0;
  }
  addListener(eventName, listener) {
    return this.events.on(eventName, listener);
  }
  /**
   * Create an animation to tween a value between the `from` and `to` properties. If an animation already exists
   * with the same `id`, immediately stop it.
   */
  animate(opts) {
    const batch = this.batch;
    try {
      batch.checkOverlappingId(opts.id);
    } catch (error2) {
      this.failsafeOnError(error2);
      return;
    }
    let { id } = opts;
    if (id == null) {
      id = `__${this.currentAnonymousAnimationId}`;
      this.currentAnonymousAnimationId += 1;
    }
    const skip = this.isSkipped() || opts.phase === "none";
    if (skip) {
      this.debug("AnimationManager - skipping animation");
    }
    const { delay, duration } = opts;
    if (!validAnimationDuration(delay)) {
      throw new Error(`Animation delay of ${delay} is unsupported (${id})`);
    }
    if (!validAnimationDuration(duration)) {
      throw new Error(`Animation duration of ${duration} is unsupported (${id})`);
    }
    const animation = new Animation({
      ...opts,
      id,
      skip,
      autoplay: this.isPlaying ? opts.autoplay : false,
      phase: opts.phase,
      defaultDuration: this.defaultDuration
    });
    if (this.forceTimeJump(animation, this.defaultDuration)) {
      return;
    }
    this.batch.addAnimation(animation);
    return animation;
  }
  play() {
    if (this.isPlaying) {
      return;
    }
    this.isPlaying = true;
    this.debug("AnimationManager.play()");
    try {
      this.batch.play();
    } catch (error2) {
      this.failsafeOnError(error2);
    }
    this.requestAnimation();
  }
  stop() {
    this.isPlaying = false;
    this.cancelAnimation();
    this.debug("AnimationManager.stop()");
    this.batch.stop();
  }
  stopByAnimationId(id) {
    try {
      this.batch.stopByAnimationId(id);
    } catch (error2) {
      this.failsafeOnError(error2);
    }
  }
  stopByAnimationGroupId(id) {
    try {
      this.batch.stopByAnimationGroupId(id);
    } catch (error2) {
      this.failsafeOnError(error2);
    }
  }
  reset() {
    if (this.isPlaying) {
      this.stop();
      this.play();
    } else {
      this.stop();
    }
  }
  skip(skip = true) {
    this.skipAnimations = skip;
  }
  isSkipped() {
    return !this.rafAvailable || this.skipAnimations || this.batch.isSkipped();
  }
  isActive() {
    return this.isPlaying && this.batch.isActive();
  }
  skipCurrentBatch() {
    if (this.debug.check()) {
      this.debug(`AnimationManager - skipCurrentBatch()`, { stack: new Error().stack });
    }
    this.batch.skip();
  }
  /** Mocking point for tests to guarantee that animation updates happen. */
  isSkippingFrames() {
    return true;
  }
  /** Mocking point for tests to capture requestAnimationFrame callbacks. */
  scheduleAnimationFrame(cb) {
    this.requestId = getWindow().requestAnimationFrame((t) => {
      cb(t).catch((e) => logger_exports.error(e));
    });
  }
  /** Mocking point for tests to skip animations to a specific point in time. */
  forceTimeJump(_animation, _defaultDuration) {
    return false;
  }
  requestAnimation() {
    if (!this.rafAvailable)
      return;
    if (!this.batch.isActive() || this.requestId !== null)
      return;
    let prevTime;
    const onAnimationFrame = async (time2) => {
      const executeAnimationFrame = () => {
        const deltaTime = time2 - (prevTime ?? time2);
        prevTime = time2;
        this.debug("AnimationManager - onAnimationFrame()", {
          controllersCount: this.batch.size,
          deltaTime
        });
        this.interactionManager.pushState(2 /* Animation */);
        try {
          this.batch.progress(deltaTime);
        } catch (error2) {
          this.failsafeOnError(error2);
        }
        this.events.emit("animation-frame", {
          type: "animation-frame",
          deltaMs: deltaTime
        });
      };
      if (this.isSkippingFrames()) {
        await this.chartUpdateMutex.acquireImmediately(executeAnimationFrame);
      } else {
        await this.chartUpdateMutex.acquire(executeAnimationFrame);
      }
      if (this.batch.isActive()) {
        this.scheduleAnimationFrame(onAnimationFrame);
      } else {
        this.batch.stop();
        this.events.emit("animation-stop", {
          type: "animation-stop",
          deltaMs: this.batch.consumedTimeMs
        });
      }
    };
    this.events.emit("animation-start", {
      type: "animation-start",
      deltaMs: 0
    });
    this.scheduleAnimationFrame(onAnimationFrame);
  }
  cancelAnimation() {
    if (this.requestId === null)
      return;
    cancelAnimationFrame(this.requestId);
    this.requestId = null;
    this.startBatch();
  }
  failsafeOnError(error2, cancelAnimation = true) {
    logger_exports.error("Error during animation, skipping animations", error2);
    if (cancelAnimation) {
      this.cancelAnimation();
    }
  }
  startBatch(skipAnimations) {
    this.debug(`AnimationManager - startBatch() with skipAnimations=${skipAnimations}.`);
    this.reset();
    this.batch.destroy();
    this.batch = new AnimationBatch(this.defaultDuration * 1.5);
    if (skipAnimations === true) {
      this.batch.skip();
    }
  }
  endBatch() {
    if (this.batch.isActive()) {
      this.batch.ready();
      this.requestAnimation();
    } else {
      this.interactionManager.popState(2 /* Animation */);
      if (this.batch.isSkipped()) {
        this.batch.skip(false);
      }
    }
  }
  onBatchStop(cb) {
    this.batch.stoppedCbs.add(cb);
  }
  destroy() {
    this.stop();
    this.events.clear();
  }
};

// packages/ag-charts-community/src/chart/interaction/chartEventManager.ts
var ChartEventManager = class extends BaseManager {
  seriesEvent(type) {
    this.listeners.dispatch(type, { type });
  }
  seriesKeyNavZoom(delta2, widgetEvent) {
    const event = { type: "series-keynav-zoom", delta: delta2, widgetEvent };
    this.listeners.dispatch("series-keynav-zoom", event);
  }
  legendItemClick(legendType, series, itemId, enabled, legendItemName) {
    const event = {
      type: "legend-item-click",
      legendType,
      series,
      itemId,
      enabled,
      legendItemName
    };
    this.listeners.dispatch("legend-item-click", event);
  }
  legendItemDoubleClick(legendType, series, itemId, enabled, numVisibleItems, legendItemName) {
    const event = {
      type: "legend-item-double-click",
      legendType,
      series,
      itemId,
      enabled,
      legendItemName,
      numVisibleItems
    };
    this.listeners.dispatch("legend-item-double-click", event);
  }
};

// packages/ag-charts-community/src/chart/interaction/contextMenuRegistry.ts
var ContextMenuRegistry = class {
  constructor() {
    this.defaultActions = [];
    this.disabledActions = /* @__PURE__ */ new Set();
    this.hiddenActions = /* @__PURE__ */ new Set();
    this.listeners = new Listeners();
  }
  static check(type, event) {
    return event.type === type;
  }
  static checkCallback(desiredType, type, _callback) {
    return desiredType === type;
  }
  dispatchContext(type, pointerEvent, context, position) {
    const { sourceEvent } = pointerEvent;
    const x = position?.x ?? pointerEvent.canvasX;
    const y = position?.y ?? pointerEvent.canvasY;
    sourceEvent.stopPropagation();
    const event = { type, x, y, context, sourceEvent };
    this.listeners.dispatch("", event);
  }
  addListener(handler) {
    return this.listeners.addListener("", handler);
  }
  filterActions(type) {
    return this.defaultActions.filter((action) => {
      return action.id != null && !this.hiddenActions.has(action.id) && ["all", type].includes(action.type);
    });
  }
  registerDefaultAction(action) {
    const didAdd = action.id != null && !this.defaultActions.some(({ id }) => id === action.id);
    if (didAdd) {
      this.defaultActions.push(action);
    }
    return () => {
      const index = didAdd ? this.defaultActions.findIndex(({ id }) => id === action.id) : -1;
      if (index !== -1) {
        this.defaultActions.splice(index, 1);
      }
    };
  }
  enableAction(actionId) {
    this.disabledActions.delete(actionId);
  }
  disableAction(actionId) {
    this.disabledActions.add(actionId);
  }
  showAction(actionId) {
    this.hiddenActions.add(actionId);
  }
  hideAction(actionId) {
    this.hiddenActions.delete(actionId);
  }
  isDisabled(actionId) {
    return this.disabledActions.has(actionId);
  }
};

// packages/ag-charts-community/src/chart/interaction/highlightManager.ts
var HighlightManager = class extends BaseManager {
  constructor() {
    super(...arguments);
    this.highlightStates = new StateTracker();
  }
  updateHighlight(callerId, highlightedDatum) {
    const { activeHighlight: previousHighlight } = this;
    this.highlightStates.set(callerId, highlightedDatum);
    this.activeHighlight = this.highlightStates.stateValue();
    if (!this.isEqual(this.activeHighlight, previousHighlight)) {
      this.listeners.dispatch("highlight-change", {
        type: "highlight-change",
        currentHighlight: this.activeHighlight,
        previousHighlight,
        callerId
      });
    }
  }
  getActiveHighlight() {
    return this.activeHighlight;
  }
  isEqual(a, b) {
    return a === b || a != null && b != null && a?.series === b?.series && a?.itemId === b?.itemId && a?.datum === b?.datum;
  }
};

// packages/ag-charts-community/src/chart/series/util.ts
function visibleRangeIndices(length, [range0, range1], xRange) {
  const xMinIndex = findMinIndex(0, length - 1, (index) => {
    const x1 = xRange(index)?.[1] ?? NaN;
    return !Number.isFinite(x1) || x1 > range0;
  }) ?? 0;
  let xMaxIndex = findMaxIndex(0, length - 1, (index) => {
    const x0 = xRange(index)?.[0] ?? NaN;
    return !Number.isFinite(x0) || x0 < range1;
  }) ?? length - 1;
  xMaxIndex = Math.min(xMaxIndex + 1, length);
  return [xMinIndex, xMaxIndex];
}
function getDatumRefPoint(series, datum) {
  const refPoint = datum.yBar?.upperPoint ?? datum.midPoint ?? series.datumMidPoint?.(datum);
  if (refPoint) {
    const { x, y } = Transformable.toCanvasPoint(series.contentGroup, refPoint.x, refPoint.y);
    return { canvasX: Math.round(x), canvasY: Math.round(y) };
  }
}
function countExpandingSearch(min, max, start2, countUntil, iteratee) {
  let i = -1;
  let count = 0;
  let shift = 0;
  let reachedAnEnd = false;
  while (count < countUntil && i <= max - min) {
    i += 1;
    const index = start2 + shift;
    if (!reachedAnEnd)
      shift *= -1;
    if (shift >= 0)
      shift += 1;
    if (reachedAnEnd && shift < 0)
      shift -= 1;
    if (index < min || index > max) {
      reachedAnEnd = true;
      continue;
    }
    if (iteratee(index))
      count += 1;
  }
  return count;
}

// packages/ag-charts-community/src/chart/interaction/tooltipManager.ts
var TooltipManager = class {
  constructor(domManager, tooltip) {
    this.domManager = domManager;
    this.tooltip = tooltip;
    this.stateTracker = new StateTracker();
    this.suppressState = new StateTracker(false);
    this.appliedState = null;
    tooltip.setup(domManager);
    domManager.addListener("hidden", () => this.tooltip.hide());
  }
  updateTooltip(callerId, meta, content) {
    if (!this.tooltip.enabled)
      return;
    content ?? (content = this.stateTracker.get(callerId)?.content);
    this.stateTracker.set(callerId, { content, meta });
    this.applyStates();
  }
  removeTooltip(callerId) {
    if (!this.tooltip.enabled)
      return;
    this.stateTracker.delete(callerId);
    this.applyStates();
  }
  suppressTooltip(callerId) {
    this.suppressState.set(callerId, true);
  }
  unsuppressTooltip(callerId) {
    this.suppressState.delete(callerId);
  }
  destroy() {
    this.domManager.removeStyles("tooltip");
  }
  applyStates() {
    const id = this.stateTracker.stateId();
    const state = id ? this.stateTracker.get(id) : null;
    if (this.suppressState.stateValue() || state?.meta == null || state?.content == null) {
      this.appliedState = null;
      this.tooltip.hide();
      return;
    }
    const canvasRect = this.domManager.getBoundingClientRect();
    const boundingRect = this.tooltip.bounds === "extended" ? this.domManager.getOverlayClientRect() : canvasRect;
    if (this.appliedState?.content === state?.content) {
      const renderInstantly = this.tooltip.isVisible();
      this.tooltip.show(boundingRect, canvasRect, state?.meta, null, renderInstantly);
    } else {
      this.tooltip.show(boundingRect, canvasRect, state?.meta, state?.content);
    }
    this.appliedState = state;
  }
  static makeTooltipMeta(event, series, datum) {
    const { canvasX, canvasY } = event;
    const tooltip = series.properties.tooltip;
    const meta = {
      canvasX,
      canvasY,
      enableInteraction: tooltip.interaction?.enabled ?? false,
      lastPointerEvent: { type: event.type, canvasX, canvasY },
      showArrow: tooltip.showArrow,
      position: {
        type: tooltip.position.type,
        xOffset: tooltip.position.xOffset,
        yOffset: tooltip.position.yOffset
      }
    };
    const refPoint = getDatumRefPoint(series, datum);
    if ((tooltip.position.type === "node" || tooltip.position.type === "sparkline") && refPoint) {
      return { ...meta, canvasX: refPoint.canvasX, canvasY: refPoint.canvasY };
    }
    return meta;
  }
  isEnteringInteractiveTooltip(event) {
    const { tooltip } = this;
    const relatedTarget = event.sourceEvent.relatedTarget;
    return tooltip.interactive && tooltip.enabled && tooltip.isVisible() && tooltip.contains(relatedTarget);
  }
};

// packages/ag-charts-community/src/chart/interaction/dragInterpreter.ts
var DRAG_THRESHOLD_PX = 3;
var DOUBLE_TAP_TIMER_MS = 505;
function makeSynthetic(device, type, event) {
  const { offsetX, offsetY, clientX, clientY, currentX, currentY, sourceEvent } = event;
  return { type, device, offsetX, offsetY, clientX, clientY, currentX, currentY, sourceEvent };
}
function checkDistanceSquared(dx, dy) {
  const distanceSquared = dx * dx + dy * dy;
  const thresholdSquared = DRAG_THRESHOLD_PX * DRAG_THRESHOLD_PX;
  return distanceSquared >= thresholdSquared;
}
var DragInterpreter = class {
  constructor(widget) {
    this.destroyFns = [];
    this.listeners = new Listeners();
    this.isDragging = false;
    this.touch = { distanceTravelledX: 0, distanceTravelledY: 0, clientX: 0, clientY: 0 };
    this.destroyFns.push(
      widget.addListener("touchstart", this.onTouchStart.bind(this)),
      widget.addListener("touchmove", this.onTouchMove.bind(this)),
      widget.addListener("touchend", this.onTouchEnd.bind(this)),
      widget.addListener("mousemove", this.onMouseMove.bind(this)),
      widget.addListener("dblclick", this.onDblClick.bind(this)),
      widget.addListener("drag-start", this.onDragStart.bind(this)),
      widget.addListener("drag-move", this.onDragMove.bind(this)),
      widget.addListener("drag-end", this.onDragEnd.bind(this))
    );
  }
  destroy() {
    this.destroyFns.forEach((fn) => fn());
    this.listeners.destroy();
  }
  addListener(type, handler) {
    return this.listeners.addListener(type, handler);
  }
  dispatch(event) {
    this.listeners.dispatch(event.type, event);
  }
  onTouchStart(e) {
    const { clientX, clientY } = e.sourceEvent.targetTouches.item(0) ?? { clientX: Infinity, clientY: Infinity };
    this.touch.distanceTravelledX = 0;
    this.touch.distanceTravelledY = 0;
    this.touch.clientX = clientX;
    this.touch.clientY = clientY;
  }
  onTouchMove(e) {
    const { clientX, clientY } = e.sourceEvent.targetTouches.item(0) ?? { clientX: Infinity, clientY: Infinity };
    this.touch.distanceTravelledX += Math.abs(this.touch.clientX - clientX);
    this.touch.distanceTravelledY += Math.abs(this.touch.clientY - clientY);
    this.touch.clientX = clientX;
    this.touch.clientY = clientY;
  }
  onTouchEnd(event) {
    event.sourceEvent.preventDefault();
  }
  onMouseMove(event) {
    this.dispatch(event);
  }
  onDblClick(event) {
    this.dispatch({ device: "mouse", ...event });
  }
  onDragStart(event) {
    this.dragStartEvent = event;
  }
  onDragMove(event) {
    if (this.dragStartEvent != null) {
      if (checkDistanceSquared(event.originDeltaX, event.originDeltaY)) {
        this.dispatch(this.dragStartEvent);
        this.dispatch({ ...this.dragStartEvent, type: "drag-move" });
        this.dragStartEvent = void 0;
        this.isDragging = true;
      }
    }
    if (this.isDragging) {
      this.dispatch(event);
    }
  }
  onDragEnd(event) {
    if (this.isDragging) {
      this.dispatch(event);
      this.isDragging = false;
      return;
    }
    if (event.device === "mouse") {
      const click = makeSynthetic("mouse", "click", event);
      this.dispatch(click);
    } else if (event.sourceEvent.type === "touchend") {
      if (checkDistanceSquared(this.touch.distanceTravelledX, this.touch.distanceTravelledY)) {
        return;
      }
      const click = makeSynthetic("touch", "click", event);
      this.dispatch(click);
      const now = Date.now();
      if (this.lastClickTime !== void 0 && now - this.lastClickTime <= DOUBLE_TAP_TIMER_MS) {
        const dblClick = makeSynthetic(event.device, "dblclick", event);
        this.dispatch(dblClick);
        this.lastClickTime = void 0;
      } else {
        this.lastClickTime = now;
      }
    }
  }
};

// packages/ag-charts-community/src/chart/interaction/widgetSet.ts
var DOMManagerWidget = class extends NativeWidget {
  constructor(elem) {
    super(elem);
  }
  addChildToDOM() {
  }
  removeChildFromDOM() {
  }
};
var WidgetSet = class {
  constructor(domManager) {
    this.seriesWidget = new DOMManagerWidget(domManager.getParent("series-area"));
    this.chartWidget = new DOMManagerWidget(domManager.getParent("canvas-proxy"));
    this.containerWidget = new DOMManagerWidget(domManager.getParent("canvas-container"));
    this.containerWidget.addChild(this.chartWidget);
    this.chartWidget.addChild(this.seriesWidget);
    this.seriesDragInterpreter = new DragInterpreter(this.seriesWidget);
  }
  destroy() {
    this.seriesDragInterpreter.destroy();
    this.seriesWidget.destroy();
    this.chartWidget.destroy();
    this.containerWidget.destroy();
  }
};

// packages/ag-charts-community/src/util/vector4.ts
var Vec4 = {
  bottomCenter,
  center,
  clone,
  collides,
  end,
  from,
  height,
  round,
  start,
  topCenter,
  origin,
  width
};
function start(a) {
  return { x: a.x1, y: a.y1 };
}
function end(a) {
  return { x: a.x2, y: a.y2 };
}
function topCenter(a) {
  return { x: (a.x1 + a.x2) / 2, y: Math.min(a.y1, a.y2) };
}
function center(a) {
  return { x: (a.x1 + a.x2) / 2, y: (a.y1 + a.y2) / 2 };
}
function bottomCenter(a) {
  return { x: (a.x1 + a.x2) / 2, y: Math.max(a.y1, a.y2) };
}
function width(a) {
  return Math.abs(a.x2 - a.x1);
}
function height(a) {
  return Math.abs(a.y2 - a.y1);
}
function round(a) {
  return { x1: Math.round(a.x1), y1: Math.round(a.y1), x2: Math.round(a.x2), y2: Math.round(a.y2) };
}
function clone(a) {
  return { x1: a.x1, y1: a.y1, x2: a.x2, y2: a.y2 };
}
function collides(a, b) {
  const an = normalise(a);
  const bn = normalise(b);
  return an.x1 <= bn.x2 && an.x2 >= bn.x1 && an.y1 <= bn.y2 && an.y2 >= bn.y1;
}
function normalise(a) {
  return { x1: Math.min(a.x1, a.x2), x2: Math.max(a.x1, a.x2), y1: Math.min(a.y1, a.y2), y2: Math.max(a.y1, a.y2) };
}
function from(a, b, c, d) {
  if (typeof a === "number") {
    return { x1: a, y1: b, x2: c, y2: d };
  }
  if ("width" in a) {
    return {
      x1: a.x,
      y1: a.y,
      x2: a.x + a.width,
      y2: a.y + a.height
    };
  }
  throw new Error(`Values can not be converted into a vector4: [${JSON.stringify(a)}] [${b}] [${c}] [${d}]`);
}
function origin() {
  return { x1: 0, y1: 0, x2: 0, y2: 0 };
}

// packages/ag-charts-community/src/util/panToBBox.ts
function normalize2(screenMin, min, screenMax, max, target) {
  return min + (max - min) * ((target - screenMin) / (screenMax - screenMin));
}
function unnormalize(screenMin, min, screenMax, max, ratio2) {
  return screenMin + (ratio2 - min) * ((screenMax - screenMin) / (max - min));
}
function calcWorldAxis(viewportMin, viewportMax, ratio2) {
  return [
    unnormalize(viewportMin, ratio2.min, viewportMax, ratio2.max, 0),
    unnormalize(viewportMin, ratio2.min, viewportMax, ratio2.max, 1)
  ];
}
function calcWorldVec4(viewport, ratioX, ratioY) {
  const [x1, x2] = calcWorldAxis(viewport.x1, viewport.x2, ratioX);
  const [y1, y2] = calcWorldAxis(viewport.y1, viewport.y2, ratioY);
  return { x1, x2, y1, y2 };
}
function panAxesUnnormalized(worldMin, worldMax, viewportMin, viewportMax, targetMin, targetMax) {
  if (viewportMin <= targetMin && targetMax <= viewportMax)
    return viewportMin;
  const minDiff = targetMin - viewportMin;
  const maxDiff = targetMax - viewportMax;
  const diff = Math.abs(minDiff) < Math.abs(maxDiff) ? minDiff : maxDiff;
  return clamp(worldMin, viewportMin + diff, worldMax);
}
function calcPanToBBoxRatios(viewportBBox, ratios, targetBBox) {
  const { x: ratioX = { min: 0, max: 1 }, y: ratioY = { min: 0, max: 1 } } = ratios;
  const target = Vec4.from(targetBBox);
  const viewport = Vec4.from(viewportBBox);
  const world = calcWorldVec4(viewport, ratioX, ratioY);
  const x = panAxesUnnormalized(world.x1, world.x2, viewport.x1, viewport.x2, target.x1, target.x2);
  const y = panAxesUnnormalized(world.y1, world.y2, viewport.y1, viewport.y2, target.y1, target.y2);
  return {
    x: {
      min: normalize2(viewport.x1, ratioX.min, viewport.x2, ratioX.max, x),
      max: normalize2(viewport.x1, ratioX.min, viewport.x2, ratioX.max, x + viewportBBox.width)
    },
    y: {
      min: normalize2(viewport.y1, ratioY.min, viewport.y2, ratioY.max, y),
      max: normalize2(viewport.y1, ratioY.min, viewport.y2, ratioY.max, y + viewportBBox.height)
    }
  };
}

// packages/ag-charts-community/src/chart/interaction/zoomManager.ts
var expectedMementoKeys = ["rangeX", "rangeY", "ratioX", "ratioY", "autoScaledAxes"];
var ZoomManagerAutoScaleAxis = class {
  constructor() {
    this.enabled = false;
    this.padding = 0;
    this.manuallyAdjusted = false;
  }
};
var rangeValidator = (axis) => attachDescription((value, context) => {
  if (!ContinuousScale.is(axis?.scale) && !OrdinalTimeScale.is(axis?.scale))
    return true;
  if (value == null || context.end == null)
    return true;
  return value <= context.end;
}, `to be less than end`);
var ZoomManager = class extends BaseManager {
  constructor(fireChartEvent, layoutManager) {
    super();
    this.fireChartEvent = fireChartEvent;
    this.mementoOriginatorKey = "zoom";
    this.axisZoomManagers = /* @__PURE__ */ new Map();
    this.state = new StateTracker(void 0, "initial");
    this.axes = [];
    this.didLayoutAxes = false;
    this.autoScaleYAxis = new ZoomManagerAutoScaleAxis();
    this.lastRestoredState = void 0;
    this.independentAxes = false;
    this.navigatorModule = false;
    this.zoomModule = false;
    // The initial state memento can not be restored until the chart has performed its first layout. Instead save it as
    // pending and restore then delete it on the first layout.
    this.pendingMemento = void 0;
    this.destroyFns.push(
      layoutManager.addListener("layout:complete", () => {
        const { pendingMemento } = this;
        const shouldPerformInitialLayout = !this.didLayoutAxes;
        this.didLayoutAxes = true;
        if (pendingMemento) {
          this.restoreMemento(pendingMemento.version, pendingMemento.mementoVersion, pendingMemento.memento);
        } else if (shouldPerformInitialLayout) {
          this.autoScaleYZoom("zoom-manager");
        }
      })
    );
  }
  createMemento() {
    return this.getMementoRanges();
  }
  guardMemento(blob, messages) {
    if (blob == null)
      return true;
    if (!isObject(blob))
      return false;
    for (const key of Object.keys(blob)) {
      if (!expectedMementoKeys.includes(key)) {
        return false;
      }
    }
    const primaryX = this.getPrimaryAxis("x" /* X */);
    const primaryY = this.getPrimaryAxis("y" /* Y */);
    const zoomMementoDefs = {
      rangeX: { start: and(or(number, date), rangeValidator(primaryX)), end: or(number, date) },
      rangeY: { start: and(or(number, date), rangeValidator(primaryY)), end: or(number, date) },
      ratioX: { start: and(ratio, lessThan("end")), end: ratio },
      ratioY: { start: and(ratio, lessThan("end")), end: ratio },
      autoScaledAxes: arrayOf(union("y"))
    };
    const result = validate(blob, zoomMementoDefs);
    if (result.errors.length > 0) {
      messages.push(...result.errors.map((e) => e.message));
      return false;
    }
    return true;
  }
  restoreMemento(version, mementoVersion, memento) {
    const { independentAxes } = this;
    if (!this.axes || !this.didLayoutAxes) {
      this.pendingMemento = { version, mementoVersion, memento };
      return;
    }
    this.pendingMemento = void 0;
    const zoom = this.getDefinedZoom();
    if (memento?.rangeX) {
      zoom.x = this.rangeToRatio(memento.rangeX, "x" /* X */) ?? { min: 0, max: 1 };
    } else if (memento?.ratioX) {
      zoom.x = {
        min: memento.ratioX.start ?? 0,
        max: memento.ratioX.end ?? 1
      };
    } else {
      zoom.x = { min: 0, max: 1 };
    }
    if (!this.navigatorModule || this.zoomModule) {
      let yAutoScale = memento?.autoScaledAxes?.includes("y");
      if (memento?.rangeY) {
        zoom.y = this.rangeToRatio(memento.rangeY, "y" /* Y */) ?? { min: 0, max: 1 };
        yAutoScale ?? (yAutoScale = false);
      } else if (memento?.ratioY) {
        zoom.y = {
          min: memento.ratioY.start ?? 0,
          max: memento.ratioY.end ?? 1
        };
        yAutoScale ?? (yAutoScale = false);
      } else {
        zoom.y = { min: 0, max: 1 };
        yAutoScale ?? (yAutoScale = true);
      }
      zoom.autoScaleYAxis = yAutoScale;
    }
    this.lastRestoredState = zoom;
    if (independentAxes !== true) {
      this.updateZoom("zoom-manager", zoom);
      return;
    }
    const primaryX = this.getPrimaryAxis("x" /* X */);
    const primaryY = this.getPrimaryAxis("y" /* Y */);
    for (const axis of [primaryX, primaryY]) {
      if (!axis)
        continue;
      this.updateAxisZoom("zoom-manager", axis.id, zoom[axis.direction]);
    }
  }
  updateAxes(axes) {
    this.axes = axes;
    const zoomManagers = new Map(axes.map((axis) => [axis.id, this.axisZoomManagers.get(axis.id)]));
    this.axisZoomManagers.clear();
    for (const axis of axes) {
      this.axisZoomManagers.set(axis.id, zoomManagers.get(axis.id) ?? new AxisZoomManager(axis));
    }
    if (this.state.size > 0 && axes.length > 0) {
      this.updateZoom(this.state.stateId(), this.state.stateValue());
    }
  }
  setIndependentAxes(independent = true) {
    this.independentAxes = independent;
  }
  setAutoScaleYAxis(enabled, padding) {
    this.autoScaleYAxis.enabled = enabled;
    this.autoScaleYAxis.padding = padding;
  }
  setNavigatorEnabled(enabled = true) {
    this.navigatorModule = enabled;
  }
  setZoomModuleEnabled(enabled = true) {
    this.zoomModule = enabled;
  }
  isZoomEnabled() {
    return this.zoomModule;
  }
  updateZoom(callerId, newZoom) {
    if (newZoom?.x && (newZoom.x.min < 0 || newZoom.x.max > 1)) {
      logger_exports.warnOnce(
        `Attempted to update x-axis zoom to an invalid ratio of [{ min: ${newZoom.x.min}, max: ${newZoom.x.max} }], expecting a ratio of 0 to 1, ignoring.`
      );
      newZoom.x = void 0;
    }
    if (newZoom?.y && (newZoom.y.min < 0 || newZoom.y.max > 1)) {
      logger_exports.warnOnce(
        `Attempted to update y-axis zoom to an invalid ratio of [{ min: ${newZoom.y.min}, max: ${newZoom.y.max} }], expecting a ratio of 0 to 1, ignoring.`
      );
      newZoom.y = void 0;
    }
    if (this.axisZoomManagers.size === 0) {
      const stateId = this.state.stateId();
      if (stateId === "initial" || stateId === callerId) {
        this.state.set(callerId, newZoom);
      }
      return;
    }
    this.state.set(callerId, newZoom);
    const autoScaleYAxis = newZoom?.autoScaleYAxis;
    if (autoScaleYAxis != null) {
      this.autoScaleYAxis.manuallyAdjusted = !autoScaleYAxis;
    }
    this.axisZoomManagers.forEach((axis) => {
      axis.updateZoom(callerId, newZoom?.[axis.getDirection()]);
    });
    this.applyChanges(callerId);
  }
  updateAxisZoom(callerId, axisId, newZoom) {
    this.axisZoomManagers.get(axisId)?.updateZoom(callerId, newZoom);
    this.applyChanges(callerId);
  }
  resetZoom(callerId) {
    this.autoScaleYAxis.manuallyAdjusted = false;
    const zoom = this.getRestoredZoom();
    this.updateZoom(callerId, {
      x: { min: zoom?.x?.min ?? 0, max: zoom?.x?.max ?? 1 },
      y: { min: zoom?.y?.min ?? 0, max: zoom?.y?.max ?? 1 },
      autoScaleYAxis: zoom?.autoScaleYAxis ?? true
    });
  }
  resetAxisZoom(callerId, axisId) {
    const axisZoomManager = this.axisZoomManagers.get(axisId);
    const direction = axisZoomManager?.getDirection();
    if (direction == null)
      return;
    const restoredZoom = this.getRestoredZoom();
    if (direction === "y" /* Y */) {
      const autoScaleYAxis = restoredZoom?.autoScaleYAxis ?? true;
      this.autoScaleYAxis.manuallyAdjusted = !autoScaleYAxis;
    }
    this.updateAxisZoom(callerId, axisId, restoredZoom?.[direction] ?? { min: 0, max: 1 });
  }
  setAxisManuallyAdjusted(_callerId, axisId) {
    const direction = this.axisZoomManagers.get(axisId)?.getDirection();
    if (direction !== "y" /* Y */)
      return;
    this.autoScaleYAxis.manuallyAdjusted = true;
  }
  updatePrimaryAxisZoom(callerId, direction, newZoom) {
    const primaryAxis = this.getPrimaryAxis(direction);
    if (!primaryAxis)
      return;
    this.updateAxisZoom(callerId, primaryAxis.id, newZoom);
  }
  panToBBox(callerId, seriesRect, target) {
    if (!this.isZoomEnabled())
      return false;
    const zoom = this.getZoom();
    if (zoom === void 0 || !zoom.x && !zoom.y)
      return false;
    const panIsPossible = seriesRect.width > 0 && seriesRect.height > 0 && Math.abs(target.width) <= Math.abs(seriesRect.width) && Math.abs(target.height) <= Math.abs(seriesRect.height);
    if (!panIsPossible) {
      logger_exports.warnOnce(`cannot pan to target BBox - chart too small?`);
      return false;
    }
    const newZoom = calcPanToBBoxRatios(seriesRect, zoom, target);
    if (this.independentAxes) {
      this.updatePrimaryAxisZoom(callerId, "x" /* X */, newZoom.x);
      this.updatePrimaryAxisZoom(callerId, "y" /* Y */, newZoom.y);
    } else {
      this.updateZoom(callerId, newZoom);
    }
    return true;
  }
  // Fire this event to signal to listeners that the view is changing through a zoom and/or pan change.
  fireZoomPanStartEvent(callerId) {
    this.listeners.dispatch("zoom-pan-start", { type: "zoom-pan-start", callerId });
  }
  extendToEnd(callerId, direction, extent2) {
    return this.extendWith(callerId, direction, (end2) => Number(end2) - extent2);
  }
  extendWith(callerId, direction, fn) {
    const axis = this.getPrimaryAxis(direction);
    if (!axis)
      return;
    const extents = this.getDomainExtents(axis);
    if (!extents)
      return;
    const [, end2] = extents;
    const start2 = fn(end2);
    const ratio2 = this.rangeToRatio({ start: start2, end: end2 }, direction);
    if (!ratio2)
      return;
    this.updateZoom(callerId, { [direction]: ratio2 });
  }
  updateWith(callerId, direction, fn) {
    const axis = this.getPrimaryAxis(direction);
    if (!axis)
      return;
    const extents = this.getDomainExtents(axis);
    if (!extents)
      return;
    let [start2, end2] = extents;
    [start2, end2] = fn(start2, end2);
    const ratio2 = this.rangeToRatio({ start: start2, end: end2 }, direction);
    if (!ratio2)
      return;
    this.updateZoom(callerId, { [direction]: ratio2 });
  }
  getZoom() {
    let x;
    let y;
    this.axisZoomManagers.forEach((axis) => {
      if (axis.getDirection() === "x" /* X */) {
        x ?? (x = axis.getZoom());
      } else if (axis.getDirection() === "y" /* Y */) {
        y ?? (y = axis.getZoom());
      }
    });
    if (x || y) {
      return { x, y };
    }
  }
  getAxisZoom(axisId) {
    return this.axisZoomManagers.get(axisId)?.getZoom() ?? { min: 0, max: 1 };
  }
  getAxisZooms() {
    const axes = {};
    for (const [axisId, axis] of this.axisZoomManagers.entries()) {
      axes[axisId] = {
        direction: axis.getDirection(),
        zoom: axis.getZoom()
      };
    }
    return axes;
  }
  getRestoredZoom() {
    return this.lastRestoredState;
  }
  getPrimaryAxisId(direction) {
    return this.getPrimaryAxis(direction)?.id;
  }
  isVisibleItemsCountAtLeast(zoom, minVisibleItems) {
    const xAxis = this.getPrimaryAxis("x" /* X */);
    const yAxis = this.getPrimaryAxis("y" /* Y */);
    const processedSeriesIds = /* @__PURE__ */ new Set();
    let visibleItemsCount = 0;
    const xVisibleRange = [zoom.x.min, zoom.x.max];
    const yVisibleRange = [zoom.y.min, zoom.y.max];
    for (const series of xAxis?.boundSeries ?? []) {
      processedSeriesIds.add(series.id);
      const remainingItems = minVisibleItems - (visibleItemsCount ?? 0);
      const seriesVisibleItems = series.getVisibleItems(xVisibleRange, yVisibleRange, remainingItems);
      visibleItemsCount += seriesVisibleItems;
      if (visibleItemsCount >= minVisibleItems)
        return true;
    }
    for (const series of yAxis?.boundSeries ?? []) {
      if (processedSeriesIds.has(series.id))
        continue;
      const remainingItems = minVisibleItems - (visibleItemsCount ?? 0);
      const seriesVisibleItems = series.getVisibleItems(xVisibleRange, yVisibleRange, remainingItems);
      visibleItemsCount += seriesVisibleItems;
      if (visibleItemsCount >= minVisibleItems)
        return true;
    }
    return processedSeriesIds.size === 0;
  }
  getMementoRanges() {
    const zoom = this.getDefinedZoom();
    let autoScaledAxes;
    if (this.autoScaleYAxis.enabled) {
      autoScaledAxes = this.autoScaleYAxis.manuallyAdjusted ? [] : ["y"];
    }
    const memento = {
      rangeX: this.getRangeDirection(zoom.x, "x" /* X */),
      rangeY: this.getRangeDirection(zoom.y, "y" /* Y */),
      ratioX: { start: zoom.x.min, end: zoom.x.max },
      ratioY: { start: zoom.y.min, end: zoom.y.max },
      autoScaledAxes
    };
    return memento;
  }
  autoScaleYZoom(callerId, applyChanges = true) {
    if (!this.isZoomEnabled())
      return;
    const { independentAxes, autoScaleYAxis } = this;
    const zoom = this.getZoom();
    if (zoom?.x == null || !autoScaleYAxis.enabled || autoScaleYAxis.manuallyAdjusted)
      return;
    const { padding } = autoScaleYAxis;
    let zoomY;
    if (zoom.x?.min === 0 && zoom.x?.max === 1) {
      zoomY = { min: 0, max: 1 };
    } else if (independentAxes) {
      zoomY = this.primaryAxisZoom("y" /* Y */, zoom.x, { padding });
    } else {
      zoomY = this.combinedAxisZoom("y" /* Y */, zoom.x, { padding });
    }
    if (zoomY == null)
      return;
    if (independentAxes) {
      const primaryAxis = this.getPrimaryAxis("y" /* Y */);
      const primaryAxisManager = primaryAxis == null ? void 0 : this.axisZoomManagers.get(primaryAxis.id);
      primaryAxisManager?.updateZoom("zoom-manager", zoomY);
    } else {
      for (const axisZoomManager of this.axisZoomManagers.values()) {
        if (axisZoomManager.getDirection() === "y" /* Y */) {
          axisZoomManager.updateZoom("zoom-manager", zoomY);
        }
      }
    }
    if (applyChanges) {
      this.applyChanges(callerId);
    }
  }
  applyChanges(callerId) {
    this.autoScaleYZoom(callerId, false);
    const changed = Array.from(this.axisZoomManagers.values(), (axis) => axis.applyChanges()).includes(true);
    if (!changed) {
      return;
    }
    const axes = {};
    for (const [axisId, axis] of this.axisZoomManagers.entries()) {
      axes[axisId] = axis.getZoom();
    }
    this.listeners.dispatch("zoom-change", { type: "zoom-change", ...this.getZoom(), axes, callerId });
    this.fireChartEvent({ type: "zoom", ...this.getMementoRanges() });
  }
  getRangeDirection(ratio2, direction) {
    const axis = this.getPrimaryAxis(direction);
    if (!axis || !ContinuousScale.is(axis.scale) && !OrdinalTimeScale.is(axis.scale))
      return;
    const extents = this.getDomainPixelExtents(axis);
    if (!extents)
      return;
    const [d0, d1] = extents;
    let start2;
    let end2;
    if (d0 <= d1) {
      start2 = axis.scale.invert(0);
      end2 = axis.scale.invert(d0 + (d1 - d0) * ratio2.max);
    } else {
      start2 = axis.scale.invert(d0 - (d0 - d1) * ratio2.min);
      end2 = axis.scale.invert(0);
    }
    return { start: start2, end: end2 };
  }
  rangeToRatio(range2, direction) {
    const axis = this.getPrimaryAxis(direction);
    if (!axis)
      return;
    const extents = this.getDomainPixelExtents(axis);
    if (!extents)
      return;
    const [d0, d1] = extents;
    let r0 = range2.start == null ? d0 : axis.scale.convert?.(range2.start);
    let r1 = range2.end == null ? d1 : axis.scale.convert?.(range2.end);
    if (!isFiniteNumber(r0) || !isFiniteNumber(r1))
      return;
    const [dMin, dMax] = [Math.min(d0, d1), Math.max(d0, d1)];
    if (r0 < dMin || r0 > dMax) {
      logger_exports.warnOnce(
        `Invalid range start [${range2.start}], expecting a value between [${axis.scale.invert?.(d0)}] and [${axis.scale.invert?.(d1)}], ignoring.`
      );
      return;
    }
    if (r1 < dMin || r1 > dMax) {
      logger_exports.warnOnce(
        `Invalid range end [${range2.end}], expecting a value between [${axis.scale.invert?.(d0)}] and [${axis.scale.invert?.(d1)}], ignoring.`
      );
      return;
    }
    r0 = Math.min(dMax, Math.max(dMin, r0));
    r1 = Math.min(dMax, Math.max(dMin, r1));
    const diff = d1 - d0;
    const min = Math.abs((r0 - d0) / diff);
    const max = Math.abs((r1 - d0) / diff);
    return { min, max };
  }
  getPrimaryAxis(direction) {
    return this.axes?.find((a) => a.direction === direction);
  }
  getDomainExtents(axis) {
    const { domain } = axis.scale;
    const d0 = domain.at(0);
    const d1 = domain.at(-1);
    if (d0 == null || d1 == null)
      return;
    return [d0, d1];
  }
  getDomainPixelExtents(axis) {
    const { domain } = axis.scale;
    const d0 = axis.scale.convert?.(domain.at(0));
    const d1 = axis.scale.convert?.(domain.at(-1));
    if (!isFiniteNumber(d0) || !isFiniteNumber(d1))
      return;
    return [d0, d1];
  }
  getDefinedZoom() {
    const zoom = this.getZoom();
    return {
      x: { min: zoom?.x?.min ?? 0, max: zoom?.x?.max ?? 1 },
      y: { min: zoom?.y?.min ?? 0, max: zoom?.y?.max ?? 1 }
    };
  }
  zoomBounds(xAxis, yAxis, zoom, padding) {
    const xScale = xAxis.scale;
    const xScaleRange = xScale.range;
    xScale.range = [0, 1];
    const yScale = yAxis.scale;
    const yScaleRange = yScale.range;
    yScale.range = [0, 1];
    let min = 1;
    let minPadding = false;
    let max = 0;
    let maxPadding = false;
    for (const series of yAxis.boundSeries) {
      const { connectsToYAxis } = series;
      const yValues = series.getRange("y" /* Y */, [zoom.min, zoom.max]);
      for (const yValue of yValues) {
        const y = yScale.convert(yValue);
        if (!Number.isFinite(y))
          continue;
        if (y < min) {
          min = y;
          minPadding = !connectsToYAxis || yValue < 0;
        }
        if (y > max) {
          max = y;
          maxPadding = !connectsToYAxis || yValue > 0;
        }
      }
    }
    if (isFiniteNumber(yAxis.min)) {
      min = 0;
    }
    if (isFiniteNumber(yAxis.max)) {
      max = 1;
    }
    xScale.range = xScaleRange;
    yScale.range = yScaleRange;
    if (min >= max)
      return;
    const totalPadding = (minPadding ? padding : 0) + (maxPadding ? padding : 0);
    const paddedDelta = Math.min((max - min) * (1 + totalPadding), 1);
    if (paddedDelta <= 0)
      return;
    if (minPadding && maxPadding) {
      const mid = (max + min) / 2;
      min = mid - paddedDelta / 2;
      max = mid + paddedDelta / 2;
    } else if (!minPadding && maxPadding) {
      max = min + paddedDelta;
    } else if (minPadding && !maxPadding) {
      min = max - paddedDelta;
    }
    if (min < 0) {
      max += -min;
      min = 0;
    } else if (max > 1) {
      min -= max - 1;
      max = 1;
    }
    return { min, max };
  }
  primaryAxisZoom(direction, zoom, { padding = 0 } = {}) {
    const crossDirection = direction === "x" /* X */ ? "y" /* Y */ : "x" /* X */;
    const xAxis = this.getPrimaryAxis(crossDirection);
    const yAxis = this.getPrimaryAxis(direction);
    if (xAxis == null || yAxis == null)
      return;
    return this.zoomBounds(xAxis, yAxis, zoom, padding);
  }
  combinedAxisZoom(direction, zoom, { padding = 0 } = {}) {
    const crossDirection = direction === "x" /* X */ ? "y" /* Y */ : "x" /* X */;
    const seriesXAxes = /* @__PURE__ */ new Map();
    for (const xAxis of this.axes) {
      if (xAxis.direction !== crossDirection)
        continue;
      for (const series of xAxis.boundSeries) {
        seriesXAxes.set(series, xAxis);
      }
    }
    let min = 1;
    let max = 0;
    for (const yAxis of this.axes) {
      if (yAxis.direction !== direction)
        continue;
      for (const series of yAxis.boundSeries) {
        const xAxis = seriesXAxes.get(series);
        if (xAxis == null)
          continue;
        const bounds = this.zoomBounds(xAxis, yAxis, zoom, padding);
        if (bounds == null)
          return;
        min = Math.min(min, bounds.min);
        max = Math.max(max, bounds.max);
      }
    }
    const delta2 = 1e-6;
    if (min < delta2)
      min = 0;
    if (max > 1 - delta2)
      max = 1;
    if (min > max)
      return;
    return { min, max };
  }
};
var AxisZoomManager = class {
  constructor(axis) {
    this.axis = axis;
    const [min = 0, max = 1] = axis.visibleRange;
    this.state = new StateTracker({ min, max });
    this.currentZoom = this.state.stateValue();
  }
  getDirection() {
    return this.axis.direction;
  }
  updateZoom(callerId, newZoom) {
    this.state.set(callerId, newZoom);
  }
  getZoom() {
    return deepClone(this.state.stateValue());
  }
  hasChanges() {
    const currentZoom = this.currentZoom;
    const pendingZoom = this.state.stateValue();
    return currentZoom.min !== pendingZoom.min || currentZoom.max !== pendingZoom.max;
  }
  applyChanges() {
    const hasChanges = this.hasChanges();
    this.currentZoom = this.state.stateValue();
    return hasChanges;
  }
};

// packages/ag-charts-community/src/chart/layout/seriesLabelLayoutManager.ts
var SeriesLabelLayoutManager = class {
  constructor() {
    this.labelData = /* @__PURE__ */ new Map();
  }
  updateLabels(placedLabelSeries, padding, seriesRect = BBox.zero) {
    const bounds = {
      x: -padding.left,
      y: -padding.top,
      width: seriesRect.width + padding.left + padding.right,
      height: seriesRect.height + padding.top + padding.bottom
    };
    const expectedSeriesId = new Set(placedLabelSeries.map((s) => s.id));
    for (const seriesId of this.labelData.keys()) {
      if (!expectedSeriesId.has(seriesId)) {
        this.labelData.delete(seriesId);
      }
    }
    for (const series of placedLabelSeries) {
      const labelData = series.getLabelData();
      if (labelData.every(isPointLabelDatum)) {
        this.labelData.set(series.id, labelData);
      }
    }
    const placedLabels = placeLabels(this.labelData, bounds, 5);
    for (const series of placedLabelSeries) {
      series.updatePlacedLabelData?.(placedLabels.get(series.id) ?? []);
    }
  }
};

// packages/ag-charts-community/src/chart/legend/legendManager.ts
var LegendManager = class extends BaseManager {
  constructor() {
    super(...arguments);
    this.mementoOriginatorKey = "legend";
    this.legendDataMap = /* @__PURE__ */ new Map();
  }
  createMemento() {
    return this.getData().filter(({ hideInLegend, isFixed }) => !hideInLegend && !isFixed).map(({ enabled, seriesId, itemId, legendItemName }) => ({
      visible: enabled,
      seriesId,
      itemId,
      legendItemName
    }));
  }
  guardMemento(blob) {
    return blob == null || isArray(blob);
  }
  restoreMemento(_version, _mementoVersion, memento) {
    memento?.forEach((datum) => {
      const { seriesId, data } = this.getRestoredData(datum) ?? {};
      if (!seriesId || !data) {
        return;
      }
      this.updateData(seriesId, data);
    });
    this.update();
  }
  getRestoredData(datum) {
    const { seriesId, itemId, legendItemName, visible } = datum;
    if (seriesId) {
      const legendData = this.legendDataMap.get(seriesId) ?? [];
      const data = legendData.map((d) => {
        const match = d.seriesId === seriesId && (!itemId || d.itemId === itemId);
        if (match && d.isFixed) {
          this.warnFixed(d.seriesId, d.itemId);
        }
        return !d.isFixed && match ? { ...d, enabled: visible } : d;
      });
      return { seriesId, data };
    }
    if (itemId == null && legendItemName == null) {
      return;
    }
    for (const legendDatum of this.getData()) {
      if (itemId != null && legendDatum.itemId !== itemId || legendItemName != null && legendDatum.legendItemName !== legendItemName) {
        continue;
      }
      if (legendDatum.isFixed) {
        this.warnFixed(legendDatum.seriesId, itemId);
        return;
      }
      const seriesLegendData = (this.legendDataMap.get(legendDatum.seriesId) ?? []).map(
        (d) => d.itemId === itemId || d.legendItemName === legendItemName ? { ...d, enabled: visible } : d
      );
      return {
        seriesId: legendDatum.seriesId,
        data: seriesLegendData
      };
    }
  }
  warnFixed(seriesId, itemId) {
    logger_exports.warnOnce(
      `The legend item with seriesId [${seriesId}] and itemId [${itemId}] is not configurable, this series item cannot be toggled through the legend.`
    );
  }
  update(data) {
    this.listeners.dispatch("legend-change", {
      type: "legend-change",
      legendData: data ?? this.getData()
    });
  }
  updateData(seriesId, data = []) {
    this.legendDataMap.set(seriesId, data);
  }
  clearData() {
    this.legendDataMap.clear();
  }
  toggleItem({
    enabled,
    seriesId,
    itemId,
    legendItemName
  }) {
    if (legendItemName) {
      this.getData().forEach((datum) => {
        const newData = (this.legendDataMap.get(datum.seriesId) ?? []).map(
          (d) => d.legendItemName === legendItemName ? { ...d, enabled } : d
        );
        this.updateData(datum.seriesId, newData);
      });
      return;
    }
    const seriesLegendData = this.getData(seriesId);
    const singleLegendItem = seriesLegendData.length === 1;
    const data = seriesLegendData.map(
      (datum) => itemId == null && singleLegendItem || datum.itemId === itemId ? { ...datum, enabled } : datum
    );
    this.updateData(seriesId, data);
  }
  getData(seriesId) {
    if (seriesId) {
      return this.legendDataMap.get(seriesId) ?? [];
    }
    return [...this.legendDataMap].reduce(
      (data, [_, legendData]) => data.concat(legendData),
      []
    );
  }
  getDatum({ seriesId, itemId } = {}) {
    return this.getData(seriesId).find((datum) => datum.itemId === itemId);
  }
  getSeriesEnabled(seriesId) {
    const data = this.getData(seriesId);
    if (data.length > 0) {
      return data.some((d) => d.enabled);
    }
  }
  getItemEnabled({ seriesId, itemId } = {}) {
    return this.getDatum({ seriesId, itemId })?.enabled ?? true;
  }
};

// packages/ag-charts-community/src/chart/series/seriesStateManager.ts
var SeriesStateManager = class {
  constructor() {
    this.groups = {};
  }
  registerSeries({ internalId, seriesGrouping, visible, type }) {
    var _a;
    if (!seriesGrouping)
      return;
    (_a = this.groups)[type] ?? (_a[type] = {});
    this.groups[type][internalId] = { grouping: seriesGrouping, visible };
  }
  updateSeries({ internalId, seriesGrouping, visible, type }) {
    if (!seriesGrouping)
      return;
    const entry = this.groups[type]?.[internalId];
    if (entry) {
      entry.grouping = seriesGrouping;
      entry.visible = visible;
    }
  }
  deregisterSeries({ internalId, type }) {
    if (this.groups[type]) {
      delete this.groups[type][internalId];
    }
    if (this.groups[type] && Object.keys(this.groups[type]).length === 0) {
      delete this.groups[type];
    }
  }
  getVisiblePeerGroupIndex({ type, seriesGrouping, visible }) {
    if (!seriesGrouping) {
      return { visibleGroupCount: visible ? 1 : 0, visibleSameStackCount: visible ? 1 : 0, index: 0 };
    }
    const visibleGroupsSet = /* @__PURE__ */ new Set();
    const visibleSameStackSet = /* @__PURE__ */ new Set();
    for (const entry of Object.values(this.groups[type] ?? {})) {
      if (!entry.visible)
        continue;
      visibleGroupsSet.add(entry.grouping.groupIndex);
      if (entry.grouping.groupIndex === seriesGrouping.groupIndex) {
        visibleSameStackSet.add(entry.grouping.stackIndex);
      }
    }
    const visibleGroups = Array.from(visibleGroupsSet);
    visibleGroups.sort((a, b) => a - b);
    return {
      visibleGroupCount: visibleGroups.length,
      visibleSameStackCount: visibleSameStackSet.size,
      index: visibleGroups.indexOf(seriesGrouping.groupIndex)
    };
  }
};

// packages/ag-charts-community/src/chart/chartUpdateType.ts
var ChartUpdateType = /* @__PURE__ */ ((ChartUpdateType2) => {
  ChartUpdateType2[ChartUpdateType2["FULL"] = 0] = "FULL";
  ChartUpdateType2[ChartUpdateType2["UPDATE_DATA"] = 1] = "UPDATE_DATA";
  ChartUpdateType2[ChartUpdateType2["PROCESS_DATA"] = 2] = "PROCESS_DATA";
  ChartUpdateType2[ChartUpdateType2["PERFORM_LAYOUT"] = 3] = "PERFORM_LAYOUT";
  ChartUpdateType2[ChartUpdateType2["SERIES_UPDATE"] = 4] = "SERIES_UPDATE";
  ChartUpdateType2[ChartUpdateType2["PRE_SCENE_RENDER"] = 5] = "PRE_SCENE_RENDER";
  ChartUpdateType2[ChartUpdateType2["SCENE_RENDER"] = 6] = "SCENE_RENDER";
  ChartUpdateType2[ChartUpdateType2["NONE"] = 7] = "NONE";
  return ChartUpdateType2;
})(ChartUpdateType || {});

// packages/ag-charts-community/src/chart/updateService.ts
var UpdateService = class {
  constructor(updateCallback) {
    this.updateCallback = updateCallback;
    this.events = new EventEmitter();
  }
  addListener(eventName, listener) {
    return this.events.on(eventName, listener);
  }
  removeListener(eventName, listener) {
    return this.events.on(eventName, listener);
  }
  destroy() {
    this.events.clear();
  }
  update(type = 0 /* FULL */, options) {
    this.updateCallback(type, options);
  }
  dispatchUpdateComplete() {
    this.events.emit("update-complete", { type: "update-complete" });
  }
  dispatchPreDomUpdate() {
    this.events.emit("pre-dom-update", { type: "pre-dom-update" });
  }
  dispatchPreSceneRender() {
    this.events.emit("pre-scene-render", { type: "pre-scene-render" });
  }
  dispatchProcessData({ series }) {
    this.events.emit("process-data", { type: "process-data", series });
  }
};

// packages/ag-charts-community/src/chart/chartContext.ts
var ChartContext = class {
  constructor(chart, vars) {
    this.callbackCache = new CallbackCache();
    this.chartEventManager = new ChartEventManager();
    this.highlightManager = new HighlightManager();
    this.layoutManager = new LayoutManager();
    this.localeManager = new LocaleManager();
    this.seriesStateManager = new SeriesStateManager();
    this.stateManager = new StateManager();
    this.seriesLabelLayoutManager = new SeriesLabelLayoutManager();
    this.contextModules = [];
    const {
      scene,
      root,
      syncManager,
      container,
      fireEvent,
      updateCallback,
      updateMutex,
      styleContainer,
      chartType
    } = vars;
    this.chartService = chart;
    this.syncManager = syncManager;
    this.domManager = new DOMManager(container, styleContainer);
    this.widgets = new WidgetSet(this.domManager);
    const canvasElement = this.domManager.addChild(
      "canvas",
      "scene-canvas",
      scene?.canvas.element
    );
    this.scene = scene ?? new Scene({ canvasElement });
    this.scene.setRoot(root);
    this.axisManager = new AxisManager(root);
    this.legendManager = new LegendManager();
    this.annotationManager = new AnnotationManager(chart.annotationRoot);
    this.chartTypeOriginator = new ChartTypeOriginator(chart);
    this.interactionManager = new InteractionManager();
    this.contextMenuRegistry = new ContextMenuRegistry();
    this.updateService = new UpdateService(updateCallback);
    this.proxyInteractionService = new ProxyInteractionService(this.localeManager, this.domManager);
    this.historyManager = new HistoryManager(this.chartEventManager);
    this.animationManager = new AnimationManager(this.interactionManager, updateMutex);
    this.dataService = new DataService(this.animationManager);
    this.tooltipManager = new TooltipManager(this.domManager, chart.tooltip);
    this.zoomManager = new ZoomManager(fireEvent, this.layoutManager);
    for (const module of moduleRegistry.byType("context")) {
      if (!module.chartTypes.includes(chartType))
        continue;
      const moduleInstance = module.moduleFactory(this);
      this.contextModules.push(moduleInstance);
      this[module.contextKey] = moduleInstance;
    }
  }
  destroy() {
    this.animationManager.destroy();
    this.highlightManager.destroy();
    this.axisManager.destroy();
    this.callbackCache.invalidateCache();
    this.chartEventManager.destroy();
    this.domManager.destroy();
    this.highlightManager.destroy();
    this.proxyInteractionService.destroy();
    this.syncManager.destroy();
    this.tooltipManager.destroy();
    this.zoomManager.destroy();
    this.widgets.destroy();
    this.contextModules.forEach((m) => m.destroy());
  }
};

// packages/ag-charts-community/src/chart/chartHighlight.ts
var ChartHighlight = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.range = "tooltip";
  }
};
__decorateClass([
  Validate(UNION(["tooltip", "node"], "a range"))
], ChartHighlight.prototype, "range", 2);

// packages/ag-charts-community/src/chart/data/caching.ts
function setsEqual(a, b) {
  if (a.size !== b.size)
    return false;
  for (const value of a) {
    if (!b.has(value))
      return false;
  }
  return true;
}
function idsMapEqual(a, b) {
  if (a == null || b == null)
    return a === b;
  if (a.size !== b.size)
    return false;
  for (const [key, aValue] of a) {
    const bValue = b.get(key);
    if (bValue == null)
      return false;
    if (!setsEqual(aValue, bValue))
      return false;
  }
  return true;
}
function propsEqual(a, b) {
  if (a.length !== b.length)
    return false;
  for (let i = 0; i < a.length; i += 1) {
    const { type: typeA, idsMap: idsMapA, scopes: scopesA, data: dataA, ...propA } = a[i];
    const { type: typeB, idsMap: idsMapB, scopes: scopesB, data: dataB, ...propB } = b[i];
    if (typeA !== typeB)
      return false;
    if (scopesA && scopesB && !arraysEqual(scopesA, scopesB))
      return false;
    if (dataA && dataB && dataA !== dataB)
      return false;
    if (!objectsEqual(propA, propB) || !idsMapEqual(idsMapA, idsMapB))
      return false;
  }
  return true;
}
function optsEqual(a, b) {
  const { props: propsA, ...restA } = a;
  const { props: propsB, ...restB } = b;
  return objectsEqual(restA, restB) && propsEqual(propsA, propsB);
}
function canReuseCachedData(cachedDataItem, data, ids, opts) {
  return data === cachedDataItem.data && arraysEqual(ids, cachedDataItem.ids) && optsEqual(opts, cachedDataItem.opts);
}

// packages/ag-charts-community/src/chart/data/dataDomain.ts
var DiscreteDomain = class _DiscreteDomain {
  constructor() {
    this.domain = /* @__PURE__ */ new Set();
  }
  static is(value) {
    return value instanceof _DiscreteDomain;
  }
  extend(val) {
    this.domain.add(val);
  }
  getDomain() {
    return Array.from(this.domain);
  }
};
var ContinuousDomain = class _ContinuousDomain {
  constructor() {
    this.domain = [Infinity, -Infinity];
  }
  static is(value) {
    return value instanceof _ContinuousDomain;
  }
  static extendDomain(values, domain = [Infinity, -Infinity]) {
    for (const value of values) {
      if (typeof value !== "number") {
        continue;
      }
      if (domain[0] > value) {
        domain[0] = value;
      }
      if (domain[1] < value) {
        domain[1] = value;
      }
    }
    return domain;
  }
  extend(value) {
    if (this.domain[0] > value) {
      this.domain[0] = value;
    }
    if (this.domain[1] < value) {
      this.domain[1] = value;
    }
  }
  getDomain() {
    return [...this.domain];
  }
};

// packages/ag-charts-community/src/chart/data/rangeLookup.ts
var MIN = 0;
var MAX = 1;
var SPAN = 2;
var RangeLookup = class {
  constructor(allValues) {
    const dataLength = allValues.reduce((acc, v) => Math.max(acc, v.length), 0);
    const sizePower = 32 - Math.clz32(dataLength);
    let maxLevelSize = 1 << sizePower;
    if (dataLength === maxLevelSize / 2) {
      maxLevelSize = maxLevelSize >>> 1;
    }
    this.maxLevelSize = maxLevelSize;
    const buffer = new Float64Array((maxLevelSize * 2 - 1) * 2).fill(NaN);
    for (const values of allValues) {
      for (let i = 0; i < values.length; i += 1) {
        const value = Number(values[i]);
        const bufferIndex = maxLevelSize + i - 1;
        const bufferMinIndex = (bufferIndex * SPAN | 0) + MIN;
        const bufferMaxIndex = (bufferIndex * SPAN | 0) + MAX;
        const prevMinValue = buffer[bufferMinIndex];
        const prevMaxValue = buffer[bufferMaxIndex];
        if (!Number.isFinite(prevMinValue) || value < prevMinValue) {
          buffer[bufferMinIndex] = value;
        }
        if (!Number.isFinite(prevMaxValue) || value > prevMaxValue) {
          buffer[bufferMaxIndex] = value;
        }
      }
    }
    for (let size = maxLevelSize / 2 | 0; size >= 1; size = size / 2 | 0) {
      const start2 = size - 1 | 0;
      const end2 = start2 + size | 0;
      for (let i = 0; i < size; i += 1) {
        const nodeIndex = start2 + i;
        const leftIndex = end2 + i * 2;
        const rightIndex = leftIndex + 1;
        const aMin = buffer[(leftIndex * SPAN | 0) + MIN];
        const bMin = buffer[(rightIndex * SPAN | 0) + MIN];
        buffer[(nodeIndex * SPAN | 0) + MIN] = !Number.isFinite(bMin) || aMin < bMin ? aMin : bMin;
        const aMax = buffer[(leftIndex * SPAN | 0) + MAX];
        const bMax = buffer[(rightIndex * SPAN | 0) + MAX];
        buffer[(nodeIndex * SPAN | 0) + MAX] = !Number.isFinite(bMax) || aMax > bMax ? aMax : bMax;
      }
    }
    this.buffer = buffer;
  }
  computeRangeInto(buffer, start2, end2, bufferIndex, currentStart, step, into) {
    const currentEnd = currentStart + step - 1;
    if (currentEnd < start2 || currentStart >= end2)
      return into;
    if (currentStart >= start2 && currentEnd < end2) {
      const min = buffer[(bufferIndex * SPAN | 0) + MIN];
      const max = buffer[(bufferIndex * SPAN | 0) + MAX];
      if (Number.isFinite(min))
        into[0] = Math.min(into[0], min);
      if (Number.isFinite(max))
        into[1] = Math.max(into[1], max);
    } else if (step > 1) {
      bufferIndex = bufferIndex * 2 | 0;
      step = step / 2 | 0;
      this.computeRangeInto(buffer, start2, end2, bufferIndex + 1 | 0, currentStart, step, into);
      this.computeRangeInto(buffer, start2, end2, bufferIndex + 2 | 0, currentStart + step, step, into);
    }
    return into;
  }
  rangeBetween(start2, end2) {
    if (start2 > end2)
      return [NaN, NaN];
    const { maxLevelSize, buffer } = this;
    const range2 = [Infinity, -Infinity];
    this.computeRangeInto(buffer, start2, end2, 0, 0, maxLevelSize, range2);
    return range2;
  }
  get range() {
    const { buffer } = this;
    return [buffer[MIN], buffer[MAX]];
  }
};

// packages/ag-charts-community/src/chart/data/dataModel.ts
var DOMAIN_RANGES = Symbol("domain-ranges");
function toKeyString(keys) {
  return keys.map((key) => isObject(key) ? JSON.stringify(key) : key).join("-");
}
function round2(val) {
  const accuracy = 1e4;
  if (Number.isInteger(val)) {
    return val;
  } else if (Math.abs(val) > accuracy) {
    return Math.trunc(val);
  }
  return Math.round(val * accuracy) / accuracy;
}
function isScoped(obj) {
  return "scopes" in obj && Array.isArray(obj.scopes);
}
var INVALID_VALUE = Symbol("invalid");
function createArray(length, value) {
  const out = [];
  for (let i = 0; i < length; i += 1) {
    out[i] = value;
  }
  return out;
}
function datumKeys(keys, columnScope, datumIndex) {
  const out = [];
  for (const k of keys) {
    const key = k.get(columnScope)?.[datumIndex];
    if (key == null)
      return;
    out.push(key);
  }
  return out;
}
function getPathComponents(path) {
  const components = [];
  let matchIndex = 0;
  let matchGroup;
  const regExp = /((?:(?:^|\.)\s*\w+|\[\s*(?:'(?:[^']|(?<!\\)\\')*'|"(?:[^"]|(?<!\\)\\")*"|-?\d+)\s*\])\s*)/g;
  while (matchGroup = regExp.exec(path)) {
    if (matchGroup.index !== matchIndex) {
      return;
    }
    matchIndex = matchGroup.index + matchGroup[0].length;
    const match = matchGroup[1].trim();
    if (match.startsWith(".")) {
      components.push(match.slice(1).trim());
    } else if (match.startsWith("[")) {
      const accessor = match.slice(1, -1).trim();
      if (accessor.startsWith(`'`)) {
        components.push(accessor.slice(1, -1).replace(/(?<!\\)\\'/g, `'`));
      } else if (accessor.startsWith(`"`)) {
        components.push(accessor.slice(1, -1).replace(/(?<!\\)\\"/g, `"`));
      } else {
        components.push(accessor);
      }
    } else {
      components.push(match);
    }
  }
  if (matchIndex !== path.length)
    return;
  return components;
}
function createPathAccessor(components) {
  return (datum) => {
    let current = datum;
    for (const component of components) {
      current = current[component];
    }
    return current;
  };
}
var DataModel = class {
  constructor(opts, mode = "standalone", suppressFieldDotNotation = false) {
    this.opts = opts;
    this.mode = mode;
    this.suppressFieldDotNotation = suppressFieldDotNotation;
    this.debug = Debug.create(true, "data-model");
    this.scopeCache = /* @__PURE__ */ new Map();
    this.keys = [];
    this.values = [];
    this.aggregates = [];
    this.groupProcessors = [];
    this.propertyProcessors = [];
    this.reducers = [];
    this.processors = [];
    this.markScopeDatumInvalid = function(scopes, data, datumIndex, invalidData) {
      for (const scope of scopes) {
        if (!invalidData.has(scope)) {
          invalidData.set(scope, createArray(data.length, false));
        }
        invalidData.get(scope)[datumIndex] = true;
      }
    };
    let keys = true;
    for (const next of opts.props) {
      if (next.type === "key" && !keys) {
        throw new Error("AG Charts - internal config error: keys must come before values.");
      }
      if (next.type === "value" && keys) {
        keys = false;
      }
    }
    const verifyMatchGroupId = ({ matchGroupIds = [] }) => {
      for (const matchGroupId of matchGroupIds) {
        if (this.values.every((def) => def.groupId !== matchGroupId)) {
          throw new Error(
            `AG Charts - internal config error: matchGroupIds properties must match defined groups (${matchGroupId}).`
          );
        }
      }
    };
    const keyScopes = /* @__PURE__ */ new Set();
    const valueScopes = /* @__PURE__ */ new Set();
    for (const def of opts.props) {
      const scopes = def.type === "key" ? keyScopes : valueScopes;
      if (isScoped(def)) {
        def.scopes?.forEach((s) => scopes.add(s));
      }
      switch (def.type) {
        case "key":
          this.keys.push({ ...def, index: this.keys.length, missing: /* @__PURE__ */ new Map() });
          break;
        case "value":
          if (def.property == null) {
            throw new Error(
              `AG Charts - internal config error: no properties specified for value definitions: ${JSON.stringify(
                def
              )}`
            );
          }
          this.values.push({ ...def, index: this.values.length, missing: /* @__PURE__ */ new Map() });
          break;
        case "aggregate":
          verifyMatchGroupId(def);
          this.aggregates.push({ ...def, index: this.aggregates.length });
          break;
        case "group-value-processor":
          verifyMatchGroupId(def);
          this.groupProcessors.push({ ...def, index: this.groupProcessors.length });
          break;
        case "property-value-processor":
          this.propertyProcessors.push({ ...def, index: this.propertyProcessors.length });
          break;
        case "reducer":
          this.reducers.push({ ...def, index: this.reducers.length });
          break;
        case "processor":
          this.processors.push({ ...def, index: this.processors.length });
          break;
      }
    }
    if (!!this.opts.groupByKeys || this.opts.groupByFn != null) {
      const ungroupedScopes = new Set(valueScopes.values());
      keyScopes.forEach((s) => ungroupedScopes.delete(s));
      if (ungroupedScopes.size > 0) {
        throw new Error(
          `AG Charts - scopes missing key for grouping, illegal configuration: ${[...ungroupedScopes.values()]}`
        );
      }
    }
  }
  resolveProcessedDataDefById(scope, searchId) {
    const def = this.scopeCache.get(scope.id)?.get(searchId);
    if (!def) {
      throw new Error(`AG Charts - didn't find property definition for [${searchId}, ${scope.id}]`);
    }
    return { index: def.index, def };
  }
  resolveProcessedDataIndexById(scope, searchId) {
    return this.resolveProcessedDataDefById(scope, searchId).index;
  }
  resolveKeysById(scope, searchId, processedData) {
    const index = this.resolveProcessedDataIndexById(scope, searchId);
    const keys = processedData.keys[index];
    if (keys == null) {
      throw new Error(`AG Charts - didn't find keys for [${searchId}, ${scope.id}]`);
    }
    return keys.get(scope.id);
  }
  hasColumnById(scope, searchId) {
    return this.scopeCache.get(scope.id)?.get(searchId) != null;
  }
  resolveColumnById(scope, searchId, processedData) {
    const index = this.resolveProcessedDataIndexById(scope, searchId);
    const column = processedData.columns?.[index];
    if (column == null) {
      throw new Error(`AG Charts - didn't find column for [${searchId}, ${scope.id}]`);
    }
    return column;
  }
  /**
   * Provides a convenience iterator to iterate over all of the extract datum values in a
   * specific DataGroup.
   *
   * @param scope to which datums should belong
   * @param group containing the datums
   * @param processedData containing the group
   */
  *forEachDatum(scope, processedData, group) {
    const columnIndex = processedData.columnScopes.findIndex((s) => s.has(scope.id));
    for (const datumIndex of group.datumIndices[columnIndex] ?? []) {
      yield processedData.columns[columnIndex][datumIndex];
    }
  }
  /**
   * Provides a convenience iterator to iterate over all of the extracted datum values in a
   * GroupedData.
   *
   * @param scope to which datums should belong
   * @param processedData to iterate through
   */
  *forEachGroupDatum(scope, processedData) {
    const columnIndex = processedData.columnScopes.findIndex((s) => s.has(scope.id));
    const output = {
      groupIndex: 0,
      columnIndex
    };
    const empty = [];
    for (const group of processedData.groups) {
      output.group = group;
      let valueIndex = 0;
      for (const datumIndex of group.datumIndices[columnIndex] ?? empty) {
        output.datumIndex = datumIndex;
        output.valueIndex = valueIndex++;
        yield output;
      }
      output.groupIndex++;
    }
  }
  /**
   * Provides a window-based convenience iterator to iterate over all of the extracted datum
   * values in a GroupedData, including the previous and next entries relative to each datum.
   *
   * @param scope to which datums should belong
   * @param processedData to iterate through
   */
  *forEachGroupDatumTuple(scope, processedData) {
    const columnIndex = processedData.columnScopes.findIndex((s) => s.has(scope.id));
    const output = { columnIndex, datumIndexes: [void 0, void 0, void 0] };
    for (const next of this.forEachGroupDatum(scope, processedData)) {
      output.group = output.nextGroup;
      output.groupIndex = output.nextGroupIndex;
      output.nextGroup = next.group;
      output.nextGroupIndex = next.groupIndex;
      output.datumIndexes[0] = output.datumIndexes[1];
      output.datumIndexes[1] = output.datumIndexes[2];
      output.datumIndexes[2] = next.datumIndex;
      if (output.group != null && output.datumIndexes[1] != null) {
        yield output;
      }
    }
    output.group = output.nextGroup;
    output.groupIndex = output.nextGroupIndex;
    output.nextGroup = void 0;
    output.nextGroupIndex = void 0;
    output.datumIndexes[0] = output.datumIndexes[1];
    output.datumIndexes[1] = output.datumIndexes[2];
    output.datumIndexes[2] = void 0;
    if (output.group != null && output.datumIndexes[1] != null) {
      yield output;
    }
  }
  getDomain(scope, searchId, type, processedData) {
    const domains = this.getDomainsByType(type ?? "value", processedData);
    return domains?.[this.resolveProcessedDataIndexById(scope, searchId)] ?? [];
  }
  getDomainBetweenRange(scope, searchIds, [i0, i1], processedData) {
    const columnIndices = searchIds.map((searchId) => this.resolveProcessedDataIndexById(scope, searchId));
    const cacheKey = columnIndices.join(":");
    const domainRanges = processedData[DOMAIN_RANGES];
    let rangeLookup = domainRanges.get(cacheKey);
    if (rangeLookup == null) {
      const values = columnIndices.map((columnIndex) => processedData.columns[columnIndex]);
      rangeLookup = new RangeLookup(values);
      domainRanges.set(cacheKey, rangeLookup);
    }
    return rangeLookup.rangeBetween(i0, i1);
  }
  getDomainsByType(type, processedData) {
    switch (type) {
      case "key":
        return processedData.domain.keys;
      case "value":
        return processedData.domain.values;
      case "aggregate":
        return processedData.domain.aggValues;
      case "group-value-processor":
        return processedData.domain.groups;
      default:
        return null;
    }
  }
  processData(sources) {
    const {
      opts: { groupByKeys, groupByFn },
      aggregates,
      groupProcessors,
      reducers,
      processors,
      propertyProcessors
    } = this;
    const start2 = performance.now();
    if (groupByKeys && this.keys.length === 0) {
      return;
    }
    let processedData = this.extractData(sources);
    if (groupByKeys) {
      processedData = this.groupData(processedData);
    } else if (groupByFn) {
      processedData = this.groupData(processedData, groupByFn(processedData));
    }
    if (groupProcessors.length > 0 && processedData.type === "grouped") {
      this.postProcessGroups(processedData);
    }
    if (aggregates.length > 0 && processedData.type === "ungrouped") {
      this.aggregateUngroupedData(processedData);
    } else if (aggregates.length > 0 && processedData.type === "grouped") {
      this.aggregateGroupedData(processedData);
    }
    if (propertyProcessors.length > 0) {
      this.postProcessProperties(processedData);
    }
    if (reducers.length > 0) {
      this.reduceData(processedData);
    }
    if (processors.length > 0) {
      this.postProcessData(processedData);
    }
    this.warnDataMissingProperties(sources);
    const end2 = performance.now();
    processedData.time = end2 - start2;
    if (this.debug.check()) {
      logProcessedData(processedData);
    }
    this.processScopeCache();
    return processedData;
  }
  warnDataMissingProperties(sources) {
    if (sources.size === 0)
      return;
    for (const def of iterate(this.keys, this.values)) {
      for (const [scope, missCount] of def.missing) {
        if (missCount < (sources.get(scope)?.length ?? Infinity))
          continue;
        const scopeHint = scope == null ? "" : ` for ${scope}`;
        logger_exports.warnOnce(`the key '${def.property}' was not found in any data element${scopeHint}.`);
      }
    }
  }
  processScopeCache() {
    this.scopeCache.clear();
    for (const def of iterate(this.keys, this.values, this.aggregates)) {
      if (!def.idsMap)
        continue;
      for (const [scope, ids] of def.idsMap) {
        for (const id of ids) {
          if (!this.scopeCache.has(scope)) {
            this.scopeCache.set(scope, /* @__PURE__ */ new Map([[id, def]]));
          } else if (this.scopeCache.get(scope)?.has(id)) {
            throw new Error("duplicate definition ids on the same scope are not allowed.");
          } else {
            this.scopeCache.get(scope).set(id, def);
          }
        }
      }
    }
  }
  valueGroupIdxLookup({ matchGroupIds }) {
    const result = [];
    for (const [index, def] of this.values.entries()) {
      if (!matchGroupIds || def.groupId && matchGroupIds.includes(def.groupId)) {
        result.push(index);
      }
    }
    return result;
  }
  valueIdxLookup(scopes, prop) {
    const noScopesToMatch = scopes == null || scopes.length === 0;
    const propId = typeof prop === "string" ? prop : prop.id;
    const hasMatchingScopeId = (def) => {
      if (def.idsMap) {
        for (const [scope, ids] of def.idsMap) {
          if (scopes?.includes(scope) && ids.has(propId)) {
            return true;
          }
        }
      }
      return false;
    };
    const result = this.values.reduce((res, def, index) => {
      const validDefScopes = def.scopes == null || noScopesToMatch && !def.scopes.length || def.scopes.some((s) => scopes?.includes(s));
      if (validDefScopes && (def.property === propId || def.id === propId || hasMatchingScopeId(def))) {
        res.push(index);
      }
      return res;
    }, []);
    if (result.length === 0) {
      throw new Error(
        `AG Charts - configuration error, unknown property ${JSON.stringify(prop)} in scope(s) ${JSON.stringify(
          scopes
        )}`
      );
    }
    return result;
  }
  extractData(sources) {
    const { dataDomain, processValue, allScopesHaveSameDefs } = this.initDataDomainProcessor();
    const { keys: keyDefs, values: valueDefs } = this;
    const { invalidData, invalidKeys, allKeyMappings } = this.extractKeys(keyDefs, sources, processValue);
    const { columns, columnScopes, partialValidDataCount, maxDataLength } = this.extractValues(
      invalidData,
      valueDefs,
      sources,
      invalidKeys,
      processValue
    );
    const propertyDomain = (def) => {
      const defDomain = dataDomain.get(def);
      const result = defDomain.getDomain();
      if (ContinuousDomain.is(defDomain) && result[0] > result[1]) {
        return [];
      }
      return result;
    };
    return {
      type: "ungrouped",
      input: { count: maxDataLength },
      scopes: new Set(sources.keys()),
      dataSources: sources,
      aggregation: void 0,
      keys: [...allKeyMappings.values()],
      columns,
      columnScopes,
      invalidKeys,
      invalidData,
      domain: {
        keys: keyDefs.map(propertyDomain),
        values: valueDefs.map(propertyDomain)
      },
      defs: {
        allScopesHaveSameDefs,
        keys: keyDefs,
        values: valueDefs
      },
      partialValidDataCount,
      time: 0,
      [DOMAIN_RANGES]: /* @__PURE__ */ new Map()
    };
  }
  extractKeys(keyDefs, sources, processValue) {
    const invalidKeys = /* @__PURE__ */ new Map();
    const invalidData = /* @__PURE__ */ new Map();
    const allKeys = /* @__PURE__ */ new Map();
    let keyDefKeys;
    let scopeDataProcessed;
    const cloneScope = (source, target) => {
      const sourceScope = scopeDataProcessed.get(source);
      keyDefKeys.set(target, keyDefKeys.get(sourceScope));
      if (invalidKeys.has(sourceScope)) {
        invalidKeys.set(target, invalidKeys.get(sourceScope));
        invalidData.set(target, invalidData.get(sourceScope));
      }
    };
    for (const keyDef of keyDefs) {
      const { invalidValue, scopes: keyScopes } = keyDef;
      keyDefKeys = /* @__PURE__ */ new Map();
      scopeDataProcessed = /* @__PURE__ */ new Map();
      allKeys.set(keyDef, keyDefKeys);
      for (const scope of keyScopes ?? []) {
        const data = sources.get(scope) ?? [];
        if (scopeDataProcessed.has(data)) {
          cloneScope(data, scope);
          continue;
        }
        const keys = [];
        keyDefKeys.set(scope, keys);
        scopeDataProcessed.set(data, scope);
        let invalidScopeKeys;
        let invalidScopeData;
        for (let datumIndex = 0; datumIndex < data.length; datumIndex++) {
          const key = processValue(keyDef, data[datumIndex], datumIndex, scope);
          if (key !== INVALID_VALUE) {
            keys.push(key);
            continue;
          }
          keys.push(invalidValue);
          invalidScopeKeys ?? (invalidScopeKeys = createArray(data.length, false));
          invalidScopeData ?? (invalidScopeData = createArray(data.length, false));
          invalidScopeKeys[datumIndex] = true;
          invalidScopeData[datumIndex] = true;
        }
        if (invalidScopeKeys && invalidScopeData) {
          invalidKeys.set(scope, invalidScopeKeys);
          invalidData.set(scope, invalidScopeData);
        }
      }
    }
    return { invalidData, invalidKeys, allKeyMappings: allKeys };
  }
  extractValues(invalidData, valueDefs, sources, invalidKeys, processValue) {
    let partialValidDataCount = 0;
    const columns = [];
    const allColumnScopes = [];
    let maxDataLength = 0;
    for (const def of valueDefs) {
      const { invalidValue } = def;
      const valueSources = new Set(def.scopes.map((s) => sources.get(s)));
      if (valueSources.size > 1) {
        throw new Error(`AG Charts - more than one data source for: ${JSON.stringify(def)}`);
      }
      const columnScopes = new Set(def.scopes);
      const columnScope = first(def.scopes);
      const columnSource = sources.get(columnScope);
      const column = columnSource.map((valueDatum, datumIndex) => {
        const invalidKey = invalidKeys.get(columnScope)?.[datumIndex];
        let value = processValue(def, valueDatum, datumIndex, def.scopes);
        if (invalidKey || value === INVALID_VALUE) {
          this.markScopeDatumInvalid(def.scopes, columnSource, datumIndex, invalidData);
        }
        if (invalidKey) {
          value = invalidValue;
        } else if (value === INVALID_VALUE) {
          partialValidDataCount += 1;
          value = invalidValue;
        }
        return value;
      });
      columns.push(column);
      allColumnScopes.push(columnScopes);
      maxDataLength = Math.max(maxDataLength, column.length);
    }
    return { columns, columnScopes: allColumnScopes, partialValidDataCount, maxDataLength };
  }
  groupData(data, groupingFn) {
    var _a;
    const groups = /* @__PURE__ */ new Map();
    const { keys: dataKeys, columns: allColumns, columnScopes, invalidKeys, invalidData } = data;
    const allScopes = data.scopes;
    const processedColumnIndexes = /* @__PURE__ */ new Set();
    for (const scope of allScopes) {
      const scopeColumnIndexes = allColumns.map((_, idx) => idx).filter((idx) => !processedColumnIndexes.has(idx) && columnScopes[idx].has(scope));
      if (scopeColumnIndexes.length === 0)
        continue;
      for (const idx of scopeColumnIndexes) {
        processedColumnIndexes.add(idx);
      }
      const siblingScopes = /* @__PURE__ */ new Set();
      for (const columnIdx of scopeColumnIndexes) {
        for (const columnScope of columnScopes[columnIdx]) {
          siblingScopes.add(columnScope);
        }
      }
      const scopeKeys = dataKeys.map((k) => k.get(scope)).filter((k) => k != null);
      const firstColumn = allColumns[first(scopeColumnIndexes)];
      const scopeInvalidData = invalidData?.get(scope);
      const scopeInvalidKeys = invalidKeys?.get(scope);
      for (let datumIndex = 0; datumIndex < firstColumn.length; datumIndex++) {
        if (scopeInvalidKeys?.[datumIndex] === true)
          continue;
        const keys = scopeKeys.map((k) => k[datumIndex]);
        if (keys == null || keys.length === 0) {
          throw new Error("AG Charts - no keys found for scope: " + scope);
        }
        const group = groupingFn?.(keys) ?? keys;
        const groupStr = toKeyString(group);
        const outputGroup = groups.get(groupStr) ?? {
          keys: group,
          datumIndices: [],
          validScopes: allScopes
        };
        if (!groups.has(groupStr)) {
          groups.set(groupStr, outputGroup);
        }
        if (scopeInvalidData?.[datumIndex] === true) {
          if (outputGroup.validScopes === allScopes) {
            outputGroup.validScopes = new Set(allScopes.values());
          }
          for (const invalidScope of siblingScopes) {
            outputGroup.validScopes.delete(invalidScope);
          }
        }
        if (outputGroup.validScopes.size === 0)
          continue;
        for (const columnIdx of scopeColumnIndexes) {
          (_a = outputGroup.datumIndices)[columnIdx] ?? (_a[columnIdx] = []);
          outputGroup.datumIndices[columnIdx].push(datumIndex);
        }
      }
    }
    const resultGroups = [];
    const resultData = [];
    for (const { keys, datumIndices, validScopes } of groups.values()) {
      if (validScopes?.size === 0)
        continue;
      resultGroups.push(keys);
      resultData.push({
        datumIndices,
        keys,
        aggregation: [],
        validScopes
      });
    }
    return {
      ...data,
      type: "grouped",
      domain: {
        ...data.domain,
        groups: resultGroups
      },
      groups: resultData
    };
  }
  aggregateUngroupedData(processedData) {
    const domainAggValues = this.aggregates.map(() => [Infinity, -Infinity]);
    processedData.domain.aggValues = domainAggValues;
    const { keys, columns, dataSources } = processedData;
    const onlyScope = first(dataSources.keys());
    const rawData = dataSources.get(onlyScope);
    processedData.aggregation = rawData?.map((_, datumIndex) => {
      const aggregation = [];
      for (const [index, def] of this.aggregates.entries()) {
        const indices = this.valueGroupIdxLookup(def);
        let groupAggValues = def.groupAggregateFunction?.() ?? [Infinity, -Infinity];
        const valuesToAgg = indices.map((columnIndex) => columns[columnIndex][datumIndex]);
        const k = datumKeys(keys, onlyScope, datumIndex);
        const valuesAgg = k != null ? def.aggregateFunction(valuesToAgg, k) : void 0;
        if (valuesAgg) {
          groupAggValues = def.groupAggregateFunction?.(valuesAgg, groupAggValues) ?? ContinuousDomain.extendDomain(valuesAgg, groupAggValues);
        }
        const finalValues = def.finalFunction?.(groupAggValues) ?? groupAggValues;
        if (def.round === true) {
          for (const idx in finalValues) {
            finalValues[idx] = round2(finalValues[idx]);
          }
        }
        aggregation[index] = finalValues;
        ContinuousDomain.extendDomain(finalValues, domainAggValues[index]);
      }
      return aggregation;
    });
  }
  aggregateGroupedData(processedData) {
    const domainAggValues = this.aggregates.map(() => [Infinity, -Infinity]);
    processedData.domain.aggValues = domainAggValues;
    const { columns } = processedData;
    for (const [index, def] of this.aggregates.entries()) {
      const indices = this.valueGroupIdxLookup(def);
      for (const group of processedData.groups) {
        group.aggregation ?? (group.aggregation = []);
        if (group.validScopes?.size === 0)
          continue;
        const groupKeys = group.keys;
        let groupAggValues = def.groupAggregateFunction?.() ?? [Infinity, -Infinity];
        const maxDatumIndex = Math.max(
          ...indices.map((columnIndex) => group.datumIndices[columnIndex]?.length ?? 0)
        );
        for (let datumIndex = 0; datumIndex < maxDatumIndex; datumIndex++) {
          const valuesToAgg = indices.map(
            (columnIndex) => columns[columnIndex][group.datumIndices[columnIndex]?.[datumIndex]]
          );
          const valuesAgg = def.aggregateFunction(valuesToAgg, groupKeys);
          if (valuesAgg) {
            groupAggValues = def.groupAggregateFunction?.(valuesAgg, groupAggValues) ?? ContinuousDomain.extendDomain(valuesAgg, groupAggValues);
          }
        }
        const finalValues = def.finalFunction?.(groupAggValues) ?? groupAggValues;
        if (def.round === true) {
          for (const idx in finalValues) {
            finalValues[idx] = round2(finalValues[idx]);
          }
        }
        group.aggregation[index] = finalValues;
        ContinuousDomain.extendDomain(finalValues, domainAggValues[index]);
      }
    }
  }
  postProcessGroups(processedData) {
    const { groupProcessors } = this;
    const { columnScopes, columns, invalidData, scopes } = processedData;
    for (const processor of groupProcessors) {
      const valueIndexes = this.valueGroupIdxLookup(processor);
      const adjustFn = processor.adjust()();
      for (const dataGroup of processedData.groups) {
        if (dataGroup.validScopes !== scopes)
          continue;
        adjustFn(columns, valueIndexes, dataGroup);
      }
      for (const valueIndex of valueIndexes) {
        const valueDef = this.values[valueIndex];
        const isDiscrete = valueDef.valueType === "category";
        const column = columns[valueIndex];
        const columnScope = first(columnScopes[valueIndex]);
        const invalidDatums = invalidData?.get(columnScope);
        const domain = isDiscrete ? new DiscreteDomain() : new ContinuousDomain();
        for (let datumIndex = 0; datumIndex < column.length; datumIndex += 1) {
          if (invalidDatums?.[datumIndex] === true)
            continue;
          domain.extend(column[datumIndex]);
        }
        processedData.domain.values[valueIndex] = domain.getDomain();
      }
    }
  }
  postProcessProperties(processedData) {
    for (const { adjust, property, scopes } of this.propertyProcessors) {
      for (const idx of this.valueIdxLookup(scopes, property)) {
        adjust()(processedData, idx);
      }
    }
  }
  reduceData(processedData) {
    processedData.reduced ?? (processedData.reduced = {});
    const { dataSources, keys } = processedData;
    for (const def of this.reducers) {
      const reducer = def.reducer();
      let accValue = def.initialValue;
      if (processedData.type === "grouped") {
        for (const group of processedData.groups) {
          accValue = reducer(accValue, group.keys);
        }
      } else {
        const onlyScope = first(dataSources.keys());
        const keyColumns = keys.map((k) => k.get(onlyScope)).filter((k) => k != null);
        const keysParam = keyColumns.map(() => void 0);
        const rawData = dataSources.get(onlyScope);
        for (let datumIndex = 0; datumIndex < rawData.length; datumIndex += 1) {
          for (let keyIdx = 0; keyIdx < keysParam.length; keyIdx++) {
            keysParam[keyIdx] = keyColumns[keyIdx]?.[datumIndex];
          }
          accValue = reducer(accValue, keysParam);
        }
      }
      processedData.reduced[def.property] = accValue;
    }
  }
  postProcessData(processedData) {
    processedData.reduced ?? (processedData.reduced = {});
    for (const def of this.processors) {
      processedData.reduced[def.property] = def.calculate(
        processedData,
        processedData.reduced[def.property]
      );
    }
  }
  initDataDomainProcessor() {
    const { keys: keyDefs, values: valueDefs } = this;
    const scopes = /* @__PURE__ */ new Set();
    for (const valueDef of valueDefs) {
      if (!valueDef.scopes)
        continue;
      for (const scope of valueDef.scopes) {
        scopes.add(scope);
      }
    }
    const dataDomain = /* @__PURE__ */ new Map();
    const processorFns = /* @__PURE__ */ new Map();
    let allScopesHaveSameDefs = true;
    const initDataDomain = () => {
      for (const def of iterate(keyDefs, valueDefs)) {
        if (def.valueType === "category") {
          dataDomain.set(def, new DiscreteDomain());
        } else {
          dataDomain.set(def, new ContinuousDomain());
          allScopesHaveSameDefs && (allScopesHaveSameDefs = (def.scopes ?? []).length === scopes.size);
        }
      }
    };
    initDataDomain();
    const accessors = this.buildAccessors(iterate(keyDefs, valueDefs));
    const processValue = (def, datum, idx, valueScopes) => {
      let valueInDatum;
      let value;
      if (accessors.has(def.property)) {
        try {
          value = accessors.get(def.property)(datum);
        } catch {
        }
        valueInDatum = value != null;
      } else {
        valueInDatum = def.property in datum;
        value = valueInDatum ? datum[def.property] : def.missingValue;
      }
      if (def.forceValue != null) {
        const valueNegative = valueInDatum && isNegative(value);
        value = valueNegative ? -1 * def.forceValue : def.forceValue;
        valueInDatum = true;
      }
      const missingValueDef = "missingValue" in def;
      if (!valueInDatum && !missingValueDef) {
        if (typeof valueScopes === "string") {
          const missCount = def.missing.get(valueScopes) ?? 0;
          def.missing.set(valueScopes, missCount + 1);
        } else {
          for (const scope of valueScopes) {
            const missCount = def.missing.get(scope) ?? 0;
            def.missing.set(scope, missCount + 1);
          }
        }
      }
      if (!dataDomain.has(def)) {
        initDataDomain();
      }
      if (valueInDatum && def.validation?.(value, datum, idx) === false) {
        if ("invalidValue" in def) {
          value = def.invalidValue;
        } else {
          if (this.mode !== "integrated") {
            logger_exports.warnOnce(
              `invalid value of type [${typeof value}] for [${def.scopes} / ${def.id}] ignored:`,
              `[${value}]`
            );
          }
          return INVALID_VALUE;
        }
      }
      if (def.processor) {
        if (!processorFns.has(def)) {
          processorFns.set(def, def.processor());
        }
        value = processorFns.get(def)?.(value);
      }
      dataDomain.get(def)?.extend(value);
      return value;
    };
    return { dataDomain, processValue, initDataDomain, scopes, allScopesHaveSameDefs };
  }
  buildAccessors(defs) {
    const result = /* @__PURE__ */ new Map();
    if (this.suppressFieldDotNotation) {
      return result;
    }
    for (const def of defs) {
      const isPath = def.property.includes(".") || def.property.includes("[");
      if (!isPath)
        continue;
      const components = getPathComponents(def.property);
      if (components == null) {
        logger_exports.warnOnce("Invalid property path [%s]", def.property);
        continue;
      }
      const accessor = createPathAccessor(components);
      result.set(def.property, accessor);
    }
    return result;
  }
};
function logProcessedData(processedData) {
  const logValues = (name, data) => {
    if (data.length > 0) {
      logger_exports.log(`DataModel.processData() - ${name}`);
      logger_exports.table(data);
    }
  };
  logger_exports.log("DataModel.processData() - processedData", processedData);
  logValues("Key Domains", processedData.domain.keys);
  logValues("Group Domains", processedData.domain.groups ?? []);
  logValues("Value Domains", processedData.domain.values);
  logValues("Aggregate Domains", processedData.domain.aggValues ?? []);
}

// packages/ag-charts-community/src/chart/data/dataController.ts
var _DataController = class _DataController {
  constructor(mode, suppressFieldDotNotation) {
    this.mode = mode;
    this.suppressFieldDotNotation = suppressFieldDotNotation;
    this.debug = Debug.create(true, "data-model");
    this.requested = [];
    this.status = "setup";
  }
  async request(id, data, opts) {
    if (this.status !== "setup") {
      throw new Error(`AG Charts - data request after data setup phase.`);
    }
    return new Promise((resolve, reject) => {
      this.requested.push({ id, opts, data, resolve, reject });
    });
  }
  execute(cachedData) {
    if (this.status !== "setup") {
      throw new Error(`AG Charts - data request after data setup phase.`);
    }
    this.status = "executed";
    this.debug("DataController.execute() - requested", this.requested);
    const valid = this.validateRequests(this.requested);
    this.debug("DataController.execute() - validated", valid);
    const merged = this.mergeRequested(valid);
    this.debug("DataController.execute() - merged", merged);
    if (this.debug.check()) {
      getWindow().processedData = [];
    }
    const nextCachedData = [];
    for (const { data, ids, opts, resolves, rejects } of merged) {
      const reusableCache = cachedData?.find((cacheItem) => canReuseCachedData(cacheItem, data, ids, opts));
      let dataModel;
      let processedData;
      if (reusableCache == null) {
        try {
          dataModel = new DataModel(opts, this.mode, this.suppressFieldDotNotation);
          const sources = new Map(valid.map((v) => [v.id, v.data]));
          processedData = dataModel.processData(sources);
        } catch (error2) {
          rejects.forEach((cb) => cb(error2));
          continue;
        }
      } else {
        ({ dataModel, processedData } = reusableCache);
      }
      nextCachedData.push({ opts, data, ids, dataModel, processedData });
      if (this.debug.check()) {
        getWindow("processedData").push(processedData);
      }
      if (processedData?.partialValidDataCount === 0) {
        resolves.forEach(
          (resolve) => resolve({
            dataModel,
            processedData
          })
        );
      } else if (processedData) {
        this.splitResult(dataModel, processedData, ids, resolves);
      } else {
        rejects.forEach((cb) => cb(new Error(`AG Charts - no processed data generated`)));
      }
    }
    return nextCachedData;
  }
  validateRequests(requested) {
    const valid = [];
    for (const [index, request] of requested.entries()) {
      if (index > 0 && request.data.length !== requested[0].data.length && request.opts.groupByData === false && request.opts.groupByKeys === false) {
        request.reject(
          new Error("all series[].data arrays must be of the same length and have matching keys.")
        );
      } else {
        valid.push(request);
      }
    }
    return valid;
  }
  mergeRequested(requested) {
    const grouped = [];
    for (const request of requested) {
      const match = grouped.find(_DataController.groupMatch(request));
      if (match) {
        match.push(request);
      } else {
        grouped.push([request]);
      }
    }
    return grouped.map(_DataController.mergeRequests);
  }
  splitResult(dataModel, processedData, scopes, resolves) {
    for (let i = 0; i < scopes.length; i++) {
      const resolve = resolves[i];
      resolve({
        dataModel,
        processedData
      });
    }
  }
  static groupMatch({ data, opts }) {
    function keys(props2) {
      return props2.filter((p) => p.type === "key").map((p) => p.property).join(";");
    }
    const { groupByData, groupByKeys = false, groupByFn, props } = opts;
    const propsKeys = keys(props);
    return ([group]) => (groupByData === false || group.data === data) && (group.opts.groupByKeys ?? false) === groupByKeys && group.opts.groupByFn === groupByFn && keys(group.opts.props) === propsKeys;
  }
  static mergeRequests(requests) {
    const crossScopeMergableTypes = /* @__PURE__ */ new Set(["key", "group-value-processor"]);
    return requests.reduce(
      (result, { id, data, resolve, reject, opts: { props, ...opts } }) => {
        result.ids.push(id);
        result.rejects.push(reject);
        result.resolves.push(resolve);
        result.data ?? (result.data = data);
        result.opts ?? (result.opts = { ...opts, props: [] });
        for (const prop of props) {
          const clone2 = { ...prop, scopes: [id], data };
          _DataController.createIdsMap(id, clone2);
          const match = result.opts.props.find(
            (existing) => existing.type === clone2.type && (crossScopeMergableTypes.has(existing.type) || existing.data === clone2.data) && _DataController.deepEqual(existing, clone2)
          );
          if (!match) {
            result.opts.props.push(clone2);
            continue;
          }
          match.scopes ?? (match.scopes = []);
          match.scopes.push(...clone2.scopes ?? []);
          if ((match.type === "key" || match.type === "value") && clone2.idsMap?.size) {
            _DataController.mergeIdsMap(clone2.idsMap, match.idsMap);
          }
        }
        return result;
      },
      { ids: [], rejects: [], resolves: [], data: null, opts: null }
    );
  }
  static mergeIdsMap(fromMap, toMap) {
    for (const [scope, ids] of fromMap) {
      const toMapValue = toMap.get(scope);
      if (toMapValue == null) {
        toMap.set(scope, new Set(ids));
      } else {
        for (const id of ids) {
          toMapValue.add(id);
        }
      }
    }
  }
  static createIdsMap(scope, prop) {
    if (prop.id == null)
      return;
    prop.idsMap ?? (prop.idsMap = /* @__PURE__ */ new Map());
    if (prop.idsMap.has(scope)) {
      prop.idsMap.get(scope).add(prop.id);
    } else {
      prop.idsMap.set(scope, /* @__PURE__ */ new Set([prop.id]));
    }
  }
  static deepEqual(a, b) {
    if (a === b) {
      return true;
    }
    if (a && b && typeof a == "object" && typeof b == "object") {
      if (a.constructor !== b.constructor) {
        return false;
      }
      let i, length;
      if (Array.isArray(a)) {
        length = a.length;
        if (length !== b.length) {
          return false;
        }
        for (i = length - 1; i >= 0; i--) {
          if (!_DataController.deepEqual(a[i], b[i])) {
            return false;
          }
        }
        return true;
      }
      const keys = Object.keys(a);
      length = keys.length;
      if (length !== Object.keys(b).length) {
        return false;
      }
      for (i = length - 1; i >= 0; i--) {
        const key = keys[i];
        if (!_DataController.skipKeys.has(key) && (!Object.hasOwn(b, key) || !_DataController.deepEqual(a[key], b[key]))) {
          return false;
        }
      }
      return true;
    }
    return false;
  }
};
// optimized version of deep equality for `mergeRequests` which can potentially loop over 1M times
_DataController.skipKeys = /* @__PURE__ */ new Set(["id", "idsMap", "type", "scopes", "data"]);
var DataController = _DataController;

// packages/ag-charts-community/src/chart/factory/axisRegistry.ts
var AxisRegistry = class {
  constructor() {
    this.axesMap = /* @__PURE__ */ new Map();
    this.themeTemplates = /* @__PURE__ */ new Map();
  }
  register(axisType, module) {
    this.axesMap.set(axisType, module.moduleFactory);
    if (module.themeTemplate) {
      this.setThemeTemplate(axisType, module.themeTemplate);
    }
  }
  create(axisType, moduleContext) {
    const axisFactory = this.axesMap.get(axisType);
    if (axisFactory) {
      return axisFactory(moduleContext);
    }
    throw new Error(`AG Charts - unknown axis type: ${axisType}`);
  }
  has(axisType) {
    return this.axesMap.has(axisType);
  }
  keys() {
    return this.axesMap.keys();
  }
  setThemeTemplate(axisType, themeTemplate) {
    this.themeTemplates.set(axisType, themeTemplate);
    return this;
  }
  getThemeTemplate(axisType) {
    return this.themeTemplates.get(axisType);
  }
};
var axisRegistry = new AxisRegistry();

// packages/ag-charts-community/src/chart/factory/expectedEnterpriseModules.ts
var EXPECTED_ENTERPRISE_MODULES = [
  {
    type: "root",
    optionsKey: "animation",
    chartTypes: ["cartesian", "polar", "hierarchy", "topology", "flow-proportion", "standalone", "gauge"]
  },
  { type: "root", optionsKey: "annotations", chartTypes: ["cartesian"] },
  {
    type: "root",
    optionsKey: "background",
    chartTypes: ["cartesian", "polar", "hierarchy", "topology", "flow-proportion", "standalone", "gauge"],
    optionsInnerKey: "image"
  },
  {
    type: "root",
    optionsKey: "foreground",
    chartTypes: ["cartesian", "polar", "hierarchy", "topology", "flow-proportion", "standalone", "gauge"],
    optionsInnerKey: "image"
  },
  {
    type: "root",
    optionsKey: "chartToolbar",
    chartTypes: ["cartesian"]
  },
  {
    type: "root",
    optionsKey: "contextMenu",
    chartTypes: ["cartesian", "polar", "hierarchy", "topology", "flow-proportion", "standalone", "gauge"]
  },
  { type: "root", optionsKey: "statusBar", chartTypes: ["cartesian"], identifier: "status-bar" },
  {
    type: "root",
    optionsKey: "dataSource",
    chartTypes: ["cartesian", "polar", "hierarchy", "topology", "flow-proportion", "standalone", "gauge"]
  },
  { type: "root", optionsKey: "sync", chartTypes: ["cartesian"] },
  { type: "root", optionsKey: "zoom", chartTypes: ["cartesian", "topology"] },
  { type: "root", optionsKey: "ranges", chartTypes: ["cartesian"] },
  {
    type: "legend",
    optionsKey: "gradientLegend",
    chartTypes: ["cartesian", "polar", "hierarchy", "topology", "flow-proportion", "standalone", "gauge"],
    identifier: "gradient"
  },
  { type: "root", optionsKey: "navigator", chartTypes: ["cartesian"] },
  { type: "axis", optionsKey: "axes[]", chartTypes: ["polar"], identifier: "angle-category" },
  { type: "axis", optionsKey: "axes[]", chartTypes: ["polar"], identifier: "angle-number" },
  { type: "axis", optionsKey: "axes[]", chartTypes: ["polar"], identifier: "radius-category" },
  { type: "axis", optionsKey: "axes[]", chartTypes: ["polar"], identifier: "radius-number" },
  { type: "series", optionsKey: "series[]", chartTypes: ["cartesian"], identifier: "bar", community: true },
  { type: "series", optionsKey: "series[]", chartTypes: ["cartesian"], identifier: "line", community: true },
  { type: "axis", optionsKey: "axes[]", chartTypes: ["cartesian"], identifier: "ordinal-time" },
  { type: "axis-option", optionsKey: "crosshair", chartTypes: ["cartesian"] },
  { type: "series", optionsKey: "series[]", chartTypes: ["cartesian"], identifier: "box-plot" },
  { type: "series", optionsKey: "series[]", chartTypes: ["cartesian"], identifier: "candlestick" },
  { type: "series", optionsKey: "series[]", chartTypes: ["cartesian"], identifier: "cone-funnel" },
  { type: "series", optionsKey: "series[]", chartTypes: ["cartesian"], identifier: "funnel" },
  { type: "series", optionsKey: "series[]", chartTypes: ["cartesian"], identifier: "ohlc" },
  { type: "series", optionsKey: "series[]", chartTypes: ["cartesian"], identifier: "heatmap" },
  { type: "series", optionsKey: "series[]", chartTypes: ["cartesian"], identifier: "range-area" },
  { type: "series", optionsKey: "series[]", chartTypes: ["cartesian"], identifier: "range-bar" },
  { type: "series", optionsKey: "series[]", chartTypes: ["cartesian"], identifier: "waterfall" },
  { type: "series", optionsKey: "series[]", chartTypes: ["polar"], identifier: "nightingale" },
  { type: "series", optionsKey: "series[]", chartTypes: ["polar"], identifier: "radar-area" },
  { type: "series", optionsKey: "series[]", chartTypes: ["polar"], identifier: "radar-line" },
  { type: "series", optionsKey: "series[]", chartTypes: ["polar"], identifier: "radial-bar" },
  { type: "series", optionsKey: "series[]", chartTypes: ["polar"], identifier: "radial-column" },
  { type: "series", optionsKey: "series[]", chartTypes: ["hierarchy"], identifier: "sunburst" },
  { type: "series", optionsKey: "series[]", chartTypes: ["hierarchy"], identifier: "treemap" },
  { type: "series", optionsKey: "series[]", chartTypes: ["topology"], identifier: "map-shape" },
  { type: "series", optionsKey: "series[]", chartTypes: ["topology"], identifier: "map-line" },
  { type: "series", optionsKey: "series[]", chartTypes: ["topology"], identifier: "map-marker" },
  { type: "series", optionsKey: "series[]", chartTypes: ["topology"], identifier: "map-shape-background" },
  { type: "series", optionsKey: "series[]", chartTypes: ["topology"], identifier: "map-line-background" },
  { type: "series", optionsKey: "series[]", chartTypes: ["flow-proportion"], identifier: "chord" },
  { type: "series", optionsKey: "series[]", chartTypes: ["flow-proportion"], identifier: "sankey" },
  { type: "series", optionsKey: "series[]", chartTypes: ["standalone"], identifier: "pyramid" },
  { type: "series", optionsKey: "series[]", chartTypes: ["gauge"], identifier: "linear-gauge" },
  { type: "series", optionsKey: "series[]", chartTypes: ["gauge"], identifier: "radial-gauge" },
  { type: "series-option", optionsKey: "errorBar", chartTypes: ["cartesian"], identifier: "error-bars" },
  { type: "context", contextKey: "sharedToolbar", chartTypes: ["cartesian"] }
];
function getEnterpriseSeriesChartTypes(type) {
  return EXPECTED_ENTERPRISE_MODULES.find((s) => s.type === "series" && s.identifier === type)?.chartTypes;
}
function isEnterpriseCartesian(seriesType) {
  const type = getEnterpriseSeriesChartTypes(seriesType)?.find((v) => v === "cartesian");
  return type === "cartesian";
}
function isEnterprisePolar(seriesType) {
  const type = getEnterpriseSeriesChartTypes(seriesType)?.find((v) => v === "polar");
  return type === "polar";
}

// packages/ag-charts-community/src/chart/factory/legendRegistry.ts
var LegendRegistry = class {
  constructor() {
    this.legendMap = /* @__PURE__ */ new Map();
    this.themeTemplates = /* @__PURE__ */ new Map();
  }
  register(legendType, { optionsKey, moduleFactory, themeTemplate }) {
    this.legendMap.set(legendType, { optionsKey, moduleFactory });
    this.themeTemplates.set(optionsKey, themeTemplate);
  }
  create(legendType, moduleContext) {
    const legendFactory = this.legendMap.get(legendType)?.moduleFactory;
    if (legendFactory) {
      return legendFactory(moduleContext);
    }
    throw new Error(`AG Charts - unknown legend type: ${legendType}`);
  }
  getThemeTemplates() {
    return Object.fromEntries(this.themeTemplates);
  }
  getKeys() {
    return Array.from(this.legendMap.entries()).reduce(
      (result, [legendType, record]) => {
        result[legendType] = record.optionsKey;
        return result;
      },
      {}
    );
  }
};
var legendRegistry = new LegendRegistry();

// packages/ag-charts-community/src/chart/factory/chartTypes.ts
var ChartTypes = class extends Map {
  get(seriesType) {
    return super.get(seriesType) ?? "unknown";
  }
  isCartesian(seriesType) {
    return this.get(seriesType) === "cartesian";
  }
  isPolar(seriesType) {
    return this.get(seriesType) === "polar";
  }
  isHierarchy(seriesType) {
    return this.get(seriesType) === "hierarchy";
  }
  isTopology(seriesType) {
    return this.get(seriesType) === "topology";
  }
  isFlowProportion(seriesType) {
    return this.get(seriesType) === "flow-proportion";
  }
  isStandalone(seriesType) {
    return this.get(seriesType) === "standalone";
  }
  isGauge(seriesType) {
    return this.get(seriesType) === "gauge";
  }
  get seriesTypes() {
    return Array.from(this.keys());
  }
  get cartesianTypes() {
    return this.seriesTypes.filter((t) => this.isCartesian(t));
  }
  get polarTypes() {
    return this.seriesTypes.filter((t) => this.isPolar(t));
  }
  get hierarchyTypes() {
    return this.seriesTypes.filter((t) => this.isHierarchy(t));
  }
  get topologyTypes() {
    return this.seriesTypes.filter((t) => this.isTopology(t));
  }
  get flowProportionTypes() {
    return this.seriesTypes.filter((t) => this.isFlowProportion(t));
  }
  get standaloneTypes() {
    return this.seriesTypes.filter((t) => this.isStandalone(t));
  }
  get gaugeTypes() {
    return this.seriesTypes.filter((t) => this.isGauge(t));
  }
};
var ChartDefaults = class extends Map {
  set(chartType, defaults) {
    return super.set(chartType, mergeDefaults(defaults, this.get(chartType)));
  }
};
var chartTypes2 = new ChartTypes();
var publicChartTypes = new ChartTypes();
var chartDefaults = new ChartDefaults();

// packages/ag-charts-community/src/chart/factory/seriesRegistry.ts
var SeriesRegistry = class {
  constructor() {
    this.seriesMap = /* @__PURE__ */ new Map();
    this.themeTemplates = /* @__PURE__ */ new Map();
  }
  register(seriesType, {
    chartTypes: [chartType],
    moduleFactory,
    tooltipDefaults,
    defaultAxes,
    themeTemplate,
    paletteFactory,
    solo,
    stackable,
    groupable,
    stackedByDefault,
    hidden
  }) {
    this.setThemeTemplate(seriesType, themeTemplate);
    this.seriesMap.set(seriesType, {
      moduleFactory,
      tooltipDefaults,
      defaultAxes,
      paletteFactory,
      solo,
      stackable,
      groupable,
      stackedByDefault
    });
    chartTypes2.set(seriesType, chartType);
    if (!hidden) {
      publicChartTypes.set(seriesType, chartType);
    }
  }
  create(seriesType, moduleContext) {
    const seriesFactory = this.seriesMap.get(seriesType)?.moduleFactory;
    if (seriesFactory) {
      return seriesFactory(moduleContext);
    }
    throw new Error(`AG Charts - unknown series type: ${seriesType}`);
  }
  cloneDefaultAxes(seriesType, options) {
    const defaultAxes = this.seriesMap.get(seriesType)?.defaultAxes;
    if (defaultAxes == null)
      return null;
    const axes = typeof defaultAxes === "function" ? defaultAxes(options) : defaultAxes;
    return { axes: deepClone(axes) };
  }
  isDerivedDefaultAxes(seriesType) {
    return typeof this.seriesMap.get(seriesType)?.defaultAxes === "function";
  }
  setThemeTemplate(seriesType, themeTemplate) {
    const currentTemplate = this.themeTemplates.get(seriesType);
    this.themeTemplates.set(seriesType, mergeDefaults(themeTemplate, currentTemplate));
  }
  getThemeTemplate(seriesType) {
    return this.themeTemplates.get(seriesType);
  }
  getPaletteFactory(seriesType) {
    return this.seriesMap.get(seriesType)?.paletteFactory;
  }
  getTooltipDefauls(seriesType) {
    return this.seriesMap.get(seriesType)?.tooltipDefaults;
  }
  isSolo(seriesType) {
    return this.seriesMap.get(seriesType)?.solo ?? false;
  }
  isGroupable(seriesType) {
    return this.seriesMap.get(seriesType)?.groupable ?? false;
  }
  isStackable(seriesType) {
    return this.seriesMap.get(seriesType)?.stackable ?? false;
  }
  isStackedByDefault(seriesType) {
    return this.seriesMap.get(seriesType)?.stackedByDefault ?? false;
  }
};
var seriesRegistry = new SeriesRegistry();

// packages/ag-charts-community/src/chart/interaction/syncManager.ts
var _SyncManager = class _SyncManager extends BaseManager {
  constructor(chart) {
    super();
    this.chart = chart;
  }
  subscribe(groupId = _SyncManager.DEFAULT_GROUP) {
    let syncGroup = this.get(groupId);
    if (!syncGroup) {
      syncGroup = /* @__PURE__ */ new Set();
      _SyncManager.chartsGroups.set(groupId, syncGroup);
    }
    syncGroup.add(this.chart);
    return this;
  }
  unsubscribe(groupId = _SyncManager.DEFAULT_GROUP) {
    this.get(groupId)?.delete(this.chart);
    return this;
  }
  getChart() {
    return this.chart;
  }
  getGroup(groupId = _SyncManager.DEFAULT_GROUP) {
    const syncGroup = this.get(groupId);
    return syncGroup ? Array.from(syncGroup) : [];
  }
  getGroupSiblings(groupId = _SyncManager.DEFAULT_GROUP) {
    return this.getGroup(groupId).filter((chart) => chart !== this.chart);
  }
  get(groupId) {
    return _SyncManager.chartsGroups.get(groupId);
  }
};
_SyncManager.chartsGroups = /* @__PURE__ */ new Map();
_SyncManager.DEFAULT_GROUP = Symbol("sync-group-default");
var SyncManager = _SyncManager;

// packages/ag-charts-community/src/chart/keyboard.ts
var Keyboard = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.enabled = false;
  }
};
__decorateClass([
  Validate(BOOLEAN)
], Keyboard.prototype, "enabled", 2);
__decorateClass([
  Validate(NUMBER)
], Keyboard.prototype, "tabIndex", 2);

// packages/ag-charts-community/src/chart/mapping/prepareAxis.ts
var CartesianAxisPositions = ["top", "right", "bottom", "left"];
function isAxisPosition(position) {
  return typeof position === "string" && CartesianAxisPositions.includes(position);
}
function guessInvalidPositions(axes) {
  const invalidAxes = [];
  const usedPositions = [];
  const guesses = [...CartesianAxisPositions];
  for (const axis of axes) {
    if (axis instanceof CartesianAxis) {
      if (isAxisPosition(axis.position)) {
        usedPositions.push(axis.position);
      } else {
        invalidAxes.push(axis);
      }
    }
  }
  for (const axis of invalidAxes) {
    let nextGuess;
    do {
      nextGuess = guesses.pop();
    } while (nextGuess && usedPositions.includes(nextGuess));
    if (nextGuess == null)
      break;
    axis.position = nextGuess;
  }
}

// packages/ag-charts-community/src/chart/mapping/prepareSeries.ts
var MATCHING_KEYS = ["direction", "xKey", "yKey", "sizeKey", "angleKey", "radiusKey", "normalizedTo"];
function matchSeriesOptions(series, optSeries, oldOptsSeries) {
  const generateKey = (type, i, opts) => {
    const result = [type];
    for (const key of MATCHING_KEYS) {
      if (key in i && i[key] != null)
        result.push(`${key}=${i[key]}`);
    }
    if (opts?.seriesGrouping) {
      result.push(`seriesGrouping.groupId=${opts?.seriesGrouping.groupId}`);
    }
    return result.join(";");
  };
  const seriesMap = /* @__PURE__ */ new Map();
  let idx = 0;
  for (const s of series) {
    const key = generateKey(s.type, s.properties, oldOptsSeries?.[idx]);
    if (!seriesMap.has(key)) {
      seriesMap.set(key, []);
    }
    seriesMap.get(key)?.push([s, idx++]);
  }
  const optsMap = /* @__PURE__ */ new Map();
  idx = 0;
  for (const o of optSeries) {
    const key = generateKey(o.type, o, o);
    if (!optsMap.has(key)) {
      optsMap.set(key, []);
    }
    optsMap.get(key)?.push([o, idx++]);
  }
  const overlap = [...seriesMap.keys()].some((k) => optsMap.has(k));
  if (!overlap) {
    return { status: "no-overlap", oldKeys: seriesMap.keys(), newKeys: optsMap.keys() };
  }
  const changes = [];
  for (const [key, optsTuples] of optsMap.entries()) {
    for (const [opts, targetIdx] of optsTuples) {
      const seriesArray = seriesMap.get(key);
      if (seriesArray == null || seriesArray.length < 1) {
        changes.push({ opts, targetIdx, idx: targetIdx, status: "add" });
        seriesMap.delete(key);
        continue;
      }
      const [outputSeries, currentIdx] = seriesArray.shift();
      const previousOpts = oldOptsSeries?.[currentIdx] ?? {};
      const diff = jsonDiff(previousOpts, opts ?? {});
      const { groupIndex, stackIndex } = diff?.seriesGrouping ?? {};
      if (groupIndex != null || stackIndex != null) {
        changes.push({
          opts,
          series: outputSeries,
          diff,
          targetIdx,
          idx: currentIdx,
          status: "series-grouping"
        });
      } else if (diff) {
        changes.push({
          opts,
          series: outputSeries,
          diff,
          targetIdx,
          idx: currentIdx,
          status: "update"
        });
      } else {
        changes.push({ opts, series: outputSeries, targetIdx, idx: currentIdx, status: "no-op" });
      }
      if (seriesArray.length === 0) {
        seriesMap.delete(key);
      }
    }
  }
  for (const seriesArray of seriesMap.values()) {
    for (const [outputSeries, currentIdx] of seriesArray) {
      changes.push({ series: outputSeries, idx: currentIdx, targetIdx: -1, status: "remove" });
    }
  }
  return { status: "overlap", changes };
}

// packages/ag-charts-community/src/chart/mapping/types.ts
function optionsType(input) {
  const { series } = input;
  if (!series)
    return;
  return series[0]?.type ?? "line";
}
function isAgCartesianChartOptions(input) {
  const specifiedType = optionsType(input);
  if (specifiedType == null) {
    return false;
  }
  return chartTypes2.isCartesian(specifiedType) || isEnterpriseCartesian(specifiedType);
}
function isAgPolarChartOptions(input) {
  const specifiedType = optionsType(input);
  if (specifiedType == null) {
    return false;
  }
  return chartTypes2.isPolar(specifiedType) || isEnterprisePolar(specifiedType);
}

// packages/ag-charts-community/src/chart/modulesManager.ts
var ModulesManager = class extends ModuleMap {
  applyOptions(options) {
    for (const m of this.moduleMap.values()) {
      if (m.module.optionsKey in options && isProperties(m.moduleInstance)) {
        m.moduleInstance.set(options[m.module.optionsKey]);
      }
    }
  }
  *legends() {
    for (const { module, moduleInstance } of this.moduleMap.values()) {
      if (module.type !== "legend")
        continue;
      yield {
        legendType: module.identifier,
        legend: moduleInstance
      };
    }
  }
};

// packages/ag-charts-community/src/chart/overlay/overlay.ts
var DEFAULT_OVERLAY_CLASS = "ag-charts-overlay";
var DEFAULT_OVERLAY_DARK_CLASS = "ag-charts-dark-overlay";
var Overlay = class extends BaseProperties {
  constructor(className, defaultMessageId) {
    super();
    this.className = className;
    this.defaultMessageId = defaultMessageId;
    this.enabled = true;
  }
  getText(localeManager) {
    return localeManager.t(this.text ?? this.defaultMessageId);
  }
  getElement(animationManager, localeManager, rect) {
    this.content?.remove();
    this.focusBox = rect;
    if (this.renderer) {
      const htmlContent = this.renderer();
      if (htmlContent instanceof HTMLElement) {
        this.content = htmlContent;
      } else {
        const tempDiv = createElement("div");
        tempDiv.innerHTML = htmlContent;
        this.content = tempDiv.firstElementChild;
      }
    } else {
      const content = createElement("div", {
        display: "flex",
        alignItems: "center",
        justifyContent: "center",
        boxSizing: "border-box",
        height: "100%",
        margin: "8px",
        fontFamily: "var(--ag-charts-font-family)",
        fontSize: "var(--ag-charts-font-size)",
        fontWeight: "var(--ag-charts-font-weight)"
      });
      content.innerText = this.getText(localeManager);
      this.content = content;
      animationManager?.animate({
        from: 0,
        to: 1,
        id: "overlay",
        phase: "add",
        groupId: "opacity",
        onUpdate(value) {
          content.style.opacity = String(value);
        },
        onStop() {
          content.style.opacity = "1";
        }
      });
    }
    return this.content;
  }
  removeElement(cleanup = () => this.content?.remove(), animationManager) {
    if (!this.content)
      return;
    if (animationManager) {
      const { content } = this;
      animationManager.animate({
        from: 1,
        to: 0,
        phase: "remove",
        id: "overlay",
        groupId: "opacity",
        onUpdate(value) {
          content.style.opacity = String(value);
        },
        onStop() {
          cleanup?.();
        }
      });
    } else {
      cleanup?.();
    }
    this.content = void 0;
    this.focusBox = void 0;
  }
};
__decorateClass([
  Validate(BOOLEAN)
], Overlay.prototype, "enabled", 2);
__decorateClass([
  Validate(STRING, { optional: true })
], Overlay.prototype, "text", 2);
__decorateClass([
  Validate(FUNCTION, { optional: true })
], Overlay.prototype, "renderer", 2);

// packages/ag-charts-community/src/chart/overlay/chartOverlays.ts
var ChartOverlays = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.darkTheme = false;
    this.loading = new Overlay("ag-charts-loading-overlay", "overlayLoadingData");
    this.noData = new Overlay("ag-charts-no-data-overlay", "overlayNoData");
    this.noVisibleSeries = new Overlay("ag-charts-no-visible-series", "overlayNoVisibleSeries");
    this.unsupportedBrowser = new Overlay("ag-charts-unsupported-browser", "overlayUnsupportedBrowser");
  }
  getFocusInfo(localeManager) {
    for (const overlay of [this.loading, this.noData, this.noVisibleSeries, this.unsupportedBrowser]) {
      if (overlay.focusBox !== void 0) {
        return { text: overlay.getText(localeManager), rect: overlay.focusBox };
      }
    }
    return void 0;
  }
  destroy() {
    this.loading.removeElement();
    this.noData.removeElement();
    this.noVisibleSeries.removeElement();
    this.unsupportedBrowser.removeElement();
  }
};
__decorateClass([
  Validate(BOOLEAN)
], ChartOverlays.prototype, "darkTheme", 2);
__decorateClass([
  Validate(OBJECT)
], ChartOverlays.prototype, "loading", 2);
__decorateClass([
  Validate(OBJECT)
], ChartOverlays.prototype, "noData", 2);
__decorateClass([
  Validate(OBJECT)
], ChartOverlays.prototype, "noVisibleSeries", 2);
__decorateClass([
  Validate(OBJECT)
], ChartOverlays.prototype, "unsupportedBrowser", 2);

// packages/ag-charts-community/src/chart/overlay/loadingSpinner.ts
function getLoadingSpinner(text, defaultDuration) {
  const { animationDuration } = PHASE_METADATA["add"];
  const duration = animationDuration * defaultDuration;
  const container = createElement("div", `${DEFAULT_OVERLAY_CLASS}--loading`, {
    display: "flex",
    alignItems: "center",
    justifyContent: "center",
    flexDirection: "column",
    height: "100%",
    boxSizing: "border-box",
    font: "13px Verdana, sans-serif",
    // FONT_SIZE.MEDIUM
    userSelect: "none",
    animation: `ag-charts-loading ${duration}ms linear 50ms both`
  });
  const matrix = createElement("span", {
    width: "45px",
    height: "40px",
    backgroundImage: [
      "linear-gradient(#0000 calc(1 * 100% / 6), #ccc 0 calc(3 * 100% / 6), #0000 0), ",
      "linear-gradient(#0000 calc(2 * 100% / 6), #ccc 0 calc(4 * 100% / 6), #0000 0), ",
      "linear-gradient(#0000 calc(3 * 100% / 6), #ccc 0 calc(5 * 100% / 6), #0000 0)"
    ].join(""),
    backgroundSize: "10px 400%",
    backgroundRepeat: "no-repeat",
    animation: "ag-charts-loading-matrix 1s infinite linear"
  });
  const label = createElement("p", { marginTop: "1em" });
  label.innerText = text;
  const background = createElement("div", `${DEFAULT_OVERLAY_CLASS}__loading-background`, {
    position: "absolute",
    inset: "0",
    opacity: "0.5",
    zIndex: "-1"
  });
  const animationStyles = createElement("style");
  animationStyles.innerText = [
    "@keyframes ag-charts-loading { from { opacity: 0 } to { opacity: 1 } }",
    "@keyframes ag-charts-loading-matrix {",
    "0% { background-position: 0% 0%, 50% 0%, 100% 0%; }",
    "100% { background-position: 0% 100%, 50% 100%, 100% 100%; }",
    "}"
  ].join(" ");
  container.replaceChildren(animationStyles, matrix, label, background);
  return container;
}

// packages/ag-charts-community/src/chart/series/series.ts
var CROSS_FILTER_MARKER_FILL_OPACITY_FACTOR = 0.25;
var CROSS_FILTER_MARKER_STROKE_OPACITY_FACTOR = 0.125;
var SeriesNodeEvent = class {
  constructor(type, event, { datum }, series) {
    this.type = type;
    this.event = event;
    this.datum = datum;
    this.seriesId = series.id;
  }
};
var SeriesGroupingChangedEvent = class {
  constructor(series, seriesGrouping, oldGrouping) {
    this.series = series;
    this.seriesGrouping = seriesGrouping;
    this.oldGrouping = oldGrouping;
    this.type = "groupingChanged";
  }
};
var Series = class extends Observable {
  constructor(seriesOpts) {
    super();
    this.destroyFns = [];
    this.usesPlacedLabels = false;
    this.seriesGrouping = void 0;
    this.NodeEvent = SeriesNodeEvent;
    this.internalId = createId(this);
    // The group node that contains the series rendering in its default (non-highlighted) state.
    this.contentGroup = new TranslatableGroup({
      name: `${this.internalId}-content`,
      zIndex: 1 /* ANY_CONTENT */
    });
    // The group node that contains all highlighted series items. This is a performance optimisation
    // for large-scale data-sets, where the only thing that routinely varies is the currently
    // highlighted node.
    this.highlightGroup = new TranslatableGroup({
      name: `${this.internalId}-highlight`,
      zIndex: 1 /* ANY_CONTENT */
    });
    // Error bars etc.
    this.annotationGroup = new TranslatableGroup({
      name: `${this.internalId}-annotation`
    });
    // Lazily initialised labelGroup for label presentation.
    this.labelGroup = new TranslatableGroup({
      name: `${this.internalId}-series-labels`
    });
    this.axes = {
      ["x" /* X */]: void 0,
      ["y" /* Y */]: void 0
    };
    this.directions = ["x" /* X */, "y" /* Y */];
    // Flag to determine if we should recalculate node data.
    this.nodeDataRefresh = true;
    this.moduleMap = new ModuleMap();
    this.datumCallbackCache = /* @__PURE__ */ new Map();
    this.connectsToYAxis = false;
    this._declarationOrder = -1;
    this.seriesListeners = new Listeners();
    this._pickNodeCache = new LRUCache();
    const {
      moduleCtx,
      pickModes,
      directionKeys = {},
      directionNames = {},
      canHaveAxes = false,
      usesPlacedLabels = false
    } = seriesOpts;
    this.ctx = moduleCtx;
    this.directionKeys = directionKeys;
    this.directionNames = directionNames;
    this.canHaveAxes = canHaveAxes;
    this.usesPlacedLabels = usesPlacedLabels;
    this.highlightGroup = new TranslatableGroup({
      name: `${this.internalId}-highlight`
    });
    this.highlightNode = this.highlightGroup.appendChild(new Group({ name: "highlightNode", zIndex: 0 }));
    this.highlightLabel = this.highlightGroup.appendChild(new Group({ name: "highlightLabel", zIndex: 10 }));
    this.pickModes = pickModes;
  }
  get pickModeAxis() {
    return "main";
  }
  get id() {
    return this.properties?.id ?? this.internalId;
  }
  get type() {
    return this.constructor.type ?? "";
  }
  get focusable() {
    return true;
  }
  get data() {
    return this._data ?? this._chartData;
  }
  set visible(newVisibility) {
    this.properties.visible = newVisibility;
    this.ctx.legendManager.toggleItem({ enabled: newVisibility, seriesId: this.id });
    this.ctx.legendManager.update();
    this.visibleMaybeChanged();
  }
  get visible() {
    return this.ctx.legendManager.getSeriesEnabled(this.id) ?? this.properties.visible;
  }
  get hasData() {
    return this.data != null && this.data.length > 0;
  }
  get tooltipEnabled() {
    return this.properties.tooltip?.enabled ?? false;
  }
  onDataChange() {
    this.nodeDataRefresh = true;
    this._pickNodeCache.clear();
  }
  setOptionsData(input) {
    this._data = input;
    this.onDataChange();
  }
  setChartData(input) {
    this._chartData = input;
    if (this.data === input) {
      this.onDataChange();
    }
  }
  onSeriesGroupingChange(prev, next) {
    const { internalId, type, visible } = this;
    if (prev) {
      this.ctx.seriesStateManager.deregisterSeries(this);
    }
    if (next) {
      this.ctx.seriesStateManager.registerSeries({ internalId, type, visible, seriesGrouping: next });
    }
    this.fireEvent(new SeriesGroupingChangedEvent(this, next, prev));
  }
  getBandScalePadding() {
    return { inner: 1, outer: 0 };
  }
  attachSeries(seriesContentNode, seriesNode, annotationNode) {
    seriesContentNode.appendChild(this.contentGroup);
    seriesNode.appendChild(this.highlightGroup);
    seriesNode.appendChild(this.labelGroup);
    annotationNode?.appendChild(this.annotationGroup);
  }
  detachSeries(seriesContentNode, seriesNode, annotationNode) {
    seriesContentNode?.removeChild(this.contentGroup);
    seriesNode.removeChild(this.highlightGroup);
    seriesNode.removeChild(this.labelGroup);
    annotationNode?.removeChild(this.annotationGroup);
  }
  setSeriesIndex(index) {
    if (index === this._declarationOrder)
      return false;
    this._declarationOrder = index;
    this.contentGroup.zIndex = [1 /* ANY_CONTENT */, index, 0 /* FOREGROUND */];
    this.highlightGroup.zIndex = [1 /* ANY_CONTENT */, index, 1 /* HIGHLIGHT */];
    this.labelGroup.zIndex = [1 /* ANY_CONTENT */, index, 2 /* LABEL */];
    this.annotationGroup.zIndex = index;
    return true;
  }
  renderToOffscreenCanvas() {
    return false;
  }
  addEventListener(type, listener) {
    return super.addEventListener(type, listener);
  }
  addListener(type, listener) {
    return this.seriesListeners.addListener(type, listener);
  }
  dispatch(type, event) {
    this.seriesListeners.dispatch(type, event);
  }
  addChartEventListeners() {
    return;
  }
  destroy() {
    this.destroyFns.forEach((f) => f());
    this.destroyFns = [];
    this.resetDatumCallbackCache();
    this.ctx.seriesStateManager.deregisterSeries(this);
  }
  getDirectionValues(direction, properties) {
    const resolvedDirection = this.resolveKeyDirection(direction);
    const keys = properties?.[resolvedDirection];
    const values = [];
    if (!keys) {
      return values;
    }
    const addValues = (...items) => {
      for (const value of items) {
        if (Array.isArray(value)) {
          addValues(...value);
        } else if (typeof value === "object") {
          addValues(...Object.values(value));
        } else {
          values.push(value);
        }
      }
    };
    addValues(...keys.map((key) => this.properties[key]));
    return values;
  }
  getKeys(direction) {
    return this.getDirectionValues(direction, this.directionKeys);
  }
  getKeyProperties(direction) {
    return this.directionKeys[this.resolveKeyDirection(direction)] ?? [];
  }
  getNames(direction) {
    return this.getDirectionValues(direction, this.directionNames);
  }
  resolveKeyDirection(direction) {
    return direction;
  }
  // The union of the series domain ('community') and series-option domains ('enterprise').
  getDomain(direction) {
    const seriesDomain = this.getSeriesDomain(direction);
    const moduleDomains = this.moduleMap.mapModules((module) => module.getDomain(direction)).flat();
    return moduleDomains.length !== 0 ? seriesDomain.concat(moduleDomains) : seriesDomain;
  }
  getRange(direction, visibleRange) {
    return this.getSeriesRange(direction, visibleRange);
  }
  getVisibleItems(_xVisibleRange, _yVisibleRange, _minVisibleItems) {
    return Infinity;
  }
  getGradientFillOptions({ bounds }, defaultColorRange) {
    const { axes } = this;
    const xAxis = axes["x" /* X */];
    const yAxis = axes["y" /* Y */];
    const xRange = xAxis?.range ?? [0, 1];
    const yRange = yAxis?.range ?? [0, 1];
    const [x1, x2] = findMinMax(xRange);
    const [y1, y2] = findMinMax(yRange);
    const width2 = x2 - x1;
    const height2 = y2 - y1;
    let domain = [0, 1];
    if (bounds === "series") {
      domain = this.getSeriesDomain("y" /* Y */);
    } else if (bounds === "axes") {
      domain = yAxis?.scale.domain ?? [0, 1];
    }
    return {
      bbox: new BBox(x1, y1, width2, height2),
      domain,
      defaultColorRange
    };
  }
  // Indicate that something external changed and we should recalculate nodeData.
  markNodeDataDirty() {
    this.nodeDataRefresh = true;
    this._pickNodeCache.clear();
    this.visibleMaybeChanged();
  }
  visibleMaybeChanged() {
    this.ctx.seriesStateManager.updateSeries(this);
  }
  getOpacity() {
    const defaultOpacity = 1;
    const { dimOpacity = 1, enabled = true } = this.properties.highlightStyle.series;
    if (!enabled || dimOpacity === defaultOpacity) {
      return defaultOpacity;
    }
    switch (this.isItemIdHighlighted()) {
      case 0 /* None */:
      case 1 /* This */:
        return defaultOpacity;
      case 2 /* Other */:
      default:
        return dimOpacity;
    }
  }
  getStrokeWidth(defaultStrokeWidth) {
    const { strokeWidth, enabled = true } = this.properties.highlightStyle.series;
    if (!enabled || strokeWidth === void 0) {
      return defaultStrokeWidth;
    }
    switch (this.isItemIdHighlighted()) {
      case 1 /* This */:
        return strokeWidth;
      case 0 /* None */:
      case 2 /* Other */:
        return defaultStrokeWidth;
    }
  }
  isItemIdHighlighted() {
    const series = this.ctx.highlightManager?.getActiveHighlight()?.series;
    if (series == null) {
      return 0 /* None */;
    }
    if (series !== this) {
      return 2 /* Other */;
    }
    return 1 /* This */;
  }
  getModuleTooltipParams() {
    return this.moduleMap.mapModules((module) => module.getTooltipParams()).reduce((total, current) => Object.assign(total, current), {});
  }
  pickNode(point, intent, exactMatchOnly = false) {
    const { pickModes, pickModeAxis, visible, contentGroup } = this;
    if (!visible || !contentGroup.visible)
      return;
    if (intent === "highlight" && !this.properties.highlight.enabled)
      return;
    if (intent === "highlight-tooltip" && !this.properties.highlight.enabled)
      return;
    let maxDistance = Infinity;
    if (intent === "tooltip" || intent === "highlight-tooltip") {
      const { tooltip } = this.properties;
      maxDistance = typeof tooltip.range === "number" ? tooltip.range : Infinity;
      exactMatchOnly || (exactMatchOnly = tooltip.range === "exact");
    } else if (intent === "event" || intent === "context-menu") {
      const { nodeClickRange } = this.properties;
      maxDistance = typeof nodeClickRange === "number" ? nodeClickRange : Infinity;
      exactMatchOnly || (exactMatchOnly = nodeClickRange === "exact");
    }
    const selectedPickModes = pickModes.filter(
      (m) => !exactMatchOnly || m === 0 /* EXACT_SHAPE_MATCH */
    );
    const { x, y } = point;
    const key = JSON.stringify({ x, y, maxDistance, selectedPickModes });
    if (this._pickNodeCache.has(key)) {
      return this._pickNodeCache.get(key);
    }
    for (const pickMode of selectedPickModes) {
      let match;
      switch (pickMode) {
        case 0 /* EXACT_SHAPE_MATCH */:
          match = this.pickNodeExactShape(point);
          break;
        case 1 /* NEAREST_NODE */:
          match = this.pickNodeClosestDatum(point);
          break;
        case 2 /* AXIS_ALIGNED */:
          match = pickModeAxis != null ? this.pickNodeMainAxisFirst(point, pickModeAxis === "main-category") : void 0;
          break;
      }
      if (match && match.distance <= maxDistance) {
        return this._pickNodeCache.set(key, { pickMode, match: match.datum, distance: match.distance });
      }
    }
    return this._pickNodeCache.set(key, void 0);
  }
  pickNodeExactShape(point) {
    const match = this.contentGroup.pickNode(point.x, point.y);
    if (match && match.datum.missing !== true) {
      return { datum: match.datum, distance: 0 };
    }
    return void 0;
  }
  pickNodeClosestDatum(_point) {
    throw new Error("AG Charts - Series.pickNodeClosestDatum() not implemented");
  }
  pickNodeNearestDistantObject(point, items) {
    const match = nearestSquared(point.x, point.y, items);
    if (match.nearest !== void 0 && match.nearest.datum.missing !== true) {
      return { datum: match.nearest.datum, distance: Math.sqrt(match.distanceSquared) };
    }
    return void 0;
  }
  pickNodeMainAxisFirst(_point, _requireCategoryAxis) {
    throw new Error("AG Charts - Series.pickNodeMainAxisFirst() not implemented");
  }
  getLabelData() {
    return [];
  }
  updatePlacedLabelData(_labels) {
    return;
  }
  fireNodeClickEvent(event, datum) {
    this.fireEvent(new this.NodeEvent("nodeClick", event, datum, this));
  }
  fireNodeDoubleClickEvent(event, datum) {
    this.fireEvent(new this.NodeEvent("nodeDoubleClick", event, datum, this));
  }
  createNodeContextMenuActionEvent(event, datum) {
    return new this.NodeEvent("nodeContextMenuAction", event, datum, this);
  }
  onLegendInitialState(legendType, initialState) {
    const { visible = true, itemId, legendItemName } = initialState ?? {};
    this.toggleSeriesItem(visible, legendType, itemId, legendItemName);
  }
  onLegendItemClick(event) {
    const { enabled, itemId, series, legendType } = event;
    const legendItemName = "legendItemName" in this.properties ? this.properties.legendItemName : void 0;
    const legendItemKey = "legendItemKey" in this.properties ? this.properties.legendItemKey : void 0;
    const matchedLegendItemName = legendItemName != void 0 && legendItemName === event.legendItemName;
    if (series.id === this.id || matchedLegendItemName || legendItemKey != void 0) {
      this.toggleSeriesItem(enabled, legendType, itemId, legendItemName, event);
    }
  }
  onLegendItemDoubleClick(event) {
    const { enabled, itemId, series, numVisibleItems, legendType } = event;
    const legendItemName = "legendItemName" in this.properties ? this.properties.legendItemName : void 0;
    const legendItemKey = "legendItemKey" in this.properties ? this.properties.legendItemKey : void 0;
    const matchedLegendItemName = legendItemName != void 0 && legendItemName === event.legendItemName;
    if (series.id === this.id || matchedLegendItemName || legendItemKey != void 0) {
      this.toggleSeriesItem(true, legendType, itemId, legendItemName, event);
    } else if (enabled && numVisibleItems === 1) {
      this.toggleSeriesItem(true, legendType, void 0, legendItemName);
    } else {
      this.toggleSeriesItem(false, legendType, void 0, legendItemName);
    }
  }
  toggleSeriesItem(enabled, legendType, itemId, legendItemName, legendEvent) {
    const seriesId = this.id;
    if (enabled || legendType !== "category") {
      this.visible = enabled;
    }
    this.nodeDataRefresh = true;
    this._pickNodeCache.clear();
    const event = {
      type: "seriesVisibilityChange",
      seriesId,
      itemId,
      legendItemName: legendEvent?.legendItemName ?? legendItemName,
      visible: enabled
    };
    this.fireEvent(event);
    this.ctx.legendManager.toggleItem({ enabled, seriesId, itemId, legendItemName });
  }
  isEnabled() {
    return this.visible;
  }
  getModuleMap() {
    return this.moduleMap;
  }
  createModuleContext() {
    return { ...this.ctx, series: this };
  }
  getLabelText(label, params, defaultFormatter = formatValue) {
    if (label.formatter) {
      return this.ctx.callbackCache.call(label.formatter, { seriesId: this.id, ...params }) ?? defaultFormatter(params.value);
    }
    return defaultFormatter(params.value);
  }
  getMarkerStyle(marker, params, defaultStyle = marker.getStyle()) {
    const defaultSize = { size: params.datum.point?.size ?? 0 };
    const markerStyle = mergeDefaults(defaultSize, defaultStyle);
    if (marker.itemStyler) {
      const style = this.ctx.callbackCache.call(marker.itemStyler, {
        seriesId: this.id,
        ...markerStyle,
        ...params,
        datum: params.datum.datum
      });
      return mergeDefaults(style, markerStyle);
    }
    return markerStyle;
  }
  updateMarkerStyle(markerNode, marker, params, defaultStyle = marker.getStyle(), { applyTranslation = true, selected = true } = {}) {
    const { point } = params.datum;
    const activeStyle = this.getMarkerStyle(marker, params, defaultStyle);
    const visible = this.visible && activeStyle.size > 0 && point && !isNaN(point.x) && !isNaN(point.y);
    if (applyTranslation) {
      markerNode.setProperties({ visible, ...activeStyle, translationX: point?.x, translationY: point?.y });
    } else {
      markerNode.setProperties({ visible, ...activeStyle });
    }
    if (!selected) {
      markerNode.fillOpacity *= CROSS_FILTER_MARKER_FILL_OPACITY_FACTOR;
      markerNode.strokeOpacity *= CROSS_FILTER_MARKER_STROKE_OPACITY_FACTOR;
    }
    if (typeof marker.shape === "function" && !markerNode.dirtyPath) {
      markerNode.path.clear(true);
      markerNode.updatePath();
      markerNode.checkPathDirty();
      const bb = markerNode.getBBox();
      if (point !== void 0 && bb.isFinite()) {
        const center2 = bb.computeCenter();
        const [dx, dy] = ["x", "y"].map(
          (key) => (activeStyle.strokeWidth ?? 0) + Math.abs(center2[key] - point[key])
        );
        point.focusSize = Math.max(bb.width + dx, bb.height + dy);
      }
    }
  }
  get nodeDataDependencies() {
    return this._nodeDataDependencies ?? { seriesRectWidth: NaN, seriesRectHeight: NaN };
  }
  checkResize(newSeriesRect) {
    const { width: seriesRectWidth, height: seriesRectHeight } = newSeriesRect ?? { width: NaN, height: NaN };
    const newNodeDataDependencies = newSeriesRect ? { seriesRectWidth, seriesRectHeight } : void 0;
    const resize = jsonDiff(this.nodeDataDependencies, newNodeDataDependencies) != null;
    if (resize) {
      this._nodeDataDependencies = newNodeDataDependencies;
      this.markNodeDataDirty();
    }
    return resize;
  }
  pickFocus(_opts) {
    return void 0;
  }
  resetDatumCallbackCache() {
    this.datumCallbackCache.clear();
  }
  cachedDatumCallback(id, fn) {
    const { datumCallbackCache } = this;
    const existing = datumCallbackCache.get(id);
    if (existing != null)
      return existing;
    const value = fn();
    datumCallbackCache.set(id, value);
    return value;
  }
};
__decorateClass([
  ActionOnSet({
    changeValue: function(newVal, oldVal) {
      this.onSeriesGroupingChange(oldVal, newVal);
    }
  })
], Series.prototype, "seriesGrouping", 2);

// packages/ag-charts-community/src/scene/polyRoots.ts
function linearRoot(a, b) {
  const t = -b / a;
  return a !== 0 && t >= 0 && t <= 1 ? [t] : [];
}
function quadraticRoots(a, b, c) {
  if (a === 0) {
    return linearRoot(b, c);
  }
  const D = b * b - 4 * a * c;
  const roots = [];
  if (D === 0) {
    const t = -b / (2 * a);
    if (t >= 0 && t <= 1) {
      roots.push(t);
    }
  } else if (D > 0) {
    const rD = Math.sqrt(D);
    const t1 = (-b - rD) / (2 * a);
    const t2 = (-b + rD) / (2 * a);
    if (t1 >= 0 && t1 <= 1) {
      roots.push(t1);
    }
    if (t2 >= 0 && t2 <= 1) {
      roots.push(t2);
    }
  }
  return roots;
}
function cubicRoots(a, b, c, d) {
  if (a === 0) {
    return quadraticRoots(b, c, d);
  }
  const A = b / a;
  const B = c / a;
  const C2 = d / a;
  const Q = (3 * B - A * A) / 9;
  const R = (9 * A * B - 27 * C2 - 2 * A * A * A) / 54;
  const D = Q * Q * Q + R * R;
  const third = 1 / 3;
  const roots = [];
  if (D >= 0) {
    const rD = Math.sqrt(D);
    const S = Math.sign(R + rD) * Math.pow(Math.abs(R + rD), third);
    const T = Math.sign(R - rD) * Math.pow(Math.abs(R - rD), third);
    const Im = Math.abs(Math.sqrt(3) * (S - T) / 2);
    const t = -third * A + (S + T);
    if (t >= 0 && t <= 1) {
      roots.push(t);
    }
    if (Im === 0) {
      const t2 = -third * A - (S + T) / 2;
      if (t2 >= 0 && t2 <= 1) {
        roots.push(t2);
      }
    }
  } else {
    const theta = Math.acos(R / Math.sqrt(-Q * Q * Q));
    const thirdA = third * A;
    const twoSqrtQ = 2 * Math.sqrt(-Q);
    const t1 = twoSqrtQ * Math.cos(third * theta) - thirdA;
    const t2 = twoSqrtQ * Math.cos(third * (theta + 2 * Math.PI)) - thirdA;
    const t3 = twoSqrtQ * Math.cos(third * (theta + 4 * Math.PI)) - thirdA;
    if (t1 >= 0 && t1 <= 1) {
      roots.push(t1);
    }
    if (t2 >= 0 && t2 <= 1) {
      roots.push(t2);
    }
    if (t3 >= 0 && t3 <= 1) {
      roots.push(t3);
    }
  }
  return roots;
}

// packages/ag-charts-community/src/scene/intersection.ts
function segmentIntersection(ax1, ay1, ax2, ay2, bx1, by1, bx2, by2) {
  const d = (ax2 - ax1) * (by2 - by1) - (ay2 - ay1) * (bx2 - bx1);
  if (d === 0) {
    return 0;
  }
  const ua = ((bx2 - bx1) * (ay1 - by1) - (ax1 - bx1) * (by2 - by1)) / d;
  const ub = ((ax2 - ax1) * (ay1 - by1) - (ay2 - ay1) * (ax1 - bx1)) / d;
  if (ua >= 0 && ua <= 1 && ub >= 0 && ub <= 1) {
    return 1;
  }
  return 0;
}
function cubicSegmentIntersections(px1, py1, px2, py2, px3, py3, px4, py4, x1, y1, x2, y2) {
  let intersections = 0;
  const A = y1 - y2;
  const B = x2 - x1;
  const C2 = x1 * (y2 - y1) - y1 * (x2 - x1);
  const bx = bezierCoefficients(px1, px2, px3, px4);
  const by = bezierCoefficients(py1, py2, py3, py4);
  const a = A * bx[0] + B * by[0];
  const b = A * bx[1] + B * by[1];
  const c = A * bx[2] + B * by[2];
  const d = A * bx[3] + B * by[3] + C2;
  const roots = cubicRoots(a, b, c, d);
  for (const t of roots) {
    const tt = t * t;
    const ttt = t * tt;
    const x = bx[0] * ttt + bx[1] * tt + bx[2] * t + bx[3];
    const y = by[0] * ttt + by[1] * tt + by[2] * t + by[3];
    let s;
    if (x1 === x2) {
      s = (y - y1) / (y2 - y1);
    } else {
      s = (x - x1) / (x2 - x1);
    }
    if (s >= 0 && s <= 1) {
      intersections++;
    }
  }
  return intersections;
}
function bezierCoefficients(P1, P2, P3, P4) {
  return [
    // Bézier expressed as matrix operations:
    -P1 + 3 * P2 - 3 * P3 + P4,
    //                 |-1  3 -3  1| |P1|
    3 * P1 - 6 * P2 + 3 * P3,
    //   [t^3 t^2 t 1] | 3 -6  3  0| |P2|
    -3 * P1 + 3 * P2,
    //                 |-3  3  0  0| |P3|
    P1
    //                 | 1  0  0  0| |P4|
  ];
}
function arcIntersections(cx, cy, r, startAngle, endAngle, counterClockwise, x1, y1, x2, y2) {
  if (isNaN(cx) || isNaN(cy)) {
    return 0;
  }
  if (counterClockwise) {
    [endAngle, startAngle] = [startAngle, endAngle];
  }
  const k = (y2 - y1) / (x2 - x1);
  const y0 = y1 - k * x1;
  const a = Math.pow(k, 2) + 1;
  const b = 2 * (k * (y0 - cy) - cx);
  const c = Math.pow(cx, 2) + Math.pow(y0 - cy, 2) - Math.pow(r, 2);
  const d = Math.pow(b, 2) - 4 * a * c;
  if (d < 0) {
    return 0;
  }
  const i1x = (-b + Math.sqrt(d)) / 2 / a;
  const i2x = (-b - Math.sqrt(d)) / 2 / a;
  let intersections = 0;
  [i1x, i2x].forEach((x) => {
    const isXInsideLine = x >= Math.min(x1, x2) && x <= Math.max(x1, x2);
    if (!isXInsideLine) {
      return;
    }
    const y = k * x + y0;
    const adjacent = x - cx;
    const opposite = y - cy;
    const angle = Math.atan2(opposite, adjacent);
    if (isBetweenAngles(angle, startAngle, endAngle)) {
      intersections++;
    }
  });
  return intersections;
}

// packages/ag-charts-community/src/scene/util/bezier.ts
function evaluateBezier(p0, p1, p2, p3, t) {
  return (1 - t) ** 3 * p0 + 3 * (1 - t) ** 2 * t * p1 + 3 * (1 - t) * t ** 2 * p2 + t ** 3 * p3;
}
function calculateDerivativeExtrema(p0, p1, p2, p3) {
  const a = -p0 + 3 * p1 - 3 * p2 + p3;
  const b = 3 * p0 - 6 * p1 + 3 * p2;
  const c = -3 * p0 + 3 * p1;
  if (a === 0) {
    if (b !== 0) {
      const t = -c / b;
      if (t > 0 && t < 1) {
        return [t];
      }
    }
    return [];
  }
  const discriminant = b * b - 4 * a * c;
  if (discriminant >= 0) {
    const sqrtDiscriminant = Math.sqrt(discriminant);
    const t1 = (-b + sqrtDiscriminant) / (2 * a);
    const t2 = (-b - sqrtDiscriminant) / (2 * a);
    return [t1, t2].filter((t) => t > 0 && t < 1);
  }
  return [];
}
function calculateDerivativeExtremaXY(sx, sy, cp1x, cp1y, cp2x, cp2y, x, y) {
  const tx = calculateDerivativeExtrema(sx, cp1x, cp2x, x);
  const ty = calculateDerivativeExtrema(sy, cp1y, cp2y, y);
  return [...tx, ...ty];
}

// packages/ag-charts-community/src/scene/extendedPath2D.ts
var ExtendedPath2D = class {
  constructor() {
    // The methods of this class will likely be called many times per animation frame,
    // and any allocation can trigger a GC cycle during animation, so we attempt
    // to minimize the number of allocations.
    this.path2d = new Path2D();
    this.previousCommands = [];
    this.previousParams = [];
    this.previousClosedPath = false;
    this.commands = [];
    this.params = [];
    this.openedPath = false;
    this.closedPath = false;
  }
  isEmpty() {
    return this.commands.length === 0;
  }
  isDirty() {
    return this.closedPath !== this.previousClosedPath || this.previousCommands.length !== this.commands.length || this.previousParams.length !== this.params.length || this.previousCommands.toString() !== this.commands.toString() || this.previousParams.toString() !== this.params.toString();
  }
  getPath2D() {
    return this.path2d;
  }
  moveTo(x, y) {
    this.openedPath = true;
    this.path2d.moveTo(x, y);
    this.commands.push(0 /* Move */);
    this.params.push(x, y);
  }
  lineTo(x, y) {
    if (this.openedPath) {
      this.path2d.lineTo(x, y);
      this.commands.push(1 /* Line */);
      this.params.push(x, y);
    } else {
      this.moveTo(x, y);
    }
  }
  rect(x, y, width2, height2) {
    this.moveTo(x, y);
    this.lineTo(x + width2, y);
    this.lineTo(x + width2, y + height2);
    this.lineTo(x, y + height2);
    this.closePath();
  }
  roundRect(x, y, width2, height2, radii) {
    radii = Math.min(radii, width2 / 2, height2 / 2);
    this.moveTo(x, y + radii);
    this.arc(x + radii, y + radii, radii, Math.PI, 1.5 * Math.PI);
    this.lineTo(x + radii, y);
    this.lineTo(x + width2 - radii, y);
    this.arc(x + width2 - radii, y + radii, radii, 1.5 * Math.PI, 2 * Math.PI);
    this.lineTo(x + width2, y + radii);
    this.lineTo(x + width2, y + height2 - radii);
    this.arc(x + width2 - radii, y + height2 - radii, radii, 0, Math.PI / 2);
    this.lineTo(x + width2 - radii, y + height2);
    this.lineTo(x + radii, y + height2);
    this.arc(x + +radii, y + height2 - radii, radii, Math.PI / 2, Math.PI);
    this.lineTo(x, y + height2 - radii);
    this.closePath();
  }
  arc(x, y, r, sAngle, eAngle, counterClockwise) {
    this.openedPath = true;
    this.path2d.arc(x, y, r, sAngle, eAngle, counterClockwise);
    this.commands.push(2 /* Arc */);
    this.params.push(x, y, r, sAngle, eAngle, counterClockwise ? 1 : 0);
  }
  cubicCurveTo(cx1, cy1, cx2, cy2, x, y) {
    if (!this.openedPath) {
      this.moveTo(cx1, cy1);
    }
    this.path2d.bezierCurveTo(cx1, cy1, cx2, cy2, x, y);
    this.commands.push(3 /* Curve */);
    this.params.push(cx1, cy1, cx2, cy2, x, y);
  }
  closePath() {
    if (this.openedPath) {
      this.path2d.closePath();
      this.commands.push(4 /* ClosePath */);
      this.openedPath = false;
      this.closedPath = true;
    }
  }
  clear(trackChanges) {
    if (trackChanges) {
      this.previousCommands = this.commands;
      this.previousParams = this.params;
      this.previousClosedPath = this.closedPath;
    }
    this.path2d = new Path2D();
    this.openedPath = false;
    this.closedPath = false;
    this.commands = [];
    this.params = [];
  }
  isPointInPath(x, y) {
    const commands = this.commands;
    const params = this.params;
    const cn = commands.length;
    const ox = -1e4;
    const oy = -1e4;
    let sx = NaN;
    let sy = NaN;
    let px = 0;
    let py = 0;
    let intersectionCount = 0;
    for (let ci = 0, pi = 0; ci < cn; ci++) {
      switch (commands[ci]) {
        case 0 /* Move */:
          intersectionCount += segmentIntersection(sx, sy, px, py, ox, oy, x, y);
          px = params[pi++];
          sx = px;
          py = params[pi++];
          sy = py;
          break;
        case 1 /* Line */:
          intersectionCount += segmentIntersection(px, py, params[pi++], params[pi++], ox, oy, x, y);
          px = params[pi - 2];
          py = params[pi - 1];
          break;
        case 3 /* Curve */:
          intersectionCount += cubicSegmentIntersections(
            px,
            py,
            params[pi++],
            params[pi++],
            params[pi++],
            params[pi++],
            params[pi++],
            params[pi++],
            ox,
            oy,
            x,
            y
          );
          px = params[pi - 2];
          py = params[pi - 1];
          break;
        case 2 /* Arc */: {
          const cx = params[pi++];
          const cy = params[pi++];
          const r = params[pi++];
          const startAngle = params[pi++];
          const endAngle = params[pi++];
          const counterClockwise = Boolean(params[pi++]);
          intersectionCount += arcIntersections(
            cx,
            cy,
            r,
            startAngle,
            endAngle,
            counterClockwise,
            ox,
            oy,
            x,
            y
          );
          if (!isNaN(sx)) {
            const startX = cx + Math.cos(startAngle) * r;
            const startY = cy + Math.sin(startAngle) * r;
            intersectionCount += segmentIntersection(px, py, startX, startY, ox, oy, x, y);
          }
          px = cx + Math.cos(endAngle) * r;
          py = cy + Math.sin(endAngle) * r;
          break;
        }
        case 4 /* ClosePath */:
          intersectionCount += segmentIntersection(sx, sy, px, py, ox, oy, x, y);
          break;
      }
    }
    return intersectionCount % 2 === 1;
  }
  distanceSquared(x, y) {
    let best = Infinity;
    const commands = this.commands;
    const params = this.params;
    const cn = commands.length;
    let sx = NaN;
    let sy = NaN;
    let px = 0;
    let py = 0;
    for (let ci = 0, pi = 0; ci < cn; ci++) {
      switch (commands[ci]) {
        case 0 /* Move */:
          px = sx = params[pi++];
          py = sy = params[pi++];
          break;
        case 1 /* Line */: {
          const nx = params[pi++];
          const ny = params[pi++];
          best = lineDistanceSquared(x, y, px, py, nx, ny, best);
          break;
        }
        case 3 /* Curve */:
          logger_exports.error("Command.Curve distanceSquare not implemented");
          break;
        case 2 /* Arc */: {
          const cx = params[pi++];
          const cy = params[pi++];
          const r = params[pi++];
          const startAngle = params[pi++];
          const endAngle = params[pi++];
          const startX = cx + Math.cos(startAngle) * r;
          const startY = cy + Math.sin(startAngle) * r;
          const counterClockwise = Boolean(params[pi++]);
          best = lineDistanceSquared(x, y, px, py, startX, startY, best);
          best = arcDistanceSquared(x, y, cx, cy, r, startAngle, endAngle, counterClockwise, best);
          px = cx + Math.cos(endAngle) * r;
          py = cy + Math.sin(endAngle) * r;
          break;
        }
        case 4 /* ClosePath */:
          best = lineDistanceSquared(x, y, px, py, sx, sy, best);
          break;
      }
    }
    return best;
  }
  // https://developer.mozilla.org/en-US/docs/Web/SVG/Attribute/d
  toSVG(transform = (x, y) => ({ x, y })) {
    const buffer = [];
    const { commands, params } = this;
    const addCommand = (command, ...points) => {
      buffer.push(command);
      for (let i = 0; i < points.length; i += 2) {
        const { x, y } = transform(points[i], points[i + 1]);
        buffer.push(x, y);
      }
    };
    let pi = 0;
    for (const command of commands) {
      switch (command) {
        case 0 /* Move */:
          addCommand("M", params[pi++], params[pi++]);
          break;
        case 1 /* Line */:
          addCommand("L", params[pi++], params[pi++]);
          break;
        case 3 /* Curve */:
          addCommand("C", params[pi++], params[pi++], params[pi++], params[pi++], params[pi++], params[pi++]);
          break;
        case 2 /* Arc */: {
          const cx = params[pi++];
          const cy = params[pi++];
          const r = params[pi++];
          const A0 = params[pi++];
          const A1 = params[pi++];
          const ccw = params[pi++];
          let sweep = ccw ? A0 - A1 : A1 - A0;
          if (sweep < 0) {
            sweep += Math.ceil(-sweep / (2 * Math.PI)) * 2 * Math.PI;
          }
          if (ccw) {
            sweep = -sweep;
          }
          const arcSections = Math.max(Math.ceil(Math.abs(sweep) / (Math.PI / 2)), 1);
          const step = sweep / arcSections;
          const h = 4 / 3 * Math.tan(step / 4);
          const move = buffer.length === 0 ? "M" : "L";
          addCommand(move, cx + Math.cos(A0) * r, cy + Math.sin(A0) * r);
          for (let i = 0; i < arcSections; i += 1) {
            const a0 = A0 + step * (i + 0);
            const a1 = A0 + step * (i + 1);
            const rSinStart = r * Math.sin(a0);
            const rCosStart = r * Math.cos(a0);
            const rSinEnd = r * Math.sin(a1);
            const rCosEnd = r * Math.cos(a1);
            addCommand(
              "C",
              cx + rCosStart - h * rSinStart,
              cy + rSinStart + h * rCosStart,
              cx + rCosEnd + h * rSinEnd,
              cy + rSinEnd - h * rCosEnd,
              cx + rCosEnd,
              cy + rSinEnd
            );
          }
          break;
        }
        case 4 /* ClosePath */:
          buffer.push("Z");
          break;
      }
    }
    return buffer.join(" ");
  }
  computeBBox() {
    const { commands, params } = this;
    let [top, left, right, bot] = [Infinity, Infinity, -Infinity, -Infinity];
    let [sx, sy] = [NaN, NaN];
    let [mx, my] = [NaN, NaN];
    const joinPoint = (x, y, updatestart) => {
      top = Math.min(y, top);
      left = Math.min(x, left);
      right = Math.max(x, right);
      bot = Math.max(y, bot);
      if (updatestart) {
        [sx, sy] = [x, y];
      }
    };
    let pi = 0;
    for (const command of commands) {
      switch (command) {
        case 0 /* Move */:
          joinPoint(params[pi++], params[pi++], true);
          [mx, my] = [sx, sy];
          break;
        case 1 /* Line */:
          joinPoint(params[pi++], params[pi++], true);
          break;
        case 3 /* Curve */: {
          const cp1x = params[pi++];
          const cp1y = params[pi++];
          const cp2x = params[pi++];
          const cp2y = params[pi++];
          const x = params[pi++];
          const y = params[pi++];
          joinPoint(x, y, true);
          const Ts = calculateDerivativeExtremaXY(sx, sy, cp1x, cp1y, cp2x, cp2y, x, y);
          Ts.forEach((t) => {
            const px = evaluateBezier(sx, cp1x, cp2x, x, t);
            const py = evaluateBezier(sy, cp1y, cp2y, y, t);
            joinPoint(px, py);
          });
          break;
        }
        case 2 /* Arc */: {
          const cx = params[pi++];
          const cy = params[pi++];
          const r = params[pi++];
          let A0 = normalizeAngle360(params[pi++]);
          let A1 = normalizeAngle360(params[pi++]);
          const ccw = params[pi++];
          if (ccw) {
            [A0, A1] = [A1, A0];
          }
          const joinAngle = (angle, updatestart) => {
            const px = cx + r * Math.cos(angle);
            const py = cy + r * Math.sin(angle);
            joinPoint(px, py, updatestart);
          };
          joinAngle(A0);
          joinAngle(A1, true);
          const criticalAngles = [0, Math.PI / 2, Math.PI, 3 * Math.PI / 2];
          for (const crit of criticalAngles) {
            if (A0 < A1 && A0 <= crit && crit <= A1 || A0 > A1 && (A0 <= crit || crit <= A1)) {
              joinAngle(crit);
            }
          }
          break;
        }
        case 4 /* ClosePath */:
          [sx, sy] = [mx, my];
          break;
      }
    }
    return new BBox(left, top, right - left, bot - top);
  }
};

// packages/ag-charts-community/src/scene/shape/path.ts
function ScenePathChangeDetection(opts) {
  const { changeCb, convertor } = opts ?? {};
  return SceneChangeDetection({ type: "path", convertor, changeCb });
}
var Path = class extends Shape {
  constructor() {
    super(...arguments);
    /**
     * Declare a path to retain for later rendering and hit testing
     * using custom Path2D class. Think of it as a TypeScript version
     * of the native Path2D (with some differences) that works in all browsers.
     */
    this.path = new ExtendedPath2D();
    this._clipX = NaN;
    this._clipY = NaN;
    this.clip = false;
    /**
     * The path only has to be updated when certain attributes change.
     * For example, if transform attributes (such as `translationX`)
     * are changed, we don't have to update the path. The `dirtyPath` flag
     * is how we keep track if the path has to be updated or not.
     */
    this._dirtyPath = true;
    this.lastPixelRatio = NaN;
  }
  set clipX(value) {
    this._clipX = value;
    this.dirtyPath = true;
  }
  set clipY(value) {
    this._clipY = value;
    this.dirtyPath = true;
  }
  set dirtyPath(value) {
    if (this._dirtyPath !== value) {
      this._dirtyPath = value;
      if (value) {
        this.markDirty();
      }
    }
  }
  get dirtyPath() {
    return this._dirtyPath;
  }
  checkPathDirty() {
    if (this._dirtyPath) {
      return;
    }
    this.dirtyPath = this.path.isDirty() || (this.fillShadow?.isDirty() ?? false) || (this._clipPath?.isDirty() ?? false);
  }
  isPointInPath(x, y) {
    this.updatePathIfDirty();
    return this.path.closedPath && this.path.isPointInPath(x, y);
  }
  distanceSquared(x, y) {
    return this.distanceSquaredTransformedPoint(x, y);
  }
  svgPathData(transform) {
    this.updatePathIfDirty();
    return this.path.toSVG(transform);
  }
  distanceSquaredTransformedPoint(x, y) {
    this.updatePathIfDirty();
    if (this.path.closedPath && this.path.isPointInPath(x, y)) {
      return 0;
    }
    return this.path.distanceSquared(x, y);
  }
  isDirtyPath() {
    return false;
  }
  updatePath() {
  }
  updatePathIfDirty() {
    if (this.dirtyPath || this.isDirtyPath()) {
      this.updatePath();
      this.dirtyPath = false;
    }
  }
  preRender(renderCtx) {
    if (renderCtx.devicePixelRatio !== this.lastPixelRatio) {
      this.dirtyPath = true;
    }
    this.lastPixelRatio = renderCtx.devicePixelRatio;
    this.updatePathIfDirty();
    return super.preRender(renderCtx, this.path.commands.length);
  }
  render(renderCtx) {
    const { ctx } = renderCtx;
    if (this.clip && !isNaN(this._clipX) && !isNaN(this._clipY)) {
      ctx.save();
      const margin = this.strokeWidth / 2;
      this._clipPath ?? (this._clipPath = new ExtendedPath2D());
      this._clipPath.clear();
      this._clipPath.rect(-margin, -margin, this._clipX + margin, this._clipY + margin + margin);
      ctx.clip(this._clipPath?.getPath2D());
      if (this._clipX > 0 && this._clipY > 0) {
        this.drawPath(ctx);
      }
      ctx.restore();
    } else {
      this._clipPath = void 0;
      this.drawPath(ctx);
    }
    this.fillShadow?.markClean();
    super.render(renderCtx);
  }
  drawPath(ctx) {
    this.fillStroke(ctx, this.path.getPath2D());
  }
  toSVG() {
    if (!this.visible)
      return;
    const element2 = createSvgElement("path");
    element2.setAttribute("d", this.svgPathData());
    this.applySvgFillAttributes(element2);
    this.applySvgStrokeAttributes(element2);
    return {
      elements: [element2]
    };
  }
};
Path.className = "Path";
__decorateClass([
  ScenePathChangeDetection()
], Path.prototype, "clip", 2);
__decorateClass([
  ScenePathChangeDetection()
], Path.prototype, "clipX", 1);
__decorateClass([
  ScenePathChangeDetection()
], Path.prototype, "clipY", 1);

// packages/ag-charts-community/src/dom/focusIndicator.ts
var FocusIndicator = class {
  constructor(swapChain) {
    this.swapChain = swapChain;
    this.div = createElement("div");
    this.svg = createSvgElement("svg");
    this.outerPath = createSvgElement("path");
    this.innerPath = createSvgElement("path");
    this.svg.append(this.outerPath);
    this.svg.append(this.innerPath);
    this.outerPath.classList.add("ag-charts-focus-svg-outer-path");
    this.innerPath.classList.add("ag-charts-focus-svg-inner-path");
    this.element = createElement("div", "ag-charts-focus-indicator");
    this.element.ariaHidden = "true";
    this.element.append(this.svg);
    this.swapChain.addListener("swap", (parent) => this.onSwap(parent));
  }
  clear() {
  }
  update(focus, rect, clip) {
    if (rect == null)
      return;
    if (focus instanceof Path) {
      const transform = (localX, localY) => {
        let { x, y } = Transformable.toCanvasPoint(focus, localX, localY);
        x -= rect.x ?? 0;
        y -= rect.y ?? 0;
        return { x, y };
      };
      const d = focus.svgPathData(transform);
      this.outerPath.setAttribute("d", d);
      this.innerPath.setAttribute("d", d);
      this.show(this.svg);
    } else {
      let bbox;
      if (clip) {
        const x0 = Math.max(focus.x - rect.x, 0);
        const y0 = Math.max(focus.y - rect.y, 0);
        const x1 = Math.min(focus.x + focus.width - rect.x, rect.width);
        const y1 = Math.min(focus.y + focus.height - rect.y, rect.height);
        bbox = new BBox(x0, y0, x1 - x0, y1 - y0);
      } else {
        bbox = new BBox(focus.x - rect.x, focus.y - rect.y, focus.width, focus.height);
      }
      setElementBBox(this.div, bbox);
      this.show(this.div);
    }
  }
  onSwap(newParent) {
    if (newParent === this.element.parentElement)
      return;
    this.element.remove();
    newParent.appendChild(this.element);
    this.overrideFocusVisible(this.focusVisible);
  }
  show(child) {
    this.element.innerHTML = "";
    this.element.append(child);
  }
  overrideFocusVisible(focusVisible) {
    this.focusVisible = focusVisible;
    const opacity = { true: "1", false: "0", undefined: "" };
    const parent = this.element.parentElement;
    parent?.style.setProperty("opacity", opacity[`${focusVisible}`]);
  }
  // Get the `:focus-visible` CSS state.
  isFocusVisible() {
    const parent = this.element.parentElement;
    return parent != null && getWindow().getComputedStyle(parent).opacity === "1";
  }
};

// packages/ag-charts-community/src/dom/focusSwapChain.ts
var FocusSwapChain = class {
  constructor(label1, label2, id, announcerRole) {
    this.label1 = label1;
    this.label2 = label2;
    this.hasFocus = false;
    this.skipDispatch = false;
    this.listeners = {
      blur: [],
      focus: [],
      swap: []
    };
    this.onBlur = (e) => {
      setElementStyle(e.target, "pointer-events", void 0);
      return !this.skipDispatch && this.dispatch("blur", e);
    };
    this.onFocus = (e) => {
      setElementStyle(e.target, "pointer-events", "auto");
      return !this.skipDispatch && this.dispatch("focus", e);
    };
    setAttribute(this.label1, "id", `${id}-label1`);
    setAttribute(this.label2, "id", `${id}-label2`);
    setElementStyle(this.label1, "display", "none");
    setElementStyle(this.label2, "display", "none");
    this.activeAnnouncer = this.createAnnouncer(announcerRole);
    this.inactiveAnnouncer = this.createAnnouncer(announcerRole);
    setAttribute(this.activeAnnouncer, "tabindex", 0);
    this.label2.insertAdjacentElement("afterend", this.activeAnnouncer);
    this.label2.insertAdjacentElement("afterend", this.inactiveAnnouncer);
    this.swap("");
  }
  createAnnouncer(role) {
    const announcer = createElement("div");
    announcer.role = role;
    announcer.className = "ag-charts-swapchain";
    announcer.addEventListener("blur", this.onBlur);
    announcer.addEventListener("focus", this.onFocus);
    return announcer;
  }
  destroy() {
    for (const announcer of [this.activeAnnouncer, this.inactiveAnnouncer]) {
      announcer.removeEventListener("blur", this.onBlur);
      announcer.removeEventListener("focus", this.onFocus);
      announcer.remove();
    }
  }
  focus(opts) {
    this.focusOptions = opts;
    this.activeAnnouncer.focus(opts);
    this.focusOptions = void 0;
  }
  update(newLabel) {
    this.skipDispatch = true;
    this.swap(newLabel);
    if (this.hasFocus) {
      this.activeAnnouncer.focus(this.focusOptions);
    }
    this.skipDispatch = false;
  }
  addListener(type, handler) {
    this.listeners[type].push(handler);
    if (type === "swap") {
      const swapHandler = handler;
      swapHandler(this.activeAnnouncer);
    }
  }
  dispatch(type, param) {
    if (type === "focus")
      this.hasFocus = true;
    else if (type === "blur")
      this.hasFocus = false;
    this.listeners[type].forEach((fn) => fn(param));
  }
  swap(newLabel) {
    const userTabIndex = this.activeAnnouncer.tabIndex;
    this.label2.textContent = newLabel;
    [this.inactiveAnnouncer, this.activeAnnouncer] = [this.activeAnnouncer, this.inactiveAnnouncer];
    [this.label1, this.label2] = [this.label2, this.label1];
    setAttributes(this.inactiveAnnouncer, {
      "aria-labelledby": this.label1.id,
      "aria-hidden": true,
      tabindex: void 0
    });
    setAttributes(this.activeAnnouncer, {
      "aria-labelledby": this.label1.id,
      "aria-hidden": false,
      tabindex: userTabIndex
    });
    this.dispatch("swap", this.activeAnnouncer);
  }
};

// packages/ag-charts-community/src/chart/interaction/keyBindings.ts
var KEY_BINDINGS = {
  arrowdown: { bindings: [{ code: "ArrowDown" }] },
  arrowleft: { bindings: [{ code: "ArrowLeft" }] },
  arrowright: { bindings: [{ code: "ArrowRight" }] },
  arrowup: { bindings: [{ code: "ArrowUp" }] },
  delete: { bindings: [{ key: "Backspace" }, { key: "Delete" }], activatesFocusIndicator: false },
  redo: {
    bindings: [
      { key: "y", ctrlOrMeta: true },
      { key: "z", ctrlOrMeta: true, shift: true }
    ],
    activatesFocusIndicator: false
  },
  undo: { bindings: [{ key: "z", ctrlOrMeta: true }], activatesFocusIndicator: false },
  submit: { bindings: [{ key: "Enter" }, { code: "Enter" }, { code: "Space" }] },
  zoomin: { bindings: [{ key: "+" }, { code: "ZoomIn" }, { code: "Add" }], activatesFocusIndicator: false },
  zoomout: { bindings: [{ key: "-" }, { code: "ZoomOut" }, { code: "Substract" }], activatesFocusIndicator: false }
};
function matchesKeyBinding(e, bindings) {
  for (const kb of bindings) {
    if ("code" in kb) {
      if (kb.code === e.code)
        return true;
    } else {
      const matches = kb.key === e.key && (kb.shift === void 0 || kb.shift === e.shiftKey) && (kb.ctrlOrMeta === void 0 || kb.ctrlOrMeta === e.ctrlKey || kb.ctrlOrMeta === e.metaKey);
      if (matches)
        return true;
    }
  }
  return false;
}
function mapKeyboardEventToAction(event) {
  for (const [actionName, { activatesFocusIndicator = true, bindings }] of Object.entries(KEY_BINDINGS)) {
    if (matchesKeyBinding(event, bindings)) {
      const name = actionName;
      return { name, activatesFocusIndicator };
    }
  }
  return void 0;
}

// packages/ag-charts-community/src/chart/keyboardUtil.ts
function computeCenter(series, hoverRect, pick) {
  const refPoint = getDatumRefPoint(series, pick.datum);
  if (refPoint != null)
    return { x: refPoint.canvasX, y: refPoint.canvasY };
  const bboxOrPath = pick.bounds;
  if (bboxOrPath == null)
    return;
  if (bboxOrPath instanceof BBox) {
    const { x: centerX, y: centerY } = bboxOrPath.computeCenter();
    return {
      x: hoverRect.x + centerX,
      y: hoverRect.y + centerY
    };
  }
  return Transformable.toCanvas(bboxOrPath).computeCenter();
}
function getPickedFocusBBox({ bounds }) {
  if (bounds instanceof BBox)
    return bounds;
  if (bounds != null)
    return Transformable.toCanvas(bounds);
  return BBox.NaN;
}
function makeKeyboardPointerEvent(series, hoverRect, pick) {
  const { x: canvasX, y: canvasY } = computeCenter(series, hoverRect, pick) ?? {};
  if (canvasX !== void 0 && canvasY !== void 0) {
    return { type: "keyboard", canvasX, canvasY };
  }
  return void 0;
}

// packages/ag-charts-community/src/util/placement.ts
function calculatePlacement(naturalWidth, naturalHeight, container, bounds) {
  let { top, right, bottom, left, width: width2, height: height2 } = bounds;
  if (left != null) {
    if (width2 != null) {
      right = container.width - left + width2;
    } else if (right != null) {
      width2 = container.width - left - right;
    }
  } else if (right != null && width2 != null) {
    left = container.width - right - width2;
  }
  if (top != null) {
    if (height2 != null) {
      bottom = container.height - top - height2;
    } else if (bottom != null) {
      height2 = container.height - bottom - top;
    }
  } else if (bottom != null && height2 != null) {
    top = container.height - bottom - height2;
  }
  if (width2 == null) {
    if (height2 == null) {
      width2 = naturalWidth;
      height2 = naturalHeight;
    } else {
      width2 = Math.ceil(naturalWidth * height2 / naturalHeight);
    }
  } else if (height2 == null) {
    height2 = Math.ceil(naturalHeight * width2 / naturalWidth);
  }
  if (left == null) {
    if (right == null) {
      left = Math.floor((container.width - width2) / 2);
    } else {
      left = container.width - right - width2;
    }
  }
  if (top == null) {
    if (bottom == null) {
      top = Math.floor((container.height - height2) / 2);
    } else {
      top = container.height - height2 - bottom;
    }
  }
  return { x: left, y: top, width: width2, height: height2 };
}

// packages/ag-charts-community/src/util/sanitize.ts
var element = null;
function sanitizeHtml(text) {
  if (text == null) {
    return;
  } else if (text === "") {
    return "";
  }
  element ?? (element = createElement("div"));
  element.textContent = String(text);
  return element.innerHTML;
}

// packages/ag-charts-community/src/chart/marker/shapes.ts
function drawMarkerUnitPolygon(params, moves) {
  const { path, size } = params;
  const { x: x0, y: y0 } = params;
  path.clear();
  let didMove = false;
  for (const [dx, dy] of moves) {
    const x = x0 + (dx - 0.5) * size;
    const y = y0 + (dy - 0.5) * size;
    if (didMove) {
      path.lineTo(x, y);
    } else {
      path.moveTo(x, y);
    }
    didMove = true;
  }
  path.closePath();
}
var MARKER_SHAPES = {
  circle({ path, x, y, size }) {
    const r = size / 2;
    path.clear();
    path.arc(x, y, r, 0, Math.PI * 2);
    path.closePath();
  },
  cross(params) {
    drawMarkerUnitPolygon(params, [
      [0.25, 0],
      [0.5, 0.25],
      [0.75, 0],
      [1, 0.25],
      [0.75, 0.5],
      [1, 0.75],
      [0.75, 1],
      [0.5, 0.75],
      [0.25, 1],
      [0, 0.75],
      [0.25, 0.5],
      [0, 0.25]
    ]);
  },
  diamond(params) {
    drawMarkerUnitPolygon(params, [
      [0.5, 0],
      [1, 0.5],
      [0.5, 1],
      [0, 0.5]
    ]);
  },
  heart({ path, x, y, size }) {
    const r = size / 4;
    y = y + r / 2;
    path.clear();
    path.arc(x - r, y - r, r, toRadians(130), toRadians(330));
    path.arc(x + r, y - r, r, toRadians(220), toRadians(50));
    path.lineTo(x, y + r);
    path.closePath();
  },
  pin({ path, x, y, size: s }) {
    const cx = 0.5;
    const cy = 0.5;
    path.moveTo(x + (0.15625 - cx) * s, y + (0.34375 - cy) * s);
    path.cubicCurveTo(
      x + (0.15625 - cx) * s,
      y + (0.151491 - cy) * s,
      x + (0.307741 - cx) * s,
      y + (0 - cy) * s,
      x + (0.5 - cx) * s,
      y + (0 - cy) * s
    );
    path.cubicCurveTo(
      x + (0.692259 - cx) * s,
      y + (0 - cy) * s,
      x + (0.84375 - cx) * s,
      y + (0.151491 - cy) * s,
      x + (0.84375 - cx) * s,
      y + (0.34375 - cy) * s
    );
    path.cubicCurveTo(
      x + (0.84375 - cx) * s,
      y + (0.493824 - cy) * s,
      x + (0.784625 - cx) * s,
      y + (0.600181 - cy) * s,
      x + (0.716461 - cx) * s,
      y + (0.695393 - cy) * s
    );
    path.cubicCurveTo(
      x + (0.699009 - cx) * s,
      y + (0.719769 - cy) * s,
      x + (0.681271 - cx) * s,
      y + (0.743104 - cy) * s,
      x + (0.663785 - cx) * s,
      y + (0.766105 - cy) * s
    );
    path.cubicCurveTo(
      x + (0.611893 - cx) * s,
      y + (0.834367 - cy) * s,
      x + (0.562228 - cx) * s,
      y + (0.899699 - cy) * s,
      x + (0.528896 - cx) * s,
      y + (0.980648 - cy) * s
    );
    path.cubicCurveTo(
      x + (0.524075 - cx) * s,
      y + (0.992358 - cy) * s,
      x + (0.512663 - cx) * s,
      y + (1 - cy) * s,
      x + (0.5 - cx) * s,
      y + (1 - cy) * s
    );
    path.cubicCurveTo(
      x + (0.487337 - cx) * s,
      y + (1 - cy) * s,
      x + (0.475925 - cx) * s,
      y + (0.992358 - cy) * s,
      x + (0.471104 - cx) * s,
      y + (0.980648 - cy) * s
    );
    path.cubicCurveTo(
      x + (0.487337 - cx) * s,
      y + (1 - cy) * s,
      x + (0.475925 - cx) * s,
      y + (0.992358 - cy) * s,
      x + (0.471104 - cx) * s,
      y + (0.980648 - cy) * s
    );
    path.cubicCurveTo(
      x + (0.437772 - cx) * s,
      y + (0.899699 - cy) * s,
      x + (0.388107 - cx) * s,
      y + (0.834367 - cy) * s,
      x + (0.336215 - cx) * s,
      y + (0.766105 - cy) * s
    );
    path.cubicCurveTo(
      x + (0.318729 - cx) * s,
      y + (0.743104 - cy) * s,
      x + (0.300991 - cx) * s,
      y + (0.719769 - cy) * s,
      x + (0.283539 - cx) * s,
      y + (0.695393 - cy) * s
    );
    path.cubicCurveTo(
      x + (0.215375 - cx) * s,
      y + (0.600181 - cy) * s,
      x + (0.15625 - cx) * s,
      y + (0.493824 - cy) * s,
      x + (0.15625 - cx) * s,
      y + (0.34375 - cy) * s
    );
    path.closePath();
  },
  plus(params) {
    drawMarkerUnitPolygon(params, [
      [1 / 3, 0],
      [2 / 3, 0],
      [2 / 3, 1 / 3],
      [1, 1 / 3],
      [1, 2 / 3],
      [2 / 3, 2 / 3],
      [2 / 3, 1],
      [1 / 3, 1],
      [1 / 3, 2 / 3],
      [0, 2 / 3],
      [0, 1 / 3],
      [1 / 3, 1 / 3]
    ]);
  },
  square({ path, x, y, size, pixelRatio }) {
    const hs = size / 2;
    path.clear();
    path.moveTo(align(pixelRatio, x - hs), align(pixelRatio, y - hs));
    path.lineTo(align(pixelRatio, x + hs), align(pixelRatio, y - hs));
    path.lineTo(align(pixelRatio, x + hs), align(pixelRatio, y + hs));
    path.lineTo(align(pixelRatio, x - hs), align(pixelRatio, y + hs));
    path.closePath();
  },
  star({ path, x, y, size }) {
    const spikes = 5;
    const outerRadius = size / 2;
    const innerRadius = outerRadius / 2;
    const rotation = Math.PI / 2;
    for (let i = 0; i < spikes * 2; i++) {
      const radius = i % 2 === 0 ? outerRadius : innerRadius;
      const angle = i * Math.PI / spikes - rotation;
      const xCoordinate = x + Math.cos(angle) * radius;
      const yCoordinate = y + Math.sin(angle) * radius;
      path.lineTo(xCoordinate, yCoordinate);
    }
    path.closePath();
  },
  triangle(params) {
    drawMarkerUnitPolygon(params, [
      [0.5, 0],
      [1, 0.87],
      [0, 0.87]
    ]);
  }
};

// packages/ag-charts-community/src/chart/marker/marker.ts
var InternalMarker = class extends Path {
  constructor() {
    super(...arguments);
    this.shape = "square";
    this.x = 0;
    this.y = 0;
    this.size = 12;
  }
  updatePath() {
    const { path, shape, x, y, size } = this;
    const pixelRatio = this.layerManager?.canvas?.pixelRatio ?? 1;
    const anchor = Marker.anchor(shape);
    const drawParams = {
      path,
      x: x - (anchor.x - 0.5) * size,
      y: y - (anchor.y - 0.5) * size,
      size,
      pixelRatio
    };
    path.clear();
    if (typeof shape === "string") {
      MARKER_SHAPES[shape](drawParams);
    } else if (typeof shape === "function") {
      shape(drawParams);
    }
  }
  computeBBox() {
    const { x, y, size } = this;
    const anchor = Marker.anchor(this.shape);
    return new BBox(x - size * anchor.x, y - size * anchor.y, size, size);
  }
  executeFill(ctx, path) {
    if (!path)
      return;
    return super.executeFill(ctx, path);
  }
  executeStroke(ctx, path) {
    if (!path)
      return;
    return super.executeStroke(ctx, path);
  }
};
__decorateClass([
  ScenePathChangeDetection()
], InternalMarker.prototype, "shape", 2);
__decorateClass([
  ScenePathChangeDetection()
], InternalMarker.prototype, "x", 2);
__decorateClass([
  ScenePathChangeDetection()
], InternalMarker.prototype, "y", 2);
__decorateClass([
  ScenePathChangeDetection({ convertor: Math.abs })
], InternalMarker.prototype, "size", 2);
var Marker = class extends Rotatable(Scalable(Translatable(InternalMarker))) {
  static anchor(shape) {
    if (shape === "pin") {
      return { x: 0.5, y: 1 };
    } else if (typeof shape === "function" && "anchor" in shape) {
      return shape.anchor;
    }
    return { x: 0.5, y: 0.5 };
  }
  constructor(options) {
    super(options);
    if (options?.shape != null) {
      this.shape = options.shape;
    }
  }
};

// packages/ag-charts-community/src/chart/legend/legendSymbol.ts
function legendSymbolSvg(symbol, size, lineSize = size * (5 / 3)) {
  const group = new Group();
  const markerStrokeWidth = Math.min(symbol.marker.strokeWidth, 2);
  const lineStrokeWidth = Math.min(symbol.line?.strokeWidth ?? 0, 2);
  const width2 = Math.max(symbol.marker.enabled === false ? 0 : size, symbol.line == null ? 0 : lineSize);
  const height2 = Math.max(symbol.marker.enabled === false ? 0 : size, lineStrokeWidth);
  if (symbol.line != null) {
    const { stroke, strokeOpacity, lineDash } = symbol.line;
    const line = new Line();
    line.x1 = 0;
    line.y1 = height2 / 2;
    line.x2 = width2;
    line.y2 = height2 / 2;
    line.stroke = stroke;
    line.strokeOpacity = strokeOpacity;
    line.strokeWidth = lineStrokeWidth;
    line.lineDash = lineDash;
    group.append(line);
  }
  if (symbol.marker.enabled !== false) {
    const { shape, fill, fillOpacity, stroke, strokeOpacity, lineDash, lineDashOffset } = symbol.marker;
    const marker = new Marker();
    marker.shape = shape ?? "square";
    marker.size = size;
    marker.fill = fill;
    marker.fillOpacity = fillOpacity;
    marker.stroke = stroke;
    marker.strokeOpacity = strokeOpacity;
    marker.strokeWidth = markerStrokeWidth;
    marker.lineDash = lineDash;
    marker.lineDashOffset = lineDashOffset;
    const anchor = Marker.anchor(shape);
    const x = width2 / 2 + (anchor.x - 0.5) * size;
    const y = height2 / 2 + (anchor.y - 0.5) * size;
    const scale = size / (size + markerStrokeWidth);
    marker.x = 0;
    marker.y = 0;
    marker.translationX = x;
    marker.translationY = y;
    marker.scalingX = scale;
    marker.scalingY = scale;
    group.append(marker);
  }
  return Group.toSVG(group, width2, height2);
}

// packages/ag-charts-community/src/chart/tooltip/springAnimation.ts
var M = 0.1;
var K = 200;
var C = 12;
var DELTA = 0.5;
var SpringAnimation = class extends Listeners {
  constructor() {
    super(...arguments);
    this.x1 = NaN;
    this.y1 = NaN;
    this.x = NaN;
    this.y = NaN;
    this.vx = 0;
    this.vy = 0;
    this.t0 = NaN;
    this.animationFrameHandle = void 0;
  }
  reset() {
    this.x = NaN;
    this.y = NaN;
    if (this.animationFrameHandle != null) {
      cancelAnimationFrame(this.animationFrameHandle);
      this.animationFrameHandle = void 0;
    }
  }
  update(x, y) {
    if (Number.isNaN(this.x) || Number.isNaN(this.y)) {
      this.x = x;
      this.y = y;
      this.vx = 0;
      this.vy = 0;
      this.emitUpdate();
      if (this.animationFrameHandle != null) {
        cancelAnimationFrame(this.animationFrameHandle);
        this.animationFrameHandle = void 0;
      }
      return;
    }
    this.x1 = x;
    this.y1 = y;
    this.t0 = Date.now();
    if (this.animationFrameHandle == null) {
      this.animationFrameHandle = requestAnimationFrame(this.onFrame.bind(this));
    }
  }
  onFrame() {
    this.animationFrameHandle = void 0;
    const { x1, y1, t0 } = this;
    const t1 = Date.now();
    const dt = t1 - t0;
    this.t0 = t1;
    const stepT = 1e-3;
    const iterations = Math.ceil(dt / (stepT * 1e3)) | 0;
    let { x, y, vx, vy } = this;
    for (let i = 0; i < iterations; i += 1) {
      const dx = x - x1;
      const dy = y - y1;
      const ax = -(K * dx + C * vx) / M;
      const ay = -(K * dy + C * vy) / M;
      vx += ax * stepT;
      vy += ay * stepT;
      x += vx * stepT;
      y += vy * stepT;
    }
    if (Math.hypot(x - x1, y - y1) < DELTA) {
      this.x = this.x1;
      this.y = this.y1;
      this.vx = 0;
      this.vy = 0;
    } else {
      this.x = x;
      this.y = y;
      this.vx = vx;
      this.vy = vy;
      this.animationFrameHandle = requestAnimationFrame(this.onFrame.bind(this));
    }
    this.emitUpdate();
  }
  emitUpdate() {
    this.dispatch("update", { type: "update", x: this.x, y: this.y });
  }
};

// packages/ag-charts-community/src/chart/tooltip/tooltip.ts
var DEFAULT_TOOLTIP_CLASS = "ag-charts-tooltip";
var DEFAULT_TOOLTIP_DARK_CLASS = "ag-charts-tooltip--dark";
function tooltipContentAriaLabel(content) {
  const ariaLabel = [];
  if (content.type === "raw")
    return "";
  if (content.heading != null)
    ariaLabel.push(content.heading);
  if (content.title != null)
    ariaLabel.push(content.title);
  content.data?.forEach((datum) => {
    ariaLabel.push(datum.label ?? datum.fallbackLabel, datum.value);
  });
  return ariaLabel.join("; ");
}
function dataHtml(label, value, inline) {
  let rowHtml = "";
  if (label == null) {
    rowHtml += `<span class="${DEFAULT_TOOLTIP_CLASS}-label">${sanitizeHtml(value)}</span>`;
  } else {
    rowHtml += `<span class="${DEFAULT_TOOLTIP_CLASS}-label">${sanitizeHtml(label)}</span>`;
    rowHtml += " ";
    rowHtml += `<span class="${DEFAULT_TOOLTIP_CLASS}-value">${sanitizeHtml(value)}</span>`;
  }
  const rowClassNames = [`${DEFAULT_TOOLTIP_CLASS}-row`];
  if (inline)
    rowClassNames.push(`${DEFAULT_TOOLTIP_CLASS}-row--inline`);
  rowHtml = `<div class="${rowClassNames.join(" ")}">${rowHtml}</div>`;
  return rowHtml;
}
function tooltipContentHtml(content) {
  if (content.type === "raw")
    return content.rawHtmlString;
  let html = "";
  if ((content.heading == null || content.title == null) && content.data?.length === 1 && content.data[0].label == null && content.data[0].value != null) {
    const datum = content.data[0];
    html += dataHtml(content.heading ?? content.title, datum.value, false);
  } else {
    const dataInline = content.title == null && content.data?.length === 1;
    if (content.heading != null) {
      html += `<span class="${DEFAULT_TOOLTIP_CLASS}-heading">${sanitizeHtml(content.heading)}</span>`;
      html += " ";
    }
    const symbol = content.symbol == null ? void 0 : legendSymbolSvg(content.symbol, 12);
    if (symbol != null && (content.title != null || content.data?.length)) {
      html += `<span class="${DEFAULT_TOOLTIP_CLASS}-symbol">${symbol}</span>`;
    }
    if (content.title != null) {
      html += `<span class="${DEFAULT_TOOLTIP_CLASS}-title">${sanitizeHtml(content.title)}</span>`;
      html += " ";
    }
    content.data?.forEach((datum) => {
      html += dataHtml(datum.label ?? datum.fallbackLabel, datum.value, dataInline);
      html += " ";
    });
  }
  html = `<div class="${DEFAULT_TOOLTIP_CLASS}-content">${html.trimEnd()}</div>`;
  return html;
}
var TooltipPosition = class extends BaseProperties {
  constructor() {
    super(...arguments);
    /** The type of positioning for the tooltip. By default, the tooltip follows the pointer. */
    this.type = "pointer";
    /** The horizontal offset in pixels for the position of the tooltip. */
    this.xOffset = 0;
    /** The vertical offset in pixels for the position of the tooltip. */
    this.yOffset = 0;
  }
};
__decorateClass([
  Validate(
    UNION(
      [
        "pointer",
        "node",
        "top",
        "right",
        "bottom",
        "left",
        "top-left",
        "top-right",
        "bottom-right",
        "bottom-left",
        { value: "sparkline", undocumented: true },
        { value: "sparkline-", undocumented: true }
      ],
      "a position type"
    )
  )
], TooltipPosition.prototype, "type", 2);
__decorateClass([
  Validate(NUMBER)
], TooltipPosition.prototype, "xOffset", 2);
__decorateClass([
  Validate(NUMBER)
], TooltipPosition.prototype, "yOffset", 2);
var Tooltip = class extends BaseProperties {
  constructor() {
    super();
    this.enabled = true;
    this.delay = 0;
    this.range = void 0;
    this.wrapping = "hyphenate";
    this.position = new TooltipPosition();
    this.darkTheme = false;
    this.bounds = "extended";
    this.destroyFns = [];
    this.springAnimation = new SpringAnimation();
    this.enableInteraction = false;
    this.wrapTypes = ["always", "hyphenate", "on-space", "never"];
    this.showTimeout = 0;
    this._showArrow = true;
    this._compact = false;
    this._visible = false;
    this.positionParams = void 0;
    this.destroyFns.push(this.springAnimation.addListener("update", this.updateTooltipPosition.bind(this)));
  }
  get interactive() {
    return this.enableInteraction;
  }
  setup(domManager) {
    if ("togglePopover" in getWindow().HTMLElement.prototype) {
      this.element = domManager.addChild("canvas-overlay", DEFAULT_TOOLTIP_CLASS);
      this.element.setAttribute("popover", "manual");
      this.element.className = DEFAULT_TOOLTIP_CLASS;
    }
  }
  destroy(domManager) {
    domManager.removeChild("canvas-overlay", DEFAULT_TOOLTIP_CLASS);
    this.destroyFns.forEach((f) => f());
  }
  isVisible() {
    return this._visible;
  }
  contains(node) {
    return this.element?.contains(node) ?? false;
  }
  updateTooltipPosition() {
    const { element: element2, positionParams } = this;
    if (element2 == null || positionParams == null)
      return;
    const { canvasRect, relativeRect, meta } = positionParams;
    const { x: canvasX, y: canvasY } = this.springAnimation;
    const positionType = meta.position?.type ?? this.position.type;
    const xOffset = meta.position?.xOffset ?? 0;
    const yOffset = meta.position?.yOffset ?? 0;
    const minX = relativeRect.x;
    const minY = relativeRect.y;
    const maxX = relativeRect.width - element2.clientWidth - 1 + minX;
    const maxY = relativeRect.height - element2.clientHeight + minY;
    let tooltipBounds = this.getTooltipBounds({ positionType, canvasX, canvasY, yOffset, xOffset, canvasRect });
    let position = calculatePlacement(element2.clientWidth, element2.clientHeight, relativeRect, tooltipBounds);
    if (positionType === "sparkline" && (position.x <= minX || position.x >= maxX)) {
      tooltipBounds = this.getTooltipBounds({
        positionType: "sparkline-constrained",
        canvasX,
        canvasY,
        yOffset,
        xOffset,
        canvasRect
      });
      position = calculatePlacement(element2.clientWidth, element2.clientHeight, relativeRect, tooltipBounds);
    }
    const left = clamp(minX, position.x, maxX);
    const top = clamp(minY, position.y, maxY);
    const constrained = left !== position.x || top !== position.y;
    const defaultShowArrow = (positionType === "node" || positionType === "pointer") && !constrained && !xOffset && !yOffset;
    const showArrow = meta.showArrow ?? this.showArrow ?? defaultShowArrow;
    this.updateShowArrow(showArrow);
    this.updateCompact(positionType === "sparkline" || positionType === "sparkline-constrained");
    element2.style.transform = `translate(${left}px, ${top}px)`;
  }
  /**
   * Shows tooltip at the given event's coordinates.
   * If the `html` parameter is missing, moves the existing tooltip to the new position.
   */
  show(boundingRect, canvasRect, meta, content, instantly = false) {
    const { element: element2 } = this;
    if (element2 != null && content != null) {
      element2.innerHTML = tooltipContentHtml(content);
    } else if (element2 == null || element2.innerHTML === "") {
      this.toggle(false);
      return;
    }
    const relativeRect = {
      x: boundingRect.x - canvasRect.x,
      y: boundingRect.y - canvasRect.y,
      width: boundingRect.width,
      height: boundingRect.height
    };
    this.positionParams = {
      canvasRect,
      relativeRect,
      meta
    };
    this.springAnimation.update(meta.canvasX, meta.canvasY);
    element2.style.top = `${canvasRect.top}px`;
    element2.style.left = `${canvasRect.left}px`;
    if (meta.enableInteraction) {
      this.enableInteraction = true;
      element2.style.pointerEvents = "auto";
      element2.removeAttribute("aria-hidden");
    } else {
      this.enableInteraction = false;
      element2.style.pointerEvents = "none";
      element2.setAttribute("aria-hidden", "true");
    }
    if (this.delay > 0 && !instantly) {
      this.toggle(false);
      this.showTimeout = setTimeout(() => {
        this.toggle(true);
      }, this.delay);
    } else {
      this.toggle(true);
    }
  }
  hide() {
    this.springAnimation.reset();
    this.toggle(false);
  }
  toggle(visible) {
    if (!this.element?.isConnected)
      return;
    this._visible = visible;
    const { classList } = this.element;
    const toggleClass = (name, include) => classList.toggle(`${DEFAULT_TOOLTIP_CLASS}--${name}`, include);
    if (!visible) {
      clearTimeout(this.showTimeout);
    }
    toggleClass("no-interaction", !this.enableInteraction);
    toggleClass("arrow", this._showArrow);
    toggleClass("compact", this._compact);
    classList.toggle(DEFAULT_TOOLTIP_DARK_CLASS, this.darkTheme);
    this.element.togglePopover(visible);
    if (visible) {
      this.updateTooltipPosition();
    }
    for (const wrapType of this.wrapTypes) {
      classList.toggle(`${DEFAULT_TOOLTIP_CLASS}--wrap-${wrapType}`, wrapType === this.wrapping);
    }
  }
  updateShowArrow(show) {
    this._showArrow = show;
  }
  updateCompact(compact) {
    this._compact = compact;
  }
  getTooltipBounds(opts) {
    if (!this.element)
      return {};
    const { positionType, canvasX, canvasY, yOffset, xOffset, canvasRect } = opts;
    const { clientWidth: tooltipWidth, clientHeight: tooltipHeight } = this.element;
    const bounds = { width: tooltipWidth, height: tooltipHeight };
    switch (positionType) {
      case "node":
      case "pointer": {
        bounds.top = canvasY + yOffset - tooltipHeight - 8;
        bounds.left = canvasX + xOffset - tooltipWidth / 2;
        return bounds;
      }
      case "top": {
        bounds.top = yOffset;
        bounds.left = canvasRect.width / 2 - tooltipWidth / 2 + xOffset;
        return bounds;
      }
      case "right": {
        bounds.top = canvasRect.height / 2 - tooltipHeight / 2 + yOffset;
        bounds.left = canvasRect.width - tooltipWidth / 2 + xOffset;
        return bounds;
      }
      case "left": {
        bounds.top = canvasRect.height / 2 - tooltipHeight / 2 + yOffset;
        bounds.left = xOffset;
        return bounds;
      }
      case "bottom": {
        bounds.top = canvasRect.height - tooltipHeight + yOffset;
        bounds.left = canvasRect.width / 2 - tooltipWidth / 2 + xOffset;
        return bounds;
      }
      case "top-left": {
        bounds.top = yOffset;
        bounds.left = xOffset;
        return bounds;
      }
      case "top-right": {
        bounds.top = yOffset;
        bounds.left = canvasRect.width - tooltipWidth + xOffset;
        return bounds;
      }
      case "bottom-right": {
        bounds.top = canvasRect.height - tooltipHeight + yOffset;
        bounds.left = canvasRect.width - tooltipWidth + xOffset;
        return bounds;
      }
      case "bottom-left": {
        bounds.top = canvasRect.height - tooltipHeight + yOffset;
        bounds.left = xOffset;
        return bounds;
      }
      case "sparkline": {
        bounds.top = canvasY + yOffset - tooltipHeight / 2;
        bounds.left = canvasX + xOffset + 8;
        return bounds;
      }
      case "sparkline-constrained": {
        bounds.top = canvasY + yOffset - tooltipHeight / 2;
        bounds.left = canvasX + xOffset - 8 - tooltipWidth;
        return bounds;
      }
    }
  }
};
__decorateClass([
  Validate(BOOLEAN)
], Tooltip.prototype, "enabled", 2);
__decorateClass([
  Validate(BOOLEAN, { optional: true })
], Tooltip.prototype, "showArrow", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER)
], Tooltip.prototype, "delay", 2);
__decorateClass([
  Validate(INTERACTION_RANGE, { optional: true })
], Tooltip.prototype, "range", 2);
__decorateClass([
  Validate(TEXT_WRAP)
], Tooltip.prototype, "wrapping", 2);
__decorateClass([
  Validate(OBJECT)
], Tooltip.prototype, "position", 2);
__decorateClass([
  Validate(BOOLEAN)
], Tooltip.prototype, "darkTheme", 2);
__decorateClass([
  Validate(UNION(["extended", "canvas"]))
], Tooltip.prototype, "bounds", 2);

// packages/ag-charts-community/src/chart/series/seriesAreaManager.ts
var SeriesAreaManager = class extends BaseManager {
  constructor(chart) {
    super();
    this.chart = chart;
    this.id = createId(this);
    this.series = [];
    this.highlight = {
      /** Last received event that still needs to be applied. */
      pendingHoverEvent: void 0,
      /** Last applied event. */
      appliedHoverEvent: void 0,
      /** Last applied event, which has been temporarily stashed during the main chart update cycle. */
      stashedHoverEvent: void 0
    };
    this.tooltip = {
      lastHover: void 0
    };
    /**
     * A11y Requirements for Tooltip/Highlight (see AG-13051 for details):
     *
     *   -   When the series-area is blurred, always the mouse to update the tooltip/highlight.
     *
     *   -   When the series-area receives a `focus` event, use `:focus-visible` to guess the input device.
     *       (this is decided by the browser).
     *
     *   -   For keyboard users, `focus` and `keydown` events always updates & shows the tooltip/highlight on
     *       the currently (or newly) focused datum.
     *
     *   -   For keyboard users, `mousemove` events update the tooltip/highlight iff `pickNode` finds a match
     *       for the mouse event offsets.
     */
    this.hoverDevice = "pointer";
    /**
     * This is the "second last" input event. It can be useful for keydown
     * events that for which don't to set the isFocusVisible state
     * (e.g. Backspace/Delete key on FC annotations, see AG-13041).
     *
     * Use with caution! The focus indicator must ALWAYS be visible for
     * keyboard-only users.
     */
    this.previousInputDevice = "keyboard";
    this.focus = {
      sortedSeries: [],
      series: void 0,
      seriesIndex: 0,
      datumIndex: 0,
      datum: void 0
    };
    this.hoverScheduler = debouncedAnimationFrame(() => {
      if (!this.tooltip.lastHover && !this.highlight.pendingHoverEvent)
        return;
      if (this.chart.getUpdateType() <= 4 /* SERIES_UPDATE */) {
        this.hoverScheduler.schedule();
        return;
      }
      if (this.highlight.pendingHoverEvent) {
        this.handleHoverHighlight(false);
      }
      if (this.tooltip.lastHover) {
        this.handleHoverTooltip(this.tooltip.lastHover, false);
      }
    });
    const label1 = chart.ctx.domManager.addChild("series-area", "series-area-aria-label1");
    const label2 = chart.ctx.domManager.addChild("series-area", "series-area-aria-label2");
    this.swapChain = new FocusSwapChain(label1, label2, this.id, "img");
    this.swapChain.addListener("blur", () => this.onBlur());
    this.swapChain.addListener("focus", () => this.onFocus());
    this.focusIndicator = new FocusIndicator(this.swapChain);
    this.focusIndicator.overrideFocusVisible(chart.mode === "integrated" ? false : void 0);
    const { seriesDragInterpreter, seriesWidget, containerWidget } = chart.ctx.widgets;
    seriesWidget.setTabIndex(-1);
    this.destroyFns.push(
      () => chart.ctx.domManager.removeChild("series-area", "series-area-aria-label1"),
      () => chart.ctx.domManager.removeChild("series-area", "series-area-aria-label2"),
      seriesWidget.addListener("focus", () => this.swapChain.focus()),
      seriesWidget.addListener("mousemove", (event) => this.onHover(event)),
      seriesWidget.addListener("wheel", (event) => this.onWheel(event)),
      seriesWidget.addListener("mouseleave", (event) => this.onLeave(event)),
      seriesWidget.addListener("keydown", (event) => this.onKeyDown(event)),
      seriesWidget.addListener("contextmenu", (event, current) => this.onContextMenu(event, current)),
      seriesDragInterpreter.addListener("drag-move", (event) => this.onDragMove(event)),
      seriesDragInterpreter.addListener("click", (event) => this.onClick(event, seriesWidget)),
      seriesDragInterpreter.addListener("dblclick", (event) => this.onClick(event, seriesWidget)),
      containerWidget.addListener("contextmenu", (event, current) => this.onContextMenu(event, current)),
      containerWidget.addListener("click", (event, current) => this.onClick(event, current)),
      containerWidget.addListener("dblclick", (event, current) => this.onClick(event, current)),
      chart.ctx.animationManager.addListener("animation-start", () => this.clearAll()),
      chart.ctx.domManager.addListener("resize", () => this.clearAll()),
      chart.ctx.highlightManager.addListener("highlight-change", (event) => this.changeHighlightDatum(event)),
      chart.ctx.layoutManager.addListener("layout:complete", (event) => this.layoutComplete(event)),
      chart.ctx.updateService.addListener("pre-scene-render", () => this.preSceneRender()),
      chart.ctx.updateService.addListener("update-complete", () => this.updateComplete()),
      chart.ctx.zoomManager.addListener("zoom-change", () => this.clearAll()),
      chart.ctx.zoomManager.addListener("zoom-pan-start", () => this.clearAll())
    );
  }
  isState(allowedStates) {
    return this.chart.ctx.interactionManager.isState(allowedStates);
  }
  isIgnoredTouch(event) {
    if (event.device !== "touch" || event.type === "click")
      return false;
    if (this.chart.ctx.chartService.touch.dragAction === "hover")
      return false;
    if (this.chart.ctx.chartService.touch.dragAction === "drag") {
      if (this.isState(9 /* AnnotationsMoveable */)) {
        return false;
      }
    }
    return true;
  }
  dataChanged() {
    var _a;
    (_a = this.highlight).stashedHoverEvent ?? (_a.stashedHoverEvent = this.highlight.appliedHoverEvent);
    this.chart.ctx.tooltipManager.removeTooltip(this.id);
    this.focusIndicator.clear();
    this.clearHighlight();
  }
  preSceneRender() {
    if (this.highlight.stashedHoverEvent != null) {
      this.highlight.pendingHoverEvent = this.highlight.stashedHoverEvent;
      this.highlight.stashedHoverEvent = void 0;
      this.handleHoverHighlight(true);
    }
    if (this.tooltip.lastHover != null) {
      this.handleHoverTooltip(this.tooltip.lastHover, true);
    }
  }
  updateComplete() {
    if (this.focusIndicator.isFocusVisible() && this.isState(34 /* Focusable */)) {
      this.handleSeriesFocus(0, 0, true);
    }
  }
  update(type, opts) {
    this.chart.ctx.updateService.update(type, opts);
  }
  seriesChanged(series) {
    this.focus.sortedSeries = [...series].sort((a, b) => {
      let fpA = a.properties.focusPriority ?? Infinity;
      let fpB = b.properties.focusPriority ?? Infinity;
      if (fpA === fpB) {
        [fpA, fpB] = [a._declarationOrder, b._declarationOrder];
      }
      if (fpA < fpB) {
        return -1;
      } else if (fpA > fpB) {
        return 1;
      }
      return 0;
    });
    this.series = series;
  }
  layoutComplete(event) {
    this.seriesRect = event.series.rect;
    this.hoverRect = event.series.paddedRect;
    this.chart.ctx.widgets.seriesWidget.setBounds(event.series.paddedRect);
    this.chart.ctx.widgets.chartWidget.setBounds(event.chart);
  }
  onContextMenu(event, current) {
    const { sourceEvent } = event;
    if (sourceEvent.currentTarget != current.getElement())
      return;
    if (sourceEvent.target == this.chart.ctx.widgets.containerWidget.getElement()) {
      if (this.isState(36 /* ContextMenuable */)) {
        const { currentX: canvasX2, currentY: canvasY2 } = event;
        this.chart.ctx.contextMenuRegistry.dispatchContext("all", { sourceEvent, canvasX: canvasX2, canvasY: canvasY2 }, {});
      }
      return;
    }
    let pickedNode;
    let position;
    if (this.focusIndicator.isFocusVisible()) {
      pickedNode = this.chart.ctx.highlightManager.getActiveHighlight();
      if (pickedNode && this.seriesRect && pickedNode.midPoint) {
        position = Transformable.toCanvasPoint(
          pickedNode.series.contentGroup,
          pickedNode.midPoint.x,
          pickedNode.midPoint.y
        );
      }
    } else if (this.isState(36 /* ContextMenuable */)) {
      const match = this.pickNode({ x: event.currentX, y: event.currentY }, "context-menu");
      if (match) {
        this.chart.ctx.highlightManager.updateHighlight(this.id);
        pickedNode = match.datum;
      }
    }
    const pickedSeries = pickedNode?.series;
    this.clearAll();
    const canvasX = event.currentX + current.cssLeft();
    const canvasY = event.currentY + current.cssTop();
    this.chart.ctx.contextMenuRegistry.dispatchContext(
      "series-area",
      { sourceEvent, canvasX, canvasY },
      { pickedSeries, pickedNode },
      position
    );
  }
  onLeave(event) {
    if (!this.isState(41 /* Clickable */))
      return;
    const relatedTarget = event.sourceEvent.relatedTarget;
    if (relatedTarget?.className === "ag-charts-text-input__textarea") {
      return;
    }
    if (this.chart.ctx.tooltipManager.isEnteringInteractiveTooltip(event)) {
      return;
    }
    this.chart.ctx.domManager.updateCursor(this.id);
    if (!this.focusIndicator.isFocusVisible())
      this.clearAll();
  }
  onWheel(_event) {
    if (!this.isState(41 /* Clickable */))
      return;
    this.focusIndicator?.overrideFocusVisible(false);
    this.previousInputDevice = "pointer";
  }
  onDragMove(event) {
    if (!this.isState(41 /* Clickable */))
      return;
    this.focusIndicator?.overrideFocusVisible(false);
    this.onHoverLikeEvent(event);
  }
  onHover(event) {
    if (!this.isState(41 /* Clickable */))
      return;
    this.onHoverLikeEvent(event);
  }
  onHoverLikeEvent(event) {
    if (this.isIgnoredTouch(event))
      return;
    if (event.device === "touch" || excludesType(event, "drag-move")) {
      this.tooltip.lastHover = event;
    }
    if (event.device === "touch" && this.chart.ctx.chartService.touch.dragAction === "hover") {
      event.sourceEvent.preventDefault();
    }
    this.hoverDevice = "pointer";
    this.previousInputDevice = "pointer";
    this.highlight.pendingHoverEvent = event;
    this.hoverScheduler.schedule();
    if (this.isState(32 /* Default */)) {
      const { currentX: x, currentY: y } = event;
      const found = this.pickNode({ x, y }, "event");
      if (found?.series.hasEventListener("nodeClick") || found?.series.hasEventListener("nodeDoubleClick")) {
        this.chart.ctx.domManager.updateCursor(this.id, "pointer");
      } else {
        this.chart.ctx.domManager.updateCursor(this.id);
      }
    }
  }
  onClick(event, current) {
    if (event.device === "touch" && current === this.chart.ctx.widgets.seriesWidget) {
      this.swapChain.focus({ preventScroll: true });
    }
    if (!this.isState(41 /* Clickable */))
      return;
    if (current === this.chart.ctx.widgets.seriesWidget) {
      if (!current.getElement().contains(event.sourceEvent.target)) {
        return;
      }
    } else if (event.sourceEvent.target != current.getElement()) {
      return;
    }
    this.focusIndicator.overrideFocusVisible(false);
    this.onHoverLikeEvent(event);
    if (!this.isState(32 /* Default */))
      return;
    if (current == this.chart.ctx.widgets.seriesWidget && this.checkSeriesNodeClick(event)) {
      this.update(4 /* SERIES_UPDATE */);
      event.sourceEvent.preventDefault();
      return;
    }
    const newEvent = { type: event.type === "click" ? "click" : "doubleClick", event: event.sourceEvent };
    this.chart.fireEvent(newEvent);
  }
  onFocus() {
    if (!this.isState(34 /* Focusable */))
      return;
    this.hoverDevice = this.focusIndicator.isFocusVisible() ? "keyboard" : "pointer";
    this.handleFocus(0, 0);
  }
  onBlur() {
    if (!this.isState(34 /* Focusable */))
      return;
    this.hoverDevice = "pointer";
    this.clearAll();
    this.focusIndicator.overrideFocusVisible(void 0);
  }
  onKeyDown(widgetEvent) {
    if (!this.isState(43 /* Keyable */))
      return;
    const action = mapKeyboardEventToAction(widgetEvent.sourceEvent);
    if (action?.activatesFocusIndicator === false) {
      this.focusIndicator.overrideFocusVisible(this.previousInputDevice === "keyboard");
    }
    switch (action?.name) {
      case "redo":
        return this.chart.ctx.chartEventManager.seriesEvent("series-redo");
      case "undo":
        return this.chart.ctx.chartEventManager.seriesEvent("series-undo");
      case "zoomin":
        return this.chart.ctx.chartEventManager.seriesKeyNavZoom(1, widgetEvent);
      case "zoomout":
        return this.chart.ctx.chartEventManager.seriesKeyNavZoom(-1, widgetEvent);
      case "arrowup":
        return this.onArrow(-1, 0, widgetEvent);
      case "arrowdown":
        return this.onArrow(1, 0, widgetEvent);
      case "arrowleft":
        return this.onArrow(0, -1, widgetEvent);
      case "arrowright":
        return this.onArrow(0, 1, widgetEvent);
      case "submit":
        return this.onSubmit(widgetEvent);
    }
  }
  onArrow(seriesIndexDelta, datumIndexDelta, event) {
    if (!this.isState(34 /* Focusable */))
      return;
    this.hoverDevice = "keyboard";
    this.previousInputDevice = "keyboard";
    this.focusIndicator.overrideFocusVisible(true);
    this.focus.seriesIndex += seriesIndexDelta;
    this.focus.datumIndex += datumIndexDelta;
    this.handleFocus(seriesIndexDelta, datumIndexDelta);
    event.sourceEvent.preventDefault();
    this.chart.ctx.chartEventManager.seriesEvent("series-focus-change");
  }
  onSubmit(event) {
    if (!this.isState(34 /* Focusable */))
      return;
    const { series, datum } = this.focus;
    const sourceEvent = event.sourceEvent;
    if (series !== void 0 && datum !== void 0) {
      series.fireNodeClickEvent(sourceEvent, datum);
    } else {
      this.chart.fireEvent({
        type: "click",
        event: sourceEvent
      });
    }
    sourceEvent.preventDefault();
  }
  checkSeriesNodeClick(event) {
    const result = this.pickNode({ x: event.currentX, y: event.currentY }, "event");
    if (result == null)
      return false;
    if (event.type === "click") {
      result.series.fireNodeClickEvent(event.sourceEvent, result.datum);
      return true;
    }
    if (event.type === "dblclick") {
      event.preventZoomDblClick = result.distance === 0;
      result.series.fireNodeDoubleClickEvent(event.sourceEvent, result.datum);
      return true;
    }
    return false;
  }
  handleFocus(seriesIndexDelta, datumIndexDelta) {
    const overlayFocus = this.chart.overlays.getFocusInfo(this.chart.ctx.localeManager);
    if (overlayFocus == null) {
      this.handleSeriesFocus(seriesIndexDelta, datumIndexDelta);
    } else {
      this.focusIndicator.update(overlayFocus.rect, this.seriesRect, false);
    }
  }
  handleSeriesFocus(otherIndexDelta, datumIndexDelta, refresh = false) {
    if (this.chart.chartType === "hierarchy" || this.chart.chartType === "gauge") {
      this.handleSoloSeriesFocus(otherIndexDelta, datumIndexDelta, refresh);
      return;
    }
    const { focus, seriesRect } = this;
    const visibleSeries = focus.sortedSeries.filter((s) => s.visible && s.focusable);
    if (visibleSeries.length === 0)
      return;
    const oldPick = {
      datumIndex: focus.datumIndex - datumIndexDelta,
      otherIndex: focus.seriesIndex - otherIndexDelta
    };
    focus.seriesIndex = clamp(0, focus.seriesIndex, visibleSeries.length - 1);
    focus.series = visibleSeries[focus.seriesIndex];
    const { datumIndex, seriesIndex: otherIndex } = focus;
    const pick = focus.series.pickFocus({ datumIndex, datumIndexDelta, otherIndex, otherIndexDelta, seriesRect });
    this.updatePickedFocus(otherIndexDelta, datumIndexDelta, oldPick, pick, refresh);
  }
  handleSoloSeriesFocus(otherIndexDelta, datumIndexDelta, refresh) {
    this.focus.series = this.focus.sortedSeries[0];
    const {
      focus: { series, seriesIndex: otherIndex, datumIndex },
      seriesRect
    } = this;
    if (series == null)
      return;
    const oldPick = {
      datumIndex: this.focus.datumIndex - datumIndexDelta,
      otherIndex: this.focus.seriesIndex - otherIndexDelta
    };
    const pick = series.pickFocus({ datumIndex, datumIndexDelta, otherIndex, otherIndexDelta, seriesRect });
    this.updatePickedFocus(otherIndexDelta, datumIndexDelta, oldPick, pick, refresh);
  }
  updatePickedFocus(otherIndexDelta, datumIndexDelta, oldPick, pick, refresh) {
    const { focus, hoverRect } = this;
    if (pick === void 0 || focus.series === void 0 || hoverRect === void 0)
      return;
    const { datum, datumIndex, otherIndex } = pick;
    if (otherIndex !== void 0) {
      focus.seriesIndex = otherIndex;
    }
    focus.datumIndex = datumIndex;
    focus.datum = datum;
    if (this.focusIndicator.isFocusVisible()) {
      this.chart.ctx.animationManager.reset();
    }
    if (this.focusIndicator.isFocusVisible()) {
      const focusBBox = getPickedFocusBBox(pick);
      const { x, y } = focusBBox.computeCenter();
      if (!hoverRect.containsPoint(x, y)) {
        const panSuccess = this.chart.ctx.zoomManager.panToBBox(this.id, hoverRect, focusBBox);
        if (panSuccess) {
          return;
        }
      }
    }
    this.focusIndicator.update(pick.bounds, this.seriesRect, pick.clipFocusBox);
    const keyboardEvent = makeKeyboardPointerEvent(focus.series, hoverRect, pick);
    if (keyboardEvent !== void 0 && this.hoverDevice === "keyboard") {
      this.tooltip.lastHover = void 0;
      this.highlight.appliedHoverEvent = void 0;
      this.highlight.pendingHoverEvent = void 0;
      this.highlight.stashedHoverEvent = void 0;
      const tooltipContent = focus.series.getTooltipContent(datum);
      const meta = TooltipManager.makeTooltipMeta(keyboardEvent, focus.series, datum);
      this.chart.ctx.highlightManager.updateHighlight(this.id, datum);
      const tooltipEnabled = this.chart.tooltip.enabled && focus.series.tooltipEnabled;
      if (tooltipEnabled) {
        this.chart.ctx.tooltipManager.updateTooltip(this.id, meta, tooltipContent);
      } else {
        this.chart.ctx.tooltipManager.removeTooltip(this.id);
      }
      if (!refresh) {
        const shouldAnnouncePick = datumIndexDelta === 0 && otherIndexDelta === 0 || oldPick.datumIndex !== pick.datumIndex || oldPick.otherIndex !== (pick.otherIndex ?? focus.seriesIndex);
        if (shouldAnnouncePick) {
          this.swapChain.update(this.getDatumAriaText(datum, tooltipContent));
        }
      }
    }
  }
  getDatumAriaText(datum, tooltipContent) {
    const description = tooltipContent == null ? "" : tooltipContentAriaLabel(tooltipContent);
    return this.chart.ctx.localeManager.t("ariaAnnounceHoverDatum", {
      datum: datum.series.getDatumAriaText?.(datum, description) ?? description
    });
  }
  clearHighlight() {
    this.highlight.pendingHoverEvent = void 0;
    this.highlight.appliedHoverEvent = void 0;
    this.chart.ctx.highlightManager.updateHighlight(this.id);
  }
  clearTooltip() {
    this.chart.ctx.tooltipManager.removeTooltip(this.id);
    this.tooltip.lastHover = void 0;
  }
  clearAll() {
    this.clearHighlight();
    this.clearTooltip();
    this.focusIndicator.clear();
  }
  handleHoverHighlight(redisplay) {
    this.highlight.appliedHoverEvent = this.highlight.pendingHoverEvent;
    this.highlight.pendingHoverEvent = void 0;
    const event = this.highlight.appliedHoverEvent;
    if (!event || !this.isState(41 /* Clickable */))
      return;
    const { currentX, currentY } = event;
    const canvasX = event.currentX + (this.hoverRect?.x ?? 0);
    const canvasY = event.currentY + (this.hoverRect?.y ?? 0);
    if (redisplay ? this.chart.ctx.animationManager.isActive() : !this.hoverRect?.containsPoint(canvasX, canvasY)) {
      this.clearHighlight();
      return;
    }
    const { range: range2 } = this.chart.highlight;
    const intent = range2 === "tooltip" ? "highlight-tooltip" : "highlight";
    const found = this.pickNode({ x: currentX, y: currentY }, intent);
    if (found) {
      this.chart.ctx.highlightManager.updateHighlight(this.id, found.datum);
      this.hoverDevice = "pointer";
      return;
    }
    this.chart.ctx.highlightManager.updateHighlight(this.id);
  }
  handleHoverTooltip(event, redisplay) {
    if (!this.isState(41 /* Clickable */))
      return;
    const { currentX, currentY } = event;
    const canvasX = currentX + (this.hoverRect?.x ?? 0);
    const canvasY = currentY + (this.hoverRect?.y ?? 0);
    const targetElement = event.sourceEvent.target;
    if (redisplay ? this.chart.ctx.animationManager.isActive() : !this.hoverRect?.containsPoint(canvasX, canvasY)) {
      if (this.hoverDevice == "pointer")
        this.clearTooltip();
      return;
    }
    if (targetElement && this.chart.tooltip.interactive && this.chart.ctx.domManager.isManagedChildDOMElement(targetElement, "canvas-overlay", DEFAULT_TOOLTIP_CLASS)) {
      return;
    }
    const pick = this.pickNode({ x: event.currentX, y: event.currentY }, "tooltip");
    if (!pick) {
      if (this.hoverDevice == "pointer")
        this.clearTooltip();
      return;
    }
    this.hoverDevice = "pointer";
    const content = pick.series.getTooltipContent(pick.datum);
    const tooltipEnabled = this.chart.tooltip.enabled && pick.series.tooltipEnabled;
    const shouldUpdateTooltip = tooltipEnabled && content != null;
    if (shouldUpdateTooltip) {
      const meta = TooltipManager.makeTooltipMeta(
        { type: "pointermove", canvasX, canvasY },
        pick.series,
        pick.datum
      );
      this.chart.ctx.tooltipManager.updateTooltip(this.id, meta, content);
    } else {
      this.chart.ctx.tooltipManager.removeTooltip(this.id);
    }
  }
  changeHighlightDatum(event) {
    const seriesToUpdate = /* @__PURE__ */ new Set();
    const { series: newSeries = void 0, datum: newDatum } = event.currentHighlight ?? {};
    const { series: lastSeries = void 0, datum: lastDatum } = event.previousHighlight ?? {};
    if (lastSeries) {
      seriesToUpdate.add(lastSeries);
    }
    if (newSeries) {
      seriesToUpdate.add(newSeries);
    }
    if (lastSeries?.properties.cursor && lastDatum) {
      this.chart.ctx.domManager.updateCursor(lastSeries.id);
    }
    if (newSeries?.properties.cursor && newSeries?.properties.cursor !== "default" && newDatum) {
      this.chart.ctx.domManager.updateCursor(newSeries.id, newSeries.properties.cursor);
    }
    const updateAll = newSeries == null || lastSeries == null;
    if (updateAll) {
      this.update(4 /* SERIES_UPDATE */);
    } else {
      this.update(4 /* SERIES_UPDATE */, { seriesToUpdate });
    }
  }
  pickNode(point, intent, exactMatchOnly) {
    const reverseSeries = [...this.series].reverse();
    let result;
    for (const series of reverseSeries) {
      if (!series.visible || !series.contentGroup.visible) {
        continue;
      }
      const { match, distance } = series.pickNode(point, intent, exactMatchOnly) ?? {};
      if (!match || distance == null) {
        continue;
      }
      if (!result || result.distance > distance) {
        result = { series, distance, datum: match };
      }
      if (distance === 0) {
        break;
      }
    }
    return result;
  }
};
function excludesType(obj, excluded) {
  return obj.type !== excluded;
}

// packages/ag-charts-community/src/chart/series/seriesLayerManager.ts
var SERIES_THRESHOLD_FOR_AGGRESSIVE_LAYER_REDUCTION = 30;
var SeriesLayerManager = class {
  constructor(seriesRoot) {
    this.seriesRoot = seriesRoot;
    this.groups = /* @__PURE__ */ new Map();
    this.series = /* @__PURE__ */ new Map();
    this.expectedSeriesCount = 1;
    this.mode = "normal";
  }
  setSeriesCount(count) {
    this.expectedSeriesCount = count;
  }
  requestGroup(seriesConfig) {
    const { internalId, type, contentGroup: seriesContentGroup, seriesGrouping } = seriesConfig;
    const { groupIndex = internalId } = seriesGrouping ?? {};
    const seriesInfo = this.series.get(internalId);
    if (seriesInfo != null) {
      throw new Error(`AG Charts - series already has an allocated layer: ${JSON.stringify(seriesInfo)}`);
    }
    if (this.series.size === 0) {
      this.mode = this.expectedSeriesCount >= SERIES_THRESHOLD_FOR_AGGRESSIVE_LAYER_REDUCTION ? "aggressive-grouping" : "normal";
    }
    let group = this.groups.get(type);
    if (group == null) {
      group = /* @__PURE__ */ new Map();
      this.groups.set(type, group);
    }
    const lookupIndex = this.lookupIdx(groupIndex);
    let groupInfo = group.get(lookupIndex);
    if (groupInfo == null) {
      groupInfo = {
        type,
        id: lookupIndex,
        seriesIds: [],
        group: this.seriesRoot.appendChild(
          new Group({
            name: `${seriesConfig.contentGroup.name ?? type}-managed-layer`,
            zIndex: seriesConfig.contentGroup.zIndex,
            // Set in updateLayerCompositing
            renderToOffscreenCanvas: false
          })
        )
      };
      group.set(lookupIndex, groupInfo);
    }
    this.series.set(internalId, { layerState: groupInfo, seriesConfig });
    groupInfo.seriesIds.push(internalId);
    groupInfo.group.appendChild(seriesContentGroup);
    return groupInfo.group;
  }
  changeGroup(seriesConfig) {
    const { internalId, seriesGrouping, type, contentGroup, oldGrouping } = seriesConfig;
    const { groupIndex = internalId } = seriesGrouping ?? {};
    if (this.groups.get(type)?.get(groupIndex)?.seriesIds.includes(internalId)) {
      return;
    }
    if (this.series.has(internalId)) {
      this.releaseGroup({
        internalId,
        seriesGrouping: oldGrouping,
        type,
        contentGroup
      });
    }
    this.requestGroup(seriesConfig);
  }
  releaseGroup(seriesConfig) {
    const { internalId, contentGroup, type } = seriesConfig;
    if (!this.series.has(internalId)) {
      throw new Error(`AG Charts - series doesn't have an allocated layer: ${internalId}`);
    }
    const groupInfo = this.series.get(internalId)?.layerState;
    if (groupInfo) {
      groupInfo.seriesIds = groupInfo.seriesIds.filter((v) => v !== internalId);
      groupInfo.group.removeChild(contentGroup);
    }
    if (groupInfo?.seriesIds.length === 0) {
      this.seriesRoot.removeChild(groupInfo.group);
      this.groups.get(groupInfo.type)?.delete(groupInfo.id);
      this.groups.get(type)?.delete(internalId);
    } else if (groupInfo != null && groupInfo.seriesIds.length > 0) {
      groupInfo.group.zIndex = this.getLowestSeriesZIndex(groupInfo.seriesIds);
    }
    this.series.delete(internalId);
  }
  updateLayerCompositing() {
    this.groups.forEach((groups) => {
      groups.forEach((groupInfo) => {
        const { group, seriesIds } = groupInfo;
        let renderToOffscreenCanvas;
        if (seriesIds.length === 0) {
          renderToOffscreenCanvas = false;
        } else if (seriesIds.length > 1) {
          renderToOffscreenCanvas = true;
        } else {
          const series = this.series.get(seriesIds[0]);
          renderToOffscreenCanvas = series?.seriesConfig.renderToOffscreenCanvas() === true;
        }
        group.renderToOffscreenCanvas = renderToOffscreenCanvas;
        group.zIndex = this.getLowestSeriesZIndex(seriesIds);
      });
    });
  }
  lookupIdx(groupIndex) {
    if (this.mode === "normal") {
      return groupIndex;
    }
    if (typeof groupIndex === "string") {
      groupIndex = Number(groupIndex.split("-").at(-1));
      if (!groupIndex) {
        return 0;
      }
    }
    return Math.floor(
      clamp(0, groupIndex / this.expectedSeriesCount, 1) * SERIES_THRESHOLD_FOR_AGGRESSIVE_LAYER_REDUCTION
    );
  }
  destroy() {
    this.groups.forEach((groups) => {
      groups.forEach((groupInfo) => {
        this.seriesRoot.removeChild(groupInfo.group);
      });
    });
    this.groups.clear();
    this.series.clear();
  }
  getLowestSeriesZIndex(seriesIds) {
    const lowestSeriesZIndex = seriesIds.reduce((currentLowest, seriesId) => {
      const series = this.series.get(seriesId);
      const zIndex = series?.seriesConfig.contentGroup.zIndex;
      if (currentLowest == null || zIndex == null)
        return zIndex;
      return compareZIndex(currentLowest, zIndex) <= 0 ? currentLowest : zIndex;
    }, void 0);
    return lowestSeriesZIndex ?? 1 /* ANY_CONTENT */;
  }
};

// packages/ag-charts-community/src/chart/touch.ts
var Touch = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.dragAction = "drag";
  }
};
__decorateClass([
  Validate(UNION(["none", "drag", "hover"]))
], Touch.prototype, "dragAction", 2);

// packages/ag-charts-community/src/chart/update/dataWindowProcessor.ts
var DataWindowProcessor = class {
  constructor(chart, dataService, updateService, zoomManager, animationManager) {
    this.chart = chart;
    this.dataService = dataService;
    this.updateService = updateService;
    this.zoomManager = zoomManager;
    this.animationManager = animationManager;
    this.dirtyZoom = false;
    this.dirtyDataSource = false;
    this.lastAxisZooms = /* @__PURE__ */ new Map();
    this.destroyFns = [];
    this.destroyFns.push(
      this.dataService.addListener("data-source-change", () => this.onDataSourceChange()),
      this.dataService.addListener("data-load", () => this.onDataLoad()),
      this.dataService.addListener("data-error", () => this.onDataError()),
      this.updateService.addListener("update-complete", () => this.onUpdateComplete()),
      this.zoomManager.addListener("zoom-change", () => this.onZoomChange())
    );
  }
  destroy() {
    this.destroyFns.forEach((cb) => cb());
  }
  onDataLoad() {
    this.animationManager.skip();
    this.updateService.update(1 /* UPDATE_DATA */);
  }
  onDataError() {
    this.updateService.update(3 /* PERFORM_LAYOUT */);
  }
  onDataSourceChange() {
    this.dirtyDataSource = true;
  }
  onUpdateComplete() {
    if (!this.dirtyZoom && !this.dirtyDataSource)
      return;
    this.updateWindow();
  }
  onZoomChange() {
    this.dirtyZoom = true;
  }
  updateWindow() {
    if (!this.dataService.isLazy())
      return;
    const axis = this.getValidAxis();
    let window2;
    let shouldRefresh = true;
    if (axis) {
      const zoom = this.zoomManager.getAxisZoom(axis.id);
      window2 = this.getAxisWindow(axis, zoom);
      shouldRefresh = this.shouldRefresh(axis, zoom);
    }
    this.dirtyZoom = false;
    this.dirtyDataSource = false;
    if (!shouldRefresh)
      return;
    this.dataService.load({ windowStart: window2?.min, windowEnd: window2?.max });
  }
  getValidAxis() {
    return this.chart.axes.find((axis) => axis.type === "time");
  }
  shouldRefresh(axis, zoom) {
    if (this.dirtyDataSource)
      return true;
    if (!this.dirtyZoom)
      return false;
    const lastZoom = this.lastAxisZooms.get(axis.id);
    if (lastZoom && zoom.min === lastZoom.min && zoom.max === lastZoom.max) {
      return false;
    }
    this.lastAxisZooms.set(axis.id, zoom);
    return true;
  }
  getAxisWindow(axis, zoom) {
    const { domain } = axis.scale;
    if (!zoom || domain.length === 0 || isNaN(Number(domain[0])))
      return;
    const diff = Number(domain[1]) - Number(domain[0]);
    const min = new Date(Number(domain[0]) + diff * zoom.min);
    const max = new Date(Number(domain[0]) + diff * zoom.max);
    return { min, max };
  }
};

// packages/ag-charts-community/src/util/browser.ts
var isSafariRegexp = /^((?!chrome|android).)*safari/i;
var safariVersionRegexp = /Version\/(\d+(\.\d+)?)/;
var isChromeRegexp = /Chrome/;
var chromeVersionRegexp = /Chrome\/(\d+)/;
var isEdge = /Edg/;
var isOpera = /OPR/;
function isUnsupportedBrowser() {
  const { userAgent } = getWindow("navigator");
  if (isSafariRegexp.test(userAgent)) {
    const version = parseFloat(safariVersionRegexp.exec(userAgent)?.[1] ?? "0");
    const supported = Math.floor(version) > 16;
    if (!supported) {
      logger_exports.warnOnce(`Unsupported Safari version: ${version}; ${userAgent}`);
    }
    return !supported;
  } else if (isChromeRegexp.test(userAgent) && !isEdge.test(userAgent) && !isOpera.test(userAgent)) {
    const version = parseInt(chromeVersionRegexp.exec(userAgent)?.[1] ?? "0", 10);
    const supported = version > 126;
    if (!supported) {
      logger_exports.warnOnce(`Unsupported Chrome version: ${version}; ${userAgent}`);
    }
    return !supported;
  }
  return false;
}

// packages/ag-charts-community/src/chart/update/overlaysProcessor.ts
var visibleIgnoredSeries = /* @__PURE__ */ new Set(["map-shape-background", "map-line-background"]);
var OverlaysProcessor = class {
  constructor(chartLike, overlays, dataService, layoutManager, localeManager, animationManager, domManager) {
    this.chartLike = chartLike;
    this.overlays = overlays;
    this.dataService = dataService;
    this.layoutManager = layoutManager;
    this.localeManager = localeManager;
    this.animationManager = animationManager;
    this.domManager = domManager;
    this.destroyFns = [];
    this.overlayElem = this.domManager.addChild("canvas-overlay", "overlay");
    this.overlayElem.role = "status";
    this.overlayElem.ariaAtomic = "false";
    this.overlayElem.ariaLive = "polite";
    this.overlayElem.classList.toggle(DEFAULT_OVERLAY_CLASS);
    this.destroyFns.push(this.layoutManager.addListener("layout:complete", (e) => this.onLayoutComplete(e)));
  }
  destroy() {
    this.destroyFns.forEach((cb) => cb());
    this.domManager.removeStyles("overlays");
    this.domManager.removeChild("canvas-overlay", "overlay");
  }
  onLayoutComplete({ series: { rect } }) {
    const isLoading = this.dataService.isLoading();
    const hasData = this.chartLike.series.some((s) => s.hasData);
    const anySeriesVisible = this.chartLike.series.some((s) => s.visible && !visibleIgnoredSeries.has(s.type));
    if (this.overlays.darkTheme) {
      this.overlayElem.classList.add(DEFAULT_OVERLAY_DARK_CLASS);
    } else {
      this.overlayElem.classList.remove(DEFAULT_OVERLAY_DARK_CLASS);
    }
    this.overlayElem.style.left = `${rect.x}px`;
    this.overlayElem.style.top = `${rect.y}px`;
    this.overlayElem.style.width = `${rect.width}px`;
    this.overlayElem.style.height = `${rect.height}px`;
    const loadingShown = isLoading;
    const noDataShown = !isLoading && !hasData;
    const noVisibleSeriesShown = hasData && !anySeriesVisible;
    const unsupportedBrowser = this.overlays.unsupportedBrowser.enabled && isUnsupportedBrowser();
    if (loadingShown) {
      this.showOverlay(this.overlays.loading, rect);
    } else {
      this.hideOverlay(this.overlays.loading);
    }
    if (noDataShown) {
      this.showOverlay(this.overlays.noData, rect);
    } else {
      this.hideOverlay(this.overlays.noData);
    }
    if (noVisibleSeriesShown) {
      this.showOverlay(this.overlays.noVisibleSeries, rect);
    } else {
      this.hideOverlay(this.overlays.noVisibleSeries);
    }
    if (unsupportedBrowser) {
      this.showOverlay(this.overlays.unsupportedBrowser, rect);
    } else {
      this.hideOverlay(this.overlays.unsupportedBrowser);
    }
    const shown = loadingShown || noDataShown || noVisibleSeriesShown || unsupportedBrowser;
    setAttribute(this.overlayElem, "aria-hidden", !shown);
  }
  showOverlay(overlay, seriesRect) {
    if (!overlay.enabled)
      return;
    const element2 = overlay.getElement(this.animationManager, this.localeManager, seriesRect);
    this.overlayElem.appendChild(element2);
  }
  hideOverlay(overlay) {
    overlay.removeElement(() => {
      this.overlayElem.innerText = "\xA0";
    }, this.animationManager);
  }
};

// packages/ag-charts-community/src/chart/chart.ts
var debug = Debug.create(true, "opts");
var SeriesArea = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.padding = new Padding(0);
  }
};
__decorateClass([
  Validate(BOOLEAN, { optional: true })
], SeriesArea.prototype, "clip", 2);
__decorateClass([
  Validate(OBJECT)
], SeriesArea.prototype, "padding", 2);
var _Chart = class _Chart extends Observable {
  constructor(options, resources) {
    var _a;
    super();
    this.id = createId(this);
    this.seriesRoot = new TranslatableGroup({
      name: `${this.id}-series-root`,
      zIndex: 5 /* SERIES_LAYER */
    });
    this.annotationRoot = new TranslatableGroup({
      name: `${this.id}-annotation-root`,
      zIndex: 9 /* SERIES_ANNOTATION */
    });
    this.titleGroup = new Group({
      name: "titles",
      zIndex: 13 /* SERIES_LABEL */
    });
    this.debug = Debug.create();
    this.extraDebugStats = {};
    this.data = [];
    this._firstAutoSize = true;
    this._autoSizeNotify = new AsyncAwaitQueue();
    this.padding = new Padding(20);
    this.seriesArea = new SeriesArea();
    this.keyboard = new Keyboard();
    this.touch = new Touch();
    this.mode = "standalone";
    this.chartCaptions = new ChartCaptions();
    this.suppressFieldDotNotation = false;
    this.destroyed = false;
    this._destroyFns = [];
    // Used to prevent infinite update loops when syncing charts.
    this.skipSync = false;
    this.chartAnimationPhase = "initial";
    this.modulesManager = new ModulesManager();
    this.processors = [];
    this.queuedUserOptions = [];
    this.queuedChartOptions = [];
    this.firstApply = true;
    this._pendingFactoryUpdatesCount = 0;
    this._performUpdateSkipAnimations = false;
    this._performUpdateNotify = new AsyncAwaitQueue();
    this.performUpdateType = 7 /* NONE */;
    this.runningUpdateType = 7 /* NONE */;
    this.updateShortcutCount = 0;
    this.seriesToUpdate = /* @__PURE__ */ new Set();
    this.updateMutex = new Mutex();
    this.updateRequestors = {};
    this.performUpdateTrigger = debouncedCallback(({ count }) => {
      if (this.destroyed)
        return;
      this.updateMutex.acquire(async () => {
        try {
          await this.performUpdate(count);
        } catch (error2) {
          logger_exports.error("update error", error2);
        }
      }).catch((e) => logger_exports.errorOnce(e));
    });
    this._performUpdateSplits = {};
    this.axes = [];
    this.series = [];
    this._cachedData = void 0;
    this.onSeriesNodeClick = (event) => {
      this.fireEvent({ ...event, type: "seriesNodeClick" });
    };
    this.onSeriesNodeDoubleClick = (event) => {
      this.fireEvent({ ...event, type: "seriesNodeDoubleClick" });
    };
    this.onSeriesVisibilityChange = (event) => {
      this.fireEvent(event);
    };
    this.seriesGroupingChanged = (event) => {
      if (!(event instanceof SeriesGroupingChangedEvent))
        return;
      const { series, seriesGrouping, oldGrouping } = event;
      if (series.contentGroup.isRoot())
        return;
      this.seriesLayerManager.changeGroup({
        internalId: series.internalId,
        type: series.type,
        contentGroup: series.contentGroup,
        renderToOffscreenCanvas: () => series.renderToOffscreenCanvas(),
        seriesGrouping,
        oldGrouping
      });
    };
    this.chartOptions = options;
    const scene = resources?.scene;
    const container = resources?.container ?? options.processedOptions.container ?? void 0;
    const styleContainer = resources?.styleContainer ?? options.specialOverrides.styleContainer;
    if (scene) {
      this._firstAutoSize = false;
      this._lastAutoSize = [scene.width, scene.height, scene.pixelRatio];
    }
    const root = new Group({ name: "root" });
    root.visible = false;
    root.append(this.seriesRoot);
    root.append(this.annotationRoot);
    root.append(this.titleGroup);
    this.titleGroup.append(this.title.node);
    this.titleGroup.append(this.subtitle.node);
    this.titleGroup.append(this.footnote.node);
    this.tooltip = new Tooltip();
    this.seriesLayerManager = new SeriesLayerManager(this.seriesRoot);
    this.mode = options.userOptions.mode ?? this.mode;
    const ctx = this.ctx = new ChartContext(this, {
      chartType: this.getChartType(),
      scene,
      root,
      container,
      styleContainer,
      syncManager: new SyncManager(this),
      fireEvent: (event) => this.fireEvent(event),
      updateCallback: (type, opts) => this.update(type, opts),
      updateMutex: this.updateMutex
    });
    this._destroyFns.push(
      ctx.domManager.addListener("resize", () => this.parentResize(ctx.domManager.containerSize))
    );
    this.overlays = new ChartOverlays();
    (_a = this.overlays.loading).renderer ?? (_a.renderer = () => getLoadingSpinner(this.overlays.loading.getText(ctx.localeManager), ctx.animationManager.defaultDuration));
    this.processors = [
      new DataWindowProcessor(this, ctx.dataService, ctx.updateService, ctx.zoomManager, ctx.animationManager),
      new OverlaysProcessor(
        this,
        this.overlays,
        ctx.dataService,
        ctx.layoutManager,
        ctx.localeManager,
        ctx.animationManager,
        ctx.domManager
      )
    ];
    this.highlight = new ChartHighlight();
    this.container = container;
    const moduleContext = this.getModuleContext();
    ctx.domManager.setDataBoolean("animating", false);
    this.seriesAreaManager = new SeriesAreaManager(this.initSeriesAreaDependencies());
    this._destroyFns.push(
      ctx.layoutManager.registerElement(0 /* Caption */, (e) => {
        e.layoutBox.shrink(this.padding.toJson());
        this.chartCaptions.positionCaptions(e);
      }),
      ctx.layoutManager.addListener("layout:complete", (e) => this.chartCaptions.positionAbsoluteCaptions(e)),
      ctx.dataService.addListener("data-load", (event) => {
        this.data = event.data;
      }),
      this.title.registerInteraction(moduleContext, "beforebegin"),
      this.subtitle.registerInteraction(moduleContext, "beforebegin"),
      this.footnote.registerInteraction(moduleContext, "afterend"),
      Widget.addWindowEvent("page-left", () => this.destroy()),
      ctx.animationManager.addListener("animation-frame", () => {
        this.update(6 /* SCENE_RENDER */);
      }),
      ctx.animationManager.addListener("animation-start", () => ctx.domManager.setDataBoolean("animating", true)),
      ctx.animationManager.addListener("animation-stop", () => ctx.domManager.setDataBoolean("animating", false)),
      ctx.zoomManager.addListener("zoom-change", () => {
        this.series.forEach((s) => s.animationState?.transition("updateData"));
        const skipAnimations = this.chartAnimationPhase !== "initial";
        this.update(3 /* PERFORM_LAYOUT */, { forceNodeDataRefresh: true, skipAnimations });
      })
    );
    this.parentResize(ctx.domManager.containerSize);
  }
  static getInstance(element2) {
    return _Chart.chartsInstances.get(element2);
  }
  /** NOTE: This is exposed for use by Integrated charts only. */
  get canvasElement() {
    return this.ctx.scene.canvas.element;
  }
  download(fileName, fileFormat) {
    this.ctx.scene.download(fileName, fileFormat);
  }
  getCanvasDataURL(fileFormat) {
    return this.ctx.scene.getDataURL(fileFormat);
  }
  toSVG() {
    return this.ctx.scene.toSVG();
  }
  getOptions() {
    return this.queuedUserOptions.at(-1) ?? this.chartOptions.userOptions;
  }
  getChartOptions() {
    return this.queuedChartOptions.at(-1) ?? this.chartOptions;
  }
  overrideFocusVisible(visible) {
    this.seriesAreaManager.focusIndicator.overrideFocusVisible(visible);
  }
  initSeriesAreaDependencies() {
    const { ctx, tooltip, highlight, overlays, seriesRoot, mode } = this;
    const chartType = this.getChartType();
    const fireEvent = this.fireEvent.bind(this);
    const getUpdateType = () => this.performUpdateType;
    return { fireEvent, getUpdateType, chartType, ctx, tooltip, highlight, overlays, seriesRoot, mode };
  }
  getModuleContext() {
    return this.ctx;
  }
  getCaptionText() {
    return [this.title, this.subtitle, this.footnote].filter((caption) => caption.enabled && caption.text).map((caption) => caption.text).join(". ");
  }
  getAriaLabel() {
    return this.ctx.localeManager.t("ariaAnnounceChart", { seriesCount: this.series.length });
  }
  resetAnimations() {
    this.chartAnimationPhase = "initial";
    for (const series of this.series) {
      series.resetAnimation(this.chartAnimationPhase);
    }
    for (const axis of this.axes) {
      axis.resetAnimation(this.chartAnimationPhase);
    }
    this.animationRect = void 0;
    this.ctx.animationManager.reset();
  }
  skipAnimations() {
    this.ctx.animationManager.skipCurrentBatch();
    this._performUpdateSkipAnimations = true;
  }
  detachAndClear() {
    this.container = void 0;
    this.ctx.scene.clear();
  }
  destroy(opts) {
    if (this.destroyed) {
      return;
    }
    const keepTransferableResources = opts?.keepTransferableResources;
    let result;
    this.performUpdateType = 7 /* NONE */;
    this._destroyFns.forEach((fn) => fn());
    this.processors.forEach((p) => p.destroy());
    this.tooltip.destroy(this.ctx.domManager);
    this.overlays.destroy();
    this.modulesManager.destroy();
    if (keepTransferableResources) {
      this.ctx.scene.strip();
      result = {
        container: this.container,
        scene: this.ctx.scene
      };
    } else {
      this.ctx.scene.destroy();
      this.container = void 0;
    }
    this.destroySeries(this.series);
    this.seriesLayerManager.destroy();
    this.axes.forEach((a) => a.destroy());
    this.axes = [];
    this.animationRect = void 0;
    this.ctx.destroy();
    this.destroyed = true;
    Object.freeze(this);
    return result;
  }
  requestFactoryUpdate(cb) {
    if (this.destroyed)
      return;
    this._pendingFactoryUpdatesCount++;
    this.updateMutex.acquire(async () => {
      if (this.destroyed)
        return;
      try {
        await cb(this);
      } finally {
        if (!this.destroyed) {
          this._pendingFactoryUpdatesCount--;
        }
      }
    }).catch((e) => logger_exports.errorOnce(e));
  }
  update(type = 0 /* FULL */, opts) {
    if (this.destroyed)
      return;
    const {
      forceNodeDataRefresh = false,
      skipAnimations,
      seriesToUpdate = this.series,
      newAnimationBatch
    } = opts ?? {};
    this.ctx.widgets.seriesWidget.setDragTouchEnabled(this.touch.dragAction !== "none");
    if (forceNodeDataRefresh) {
      this.series.forEach((series) => series.markNodeDataDirty());
    }
    for (const series of seriesToUpdate) {
      this.seriesToUpdate.add(series);
    }
    if (skipAnimations) {
      this.ctx.animationManager.skipCurrentBatch();
      this._performUpdateSkipAnimations = true;
    }
    if (newAnimationBatch && this.ctx.animationManager.isActive()) {
      this._performUpdateSkipAnimations = true;
    }
    this.skipSync = opts?.skipSync ?? false;
    if (this.debug.check()) {
      let stack = new Error().stack ?? "<unknown>";
      stack = stack.replace(/\([^)]*/g, "");
      this.updateRequestors[stack] = type;
    }
    if (type < this.performUpdateType) {
      this.performUpdateType = type;
      this.ctx.domManager.setDataBoolean("updatePending", true);
      this.performUpdateTrigger.schedule(opts?.backOffMs);
    }
  }
  async performUpdate(count) {
    const { performUpdateType, extraDebugStats, _performUpdateSplits: splits, ctx } = this;
    const seriesToUpdate = [...this.seriesToUpdate];
    this.performUpdateType = 7 /* NONE */;
    this.seriesToUpdate.clear();
    this.runningUpdateType = performUpdateType;
    if (this.updateShortcutCount === 0 && performUpdateType < 6 /* SCENE_RENDER */) {
      ctx.animationManager.startBatch(this._performUpdateSkipAnimations);
      ctx.animationManager.onBatchStop(() => this.chartAnimationPhase = "ready");
    }
    this.debug("Chart.performUpdate() - start", ChartUpdateType[performUpdateType]);
    let previousSplit = performance.now();
    splits.start ?? (splits.start = previousSplit);
    const updateSplits = (splitName) => {
      splits[splitName] ?? (splits[splitName] = 0);
      splits[splitName] += performance.now() - previousSplit;
      previousSplit = performance.now();
    };
    switch (performUpdateType) {
      case 0 /* FULL */:
        if (this.checkUpdateShortcut(0 /* FULL */))
          break;
        this.ctx.updateService.dispatchPreDomUpdate();
        this.updateDOM();
      case 1 /* UPDATE_DATA */:
        if (this.checkUpdateShortcut(1 /* UPDATE_DATA */))
          break;
        await this.updateData();
        updateSplits("\u2B07\uFE0F");
      case 2 /* PROCESS_DATA */:
        if (this.checkUpdateShortcut(2 /* PROCESS_DATA */))
          break;
        await this.processData();
        this.seriesAreaManager.dataChanged();
        updateSplits("\u{1F3ED}");
      case 3 /* PERFORM_LAYOUT */:
        await this.checkFirstAutoSize();
        if (this.checkUpdateShortcut(3 /* PERFORM_LAYOUT */))
          break;
        await this.processLayout();
        updateSplits("\u2316");
      case 4 /* SERIES_UPDATE */: {
        if (this.checkUpdateShortcut(4 /* SERIES_UPDATE */))
          break;
        await this.updateSeries(seriesToUpdate);
        updateSplits("\u{1F914}");
        this.updateAriaLabels();
        this.seriesLayerManager.updateLayerCompositing();
      }
      case 5 /* PRE_SCENE_RENDER */:
        if (this.checkUpdateShortcut(5 /* PRE_SCENE_RENDER */))
          break;
        ctx.updateService.dispatchPreSceneRender();
        updateSplits("\u2196");
      case 6 /* SCENE_RENDER */:
        if (this.checkUpdateShortcut(6 /* SCENE_RENDER */))
          break;
        ctx.animationManager.endBatch();
        extraDebugStats["updateShortcutCount"] = this.updateShortcutCount;
        ctx.scene.render({ debugSplitTimes: splits, extraDebugStats, seriesRect: this.seriesRect });
        this.extraDebugStats = {};
        for (const key of Object.keys(splits)) {
          delete splits[key];
        }
        this.ctx.domManager.incrementDataCounter("sceneRenders");
      case 7 /* NONE */:
        this.updateShortcutCount = 0;
        this.updateRequestors = {};
        this._performUpdateSkipAnimations = false;
        ctx.animationManager.endBatch();
    }
    if (!this.destroyed) {
      ctx.updateService.dispatchUpdateComplete();
      this.ctx.domManager.setDataBoolean("updatePending", false);
      this.runningUpdateType = 7 /* NONE */;
    }
    this._performUpdateNotify.notify();
    const end2 = performance.now();
    this.debug("Chart.performUpdate() - end", {
      chart: this,
      durationMs: Math.round((end2 - splits["start"]) * 100) / 100,
      count,
      performUpdateType: ChartUpdateType[performUpdateType]
    });
  }
  updateThemeClassName() {
    const themeClassNamePrefix = "ag-charts-theme-";
    const validThemeClassNames = [`${themeClassNamePrefix}default`, `${themeClassNamePrefix}default-dark`];
    let themeClassName = validThemeClassNames[0];
    let isDark = false;
    let { theme } = this.chartOptions.processedOptions;
    while (typeof theme !== "string" && theme != null) {
      theme = theme.baseTheme;
    }
    if (typeof theme === "string") {
      themeClassName = theme.replace("ag-", themeClassNamePrefix);
      isDark = theme.includes("-dark");
    }
    if (!validThemeClassNames.includes(themeClassName)) {
      themeClassName = isDark ? validThemeClassNames[1] : validThemeClassNames[0];
    }
    this.ctx.domManager.setThemeClass(themeClassName);
  }
  updateDOM() {
    this.updateThemeClassName();
    const { enabled, tabIndex } = this.keyboard;
    this.ctx.domManager.setTabGuardIndex(enabled ? tabIndex ?? 0 : -1);
    this.ctx.domManager.setThemeParameters(this.chartOptions.themeParameters);
  }
  updateAriaLabels() {
    this.ctx.domManager.updateCanvasLabel(this.getAriaLabel());
  }
  checkUpdateShortcut(checkUpdateType) {
    const maxShortcuts = 3;
    if (this.destroyed)
      return true;
    if (this.updateShortcutCount > maxShortcuts) {
      logger_exports.warn(
        `exceeded the maximum number of simultaneous updates (${maxShortcuts + 1}), discarding changes and rendering`,
        this.updateRequestors
      );
      return false;
    }
    if (this.performUpdateType <= checkUpdateType) {
      this.updateShortcutCount++;
      return true;
    }
    return false;
  }
  async checkFirstAutoSize() {
    if (this.width != null && this.height != null) {
    } else if (!this._lastAutoSize) {
      const success = await this._autoSizeNotify.await(500);
      if (!success) {
        this.debug("Chart.checkFirstAutoSize() - timeout for first size update.");
      }
    }
  }
  onAxisChange(newValue, oldValue) {
    if (oldValue == null && newValue.length === 0)
      return;
    this.ctx.axisManager.updateAxes(oldValue ?? [], newValue);
  }
  onSeriesChange(newValue, oldValue) {
    const seriesToDestroy = oldValue?.filter((series) => !newValue.includes(series)) ?? [];
    this.destroySeries(seriesToDestroy);
    this.seriesLayerManager?.setSeriesCount(newValue.length);
    for (const series of newValue) {
      if (oldValue?.includes(series))
        continue;
      const seriesContentNode = this.seriesLayerManager.requestGroup(series);
      series.attachSeries(seriesContentNode, this.seriesRoot, this.annotationRoot);
      const chart = this;
      series.chart = {
        get mode() {
          return chart.mode;
        },
        get isMiniChart() {
          return false;
        },
        get seriesRect() {
          return chart.seriesRect;
        }
      };
      series.resetAnimation(this.chartAnimationPhase);
      this.addSeriesListeners(series);
      series.addChartEventListeners();
    }
    this.seriesAreaManager?.seriesChanged(newValue);
  }
  destroySeries(allSeries) {
    allSeries?.forEach((series) => {
      series.removeEventListener("nodeClick", this.onSeriesNodeClick);
      series.removeEventListener("nodeDoubleClick", this.onSeriesNodeDoubleClick);
      series.removeEventListener("groupingChanged", this.seriesGroupingChanged);
      series.destroy();
      this.seriesLayerManager.releaseGroup(series);
      series.detachSeries(void 0, this.seriesRoot, this.annotationRoot);
      series.chart = void 0;
    });
  }
  addSeriesListeners(series) {
    if (this.hasEventListener("seriesNodeClick")) {
      series.addEventListener("nodeClick", this.onSeriesNodeClick);
    }
    if (this.hasEventListener("seriesNodeDoubleClick")) {
      series.addEventListener("nodeDoubleClick", this.onSeriesNodeDoubleClick);
    }
    if (this.hasEventListener("seriesVisibilityChange")) {
      series.addEventListener("seriesVisibilityChange", this.onSeriesVisibilityChange);
    }
    series.addEventListener("groupingChanged", this.seriesGroupingChanged);
  }
  assignSeriesToAxes() {
    for (const axis of this.axes) {
      axis.boundSeries = this.series.filter((s) => s.axes[axis.direction] === axis);
    }
  }
  assignAxesToSeries() {
    const directionToAxesMap = groupBy(this.axes, (axis) => axis.direction);
    for (const series of this.series) {
      for (const direction of series.directions) {
        const directionAxes = directionToAxesMap[direction];
        if (!directionAxes) {
          logger_exports.warnOnce(
            `no available axis for direction [${direction}]; check series and axes configuration.`
          );
          return;
        }
        const seriesKeys = series.getKeys(direction);
        const newAxis = directionAxes.find(
          (axis) => !axis.keys.length || seriesKeys.some((key) => axis.keys.includes(key))
        );
        if (!newAxis) {
          logger_exports.warnOnce(
            `no matching axis for direction [${direction}] and keys [${seriesKeys}]; check series and axes configuration.`
          );
          return;
        }
        series.axes[direction] = newAxis;
      }
    }
  }
  parentResize(size) {
    if (size == null || this.width != null && this.height != null)
      return;
    let { width: width2, height: height2 } = size;
    const { pixelRatio } = size;
    width2 = Math.floor(width2);
    height2 = Math.floor(height2);
    if (width2 === 0 && height2 === 0)
      return;
    const [autoWidth = 0, autoHeight = 0, autoPixelRatio = 1] = this._lastAutoSize ?? [];
    if (autoWidth === width2 && autoHeight === height2 && autoPixelRatio === pixelRatio)
      return;
    this._lastAutoSize = [width2, height2, pixelRatio];
    this.resize("SizeMonitor", {});
  }
  resize(source, opts) {
    const { scene, animationManager } = this.ctx;
    const { inWidth, inHeight, inMinWidth, inMinHeight, inOverrideDevicePixelRatio } = opts;
    this.ctx.domManager.setSizeOptions(
      inMinWidth ?? this.minWidth,
      inMinHeight ?? this.minHeight,
      inWidth ?? this.width,
      inHeight ?? this.height
    );
    const width2 = inWidth ?? this.width ?? this._lastAutoSize?.[0];
    const height2 = inHeight ?? this.height ?? this._lastAutoSize?.[1];
    const pixelRatio = inOverrideDevicePixelRatio ?? this.overrideDevicePixelRatio ?? this._lastAutoSize?.[2];
    this.debug(`Chart.resize() from ${source}`, { width: width2, height: height2, pixelRatio, stack: new Error().stack });
    if (width2 == null || height2 == null || !isFiniteNumber(width2) || !isFiniteNumber(height2))
      return;
    if (scene.resize(width2, height2, pixelRatio)) {
      animationManager.reset();
      let skipAnimations = true;
      if ((this.width == null || this.height == null) && this._firstAutoSize) {
        skipAnimations = false;
        this._firstAutoSize = false;
      }
      this.update(3 /* PERFORM_LAYOUT */, { forceNodeDataRefresh: true, skipAnimations });
      this._autoSizeNotify.notify();
    }
  }
  async updateData() {
    this.series.forEach((s) => s.setChartData(this.data));
    const modulePromises = this.modulesManager.mapModules((m) => m.updateData?.(this.data));
    await Promise.all(modulePromises);
  }
  async processData() {
    if (this.series.some((s) => s.canHaveAxes)) {
      this.assignAxesToSeries();
      this.assignSeriesToAxes();
    }
    const dataController = new DataController(this.mode, this.suppressFieldDotNotation);
    const seriesPromises = this.series.map((s) => {
      s.resetDatumCallbackCache();
      return s.processData(dataController);
    });
    const modulePromises = this.modulesManager.mapModules((m) => m.processData?.(dataController));
    this._cachedData = dataController.execute(this._cachedData);
    await Promise.all([...seriesPromises, ...modulePromises]);
    for (const axis of this.axes) {
      axis.processData();
    }
    this.updateLegends();
  }
  updateLegends(initialStateLegend) {
    for (const { legend, legendType } of this.modulesManager.legends()) {
      if (legendType === "category") {
        this.setCategoryLegendData(initialStateLegend);
      } else {
        this.setLegendData(legendType, legend);
      }
    }
  }
  setCategoryLegendData(initialState) {
    var _a;
    const {
      ctx: { legendManager, stateManager }
    } = this;
    if (initialState) {
      this.series.forEach((s) => {
        const seriesState = initialState.find((init) => init.seriesId === s.id);
        s.onLegendInitialState("category", seriesState);
      });
    }
    const legendData = this.series.flatMap((s) => {
      const seriesLegendData = s.getLegendData("category");
      legendManager.updateData(s.id, seriesLegendData);
      return seriesLegendData;
    });
    if (initialState) {
      stateManager.setStateAndRestore(legendManager, initialState);
      return;
    }
    if (this.mode !== "integrated") {
      const seriesMarkerFills = {};
      const seriesTypeMap = new Map(this.series.map((s) => [s.id, s.type]));
      for (const {
        seriesId,
        symbol: { marker },
        label
      } of legendData.filter((d) => !d.hideInLegend)) {
        if (marker.fill == null)
          continue;
        const seriesType = seriesTypeMap.get(seriesId);
        const markerFill = seriesMarkerFills[seriesType] ?? (seriesMarkerFills[seriesType] = {});
        markerFill[_a = label.text] ?? (markerFill[_a] = marker.fill);
        if (markerFill[label.text] !== marker.fill) {
          logger_exports.warnOnce(
            `legend item '${label.text}' has multiple fill colors, this may cause unexpected behaviour.`
          );
        }
      }
    }
    legendManager.update();
  }
  setLegendData(legendType, legend) {
    legend.data = this.series.filter((s) => s.properties.showInLegend).flatMap((s) => s.getLegendData(legendType));
  }
  async processLayout() {
    const oldRect = this.animationRect;
    const { width: width2, height: height2 } = this.ctx.scene;
    const ctx = this.ctx.layoutManager.createContext(width2, height2);
    await this.performLayout(ctx);
    if (oldRect && !this.animationRect?.equals(oldRect)) {
      this.ctx.animationManager.skipCurrentBatch();
    }
    this.debug("Chart.performUpdate() - seriesRect", this.seriesRect);
  }
  async updateSeries(seriesToUpdate) {
    const { seriesRect } = this;
    await Promise.all(seriesToUpdate.map((series) => series.update({ seriesRect })));
    this.ctx.seriesLabelLayoutManager.updateLabels(
      this.series.filter((s) => s.visible && s.usesPlacedLabels),
      this.padding,
      this.seriesRect
    );
  }
  async waitForUpdate(timeoutMs = 1e4, failOnTimeout = false) {
    const start2 = performance.now();
    while (this._pendingFactoryUpdatesCount > 0 || this.performUpdateType !== 7 /* NONE */ || this.runningUpdateType !== 7 /* NONE */) {
      if (this.destroyed)
        break;
      if (this._pendingFactoryUpdatesCount > 0) {
        await this.updateMutex.waitForClearAcquireQueue();
      }
      if (this.performUpdateType !== 7 /* NONE */ || this.runningUpdateType !== 7 /* NONE */) {
        await this._performUpdateNotify.await();
      }
      if (performance.now() - start2 > timeoutMs) {
        const message = `Chart.waitForUpdate() timeout of ${timeoutMs} reached - first chart update taking too long.`;
        if (failOnTimeout) {
          throw new Error(message);
        } else {
          logger_exports.warnOnce(message);
        }
      }
      if (isInputPending()) {
        await pause();
      }
    }
  }
  filterMiniChartSeries(series) {
    return series?.filter((s) => s.showInMiniChart !== false);
  }
  applyOptions(newChartOptions) {
    const deltaOptions = this.firstApply ? newChartOptions.processedOptions : newChartOptions.diffOptions(this.chartOptions);
    if (deltaOptions == null || Object.keys(deltaOptions).length === 0)
      return;
    const oldOpts = this.firstApply ? {} : this.chartOptions.processedOptions;
    const newOpts = newChartOptions.processedOptions;
    debug("Chart.applyOptions() - applying delta", deltaOptions);
    const modulesChanged = this.applyModules(newOpts);
    const skip = [
      "type",
      "data",
      "series",
      "listeners",
      "preset",
      "theme",
      "legend.listeners",
      "navigator.miniChart.series",
      "navigator.miniChart.label",
      "locale.localeText",
      "axes",
      "topology",
      "nodes",
      "initialState",
      "styleContainer"
    ];
    if (deltaOptions.listeners) {
      this.registerListeners(this, deltaOptions.listeners);
    }
    jsonApply(this, deltaOptions, { skip });
    let forceNodeDataRefresh = false;
    let seriesStatus = "no-op";
    if (deltaOptions.series != null) {
      seriesStatus = this.applySeries(this, deltaOptions.series, oldOpts?.series);
      forceNodeDataRefresh = true;
    }
    if (seriesStatus === "replaced") {
      this.resetAnimations();
    }
    if (this.applyAxes(this, newOpts, oldOpts, seriesStatus, [])) {
      forceNodeDataRefresh = true;
    }
    if (deltaOptions.data) {
      this.data = deltaOptions.data;
    }
    if (deltaOptions.legend?.listeners && this.modulesManager.isEnabled("legend")) {
      Object.assign(this.legend.listeners, deltaOptions.legend.listeners);
    }
    if (deltaOptions.locale?.localeText) {
      this.modulesManager.getModule("locale").localeText = deltaOptions.locale?.localeText;
    }
    this.chartOptions = newChartOptions;
    const navigatorModule = this.modulesManager.getModule("navigator");
    const zoomModule = this.modulesManager.getModule("zoom");
    if (!navigatorModule?.enabled && !zoomModule?.enabled) {
      this.ctx.zoomManager.updateZoom("chart");
    }
    const miniChart = navigatorModule?.miniChart;
    const miniChartSeries = newOpts.navigator?.miniChart?.series ?? newOpts.series;
    if (miniChart?.enabled === true && miniChartSeries != null) {
      this.applyMiniChartOptions(miniChart, miniChartSeries, newOpts, oldOpts);
    } else if (miniChart?.enabled === false) {
      miniChart.series = [];
      miniChart.axes = [];
    }
    this.ctx.annotationManager.setAnnotationStyles(newChartOptions.annotationThemes);
    forceNodeDataRefresh || (forceNodeDataRefresh = this.shouldForceNodeDataRefresh(deltaOptions, seriesStatus));
    const majorChange = forceNodeDataRefresh || modulesChanged;
    const updateType = majorChange ? 0 /* FULL */ : 3 /* PERFORM_LAYOUT */;
    this.maybeResetAnimations(seriesStatus);
    if (this.shouldClearLegendData(newOpts, oldOpts, seriesStatus)) {
      this.ctx.legendManager.clearData();
    }
    this.applyInitialState(newOpts);
    debug("Chart.applyOptions() - update type", ChartUpdateType[updateType], {
      seriesStatus,
      forceNodeDataRefresh
    });
    this.update(updateType, { forceNodeDataRefresh, newAnimationBatch: true });
    this.firstApply = false;
  }
  applyInitialState(options) {
    const { annotationManager, chartTypeOriginator, historyManager, stateManager, zoomManager } = this.ctx;
    const { initialState } = options;
    if ("annotations" in options && options.annotations?.enabled && initialState?.annotations != null) {
      const annotations = initialState.annotations.map((annotation) => {
        const annotationTheme = annotationManager.getAnnotationTypeStyles(annotation.type);
        return mergeDefaults(annotation, annotationTheme);
      });
      stateManager.setState(annotationManager, annotations);
    }
    if (initialState?.chartType != null) {
      stateManager.setState(chartTypeOriginator, initialState.chartType);
    }
    if ((options.navigator?.enabled || options.zoom?.enabled) && initialState?.zoom != null) {
      stateManager.setState(zoomManager, initialState.zoom);
    }
    if (initialState?.legend != null) {
      this.updateLegends(initialState.legend);
    }
    if (initialState != null) {
      historyManager.clear();
    }
  }
  maybeResetAnimations(seriesStatus) {
    if (this.mode !== "standalone")
      return;
    switch (seriesStatus) {
      case "series-grouping-change":
      case "replaced":
        this.resetAnimations();
        break;
      default:
    }
  }
  shouldForceNodeDataRefresh(deltaOptions, seriesStatus) {
    const seriesDataUpdate = !!deltaOptions.data || seriesStatus === "data-change" || seriesStatus === "replaced";
    const legendKeys = legendRegistry.getKeys();
    const optionsHaveLegend = Object.values(legendKeys).some(
      (legendKey) => deltaOptions[legendKey] != null
    );
    const otherRefreshUpdate = deltaOptions.title != null && deltaOptions.subtitle != null;
    return seriesDataUpdate || optionsHaveLegend || otherRefreshUpdate;
  }
  shouldClearLegendData(options, oldOpts, seriesStatus) {
    const seriesChanged = seriesStatus === "replaced" || seriesStatus === "series-grouping-change" || seriesStatus === "updated";
    const legendRemoved = oldOpts.legend != null && options.legend == null;
    return seriesChanged || legendRemoved;
  }
  applyMiniChartOptions(miniChart, miniChartSeries, completeOptions, oldOpts) {
    const oldSeries = oldOpts?.navigator?.miniChart?.series ?? oldOpts?.series;
    const miniChartSeriesStatus = this.applySeries(
      miniChart,
      this.filterMiniChartSeries(miniChartSeries),
      this.filterMiniChartSeries(oldSeries)
    );
    this.applyAxes(miniChart, completeOptions, oldOpts, miniChartSeriesStatus, [
      "axes[].tick",
      "axes[].thickness",
      "axes[].title",
      "axes[].crosshair",
      "axes[].gridLine",
      "axes[].label"
    ]);
    const series = miniChart.series;
    for (const s of series) {
      s.properties.id = void 0;
    }
    const axes = miniChart.axes;
    const horizontalAxis = axes.find((axis) => axis.direction === "x" /* X */);
    for (const axis of axes) {
      axis.nice = false;
      axis.gridLine.enabled = false;
      axis.label.enabled = axis === horizontalAxis;
      axis.tick.enabled = false;
      axis.interactionEnabled = false;
    }
    if (horizontalAxis != null) {
      const miniChartOpts = completeOptions.navigator?.miniChart;
      const labelOptions = miniChartOpts?.label;
      const intervalOptions = miniChartOpts?.label?.interval;
      horizontalAxis.line.enabled = false;
      horizontalAxis.label.set(
        without(labelOptions, ["interval", "rotation", "minSpacing", "autoRotate", "autoRotateAngle"])
      );
      horizontalAxis.tick.set(
        without(intervalOptions, ["enabled", "width", "size", "color", "interval", "step"])
      );
      if (horizontalAxis.type === "grouped-category") {
        horizontalAxis.label.enabled = false;
        horizontalAxis.label.rotation = 0;
        const { depthOptions } = horizontalAxis;
        if (depthOptions.length === 0) {
          depthOptions.set([{ label: { enabled: true } }]);
        } else {
          for (let i = 1; i < depthOptions.length; i++) {
            depthOptions[i].label.enabled = false;
          }
        }
      }
      const step = intervalOptions?.step;
      if (step != null) {
        horizontalAxis.interval.step = step;
      }
    }
  }
  applyModules(options) {
    const { type: chartType } = this.constructor;
    let modulesChanged = false;
    for (const module of moduleRegistry.byType("root", "legend")) {
      const isConfigured = options[module.optionsKey] != null;
      const shouldBeEnabled = isConfigured && module.chartTypes.includes(chartType);
      if (shouldBeEnabled === this.modulesManager.isEnabled(module))
        continue;
      if (shouldBeEnabled) {
        this.modulesManager.addModule(module, (m) => m.moduleFactory(this.getModuleContext()));
        if (module.type === "legend") {
          this.modulesManager.getModule(module)?.attachLegend(this.ctx.scene);
        }
        this[module.optionsKey] = this.modulesManager.getModule(module);
      } else {
        this.modulesManager.removeModule(module);
        delete this[module.optionsKey];
      }
      modulesChanged = true;
    }
    return modulesChanged;
  }
  initSeriesDeclarationOrder(series) {
    for (let idx = 0; idx < series.length; idx++) {
      series[idx].setSeriesIndex(idx);
    }
  }
  applySeries(chart, optSeries, oldOptSeries) {
    if (!optSeries) {
      return "no-change";
    }
    const matchResult = matchSeriesOptions(chart.series, optSeries, oldOptSeries);
    if (matchResult.status === "no-overlap") {
      debug(`Chart.applySeries() - creating new series instances, status: ${matchResult.status}`, matchResult);
      const chartSeries = optSeries.map((opts) => this.createSeries(opts));
      this.initSeriesDeclarationOrder(chartSeries);
      chart.series = chartSeries;
      return "replaced";
    }
    debug(`Chart.applySeries() - matchResult`, matchResult);
    const seriesInstances = [];
    let dataChanged = false;
    let groupingChanged = false;
    let isUpdated = false;
    const changes = matchResult.changes.toSorted((a, b) => a.targetIdx - b.targetIdx);
    for (const change of changes) {
      groupingChanged || (groupingChanged = change.status === "series-grouping");
      dataChanged || (dataChanged = change.diff?.data != null);
      isUpdated || (isUpdated = change.status !== "no-op");
      switch (change.status) {
        case "add": {
          const newSeries = this.createSeries(change.opts);
          seriesInstances.push(newSeries);
          debug(`Chart.applySeries() - created new series`, newSeries);
          break;
        }
        case "remove":
          debug(`Chart.applySeries() - removing series at previous idx ${change.idx}`, change.series);
          break;
        case "no-op":
          seriesInstances.push(change.series);
          debug(`Chart.applySeries() - no change to series at previous idx ${change.idx}`, change.series);
          break;
        case "series-grouping":
        case "update":
        default: {
          const { series, diff, idx } = change;
          debug(`Chart.applySeries() - applying series diff previous idx ${idx}`, diff, series);
          this.applySeriesValues(series, diff);
          series.markNodeDataDirty();
          seriesInstances.push(series);
        }
      }
    }
    this.initSeriesDeclarationOrder(seriesInstances);
    debug(`Chart.applySeries() - final series instances`, seriesInstances);
    chart.series = seriesInstances;
    if (groupingChanged) {
      return "series-grouping-change";
    }
    if (dataChanged) {
      return "data-change";
    }
    return isUpdated ? "updated" : "no-op";
  }
  applyAxes(chart, options, oldOpts, seriesStatus, skip = []) {
    if (!("axes" in options) || !options.axes) {
      return false;
    }
    skip = ["axes[].type", ...skip];
    const { axes } = options;
    const forceRecreate = seriesStatus === "replaced";
    const matchingTypes = !forceRecreate && chart.axes.length === axes.length && chart.axes.every((a, i) => a.type === axes[i].type);
    if (matchingTypes && isAgCartesianChartOptions(oldOpts)) {
      chart.axes.forEach((axis, index) => {
        const previousOpts = oldOpts.axes?.[index] ?? {};
        const axisDiff = jsonDiff(previousOpts, axes[index]);
        debug(`Chart.applyAxes() - applying axis diff idx ${index}`, axisDiff);
        const path = `axes[${index}]`;
        jsonApply(axis, axisDiff, { path, skip });
      });
      return true;
    }
    debug(`Chart.applyAxes() - creating new axes instances; seriesStatus: ${seriesStatus}`);
    chart.axes = this.createAxis(axes, skip);
    return true;
  }
  createSeries(seriesOptions) {
    const seriesInstance = seriesRegistry.create(seriesOptions.type, this.getModuleContext());
    this.applySeriesOptionModules(seriesInstance, seriesOptions);
    this.applySeriesValues(seriesInstance, seriesOptions);
    return seriesInstance;
  }
  applySeriesOptionModules(series, options) {
    const moduleContext = series.createModuleContext();
    const moduleMap = series.getModuleMap();
    for (const module of moduleRegistry.byType("series-option")) {
      if (module.optionsKey in options && module.seriesTypes.includes(series.type)) {
        moduleMap.addModule(module, (m) => m.moduleFactory(moduleContext));
      }
    }
  }
  applySeriesValues(target, options) {
    const moduleMap = target.getModuleMap();
    const { type: _, data, listeners, seriesGrouping, showInMiniChart: __, ...seriesOptions } = options;
    for (const moduleDef of EXPECTED_ENTERPRISE_MODULES) {
      if (moduleDef.type !== "series-option")
        continue;
      if (moduleDef.optionsKey in seriesOptions) {
        const module = moduleMap.getModule(moduleDef.optionsKey);
        if (module) {
          const moduleOptions = seriesOptions[moduleDef.optionsKey];
          delete seriesOptions[moduleDef.optionsKey];
          module.properties.set(moduleOptions);
        }
      }
    }
    target.properties.set(seriesOptions);
    if ("data" in options) {
      target.setOptionsData(data);
    }
    if (listeners) {
      this.registerListeners(target, listeners);
    }
    if ("seriesGrouping" in options) {
      if (seriesGrouping == null) {
        target.seriesGrouping = void 0;
      } else {
        target.seriesGrouping = { ...target.seriesGrouping, ...seriesGrouping };
      }
    }
  }
  createAxis(options, skip) {
    const newAxes = [];
    const moduleContext = this.getModuleContext();
    for (let index = 0; index < options.length; index++) {
      const axisOptions = options[index];
      const axis = axisRegistry.create(axisOptions.type, moduleContext);
      this.applyAxisModules(axis, axisOptions);
      jsonApply(axis, axisOptions, { path: `axes[${index}]`, skip });
      newAxes.push(axis);
    }
    guessInvalidPositions(newAxes);
    return newAxes;
  }
  applyAxisModules(axis, options) {
    const moduleContext = axis.createModuleContext();
    const moduleMap = axis.getModuleMap();
    for (const module of moduleRegistry.byType("axis-option")) {
      const shouldBeEnabled = options[module.optionsKey] != null;
      if (shouldBeEnabled === moduleMap.isEnabled(module))
        continue;
      if (shouldBeEnabled) {
        moduleMap.addModule(module, (m) => m.moduleFactory(moduleContext));
        axis[module.optionsKey] = moduleMap.getModule(module);
      } else {
        moduleMap.removeModule(module);
        delete axis[module.optionsKey];
      }
    }
  }
  registerListeners(source, listeners) {
    source.clearEventListeners();
    for (const [property, listener] of Object.entries(listeners)) {
      if (isFunction(listener)) {
        source.addEventListener(property, listener);
      }
    }
  }
};
_Chart.chartsInstances = /* @__PURE__ */ new WeakMap();
__decorateClass([
  ActionOnSet({
    newValue(value) {
      if (this.destroyed)
        return;
      this.ctx.domManager.setContainer(value);
      _Chart.chartsInstances.set(value, this);
    },
    oldValue(value) {
      _Chart.chartsInstances.delete(value);
    }
  })
], _Chart.prototype, "container", 2);
__decorateClass([
  ActionOnSet({
    newValue(value) {
      this.resize("width option", { inWidth: value });
    }
  })
], _Chart.prototype, "width", 2);
__decorateClass([
  ActionOnSet({
    newValue(value) {
      this.resize("height option", { inHeight: value });
    }
  })
], _Chart.prototype, "height", 2);
__decorateClass([
  ActionOnSet({
    newValue(value) {
      this.resize("minWidth option", { inMinWidth: value });
    }
  })
], _Chart.prototype, "minWidth", 2);
__decorateClass([
  ActionOnSet({
    newValue(value) {
      this.resize("minHeight option", { inMinHeight: value });
    }
  })
], _Chart.prototype, "minHeight", 2);
__decorateClass([
  ActionOnSet({
    newValue(value) {
      this.resize("overrideDevicePixelRatio option", { inOverrideDevicePixelRatio: value });
    }
  })
], _Chart.prototype, "overrideDevicePixelRatio", 2);
__decorateClass([
  Validate(OBJECT)
], _Chart.prototype, "padding", 2);
__decorateClass([
  Validate(OBJECT)
], _Chart.prototype, "seriesArea", 2);
__decorateClass([
  Validate(OBJECT)
], _Chart.prototype, "keyboard", 2);
__decorateClass([
  Validate(OBJECT)
], _Chart.prototype, "touch", 2);
__decorateClass([
  Validate(UNION(["standalone", "integrated"], "a chart mode"))
], _Chart.prototype, "mode", 2);
__decorateClass([
  ProxyProperty("chartCaptions.title")
], _Chart.prototype, "title", 2);
__decorateClass([
  ProxyProperty("chartCaptions.subtitle")
], _Chart.prototype, "subtitle", 2);
__decorateClass([
  ProxyProperty("chartCaptions.footnote")
], _Chart.prototype, "footnote", 2);
__decorateClass([
  Validate(BOOLEAN)
], _Chart.prototype, "suppressFieldDotNotation", 2);
__decorateClass([
  ActionOnSet({
    changeValue(newValue, oldValue) {
      this.onAxisChange(newValue, oldValue);
    }
  })
], _Chart.prototype, "axes", 2);
__decorateClass([
  ActionOnSet({
    changeValue(newValue, oldValue) {
      this.onSeriesChange(newValue, oldValue);
    }
  })
], _Chart.prototype, "series", 2);
var Chart = _Chart;

// packages/ag-charts-community/src/scale/logScale.ts
var logFunctions = {
  2: (_base, x) => Math.log2(x),
  [Math.E]: (_base, x) => Math.log(x),
  10: (_base, x) => Math.log10(x)
};
var DEFAULT_LOG = (base, x) => Math.log(x) / Math.log(base);
function log2(base, domain, x) {
  const start2 = Math.min(...domain);
  const fn = logFunctions[base] ?? DEFAULT_LOG;
  return start2 >= 0 ? fn(base, x) : -fn(base, -x);
}
var powFunctions = {
  [Math.E]: (_base, x) => Math.exp(x),
  10: (_base, x) => x >= 0 ? 10 ** x : 1 / 10 ** -x
};
var DEFAULT_POW = (base, x) => base ** x;
function pow(base, domain, x) {
  const start2 = Math.min(...domain);
  const fn = powFunctions[base] ?? DEFAULT_POW;
  return start2 >= 0 ? fn(base, x) : -fn(base, -x);
}
var LogScale = class extends ContinuousScale {
  constructor(d = [1, 10], r = [0, 1]) {
    super(d, r);
    this.type = "log";
    // Handling <1 and crossing 0 cases is tricky, easiest solution is to default to clamping.
    this.defaultClamp = true;
    this.base = 10;
    this.log = (x) => log2(this.base, this.domain, x);
    this.pow = (x) => pow(this.base, this.domain, x);
  }
  toDomain(d) {
    return d;
  }
  transform(x) {
    const [min, max] = findMinMax(this.domain);
    if (min >= 0 !== max >= 0)
      return NaN;
    return min >= 0 ? Math.log(x) : -Math.log(-x);
  }
  transformInvert(x) {
    const [min, max] = findMinMax(this.domain);
    if (min >= 0 !== max >= 0)
      return NaN;
    return min >= 0 ? Math.exp(x) : -Math.exp(-x);
  }
  niceDomain(_ticks, domain = this.domain) {
    if (domain.length < 2)
      return [];
    const { base } = this;
    const [d0, d1] = domain;
    const roundStart = d0 > d1 ? Math.ceil : Math.floor;
    const roundStop = d0 > d1 ? Math.floor : Math.ceil;
    const n0 = pow(base, domain, roundStart(log2(base, domain, d0)));
    const n1 = pow(base, domain, roundStop(log2(base, domain, d1)));
    return [n0, n1];
  }
  ticks({ interval, tickCount = ContinuousScale.defaultTickCount }, domain = this.domain, visibleRange) {
    if (!domain || domain.length < 2 || tickCount < 1) {
      return [];
    }
    const base = this.base;
    const [d0, d1] = domain;
    const start2 = Math.min(d0, d1);
    const stop = Math.max(d0, d1);
    let p0 = this.log(start2);
    let p1 = this.log(stop);
    if (interval) {
      const inBounds = (tick) => tick >= start2 && tick <= stop;
      const step = Math.min(Math.abs(interval), Math.abs(p1 - p0));
      const ticks2 = range(p0, p1, step).map(this.pow).filter(inBounds);
      if (!isDenseInterval(ticks2.length, this.getPixelRange())) {
        return ticks2;
      }
    }
    if (!isInteger(base) || p1 - p0 >= tickCount) {
      return createTicks(p0, p1, Math.min(p1 - p0, tickCount)).map(this.pow);
    }
    let ticks = [];
    const isPositive = start2 > 0;
    p0 = Math.floor(p0) - 1;
    p1 = Math.round(p1) + 1;
    const availableSpacing = findRangeExtent(this.range) / tickCount;
    let lastTickPosition = Infinity;
    for (let p = p0; p <= p1; p++) {
      const nextMagnitudeTickPosition = this.convert(this.pow(p + 1));
      for (let k = 1; k < base; k++) {
        const q = isPositive ? k : base - k + 1;
        const t = this.pow(p) * q;
        const tickPosition = this.convert(t);
        const prevSpacing = Math.abs(lastTickPosition - tickPosition);
        const nextSpacing = Math.abs(tickPosition - nextMagnitudeTickPosition);
        const fits = prevSpacing >= availableSpacing && nextSpacing >= availableSpacing;
        if (t >= start2 && t <= stop && (k === 1 || fits || ticks.length === 0)) {
          ticks.push(t);
          lastTickPosition = tickPosition;
        }
      }
    }
    ticks = filterVisibleTicks(ticks, isPositive, visibleRange);
    return ticks;
  }
  tickFormatter({ specifier }) {
    return specifier != null ? numberFormat(specifier) : String;
  }
  datumFormatter(params) {
    return this.tickFormatter(params);
  }
};

// packages/ag-charts-community/src/scene/util/quadtree.ts
var QuadtreeNearest = class {
  constructor(capacity, maxdepth, boundary) {
    this.root = new QuadtreeNodeNearest(capacity, maxdepth, boundary);
  }
  clear(boundary) {
    this.root.clear(boundary);
  }
  addValue(hitTester, value) {
    const elem = {
      hitTester,
      value,
      distanceSquared: (x, y) => {
        return hitTester.distanceSquared(x, y);
      }
    };
    this.root.addElem(elem);
  }
  find(x, y) {
    const arg = { best: { nearest: void 0, distanceSquared: Infinity } };
    this.root.find(x, y, arg);
    return arg.best;
  }
};
var QuadtreeSubdivisions = class {
  constructor(nw, ne, sw, se) {
    this.nw = nw;
    this.ne = ne;
    this.sw = sw;
    this.se = se;
  }
  addElem(elem) {
    this.nw.addElem(elem);
    this.ne.addElem(elem);
    this.sw.addElem(elem);
    this.se.addElem(elem);
  }
  find(x, y, arg) {
    this.nw.find(x, y, arg);
    this.ne.find(x, y, arg);
    this.sw.find(x, y, arg);
    this.se.find(x, y, arg);
  }
};
var QuadtreeNode = class {
  constructor(capacity, maxdepth, boundary) {
    this.capacity = capacity;
    this.maxdepth = maxdepth;
    this.boundary = boundary ?? BBox.NaN;
    this.elems = [];
    this.subdivisions = void 0;
  }
  clear(boundary) {
    this.elems.length = 0;
    this.boundary = boundary;
    this.subdivisions = void 0;
  }
  addElem(e) {
    if (this.addCondition(e)) {
      if (this.subdivisions === void 0) {
        if (this.maxdepth === 0 || this.elems.length < this.capacity) {
          this.elems.push(e);
        } else {
          this.subdivide(e);
        }
      } else {
        this.subdivisions.addElem(e);
      }
    }
  }
  find(x, y, arg) {
    if (this.findCondition(x, y, arg)) {
      if (this.subdivisions === void 0) {
        this.findAction(x, y, arg);
      } else {
        this.subdivisions.find(x, y, arg);
      }
    }
  }
  subdivide(newElem) {
    this.subdivisions = this.makeSubdivisions();
    for (const e of this.elems) {
      this.subdivisions.addElem(e);
    }
    this.subdivisions.addElem(newElem);
    this.elems.length = 0;
  }
  makeSubdivisions() {
    const { x, y, width: width2, height: height2 } = this.boundary;
    const { capacity } = this;
    const depth = this.maxdepth - 1;
    const halfWidth = width2 / 2;
    const halfHeight = height2 / 2;
    const nwBoundary = new BBox(x, y, halfWidth, halfHeight);
    const neBoundary = new BBox(x + halfWidth, y, halfWidth, halfHeight);
    const swBoundary = new BBox(x, y + halfHeight, halfWidth, halfHeight);
    const seBoundary = new BBox(x + halfWidth, y + halfHeight, halfWidth, halfHeight);
    return new QuadtreeSubdivisions(
      this.child(capacity, depth, nwBoundary),
      this.child(capacity, depth, neBoundary),
      this.child(capacity, depth, swBoundary),
      this.child(capacity, depth, seBoundary)
    );
  }
};
var QuadtreeNodeNearest = class _QuadtreeNodeNearest extends QuadtreeNode {
  addCondition(e) {
    const { x, y } = e.hitTester.midPoint;
    return this.boundary.containsPoint(x, y);
  }
  findCondition(x, y, arg) {
    const { best } = arg;
    return best.distanceSquared !== 0 && this.boundary.distanceSquared(x, y) < best.distanceSquared;
  }
  findAction(x, y, arg) {
    const other = nearestSquared(x, y, this.elems, arg.best.distanceSquared);
    if (other.nearest !== void 0 && other.distanceSquared < arg.best.distanceSquared) {
      arg.best = other;
    }
  }
  child(capacity, depth, boundary) {
    return new _QuadtreeNodeNearest(capacity, depth, boundary);
  }
};

// packages/ag-charts-community/src/scale/linearScale.ts
var LinearScale = class _LinearScale extends ContinuousScale {
  constructor() {
    super([0, 1], [0, 1]);
    this.type = "number";
  }
  static getTickStep(start2, stop, ticks) {
    const { interval, tickCount = ContinuousScale.defaultTickCount, minTickCount, maxTickCount } = ticks;
    return interval ?? tickStep(start2, stop, tickCount, minTickCount, maxTickCount);
  }
  toDomain(d) {
    return d;
  }
  ticks({ interval, tickCount = ContinuousScale.defaultTickCount, minTickCount, maxTickCount }, domain = this.domain, visibleRange) {
    if (!domain || domain.length < 2 || tickCount < 1 || !domain.every(isFinite)) {
      return [];
    }
    const [d0, d1] = domain;
    if (interval) {
      const step = Math.abs(interval);
      if (!isDenseInterval((d1 - d0) / step, this.getPixelRange())) {
        return range(d0, d1, step);
      }
    }
    return createTicks(d0, d1, tickCount, minTickCount, maxTickCount, visibleRange);
  }
  niceDomain(ticks, domain = this.domain) {
    if (domain.length < 2)
      return [];
    const { tickCount = ContinuousScale.defaultTickCount } = ticks;
    let [start2, stop] = domain;
    if (tickCount === 1) {
      [start2, stop] = niceTicksDomain(start2, stop);
    } else if (tickCount > 1) {
      const roundStart = start2 > stop ? Math.ceil : Math.floor;
      const roundStop = start2 > stop ? Math.floor : Math.ceil;
      const maxAttempts = 4;
      for (let i = 0; i < maxAttempts; i++) {
        const prev0 = start2;
        const prev1 = stop;
        const step = _LinearScale.getTickStep(start2, stop, ticks);
        const [d0, d1] = domain;
        start2 = roundStart(d0 / step) * step;
        stop = roundStop(d1 / step) * step;
        if (start2 === prev0 && stop === prev1)
          break;
      }
    }
    return [start2, stop];
  }
  tickFormatter({ ticks: specifiedTicks, fractionDigits, specifier }) {
    return specifier != null ? tickFormat(specifiedTicks, specifier) : (x) => formatValue(x, fractionDigits);
  }
  datumFormatter({ ticks: specifiedTicks, fractionDigits, specifier }) {
    return specifier != null ? tickFormat(specifiedTicks, specifier) : (x) => formatValue(x, fractionDigits + 1);
  }
};

// packages/ag-charts-community/src/util/extent.ts
function extent(values) {
  if (values.length === 0) {
    return null;
  }
  let min = Infinity;
  let max = -Infinity;
  for (const n of values) {
    const v = n instanceof Date ? n.getTime() : n;
    if (typeof v !== "number")
      continue;
    if (v < min) {
      min = v;
    }
    if (v > max) {
      max = v;
    }
  }
  const result = [min, max];
  return result.every(isFinite) ? result : null;
}
function normalisedExtentWithMetadata(d, min, max) {
  let clipped = false;
  if (d.length > 2) {
    d = extent(d) ?? [NaN, NaN];
  }
  if (!isNaN(min)) {
    clipped || (clipped = min > d[0]);
    d = [min, d[1]];
  }
  if (!isNaN(max)) {
    clipped || (clipped = max < d[1]);
    d = [d[0], max];
  }
  if (d[0] > d[1]) {
    d = [];
  }
  return { extent: d, clipped };
}

// packages/ag-charts-community/src/chart/axis/numberAxis.ts
var NumberAxis = class extends CartesianAxis {
  constructor(moduleCtx, scale = new LinearScale()) {
    super(moduleCtx, scale);
    this.min = NaN;
    this.max = NaN;
  }
  normaliseDataDomain(d) {
    const { min, max } = this;
    const { extent: extent2, clipped } = normalisedExtentWithMetadata(d, min, max);
    return { domain: extent2, clipped };
  }
};
NumberAxis.className = "NumberAxis";
NumberAxis.type = "number";
__decorateClass([
  Validate(AND(NUMBER_OR_NAN, LESS_THAN("max"))),
  Default(NaN)
], NumberAxis.prototype, "min", 2);
__decorateClass([
  Validate(AND(NUMBER_OR_NAN, GREATER_THAN("min"))),
  Default(NaN)
], NumberAxis.prototype, "max", 2);

// packages/ag-charts-community/src/chart/axis/timeAxis.ts
var TimeAxis = class extends CartesianAxis {
  constructor(moduleCtx) {
    super(moduleCtx, new TimeScale());
    this.min = void 0;
    this.max = void 0;
  }
  normaliseDataDomain(d) {
    let { min, max } = this;
    let clipped = false;
    if (typeof min === "number") {
      min = new Date(min);
    }
    if (typeof max === "number") {
      max = new Date(max);
    }
    if (d.length > 2) {
      d = extent(d)?.map((x) => new Date(x)) ?? [];
    }
    if (min instanceof Date) {
      clipped || (clipped = min > d[0]);
      d = [min, d[1]];
    }
    if (max instanceof Date) {
      clipped || (clipped = max < d[1]);
      d = [d[0], max];
    }
    if (d[0] > d[1]) {
      d = [];
    }
    return { domain: d, clipped };
  }
};
TimeAxis.className = "TimeAxis";
TimeAxis.type = "time";
__decorateClass([
  Validate(AND(DATE_OR_DATETIME_MS, LESS_THAN("max")), { optional: true })
], TimeAxis.prototype, "min", 2);
__decorateClass([
  Validate(AND(DATE_OR_DATETIME_MS, GREATER_THAN("min")), { optional: true })
], TimeAxis.prototype, "max", 2);

// packages/ag-charts-community/src/chart/series/dataModelSeries.ts
var DataModelSeries = class extends Series {
  constructor() {
    super(...arguments);
    this.clipFocusBox = true;
  }
  getScaleInformation({
    xScale,
    yScale
  }) {
    const isContinuousX = ContinuousScale.is(xScale);
    const isContinuousY = ContinuousScale.is(yScale);
    return { isContinuousX, isContinuousY, xScaleType: xScale?.type, yScaleType: yScale?.type };
  }
  getModulePropertyDefinitions() {
    const xScale = this.axes["x" /* X */]?.scale;
    const yScale = this.axes["y" /* Y */]?.scale;
    return this.moduleMap.mapModules((mod) => mod.getPropertyDefinitions(this.getScaleInformation({ xScale, yScale }))).flat();
  }
  // Request data, but with message dispatching to series-options (modules).
  async requestDataModel(dataController, data, opts) {
    opts.props.push(...this.getModulePropertyDefinitions());
    const { dataModel, processedData } = await dataController.request(this.id, data ?? [], opts);
    this.dataModel = dataModel;
    this.processedData = processedData;
    this.dispatch("data-processed", { dataModel, processedData });
    return { dataModel, processedData };
  }
  isProcessedDataAnimatable() {
    const validationResults = this.processedData?.reduced?.animationValidation;
    if (!validationResults) {
      return true;
    }
    const { orderedKeys, uniqueKeys } = validationResults;
    return orderedKeys && uniqueKeys;
  }
  checkProcessedDataAnimatable() {
    if (!this.isProcessedDataAnimatable()) {
      this.ctx.animationManager.skipCurrentBatch();
    }
  }
  pickFocus(opts) {
    const nodeData = this.getNodeData();
    if (nodeData === void 0 || nodeData.length === 0) {
      return;
    }
    const datumIndex = this.computeFocusDatumIndex(opts, nodeData);
    if (datumIndex === void 0) {
      return;
    }
    const { clipFocusBox } = this;
    const datum = nodeData[datumIndex];
    const derivedOpts = { ...opts, datumIndex };
    const bounds = this.computeFocusBounds(derivedOpts);
    if (bounds !== void 0) {
      return { bounds, clipFocusBox, datum, datumIndex };
    }
  }
  computeFocusDatumIndex(opts, nodeData) {
    const isDatumEnabled = (datumIndex2) => {
      const { missing = false, enabled = true, focusable = true } = nodeData[datumIndex2];
      return !missing && enabled && focusable;
    };
    const searchBackward = (datumIndex2, delta2) => {
      while (datumIndex2 >= 0 && !isDatumEnabled(datumIndex2)) {
        datumIndex2 += delta2;
      }
      return datumIndex2 === -1 ? void 0 : datumIndex2;
    };
    const searchForward = (datumIndex2, delta2) => {
      while (datumIndex2 < nodeData.length && !isDatumEnabled(datumIndex2)) {
        datumIndex2 += delta2;
      }
      return datumIndex2 === nodeData.length ? void 0 : datumIndex2;
    };
    let datumIndex;
    const clampedIndex = clamp(0, opts.datumIndex, nodeData.length - 1);
    if (opts.datumIndexDelta < 0) {
      datumIndex = searchBackward(clampedIndex, opts.datumIndexDelta);
    } else if (opts.datumIndexDelta > 0) {
      datumIndex = searchForward(clampedIndex, opts.datumIndexDelta);
    } else {
      datumIndex = searchForward(clampedIndex, 1) ?? searchBackward(clampedIndex, -1);
    }
    if (datumIndex === void 0) {
      if (opts.datumIndexDelta === 0) {
        return;
      } else {
        return opts.datumIndex - opts.datumIndexDelta;
      }
    } else {
      return datumIndex;
    }
  }
};

// packages/ag-charts-community/src/chart/series/seriesProperties.ts
var SeriesItemHighlightStyle = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.fill = "rgba(255,255,255, 0.33)";
    this.stroke = `rgba(0, 0, 0, 0.4)`;
    this.strokeWidth = 2;
  }
};
__decorateClass([
  Validate(COLOR_STRING, { optional: true })
], SeriesItemHighlightStyle.prototype, "fill", 2);
__decorateClass([
  Validate(RATIO, { optional: true })
], SeriesItemHighlightStyle.prototype, "fillOpacity", 2);
__decorateClass([
  Validate(COLOR_STRING, { optional: true })
], SeriesItemHighlightStyle.prototype, "stroke", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER, { optional: true })
], SeriesItemHighlightStyle.prototype, "strokeWidth", 2);
__decorateClass([
  Validate(RATIO, { optional: true })
], SeriesItemHighlightStyle.prototype, "strokeOpacity", 2);
__decorateClass([
  Validate(LINE_DASH, { optional: true })
], SeriesItemHighlightStyle.prototype, "lineDash", 2);
__decorateClass([
  Validate(POSITIVE_NUMBER, { optional: true })
], SeriesItemHighlightStyle.prototype, "lineDashOffset", 2);
var SeriesHighlightStyle = class extends BaseProperties {
};
__decorateClass([
  Validate(POSITIVE_NUMBER, { optional: true })
], SeriesHighlightStyle.prototype, "strokeWidth", 2);
__decorateClass([
  Validate(RATIO, { optional: true })
], SeriesHighlightStyle.prototype, "dimOpacity", 2);
__decorateClass([
  Validate(BOOLEAN, { optional: true })
], SeriesHighlightStyle.prototype, "enabled", 2);
var TextHighlightStyle = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.color = "black";
  }
};
__decorateClass([
  Validate(COLOR_STRING, { optional: true })
], TextHighlightStyle.prototype, "color", 2);
var HighlightProperties = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.enabled = true;
  }
};
__decorateClass([
  Validate(BOOLEAN, { optional: true })
], HighlightProperties.prototype, "enabled", 2);
var HighlightStyle = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.item = new SeriesItemHighlightStyle();
    this.series = new SeriesHighlightStyle();
    this.text = new TextHighlightStyle();
  }
};
__decorateClass([
  Validate(OBJECT)
], HighlightStyle.prototype, "item", 2);
__decorateClass([
  Validate(OBJECT)
], HighlightStyle.prototype, "series", 2);
__decorateClass([
  Validate(OBJECT)
], HighlightStyle.prototype, "text", 2);
var SeriesProperties = class extends BaseProperties {
  constructor() {
    super(...arguments);
    this.visible = true;
    this.focusPriority = Infinity;
    this.showInLegend = true;
    this.cursor = "default";
    this.nodeClickRange = "exact";
    this.highlight = new HighlightProperties();
    this.highlightStyle = new HighlightStyle();
  }
};
__decorateClass([
  Validate(STRING, { optional: true })
], SeriesProperties.prototype, "id", 2);
__decorateClass([
  Validate(BOOLEAN)
], SeriesProperties.prototype, "visible", 2);
__decorateClass([
  Validate(REAL_NUMBER, { optional: true })
], SeriesProperties.prototype, "focusPriority", 2);
__decorateClass([
  Validate(BOOLEAN)
], SeriesProperties.prototype, "showInLegend", 2);
__decorateClass([
  Validate(STRING)
], SeriesProperties.prototype, "cursor", 2);
__decorateClass([
  Validate(INTERACTION_RANGE)
], SeriesProperties.prototype, "nodeClickRange", 2);
__decorateClass([
  Validate(OBJECT)
], SeriesProperties.prototype, "highlight", 2);
__decorateClass([
  Validate(OBJECT)
], SeriesProperties.prototype, "highlightStyle", 2);

// packages/ag-charts-community/src/chart/series/cartesian/cartesianSeries.ts
var DEFAULT_CARTESIAN_DIRECTION_KEYS = {
  ["x" /* X */]: ["xKey"],
  ["y" /* Y */]: ["yKey"]
};
var DEFAULT_CARTESIAN_DIRECTION_NAMES = {
  ["x" /* X */]: ["xName"],
  ["y" /* Y */]: ["yName"]
};
var CartesianSeriesNodeEvent = class extends SeriesNodeEvent {
  constructor(type, nativeEvent, datum, series) {
    super(type, nativeEvent, datum, series);
    this.xKey = series.properties.xKey;
    this.yKey = series.properties.yKey;
  }
};
var CartesianSeriesProperties = class extends SeriesProperties {
  constructor() {
    super(...arguments);
    this.pickOutsideVisibleMinorAxis = false;
  }
};
__decorateClass([
  Validate(STRING, { optional: true })
], CartesianSeriesProperties.prototype, "legendItemName", 2);
__decorateClass([
  Validate(BOOLEAN, { optional: true })
], CartesianSeriesProperties.prototype, "pickOutsideVisibleMinorAxis", 2);
var RENDER_TO_OFFSCREEN_CANVAS_THRESHOLD = 100;
var CartesianSeries = class extends DataModelSeries {
  constructor({
    pathsPerSeries = ["path"],
    hasMarkers = false,
    hasHighlightedLabels = false,
    pathsZIndexSubOrderOffset = [],
    datumSelectionGarbageCollection = true,
    markerSelectionGarbageCollection = true,
    animationAlwaysUpdateSelections = false,
    animationResetFns,
    directionKeys,
    directionNames,
    ...otherOpts
  }) {
    super({
      directionKeys,
      directionNames,
      canHaveAxes: true,
      ...otherOpts
    });
    this.NodeEvent = CartesianSeriesNodeEvent;
    this.dataNodeGroup = this.contentGroup.appendChild(
      new Group({ name: `${this.id}-series-dataNodes`, zIndex: 0 })
    );
    this.markerGroup = this.contentGroup.appendChild(
      new Group({ name: `${this.id}-series-markers`, zIndex: 1 })
    );
    this.labelGroup = this.contentGroup.appendChild(
      new TranslatableGroup({ name: `${this.id}-series-labels` })
    );
    this.labelSelection = Selection.select(this.labelGroup, Text);
    this.highlightSelection = Selection.select(
      this.highlightNode,
      () => this.opts.hasMarkers ? new Marker() : this.nodeFactory()
    );
    this.highlightLabelSelection = Selection.select(this.highlightLabel, Text);
    this.annotationSelections = /* @__PURE__ */ new Set();
    this.debug = Debug.create();
    if (!directionKeys || !directionNames)
      throw new Error(`Unable to initialise series type ${this.type}`);
    this.opts = {
      pathsPerSeries,
      hasMarkers,
      hasHighlightedLabels,
      pathsZIndexSubOrderOffset,
      directionKeys,
      directionNames,
      animationResetFns,
      animationAlwaysUpdateSelections,
      datumSelectionGarbageCollection,
      markerSelectionGarbageCollection
    };
    this.paths = pathsPerSeries.map((path) => {
      return new Path({ name: `${this.id}-${path}` });
    });
    this.datumSelection = Selection.select(
      this.dataNodeGroup,
      () => this.nodeFactory(),
      datumSelectionGarbageCollection
    );
    this.markerSelection = Selection.select(this.markerGroup, Marker, markerSelectionGarbageCollection);
    this.animationState = new StateMachine(
      "empty",
      {
        empty: {
          update: {
            target: "ready",
            action: (data) => this.animateEmptyUpdateReady(data)
          },
          reset: "empty",
          skip: "ready",
          disable: "disabled"
        },
        ready: {
          updateData: "waiting",
          clear: "clearing",
          highlight: (data) => this.animateReadyHighlight(data),
          highlightMarkers: (data) => this.animateReadyHighlightMarkers(data),
          resize: (data) => this.animateReadyResize(data),
          reset: "empty",
          skip: "ready",
          disable: "disabled"
        },
        waiting: {
          update: {
            target: "ready",
            action: (data) => {
              if (this.ctx.animationManager.isSkipped()) {
                this.resetAllAnimation(data);
              } else {
                this.animateWaitingUpdateReady(data);
              }
            }
          },
          reset: "empty",
          skip: "ready",
          disable: "disabled"
        },
        disabled: {
          update: (data) => this.resetAllAnimation(data),
          reset: "empty"
        },
        clearing: {
          update: {
            target: "empty",
            action: (data) => this.animateClearingUpdateEmpty(data)
          },
          reset: "empty",
          skip: "ready"
        }
      },
      () => this.checkProcessedDataAnimatable()
    );
  }
  get contextNodeData() {
    return this._contextNodeData;
  }
  getNodeData() {
    return this.contextNodeData?.nodeData;
  }
  attachSeries(seriesContentNode, seriesNode, annotationNode) {
    super.attachSeries(seriesContentNode, seriesNode, annotationNode);
    this.attachPaths(this.paths, seriesNode, annotationNode);
  }
  detachSeries(seriesContentNode, seriesNode, annotationNode) {
    super.detachSeries(seriesContentNode, seriesNode, annotationNode);
    this.detachPaths(this.paths, seriesNode, annotationNode);
  }
  attachPaths(paths, _seriesNode, _annotationNode) {
    for (const path of paths) {
      this.contentGroup.appendChild(path);
    }
  }
  detachPaths(paths, _seriesNode, _annotationNode) {
    for (const path of paths) {
      this.contentGroup.removeChild(path);
    }
  }
  renderToOffscreenCanvas() {
    const nodeData = this.getNodeData();
    return nodeData != null && nodeData.length > RENDER_TO_OFFSCREEN_CANVAS_THRESHOLD;
  }
  resetAnimation(phase) {
    if (phase === "initial") {
      this.animationState.transition("reset");
    } else if (phase === "ready") {
      this.animationState.transition("skip");
    } else if (phase === "disabled") {
      this.animationState.transition("disable");
    }
  }
  addChartEventListeners() {
    this.destroyFns.push(
      this.ctx.chartEventManager.addListener("legend-item-click", (event) => this.onLegendItemClick(event)),
      this.ctx.chartEventManager.addListener(
        "legend-item-double-click",
        (event) => this.onLegendItemDoubleClick(event)
      )
    );
  }
  destroy() {
    super.destroy();
    this._contextNodeData = void 0;
  }
  update({ seriesRect }) {
    const { visible, _contextNodeData: previousContextData } = this;
    const series = this.ctx.highlightManager?.getActiveHighlight()?.series;
    const seriesHighlighted = series === this;
    const resize = this.checkResize(seriesRect);
    const highlightItems = this.updateHighlightSelection(seriesHighlighted);
    this.updateSelections(visible);
    this.updateNodes(highlightItems, seriesHighlighted, visible);
    const animationData = this.getAnimationData(seriesRect, previousContextData);
    if (!animationData)
      return;
    if (resize) {
      this.animationState.transition("resize", animationData);
    }
    this.animationState.transition("update", animationData);
  }
  updateSelections(anySeriesItemEnabled) {
    var _a;
    const animationSkipUpdate = !this.opts.animationAlwaysUpdateSelections && this.ctx.animationManager.isSkipped();
    if (!anySeriesItemEnabled && animationSkipUpdate) {
      return;
    }
    if (!this.nodeDataRefresh && !this.isPathOrSelectionDirty()) {
      return;
    }
    if (this.nodeDataRefresh) {
      this.nodeDataRefresh = false;
      this.debug(`CartesianSeries.updateSelections() - calling createNodeData() for`, this.id);
      this.markQuadtreeDirty();
      this._contextNodeData = this.createNodeData();
      const animationValid = this.isProcessedDataAnimatable();
      if (this._contextNodeData) {
        (_a = this._contextNodeData).animationValid ?? (_a.animationValid = animationValid);
      }
      const { dataModel, processedData } = this;
      if (dataModel !== void 0 && processedData !== void 0) {
        this.dispatch("data-update", { dataModel, processedData });
      }
    }
    this.updateSeriesSelections();
  }
  updateSeriesSelections(seriesHighlighted) {
    const { datumSelection, labelSelection, markerSelection, paths } = this;
    const contextData = this._contextNodeData;
    if (!contextData)
      return;
    const { nodeData, labelData, itemId } = contextData;
    this.updatePaths({ seriesHighlighted, itemId, contextData, paths });
    this.datumSelection = this.updateDatumSelection({ nodeData, datumSelection });
    this.labelSelection = this.updateLabelSelection({ labelData, labelSelection }) ?? labelSelection;
    if (this.opts.hasMarkers) {
      this.markerSelection = this.updateMarkerSelection({ nodeData, markerSelection });
    }
  }
  updateNodes(highlightedItems, seriesHighlighted, anySeriesItemEnabled) {
    const {
      highlightSelection,
      highlightLabelSelection,
      opts: { hasMarkers, hasHighlightedLabels }
    } = this;
    const animationEnabled = !this.ctx.animationManager.isSkipped();
    const visible = this.visible && this._contextNodeData != null && anySeriesItemEnabled;
    this.contentGroup.visible = animationEnabled || visible;
    this.highlightGroup.visible = (animationEnabled || visible) && seriesHighlighted;
    const opacity = this.getOpacity();
    if (hasMarkers) {
      this.updateMarkerNodes({
        markerSelection: highlightSelection,
        isHighlight: true
      });
      this.animationState.transition("highlightMarkers", highlightSelection);
    } else {
      this.updateDatumNodes({
        datumSelection: highlightSelection,
        isHighlight: true
      });
      this.animationState.transition("highlight", highlightSelection);
    }
    if (hasHighlightedLabels) {
      this.updateLabelNodes({ labelSelection: highlightLabelSelection });
    }
    const { dataNodeGroup, markerGroup, datumSelection, labelSelection, markerSelection, paths, labelGroup } = this;
    const { itemId } = this.contextNodeData ?? {};
    dataNodeGroup.opacity = opacity;
    dataNodeGroup.visible = animationEnabled || visible;
    labelGroup.visible = visible;
    if (hasMarkers) {
      markerGroup.opacity = opacity;
      markerGroup.visible = visible;
    }
    if (labelGroup) {
      labelGroup.opacity = opacity;
    }
    this.updatePathNodes({
      seriesHighlighted,
      itemId,
      paths,
      opacity,
      visible,
      animationEnabled
    });
    if (!dataNodeGroup.visible) {
      return;
    }
    this.updateDatumNodes({ datumSelection, highlightedItems, isHighlight: false });
    if (!this.usesPlacedLabels) {
      this.updateLabelNodes({ labelSelection });
    }
    if (hasMarkers) {
      this.updateMarkerNodes({ markerSelection, isHighlight: false });
    }
  }
  getHighlightLabelData(labelData, highlightedItem) {
    const labelItems = labelData.filter(
      (ld) => ld.datum === highlightedItem.datum && ld.itemId === highlightedItem.itemId
    );
    return labelItems.length === 0 ? void 0 : labelItems;
  }
  getHighlightData(_nodeData, highlightedItem) {
    return highlightedItem ? [highlightedItem] : void 0;
  }
  updateHighlightSelection(seriesHighlighted) {
    const { highlightSelection, highlightLabelSelection, _contextNodeData: contextNodeData } = this;
    if (!contextNodeData)
      return;
    const highlightedDatum = this.ctx.highlightManager?.getActiveHighlight();
    const item = seriesHighlighted && highlightedDatum?.datum ? highlightedDatum : void 0;
    let labelItems;
    let highlightItems;
    if (item != null) {
      const labelsEnabled = this.isLabelEnabled();
      const { labelData, nodeData } = contextNodeData;
      highlightItems = this.getHighlightData(nodeData, item);
      labelItems = labelsEnabled ? this.getHighlightLabelData(labelData, item) : void 0;
    }
    this.highlightSelection = this.updateHighlightSelectionItem({
      items: highlightItems,
      highlightSelection
    });
    this.highlightLabelSelection = this.updateHighlightSelectionLabel({
      items: labelItems,
      highlightLabelSelection
    });
    return highlightItems;
  }
  markQuadtreeDirty() {
    this.quadtree = void 0;
  }
  *datumNodesIter() {
    for (const { node } of this.datumSelection) {
      if (node.datum.missing === true)
        continue;
      yield node;
    }
  }
  getQuadTree() {
    if (this.quadtree === void 0) {
      const { width: width2, height: height2 } = this.ctx.scene.canvas;
      const canvasRect = new BBox(0, 0, width2, height2);
      this.quadtree = new QuadtreeNearest(100, 10, canvasRect);
      this.initQuadTree(this.quadtree);
    }
    return this.quadtree;
  }
  initQuadTree(_quadtree) {
  }
  pickNodeExactShape(point) {
    const result = super.pickNodeExactShape(point);
    if (result) {
      return result;
    }
    const { x, y } = point;
    const {
      opts: { hasMarkers }
    } = this;
    let match;
    const { dataNodeGroup, markerGroup } = this;
    match = dataNodeGroup.pickNode(x, y);
    if (!match && hasMarkers) {
      match = markerGroup?.pickNode(x, y);
    }
    if (match && match.datum.missing !== true) {
      return { datum: match.datum, distance: 0 };
    }
    for (const mod of this.moduleMap.modules()) {
      const { datum } = mod.pickNodeExact(point) ?? {};
      if (datum == null)
        continue;
      if (datum?.missing === true)
        continue;
      return { datum, distance: 0 };
    }
  }
  pickNodeClosestDatum(point) {
    const { x, y } = point;
    const { axes, _contextNodeData: contextNodeData } = this;
    if (!contextNodeData)
      return;
    const xAxis = axes["x" /* X */];
    const yAxis = axes["y" /* Y */];
    const hitPoint = { x, y };
    let minDistance = Infinity;
    let closestDatum;
    for (const datum of contextNodeData.nodeData) {
      const { point: { x: datumX = NaN, y: datumY = NaN } = {} } = datum;
      if (isNaN(datumX) || isNaN(datumY)) {
        continue;
      }
      const isInRange = xAxis?.inRange(datumX) && yAxis?.inRange(datumY);
      if (!isInRange) {
        continue;
      }
      const distance = Math.max((hitPoint.x - datumX) ** 2 + (hitPoint.y - datumY) ** 2, 0);
      if (distance < minDistance) {
        minDistance = distance;
        closestDatum = datum;
      }
    }
    for (const mod of this.moduleMap.modules()) {
      const modPick = mod.pickNodeNearest(point);
      if (modPick !== void 0 && modPick.distanceSquared < minDistance) {
        minDistance = modPick.distanceSquared;
        closestDatum = modPick.datum;
        break;
      }
    }
    if (closestDatum) {
      const distance = Math.max(Math.sqrt(minDistance) - (closestDatum.point?.size ?? 0), 0);
      return { datum: closestDatum, distance };
    }
  }
  pickNodeMainAxisFirst(point, requireCategoryAxis) {
    const { x, y } = point;
    const { axes, _contextNodeData: contextNodeData } = this;
    const { pickOutsideVisibleMinorAxis } = this.properties;
    if (!contextNodeData)
      return;
    const xAxis = axes["x" /* X */];
    const yAxis = axes["y" /* Y */];
    const directions2 = [xAxis, yAxis].filter(CategoryAxis.is).map((a) => a.direction);
    if (requireCategoryAxis && directions2.length === 0)
      return;
    const [majorDirection = "x" /* X */] = directions2;
    const hitPointCoords = [x, y];
    if (majorDirection !== "x" /* X */)
      hitPointCoords.reverse();
    const minDistance = [Infinity, Infinity];
    let closestDatum;
    for (const datum of contextNodeData.nodeData) {
      const { x: datumX = NaN, y: datumY = NaN } = datum.point ?? datum.midPoint ?? {};
      if (isNaN(datumX) || isNaN(datumY) || datum.missing === true)
        continue;
      const visible = [xAxis?.inRange(datumX), yAxis?.inRange(datumY)];
      if (majorDirection !== "x" /* X */) {
        visible.reverse();
      }
      if (!visible[0] || !pickOutsideVisibleMinorAxis && !visible[1])
        continue;
      const datumPoint = [datumX, datumY];
      if (majorDirection !== "x" /* X */) {
        datumPoint.reverse();
      }
      let newMinDistance = true;
      for (let i = 0; i < datumPoint.length; i++) {
        const dist = Math.abs(datumPoint[i] - hitPointCoords[i]);
        if (dist > minDistance[i]) {
          newMinDistance = false;
          break;
        } else if (dist < minDistance[i]) {
          minDistance[i] = dist;
          minDistance.fill(Infinity, i + 1, minDistance.length);
        }
      }
      if (newMinDistance) {
        closestDatum = datum;
      }
    }
    if (closestDatum) {
      let closestDistanceSquared = Math.max(
        minDistance[0] ** 2 + minDistance[1] ** 2 - (closestDatum.point?.size ?? 0),
        0
      );
      for (const mod of this.moduleMap.modules()) {
        const modPick = mod.pickNodeMainAxisFirst(point);
        if (modPick !== void 0 && modPick.distanceSquared < closestDistanceSquared) {
          closestDatum = modPick.datum;
          closestDistanceSquared = modPick.distanceSquared;
          break;
        }
      }
      return { datum: closestDatum, distance: Math.sqrt(closestDistanceSquared) };
    }
  }
  isPathOrSelectionDirty() {
    return false;
  }
  shouldFlipXY() {
    return false;
  }
  // Workaround - it would be nice if this difference didn't exist
  keysOrValues(xKey) {
    const key = this.dataModel.resolveProcessedDataIndexById(this, xKey);
    return this.processedData?.keys[key]?.get(this.id) ?? this.processedData?.columns[key] ?? [];
  }
  visibleRange(axisKey, visibleRange, indices) {
    const xValues = this.keysOrValues(axisKey);
    const pixelSize = 0;
    return visibleRangeIndices(indices?.length ?? xValues.length, visibleRange, (topIndex) => {
      const datumIndex = indices?.[topIndex] ?? topIndex;
      return this.xCoordinateRange(xValues[datumIndex], pixelSize, datumIndex);
    });
  }
  domainForVisibleRange(_direction, axisKeys, crossAxisKey, visibleRange, sorted, indices) {
    const { processedData, dataModel } = this;
    const [r0, r1] = visibleRange;
    const crossAxisValues = this.keysOrValues(crossAxisKey);
    if (sorted) {
      const crossAxisRange = this.visibleRange(crossAxisKey, visibleRange, indices);
      return dataModel.getDomainBetweenRange(this, axisKeys, crossAxisRange, processedData);
    }
    const allAxisValues = axisKeys.map((axisKey) => this.keysOrValues(axisKey));
    let axisMin = Infinity;
    let axisMax = -Infinity;
    crossAxisValues.forEach((crossAxisValue, i) => {
      const [x0, x1] = this.xCoordinateRange(crossAxisValue, 0, i);
      if (x1 < r0 || x0 > r1)
        return;
      for (let j = 0; j < axisKeys.length; j++) {
        const axisValue = allAxisValues[j][i];
        axisMin = Math.min(axisMin, axisValue);
        axisMax = Math.max(axisMax, axisValue);
      }
    });
    if (axisMin > axisMax)
      return [NaN, NaN];
    return [axisMin, axisMax];
  }
  domainForClippedRange(direction, axisKeys, crossAxisKey, sorted) {
    const { processedData, dataModel, axes } = this;
    const crossDirection = direction === "x" /* X */ ? "y" /* Y */ : "x" /* X */;
    const crossAxisRange = axisExtent(axes[crossDirection]);
    if (!crossAxisRange) {
      return axisKeys.flatMap((axisKey) => dataModel.getDomain(this, axisKey, "value", processedData));
    }
    const crossAxisValues = this.keysOrValues(crossAxisKey);
    if (sorted) {
      const crossRange = clippedRangeIndices(
        crossAxisValues.length,
        crossAxisRange,
        (index) => crossAxisValues[index]
      );
      return dataModel.getDomainBetweenRange(this, axisKeys, crossRange, processedData);
    }
    const allAxisValues = axisKeys.map((axisKey) => this.keysOrValues(axisKey));
    const range0 = crossAxisRange[0].valueOf();
    const range1 = crossAxisRange[1].valueOf();
    const axisValues = [];
    crossAxisValues.forEach((crossAxisValue, i) => {
      const c = crossAxisValue.valueOf();
      if (c < range0 || c > range1)
        return;
      const values = allAxisValues.map((v) => v[i]);
      if (c >= range0) {
        axisValues.push(...values);
      }
      if (c <= range1) {
        axisValues.push(...values);
      }
    });
    return axisValues;
  }
  countVisibleItems(crossAxisKey, axisKeys, xVisibleRange, yVisibleRange, minVisibleItems) {
    const { dataModel, processedData } = this;
    if (!dataModel || !processedData)
      return Infinity;
    const crossValues = this.keysOrValues(crossAxisKey);
    const allAxisValues = axisKeys.map((axisKey) => dataModel.resolveColumnById(this, axisKey, processedData));
    const crossAxis = this.axes["x" /* X */];
    const axis = this.axes["y" /* Y */];
    const shouldFlipXY = this.shouldFlipXY();
    const crossRange = crossAxis.range;
    const range2 = axis.range;
    const convert = (d, r, v) => {
      return d[0] + (v - r[0]) / (r[1] - r[0]) * (d[1] - d[0]);
    };
    const crossMin = convert(crossRange, crossAxis.visibleRange, xVisibleRange[0]);
    const crossMax = convert(crossRange, crossAxis.visibleRange, xVisibleRange[1]);
    const axisMin = convert(range2, axis.visibleRange, shouldFlipXY ? yVisibleRange[0] : yVisibleRange[1]);
    const axisMax = convert(range2, axis.visibleRange, shouldFlipXY ? yVisibleRange[1] : yVisibleRange[0]);
    const startIndex = Math.round(
      (xVisibleRange[0] + (xVisibleRange[1] - xVisibleRange[0]) / 2) * crossValues.length
    );
    const pixelSize = 0;
    return countExpandingSearch(0, crossValues.length - 1, startIndex, minVisibleItems, (index) => {
      let [x0, x1] = this.xCoordinateRange(crossValues[index], pixelSize, index);
      let [y0, y1] = this.yCoordinateRange(
        allAxisValues.map((axisValues) => axisValues[index]),
        pixelSize,
        index
      );
      if (!isFiniteNumber(x0) || !isFiniteNumber(x1) || !isFiniteNumber(y0) || !isFiniteNumber(y1)) {
        return false;
      }
      if (shouldFlipXY)
        [x0, x1, y0, y1] = [y0, y1, x0, x1];
      return x0 >= crossMin && x1 <= crossMax && y0 >= axisMin && y1 <= axisMax;
    });
  }
  updateHighlightSelectionItem(opts) {
    const {
      opts: { hasMarkers }
    } = this;
    const { items, highlightSelection } = opts;
    const nodeData = items ?? [];
    if (hasMarkers) {
      const markerSelection = highlightSelection;
      return this.updateMarkerSelection({ nodeData, markerSelection });
    } else {
      return this.updateDatumSelection({
        nodeData,
        datumSelection: highlightSelection
      });
    }
  }
  updateHighlightSelectionLabel(opts) {
    return this.updateLabelSelection({
      labelData: opts.items ?? [],
      labelSelection: opts.highlightLabelSelection
    });
  }
  updateDatumSelection(opts) {
    return opts.datumSelection;
  }
  updateDatumNodes(_opts) {
  }
  updateMarkerSelection(opts) {
    return opts.markerSelection;
  }
  updateMarkerNodes(_opts) {
  }
  updatePaths(opts) {
    opts.paths.forEach((p) => p.visible = false);
  }
  updatePathNodes(opts) {
    const { paths, opacity, visible } = opts;
    for (const path of paths) {
      path.opacity = opacity;
      path.visible = visible;
    }
  }
  resetPathAnimation(data) {
    const { path } = this.opts?.animationResetFns ?? {};
    if (path) {
      data.paths.forEach((paths) => {
        resetMotion([paths], path);
      });
    }
  }
  resetDatumAnimation(data) {
    const { datum } = this.opts?.animationResetFns ?? {};
    if (datum) {
      resetMotion([data.datumSelection], datum);
    }
  }
  resetLabelAnimation(data) {
    const { label } = this.opts?.animationResetFns ?? {};
    if (label) {
      resetMotion([data.labelSelection], label);
    }
  }
  resetMarkerAnimation(data) {
    const { marker } = this.opts?.animationResetFns ?? {};
    if (marker && this.opts.hasMarkers) {
      resetMotion([data.markerSelection], marker);
    }
  }
  resetAllAnimation(data) {
    this.ctx.animationManager.stopByAnimationGroupId(this.id);
    this.resetPathAnimation(data);
    this.resetDatumAnimation(data);
    this.resetLabelAnimation(data);
    this.resetMarkerAnimation(data);
    if (data.contextData?.animationValid === false) {
      this.ctx.animationManager.skipCurrentBatch();
    }
  }
  animateEmptyUpdateReady(data) {
    this.ctx.animationManager.skipCurrentBatch();
    this.resetAllAnimation(data);
  }
  animateWaitingUpdateReady(data) {
    this.ctx.animationManager.skipCurrentBatch();
    this.resetAllAnimation(data);
  }
  animateReadyHighlight(data) {
    const { datum } = this.opts?.animationResetFns ?? {};
    if (datum) {
      resetMotion([data], datum);
    }
  }
  animateReadyHighlightMarkers(data) {
    const { marker } = this.opts?.animationResetFns ?? {};
    if (marker) {
      resetMotion([data], marker);
    }
  }
  animateReadyResize(data) {
    this.resetAllAnimation(data);
  }
  animateClearingUpdateEmpty(data) {
    this.ctx.animationManager.skipCurrentBatch();
    this.resetAllAnimation(data);
  }
  getAnimationData(seriesRect, previousContextData) {
    const { _contextNodeData: contextData } = this;
    if (!contextData)
      return;
    const animationData = {
      datumSelection: this.datumSelection,
      markerSelection: this.markerSelection,
      labelSelection: this.labelSelection,
      annotationSelections: [...this.annotationSelections],
      contextData,
      previousContextData,
      paths: this.paths,
      seriesRect
    };
    return animationData;
  }
  updateLabelSelection(opts) {
    return opts.labelSelection;
  }
  getScaling(scale) {
    if (scale instanceof LogScale) {
      const { range: range2, domain } = scale;
      return {
        type: "log",
        convert: (d) => scale.convert(d),
        domain: [domain[0], domain[1]],
        range: [range2[0], range2[1]]
      };
    } else if (scale instanceof ContinuousScale) {
      const { range: range2, domain } = scale;
      return {
        type: "continuous",
        domain: [domain[0], domain[1]],
        range: [range2[0], range2[1]]
      };
    } else if (scale instanceof BandScale) {
      const { domain } = scale;
      return {
        type: "category",
        domain,
        inset: scale.inset,
        step: scale.step
      };
    }
  }
  calculateScaling() {
    const result = {};
    for (const direction of Object.values(ChartAxisDirection)) {
      const axis = this.axes[direction];
      if (!axis)
        continue;
      const scalingResult = this.getScaling(axis.scale);
      if (scalingResult != null) {
        result[direction] = scalingResult;
      }
    }
    return result;
  }
};
function axisExtent(axis) {
  let min;
  let max;
  if (axis instanceof NumberAxis && (Number.isFinite(axis.min) || Number.isFinite(axis.max))) {
    min = Number.isFinite(axis.min) ? axis.min : void 0;
    max = Number.isFinite(axis.max) ? axis.max : void 0;
  } else if (axis instanceof TimeAxis && (axis.min != null || axis.max != null)) {
    ({ min, max } = axis);
  }
  if (min == null && max == null)
    return;
  min ?? (min = -Infinity);
  max ?? (max = Infinity);
  return [min, max];
}
function clippedRangeIndices(length, range2, xValue) {
  const range0 = range2[0].valueOf();
  const range1 = range2[1].valueOf();
  const xMinIndex = findMinIndex(0, length - 1, (index) => {
    const x = xValue(index)?.valueOf();
    return !Number.isFinite(x) || x >= range0;
  });
  let xMaxIndex = findMaxIndex(0, length - 1, (index) => {
    const x = xValue(index)?.valueOf();
    return !Number.isFinite(x) || x <= range1;
  });
  if (xMinIndex == null || xMaxIndex == null)
    return [0, 0];
  xMaxIndex = Math.min(xMaxIndex + 1, length);
  return [xMinIndex, xMaxIndex];
}

// packages/ag-charts-community/src/chart/cartesianChart.ts
var directions = ["top", "right", "bottom", "left"];
var _CartesianChart = class _CartesianChart extends Chart {
  constructor(options, resources) {
    super(options, resources);
    // TODO should come from theme
    /** Integrated Charts feature state - not used in Standalone Charts. */
    this.paired = true;
    this.lastUpdateClipRect = void 0;
    this.lastLayoutWidth = NaN;
    this.lastLayoutHeight = NaN;
  }
  onAxisChange(newValue, oldValue) {
    super.onAxisChange(newValue, oldValue);
    if (this.ctx != null) {
      this.ctx.zoomManager.updateAxes(newValue);
    }
  }
  destroySeries(series) {
    super.destroySeries(series);
    this.lastLayoutWidth = NaN;
    this.lastLayoutHeight = NaN;
  }
  getChartType() {
    return "cartesian";
  }
  setRootClipRects(clipRect) {
    const { seriesRoot, annotationRoot } = this;
    seriesRoot.setClipRect(clipRect);
    annotationRoot.setClipRect(clipRect);
  }
  async processData() {
    await super.processData();
    for (const axis of this.axes) {
      const syncedDomain = this.getSyncedDomain(axis);
      if (syncedDomain != null) {
        axis.setDomains(syncedDomain);
      }
    }
    this.ctx.updateService.dispatchProcessData({ series: { shouldFlipXY: this.shouldFlipXY() } });
  }
  performLayout(ctx) {
    const { seriesRoot, annotationRoot } = this;
    const { clipSeries, seriesRect, visible } = this.updateAxes(ctx.layoutBox);
    this.seriesRoot.visible = visible;
    this.seriesRect = seriesRect;
    this.animationRect = ctx.layoutBox;
    const { x, y } = seriesRect;
    if (ctx.width !== this.lastLayoutWidth || ctx.height !== this.lastLayoutHeight) {
      for (const group of [seriesRoot, annotationRoot]) {
        group.translationX = Math.floor(x);
        group.translationY = Math.floor(y);
      }
    } else {
      const { translationX, translationY } = seriesRoot;
      staticFromToMotion(
        this.id,
        "seriesRect",
        this.ctx.animationManager,
        [seriesRoot, annotationRoot],
        { translationX, translationY },
        { translationX: Math.floor(x), translationY: Math.floor(y) },
        { phase: "update" }
      );
    }
    this.lastLayoutWidth = ctx.width;
    this.lastLayoutHeight = ctx.height;
    const seriesPaddedRect = seriesRect.clone().grow(this.seriesArea.padding);
    const clipRect = this.seriesArea.clip || clipSeries ? seriesPaddedRect : void 0;
    const { lastUpdateClipRect } = this;
    this.lastUpdateClipRect = clipRect;
    if (this.ctx.animationManager.isActive() && lastUpdateClipRect != null) {
      this.ctx.animationManager.animate({
        id: this.id,
        groupId: "clip-rect",
        phase: "update",
        from: lastUpdateClipRect,
        to: seriesPaddedRect,
        onUpdate: (interpolatedClipRect) => this.setRootClipRects(interpolatedClipRect),
        onComplete: () => this.setRootClipRects(clipRect)
      });
    } else {
      this.setRootClipRects(clipRect);
    }
    this.ctx.layoutManager.emitLayoutComplete(ctx, {
      axes: this.axes.map((axis) => axis.getLayoutState()),
      series: {
        visible,
        rect: seriesRect,
        paddedRect: seriesPaddedRect
      },
      clipSeries
    });
  }
  updateAxes(layoutBox) {
    const { clipSeries, seriesRect, overflows } = this.resolveAxesLayout(layoutBox);
    for (const axis of this.axes) {
      axis.update();
      axis.setCrossLinesVisible(!overflows);
      this.clipAxis(axis, seriesRect, layoutBox);
    }
    return { clipSeries, seriesRect, visible: !overflows };
  }
  // Iteratively try to resolve axis widths - since X axis width affects Y axis range,
  // and vice-versa, we need to iteratively try and find a fit for the axes and their
  // ticks/labels.
  resolveAxesLayout(layoutBox) {
    let newState;
    let prevState;
    let iterations = 0;
    const maxIterations = 10;
    do {
      prevState = newState ?? this.getDefaultState();
      newState = this.updateAxesPass(new Map(prevState.axisAreaWidths), layoutBox.clone());
      if (iterations++ > maxIterations) {
        logger_exports.warn("Max iterations reached. Unable to stabilize axes layout.");
        break;
      }
    } while (!this.isLayoutStable(newState, prevState));
    this.lastAreaWidths = newState.axisAreaWidths;
    return newState;
  }
  updateAxesPass(axisAreaWidths, axisAreaBound) {
    const axisWidths = /* @__PURE__ */ new Map();
    const primaryTickCounts = {};
    let overflows = false;
    let clipSeries = false;
    for (const dir of directions) {
      const padding = this.seriesArea.padding[dir];
      const axis = this.axes.findLast((a) => a.position === dir);
      if (axis) {
        axis.seriesAreaPadding = padding;
      } else {
        axisAreaBound.shrink(padding, dir);
      }
    }
    const totalWidth = (axisAreaWidths.get("left") ?? 0) + (axisAreaWidths.get("right") ?? 0);
    const totalHeight = (axisAreaWidths.get("top") ?? 0) + (axisAreaWidths.get("bottom") ?? 0);
    const crossLinePadding = this.buildCrossLinePadding(axisAreaWidths);
    if (axisAreaBound.width <= totalWidth + crossLinePadding.hPadding || axisAreaBound.height <= totalHeight + crossLinePadding.vPadding) {
      overflows = true;
    } else {
      axisAreaBound.shrink(crossLinePadding);
    }
    const seriesRect = axisAreaBound.clone().shrink(Object.fromEntries(axisAreaWidths));
    for (const axis of this.axes) {
      const { position = "left", direction } = axis;
      this.sizeAxis(axis, seriesRect, position);
      const isVertical = direction === "y" /* Y */;
      const { primaryTickCount, bbox } = axis.calculateLayout(
        axis.nice ? primaryTickCounts[direction] : void 0
      );
      primaryTickCounts[direction] ?? (primaryTickCounts[direction] = primaryTickCount);
      clipSeries || (clipSeries = axis.dataDomain.clipped || axis.visibleRange[0] > 0 || axis.visibleRange[1] < 1);
      axisWidths.set(axis.id, Math.ceil(axis.thickness ?? (isVertical ? bbox?.width : bbox?.height) ?? 0));
    }
    const axisGroups = Object.entries(groupBy(this.axes, (axis) => axis.position ?? "left"));
    const { width: width2, height: height2, pixelRatio } = this.ctx.scene;
    const newAxisAreaWidths = /* @__PURE__ */ new Map();
    const axisOffsets = /* @__PURE__ */ new Map();
    for (const [position, axes] of axisGroups) {
      const isVertical = position === "left" || position === "right";
      let currentOffset = isVertical ? height2 % pixelRatio : width2 % pixelRatio;
      let totalAxisWidth = 0;
      for (const axis of axes) {
        axisOffsets.set(axis.id, currentOffset);
        const axisThickness = axisWidths.get(axis.id) ?? 0;
        totalAxisWidth = Math.max(totalAxisWidth, currentOffset + axisThickness);
        if (axis.layoutConstraints.stacked) {
          currentOffset += axisThickness + _CartesianChart.AxesPadding;
        }
      }
      newAxisAreaWidths.set(position, Math.ceil(totalAxisWidth));
    }
    for (const [position, axes] of axisGroups) {
      this.positionAxes({
        axes,
        position,
        axisWidths,
        axisOffsets,
        axisAreaWidths: newAxisAreaWidths,
        axisBound: axisAreaBound,
        seriesRect
      });
    }
    return { clipSeries, seriesRect, axisAreaWidths: newAxisAreaWidths, overflows };
  }
  buildCrossLinePadding(axisAreaSize) {
    const crossLinePadding = { top: 0, right: 0, bottom: 0, left: 0, hPadding: 0, vPadding: 0 };
    this.axes.forEach((axis) => {
      axis.crossLines?.forEach((crossLine) => {
        crossLine.calculatePadding?.(crossLinePadding);
      });
    });
    for (const [side, padding = 0] of Object.entries(crossLinePadding)) {
      crossLinePadding[side] = Math.max(padding - (axisAreaSize.get(side) ?? 0), 0);
    }
    crossLinePadding.hPadding = crossLinePadding.left + crossLinePadding.right;
    crossLinePadding.vPadding = crossLinePadding.top + crossLinePadding.bottom;
    return crossLinePadding;
  }
  clampToOutsideSeriesRect(seriesRect, value, dimension, direction) {
    const bound = dimension === "x" ? seriesRect.x : seriesRect.y;
    const size = dimension === "x" ? seriesRect.width : seriesRect.height;
    return direction === 1 ? Math.min(value, bound + size) : Math.max(value, bound);
  }
  getSyncedDomain(axis) {
    const syncModule = this.modulesManager.getModule("sync");
    if (!syncModule?.enabled)
      return;
    const syncedDomain = syncModule.getSyncedDomain(axis);
    if (syncedDomain && axis.dataDomain.domain.length) {
      let shouldUpdate;
      const { domain } = axis.scale;
      if (ContinuousScale.is(axis.scale)) {
        const [min, max] = findMinMax(syncedDomain);
        shouldUpdate = min !== domain[0] || max !== domain[1];
      } else {
        shouldUpdate = !arraysEqual(syncedDomain, domain);
      }
      if (shouldUpdate && !this.skipSync) {
        syncModule.updateSiblings();
      }
    }
    return syncedDomain;
  }
  sizeAxis(axis, seriesRect, position) {
    const isCategory = axis instanceof CategoryAxis;
    const isLeftRight = position === "left" || position === "right";
    const { width: width2, height: height2 } = seriesRect;
    const maxEnd = isLeftRight ? height2 : width2;
    let start2 = 0;
    let end2 = maxEnd;
    let { min, max } = this.ctx.zoomManager.getAxisZoom(axis.id);
    const { width: axisWidth, unit, align: align2 } = axis.layoutConstraints;
    if (unit === "px") {
      end2 = start2 + axisWidth;
    } else {
      end2 = end2 * axisWidth / 100;
    }
    if (align2 === "end") {
      start2 = maxEnd - (end2 - start2);
      end2 = maxEnd;
    }
    if (isLeftRight) {
      if (isCategory) {
        [min, max] = [1 - max, 1 - min];
      } else {
        [start2, end2] = [end2, start2];
      }
    }
    axis.range = [start2, end2];
    axis.visibleRange = [min, max];
    axis.gridLength = isLeftRight ? width2 : height2;
  }
  positionAxes(opts) {
    const { axes, axisBound, axisWidths, axisOffsets, axisAreaWidths, seriesRect, position } = opts;
    const axisAreaWidth = axisAreaWidths.get(position) ?? 0;
    let mainDimension = "x";
    let minorDimension = "y";
    let direction = 1;
    if (position === "top" || position === "bottom") {
      mainDimension = "y";
      minorDimension = "x";
    }
    let axisBoundMainOffset = axisBound[mainDimension];
    if (position === "right" || position === "bottom") {
      direction = -1;
      axisBoundMainOffset += mainDimension === "x" ? axisBound.width : axisBound.height;
    }
    for (const axis of axes) {
      const minorOffset = axisAreaWidths.get(minorDimension === "x" ? "left" : "top") ?? 0;
      const axisThickness = axisWidths.get(axis.id) ?? 0;
      const axisOffset = axisOffsets.get(axis.id) ?? 0;
      axis.gridPadding = axisAreaWidth - axisOffset - axisThickness;
      axis.translation[minorDimension] = axisBound[minorDimension] + minorOffset;
      axis.translation[mainDimension] = this.clampToOutsideSeriesRect(
        seriesRect,
        axisBoundMainOffset + direction * (axisOffset + axisThickness),
        mainDimension,
        direction
      );
    }
  }
  shouldFlipXY() {
    return this.series.every((series) => series instanceof CartesianSeries && series.shouldFlipXY());
  }
  getDefaultState() {
    const axisAreaWidths = /* @__PURE__ */ new Map();
    if (this.lastAreaWidths) {
      for (const { position = "left" } of this.axes) {
        const areaWidth = this.lastAreaWidths.get(position);
        if (areaWidth != null) {
          axisAreaWidths.set(position, areaWidth);
        }
      }
    }
    return { axisAreaWidths, clipSeries: false, overflows: false };
  }
  isLayoutStable(newState, prevState) {
    if (prevState.overflows !== newState.overflows || prevState.clipSeries !== newState.clipSeries) {
      return false;
    }
    for (const key of newState.axisAreaWidths.keys()) {
      if (!prevState.axisAreaWidths.has(key)) {
        return false;
      }
    }
    for (const [p, w] of prevState.axisAreaWidths.entries()) {
      const otherW = newState.axisAreaWidths.get(p);
      if ((w != null || otherW != null) && w !== otherW) {
        return false;
      }
    }
    return true;
  }
  clipAxis(axis, seriesRect, layoutBBox) {
    const gridLinePadding = Math.ceil(axis.gridLine?.width ?? 0);
    const axisLinePadding = Math.ceil(axis.line?.width ?? 0);
    let { width: width2, height: height2 } = seriesRect;
    width2 += axis.direction === "x" /* X */ ? gridLinePadding : axisLinePadding;
    height2 += axis.direction === "y" /* Y */ ? gridLinePadding : axisLinePadding;
    axis.clipGrid(seriesRect.x, seriesRect.y, width2, height2);
    switch (axis.position) {
      case "left":
      case "right":
        axis.clipTickLines(
          layoutBBox.x,
          seriesRect.y - gridLinePadding,
          layoutBBox.width + gridLinePadding,
          seriesRect.height + gridLinePadding * 2
        );
        break;
      case "top":
      case "bottom":
        axis.clipTickLines(
          seriesRect.x - gridLinePadding,
          layoutBBox.y,
          seriesRect.width + gridLinePadding * 2,
          layoutBBox.height + gridLinePadding
        );
        const { label, labelNodes, scale } = axis;
        if (ContinuousScale.is(scale) && label.enabled && label.avoidCollisions && labelNodes.length > 1) {
          const sortedLabels = labelNodes.toSorted((a, b) => a.translationY - b.translationY);
          const lastLabel = sortedLabels.at(-1);
          const lastLabelBBox = lastLabel.getBBox();
          const lastLabelInBounds = seriesRect.x + lastLabelBBox.y + lastLabelBBox.height <= layoutBBox.x + layoutBBox.width + this.padding.right;
          lastLabel.visible = lastLabelInBounds;
          if (lastLabelInBounds || axis.visibleRange[0] > 0 || axis.visibleRange[1] < 1) {
            sortedLabels[0].visible = true;
          } else {
            const firstLabelBBox = sortedLabels[0].getBBox();
            sortedLabels[0].visible = firstLabelBBox.y >= 0;
          }
        }
        break;
    }
  }
};
_CartesianChart.className = "CartesianChart";
_CartesianChart.type = "cartesian";
_CartesianChart.AxesPadding = 15;
var CartesianChart = _CartesianChart;

// packages/ag-charts-community/src/chart/cartesianChartModule.ts
var CartesianChartModule = {
  type: "chart",
  name: "cartesian",
  detect: isAgCartesianChartOptions,
  create(options, resources) {
    return new CartesianChart(options, resources);
  }
};

// packages/ag-charts-community/src/chart/axis/polarAxis.ts
var PolarAxis = class extends Axis {
  constructor() {
    super(...arguments);
    this.shape = "polygon";
    this.innerRadiusRatio = 0;
    this.defaultTickMinSpacing = 20;
  }
  updatePosition() {
    super.updatePosition();
    const selectionCtx = prepareAxisAnimationContext(this);
    const resetAxisFn = resetAxisSelectionFn(selectionCtx);
    this.axisGroup.setProperties(this.getAxisTransform());
    this.gridLineGroupSelection.each(resetAxisFn);
    this.tickLineGroupSelection.each(resetAxisFn);
    this.tickLabelGroupSelection.each(resetAxisLabelSelectionFn());
  }
  computeLabelsBBox(_options, _seriesRect) {
    return null;
  }
  computeRange() {
  }
  getAxisLinePoints() {
    return void 0;
  }
};
__decorateClass([
  Validate(UNION(["polygon", "circle"], "a polar axis shape"))
], PolarAxis.prototype, "shape", 2);
__decorateClass([
  Validate(RATIO)
], PolarAxis.prototype, "innerRadiusRatio", 2);

// packages/ag-charts-community/src/chart/series/polar/polarSeries.ts
var PolarSeries = class extends DataModelSeries {
  constructor({
    useLabelLayer = false,
    pickModes = [1 /* NEAREST_NODE */, 0 /* EXACT_SHAPE_MATCH */],
    canHaveAxes = false,
    animationResetFns,
    ...opts
  }) {
    super({
      ...opts,
      useLabelLayer,
      pickModes,
      directionKeys: {
        ["x" /* X */]: ["angleKey"],
        ["y" /* Y */]: ["radiusKey"]
      },
      directionNames: {
        ["x" /* X */]: ["angleName"],
        ["y" /* Y */]: ["radiusName"]
      },
      canHaveAxes
    });
    this.itemGroup = this.contentGroup.appendChild(new Group());
    this.nodeData = [];
    this.itemSelection = Selection.select(
      this.itemGroup,
      () => this.nodeFactory(),
      false
    );
    this.labelSelection = Selection.select(
      this.labelGroup,
      () => this.labelFactory(),
      false
    );
    this.highlightSelection = Selection.select(
      this.highlightGroup,
      () => this.nodeFactory()
    );
    this.highlightLabelSelection = Selection.select(
      this.highlightLabel,
      () => this.labelFactory()
    );
    /**
     * The center of the polar series (for example, the center of a pie).
     * If the polar chart has multiple series, all of them will have their
     * center set to the same value as a result of the polar chart layout.
     * The center coordinates are not supposed to be set by the user.
     */
    this.centerX = 0;
    this.centerY = 0;
    /**
     * The maximum radius the series can use.
     * This value is set automatically as a result of the polar chart layout
     * and is not supposed to be set by the user.
     */
    this.radius = 0;
    this.animationResetFns = animationResetFns;
    this.animationState = new StateMachine(
      "empty",
      {
        empty: {
          update: {
            target: "ready",
            action: (data) => this.animateEmptyUpdateReady(data)
          },
          reset: "empty",
          skip: "ready"
        },
        ready: {
          updateData: "waiting",
          clear: "clearing",
          highlight: (data) => this.animateReadyHighlight(data),
          highlightMarkers: (data) => this.animateReadyHighlightMarkers(data),
          resize: (data) => this.animateReadyResize(data),
          reset: "empty",
          skip: "ready"
        },
        waiting: {
          update: {
            target: "ready",
            action: (data) => this.animateWaitingUpdateReady(data)
          },
          reset: "empty",
          skip: "ready"
        },
        clearing: {
          update: {
            target: "empty",
            action: (data) => this.animateClearingUpdateEmpty(data)
          },
          reset: "empty",
          skip: "ready"
        }
      },
      () => this.checkProcessedDataAnimatable()
    );
  }
  getItemNodes() {
    return [...this.itemGroup.children()];
  }
  getNodeData() {
    return this.nodeData;
  }
  setSeriesIndex(index) {
    if (!super.setSeriesIndex(index))
      return false;
    this.contentGroup.zIndex = [index, 1 /* FOREGROUND */];
    this.highlightGroup.zIndex = [index, 2 /* HIGHLIGHT */];
    this.labelGroup.zIndex = [index, 3 /* LABEL */];
    return true;
  }
  resetAnimation(phase) {
    if (phase === "initial") {
      this.animationState.transition("reset");
    } else if (phase === "ready") {
      this.animationState.transition("skip");
    }
  }
  labelFactory() {
    const text = new Text();
    text.pointerEvents = 1 /* None */;
    return text;
  }
  addChartEventListeners() {
    this.destroyFns.push(
      this.ctx.chartEventManager?.addListener("legend-item-click", (event) => this.onLegendItemClick(event))
    );
  }
  getInnerRadius() {
    return 0;
  }
  computeLabelsBBox(_options, _seriesRect) {
    return null;
  }
  resetAllAnimation() {
    const { item, label } = this.animationResetFns ?? {};
    this.ctx.animationManager.stopByAnimationGroupId(this.id);
    if (item) {
      resetMotion([this.itemSelection, this.highlightSelection], item);
    }
    if (label) {
      resetMotion([this.labelSelection, this.highlightLabelSelection], label);
    }
    this.itemSelection.cleanup();
    this.labelSelection.cleanup();
    this.highlightSelection.cleanup();
    this.highlightLabelSelection.cleanup();
  }
  animateEmptyUpdateReady(_data) {
    this.ctx.animationManager.skipCurrentBatch();
    this.resetAllAnimation();
  }
  animateWaitingUpdateReady(_data) {
    this.ctx.animationManager.skipCurrentBatch();
    this.resetAllAnimation();
  }
  animateReadyHighlight(_data) {
    const { item, label } = this.animationResetFns ?? {};
    if (item) {
      resetMotion([this.highlightSelection], item);
    }
    if (label) {
      resetMotion([this.highlightLabelSelection], label);
    }
  }
  animateReadyHighlightMarkers(_data) {
  }
  animateReadyResize(_data) {
    this.resetAllAnimation();
  }
  animateClearingUpdateEmpty(_data) {
    this.ctx.animationManager.skipCurrentBatch();
    this.resetAllAnimation();
  }
  computeFocusBounds(opts) {
    const datum = this.getNodeData()?.[opts.datumIndex];
    if (datum !== void 0) {
      return this.itemSelection.select((node) => node instanceof Path && node.datum === datum)[0];
    }
    return void 0;
  }
  getSeriesRange(_direction, _visibleRange) {
    return [NaN, NaN];
  }
};

// packages/ag-charts-community/src/chart/polarChart.ts
var PolarChart = class extends Chart {
  constructor(options, resources) {
    super(options, resources);
    this.padding = new Padding(40);
    this.ctx.axisManager.axisGroup.zIndex = 6 /* AXIS_FOREGROUND */;
  }
  getChartType() {
    return "polar";
  }
  async performLayout(ctx) {
    const { layoutBox } = ctx;
    layoutBox.shrink(this.seriesArea.padding.toJson());
    const seriesRect = layoutBox.clone();
    this.seriesRect = seriesRect;
    this.animationRect = seriesRect;
    this.seriesRoot.translationX = seriesRect.x;
    this.seriesRoot.translationY = seriesRect.y;
    await this.computeCircle(seriesRect);
    this.axes.forEach((axis) => axis.update());
    this.ctx.layoutManager.emitLayoutComplete(ctx, {
      series: { visible: true, rect: seriesRect, paddedRect: seriesRect }
    });
  }
  updateAxes(seriesBox, cx, cy, radius) {
    const angleAxis = this.axes.find((axis) => axis.direction === "x" /* X */);
    const radiusAxis = this.axes.find((axis) => axis.direction === "y" /* Y */);
    if (!(angleAxis instanceof PolarAxis) || !(radiusAxis instanceof PolarAxis))
      return;
    const angleScale = angleAxis.scale;
    const innerRadiusRatio = radiusAxis.innerRadiusRatio;
    angleAxis.innerRadiusRatio = innerRadiusRatio;
    angleAxis.computeRange();
    angleAxis.gridLength = radius;
    radiusAxis.gridAngles = angleScale.ticks({
      nice: angleAxis.nice,
      interval: void 0,
      tickCount: void 0,
      minTickCount: 0,
      maxTickCount: Infinity
    })?.map((value) => angleScale.convert(value));
    radiusAxis.gridRange = angleAxis.range;
    radiusAxis.range = [radius, radius * innerRadiusRatio];
    [angleAxis, radiusAxis].forEach((axis) => {
      axis.translation.x = seriesBox.x + cx;
      axis.translation.y = seriesBox.y + cy;
      axis.calculateLayout();
    });
  }
  async computeCircle(seriesBox) {
    const polarSeries = this.series.filter(isPolarSeries);
    const polarAxes = this.axes.filter(isPolarAxis);
    const setSeriesCircle = (cx, cy, r) => {
      this.updateAxes(seriesBox, cx, cy, r);
      polarSeries.forEach((series) => {
        series.centerX = cx;
        series.centerY = cy;
        series.radius = r;
      });
      const pieSeries = polarSeries.filter((s) => s.type === "donut" || s.type === "pie");
      if (pieSeries.length > 1) {
        const innerRadii = pieSeries.map((series) => {
          const innerRadius = series.getInnerRadius();
          return { series, innerRadius };
        }).sort((a, b) => a.innerRadius - b.innerRadius);
        innerRadii.at(-1).series.surroundingRadius = void 0;
        for (let i = 0; i < innerRadii.length - 1; i++) {
          innerRadii[i].series.surroundingRadius = innerRadii[i + 1].innerRadius;
        }
      }
    };
    const centerX = seriesBox.width / 2;
    const centerY = seriesBox.height / 2;
    const initialRadius = Math.max(0, Math.min(seriesBox.width, seriesBox.height) / 2);
    let radius = initialRadius;
    setSeriesCircle(centerX, centerY, radius);
    const shake = async ({ hideWhenNecessary = false } = {}) => {
      const labelBoxes = [];
      for (const series of iterate(polarAxes, polarSeries)) {
        const box = await series.computeLabelsBBox({ hideWhenNecessary }, seriesBox);
        if (box) {
          labelBoxes.push(box);
        }
      }
      if (labelBoxes.length === 0) {
        setSeriesCircle(centerX, centerY, initialRadius);
        return;
      }
      const labelBox = BBox.merge(labelBoxes);
      const refined = this.refineCircle(labelBox, radius, seriesBox);
      setSeriesCircle(refined.centerX, refined.centerY, refined.radius);
      radius = refined.radius;
    };
    await shake();
    await shake();
    await shake();
    await shake({ hideWhenNecessary: true });
    await shake({ hideWhenNecessary: true });
    for (const series of iterate(polarAxes, polarSeries)) {
      await series.computeLabelsBBox({ hideWhenNecessary: true }, seriesBox);
    }
    return { radius, centerX, centerY };
  }
  refineCircle(labelsBox, radius, seriesBox) {
    const minCircleRatio = 0.5;
    const circleLeft = -radius;
    const circleTop = -radius;
    const circleRight = radius;
    const circleBottom = radius;
    let padLeft = Math.max(0, circleLeft - labelsBox.x);
    let padTop = Math.max(0, circleTop - labelsBox.y);
    let padRight = Math.max(0, labelsBox.x + labelsBox.width - circleRight);
    let padBottom = Math.max(0, labelsBox.y + labelsBox.height - circleBottom);
    padLeft = padRight = Math.max(padLeft, padRight);
    padTop = padBottom = Math.max(padTop, padBottom);
    const availCircleWidth = seriesBox.width - padLeft - padRight;
    const availCircleHeight = seriesBox.height - padTop - padBottom;
    let newRadius = Math.min(availCircleWidth, availCircleHeight) / 2;
    const minHorizontalRadius = minCircleRatio * seriesBox.width / 2;
    const minVerticalRadius = minCircleRatio * seriesBox.height / 2;
    const minRadius = Math.min(minHorizontalRadius, minVerticalRadius);
    if (newRadius < minRadius) {
      newRadius = minRadius;
      const horizontalPadding = padLeft + padRight;
      const verticalPadding = padTop + padBottom;
      if (2 * newRadius + verticalPadding > seriesBox.height) {
        const padHeight = seriesBox.height - 2 * newRadius;
        if (Math.min(padTop, padBottom) * 2 > padHeight) {
          padTop = padHeight / 2;
          padBottom = padHeight / 2;
        } else if (padTop > padBottom) {
          padTop = padHeight - padBottom;
        } else {
          padBottom = padHeight - padTop;
        }
      }
      if (2 * newRadius + horizontalPadding > seriesBox.width) {
        const padWidth = seriesBox.width - 2 * newRadius;
        if (Math.min(padLeft, padRight) * 2 > padWidth) {
          padLeft = padWidth / 2;
          padRight = padWidth / 2;
        } else if (padLeft > padRight) {
          padLeft = padWidth - padRight;
        } else {
          padRight = padWidth - padLeft;
        }
      }
    }
    const newWidth = padLeft + 2 * newRadius + padRight;
    const newHeight = padTop + 2 * newRadius + padBottom;
    return {
      centerX: (seriesBox.width - newWidth) / 2 + padLeft + newRadius,
      centerY: (seriesBox.height - newHeight) / 2 + padTop + newRadius,
      radius: newRadius
    };
  }
};
PolarChart.className = "PolarChart";
PolarChart.type = "polar";
function isPolarSeries(series) {
  return series instanceof PolarSeries;
}
function isPolarAxis(axis) {
  return axis instanceof PolarAxis;
}

// packages/ag-charts-community/src/chart/polarChartModule.ts
var PolarChartModule = {
  type: "chart",
  name: "polar",
  detect: isAgPolarChartOptions,
  create(options, resources) {
    return new PolarChart(options, resources);
  }
};
export {
  CartesianChartModule,
  ModuleRegistry,
  PolarChartModule
};
