declare const descriptionSymbol: unique symbol;
declare const requiredSymbol: unique symbol;
type ObjectLikeDef<T> = T extends object ? (keyof T extends never ? never : OptionsDefs<T>) : never;
export type OptionsDefs<T> = {
    [K in keyof T]-?: Validator | ObjectLikeDef<T[K]>;
} & {
    [descriptionSymbol]?: string;
    [requiredSymbol]?: boolean;
};
interface Validator extends Function {
    (value: unknown, context?: any): boolean;
    [descriptionSymbol]?: string;
    [requiredSymbol]?: boolean;
}
interface ValidationError {
    key?: string;
    path: string;
    message: string;
    unknown?: boolean;
    value?: any;
}
/**
 * Validates the provided options against the specified definitions.
 * @param options The options object to validate.
 * @param optionsDefs The definitions against which to validate the options.
 * @param path The current path in the options object, for nested properties.
 * @returns An object containing valid options and validation errors.
 */
export declare function validate<T>(options: unknown, optionsDefs: OptionsDefs<T>, path?: string): {
    valid: null;
    errors: {
        path: string;
        value: unknown;
        message: string;
    }[];
} | {
    valid: Partial<T>;
    errors: ValidationError[];
};
/**
 * Validates the provided options object against the specified options definitions. Logs warnings for any invalid options encountered.
 * @param options The options object to validate.
 * @param optionsDefs The definitions against which to validate the options.
 * @param path (Optional) The current path in the options object, for nested properties.
 * @returns A boolean indicating whether the options are valid.
 */
export declare function isValid<T extends object>(options: unknown, optionsDefs: OptionsDefs<T>, path?: string): options is T;
/**
 * Attaches a descriptive message to a validator function.
 * @param validator The validator function to which to attach a description.
 * @param description The description to attach.
 * @returns A new validator function with the attached description.
 */
export declare function attachDescription(validator: Validator, description: string): Validator;
/**
 * Marks a validator or option definitions object as required.
 * @param validatorOrDefs The validator or option definitions to mark as required.
 * @returns The modified validator or option definitions, marked as required.
 */
export declare function required<T extends OptionsDefs<any>>(validatorOrDefs: T): T;
export declare function required<T extends OptionsDefs<any>[]>(validatorOrDefs: T): T;
export declare function required(validatorOrDefs: Validator): Validator;
/**
 * Creates a validator for ensuring an object matches the provided option definitions.
 * @param defs The option definitions against which to validate an object.
 * @param description (Optional) A description for the validator, defaulting to 'an object'.
 * @returns A validator function for the given option definitions.
 */
export declare const optionsDefs: <T>(defs: OptionsDefs<T>, description?: string) => Validator;
/**
 * Combines multiple validators, requiring all to pass.
 * @param validators An array of validators to combine.
 * @returns A validator that requires all specified validators to pass.
 */
export declare const and: (...validators: Validator[]) => Validator;
/**
 * Combines multiple validators, passing if any one of them does.
 * @param validators An array of validators to combine.
 * @returns A validator that passes if any one of the specified validators does.
 */
export declare const or: (...validators: Validator[]) => Validator;
export declare const array: Validator;
export declare const boolean: Validator;
export declare const callback: Validator;
export declare const number: Validator;
export declare const object: Validator;
export declare const string: Validator;
export declare const date: Validator;
export declare const positiveNumber: Validator;
export declare const ratio: Validator;
export declare const lessThan: (otherField: string) => Validator;
/**
 * Creates a validator for a union of allowed values.
 * @param allowed An array of allowed values.
 * @returns A validator function that checks if a value is among the allowed ones.
 */
export declare function union(allowed: object): Validator;
export declare function union(...allowed: any[]): Validator;
/**
 * Creates a validator for a single constant value.
 * @param allowed The allowed constant value.
 * @returns A validator function that checks for equality with the allowed value.
 */
export declare const constant: (allowed: boolean | number | string) => Validator;
/**
 * Creates a validator for instances of a specific class.
 * @param instanceType The constructor of the class to check instances against.
 * @param description An optional description string.
 * @returns A validator function that checks if a value is an instance of the specified class.
 */
export declare const instanceOf: (instanceType: Function, description?: string) => Validator;
/**
 * Creates a validator for arrays where every element must pass a given validator.
 * @param validator The validator to apply to each array element.
 * @param description An optional description string.
 * @returns A validator function for arrays with elements validated by the specified validator.
 */
export declare const arrayOf: (validator: Validator, description?: string) => Validator;
export {};
