import type { BBox } from './bbox';
import { Matrix } from './matrix';
import { Node } from './node';
type Constructor<T> = new (...args: any[]) => T;
interface LocalToParentCoordinateSpaceTransforms {
    /** Apply local node transforms to the given BBox. */
    toParent(bbox: BBox): BBox;
    /** Apply local node transforms to the given point. */
    toParentPoint(x: number, y: number): {
        x: number;
        y: number;
    };
}
interface ParentToLocalCoordinateSpaceTransforms {
    /** Apply local node inverse transforms to the given BBox. */
    fromParent(bbox: BBox): BBox;
    /** Apply local node inverse transforms to the given point. */
    fromParentPoint(x: number, y: number): {
        x: number;
        y: number;
    };
}
type MatrixTransformType<T> = T & LocalToParentCoordinateSpaceTransforms & ParentToLocalCoordinateSpaceTransforms & {
    updateMatrix(matrix: Matrix): void;
    computeBBoxWithoutTransforms(): BBox | undefined;
};
export type RotatableType<T> = MatrixTransformType<T & {
    rotationCenterX: number | null;
    rotationCenterY: number | null;
    rotation: number;
}>;
/** Mixin type for scene Nodes that are rotatable. */
export declare function Rotatable<N extends Node>(Parent: Constructor<N>): Constructor<RotatableType<N>>;
export type ScalableType<T> = MatrixTransformType<T & {
    scalingX: number;
    scalingY: number;
    scalingCenterX: number | null;
    scalingCenterY: number | null;
}>;
/** Mixin type for scene Nodes that are scalable. */
export declare function Scalable<N extends Node>(Parent: Constructor<N>): Constructor<ScalableType<N>>;
export type TranslatableType<T> = MatrixTransformType<T & {
    translationX: number;
    translationY: number;
}>;
/** Mixin type for scene Nodes that are translatable. */
export declare function Translatable<N extends Node>(Parent: Constructor<N>): Constructor<TranslatableType<N>>;
/** Utility class for operations relating to matrix-transformable mixin types. */
export declare class Transformable {
    /**
     * Converts a BBox from canvas coordinate space into the coordinate space of the given Node.
     */
    static fromCanvas(node: Node, bbox: BBox): BBox;
    /**
     * Converts a Nodes BBox (or an arbitrary BBox if supplied) from local Node coordinate space
     * into the Canvas coordinate space.
     */
    static toCanvas(node: Node, bbox?: BBox): BBox;
    /**
     * Converts a point from canvas coordinate space into the coordinate space of the given Node.
     */
    static fromCanvasPoint(node: Node, x: number, y: number): {
        x: number;
        y: number;
    };
    /**
     * Converts a point from a Nodes local coordinate space into the Canvas coordinate space.
     */
    static toCanvasPoint(node: Node, x: number, y: number): {
        x: number;
        y: number;
    };
}
export {};
