import { LinePattern } from "../render/patterns/LinePattern";
import { RectanglePattern } from "../render/patterns/RectanglePattern";
import { CirclePattern } from "../render/patterns/CirclePattern";
import { Entity } from "./Entity";
/**
 * An object which holds list of [[Pattern]] objects and can serve them up in
 * an interative way.
 *
 * @see {@link https://www.amcharts.com/docs/v5/concepts/colors-gradients-and-patterns/patterns/#Pattern_sets} for more info
 * @since 5.10.0
 */
export class PatternSet extends Entity {
    _afterNew() {
        // Applying themes because pattern set will not have parent
        super._afterNewApplyThemes();
        if (this.get("patterns", []).length === 0) {
            const color = this.get("color", this.root.interfaceColors.get("stroke"));
            this.set("patterns", [
                this.getLinePattern({
                    width: 1000,
                    height: 1000,
                    rotation: 45,
                    strokeWidth: 1,
                    //gap: 6,
                    color: color
                }),
                this.getRectanglePattern({
                    width: 10,
                    height: 10,
                    rotation: 0,
                    maxWidth: 4,
                    maxHeight: 4,
                    color: color
                }),
                this.getLinePattern({
                    width: 1000,
                    height: 1000,
                    rotation: -45,
                    strokeWidth: 1,
                    gap: 6,
                    color: color
                }),
                this.getCirclePattern({
                    width: 11,
                    height: 11,
                    radius: 2,
                    color: color
                }),
                this.getLinePattern({
                    width: 6,
                    height: 6,
                    rotation: 90,
                    strokeWidth: 1,
                    color: color
                }),
                this.getRectanglePattern({
                    width: 14,
                    height: 14,
                    rotation: 45,
                    gap: 4,
                    maxWidth: 6,
                    maxHeight: 6,
                    checkered: true,
                    color: color
                }),
                this.getLinePattern({
                    width: 6,
                    height: 6,
                    rotation: 0,
                    strokeWidth: 1,
                    color: color
                }),
                this.getRectanglePattern({
                    width: 15,
                    height: 15,
                    rotation: 0,
                    gap: 5,
                    maxWidth: 5,
                    maxHeight: 5,
                    checkered: true,
                    color: color
                }),
                this.getLinePattern({
                    width: 1000,
                    height: 1000,
                    rotation: 45,
                    strokeWidth: 2,
                    gap: 3,
                    strokeDasharray: [4, 2],
                    color: color
                }),
                this.getCirclePattern({
                    width: 20,
                    height: 20,
                    radius: 3,
                    gap: 4,
                    checkered: true,
                    color: color
                }),
                this.getLinePattern({
                    width: 1000,
                    height: 1000,
                    rotation: -45,
                    strokeWidth: 2,
                    gap: 3,
                    strokeDasharray: [4, 2],
                    color: color
                }),
                this.getRectanglePattern({
                    width: 10,
                    height: 10,
                    rotation: 0,
                    gap: 1,
                    maxWidth: 9,
                    maxHeight: 9,
                    color: color
                }),
                this.getLinePattern({
                    width: 1000,
                    height: 1000,
                    rotation: -45,
                    strokeWidth: 2,
                    gap: 1,
                    color: color
                }),
                this.getLinePattern({
                    width: 1000,
                    height: 1000,
                    rotation: 45,
                    strokeWidth: 2,
                    gap: 1,
                    color: color
                }),
                this.getLinePattern({
                    width: 1000,
                    height: 1000,
                    rotation: 0,
                    strokeWidth: 3,
                    gap: 1,
                    color: color
                }),
                this.getLinePattern({
                    width: 1000,
                    height: 1000,
                    rotation: 90,
                    strokeWidth: 3,
                    gap: 1,
                    color: color
                }),
            ]);
        }
        this._dirty["patterns"] = false;
    }
    _beforeChanged() {
        if (this.isDirty("patterns")) {
            this.reset();
        }
    }
    /**
     * Returns a [[Pattern]] at specific index.
     *
     * @param   index  Index
     * @return         Color
     */
    getIndex(index) {
        const patterns = this.get("patterns", []);
        if ((index < patterns.length) && patterns[index] !== null) {
            return patterns[index];
        }
        if (index > (patterns.length - 1)) {
            const adjustedIndex = index - Math.floor(index * (index / patterns.length));
            return patterns[adjustedIndex];
        }
        return patterns[index];
    }
    /**
     * Returns next [[Color]] in the list.
     *
     * If the list is out of colors, new ones are generated dynamically.
     */
    next() {
        let currentStep = this.getPrivate("currentStep", this.get("startIndex", 0));
        this.setPrivate("currentStep", currentStep + this.get("step", 1));
        return this.getIndex(currentStep);
    }
    /**
     * Resets counter to the start of the list, so the next call for `next()` will
     * return the first pattern.
     */
    reset() {
        this.setPrivate("currentStep", this.get("startIndex", 0));
    }
    /**
     * Returns a [[LinePattern].
     *
     * @param   settings  Pattern settings
     * @return            Pattern object
     */
    getLinePattern(settings) {
        let pattern = LinePattern.new(this.root, settings);
        return pattern;
    }
    /**
     * Returns a [[RectanglePattern].
     *
     * @param   settings  Pattern settings
     * @return            Pattern object
     */
    getRectanglePattern(settings) {
        let pattern = RectanglePattern.new(this.root, settings);
        return pattern;
    }
    /**
     * Returns a [[CirclePattern].
     *
     * @param   settings  Pattern settings
     * @return            Pattern object
     */
    getCirclePattern(settings) {
        let pattern = CirclePattern.new(this.root, settings);
        return pattern;
    }
}
Object.defineProperty(PatternSet, "className", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: "PatternSet"
});
Object.defineProperty(PatternSet, "classNames", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: Entity.classNames.concat([PatternSet.className])
});
//# sourceMappingURL=PatternSet.js.map