import { ChartIndicator } from "./ChartIndicator";
import { LineSeries } from "../../xy/series/LineSeries";
import * as $type from "../../../core/util/Type";
/**
 * An implementation of a [[StockChart]] indicator.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/stock/indicators/} for more info
 */
export class PVT extends ChartIndicator {
    constructor() {
        super(...arguments);
        Object.defineProperty(this, "_editableSettings", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: [{
                    key: "seriesColor",
                    name: this.root.language.translateAny("Color"),
                    type: "color"
                }]
        });
    }
    _afterNew() {
        this._themeTags.push("pvt");
        super._afterNew();
        this.yAxis.set("numberFormat", "#.###a");
    }
    _createSeries() {
        return this.panel.series.push(LineSeries.new(this._root, {
            themeTags: ["indicator"],
            xAxis: this.xAxis,
            yAxis: this.yAxis,
            valueXField: "valueX",
            valueYField: "pvt",
            stroke: this.get("seriesColor"),
            fill: undefined
        }));
    }
    /**
     * @ignore
     */
    prepareData() {
        if (this.series) {
            this.setRaw("field", "close");
            const dataItems = this.get("stockSeries").dataItems;
            const volumeSeries = this.get("volumeSeries");
            let data = this._getDataArray(dataItems);
            let previous = 0;
            let len = data.length;
            //PVT = [((CurrentClose - PreviousClose) / PreviousClose) x Volume] + PreviousPVT
            if (volumeSeries && len > 1) {
                let cy = data[0].value_y;
                for (let i = 1; i < len; i++) {
                    const dataItem = data[i];
                    let c = dataItem.value_y;
                    if (c != null && $type.isNumber(c) && c != 0) {
                        const volumeDI = volumeSeries.dataItems[i];
                        let volume = 0;
                        if (volumeDI) {
                            volume = volumeDI.get("valueY", 1);
                        }
                        let pvt = (((c - cy) / cy) * volume) + previous;
                        dataItem.pvt = pvt;
                        previous = pvt;
                        cy = c;
                    }
                }
            }
            this.series.data.setAll(data);
        }
    }
}
Object.defineProperty(PVT, "className", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: "PVT"
});
Object.defineProperty(PVT, "classNames", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: ChartIndicator.classNames.concat([PVT.className])
});
//# sourceMappingURL=PVT.js.map