package com.logship.mobile;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.util.Log;
import android.webkit.WebView;

/**
 * Broadcast receiver for enterprise barcode scanners
 * Supports: Zebra DataWedge, Honeywell, Urovo, Newland, Netum, and generic Android PDA scanners
 * Receives scanner intents and forwards them to the WebView via JavaScript
 */
public class ScannerBroadcastReceiver extends BroadcastReceiver {
    private static final String TAG = "ScannerReceiver";
    private WebView webView;

    public ScannerBroadcastReceiver(WebView webView) {
        this.webView = webView;
    }

    @Override
    public void onReceive(Context context, Intent intent) {
        if (intent == null) return;

        String action = intent.getAction();
        Log.d(TAG, "Received broadcast action: " + action);

        // Log all extras for debugging
        Bundle extras = intent.getExtras();
        if (extras != null) {
            for (String key : extras.keySet()) {
                Object value = extras.get(key);
                Log.d(TAG, "  Extra: " + key + " = " + (value != null ? value.toString() : "null"));
            }
        }

        String barcode = extractBarcode(intent);

        if (barcode != null && !barcode.isEmpty()) {
            Log.d(TAG, "Barcode extracted: " + barcode);
            sendToWebView(barcode);
        } else {
            Log.d(TAG, "No barcode found in intent");
        }
    }

    private String extractBarcode(Intent intent) {
        String barcode = null;

        // Zebra DataWedge
        if (barcode == null && intent.hasExtra("com.symbol.datawedge.data_string")) {
            barcode = intent.getStringExtra("com.symbol.datawedge.data_string");
            Log.d(TAG, "Found Zebra DataWedge barcode");
        }

        // Urovo devices
        if (barcode == null && intent.hasExtra("barcode_string")) {
            barcode = intent.getStringExtra("barcode_string");
            Log.d(TAG, "Found Urovo barcode_string");
        }

        // Urovo byte array fallback
        if (barcode == null && intent.hasExtra("decode_data")) {
            byte[] data = intent.getByteArrayExtra("decode_data");
            if (data != null) {
                barcode = new String(data);
                Log.d(TAG, "Found Urovo decode_data");
            }
        }

        // Honeywell
        if (barcode == null && intent.hasExtra("data")) {
            barcode = intent.getStringExtra("data");
            Log.d(TAG, "Found Honeywell data");
        }

        // Newland
        if (barcode == null && intent.hasExtra("SCAN_BARCODE1")) {
            barcode = intent.getStringExtra("SCAN_BARCODE1");
            Log.d(TAG, "Found Newland SCAN_BARCODE1");
        }

        // Generic Android PDA scanners
        if (barcode == null && intent.hasExtra("scannerdata")) {
            barcode = intent.getStringExtra("scannerdata");
            Log.d(TAG, "Found scannerdata");
        }

        if (barcode == null && intent.hasExtra("scan_result")) {
            barcode = intent.getStringExtra("scan_result");
            Log.d(TAG, "Found scan_result");
        }

        if (barcode == null && intent.hasExtra("value")) {
            barcode = intent.getStringExtra("value");
            Log.d(TAG, "Found value");
        }

        // Common fallback names
        if (barcode == null && intent.hasExtra("SCAN_BARCODE_DATA")) {
            barcode = intent.getStringExtra("SCAN_BARCODE_DATA");
            Log.d(TAG, "Found SCAN_BARCODE_DATA");
        }

        if (barcode == null && intent.hasExtra("barcode")) {
            barcode = intent.getStringExtra("barcode");
            Log.d(TAG, "Found barcode");
        }

        if (barcode == null && intent.hasExtra("scanData")) {
            barcode = intent.getStringExtra("scanData");
            Log.d(TAG, "Found scanData");
        }

        if (barcode == null && intent.hasExtra("Barcode")) {
            barcode = intent.getStringExtra("Barcode");
            Log.d(TAG, "Found Barcode (capital)");
        }

        if (barcode == null && intent.hasExtra("EXTRA_BARCODE_DECODING_DATA")) {
            barcode = intent.getStringExtra("EXTRA_BARCODE_DECODING_DATA");
            Log.d(TAG, "Found EXTRA_BARCODE_DECODING_DATA");
        }

        // Honeywell HS7 specific (used in Netum Q900)
        if (barcode == null && intent.hasExtra("aimId")) {
            barcode = intent.getStringExtra("data");
            Log.d(TAG, "Found Honeywell aimId/data");
        }

        // Netum Q900 custom extras (as configured by user)
        if (barcode == null && intent.hasExtra("barcode_string")) {
            barcode = intent.getStringExtra("barcode_string");
            Log.d(TAG, "Found barcode_string (Netum custom)");
        }

        if (barcode == null && intent.hasExtra("barcode_data")) {
            byte[] data = intent.getByteArrayExtra("barcode_data");
            if (data != null) {
                barcode = new String(data);
                Log.d(TAG, "Found barcode_data bytes (Netum custom)");
            }
        }

        // Last resort - check for any string extra that looks like a barcode
        if (barcode == null) {
            Bundle extras = intent.getExtras();
            if (extras != null) {
                for (String key : extras.keySet()) {
                    Object value = extras.get(key);
                    if (value instanceof String) {
                        String strValue = (String) value;
                        // If it looks like barcode data (alphanumeric, reasonable length)
                        if (strValue.length() > 3 && strValue.length() < 100 && strValue.matches("[A-Za-z0-9\\-\\_\\.]+")) {
                            barcode = strValue;
                            Log.d(TAG, "Found potential barcode in extra '" + key + "'");
                            break;
                        }
                    }
                }
            }
        }

        return barcode;
    }

    private void sendToWebView(final String barcode) {
        if (webView != null) {
            // Escape the barcode for JavaScript
            final String escapedBarcode = barcode.replace("\\", "\\\\")
                                                  .replace("'", "\\'")
                                                  .replace("\"", "\\\"")
                                                  .replace("\n", "\\n")
                                                  .replace("\r", "\\r");

            webView.post(new Runnable() {
                @Override
                public void run() {
                    // Dispatch custom event to the webview
                    String js = "console.log('[NativeScan] Received: " + escapedBarcode + "'); window.dispatchEvent(new CustomEvent('nativeScan', { detail: { barcode: '" + escapedBarcode + "' } }));";
                    webView.evaluateJavascript(js, null);
                    Log.d(TAG, "Sent barcode to WebView: " + escapedBarcode);
                }
            });
        }
    }
}
